/*********************************************************************
----------------------------------------------------------------------
File    : main.c
Purpose : Sample program for CortexM
--------- END-OF-HEADER --------------------------------------------*/

#include "main.h"

/*********************************************************************
*
*       Defines
*
**********************************************************************
*/
#define U8  unsigned char
#define U16 unsigned short
#define U32 unsigned long
#define U64 unsigned long long

#define I8  signed char
#define I16 signed short
#define I32 signed long

/*********************************************************************
*
*       Types
*
*********************************************************************/
typedef struct {
  U32 securityMode;
  U32 mainPasswd;
  U32 commonCfg0;
  U32 commonCfg1;
} BOOT_INFO_HEADER;

typedef struct {
  U32 codeSig;
  U32 codeLen;
  U32 flashStartAddr;
  U32 sramStartAddr;
  U32 sramEntryAddr;
  U32 bootCfg0;
  U32 bootCfg1;
  U32 reserved;
} SECTION_HEADER;

typedef struct {
  BOOT_INFO_HEADER InfoHeader;
  SECTION_HEADER   FlashSectionHeader;
  SECTION_HEADER   SPISectionHeader;
  SECTION_HEADER   USBSectionHeader;
} BOOT_INFO;

/*********************************************************************
*
*       Global data
*
*********************************************************************/

volatile int Cnt;

/*********************************************************************
*
*       Const data
*
*********************************************************************/

const BOOT_INFO BootInfo __attribute__((section(".BootCfg"))) = {
  //
  // Boot info header
  //
  {
    0x00000000,                    // securityMode: No security
    0x00000000,                    // mainPasswd: No Password
      (0x00   << 0)                // Select flash boot header (first section header) to be used for boot
    | (0x03   << 8)                // System clock source == RC32M
    | (0x7F   << 10)               // Keep reserved bits at 1
    | (0x01   << 17)               // Use default SPI prescaler (2, otherwise bits 22:18 of commonCfg0 determine prescaler)
    | (0x02   << 18)               // SPI prescaler == 2 (do not care, since
    | (0x3F   << 23)               // Keep at 1s
    | (0x03   << 29)               // Keep at 1s
    | (0x01uL << 31),              // No change to flash status register
    0xFFFFFFFF                     // (CommCfg1)
  },
  //
  // Flash section header (we want to boot from flash)
  //
  {
    0x00000000, // codeSig: We do not use CRC for now
    0x00000400, // codeLen: Size of image to be copied to SRAM on boot
    0x00000800, // flashStartAddr: Offset into flash (from base addr.) where image to be copied into SRAM can be found. We define a 32 KB boot header section, so second level bootloader starts right behind this
    0x00100000, // sramStartAddr: Address in SRAM where image will be copied to. Our mini 2nd level bootloader being copied by ROM BTL is position independend, so start at SRAM base address
    0x00100001, // sramEntryAddr: Entry point into image loaded to SRAM. 2nd level bootloader starts at SRAM base address
    0xFFFFFFFF, // bootCfg0: Leave at default (1s)
    0xFFFFFFFF, // bootCfg1: Leave at default (1s)
    0xFFFFFFFF, // reserved: Leave at default (1s)
  },
  //
  // SPI section header, all empty because not used
  //
  { 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF },
  //
  // USB section header
  //
  { 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF }
};

/*********************************************************************
*
*       Function Declarations
*
*********************************************************************/
void MiniBTL(void) __attribute__((naked, section(".2ndLevelBTL")));

/*********************************************************************
*
*       Function Definitions
*
*********************************************************************/

/*********************************************************************
*
*       MiniBTL
*
*  Function description
*    Placed at start of flash (extra region, see linker file)
*    Copied by ROM bootloader of 88MC200 into internal RAM.
*    Just contains a mini bootloader which makes sure that flash clocks are enabled & flash is in memory mapped mode
*    Then simply jumps to the real application in memory-mapped SPI flash
*/
void MiniBTL(void) {
  asm(
      "LDR R3, =__PMU\r\n"
      //
      // Set bits 22 & 25 in PMU_IO_PAD_PWR_CFG to enable flash power domain
      //
      "LDR R2, [R3, #+0xA4]\r\n"    // PMU_IO_PAD_PWR_CFG
      "MOVS R1, #+9\r\n"
      "LSLS R1, R1, #+22\r\n"
      "ORRS R2, R1\r\n"
      "STR R2, [R3, #+0xA4]\r\n"
      //
      // Enable QSPI clock by setting PMU_PERI_CLK_EN[1:1]
      //
      "LDR R2, [R3, #+0x7C]\r\n"
      "MOVS R1, #+2\r\n"
      "BIC R2, R1\r\n"
      "STR R2, [R3, #+0x7C]\r\n"
      //
      // Put flash into memory-mapped mode (set FLASHC[31:31]
      //
      "LDR R3, =__FLASHC\r\n"
      "LDR R2, [R3, #+0]\r\n"
      "MOVS R1, #+1\r\n"
      "LSLS R1, R1, #+31\r\n"
      "ORRS R2, R1\r\n"
      "STR R2, [R3, #+0]\r\n"
      //
      // Setup stackpointer for main application
      //
      "LDR r3, =__StackTop\r\n"    // Take global stackpointer symbol for main application stackpointer
      "MOV sp, r3\r\n"
      //
      // Jump to main application
      //
      "LDR r3, =Reset_Handler\r\n"   // Jump to start of application an make sure that complete startup code is executed
      "BX r3\r\n"
      //
      // Data table for register base addresses
      //
      ".equ __PMU, 0x480A0000\r\n"
      ".equ __FLASHC, 0x44003000\r\n"
      );
}

/*********************************************************************
*
*       Public functions
*
*********************************************************************/

/*********************************************************************
*
*       SystemInit
*/
void SystemInit(void) {
  //
  // Add system init here
  //
}

/*********************************************************************
*
*       main
*/
int main(void) {
  Cnt = 0;
  while (1) {
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
    Cnt++;
  }
  return 0;
}

