/****************************************************************************/
/**
* @file     XMC4500.h
*           XMC4000 Device Series
* @version  V1.16
* @date     02 Apr 2012
*
Copyright (C) 2011-2012 Infineon Technologies AG. All rights reserved.
* 
*
* @par
* Infineon Technologies AG (Infineon) is supplying this software for use with Infineon's
* microcontrollers.  This file can be freely distributed
* within development tools that are supporting such microcontrollers.
*
* @par
* THIS SOFTWARE IS PROVIDED AS IS.  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
* OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
* INFINEON SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
* CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
*
******************************************************************************/

/** @mainpage  CMSIS device specific (XMC4500) peripheral access layer.
*
* @par
* This file provides the XMC4500 Device specific peripheral access layer for
* all peripherals. This file contains all the data structures and the address
* mapping of the device specific peripherals.
* @par
* The file also provides interrupt numbers (IRQHandler) for all core and device
* specific exceptions and interrupts.
*
*/

#ifndef __XMC4500_H__
#define __XMC4500_H__

#ifdef __cplusplus
extern "C" {
#endif 



/********************************************
** Start of section using anonymous unions **
*********************************************/
#if defined(__CC_ARM)
  #pragma push
  #pragma anon_unions
#elif defined(__ICCARM__)
  #pragma language=extended
#elif defined(__GNUC__)
  /* anonymous unions are enabled by default */ 
  #elif defined(__TMS470__)
/* anonymous unions are enabled by default */ 
#elif defined(__TASKING__)
  #pragma warning 586 
#else
  #warning Not supported compiler type
#endif

/*
* ==========================================================================
* --------------------- __NVIC_PRIO_BITS -----------------------------------
* ==========================================================================
*/
#ifndef __NVIC_PRIO_BITS
#define __NVIC_PRIO_BITS (6)
#endif

/*
* ==========================================================================
* ----------------------------- M4 Stuff -----------------------------------
* ==========================================================================
*/
#define __CM4_REV                      0x0001    /**< Core revision r0p1 */
#define __MPU_PRESENT                  1         /**< MPU present or not */
#define __Vendor_SysTickConfig         0         /**< Set to 1 if different SysTick Config is used */
#define __FPU_PRESENT                  1         /**< FPU present or not */

/*
* ==========================================================================
* ---------- Interrupt Number Definition -----------------------------------
* ==========================================================================
*/

typedef enum IRQn
{
/******  Cortex-M3 Processor Exceptions Numbers **********************************/
NMI_IRQn            = -14,      /*!<  2 Non Maskable Interrupt                */
MemManage_IRQn         = -12,      /*!<  4 Cortex-M4 Memory Management Interrupt */
BusFault_IRQn          = -11,      /*!<  5 Cortex-M4 Bus Fault Interrupt         */
UsageFault_IRQn        = -10,      /*!<  6 Cortex-M4 Usage Fault Interrupt       */
SVC_IRQn               = -5,       /*!< 11 Cortex-M4 SV Call Interrupt           */
DebugMon_IRQn          = -4,       /*!< 12 Cortex-M4 Debug Monitor Interrupt     */
PendSV_IRQn            = -2,       /*!< 14 Cortex-M4 Pend SV Interrupt           */
SysTick_IRQn           = -1,       /*!< 15 Cortex-M4 System Tick Interrupt       */

/******  XMC45xx Specific Interrupt Numbers ***************************************/
SCU_0_IRQn              =   0,      /*!< SCU SR0 Interrupt                        */
ERU0_0_IRQn             =   1,      /*!< SCU SR1 Interrupt                        */
ERU0_1_IRQn             =   2,      /*!< SCU SR2 Interrupt                        */
ERU0_2_IRQn             =   3,      /*!< SCU SR3 Interrupt                        */
ERU0_3_IRQn             =   4,      /*!< SCU SR4 Interrupt                        */
ERU1_0_IRQn             =   5,      /*!< SCU SR5 Interrupt                        */
ERU1_1_IRQn             =   6,      /*!< SCU SR6 Interrupt                        */
ERU1_2_IRQn             =   7,      /*!< SCU SR7 Interrupt                        */
ERU1_3_IRQn             =   8,      /*!< SCU SR8 Interrupt                        */
PMU0_0_IRQn             =  12,      /*!< PMU SR0 Interrupt                        */
VADC0_C0_0_IRQn         =  14,      /*!< VADC SR0 Interrupt                       */
VADC0_C0_1_IRQn         =  15,      /*!< VADC SR1 Interrupt                       */
VADC0_C0_2_IRQn         =  16,      /*!< VADC SR2 Interrupt                       */
VADC0_C0_3_IRQn         =  17,      /*!< VADC SR3 Interrupt                       */
VADC0_G0_0_IRQn         =  18,      /*!< VADC SR4 Interrupt                       */
VADC0_G0_1_IRQn         =  19,      /*!< VADC SR5 Interrupt                       */
VADC0_G0_2_IRQn         =  20,      /*!< VADC SR6 Interrupt                       */
VADC0_G0_3_IRQn         =  21,      /*!< VADC SR7 Interrupt                       */
VADC0_G1_0_IRQn         =  22,      /*!< VADC SR8 Interrupt                       */
VADC0_G1_1_IRQn         =  23,      /*!< VADC SR9 Interrupt                       */
VADC0_G1_2_IRQn         =  24,      /*!< VADC SR10 Interrupt                      */
VADC0_G1_3_IRQn         =  25,      /*!< VADC SR11 Interrupt                      */
VADC0_G2_0_IRQn         =  26,      /*!< VADC SR12 Interrupt                      */
VADC0_G2_1_IRQn         =  27,      /*!< VADC SR13 Interrupt                      */
VADC0_G2_2_IRQn         =  28,      /*!< VADC SR14 Interrupt                      */
VADC0_G2_3_IRQn         =  29,      /*!< VADC SR15 Interrupt                      */
VADC0_G3_0_IRQn         =  30,      /*!< VADC SR16 Interrupt                      */
VADC0_G3_1_IRQn         =  31,      /*!< VADC SR17 Interrupt                      */
VADC0_G3_2_IRQn         =  32,      /*!< VADC SR18 Interrupt                      */
VADC0_G3_3_IRQn         =  33,      /*!< VADC SR19 Interrupt                      */
DSD0_M_0_IRQn           =  34,      /*!< DSD SR0 Interrupt                        */
DSD0_M_1_IRQn           =  35,      /*!< DSD SR1 Interrupt                        */
DSD0_M_2_IRQn           =  36,      /*!< DSD SR2 Interrupt                        */
DSD0_M_3_IRQn           =  37,      /*!< DSD SR3 Interrupt                        */
DSD0_A_4_IRQn           =  38,      /*!< DSD SR4 Interrupt                        */
DSD0_A_5_IRQn           =  39,      /*!< DSD SR5 Interrupt                        */
DSD0_A_6_IRQn           =  40,      /*!< DSD SR6 Interrupt                        */
DSD0_A_7_IRQn           =  41,      /*!< DSD SR7 Interrupt                        */
DAC0_0_IRQn             =  42,      /*!< DAC SR0 Interrupt                        */
DAC1_1_IRQn             =  43,      /*!< DAC SR1 Interrupt                        */
CCU40_0_IRQn            =  44,      /*!< CCU40 SR0 Interrupt                      */
CCU40_1_IRQn            =  45,      /*!< CCU40 SR1 Interrupt                      */
CCU40_2_IRQn            =  46,      /*!< CCU40 SR2 Interrupt                      */
CCU40_3_IRQn            =  47,      /*!< CCU40 SR3 Interrupt                      */
CCU41_0_IRQn            =  48,      /*!< CCU41 SR0 Interrupt                      */
CCU41_1_IRQn            =  49,      /*!< CCU41 SR1 Interrupt                      */
CCU41_2_IRQn            =  50,      /*!< CCU41 SR2 Interrupt                      */
CCU41_3_IRQn            =  51,      /*!< CCU41 SR3 Interrupt                      */
CCU42_0_IRQn            =  52,      /*!< CCU42 SR0 Interrupt                      */
CCU42_1_IRQn            =  53,      /*!< CCU42 SR1 Interrupt                      */
CCU42_2_IRQn            =  54,      /*!< CCU42 SR2 Interrupt                      */
CCU42_3_IRQn            =  55,      /*!< CCU42 SR3 Interrupt                      */
CCU43_0_IRQn            =  56,      /*!< CCU43 SR0 Interrupt                      */
CCU43_1_IRQn            =  57,      /*!< CCU43 SR1 Interrupt                      */
CCU43_2_IRQn            =  58,      /*!< CCU43 SR2 Interrupt                      */
CCU43_3_IRQn            =  59,      /*!< CCU43 SR3 Interrupt                      */
CCU80_0_IRQn            =  60,      /*!< CCU80 SR0 Interrupt                      */
CCU80_1_IRQn            =  61,      /*!< CCU80 SR1 Interrupt                      */
CCU80_2_IRQn            =  62,      /*!< CCU80 SR2 Interrupt                      */
CCU80_3_IRQn            =  63,      /*!< CCU80 SR3 Interrupt                      */
CCU81_0_IRQn            =  64,      /*!< CCU81 SR0 Interrupt                      */
CCU81_1_IRQn            =  65,      /*!< CCU81 SR1 Interrupt                      */
CCU81_2_IRQn            =  66,      /*!< CCU81 SR2 Interrupt                      */
CCU81_3_IRQn            =  67,      /*!< CCU81 SR3 Interrupt                      */
POSIF0_0_IRQn           =  68,      /*!< POSIF0 SR0 Interrupt                     */
POSIF0_1_IRQn           =  69,      /*!< POSIF0 SR1 Interrupt                     */
POSIF1_0_IRQn           =  70,      /*!< POSIF1 SR0 Interrupt                     */
POSIF1_1_IRQn           =  71,      /*!< POSIF1 SR1 Interrupt                     */
CAN0_0_IRQn             =  76,      /*!< MCAN SR0 Interrupt                       */
CAN0_1_IRQn             =  77,      /*!< MCAN SR1 Interrupt                       */
CAN0_2_IRQn             =  78,      /*!< MCAN SR2 Interrupt                       */
CAN0_3_IRQn             =  79,      /*!< MCAN SR3 Interrupt                       */
CAN0_4_IRQn             =  80,      /*!< MCAN SR4 Interrupt                       */
CAN0_5_IRQn             =  81,      /*!< MCAN SR5 Interrupt                       */
CAN0_6_IRQn             =  82,      /*!< MCAN SR6 Interrupt                       */
CAN0_7_IRQn             =  83,      /*!< MCAN SR7 Interrupt                       */
USIC0_0_IRQn            =  84,      /*!< USIC0 SR0 Interrupt                      */
USIC0_1_IRQn            =  85,      /*!< USIC0 SR1 Interrupt                      */
USIC0_2_IRQn            =  86,      /*!< USIC0 SR2 Interrupt                      */
USIC0_3_IRQn            =  87,      /*!< USIC0 SR3 Interrupt                      */
USIC0_4_IRQn            =  88,      /*!< USIC0 SR4 Interrupt                      */
USIC0_5_IRQn            =  89,      /*!< USIC0 SR5 Interrupt                      */
USIC1_0_IRQn            =  90,      /*!< USIC1 SR0 Interrupt                      */
USIC1_1_IRQn            =  91,      /*!< USIC1 SR1 Interrupt                      */
USIC1_2_IRQn            =  92,      /*!< USIC1 SR2 Interrupt                      */
USIC1_3_IRQn            =  93,      /*!< USIC1 SR3 Interrupt                      */
USIC1_4_IRQn            =  94,      /*!< USIC1 SR4 Interrupt                      */
USIC1_5_IRQn            =  95,      /*!< USIC1 SR5 Interrupt                      */
USIC2_0_IRQn            =  96,      /*!< USIC2 SR0 Interrupt                      */   
USIC2_1_IRQn            =  97,      /*!< USIC2 SR1 Interrupt                      */
USIC2_2_IRQn            =  98,      /*!< USIC2 SR2 Interrupt                      */
USIC2_3_IRQn            =  99,      /*!< USIC2 SR3 Interrupt                      */
USIC2_4_IRQn            = 100,      /*!< USIC2 SR4 Interrupt                      */
USIC2_5_IRQn					   = 101,      /*!< USIC2 SR5 Interrupt                      */
LEDTS0_0_IRQn           = 102,      /*!< LEDTSU SR0 Interrupt                     */
FCE0_0_IRQn             = 104,      /*!< FCE SR0 Interrupt                        */
GPDMA0_0_IRQn           = 105,      /*!< GPDMA SR0 Interrupt                      */
SDMMC0_0_IRQn           = 106,      /*!< MMCI SR0 Interrupt                       */
USB0_0_IRQn             = 107,      /*!< USB SR0 Interrupt                        */
ETH0_0_IRQn             = 108,      /*!< ETH SR0 Interrupt                        */
GPDMA1_0_IRQn           = 110       /*!< GPDMA1 SR0 Interrupt                     */
}IRQn_Type;


/*
* ==========================================================================
* ----------- Processor and Core Peripheral Section ------------------------
* ==========================================================================
*/

/** Macro to write new value to the bits in register */
#define WR_REG(reg, mask, pos, val) {  \
                     reg &= ~(mask);\
                     reg |= (val << pos) & mask;\
                    }
                  
/** Macro to read the bits in register */
#define RD_REG(reg, mask, pos)            (((reg)&mask) >> pos)
/** Macro to set the particular bit in register */
#define SET_BIT(reg, pos)                 (reg) |= (1U<<pos)
/** Macro to clear the particular bit in register */
#define CLR_BIT(reg, pos)                 (reg) &= ~(1U<<pos)

#include <core_cm4.h>    /* Cortex-M4 processor and core peripherals       */
#include "system_XMC4500.h"

/******************************************************************************/
/*                Device Specific Peripheral registers structures             */
/******************************************************************************/

/***************************************************************************/
/*                     CAN                          */
/***************************************************************************/

  /* 
  *CAN GLOBAL registers
  */
typedef struct {
__IO uint32_t	CLC; /*!<CAN Clock Control Register Offset  0x0000*/
__I  uint32_t	RESERVED1;
__I uint32_t	ID; /*!<Module Identification Register Offset  0x0008*/
__IO uint32_t	FDR; /*!<CAN Fractional Divider Register Offset  0x000C*/
__I  uint32_t	RESERVED2[60];
__I uint32_t	LIST[8]; /*!<List Register 0 Offset  0x0100*/
__I  uint32_t	RESERVED3[8];
__IO uint32_t	MSPND[8]; /*!<Message Pending Register 0 Offset  0x0140*/
__I  uint32_t	RESERVED4[8];
__I uint32_t	MSID[8]; /*!<Message Index Register 0 Offset  0x0180*/
__I  uint32_t	RESERVED5[8];
__IO uint32_t	MSIMASK; /*!<Message Index Mask Register Offset  0x01C0*/
__IO uint32_t	PANCTR; /*!<Panel Control Register Offset  0x01C4*/
__IO uint32_t	MCR; /*!<Module Control Register Offset  0x01C8*/
__O uint32_t	MITR; /*!<Module Interrupt Trigger Register Offset  0x01CC*/
}CAN_GLOBAL_TypeDef;

  /* 
  *CAN NODE registers
  */
typedef struct {
__IO uint32_t	NCR; /*!<Node 0 Control Register Offset  0x0200*/
__IO uint32_t	NSR; /*!<Node 0 Status Register Offset  0x0204*/
__IO uint32_t	NIPR; /*!<Node 0 Interrupt Pointer Register Offset  0x0208*/
__IO uint32_t	NPCR; /*!<Node 0 Port Control Register Offset  0x020C*/
__IO uint32_t	NBTR; /*!<Node 0 Bit Timing Register Offset  0x0210*/
__IO uint32_t	NECNT; /*!<Node 0 Error Counter Register Offset  0x0214*/
__IO uint32_t	NFCR; /*!<Node 0 Frame Counter Register Offset  0x0218*/
}CAN_NODE_TypeDef;

  /* 
  *CAN MO registers
  */
typedef struct {
__IO uint32_t	MOFCR; /*!<Message Object 0 Function Control Register Offset  0x1000*/
__IO uint32_t	MOFGPR; /*!<Message Object 0 FIFO/Gateway Pointer Register Offset  0x1004*/
__IO uint32_t	MOIPR; /*!<Message Object 0 Interrupt Pointer Register Offset  0x1008*/
__IO uint32_t	MOAMR; /*!<Message Object 0 Acceptance Mask Register Offset  0x100C*/
__IO uint32_t	MODATAL; /*!<Message Object 0 Data Register Low Offset  0x1010*/
__IO uint32_t	MODATAH; /*!<Message Object 0 Data Register High Offset  0x1014*/
__IO uint32_t	MOAR; /*!<Message Object 0 Arbitration Register Offset  0x1018*/
union
 {
__O uint32_t	MOCTR; /*!<Message Object 0 Control Register Offset  0x101C*/
__I uint32_t	MOSTAT; /*!<Message Object 0 Status Register Offset  0x101C*/
 };
}CAN_MO_TypeDef;

#define   CAN_BASE			 ((uint32_t)0x48014000U)
#define   CAN_GLOBAL_BASE		 (0x0000U)
#define   CAN_NODE0_BASE		 (0x0200U)
#define   CAN_NODE1_BASE		 (0x0300U)
#define   CAN_NODE2_BASE		 (0x0400U)
#define   CAN_MO0_BASE		 (0x1000U)
#define   CAN_MO1_BASE		 (0x1020U)
#define   CAN_MO2_BASE		 (0x1040U)
#define   CAN_MO3_BASE		 (0x1060U)
#define   CAN_MO4_BASE		 (0x1080U)
#define   CAN_MO5_BASE		 (0x10A0U)
#define   CAN_MO6_BASE		 (0x10C0U)
#define   CAN_MO7_BASE		 (0x10E0U)
#define   CAN_MO8_BASE		 (0x1100U)
#define   CAN_MO9_BASE		 (0x1120U)
#define   CAN_MO10_BASE		 (0x1140U)
#define   CAN_MO11_BASE		 (0x1160U)
#define   CAN_MO12_BASE		 (0x1180U)
#define   CAN_MO13_BASE		 (0x11A0U)
#define   CAN_MO14_BASE		 (0x11C0U)
#define   CAN_MO15_BASE		 (0x11E0U)
#define   CAN_MO16_BASE		 (0x1200U)
#define   CAN_MO17_BASE		 (0x1220U)
#define   CAN_MO18_BASE		 (0x1240U)
#define   CAN_MO19_BASE		 (0x1260U)
#define   CAN_MO20_BASE		 (0x1280U)
#define   CAN_MO21_BASE		 (0x12A0U)
#define   CAN_MO22_BASE		 (0x12C0U)
#define   CAN_MO23_BASE		 (0x12E0U)
#define   CAN_MO24_BASE		 (0x1300U)
#define   CAN_MO25_BASE		 (0x1320U)
#define   CAN_MO26_BASE		 (0x1340U)
#define   CAN_MO27_BASE		 (0x1360U)
#define   CAN_MO28_BASE		 (0x1380U)
#define   CAN_MO29_BASE		 (0x13A0U)
#define   CAN_MO30_BASE		 (0x13C0U)
#define   CAN_MO31_BASE		 (0x13E0U)
#define   CAN_MO32_BASE		 (0x1400U)
#define   CAN_MO33_BASE		 (0x1420U)
#define   CAN_MO34_BASE		 (0x1440U)
#define   CAN_MO35_BASE		 (0x1460U)
#define   CAN_MO36_BASE		 (0x1480U)
#define   CAN_MO37_BASE		 (0x14A0U)
#define   CAN_MO38_BASE		 (0x14C0U)
#define   CAN_MO39_BASE		 (0x14E0U)
#define   CAN_MO40_BASE		 (0x1500U)
#define   CAN_MO41_BASE		 (0x1520U)
#define   CAN_MO42_BASE		 (0x1540U)
#define   CAN_MO43_BASE		 (0x1560U)
#define   CAN_MO44_BASE		 (0x1580U)
#define   CAN_MO45_BASE		 (0x15A0U)
#define   CAN_MO46_BASE		 (0x15C0U)
#define   CAN_MO47_BASE		 (0x15E0U)
#define   CAN_MO48_BASE		 (0x1600U)
#define   CAN_MO49_BASE		 (0x1620U)
#define   CAN_MO50_BASE		 (0x1640U)
#define   CAN_MO51_BASE		 (0x1660U)
#define   CAN_MO52_BASE		 (0x1680U)
#define   CAN_MO53_BASE		 (0x16A0U)
#define   CAN_MO54_BASE		 (0x16C0U)
#define   CAN_MO55_BASE		 (0x16E0U)
#define   CAN_MO56_BASE		 (0x1700U)
#define   CAN_MO57_BASE		 (0x1720U)
#define   CAN_MO58_BASE		 (0x1740U)
#define   CAN_MO59_BASE		 (0x1760U)
#define   CAN_MO60_BASE		 (0x1780U)
#define   CAN_MO61_BASE		 (0x17A0U)
#define   CAN_MO62_BASE		 (0x17C0U)
#define   CAN_MO63_BASE		 (0x17E0U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   CAN		 ((CAN_GLOBAL_TypeDef*)(CAN_BASE + CAN_GLOBAL_BASE))
#define   CAN_NODE0		 ((CAN_NODE_TypeDef*)(CAN_BASE + CAN_NODE0_BASE))
#define   CAN_NODE1		 ((CAN_NODE_TypeDef*)(CAN_BASE + CAN_NODE1_BASE))
#define   CAN_NODE2		 ((CAN_NODE_TypeDef*)(CAN_BASE + CAN_NODE2_BASE))
#define   CAN_MO0		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO0_BASE))
#define   CAN_MO1		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO1_BASE))
#define   CAN_MO2		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO2_BASE))
#define   CAN_MO3		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO3_BASE))
#define   CAN_MO4		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO4_BASE))
#define   CAN_MO5		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO5_BASE))
#define   CAN_MO6		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO6_BASE))
#define   CAN_MO7		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO7_BASE))
#define   CAN_MO8		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO8_BASE))
#define   CAN_MO9		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO9_BASE))
#define   CAN_MO10		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO10_BASE))
#define   CAN_MO11		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO11_BASE))
#define   CAN_MO12		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO12_BASE))
#define   CAN_MO13		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO13_BASE))
#define   CAN_MO14		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO14_BASE))
#define   CAN_MO15		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO15_BASE))
#define   CAN_MO16		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO16_BASE))
#define   CAN_MO17		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO17_BASE))
#define   CAN_MO18		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO18_BASE))
#define   CAN_MO19		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO19_BASE))
#define   CAN_MO20		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO20_BASE))
#define   CAN_MO21		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO21_BASE))
#define   CAN_MO22		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO22_BASE))
#define   CAN_MO23		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO23_BASE))
#define   CAN_MO24		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO24_BASE))
#define   CAN_MO25		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO25_BASE))
#define   CAN_MO26		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO26_BASE))
#define   CAN_MO27		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO27_BASE))
#define   CAN_MO28		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO28_BASE))
#define   CAN_MO29		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO29_BASE))
#define   CAN_MO30		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO30_BASE))
#define   CAN_MO31		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO31_BASE))
#define   CAN_MO32		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO32_BASE))
#define   CAN_MO33		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO33_BASE))
#define   CAN_MO34		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO34_BASE))
#define   CAN_MO35		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO35_BASE))
#define   CAN_MO36		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO36_BASE))
#define   CAN_MO37		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO37_BASE))
#define   CAN_MO38		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO38_BASE))
#define   CAN_MO39		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO39_BASE))
#define   CAN_MO40		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO40_BASE))
#define   CAN_MO41		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO41_BASE))
#define   CAN_MO42		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO42_BASE))
#define   CAN_MO43		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO43_BASE))
#define   CAN_MO44		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO44_BASE))
#define   CAN_MO45		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO45_BASE))
#define   CAN_MO46		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO46_BASE))
#define   CAN_MO47		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO47_BASE))
#define   CAN_MO48		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO48_BASE))
#define   CAN_MO49		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO49_BASE))
#define   CAN_MO50		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO50_BASE))
#define   CAN_MO51		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO51_BASE))
#define   CAN_MO52		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO52_BASE))
#define   CAN_MO53		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO53_BASE))
#define   CAN_MO54		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO54_BASE))
#define   CAN_MO55		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO55_BASE))
#define   CAN_MO56		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO56_BASE))
#define   CAN_MO57		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO57_BASE))
#define   CAN_MO58		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO58_BASE))
#define   CAN_MO59		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO59_BASE))
#define   CAN_MO60		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO60_BASE))
#define   CAN_MO61		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO61_BASE))
#define   CAN_MO62		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO62_BASE))
#define   CAN_MO63		 ((CAN_MO_TypeDef*)(CAN_BASE + CAN_MO63_BASE))

/***************************************************************************/
/*                     CCU4x                          */
/***************************************************************************/

  /* 
  *CAPCOM4 Global Registers
  */
typedef struct {
__IO uint32_t	GCTRL; /*!<Global Control Register Offset  0x0000*/
__I uint32_t	GSTAT; /*!<Global Status Register Offset  0x0004*/
__O uint32_t	GIDLS; /*!<Global Idle Set Offset  0x0008*/
__O uint32_t	GIDLC; /*!<Global Idle Clear Offset  0x000C*/
__O uint32_t	GCSS; /*!<Global Channel Set Offset  0x0010*/
__O uint32_t	GCSC; /*!<Global Channel Clear Offset  0x0014*/
__I uint32_t	GCST; /*!<Global Channel Status Offset  0x0018*/
__I  uint32_t	RESERVED1[13];
__I uint32_t	ECRD; /*!<Extended Capture Mode Read Offset  0x0050*/
__I  uint32_t	RESERVED2[11];
__I uint32_t	MIDR; /*!<Module Identification Offset  0x0080*/
}CCU4_GLOBAL_TypeDef;

  /* 
  *CC40 Registers
  */
typedef struct {
__IO uint32_t	INS; /*!<Input Selector Configuration Offset  0x0100*/
__IO uint32_t	CMC; /*!<Connection Matrix Control Offset  0x0104*/
__I uint32_t	TCST; /*!<Slice Timer Status Offset  0x0108*/
__O uint32_t	TCSET; /*!<Slice Timer Run Set Offset  0x010C*/
__O uint32_t	TCCLR; /*!<Slice Timer Clear Offset  0x0110*/
__IO uint32_t	TC; /*!<Slice Timer Control Offset  0x0114*/
__IO uint32_t	PSL; /*!<Passive Level Config Offset  0x0118*/
__I uint32_t	DIT; /*!<Dither Config Offset  0x011C*/
__IO uint32_t	DITS; /*!<Dither Shadow Register Offset  0x0120*/
__IO uint32_t	PSC; /*!<Prescaler Control Offset  0x0124*/
__IO uint32_t	FPC; /*!<Floating Prescaler Control Offset  0x0128*/
__IO uint32_t	FPCS; /*!<Floating Prescaler Shadow Offset  0x012C*/
__I uint32_t	PR; /*!<Timer Period Value Offset  0x0130*/
__IO uint32_t	PRS; /*!<Timer Shadow Period Value Offset  0x0134*/
__I uint32_t	CR; /*!<Timer Compare Value Offset  0x0138*/
__IO uint32_t	CRS; /*!<Timer Shadow Compare Value Offset  0x013C*/
__I  uint32_t	RESERVED1[12];
__IO uint32_t	TIMER; /*!<Timer Value Offset  0x0170*/
__I uint32_t	CV[4]; /*!<Capture Register 0 Offset  0x0174*/
__I  uint32_t	RESERVED2[7];
__I uint32_t	INTS; /*!<Interrupt Status Offset  0x01A0*/
__IO uint32_t	INTE; /*!<Interrupt Enable Control Offset  0x01A4*/
__IO uint32_t	SRS; /*!<Service Request Selector Offset  0x01A8*/
__O uint32_t	SWS; /*!<Interrupt Status Set Offset  0x01AC*/
__O uint32_t	SWR; /*!<Interrupt Status Clear Offset  0x01B0*/
}CCU4_CC4_TypeDef;

#define   CCU40_BASE			 ((uint32_t)0x4000C000U)
#define   CCU41_BASE			 ((uint32_t)0x40010000U)
#define   CCU42_BASE			 ((uint32_t)0x40014000U)
#define   CCU43_BASE			 ((uint32_t)0x48004000U)
#define   CCU4_GLOBAL_BASE		 (0x0000U)
#define   CCU40_CC40_BASE		 (0x0100U)
#define   CCU40_CC41_BASE		 (0x0200U)
#define   CCU40_CC42_BASE		 (0x0300U)
#define   CCU40_CC43_BASE		 (0x0400U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   CCU40		 ((CCU4_GLOBAL_TypeDef*)(CCU40_BASE + CCU4_GLOBAL_BASE))
#define   CCU40_CC40		 ((CCU4_CC4_TypeDef*)(CCU40_BASE + CCU40_CC40_BASE))
#define   CCU40_CC41		 ((CCU4_CC4_TypeDef*)(CCU40_BASE + CCU40_CC41_BASE))
#define   CCU40_CC42		 ((CCU4_CC4_TypeDef*)(CCU40_BASE + CCU40_CC42_BASE))
#define   CCU40_CC43		 ((CCU4_CC4_TypeDef*)(CCU40_BASE + CCU40_CC43_BASE))
#define   CCU41		 ((CCU4_GLOBAL_TypeDef*)(CCU41_BASE + CCU4_GLOBAL_BASE))
#define   CCU41_CC40		 ((CCU4_CC4_TypeDef*)(CCU41_BASE + CCU40_CC40_BASE))
#define   CCU41_CC41		 ((CCU4_CC4_TypeDef*)(CCU41_BASE + CCU40_CC41_BASE))
#define   CCU41_CC42		 ((CCU4_CC4_TypeDef*)(CCU41_BASE + CCU40_CC42_BASE))
#define   CCU41_CC43		 ((CCU4_CC4_TypeDef*)(CCU41_BASE + CCU40_CC43_BASE))
#define   CCU42		 ((CCU4_GLOBAL_TypeDef*)(CCU42_BASE + CCU4_GLOBAL_BASE))
#define   CCU42_CC40		 ((CCU4_CC4_TypeDef*)(CCU42_BASE + CCU40_CC40_BASE))
#define   CCU42_CC41		 ((CCU4_CC4_TypeDef*)(CCU42_BASE + CCU40_CC41_BASE))
#define   CCU42_CC42		 ((CCU4_CC4_TypeDef*)(CCU42_BASE + CCU40_CC42_BASE))
#define   CCU42_CC43		 ((CCU4_CC4_TypeDef*)(CCU42_BASE + CCU40_CC43_BASE))
#define   CCU43		 ((CCU4_GLOBAL_TypeDef*)(CCU43_BASE + CCU4_GLOBAL_BASE))
#define   CCU43_CC40		 ((CCU4_CC4_TypeDef*)(CCU43_BASE + CCU40_CC40_BASE))
#define   CCU43_CC41		 ((CCU4_CC4_TypeDef*)(CCU43_BASE + CCU40_CC41_BASE))
#define   CCU43_CC42		 ((CCU4_CC4_TypeDef*)(CCU43_BASE + CCU40_CC42_BASE))
#define   CCU43_CC43		 ((CCU4_CC4_TypeDef*)(CCU43_BASE + CCU40_CC43_BASE))

/***************************************************************************/
/*                     CCU8x                          */
/***************************************************************************/

  /* 
  *CAPCOM8 Global Registers
  */
typedef struct {
__IO uint32_t	GCTRL; /*!<Global Control Register Offset  0x0000*/
__I uint32_t	GSTAT; /*!<Global Status Register Offset  0x0004*/
__O uint32_t	GIDLS; /*!<Global Idle Set Offset  0x0008*/
__O uint32_t	GIDLC; /*!<Global Idle Clear Offset  0x000C*/
__O uint32_t	GCSS; /*!<Global Channel Set Offset  0x0010*/
__O uint32_t	GCSC; /*!<Global Channel Clear Offset  0x0014*/
__I uint32_t	GCST; /*!<Global Channel status Offset  0x0018*/
__IO uint32_t	GPCHK; /*!<Parity Checker Configuration Offset  0x001C*/
__I  uint32_t	RESERVED1[12];
__I uint32_t	ECRD; /*!<Extended Capture Mode Read Offset  0x0050*/
__I  uint32_t	RESERVED2[11];
__I uint32_t	MIDR; /*!<Module Identification Offset  0x0080*/
}CCU8_GLOBAL_TypeDef;

  /* 
  *CC8x Registers
  */
typedef struct {
__IO uint32_t	INS; /*!<Input Selector Configuration Offset  0x0100*/
__IO uint32_t	CMC; /*!<Connection Matrix Control Offset  0x0104*/
__I uint32_t	TCST; /*!<Slice Timer Status Offset  0x0108*/
__O uint32_t	TCSET; /*!<Slice Timer Run Set Offset  0x010C*/
__O uint32_t	TCCLR; /*!<Slice Timer Clear Offset  0x0110*/
__IO uint32_t	TC; /*!<Slice Timer Control Offset  0x0114*/
__IO uint32_t	PSL; /*!<Passive Level Config Offset  0x0118*/
__I uint32_t	DIT; /*!<Dither Config Offset  0x011C*/
__IO uint32_t	DITS; /*!<Dither Shadow Register Offset  0x0120*/
__IO uint32_t	PSC; /*!<Prescaler Control Offset  0x0124*/
__IO uint32_t	FPC; /*!<Floating Prescaler Control Offset  0x0128*/
__IO uint32_t	FPCS; /*!<Floating Prescaler Shadow Offset  0x012C*/
__I uint32_t	PR; /*!<Timer Period Value Offset  0x0130*/
__IO uint32_t	PRS; /*!<Timer Shadow Period Value Offset  0x0134*/
__I uint32_t	CR1; /*!<Channel 1 Compare Value Offset  0x0138*/
__IO uint32_t	CR1S; /*!<Channel 1 Compare Shadow Value Offset  0x013C*/
__I uint32_t	CR2; /*!<Channel 2 Compare Value Offset  0x0140*/
__IO uint32_t	CR2S; /*!<Channel 2 Compare Shadow Value Offset  0x0144*/
__IO uint32_t	CHC; /*!<Channel Control Offset  0x0148*/
__IO uint32_t	DTC; /*!<Dead Time Control Offset  0x014C*/
__IO uint32_t	DC1R; /*!<Channel 1 Dead Time Values Offset  0x0150*/
__IO uint32_t	DC2R; /*!<Channel 2 Dead Time Values Offset  0x0154*/
__I  uint32_t	RESERVED1[6];
__IO uint32_t	TIMER; /*!<Timer Value Offset  0x0170*/
__I uint32_t	CV[4]; /*!<Capture Register 0 Offset  0x0174*/
__I  uint32_t	RESERVED2[7];
__I uint32_t	INTS; /*!<Interrupt Status Offset  0x01A0*/
__IO uint32_t	INTE; /*!<Interrupt Enable Control Offset  0x01A4*/
__IO uint32_t	SRS; /*!<Service Request Selector Offset  0x01A8*/
__O uint32_t	SWS; /*!<Interrupt Status Set Offset  0x01AC*/
__O uint32_t	SWR; /*!<Interrupt Status Clear Offset  0x01B0*/
}CCU8_CC8_TypeDef;

#define   CCU80_BASE			 ((uint32_t)0x40020000U)
#define   CCU81_BASE			 ((uint32_t)0x40024000U)
#define   CCU8_GLOBAL_BASE		 (0x0000U)
#define   CCU80_CC80_BASE		 (0x0100U)
#define   CCU80_CC81_BASE		 (0x0200U)
#define   CCU80_CC82_BASE		 (0x0300U)
#define   CCU80_CC83_BASE		 (0x0400U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   CCU80		 ((CCU8_GLOBAL_TypeDef*)(CCU80_BASE + CCU8_GLOBAL_BASE))
#define   CCU80_CC80		 ((CCU8_CC8_TypeDef*)(CCU80_BASE + CCU80_CC80_BASE))
#define   CCU80_CC81		 ((CCU8_CC8_TypeDef*)(CCU80_BASE + CCU80_CC81_BASE))
#define   CCU80_CC82		 ((CCU8_CC8_TypeDef*)(CCU80_BASE + CCU80_CC82_BASE))
#define   CCU80_CC83		 ((CCU8_CC8_TypeDef*)(CCU80_BASE + CCU80_CC83_BASE))
#define   CCU81		 ((CCU8_GLOBAL_TypeDef*)(CCU81_BASE + CCU8_GLOBAL_BASE))
#define   CCU81_CC80		 ((CCU8_CC8_TypeDef*)(CCU81_BASE + CCU80_CC80_BASE))
#define   CCU81_CC81		 ((CCU8_CC8_TypeDef*)(CCU81_BASE + CCU80_CC81_BASE))
#define   CCU81_CC82		 ((CCU8_CC8_TypeDef*)(CCU81_BASE + CCU80_CC82_BASE))
#define   CCU81_CC83		 ((CCU8_CC8_TypeDef*)(CCU81_BASE + CCU80_CC83_BASE))

/***************************************************************************/
/*                     DAC                          */
/***************************************************************************/

  /* 
  *DAC Kernel Registers
  */
typedef struct {
__I uint32_t	ID; /*!<Module Identification Register Offset  0x0000*/
__IO uint32_t	DAC0CFG0; /*!<DAC0 Configuration Register 0 Offset  0x0004*/
__IO uint32_t	DAC0CFG1; /*!<DAC0 Configuration Register 1 Offset  0x0008*/
__IO uint32_t	DAC1CFG0; /*!<DAC1 Configuration Register 0 Offset  0x000C*/
__IO uint32_t	DAC1CFG1; /*!<DAC1 Configuration Register 1 Offset  0x0010*/
__IO uint32_t	DAC0DATA; /*!<DAC0 Data Register Offset  0x0014*/
__IO uint32_t	DAC1DATA; /*!<DAC1 Data Register Offset  0x0018*/
__IO uint32_t	DAC01DATA; /*!<DAC01 Data Register Offset  0x001C*/
__IO uint32_t	DAC0PATL; /*!<DAC0 Lower Pattern Register Offset  0x0020*/
__IO uint32_t	DAC0PATH; /*!<DAC0 Higher Pattern Register Offset  0x0024*/
__IO uint32_t	DAC1PATL; /*!<DAC1 Lower Pattern Register Offset  0x0028*/
__IO uint32_t	DAC1PATH; /*!<DAC1 Higher Pattern Register Offset  0x002C*/
}DAC_GLOBAL_TypeDef;

#define   DAC_BASE			 ((uint32_t)0x48018000U)
#define   DAC_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   DAC		 ((DAC_GLOBAL_TypeDef*)(DAC_BASE + DAC_GLOBAL_BASE))

/***************************************************************************/
/*                     DLR                          */
/***************************************************************************/

  /* 
  *DLR registers
  */
typedef struct {
__I uint32_t	OVRSTAT; /*!<GPDMA Overrun Status Offset  0x0000*/
__O uint32_t	OVRCLR; /*!<GPDMA Overrun Clear Offset  0x0004*/
__IO uint32_t	SRSEL0; /*!<GPDMA Service Request Selection 0 Offset  0x0008*/
__IO uint32_t	SRSEL1; /*!<GPDMA Service Request Selection 1 Offset  0x000C*/
__IO uint32_t	LNEN; /*!<GPDMA Line Enable Offset  0x0010*/
}DLR_GLOBAL_TypeDef;

#define   DLR_BASE			 ((uint32_t)0x50004900U)
#define   DLR_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   DLR		 ((DLR_GLOBAL_TypeDef*)(DLR_BASE + DLR_GLOBAL_BASE))

/***************************************************************************/
/*                     GPDMA1                          */
/***************************************************************************/

  /* 
  *Channel Related registers
  */
typedef struct {
__IO uint32_t	SAR; /*!<Source Address Register for Channel 0 Offset  0x0000*/
__I  uint32_t	RESERVED1;
__IO uint32_t	DAR; /*!<Destination Address Register for Channel 0 Offset  0x0008*/
__I  uint32_t	RESERVED2[3];
__IO uint32_t	CTLL; /*!<Control Register for Channel 0 Low Word Offset  0x0018*/
__IO uint32_t	CTLH; /*!<Control Register for Channel 0 High Word Offset  0x001C*/
__I  uint32_t	RESERVED3[8];
__IO uint32_t	CFGL; /*!<Configuration Register for Channel 0 Low Word Offset  0x0040*/
__IO uint32_t	CFGH; /*!<Configuration Register for Channel 0 High Word Offset  0x0044*/
}GPDMA1_CH_TypeDef;

  /* 
  *GPDMA1 Global registers
  */
typedef struct {
__IO uint32_t	RAWTFR; /*!<Raw IntTfr Status Offset  0x02C0*/
__I  uint32_t	RESERVED1;
__IO uint32_t	RAWBLOCK; /*!<Raw IntBlock Status Offset  0x02C8*/
__I  uint32_t	RESERVED2;
__IO uint32_t	RAWSRCTRAN; /*!<Raw IntSrcTran Status Offset  0x02D0*/
__I  uint32_t	RESERVED3;
__IO uint32_t	RAWDSTTRAN; /*!<Raw IntBlock Status Offset  0x02D8*/
__I  uint32_t	RESERVED4;
__IO uint32_t	RAWERR; /*!<Raw IntErr Status Offset  0x02E0*/
__I  uint32_t	RESERVED5;
__I uint32_t	STATUSTFR; /*!<IntTfr Status Offset  0x02E8*/
__I  uint32_t	RESERVED6;
__I uint32_t	STATUSBLOCK; /*!<IntBlock Status Offset  0x02F0*/
__I  uint32_t	RESERVED7;
__I uint32_t	STATUSSRCTRAN; /*!<IntSrcTran Status Offset  0x02F8*/
__I  uint32_t	RESERVED8;
__I uint32_t	STATUSDSTTRAN; /*!<IntBlock Status Offset  0x0300*/
__I  uint32_t	RESERVED9;
__I uint32_t	STATUSERR; /*!<IntErr Status Offset  0x0308*/
__I  uint32_t	RESERVED10;
__IO uint32_t	MASKTFR; /*!<Mask for Raw IntTfr Status Offset  0x0310*/
__I  uint32_t	RESERVED11;
__IO uint32_t	MASKBLOCK; /*!<Mask for Raw IntBlock Status Offset  0x0318*/
__I  uint32_t	RESERVED12;
__IO uint32_t	MASKSRCTRAN; /*!<Mask for Raw IntSrcTran Status Offset  0x0320*/
__I  uint32_t	RESERVED13;
__IO uint32_t	MASKDSTTRAN; /*!<Mask for Raw IntBlock Status Offset  0x0328*/
__I  uint32_t	RESERVED14;
__IO uint32_t	MASKERR; /*!<Mask for Raw IntErr Status Offset  0x0330*/
__I  uint32_t	RESERVED15;
__O uint32_t	CLEARTFR; /*!<IntTfr Status Offset  0x0338*/
__I  uint32_t	RESERVED16;
__O uint32_t	CLEARBLOCK; /*!<IntBlock Status Offset  0x0340*/
__I  uint32_t	RESERVED17;
__O uint32_t	CLEARSRCTRAN; /*!<IntSrcTran Status Offset  0x0348*/
__I  uint32_t	RESERVED18;
__O uint32_t	CLEARDSTTRAN; /*!<IntBlock Status Offset  0x0350*/
__I  uint32_t	RESERVED19;
__O uint32_t	CLEARERR; /*!<IntErr Status Offset  0x0358*/
__I  uint32_t	RESERVED20;
__I uint32_t	STATUSINT; /*!<Combined Interrupt Status Register Offset  0x0360*/
__I  uint32_t	RESERVED21;
__IO uint32_t	REQSRCREG; /*!<Source Software Transaction Request Register Offset  0x0368*/
__I  uint32_t	RESERVED22;
__IO uint32_t	REQDSTREG; /*!<Destination Software Transaction Request Register Offset  0x0370*/
__I  uint32_t	RESERVED23;
__IO uint32_t	SGLREQSRCREG; /*!<Single Source Transaction Request Register Offset  0x0378*/
__I  uint32_t	RESERVED24;
__IO uint32_t	SGLREQDSTREG; /*!<Single Destination Transaction Request Register Offset  0x0380*/
__I  uint32_t	RESERVED25;
__IO uint32_t	LSTSRCREG; /*!<Last Source Transaction Request Register Offset  0x0388*/
__I  uint32_t	RESERVED26;
__IO uint32_t	LSTDSTREG; /*!<Last Destination Transaction Request Register Offset  0x0390*/
__I  uint32_t	RESERVED27;
__IO uint32_t	DMACFGREG; /*!<GPDMA Configuration Register Offset  0x0398*/
__I  uint32_t	RESERVED28;
__IO uint32_t	CHENREG; /*!<GPDMA Channel Enable Register Offset  0x03A0*/
__I  uint32_t	RESERVED29;
__I uint32_t	ID; /*!<GPDMA1 ID Register Offset  0x03A8*/
__I  uint32_t	RESERVED30[19];
__I uint32_t	TYPE; /*!<GPDMA Component Type Offset  0x03F8*/
__I uint32_t	VERSION; /*!<DMA Component Version Offset  0x03FC*/
}GPDMA1_GLOBAL_TypeDef;

#define   GPDMA1_BASE			 ((uint32_t)0x50018000U)
#define   GPDMA1_CH0_BASE		 (0x0000U)
#define   GPDMA1_CH1_BASE		 (0x0058U)
#define   GPDMA1_CH2_BASE		 (0x00B0U)
#define   GPDMA1_CH3_BASE		 (0x0108U)
#define   GPDMA1_GLOBAL_BASE		 (0x02C0U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   GPDMA1_CH0		 ((GPDMA1_CH_TypeDef*)(GPDMA1_BASE + GPDMA1_CH0_BASE))
#define   GPDMA1_CH1		 ((GPDMA1_CH_TypeDef*)(GPDMA1_BASE + GPDMA1_CH1_BASE))
#define   GPDMA1_CH2		 ((GPDMA1_CH_TypeDef*)(GPDMA1_BASE + GPDMA1_CH2_BASE))
#define   GPDMA1_CH3		 ((GPDMA1_CH_TypeDef*)(GPDMA1_BASE + GPDMA1_CH3_BASE))
#define   GPDMA1		 ((GPDMA1_GLOBAL_TypeDef*)(GPDMA1_BASE + GPDMA1_GLOBAL_BASE))

/***************************************************************************/
/*                     GPDMA0                          */
/***************************************************************************/

  /* 
  *Channel Related registers
  */
typedef struct {
__IO uint32_t	SAR; /*!<Source Address Register for Channel 0 Offset  0x0000*/
__I  uint32_t	RESERVED1;
__IO uint32_t	DAR; /*!<Destination Address Register for Channel 0 Offset  0x0008*/
__I  uint32_t	RESERVED2;
__IO uint32_t	LLP; /*!<Linked List Pointer Register for Channel 0 Offset  0x0010*/
__I  uint32_t	RESERVED3;
__IO uint32_t	CTLL; /*!<Control Register for Channel 0 Low Word Offset  0x0018*/
__IO uint32_t	CTLH; /*!<Control Register for Channel 0 High Word Offset  0x001C*/
__IO uint32_t	SSTAT; /*!<Source Status Register for Channel 0 Offset  0x0020*/
__I  uint32_t	RESERVED4;
__IO uint32_t	DSTAT; /*!<Destination Status Register for Channel 0 Offset  0x0028*/
__I  uint32_t	RESERVED5;
__IO uint32_t	SSTATAR; /*!<Source Status Address Register for Channel 0 Offset  0x0030*/
__I  uint32_t	RESERVED6;
__IO uint32_t	DSTATAR; /*!<Destination Status Address Register for Channel 0 Offset  0x0038*/
__I  uint32_t	RESERVED7;
__IO uint32_t	CFGL; /*!<Configuration Register for Channel 0 Low Word Offset  0x0040*/
__IO uint32_t	CFGH; /*!<Configuration Register for Channel 0 High Word Offset  0x0044*/
__IO uint32_t	SGR; /*!<Source Gather Register for Channel 0 Offset  0x0048*/
__I  uint32_t	RESERVED8;
__IO uint32_t	DSR; /*!<Destination Scatter Register for Channel 0 Offset  0x0050*/
}GPDMA0_CH_TypeDef;

  /* 
  *GPDMA0 Global registers
  */
typedef struct {
__IO uint32_t	RAWTFR; /*!<Raw IntTfr Status Offset  0x02C0*/
__I  uint32_t	RESERVED1;
__IO uint32_t	RAWBLOCK; /*!<Raw IntBlock Status Offset  0x02C8*/
__I  uint32_t	RESERVED2;
__IO uint32_t	RAWSRCTRAN; /*!<Raw IntSrcTran Status Offset  0x02D0*/
__I  uint32_t	RESERVED3;
__IO uint32_t	RAWDSTTRAN; /*!<Raw IntBlock Status Offset  0x02D8*/
__I  uint32_t	RESERVED4;
__IO uint32_t	RAWERR; /*!<Raw IntErr Status Offset  0x02E0*/
__I  uint32_t	RESERVED5;
__I uint32_t	STATUSTFR; /*!<IntTfr Status Offset  0x02E8*/
__I  uint32_t	RESERVED6;
__I uint32_t	STATUSBLOCK; /*!<IntBlock Status Offset  0x02F0*/
__I  uint32_t	RESERVED7;
__I uint32_t	STATUSSRCTRAN; /*!<IntSrcTran Status Offset  0x02F8*/
__I  uint32_t	RESERVED8;
__I uint32_t	STATUSDSTTRAN; /*!<IntBlock Status Offset  0x0300*/
__I  uint32_t	RESERVED9;
__I uint32_t	STATUSERR; /*!<IntErr Status Offset  0x0308*/
__I  uint32_t	RESERVED10;
__IO uint32_t	MASKTFR; /*!<Mask for Raw IntTfr Status Offset  0x0310*/
__I  uint32_t	RESERVED11;
__IO uint32_t	MASKBLOCK; /*!<Mask for Raw IntBlock Status Offset  0x0318*/
__I  uint32_t	RESERVED12;
__IO uint32_t	MASKSRCTRAN; /*!<Mask for Raw IntSrcTran Status Offset  0x0320*/
__I  uint32_t	RESERVED13;
__IO uint32_t	MASKDSTTRAN; /*!<Mask for Raw IntBlock Status Offset  0x0328*/
__I  uint32_t	RESERVED14;
__IO uint32_t	MASKERR; /*!<Mask for Raw IntErr Status Offset  0x0330*/
__I  uint32_t	RESERVED15;
__O uint32_t	CLEARTFR; /*!<IntTfr Status Offset  0x0338*/
__I  uint32_t	RESERVED16;
__O uint32_t	CLEARBLOCK; /*!<IntBlock Status Offset  0x0340*/
__I  uint32_t	RESERVED17;
__O uint32_t	CLEARSRCTRAN; /*!<IntSrcTran Status Offset  0x0348*/
__I  uint32_t	RESERVED18;
__O uint32_t	CLEARDSTTRAN; /*!<IntBlock Status Offset  0x0350*/
__I  uint32_t	RESERVED19;
__O uint32_t	CLEARERR; /*!<IntErr Status Offset  0x0358*/
__I  uint32_t	RESERVED20;
__I uint32_t	STATUSINT; /*!<Combined Interrupt Status Register Offset  0x0360*/
__I  uint32_t	RESERVED21;
__IO uint32_t	REQSRCREG; /*!<Source Software Transaction Request Register Offset  0x0368*/
__I  uint32_t	RESERVED22;
__IO uint32_t	REQDSTREG; /*!<Destination Software Transaction Request Register Offset  0x0370*/
__I  uint32_t	RESERVED23;
__IO uint32_t	SGLREQSRCREG; /*!<Single Source Transaction Request Register Offset  0x0378*/
__I  uint32_t	RESERVED24;
__IO uint32_t	SGLREQDSTREG; /*!<Single Destination Transaction Request Register Offset  0x0380*/
__I  uint32_t	RESERVED25;
__IO uint32_t	LSTSRCREG; /*!<Last Source Transaction Request Register Offset  0x0388*/
__I  uint32_t	RESERVED26;
__IO uint32_t	LSTDSTREG; /*!<Last Destination Transaction Request Register Offset  0x0390*/
__I  uint32_t	RESERVED27;
__IO uint32_t	DMACFGREG; /*!<GPDMA Configuration Register Offset  0x0398*/
__I  uint32_t	RESERVED28;
__IO uint32_t	CHENREG; /*!<GPDMA Channel Enable Register Offset  0x03A0*/
__I  uint32_t	RESERVED29;
__I uint32_t	ID; /*!<GPDMA0 ID Register Offset  0x03A8*/
__I  uint32_t	RESERVED30[19];
__I uint32_t	TYPE; /*!<GPDMA Component Type Offset  0x03F8*/
__I uint32_t	VERSION; /*!<DMA Component Version Offset  0x03FC*/
}GPDMA0_GLOBAL_TypeDef;

#define   GPDMA0_BASE			 ((uint32_t)0x50014000U)
#define   GPDMA0_CH0_BASE		 (0x0000U)
#define   GPDMA0_CH1_BASE		 (0x0058U)
#define   GPDMA0_CH2_BASE		 (0x00B0U)
#define   GPDMA0_CH3_BASE		 (0x0108U)
#define   GPDMA0_CH4_BASE		 (0x0160U)
#define   GPDMA0_CH5_BASE		 (0x01B8U)
#define   GPDMA0_CH6_BASE		 (0x0210U)
#define   GPDMA0_CH7_BASE		 (0x0268U)
#define   GPDMA0_GLOBAL_BASE		 (0x02C0U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   GPDMA0_CH0		 ((GPDMA0_CH_TypeDef*)(GPDMA0_BASE + GPDMA0_CH0_BASE))
#define   GPDMA0_CH1		 ((GPDMA0_CH_TypeDef*)(GPDMA0_BASE + GPDMA0_CH1_BASE))
#define   GPDMA0_CH2		 ((GPDMA0_CH_TypeDef*)(GPDMA0_BASE + GPDMA0_CH2_BASE))
#define   GPDMA0_CH3		 ((GPDMA0_CH_TypeDef*)(GPDMA0_BASE + GPDMA0_CH3_BASE))
#define   GPDMA0_CH4		 ((GPDMA0_CH_TypeDef*)(GPDMA0_BASE + GPDMA0_CH4_BASE))
#define   GPDMA0_CH5		 ((GPDMA0_CH_TypeDef*)(GPDMA0_BASE + GPDMA0_CH5_BASE))
#define   GPDMA0_CH6		 ((GPDMA0_CH_TypeDef*)(GPDMA0_BASE + GPDMA0_CH6_BASE))
#define   GPDMA0_CH7		 ((GPDMA0_CH_TypeDef*)(GPDMA0_BASE + GPDMA0_CH7_BASE))
#define   GPDMA0		 ((GPDMA0_GLOBAL_TypeDef*)(GPDMA0_BASE + GPDMA0_GLOBAL_BASE))

/***************************************************************************/
/*                     DSD                          */
/***************************************************************************/

  /* 
  *DSD Global Registers
  */
typedef struct {
__IO uint32_t	CLC; /*!<Clock Control Register Offset  0x0000*/
__I  uint32_t	RESERVED1;
__I uint32_t	ID; /*!<Module Identification Register Offset  0x0008*/
__I  uint32_t	RESERVED2[7];
__IO uint32_t	OCS; /*!<OCDS Control and Status Register Offset  0x0028*/
__I  uint32_t	RESERVED3[21];
__IO uint32_t	GLOBCFG; /*!<Global Configuration Register Offset  0x0080*/
__I  uint32_t	RESERVED4;
__IO uint32_t	GLOBRC; /*!<Global Run Control Register Offset  0x0088*/
__I  uint32_t	RESERVED5[5];
__IO uint32_t	CGCFG; /*!<Carrier Generator Configuration Register Offset  0x00A0*/
__I  uint32_t	RESERVED6[15];
__IO uint32_t	EVFLAG; /*!<Event Flag Register Offset  0x00E0*/
__O uint32_t	EVFLAGCLR; /*!<Event Flag Clear Register Offset  0x00E4*/
}DSD_GLOBAL_TypeDef;

  /* 
  *DSD CHANNEL Registers
  */
typedef struct {
__IO uint32_t	MODCFG; /*!<Modulator Configuration Register 0 Offset  0x0100*/
__I  uint32_t	RESERVED1;
__IO uint32_t	DICFG; /*!<Demodulator Input Configuration Register 0 Offset  0x0108*/
__I  uint32_t	RESERVED2[2];
__IO uint32_t	FCFGC; /*!<Filter Configuration Register 0, Main Comb Filter Offset  0x0114*/
__IO uint32_t	FCFGA; /*!<Filter Configuration Register 0, Auxiliary Filter Offset  0x0118*/
__I  uint32_t	RESERVED3;
__IO uint32_t	IWCTR; /*!<Integration Window Control Register 0 Offset  0x0120*/
__I  uint32_t	RESERVED4;
__IO uint32_t	BOUNDSEL; /*!<Boundary Select Register 0 Offset  0x0128*/
__I  uint32_t	RESERVED5;
__I uint32_t	RESM; /*!<Result Register 0 Main Filter Offset  0x0130*/
__I  uint32_t	RESERVED6;
__IO uint32_t	OFFM; /*!<Offset Register 0 Main Filter Offset  0x0138*/
__I  uint32_t	RESERVED7;
__I uint32_t	RESA; /*!<Result Register 0 Auxiliary Filter Offset  0x0140*/
__I  uint32_t	RESERVED8[3];
__I uint32_t	TSTMP; /*!<Time-Stamp Register 0 Offset  0x0150*/
__I  uint32_t	RESERVED9[19];
__IO uint32_t	CGSYNC; /*!<Carrier Generator Synchronization Register 0 Offset  0x01A0*/
__I  uint32_t	RESERVED10;
__IO uint32_t	RECTCFG; /*!<Rectification Configuration Register 0 Offset  0x01A8*/
}DSD_CH_TypeDef;

#define   DSD_BASE			 ((uint32_t)0x40008000U)
#define   DSD_GLOBAL_BASE		 (0x0000U)
#define   DSD_CH0_BASE		 (0x0100U)
#define   DSD_CH1_BASE		 (0x0200U)
#define   DSD_CH2_BASE		 (0x0300U)
#define   DSD_CH3_BASE		 (0x0400U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   DSD		 ((DSD_GLOBAL_TypeDef*)(DSD_BASE + DSD_GLOBAL_BASE))
#define   DSD_CH0		 ((DSD_CH_TypeDef*)(DSD_BASE + DSD_CH0_BASE))
#define   DSD_CH1		 ((DSD_CH_TypeDef*)(DSD_BASE + DSD_CH1_BASE))
#define   DSD_CH2		 ((DSD_CH_TypeDef*)(DSD_BASE + DSD_CH2_BASE))
#define   DSD_CH3		 ((DSD_CH_TypeDef*)(DSD_BASE + DSD_CH3_BASE))

/***************************************************************************/
/*                     DWT                          */
/***************************************************************************/

  /* 
  *Cortex M4 - Data Watchpoint and Trace
  */
#if __CM4_CMSIS_VERSION_MAIN < 3
typedef struct {                            /*!< DWT Structure                         */
  __IO uint32_t  DWT_CTRL;                  /*!< Use the DWT Control Register to enable the DWT unit. */
  __IO uint32_t  DWT_CYCCNT;                /*!< The DWT_CYCCNT register characteristics are:Purpose Shows or sets the value of the processor cycle counter, CYCCNT.Usage constraints The DWT unit suspends CYCCNT counting when the processor is in Debug state.Configurations Implemented only when DWT_CTRL.NOCYCCNT is RAZ, see Control register, DWT_CTRL on page C1-879.When DWT_CTRL.NOCYCCNT is RAO no cycle counter is implemented and this register is UNK/SBZP.For more information see CYCCNT cycle counter and related timers on page C1-871. */
  __IO uint32_t  DWT_CPICNT;                /*!< The DWT_CPICNT register characteristics are:Purpose Counts additional cycles required to execute multi-cycle instructions and instruction fetch stalls.Usage constraints The counter initializes to 0 when software enables its counter overflow event by setting the DWT_CTRL.CPIEVTENA bit to 1.Configurations Implemented only when DWT_CTRL.NOPRFCNT is RAZ, see Control register, DWT_CTRL on page C1-879.If DWT_CTRL.NOPRFCNT is RAO, indicating that the implementation does not include the profiling counters, this re */
  __IO uint32_t  DWT_EXCCNT;                /*!< The DWT_EXCCNT register characteristics are:Purpose Counts the total cycles spent in exception processingUsage constraints The counter initializes to 0 when software enables its counter overflow event by setting the DWT_CTRL.EXCEVTENA bit to 1.Configurations Implemented only when DWT_CTRL.NOPRFCNT is RAZ, see Control register, DWT_CTRL on page C1-879.If DWT_CTRL.NOPRFCNT is RAO, indicating that the implementation does not include the profiling counters, this register is UNK/SBZP.For more information see Pr */
  __IO uint32_t  DWT_SLEEPCNT;              /*!< The DWT_SLEEPCNT register characteristics are:Purpose Counts the total number of cycles that the processor is sleeping.Usage constraints The counter initializes to 0 when software enables its counter overflow event by setting the DWT_CTRL.SLEEPEVTENA bit to 1.Configurations Implemented only when DWT_CTRL.NOPRFCNT is RAZ, see Control register, DWT_CTRL on page C1-879.If DWT_CTRL.NOPRFCNT is RAO, indicating that the implementation does not include the profiling counters, this register is UNK/SBZP.ARM recomme */
  __IO uint32_t  DWT_LSUCNT;                /*!< The DWT_LSUCNT register characteristics are:Purpose Increments on the additional cycles required to execute all load or store instructionsUsage constraints The counter initializes to 0 when software enables its counter overflow event by setting the DWT_CTRL.LSUEVTENA bit to 1.Configurations Implemented only when DWT_CTRL.NOPRFCNT is RAZ, see Control register, DWT_CTRL on page C1-879.If DWT_CTRL.NOPRFCNT is RAO, indicating that the implementation does not include the profiling counters, this register is UNK */
  __IO uint32_t  DWT_FOLDCNT;               /*!< The DWT_FOLDCNT register characteristics are:Purpose Increments on each instruction that takes 0 cycles.Usage constraints The counter initializes to 0 when software enables its counter overflow event by setting the DWT_CTRL.FOLDEVTENA bit to 1. If an implementation includes profiling counters but does not support instruction folding, this counter can be RAZ/WI.Configurations Implemented only when DWT_CTRL.NOPRFCNT is RAZ, see Control register, DWT_CTRL on page C1-879.If DWT_CTRL.NOPRFCNT is RAO, indicating */
  __I  uint32_t  DWT_PCSR;                  /*!< The DWT_PCSR characteristics are:Purpose Samples the current value of the program counter.Usage constraints There are no usage constraints.NoteBit [0] of any sampled value is RAZ and does not reflect instruction set state as it does in a PC sample on the ARMv7-A and ARMv7-R architecture profiles.Configurations An optional feature. Register is RAZ/WI if not implemented.For more information see Program counter sampling support on page C1-877. */
  __IO uint32_t  DWT_COMP0;                 /*!< The DWT_COMP0 register characteristics are:Purpose Provides a reference value for use by comparator 0.Usage constraints The operation of comparator 0 depends also on the registers DWT_MASK0 and DWT_FUNCTION0, see Comparator Mask registers, DWT_MASKn on page C1-892 and Comparator Function registers, DWT_FUNCTIONn on page C1-893.Configurations Implemented only when DWT_CTRL.NUMCOMP is nonzero, see Control register, DWT_CTRL on page C1-879.DWT_CTRL.NUMCOMP defines the number of implemented DWT_COMPn registers */
  __IO uint32_t  DWT_MASK0;                 /*!< The DWT_MASK0 register characteristics are:Purpose Provides the size of the ignore mask applied to the access address for address range matching by comparator 0.Usage constraints The operation of comparator 0 depends also on the registers DWT_COMP0 and DWT_FUNCTION0, see Comparator registers, DWT_COMPn on page C1-891 and Comparator Function registers, DWT_FUNCTIONn on page C1-893.Configurations Implemented only when DWT_CTRL.NUMCOMP is nonzero, see Control register, DWT_CTRL on page C1-879.DWT_CTRL.NUMCOMP */
  __IO uint32_t  DWT_FUNCTION0;             /*!< The DWT_FUNCTION0 register characteristics are:Purpose Controls the operation of comparator 0.Usage constraints The operation of comparator 0 depends also on the registers DWT_COMP0 and DWT_MASK0, see Comparator registers, DWT_COMPn on page C1-891 and Comparator Mask registers, DWT_MASKn on page C1-892.Reading this register clears some fields to zero. See the field descriptions in Table C1-39 on page C1-894 for more information, and for the usage constraints of individual fields.Configurations Implemented  */
  __I  uint32_t  RESERVED0;
  __IO uint32_t  DWT_COMP1;                 /*!< The DWT_COMP1 register characteristics are:Purpose Provides a reference value for use by comparator n.Usage constraints The operation of comparator 1 depends also on the registers DWT_MASK1 and DWT_FUNCTION1, see Comparator Mask registers, DWT_MASKn on page C1-892 and Comparator Function registers, DWT_FUNCTIONn on page C1-893.Configurations Implemented only when DWT_CTRL.NUMCOMP is nonzero, see Control register, DWT_CTRL on page C1-879.DWT_CTRL.NUMCOMP defines the number of implemented DWT_COMPn registers */
  __IO uint32_t  DWT_MASK1;                 /*!< The DWT_MASK1 register characteristics are:Purpose Provides the size of the ignore mask applied to the access address for address range matching by comparator 1.Usage constraints The operation of comparator 1 depends also on the registers DWT_COMP1 and DWT_FUNCTION1, see Comparator registers, DWT_COMPn on page C1-891 and Comparator Function registers, DWT_FUNCTIONn on page C1-893.Configurations Implemented only when DWT_CTRL.NUMCOMP is nonzero, see Control register, DWT_CTRL on page C1-879.DWT_CTRL.NUMCOMP */
  __IO uint32_t  DWT_FUNCTION1;             /*!< The DWT_FUNCTION1 register characteristics are:Purpose Controls the operation of comparator 1.Usage constraints The operation of comparator 1 depends also on the registers DWT_COMP1 and DWT_MASK1, see Comparator registers, DWT_COMPn on page C1-891 and Comparator Mask registers, DWT_MASKn on page C1-892.Reading this register clears some fields to zero. See the field descriptions in Table C1-39 on page C1-894 for more information, and for the usage constraints of individual fields.Configurations Implemented  */
  __I  uint32_t  RESERVED1;
  __IO uint32_t  DWT_COMP2;                 /*!< The DWT_COMP2 register characteristics are:Purpose Provides a reference value for use by comparator n.Usage constraints The operation of comparator n depends also on the registers DWT_MASK2 and DWT_FUNCTION2, see Comparator Mask registers, DWT_MASKn on page C1-892 and Comparator Function registers, DWT_FUNCTIONn on page C1-893.Configurations Implemented only when DWT_CTRL.NUMCOMP is nonzero, see Control register, DWT_CTRL on page C1-879.DWT_CTRL.NUMCOMP defines the number of implemented DWT_COMPn registers */
  __IO uint32_t  DWT_MASK2;                 /*!< The DWT_MASK2 register characteristics are:Purpose Provides the size of the ignore mask applied to the access address for address range matching by comparator 2.Usage constraints The operation of comparator 2 depends also on the registers DWT_COMP2 and DWT_FUNCTION2, see Comparator registers, DWT_COMPn on page C1-891 and Comparator Function registers, DWT_FUNCTIONn on page C1-893.Configurations Implemented only when DWT_CTRL.NUMCOMP is nonzero, see Control register, DWT_CTRL on page C1-879.DWT_CTRL.NUMCOMP */
  __IO uint32_t  DWT_FUNCTION2;             /*!< The DWT_FUNCTION2 register characteristics are:Purpose Controls the operation of comparator 2.Usage constraints The operation of comparator 2 depends also on the registers DWT_COMP2 and DWT_MASK2, see Comparator registers, DWT_COMPn on page C1-891 and Comparator Mask registers, DWT_MASKn on page C1-892.Reading this register clears some fields to zero. See the field descriptions in Table C1-39 on page C1-894 for more information, and for the usage constraints of individual fields.Configurations Implemented  */
  __I  uint32_t  RESERVED2;
  __IO uint32_t  DWT_COMP3;                 /*!< The DWT_COMP3 register characteristics are:Purpose Provides a reference value for use by comparator n.Usage constraints The operation of comparator 3 depends also on the registers DWT_MASK3 and DWT_FUNCTION3, see Comparator Mask registers, DWT_MASKn on page C1-892 and Comparator Function registers, DWT_FUNCTIONn on page C1-893.Configurations Implemented only when DWT_CTRL.NUMCOMP is nonzero, see Control register, DWT_CTRL on page C1-879.DWT_CTRL.NUMCOMP defines the number of implemented DWT_COMPn registers */
  __IO uint32_t  DWT_MASK3;                 /*!< The DWT_MASK3 register characteristics are:Purpose Provides the size of the ignore mask applied to the access address for address range matching by comparator 3.Usage constraints The operation of comparator 3 depends also on the registers DWT_COMP3 and DWT_FUNCTION3, see Comparator registers, DWT_COMPn on page C1-891 and Comparator Function registers, DWT_FUNCTIONn on page C1-893.Configurations Implemented only when DWT_CTRL.NUMCOMP is nonzero, see Control register, DWT_CTRL on page C1-879.DWT_CTRL.NUMCOMP */
  __IO uint32_t  DWT_FUNCTION3;             /*!< The DWT_FUNCTION3 register characteristics are:Purpose Controls the operation of comparator 3.Usage constraints The operation of comparator 3 depends also on the registers DWT_COMP3 and DWT_MASK3, see Comparator registers, DWT_COMPn on page C1-891 and Comparator Mask registers, DWT_MASKn on page C1-892.Reading this register clears some fields to zero. See the field descriptions in Table C1-39 on page C1-894 for more information, and for the usage constraints of individual fields.Configurations Implemented  */
  __I  uint32_t  RESERVED3[989];
  __I  uint32_t  DWTPID4;                   /*!< Peripheral identification register4   */
  __I  uint32_t  DWTPID5;                   /*!< Peripheral identification register5   */
  __I  uint32_t  DWTPID6;                   /*!< Peripheral identification register6   */
  __I  uint32_t  DWTPID7;                   /*!< Peripheral identification register7   */
  __I  uint32_t  DWTPID0;                   /*!< Peripheral identification register0   */
  __I  uint32_t  DWTPID1;                   /*!< Peripheral identification register1   */
  __I  uint32_t  DWTPID2;                   /*!< Peripheral identification register2   */
  __I  uint32_t  DWTPID3;                   /*!< Peripheral identification register3   */
  __I  uint32_t  DWTCID0;                   /*!< Component identification register0    */
  __I  uint32_t  DWTCID1;                   /*!< Component identification register1    */
  __I  uint32_t  DWTCID2;                   /*!< Component identification register2    */
  __I  uint32_t  DWTCID3;                   /*!< Component identification register3    */
} DWT_Type;

#define   DWT_BASE			 ((uint32_t)0xE0001000U)
#define   DWT_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   DWT		 ((DWT_Type *)(DWT_BASE + DWT_GLOBAL_BASE))

#endif

/***************************************************************************/
/*                     EBU                          */
/***************************************************************************/

  /* 
  *EBU GLOBAL registers
  */
typedef struct {
__IO uint32_t	CLC; /*!<EBU Clock Control Register Offset  0x0000*/
__IO uint32_t	MODCON; /*!<EBU Configuration Register Offset  0x0004*/
__I uint32_t	ID; /*!<EBU Module Identification Register Offset  0x0008*/
__IO uint32_t	USERCON; /*!<EBU Test/Control Configuration Register Offset  0x000C*/
}EBU_GLOBAL_TypeDef;

  /* 
  *EBU chipselect registers
  */
typedef struct {
__IO uint32_t	ADDRSEL; /*!<EBU Address Select Register 0 Offset  0x0018*/
__I  uint32_t	RESERVED1[3];
__IO uint32_t	BUSRCON; /*!<EBU Bus Configuration Register Offset  0x0028*/
__IO uint32_t	BUSRAP; /*!<EBU Bus Read Access Parameter Register Offset  0x002C*/
__IO uint32_t	BUSWCON; /*!<EBU Bus Write Configuration Register Offset  0x0030*/
__IO uint32_t	BUSWAP; /*!<EBU Bus Write Access Parameter Register Offset  0x0034*/
}EBU_CS0_TypeDef;

typedef struct {
__IO uint32_t	ADDRSEL; /*!<EBU Address Select Register 0 Offset  0x001C*/
__I  uint32_t	RESERVED1[6];
__IO uint32_t	BUSRCON; /*!<EBU Bus Configuration Register Offset  0x0038*/
__IO uint32_t	BUSRAP; /*!<EBU Bus Read Access Parameter Register Offset*/
__IO uint32_t	BUSWCON; /*!<EBU Bus Write Configuration Register Offset*/  
__IO uint32_t	BUSWAP; /*!<EBU Bus Write Access Parameter Register Offset*/  
}EBU_CS1_TypeDef;

typedef struct {
__IO uint32_t	ADDRSEL; /*!<EBU Address Select Register 0 Offset  0x0020*/
__I  uint32_t	RESERVED1[9];
__IO uint32_t	BUSRCON; /*!<EBU Bus Configuration Register Offset  0x0048*/
__IO uint32_t	BUSRAP; /*!<EBU Bus Read Access Parameter Register Offset*/  
__IO uint32_t	BUSWCON; /*!<EBU Bus Write Configuration Register Offset*/  
__IO uint32_t	BUSWAP; /*!<EBU Bus Write Access Parameter Register Offset*/ 
}EBU_CS2_TypeDef;

typedef struct {
__IO uint32_t	ADDRSEL; /*!<EBU Address Select Register 0 Offset  0x0024*/
__I  uint32_t	RESERVED1[12];
__IO uint32_t	BUSRCON; /*!<EBU Bus Configuration Register Offset  0x0058*/
__IO uint32_t	BUSRAP; /*!<EBU Bus Read Access Parameter Register Offset*/ 
__IO uint32_t	BUSWCON; /*!<EBU Bus Write Configuration Register Offset*/  
__IO uint32_t	BUSWAP; /*!<EBU Bus Write Access Parameter Register Offset*/  
}EBU_CS3_TypeDef;

  /* 
  *EBU SDRAM registers
  */
typedef struct {
__IO uint32_t	SDRMCON; /*!<EBU SDRAM Control Register Offset  0x0068*/
__IO uint32_t	SDRMOD; /*!<EBU SDRAM Mode Register Offset  0x006C*/
__IO uint32_t	SDRMREF; /*!<EBU SDRAM Refresh Control Register Offset  0x0070*/
__I uint32_t	SDRSTAT; /*!<EBU SDRAM Status Register Offset  0x0074*/
}EBU_SDRAM_TypeDef;

#define   EBU_BASE			 ((uint32_t)0x58008000U)
#define   EBU_GLOBAL_BASE	(0x0000U)
#define   EBU_CS0_BASE		 (0x0018U)
#define   EBU_CS1_BASE		 (0x001CU)
#define   EBU_CS2_BASE		 (0x0020U)
#define   EBU_CS3_BASE		 (0x0024U)
#define   EBU_SDRAM_BASE	 (0x0068U)

/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   EBU		 ((EBU_GLOBAL_TypeDef*)(EBU_BASE + EBU_GLOBAL_BASE))
#define   EBU_CS0		 ((EBU_CS0_TypeDef*)(EBU_BASE + EBU_CS0_BASE))
#define   EBU_CS1		 ((EBU_CS1_TypeDef*)(EBU_BASE + EBU_CS1_BASE))
#define   EBU_CS2		 ((EBU_CS2_TypeDef*)(EBU_BASE + EBU_CS2_BASE))
#define   EBU_CS3		 ((EBU_CS3_TypeDef*)(EBU_BASE + EBU_CS3_BASE))
#define   EBU_SDRAM		 ((EBU_SDRAM_TypeDef*)(EBU_BASE + EBU_SDRAM_BASE))

/***************************************************************************/
/*                     ERUx                          */
/***************************************************************************/

  /* 
  *ERU Registers
  */
typedef struct {
__IO uint32_t	EXISEL; /*!<Event Input Select Register 0 Offset  0x0000*/
__I  uint32_t	RESERVED1[3];
__IO uint32_t	EXICON[4]; /*!<Event Input Control 0 Register Offset  0x0010*/
__IO uint32_t	EXOCON[4]; /*!<Event Output Trigger Control 0 Register Offset  0x0020*/
}ERU_GLOBAL_TypeDef;

#define   ERU0_BASE			 ((uint32_t)0x50004800U)
#define   ERU1_BASE			 ((uint32_t)0x40044000U)
#define   ERU_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   ERU0		 ((ERU_GLOBAL_TypeDef*)(ERU0_BASE + ERU_GLOBAL_BASE))
#define   ERU1		 ((ERU_GLOBAL_TypeDef*)(ERU1_BASE + ERU_GLOBAL_BASE))

/***************************************************************************/
/*                     ETHx                          */
/***************************************************************************/

  /* 
  *Ethernet Unit
  */
typedef struct {
__IO uint32_t	MAC_CONFIGURATION; /*!<Register 0 [MAC Configuration Register] Offset  0x0000*/
__IO uint32_t	MAC_FRAME_FILTER; /*!<Register 1 [MAC Frame Filter] Offset  0x0004*/
__IO uint32_t	HASH_TABLE_HIGH; /*!<Register 2 [Hash Table High Register] Offset  0x0008*/
__IO uint32_t	HASH_TABLE_LOW; /*!<Register 3 [Hash Table Low Register] Offset  0x000C*/
__IO uint32_t	GMII_ADDRESS; /*!<Register 4 [MII Address Register) Offset  0x0010*/
__IO uint32_t	GMII_DATA; /*!<Register 5 [MII Data Register] Offset  0x0014*/
__IO uint32_t	FLOW_CONTROL; /*!<Register 6 [Flow Control Register] Offset  0x0018*/
__IO uint32_t	VLAN_TAG; /*!<Register 7 [VLAN Tag Register] Offset  0x001C*/
__I uint32_t	VERSION; /*!<Register 8 [Version Register] Offset  0x0020*/
__I uint32_t	DEBUG; /*!<Register 9 [Debug Register] Offset  0x0024*/
__IO uint32_t	REMOTE_WAKE_UP_FRAME_FILTER; /*!< Offset  0x0028*/
__IO uint32_t	PMT_CONTROL_STATUS; /*!<Register 11 [PMT Control and Status Register] Offset  0x002C*/
__I  uint32_t	RESERVED1[2];
__I uint32_t	INTERRUPT_STATUS; /*!<Register 14 [Interrupt Register] Offset  0x0038*/
__IO uint32_t	INTERRUPT_MASK; /*!<Register 15 [Interrupt Mask Register] Offset  0x003C*/
__IO uint32_t	MAC_ADDRESS0_HIGH; /*!<Register 16 [MAC Address0 High Register] Offset  0x0040*/
__IO uint32_t	MAC_ADDRESS0_LOW; /*!<Register 17 [MAC Address0 Low Register] Offset  0x0044*/
__IO uint32_t	MAC_ADDRESS1_HIGH; /*!<Register 18 [MAC Address1 High Register] Offset  0x0048*/
__IO uint32_t	MAC_ADDRESS1_LOW; /*!<Register 19 [MAC Address1 Low Register] Offset  0x004C*/
__IO uint32_t	MAC_ADDRESS2_HIGH; /*!<Register 20 [MAC Address2 High Register] Offset  0x0050*/
__IO uint32_t	MAC_ADDRESS2_LOW; /*!<Register 21 [MAC Address2 Low Register] Offset  0x0054*/
__IO uint32_t	MAC_ADDRESS3_HIGH; /*!<Register 22 [MAC Address3 High Register] Offset  0x0058*/
__IO uint32_t	MAC_ADDRESS3_LOW; /*!<Register 23 [MAC Address3 Low Register] Offset  0x005C*/
__I  uint32_t	RESERVED2[40];
__IO uint32_t	MMC_CONTROL; /*!<Register 64 [MMC Control Register] Offset  0x0100*/
__I uint32_t	MMC_RECEIVE_INTERRUPT; /*!<Register 65 [MMC Receive Interrupt Register] Offset  0x0104*/
__I uint32_t	MMC_TRANSMIT_INTERRUPT; /*!<Register 66 [MMC Transmit Interrupt Register] Offset  0x0108*/
__IO uint32_t	MMC_RECEIVE_INTERRUPT_MASK; /*!< Offset  0x010C*/
__IO uint32_t	MMC_TRANSMIT_INTERRUPT_MASK; /*!< Offset  0x0110*/
__I uint32_t	TX_OCTET_COUNT_GOOD_BAD; /*!<Register 69 [Transmit Octet Count for Good and Bad Frames] Offset  0x0114*/
__I uint32_t	TX_FRAME_COUNT_GOOD_BAD; /*!<Register 70 [Transmit Frame Count for Good and Bad Frames] Offset  0x0118*/
__I uint32_t	TX_BROADCAST_FRAMES_GOOD; /*!<Register 71 [Transmit Frame Count for Good Broadcast Frames] Offset  0x011C*/
__I uint32_t	TX_MULTICAST_FRAMES_GOOD; /*!<Register 72 [Transmit Frame Count for Good Multicast Frames] Offset  0x0120*/
__I uint32_t	TX_64OCTETS_FRAMES_GOOD_BAD; /*!<Register 73 [Transmit Octet Count for Good and Bad 64 Byte Frames] Offset  0x0124*/
__I uint32_t	TX_65TO127OCTETS_FRAMES_GOOD_BAD; /*!<Register 74 [Transmit Octet Count for Good and Bad 65 to 127 Bytes Frames] Offset  0x0128*/
__I uint32_t	TX_128TO255OCTETS_FRAMES_GOOD_BAD; /*!<Register 75 [Transmit Octet Count for Good and Bad 128 to 255 Bytes Frames] Offset  0x012C*/
__I uint32_t	TX_256TO511OCTETS_FRAMES_GOOD_BAD; /*!<Register 76 [Transmit Octet Count for Good and Bad 256 to 511 Bytes Frames] Offset  0x0130*/
__I uint32_t	TX_512TO1023OCTETS_FRAMES_GOOD_BAD; /*!<Register 77 [Transmit Octet Count for Good and Bad 512 to 1023 Bytes Frames] Offset  0x0134*/
__I uint32_t	TX_1024TOMAXOCTETS_FRAMES_GOOD_BAD; /*!<Register 78 [Transmit Octet Count for Good and Bad 1024 to Maxsize Bytes Frames] Offset  0x0138*/
__I uint32_t	TX_UNICAST_FRAMES_GOOD_BAD; /*!<Register 79 [Transmit Frame Count for Good and Bad Unicast Frames] Offset  0x013C*/
__I uint32_t	TX_MULTICAST_FRAMES_GOOD_BAD; /*!<Register 80 [Transmit Frame Count for Good and Bad Multicast Frames] Offset  0x0140*/
__I uint32_t	TX_BROADCAST_FRAMES_GOOD_BAD; /*!<Register 81 [Transmit Frame Count for Good and Bad Broadcast Frames] Offset  0x0144*/
__I uint32_t	TX_UNDERFLOW_ERROR_FRAMES; /*!<Register 82 [Transmit Frame Count for Underflow Error Frames] Offset  0x0148*/
__I uint32_t	TX_SINGLE_COLLISION_GOOD_FRAMES; /*!<Register 83 [Transmit Frame Count for Frames Transmitted after Single Collision] Offset  0x014C*/
__I uint32_t	TX_MULTIPLE_COLLISION_GOOD_FRAMES; /*!<Register 84 [Transmit Frame Count for Frames Transmitted after Multiple Collision] Offset  0x0150*/
__I uint32_t	TX_DEFERRED_FRAMES; /*!< Offset  0x0154*/
__I uint32_t	TX_LATE_COLLISION_FRAMES; /*!<Register 86 [Transmit Frame Count for Late Collision Error Frames] Offset  0x0158*/
__I uint32_t	TX_EXCESSIVE_COLLISION_FRAMES; /*!<Register 87 [Transmit Frame Count for Excessive Collision Error Frames] Offset  0x015C*/
__I uint32_t	TX_CARRIER_ERROR_FRAMES; /*!<Register 88 [Transmit Frame Count for Carrier Sense Error Frames] Offset  0x0160*/
__I uint32_t	TX_OCTET_COUNT_GOOD; /*!< Offset  0x0164*/
__I uint32_t	TX_FRAME_COUNT_GOOD; /*!< Offset  0x0168*/
__I uint32_t	TX_EXCESSIVE_DEFERRAL_ERROR; /*!<Register 91 [Transmit Frame Count for Excessive Deferral Error Frames] Offset  0x016C*/
__I uint32_t	TX_PAUSE_FRAMES; /*!<Register 92 [Transmit Frame Count for Good PAUSE Frames] Offset  0x0170*/
__I uint32_t	TX_VLAN_FRAMES_GOOD; /*!<Register 93 [Transmit Frame Count for Good VLAN Frames] Offset  0x0174*/
__I uint32_t	TX_OSIZE_FRAMES_GOOD; /*!<Register 94 [Transmit Frame Count for Good Oversize Frames] Offset  0x0178*/
__I  uint32_t	RESERVED3;
__I uint32_t	RX_FRAMES_COUNT_GOOD_BAD; /*!<Register 96 [Receive Frame Count for Good and Bad Frames] Offset  0x0180*/
__I uint32_t	RX_OCTET_COUNT_GOOD_BAD; /*!<Register 97 [Receive Octet Count for Good and Bad Frames] Offset  0x0184*/
__I uint32_t	RX_OCTET_COUNT_GOOD; /*!< Offset  0x0188*/
__I uint32_t	RX_BROADCAST_FRAMES_GOOD; /*!<Register 99 [Receive Frame Count for Good Broadcast Frames] Offset  0x018C*/
__I uint32_t	RX_MULTICAST_FRAMES_GOOD; /*!<Register 100 [Receive Frame Count for Good Multicast Frames] Offset  0x0190*/
__I uint32_t	RX_CRC_ERROR_FRAMES; /*!<Register 101 [Receive Frame Count for CRC Error Frames] Offset  0x0194*/
__I uint32_t	RX_ALIGNMENT_ERROR_FRAMES; /*!<Register 102 [Receive Frame Count for Alignment Error Frames] Offset  0x0198*/
__I uint32_t	RX_RUNT_ERROR_FRAMES; /*!<Register 103 [Receive Frame Count for Runt Error Frames] Offset  0x019C*/
__I uint32_t	RX_JABBER_ERROR_FRAMES; /*!<Register 104 [Receive Frame Count for Jabber Error Frames] Offset  0x01A0*/
__I uint32_t	RX_UNDERSIZE_FRAMES_GOOD; /*!<Register 105 [Receive Frame Count for Undersize Frames] Offset  0x01A4*/
__I uint32_t	RX_OVERSIZE_FRAMES_GOOD; /*!< Offset  0x01A8*/
__I uint32_t	RX_64OCTETS_FRAMES_GOOD_BAD; /*!<Register 107 [Receive Frame Count for Good and Bad 64 Byte Frames] Offset  0x01AC*/
__I uint32_t	RX_65TO127OCTETS_FRAMES_GOOD_BAD; /*!<Register 108 [Receive Frame Count for Good and Bad 65 to 127 Bytes Frames] Offset  0x01B0*/
__I uint32_t	RX_128TO255OCTETS_FRAMES_GOOD_BAD; /*!<Register 109 [Receive Frame Count for Good and Bad 128 to 255 Bytes Frames] Offset  0x01B4*/
__I uint32_t	RX_256TO511OCTETS_FRAMES_GOOD_BAD; /*!<Register 110 [Receive Frame Count for Good and Bad 256 to 511 Bytes Frames] Offset  0x01B8*/
__I uint32_t	RX_512TO1023OCTETS_FRAMES_GOOD_BAD; /*!<Register 111 [Receive Frame Count for Good and Bad 512 to 1,023 Bytes Frames] Offset  0x01BC*/
__I uint32_t	RX_1024TOMAXOCTETS_FRAMES_GOOD_BAD; /*!<Register 112 [Receive Frame Count for Good and Bad 1,024 to Maxsize Bytes Frames] Offset  0x01C0*/
__I uint32_t	RX_UNICAST_FRAMES_GOOD; /*!<Register 113 [Receive Frame Count for Good Unicast Frames] Offset  0x01C4*/
__I uint32_t	RX_LENGTH_ERROR_FRAMES; /*!<Register 114 [Receive Frame Count for Length Error Frames] Offset  0x01C8*/
__I uint32_t	RX_OUT_OF_RANGE_TYPE_FRAMES; /*!<Register 115 [Receive Frame Count for Out of Range Frames] Offset  0x01CC*/
__I uint32_t	RX_PAUSE_FRAMES; /*!<Register 116 [Receive Frame Count for PAUSE Frames] Offset  0x01D0*/
__I uint32_t	RX_FIFO_OVERFLOW_FRAMES; /*!<Register 117 [Receive Frame Count for FIFO Overflow Frames] Offset  0x01D4*/
__I uint32_t	RX_VLAN_FRAMES_GOOD_BAD; /*!<Register 118 [Receive Frame Count for Good and Bad VLAN Frames] Offset  0x01D8*/
__I uint32_t	RX_WATCHDOG_ERROR_FRAMES; /*!<Register 119 [Receive Frame Count for Watchdog Error Frames] Offset  0x01DC*/
__I uint32_t	RX_RECEIVE_ERROR_FRAMES; /*!<Register 120 [Receive Frame Count for Receive Error Frames] Offset  0x01E0*/
__I uint32_t	RX_CONTROL_FRAMES_GOOD; /*!<Register 121 [Receive Frame Count for Good Control Frames Frames] Offset  0x01E4*/
__I  uint32_t	RESERVED4[6];
__IO uint32_t	MMC_IPC_RECEIVE_INTERRUPT_MASK; /*!<Register 128 [MMC Receive Checksum Offload Interrupt Mask Register] Offset  0x0200*/
__I  uint32_t	RESERVED5;
__I uint32_t	MMC_IPC_RECEIVE_INTERRUPT; /*!<Register 130 [MMC Receive Checksum Offload Interrupt Register] Offset  0x0208*/
__I  uint32_t	RESERVED6;
__I uint32_t	RXIPV4_GOOD_FRAMES; /*!< Offset  0x0210*/
__I uint32_t	RXIPV4_HEADER_ERROR_FRAMES; /*!<Register 133 [Receive IPV4 Header Error Frame Counter Register] Offset  0x0214*/
__I uint32_t	RXIPV4_NO_PAYLOAD_FRAMES; /*!<Register 134 [Receive IPV4 No Payload Frame Counter Register] Offset  0x0218*/
__I uint32_t	RXIPV4_FRAGMENTED_FRAMES; /*!<Register 135 [Receive IPV4 Fragmented Frame Counter Register] Offset  0x021C*/
__I uint32_t	RXIPV4_UDP_CHECKSUM_DISABLED_FRAMES; /*!<Register 136 [Receive IPV4 UDP Checksum Disabled Frame Counter Register] Offset  0x0220*/
__I uint32_t	RXIPV6_GOOD_FRAMES; /*!< Offset  0x0224*/
__I uint32_t	RXIPV6_HEADER_ERROR_FRAMES; /*!<Register 138 [Receive IPV6 Header Error Frame Counter Register] Offset  0x0228*/
__I uint32_t	RXIPV6_NO_PAYLOAD_FRAMES; /*!<Register 139 [Receive IPV6 No Payload Frame Counter Register] Offset  0x022C*/
__I uint32_t	RXUDP_GOOD_FRAMES; /*!< Offset  0x0230*/
__I uint32_t	RXUDP_ERROR_FRAMES; /*!< Offset  0x0234*/
__I uint32_t	RXTCP_GOOD_FRAMES; /*!< Offset  0x0238*/
__I uint32_t	RXTCP_ERROR_FRAMES; /*!< Offset  0x023C*/
__I uint32_t	RXICMP_GOOD_FRAMES; /*!< Offset  0x0240*/
__I uint32_t	RXICMP_ERROR_FRAMES; /*!< Offset  0x0244*/
__I  uint32_t	RESERVED7[2];
__I uint32_t	RXIPV4_GOOD_OCTETS; /*!< Offset  0x0250*/
__I uint32_t	RXIPV4_HEADER_ERROR_OCTETS; /*!<Register 149 [Receive IPV4 Header Error Octet Counter Register] Offset  0x0254*/
__I uint32_t	RXIPV4_NO_PAYLOAD_OCTETS; /*!<Register 150 [Receive IPV4 No Payload Octet Counter Register] Offset  0x0258*/
__I uint32_t	RXIPV4_FRAGMENTED_OCTETS; /*!<Register 151 [Receive IPV4 Fragmented Octet Counter Register] Offset  0x025C*/
__I uint32_t	RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS; /*!<Register 152 [Receive IPV4 Fragmented Octet Counter Register] Offset  0x0260*/
__I uint32_t	RXIPV6_GOOD_OCTETS; /*!< Offset  0x0264*/
__I uint32_t	RXIPV6_HEADER_ERROR_OCTETS; /*!<Register 154 [Receive IPV6 Header Error Octet Counter Register] Offset  0x0268*/
__I uint32_t	RXIPV6_NO_PAYLOAD_OCTETS; /*!<Register 155 [Receive IPV6 No Payload Octet Counter Register] Offset  0x026C*/
__I uint32_t	RXUDP_GOOD_OCTETS; /*!< Offset  0x0270*/
__I uint32_t	RXUDP_ERROR_OCTETS; /*!< Offset  0x0274*/
__I uint32_t	RXTCP_GOOD_OCTETS; /*!< Offset  0x0278*/
__I uint32_t	RXTCP_ERROR_OCTETS; /*!< Offset  0x027C*/
__I uint32_t	RXICMP_GOOD_OCTETS; /*!< Offset  0x0280*/
__I uint32_t	RXICMP_ERROR_OCTETS; /*!< Offset  0x0284*/
__I  uint32_t	RESERVED8[286];
__IO uint32_t	TIMESTAMP_CONTROL; /*!<Register 448 [Timestamp Control Register] Offset  0x0700*/
__IO uint32_t	SUB_SECOND_INCREMENT; /*!<Register 449 [Sub-Second Increment Register] Offset  0x0704*/
__I uint32_t	SYSTEM_TIME_SECONDS; /*!<Register 450 [System Time - Seconds Register] Offset  0x0708*/
__I uint32_t	SYSTEM_TIME_NANOSECONDS; /*!< Offset  0x070C*/
__IO uint32_t	SYSTEM_TIME_SECONDS_UPDATE; /*!<Register 452 [System Time - Seconds Update Register] Offset  0x0710*/
__IO uint32_t	SYSTEM_TIME_NANOSECONDS_UPDATE; /*!< Offset  0x0714*/
__IO uint32_t	TIMESTAMP_ADDEND; /*!<Register 454 [Timestamp Addend Register] Offset  0x0718*/
__IO uint32_t	TARGET_TIME_SECONDS; /*!<Register 455 [Target Time Seconds Register] Offset  0x071C*/
__IO uint32_t	TARGET_TIME_NANOSECONDS; /*!< Offset  0x0720*/
__IO uint32_t	SYSTEM_TIME_HIGHER_WORD_SECONDS; /*!<Register 457 [System Time - Higher Word Seconds Register] Offset  0x0724*/
__I uint32_t	TIMESTAMP_STATUS; /*!<Register 458 [Timestamp Status Register] Offset  0x0728*/
__IO uint32_t	PPS_CONTROL; /*!<Register 459 [PPS Control Register] Offset  0x072C*/
__I  uint32_t	RESERVED9[564];
__IO uint32_t	BUS_MODE; /*!<Register 0 [Bus Mode Register] Offset  0x1000*/
__IO uint32_t	TRANSMIT_POLL_DEMAND; /*!<Register 1 [Transmit Poll Demand Register] Offset  0x1004*/
__IO uint32_t	RECEIVE_POLL_DEMAND; /*!< Offset  0x1008*/
__IO uint32_t	RECEIVE_DESCRIPTOR_LIST_ADDRESS; /*!< Offset  0x100C*/
__IO uint32_t	TRANSMIT_DESCRIPTOR_LIST_ADDRESS; /*!< Offset  0x1010*/
__IO uint32_t	STATUS; /*!< Offset  0x1014*/
__IO uint32_t	OPERATION_MODE; /*!<Register 6 [Operation Mode Register] Offset  0x1018*/
__IO uint32_t	INTERRUPT_ENABLE; /*!<Register 7 [Interrupt Enable Register] Offset  0x101C*/
__I uint32_t	MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER; /*!< Offset  0x1020*/
__IO uint32_t	RECEIVE_INTERRUPT_WATCHDOG_TIMER; /*!< Offset  0x1024*/
__I  uint32_t	RESERVED10;
__I uint32_t	AHB_STATUS; /*!<Register 11 [AHB Status Register] Offset  0x102C*/
__I  uint32_t	RESERVED11[6];
__I uint32_t	CURRENT_HOST_TRANSMIT_DESCRIPTOR; /*!< Offset  0x1048*/
__I uint32_t	CURRENT_HOST_RECEIVE_DESCRIPTOR; /*!< Offset  0x104C*/
__I uint32_t	CURRENT_HOST_TRANSMIT_BUFFER_ADDRESS; /*!<Register 20 [Current Host Transmit Buffer Address Register] Offset  0x1050*/
__I uint32_t	CURRENT_HOST_RECEIVE_BUFFER_ADDRESS; /*!<Register 21 [Current Host Receive Buffer Address Register] Offset  0x1054*/
__IO uint32_t	HW_FEATURE; /*!<Register 22 [HW Feature Register] Offset  0x1058*/
}ETH_GLOBAL_TypeDef;

#define   ETH0_BASE			 ((uint32_t)0x5000C000U)
#define   ETH_GLOBAL_BASE		 (0x0000U)

/***************************************************************************/
/*                     ETH0_CON                          */
/***************************************************************************/

  /* 
  *Ethernet I/O Control Register
  */
typedef struct {
__IO uint32_t	CON; /*!<Ethernet 0 Port Control Register Offset  0x0000*/
}ETH0_CON_GLOBAL_TypeDef;

#define   ETH0_CON_BASE			 ((uint32_t)0x50004040U)
#define   ETH0_CON_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   ETH0_CON		 ((ETH0_CON_GLOBAL_TypeDef*)(ETH0_CON_BASE + ETH0_CON_GLOBAL_BASE))
#define   ETH0		 ((ETH_GLOBAL_TypeDef*)(ETH0_BASE + ETH_GLOBAL_BASE))

/***************************************************************************/
/*                     ETM                          */
/***************************************************************************/

  /* 
  *Cortex M4 - Embedded Trace Macrocell
  */
typedef struct {                            /*!< ETM Structure                         */
  __IO uint32_t  ETMCR;                     /*!< The ETMCR characteristics are:Purpose Controls general operation of the ETM, such as whether tracing is enabled.Configurations This register is only available if the processor is configured to use the ETM. */
  __I  uint32_t  ETMCCR;                    /*!< The ETM Configuration Code Register characteristics are:Purpose Enables software to read the implementation-specific configuration of the ETM.Configurations This register is only available if the processor is configured to use the ETM. */
  __IO uint32_t  ETMTRIGGER;                /*!< The Trigger Event Register defines the event that controls the trigger. It is: register 0x02, at offset 0x08 in a memory-mapped implementation a write-only register, although in architecture versions 3.1 and later it is read/write when bit [11] ofthe Configuration Code Extension Register is set, see Configuration Code Extension Register,ETMv3.1 and later on page 3-76. */
  __I  uint32_t  RESERVED0;
  __IO uint32_t  ETMSR;                     /*!< The ETM Status Register provides information about the current status of the trace and trigger logic. It is: register 0x04, at offset 0x10 in a memory-mapped implementation a read/write register */
  __I  uint32_t  ETMSCR;                    /*!< The ETMSCR characteristics are:Purpose Shows the ETM features supported by the implementation of the ETM macrocell.Configurations This register is only available if the processor is configured to use the ETM. */
  __I  uint32_t  RESERVED1[2];
  __IO uint32_t  ETMTEEVR;                  /*!< The TraceEnable Event Register defines the TraceEnable enabling event. It is: register 0x08, at offset 0x20 in a memory-mapped implementation a write-only register, although in architecture versions 3.1 and later it is read/write when bit [11] of the Configuration Code Extension Register is set, see Configuration Code Extension Register, ETMv3.1 and later on page 3-76. */
  __IO uint32_t  ETMTECR1;                  /*!< The ETMTECR1 characteristics are:Purpose Enables the start/stop logic used for trace enable.Configurations This register is only available if the processor is configured to use the ETM. */
  __IO uint32_t  ETMFFLR;                   /*!< The FIFOFULL Level Register holds the level below which the FIFO is considered full, although its function varies for different ETM architectures. From ETMv3.0 the value in this register also controls the point at which data trace suppression occurs.The FIFOFULL Level Register is: register 0x0B, at offset 0x2C in a memory-mapped implementation a read/write register in ETM architecture versions 2.0 and later. */
  __I  uint32_t  RESERVED2[69];
  __IO uint32_t  ETMCNTRLDVR1;              /*!< A Counter Reload Value Register specifies the starting value of the associated counter. The counter is automatically loaded with this value when this register is programmed and when the ETM Programming bit is set. It is then reloaded with this value whenever the counter reload event, specified by the Counter Reload Event Register, is active. The Counter Reload Value Registers are: registers 0x50 to 0x53, at offsets 0x140, 0x144, 0x148, and 0x14C in a memory-mapped implementation write-only registers, altho */
  __I  uint32_t  RESERVED3[39];
  __I  uint32_t  ETMSYNCFR;                 /*!< The ETMSYNCFR characteristics are:Purpose Holds the trace synchronization frequency value.From ETMv3.5, this value must be set to a value greater than the size of the FIFO, or to zero. Values greater than zero but less than the FIFO size are still not permitted.A value of zero disables periodic synchronization based on the synchronization frequency counter. This does not affect other sources of synchronization, such as external requests from a CoreSight system. */
  __I  uint32_t  ETMIDR;                    /*!< The ETMIDR characteristics are:Purpose Holds the ETM architecture variant, and defines the programmers model for the ETM.Configurations This register is only available if the processor is configured to use the ETM. */
  __I  uint32_t  ETMCCER;                   /*!< The ETMCCER characteristics are:Purpose Holds ETM configuration information additional to that in the ETMCCR.See Configuration Code Register, ETMCCR on page 10-14.Configurations This register is only available if the processor is configured to use the ETM. */
  __I  uint32_t  RESERVED4;
  __IO uint32_t  ETMTESSEICR;               /*!< The Trace Start/Stop EmbeddedICE Control Register specifies the EmbeddedICE watchpoint comparator inputs that are used as trace start and stop resources. It is: register 0x7C, at offset 0x1F0 in a memory-mapped implementation a write-only register, that is read/write when bit [11] of the Configuration Code Extension Register is set, see Configuration Code Extension Register, ETMv3.1 and later on page 3-76. */
  __I  uint32_t  RESERVED5;
  __IO uint32_t  ETMTSEVR;                  /*!< The ETMTSEVR characteristics are:Purpose Defines an event that requests the insertion of a timestamp into the trace stream. */
  __I  uint32_t  RESERVED6;
  __IO uint32_t  ETMTRACEIDR;               /*!< The ETMTRACEIDR characteristics are:Purpose Defines the 7-bit Trace ID, for output to the trace bus.This register is used in systems where multiple trace sources are present and tracing simultaneously. For example, where an ETM outputs trace onto the AMBA version 3 Advanced Trace Bus (ATB), a unique ID is required for each trace source so that the trace can be uniquely identified as coming from a particular trace source. For more information about the AMBA version 3 ATB, see the CoreSight Architecture Spec */
  __I  uint32_t  RESERVED7;
  __I  uint32_t  ETMIDR2;                   /*!< The ETMIDR2 characteristics are:Purpose Provides an extension to the ETM ID register, ETMIDR. */
  __I  uint32_t  RESERVED8[66];
  __I  uint32_t  ETMPDSR;                   /*!< The ETMPDSR characteristics are:Purpose Indicates the power-down status of the ETM.Configurations This register is only available if the processor is configured to use an ETM. */
  __I  uint32_t  RESERVED9[754];
  __I  uint32_t  ITMISCIN;                  /*!< The ITMISCIN characteristics are:Purpose Integration test.Configurations This register is only available if the processor is configured to use the ETM. */
  __I  uint32_t  RESERVED10;
  __IO uint32_t  ITTRIGOUT;                 /*!< The ITMISCIN characteristics are:Purpose Integration test.Usage constraints You must set bit [0] of ETMITCTRL to use this register.Configurations This register is only available if the processor is configured to use the ETM. */
  __I  uint32_t  RESERVED11;
  __I  uint32_t  ETM_ITATBCTR2;             /*!< The ETM_ITATBCTR2 characteristics are:Purpose Integration test.Usage constraints You must set bit [0] of ETMITCTRL to use this register.Configurations This register is only available if the processor is configured to use the ETM. */
  __I  uint32_t  RESERVED12;
  __IO uint32_t  ETM_ITATBCTR0;             /*!< The Integration Test ATB Control (ETM_ITATBCTR0) characteristics are:Purpose Integration test.Usage constraints You must set bit [0] of ETMITCTRL to use this register.Configurations This register is only available if the processor is configured to use the ETM. */
  __I  uint32_t  RESERVED13;
  __IO uint32_t  ETMITCTRL;                 /*!< The Integration Mode Control Register (ITCTRL) is used to enable topology detection or to check integration testing. It is: register 0x3C0, at offset 0xF00 in a memory-mapped implementation */
  __I  uint32_t  RESERVED14[39];
  __IO uint32_t  ETMCLAIMSET;               /*!< The Claim Tag Set Register (CLAIMSET) is used to set bits in the claim tag and find the number of bits supported by the claim tag. It is: register 0x3E8, at offset 0xFA0 in a memory-mapped implementation */
  __IO uint32_t  ETMCLAIMCLR;               /*!< The Claim Tag Clear Register (CLAIMCLR) is used to clear bits in the claim tag to 0, and to find the current value of the claim tag. It is: register 0x3E9, at offset 0xFA4 in a memory-mapped implementation */
  __I  uint32_t  RESERVED15[2];
  __IO uint32_t  ETMLAR;                    /*!< The Lock Access Register (LAR or LOCKACCESS) is used to lock and unlock access to all other ETM registers. It is: register 0x3EC, at offset 0xFB0 in a memory-mapped implementation */
  __I  uint32_t  ETMLSR;                    /*!< The Lock Status Register (LSR or LOCKSTATUS) has two uses: If you read this register from any interface, you can check bit [0] to find out whether the lock registers are implemented for the interface you are using. If you read this register from an interface for which lock registers are implemented, you can check bit [1] to find out whether the registers are currently locked.The Lock Status Register is: register 0x3ED, at offset 0xFB4 in a memory-mapped implementation */
  __I  uint32_t  ETMAUTHSTATUS;             /*!< The Authentication Status Register (AUTHSTATUS) reports the level of tracing currently permitted by the authentication signals provided to the ETM. It is: register 0x3EE, at offset 0xFB8 in a memory-mapped implementation */
  __I  uint32_t  RESERVED16[4];
  __I  uint32_t  ETMDEVTYPE;                /*!< The Device Type Register (DEVTYPE) returns the CoreSight device type of the component. It is: register 0x3F3, at offset 0xFCC in a memory-mapped implementation present in all CoreSight components. */
  __I  uint32_t  ETMPIDR4;                  /*!< The Peripheral ID4 Register holds peripheral identification information. */
  __I  uint32_t  ETMPIDR5;                  /*!< Peripheral Identification register5. See ARM Embedded TraceMacrocell Architecture Specification */
  __I  uint32_t  ETMPIDR6;                  /*!< Peripheral Identification register6. See ARM Embedded TraceMacrocell Architecture Specification */
  __I  uint32_t  ETMPIDR7;                  /*!< Peripheral Identification register7. See ARM Embedded TraceMacrocell Architecture Specification */
  __I  uint32_t  ETMPIDR0;                  /*!< The Peripheral ID0 Register holds peripheral identification information. */
  __I  uint32_t  ETMPIDR1;                  /*!< The Peripheral ID1 Register holds peripheral identification information. */
  __I  uint32_t  ETMPIDR2;                  /*!< The Peripheral ID2 Register holds peripheral identification information. */
  __I  uint32_t  ETMPIDR3;                  /*!< The Peripheral ID3 Register holds peripheral identification information. */
  __I  uint32_t  ETMCIDR0;                  /*!< The Component ID0 Register holds byte 0 of the CoreSight preamble information. */
  __I  uint32_t  ETMCIDR1;                  /*!< The Component ID1 Register holds byte 1 of the CoreSight preamble information. */
  __I  uint32_t  ETMCIDR2;                  /*!< The Component ID2 Register holds byte 2 of the CoreSight preamble information. */
  __I  uint32_t  ETMCIDR3;                  /*!< The Component ID3 Register holds byte 3 of the CoreSight preamble information. */
} ETM_Type;

#define   ETM_BASE			 ((uint32_t)0xE0041000U)
#define   ETM_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   ETM		 ((ETM_Type *)(ETM_BASE + ETM_GLOBAL_BASE))

/***************************************************************************/
/*                     FCE                          */
/***************************************************************************/

  /* 
  *Global registers
  */
typedef struct {
__IO uint32_t	CLC; /*!<Clock Control Register Offset  0x0000*/
__I  uint32_t	RESERVED1;
__I uint32_t	ID; /*!<Module Identification Register Offset  0x0008*/
}FCE_GLOBAL_TypeDef;

  /* 
  *Generic CRC Engine registers
  */
typedef struct {
__IO uint32_t	IR; /*!<Input Register 0 Offset  0x0020*/
__I uint32_t	RES; /*!<CRC Result Register 0 Offset  0x0024*/
__IO uint32_t	CFG; /*!<CRC Configuration Register 0 Offset  0x0028*/
__IO uint32_t	STS; /*!<CRC Status Register 0 Offset  0x002C*/
__IO uint32_t	LENGTH; /*!<CRC Length Register 0 Offset  0x0030*/
__IO uint32_t	CHECK; /*!<CRC Check Register 0 Offset  0x0034*/
__IO uint32_t	CRC; /*!<CRC Register 0 Offset  0x0038*/
__IO uint32_t	CTR; /*!<CRC Test Register 0 Offset  0x003C*/
}FCE_KE_TypeDef;

#define   FCE_BASE			 ((uint32_t)0x50020000U)
#define   FCE_GLOBAL_BASE		 (0x0000U)
#define   FCE_KE0_BASE		 (0x0020U)
#define   FCE_KE1_BASE		 (0x0040U)
#define   FCE_KE2_BASE		 (0x0060U)
#define   FCE_KE3_BASE		 (0x0080U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   FCE		 ((FCE_GLOBAL_TypeDef*)(FCE_BASE + FCE_GLOBAL_BASE))
#define   FCE_KE0		 ((FCE_KE_TypeDef*)(FCE_BASE + FCE_KE0_BASE))
#define   FCE_KE1		 ((FCE_KE_TypeDef*)(FCE_BASE + FCE_KE1_BASE))
#define   FCE_KE2		 ((FCE_KE_TypeDef*)(FCE_BASE + FCE_KE2_BASE))
#define   FCE_KE3		 ((FCE_KE_TypeDef*)(FCE_BASE + FCE_KE3_BASE))

/***************************************************************************/
/*                     FLASH0                          */
/***************************************************************************/

  /* 
  *FLASH0 GLOBAL registers
  */
typedef struct {
__I uint32_t	ID; /*!<Flash Module Identification Register Offset  0x1008*/
__I  uint32_t	RESERVED1;
__I uint32_t	FSR; /*!<Flash Status Register Offset  0x1010*/
__IO uint32_t	FCON; /*!<Flash Configuration Register Offset  0x1014*/
__IO uint32_t	MARP; /*!<Margin Control Register PFLASH Offset  0x1018*/
__I  uint32_t	RESERVED2;
__I uint32_t	PROCON0; /*!<Flash Protection Configuration Register User 0 Offset  0x1020*/
__I uint32_t	PROCON1; /*!<Flash Protection Configuration Register User 1 Offset  0x1024*/
__I uint32_t	PROCON2; /*!<Flash Protection Configuration Register User 2 Offset  0x1028*/
}FLASH0_GLOBAL_TypeDef;

#define   FLASH0_BASE			 ((uint32_t)0x58001000U)
#define   FLASH0_GLOBAL_BASE		 (0x1008U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   FLASH0		 ((FLASH0_GLOBAL_TypeDef*)(FLASH0_BASE + FLASH0_GLOBAL_BASE))

/***************************************************************************/
/*                     FPB                          */
/***************************************************************************/

  /* 
  *Cortex M4 - Flash Patch and Breakpoint
  */
typedef struct {                            /*!< FPB Structure                         */
  __IO uint32_t  FP_CTRL;                   /*!< The FP_CTRL Register characteristics are:Purpose Provides FPB implementation information, and the global enable for the FPB unit.If implemented: The instruction address comparators start at FP_COMP0. This means the last instruction address comparator is FP_COMPn, where n = (NUM_CODE-1). The maximum number of instruction address comparators is 127. The literal address comparators start at FP_COMPm, where m=NUM_CODE. This means the last literal address comparator is at FP_COMPp, where p=(NUM_CODE+NUM_LIT-1). */
  __IO uint32_t  FP_REMAP;                  /*!< The FP_REMAP register characteristics are:Purpose Indicates whether the implementation supports flash patch remap, and if it does, holds the SRAM address for remap.The remap base address must be aligned to the number of words required to support the implemented comparators, that is to (NUM_CODE+NUM_LIT) words, with a minimum alignment of 8 words. Because remap is into the SRAM memory region, 0x20000000-0x3FFFFFFF, bits [31:29] of the remap address are 0b001. */
  __IO uint32_t  FP_COMP0;                  /*!< The FP_COMP0 register characteristics are:Purpose Holds an address for comparison with addresses in the Code memory region, see The system address map on page B3-704. The effect of a match depends on whether the comparator is an instruction address comparator or a literal address comparator:Instruction address comparatorsEither: defines an instruction address to remap to an address based on the address specified in the FP_REMAP register defines a breakpoint address.Literal address comparatorsDefines a lite */
  __IO uint32_t  FP_COMP1;                  /*!< The FP_COMP1 register characteristics are:Purpose Holds an address for comparison with addresses in the Code memory region, see The system address map on page B3-704. The effect of a match depends on whether the comparator is an instruction address comparator or a literal address comparator:Instruction address comparatorsEither: defines an instruction address to remap to an address based on the address specified in the FP_REMAP register defines a breakpoint address.Literal address comparatorsDefines a lite */
  __IO uint32_t  FP_COMP2;                  /*!< The FP_COMP2 register characteristics are:Purpose Holds an address for comparison with addresses in the Code memory region, see The system address map on page B3-704. The effect of a match depends on whether the comparator is an instruction address comparator or a literal address comparator:Instruction address comparatorsEither: defines an instruction address to remap to an address based on the address specified in the FP_REMAP register defines a breakpoint address.Literal address comparatorsDefines a lite */
  __IO uint32_t  FP_COMP3;                  /*!< The FP_COMP3 register characteristics are:Purpose Holds an address for comparison with addresses in the Code memory region, see The system address map on page B3-704. The effect of a match depends on whether the comparator is an instruction address comparator or a literal address comparator:Instruction address comparatorsEither: defines an instruction address to remap to an address based on the address specified in the FP_REMAP register defines a breakpoint address.Literal address comparatorsDefines a lite */
  __IO uint32_t  FP_COMP4;                  /*!< The FP_COMP4 register characteristics are:Purpose Holds an address for comparison with addresses in the Code memory region, see The system address map on page B3-704. The effect of a match depends on whether the comparator is an instruction address comparator or a literal address comparator:Instruction address comparatorsEither: defines an instruction address to remap to an address based on the address specified in the FP_REMAP register defines a breakpoint address.Literal address comparatorsDefines a lite */
  __IO uint32_t  FP_COMP5;                  /*!< The FP_COMP5 register characteristics are:Purpose Holds an address for comparison with addresses in the Code memory region, see The system address map on page B3-704. The effect of a match depends on whether the comparator is an instruction address comparator or a literal address comparator:Instruction address comparatorsEither: defines an instruction address to remap to an address based on the address specified in the FP_REMAP register defines a breakpoint address.Literal address comparatorsDefines a lite */
  __IO uint32_t  FP_COMP6;                  /*!< The FP_COMP6 register characteristics are:Purpose Holds an address for comparison with addresses in the Code memory region, see The system address map on page B3-704. The effect of a match depends on whether the comparator is an instruction address comparator or a literal address comparator:Instruction address comparatorsEither: defines an instruction address to remap to an address based on the address specified in the FP_REMAP register defines a breakpoint address.Literal address comparatorsDefines a lite */
  __IO uint32_t  FP_COMP7;                  /*!< The FP_COMP7 register characteristics are:Purpose Holds an address for comparison with addresses in the Code memory region, see The system address map on page B3-704. The effect of a match depends on whether the comparator is an instruction address comparator or a literal address comparator:Instruction address comparatorsEither: defines an instruction address to remap to an address based on the address specified in the FP_REMAP register defines a breakpoint address.Literal address comparatorsDefines a lite */
  __I  uint32_t  RESERVED0[1002];
  __I  uint32_t  FPBPID4;                   /*!< Peripheral identification register4   */
  __I  uint32_t  FPBPID5;                   /*!< Peripheral identification register5   */
  __I  uint32_t  FPBPID6;                   /*!< Peripheral identification register6   */
  __I  uint32_t  FPBPID7;                   /*!< Peripheral identification register7   */
  __I  uint32_t  FPBPID0;                   /*!< Peripheral identification register0   */
  __I  uint32_t  FPBPID1;                   /*!< Peripheral identification register1   */
  __I  uint32_t  FPBPID2;                   /*!< Peripheral identification register2   */
  __I  uint32_t  FPBPID3;                   /*!< Peripheral identification register3   */
  __I  uint32_t  FPBCID0;                   /*!< Component identification register0    */
  __I  uint32_t  FPBCID1;                   /*!< Component identification register1    */
  __I  uint32_t  FPBCID2;                   /*!< Component identification register2    */
  __I  uint32_t  FPBCID3;                   /*!< Component identification register3    */
} FPB_Type;

#define   FPB_BASE			 ((uint32_t)0xE0002000U)
#define   FPB_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   FPB		 ((FPB_Type *)(FPB_BASE + FPB_GLOBAL_BASE))

/***************************************************************************/
/*                     LEDTS0                          */
/***************************************************************************/

  /* 
  *LEDTS0 KERNEL registers
  */
typedef struct {
__I uint32_t	ID; /*!<Module Identification Register Offset  0x0000*/
__IO uint32_t	GLOBCTL; /*!<Global Control Register Offset  0x0004*/
__IO uint32_t	FNCTL; /*!<Function Control Register Offset  0x0008*/
__IO uint32_t	EVFR; /*!<Event Flag Register Offset  0x000C*/
__IO uint32_t	TSVAL; /*!<Touch-sense TS-Counter Value Offset  0x0010*/
__IO uint32_t	LINE0; /*!<Line Pattern Register 0 Offset  0x0014*/
__IO uint32_t	LINE1; /*!<Line Pattern Register 1 Offset  0x0018*/
__IO uint32_t	LDCMP0; /*!<LED Compare Register 0 Offset  0x001C*/
__IO uint32_t	LDCMP1; /*!<LED Compare Register 1 Offset  0x0020*/
__IO uint32_t	TSCMP0; /*!<Touch-sense Compare Register 0 Offset  0x0024*/
__IO uint32_t	TSCMP1; /*!<Touch-sense Compare Register 1 Offset  0x0028*/
}LEDTS0_GLOBAL_TypeDef;

#define   LEDTS0_BASE			 ((uint32_t)0x48010000U)
#define   LEDTS0_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   LEDTS0		 ((LEDTS0_GLOBAL_TypeDef*)(LEDTS0_BASE + LEDTS0_GLOBAL_BASE))

/***************************************************************************/
/*                     PBAx                          */
/***************************************************************************/

  /* 
  *Peripheral Bridge Status and Write Error Address registers
  */
typedef struct {
__IO uint32_t	STS; /*!<Peripheral Bridge Status Register Offset  0x0000*/
__I uint32_t	WADDR; /*!<PBA Write Error Address Register Offset  0x0004*/
}PBA_GLOBAL_TypeDef;

#define   PBA0_BASE			 ((uint32_t)0x40000000U)
#define   PBA1_BASE			 ((uint32_t)0x48000000U)
#define   PBA_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   PBA0		 ((PBA_GLOBAL_TypeDef*)(PBA0_BASE + PBA_GLOBAL_BASE))
#define   PBA1		 ((PBA_GLOBAL_TypeDef*)(PBA1_BASE + PBA_GLOBAL_BASE))

/***************************************************************************/
/*                     PMU0                          */
/***************************************************************************/

  /* 
  *PMU0 ID register
  */
typedef struct {
__I uint32_t	ID; /*!<PMU0 Identification Register Offset  0x0508*/
}PMU0_GLOBAL_TypeDef;

#define   PMU0_BASE			 ((uint32_t)0x58000000U)
#define   PMU0_GLOBAL_BASE		 (0x0508U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   PMU0		 ((PMU0_GLOBAL_TypeDef*)(PMU0_BASE + PMU0_GLOBAL_BASE))

// ------------------------------------------------------------------------------------------------
// -----                                         PORT0                                        -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Port 0 (PORT0)
  */

typedef struct {                            /*!< PORT0 Structure                       */
  __IO uint32_t  OUT;                       /*!< Port 0 Output Register                */
  __O  uint32_t  OMR;                       /*!< Port 0 Output Modification Register   */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  IOCR0;                     /*!< Port 0 Input/Output Control Register 0 */
  __IO uint32_t  IOCR4;                     /*!< Port 0 Input/Output Control Register 4 */
  __IO uint32_t  IOCR8;                     /*!< Port 0 Input/Output Control Register 8 */
  __IO uint32_t  IOCR12;                    /*!< Port 0 Input/Output Control Register 12 */
  __I  uint32_t  RESERVED1;
  __I  uint32_t  IN;                        /*!< Port 0 Input Register                 */
  __I  uint32_t  RESERVED2[6];
  __IO uint32_t  PDR0;                      /*!< Port 0 Pad Driver Mode 0 Register     */
  __IO uint32_t  PDR1;                      /*!< Port 0 Pad Driver Mode 1 Register     */
  __I  uint32_t  RESERVED3[6];
  __I  uint32_t  PDISC;                     /*!< Port 0 Pin Function Decision Control Register */
  __I  uint32_t  RESERVED4[3];
  __IO uint32_t  PPS;                       /*!< Port 0 Pin Power Save Register        */
  __IO uint32_t  HWSEL;                     /*!< Port 0 Pin Hardware Select Register   */
} PORT0_Type;


// ------------------------------------------------------------------------------------------------
// -----                                         PORT1                                        -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Port 1 (PORT1)
  */

typedef struct {                            /*!< PORT1 Structure                       */
  __IO uint32_t  OUT;                       /*!< Port 1 Output Register                */
  __O  uint32_t  OMR;                       /*!< Port 1 Output Modification Register   */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  IOCR0;                     /*!< Port 1 Input/Output Control Register 0 */
  __IO uint32_t  IOCR4;                     /*!< Port 1 Input/Output Control Register 4 */
  __IO uint32_t  IOCR8;                     /*!< Port 1 Input/Output Control Register 8 */
  __IO uint32_t  IOCR12;                    /*!< Port 1 Input/Output Control Register 12 */
  __I  uint32_t  RESERVED1;
  __I  uint32_t  IN;                        /*!< Port 1 Input Register                 */
  __I  uint32_t  RESERVED2[6];
  __IO uint32_t  PDR0;                      /*!< Port 1 Pad Driver Mode 0 Register     */
  __IO uint32_t  PDR1;                      /*!< Port 1 Pad Driver Mode 1 Register     */
  __I  uint32_t  RESERVED3[6];
  __I  uint32_t  PDISC;                     /*!< Port 1 Pin Function Decision Control Register */
  __I  uint32_t  RESERVED4[3];
  __IO uint32_t  PPS;                       /*!< Port 1 Pin Power Save Register        */
  __IO uint32_t  HWSEL;                     /*!< Port 1 Pin Hardware Select Register   */
} PORT1_Type;


// ------------------------------------------------------------------------------------------------
// -----                                         PORT2                                        -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Port 2 (PORT2)
  */

typedef struct {                            /*!< PORT2 Structure                       */
  __IO uint32_t  OUT;                       /*!< Port 2 Output Register                */
  __O  uint32_t  OMR;                       /*!< Port 2 Output Modification Register   */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  IOCR0;                     /*!< Port 2 Input/Output Control Register 0 */
  __IO uint32_t  IOCR4;                     /*!< Port 2 Input/Output Control Register 4 */
  __IO uint32_t  IOCR8;                     /*!< Port 2 Input/Output Control Register 8 */
  __IO uint32_t  IOCR12;                    /*!< Port 2 Input/Output Control Register 12 */
  __I  uint32_t  RESERVED1;
  __I  uint32_t  IN;                        /*!< Port 2 Input Register                 */
  __I  uint32_t  RESERVED2[6];
  __IO uint32_t  PDR0;                      /*!< Port 2 Pad Driver Mode 0 Register     */
  __IO uint32_t  PDR1;                      /*!< Port 2 Pad Driver Mode 1 Register     */
  __I  uint32_t  RESERVED3[6];
  __I  uint32_t  PDISC;                     /*!< Port 2 Pin Function Decision Control Register */
  __I  uint32_t  RESERVED4[3];
  __IO uint32_t  PPS;                       /*!< Port 2 Pin Power Save Register        */
  __IO uint32_t  HWSEL;                     /*!< Port 2 Pin Hardware Select Register   */
} PORT2_Type;


// ------------------------------------------------------------------------------------------------
// -----                                         PORT3                                        -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Port 3 (PORT3)
  */

typedef struct {                            /*!< PORT3 Structure                       */
  __IO uint32_t  OUT;                       /*!< Port 3 Output Register                */
  __O  uint32_t  OMR;                       /*!< Port 3 Output Modification Register   */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  IOCR0;                     /*!< Port 3 Input/Output Control Register 0 */
  __IO uint32_t  IOCR4;                     /*!< Port 3 Input/Output Control Register 4 */
  __IO uint32_t  IOCR8;                     /*!< Port 3 Input/Output Control Register 8 */
  __IO uint32_t  IOCR12;                    /*!< Port 3 Input/Output Control Register 12 */
  __I  uint32_t  RESERVED1;
  __I  uint32_t  IN;                        /*!< Port 3 Input Register                 */
  __I  uint32_t  RESERVED2[6];
  __IO uint32_t  PDR0;                      /*!< Port 3 Pad Driver Mode 0 Register     */
  __IO uint32_t  PDR1;                      /*!< Port 3 Pad Driver Mode 1 Register     */
  __I  uint32_t  RESERVED3[6];
  __I  uint32_t  PDISC;                     /*!< Port 3 Pin Function Decision Control Register */
  __I  uint32_t  RESERVED4[3];
  __IO uint32_t  PPS;                       /*!< Port 3 Pin Power Save Register        */
  __IO uint32_t  HWSEL;                     /*!< Port 3 Pin Hardware Select Register   */
} PORT3_Type;


// ------------------------------------------------------------------------------------------------
// -----                                         PORT4                                        -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Port 4 (PORT4)
  */

typedef struct {                            /*!< PORT4 Structure                       */
  __IO uint32_t  OUT;                       /*!< Port 4 Output Register                */
  __O  uint32_t  OMR;                       /*!< Port 4 Output Modification Register   */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  IOCR0;                     /*!< Port 4 Input/Output Control Register 0 */
  __IO uint32_t  IOCR4;                     /*!< Port 4 Input/Output Control Register 4 */
  __I  uint32_t  RESERVED1[3];
  __I  uint32_t  IN;                        /*!< Port 4 Input Register                 */
  __I  uint32_t  RESERVED2[6];
  __IO uint32_t  PDR0;                      /*!< Port 4 Pad Driver Mode 0 Register     */
  __I  uint32_t  RESERVED3[7];
  __I  uint32_t  PDISC;                     /*!< Port 4 Pin Function Decision Control Register */
  __I  uint32_t  RESERVED4[3];
  __IO uint32_t  PPS;                       /*!< Port 4 Pin Power Save Register        */
  __IO uint32_t  HWSEL;                     /*!< Port 4 Pin Hardware Select Register   */
} PORT4_Type;


// ------------------------------------------------------------------------------------------------
// -----                                         PORT5                                        -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Port 5 (PORT5)
  */

typedef struct {                            /*!< PORT5 Structure                       */
  __IO uint32_t  OUT;                       /*!< Port 5 Output Register                */
  __O  uint32_t  OMR;                       /*!< Port 5 Output Modification Register   */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  IOCR0;                     /*!< Port 5 Input/Output Control Register 0 */
  __IO uint32_t  IOCR4;                     /*!< Port 5 Input/Output Control Register 4 */
  __IO uint32_t  IOCR8;                     /*!< Port 5 Input/Output Control Register 8 */
  __I  uint32_t  RESERVED1[2];
  __I  uint32_t  IN;                        /*!< Port 5 Input Register                 */
  __I  uint32_t  RESERVED2[6];
  __IO uint32_t  PDR0;                      /*!< Port 5 Pad Driver Mode 0 Register     */
  __IO uint32_t  PDR1;                      /*!< Port 5 Pad Driver Mode 1 Register     */
  __I  uint32_t  RESERVED3[6];
  __I  uint32_t  PDISC;                     /*!< Port 5 Pin Function Decision Control Register */
  __I  uint32_t  RESERVED4[3];
  __IO uint32_t  PPS;                       /*!< Port 5 Pin Power Save Register        */
  __IO uint32_t  HWSEL;                     /*!< Port 5 Pin Hardware Select Register   */
} PORT5_Type;


// ------------------------------------------------------------------------------------------------
// -----                                         PORT6                                        -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Port 6 (PORT6)
  */

typedef struct {                            /*!< PORT6 Structure                       */
  __IO uint32_t  OUT;                       /*!< Port 6 Output Register                */
  __O  uint32_t  OMR;                       /*!< Port 6 Output Modification Register   */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  IOCR0;                     /*!< Port 6 Input/Output Control Register 0 */
  __IO uint32_t  IOCR4;                     /*!< Port 6 Input/Output Control Register 4 */
  __I  uint32_t  RESERVED1[3];
  __I  uint32_t  IN;                        /*!< Port 6 Input Register                 */
  __I  uint32_t  RESERVED2[6];
  __IO uint32_t  PDR0;                      /*!< Port 6 Pad Driver Mode 0 Register     */
  __I  uint32_t  RESERVED3[7];
  __I  uint32_t  PDISC;                     /*!< Port 6 Pin Function Decision Control Register */
  __I  uint32_t  RESERVED4[3];
  __IO uint32_t  PPS;                       /*!< Port 6 Pin Power Save Register        */
  __IO uint32_t  HWSEL;                     /*!< Port 6 Pin Hardware Select Register   */
} PORT6_Type;


// ------------------------------------------------------------------------------------------------
// -----                                        PORT14                                        -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Port 14 (PORT14)
  */

typedef struct {                            /*!< PORT14 Structure                      */
  __IO uint32_t  OUT;                       /*!< Port 14 Output Register               */
  __O  uint32_t  OMR;                       /*!< Port 14 Output Modification Register  */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  IOCR0;                     /*!< Port 14 Input/Output Control Register 0 */
  __IO uint32_t  IOCR4;                     /*!< Port 14 Input/Output Control Register 4 */
  __IO uint32_t  IOCR8;                     /*!< Port 14 Input/Output Control Register 8 */
  __IO uint32_t  IOCR12;                    /*!< Port 14 Input/Output Control Register 12 */
  __I  uint32_t  RESERVED1;
  __I  uint32_t  IN;                        /*!< Port 14 Input Register                */
  __I  uint32_t  RESERVED2[14];
  __IO uint32_t  PDISC;                     /*!< Port 14 Pin Function Decision Control Register */
  __I  uint32_t  RESERVED3[3];
  __IO uint32_t  PPS;                       /*!< Port 14 Pin Power Save Register       */
  __IO uint32_t  HWSEL;                     /*!< Port 14 Pin Hardware Select Register  */
} PORT14_Type;


// ------------------------------------------------------------------------------------------------
// -----                                        PORT15                                        -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Port 15 (PORT15)
  */

typedef struct {                            /*!< PORT15 Structure                      */
  __IO uint32_t  OUT;                       /*!< Port 15 Output Register               */
  __O  uint32_t  OMR;                       /*!< Port 15 Output Modification Register  */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  IOCR0;                     /*!< Port 15 Input/Output Control Register 0 */
  __IO uint32_t  IOCR4;                     /*!< Port 15 Input/Output Control Register 4 */
  __IO uint32_t  IOCR8;                     /*!< Port 15 Input/Output Control Register 8 */
  __IO uint32_t  IOCR12;                    /*!< Port 15 Input/Output Control Register 12 */
  __I  uint32_t  RESERVED1;
  __I  uint32_t  IN;                        /*!< Port 15 Input Register                */
  __I  uint32_t  RESERVED2[14];
  __IO uint32_t  PDISC;                     /*!< Port 15 Pin Function Decision Control Register */
  __I  uint32_t  RESERVED3[3];
  __IO uint32_t  PPS;                       /*!< Port 15 Pin Power Save Register       */
  __IO uint32_t  HWSEL;                     /*!< Port 15 Pin Hardware Select Register  */
} PORT15_Type;

#define PORT0_BASE                0x48028000
#define PORT1_BASE                0x48028100
#define PORT2_BASE                0x48028200
#define PORT3_BASE                0x48028300
#define PORT4_BASE                0x48028400
#define PORT5_BASE                0x48028500
#define PORT6_BASE                0x48028600
#define PORT14_BASE               0x48028E00
#define PORT15_BASE               0x48028F00

#define PORT0                     ((PORT0_Type              *) PORT0_BASE)
#define PORT1                     ((PORT1_Type              *) PORT1_BASE)
#define PORT2                     ((PORT2_Type              *) PORT2_BASE)
#define PORT3                     ((PORT3_Type              *) PORT3_BASE)
#define PORT4                     ((PORT4_Type              *) PORT4_BASE)
#define PORT5                     ((PORT5_Type              *) PORT5_BASE)
#define PORT6                     ((PORT6_Type              *) PORT6_BASE)
#define PORT14                    ((PORT14_Type             *) PORT14_BASE)
#define PORT15                    ((PORT15_Type             *) PORT15_BASE)


/***************************************************************************/
/*                     POSIFx                          */
/***************************************************************************/

  /* 
  *Position Interface
  */
typedef struct {
__IO uint32_t	PCONF; /*!<Architectural Overview configuration Offset  0x0000*/
__IO uint32_t	PSUS; /*!<Architectural Overview Suspend Config Offset  0x0004*/
__O uint32_t	PRUNS; /*!<Architectural Overview Run Bit Set Offset  0x0008*/
__O uint32_t	PRUNC; /*!<Architectural Overview Run Bit Clear Offset  0x000C*/
__I uint32_t	PRUN; /*!<Architectural Overview Run Bit Status Offset  0x0010*/
__I  uint32_t	RESERVED1[3];
__I uint32_t	MIDR; /*!<Module Identification register Offset  0x0020*/
__I  uint32_t	RESERVED2[3];
__I uint32_t	HALP; /*!<Hall Sensor Patterns Offset  0x0030*/
__IO uint32_t	HALPS; /*!<Hall Sensor Shadow Patterns Offset  0x0034*/
__I  uint32_t	RESERVED3[2];
__I uint32_t	MCM; /*!<Multi Channel Pattern Offset  0x0040*/
__IO uint32_t	MCSM; /*!<Multi Channel Shadow Pattern Offset  0x0044*/
__O uint32_t	MCMS; /*!<Multi Channel Pattern Control set Offset  0x0048*/
__O uint32_t	MCMC; /*!<Multi Channel Pattern Control clear Offset  0x004C*/
__I uint32_t	MCMF; /*!<Multi Channel Pattern Control flag Offset  0x0050*/
__I  uint32_t	RESERVED4[3];
__IO uint32_t	QDC; /*!<Quadrature Decoder Control Offset  0x0060*/
__I  uint32_t	RESERVED5[3];
__I uint32_t	PFLG; /*!<Architectural Overview Interrupt Flags Offset  0x0070*/
__IO uint32_t	PFLGE; /*!<Architectural Overview Interrupt Enable Offset  0x0074*/
__O uint32_t	SPFLG; /*!<Architectural Overview Interrupt Set Offset  0x0078*/
__O uint32_t	RPFLG; /*!<Architectural Overview Interrupt Clear Offset  0x007C*/
}POSIF_GLOBAL_TypeDef;

#define   POSIF0_BASE			 ((uint32_t)0x40028000U)
#define   POSIF1_BASE			 ((uint32_t)0x4002C000U)
#define   POSIF_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   POSIF0		 ((POSIF_GLOBAL_TypeDef*)(POSIF0_BASE + POSIF_GLOBAL_BASE))
#define   POSIF1		 ((POSIF_GLOBAL_TypeDef*)(POSIF1_BASE + POSIF_GLOBAL_BASE))

// ------------------------------------------------------------------------------------------------
// -----                                          PPB                                         -----
// ------------------------------------------------------------------------------------------------


/**
  * @brief Cortex-M4 Private Peripheral Block (PPB)
  */

typedef struct {                            /*!< (@ 0xE000E000) PPB Structure          */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  ACTLR;                     /*!< (@ 0xE000E008) Auxiliary Control Register */
  __I  uint32_t  RESERVED1;
  __IO uint32_t  SYST_CSR;                  /*!< (@ 0xE000E010) SysTick Control and Status Register */
  __IO uint32_t  SYST_RVR;                  /*!< (@ 0xE000E014) SysTick Reload Value Register */
  __IO uint32_t  SYST_CVR;                  /*!< (@ 0xE000E018) SysTick Current Value Register */
  __IO uint32_t  SYST_CALIB;                /*!< (@ 0xE000E01C) SysTick Calibration Value Register r */
  __I  uint32_t  RESERVED2[56];
  __IO uint32_t  NVIC_ISER0;                /*!< (@ 0xE000E100) Interrupt Set-enable Register 0 */
  __IO uint32_t  NVIC_ISER1;                /*!< (@ 0xE000E104) Interrupt Set-enable Register 1 */
  __IO uint32_t  NVIC_ISER2;                /*!< (@ 0xE000E108) Interrupt Set-enable Register 2 */
  __IO uint32_t  NVIC_ISER3;                /*!< (@ 0xE000E10C) Interrupt Set-enable Register 3 */
  __I  uint32_t  RESERVED3[28];
  __IO uint32_t  NVIC_ISCER0;               /*!< (@ 0xE000E180) Interrupt Clear-enable Register 0 */
  __IO uint32_t  NVIC_ISCER1;               /*!< (@ 0xE000E184) Interrupt Clear-enable Register 1 */
  __IO uint32_t  NVIC_ISCER2;               /*!< (@ 0xE000E188) Interrupt Clear-enable Register 2 */
  __IO uint32_t  NVIC_ISCER3;               /*!< (@ 0xE000E18C) Interrupt Clear-enable Register 3 */
  __I  uint32_t  RESERVED4[28];
  __IO uint32_t  NVIC_ISSPR0;               /*!< (@ 0xE000E200) Interrupt Set-pending Register 0 */
  __IO uint32_t  NVIC_ISSPR1;               /*!< (@ 0xE000E204) Interrupt Set-pending Register 1 */
  __IO uint32_t  NVIC_ISSPR2;               /*!< (@ 0xE000E208) Interrupt Set-pending Register 2 */
  __IO uint32_t  NVIC_ISSPR3;               /*!< (@ 0xE000E20C) Interrupt Set-pending Register 3 */
  __I  uint32_t  RESERVED5[28];
  __IO uint32_t  NVIC_ICPR0;                /*!< (@ 0xE000E280) Interrupt Clear-pending Register 0 */
  __IO uint32_t  NVIC_ICPR1;                /*!< (@ 0xE000E284) Interrupt Clear-pending Register 1 */
  __IO uint32_t  NVIC_ICPR2;                /*!< (@ 0xE000E288) Interrupt Clear-pending Register 2 */
  __IO uint32_t  NVIC_ICPR3;                /*!< (@ 0xE000E28C) Interrupt Clear-pending Register 3 */
  __I  uint32_t  RESERVED6[28];
  __IO uint32_t  NVIC_IABR0;                /*!< (@ 0xE000E300) Interrupt Active Bit Register 0 */
  __IO uint32_t  NVIC_IABR1;                /*!< (@ 0xE000E304) Interrupt Active Bit Register 1 */
  __IO uint32_t  NVIC_IABR2;                /*!< (@ 0xE000E308) Interrupt Active Bit Register 2 */
  __IO uint32_t  NVIC_IABR3;                /*!< (@ 0xE000E30C) Interrupt Active Bit Register 3 */
  __I  uint32_t  RESERVED7[60];
  __IO uint32_t  NVIC_IPR0;                 /*!< (@ 0xE000E400) Interrupt Priority Register 0 */
  __IO uint32_t  NVIC_IPR1;                 /*!< (@ 0xE000E404) Interrupt Priority Register 1 */
  __IO uint32_t  NVIC_IPR2;                 /*!< (@ 0xE000E408) Interrupt Priority Register 2 */
  __IO uint32_t  NVIC_IPR3;                 /*!< (@ 0xE000E40C) Interrupt Priority Register 3 */
  __IO uint32_t  NVIC_IPR4;                 /*!< (@ 0xE000E410) Interrupt Priority Register 4 */
  __IO uint32_t  NVIC_IPR5;                 /*!< (@ 0xE000E414) Interrupt Priority Register 5 */
  __IO uint32_t  NVIC_IPR6;                 /*!< (@ 0xE000E418) Interrupt Priority Register 6 */
  __IO uint32_t  NVIC_IPR7;                 /*!< (@ 0xE000E41C) Interrupt Priority Register 7 */
  __IO uint32_t  NVIC_IPR8;                 /*!< (@ 0xE000E420) Interrupt Priority Register 8 */
  __IO uint32_t  NVIC_IPR9;                 /*!< (@ 0xE000E424) Interrupt Priority Register 9 */
  __IO uint32_t  NVIC_IPR10;                /*!< (@ 0xE000E428) Interrupt Priority Register 10 */
  __IO uint32_t  NVIC_IPR11;                /*!< (@ 0xE000E42C) Interrupt Priority Register 11 */
  __IO uint32_t  NVIC_IPR12;                /*!< (@ 0xE000E430) Interrupt Priority Register 12 */
  __IO uint32_t  NVIC_IPR13;                /*!< (@ 0xE000E434) Interrupt Priority Register 13 */
  __IO uint32_t  NVIC_IPR14;                /*!< (@ 0xE000E438) Interrupt Priority Register 14 */
  __IO uint32_t  NVIC_IPR15;                /*!< (@ 0xE000E43C) Interrupt Priority Register 15 */
  __IO uint32_t  NVIC_IPR16;                /*!< (@ 0xE000E440) Interrupt Priority Register 16 */
  __IO uint32_t  NVIC_IPR17;                /*!< (@ 0xE000E444) Interrupt Priority Register 17 */
  __IO uint32_t  NVIC_IPR18;                /*!< (@ 0xE000E448) Interrupt Priority Register 18 */
  __IO uint32_t  NVIC_IPR19;                /*!< (@ 0xE000E44C) Interrupt Priority Register 19 */
  __IO uint32_t  NVIC_IPR20;                /*!< (@ 0xE000E450) Interrupt Priority Register 20 */
  __IO uint32_t  NVIC_IPR21;                /*!< (@ 0xE000E454) Interrupt Priority Register 21 */
  __IO uint32_t  NVIC_IPR22;                /*!< (@ 0xE000E458) Interrupt Priority Register 22 */
  __IO uint32_t  NVIC_IPR23;                /*!< (@ 0xE000E45C) Interrupt Priority Register 23 */
  __IO uint32_t  NVIC_IPR24;                /*!< (@ 0xE000E460) Interrupt Priority Register 24 */
  __IO uint32_t  NVIC_IPR25;                /*!< (@ 0xE000E464) Interrupt Priority Register 25 */
  __IO uint32_t  NVIC_IPR26;                /*!< (@ 0xE000E468) Interrupt Priority Register 26 */
  __IO uint32_t  NVIC_IPR27;                /*!< (@ 0xE000E46C) Interrupt Priority Register 27 */
  __I  uint32_t  RESERVED8[548];
  __I  uint32_t  CPUID;                     /*!< (@ 0xE000ED00) CPUID Base Register    */
  __IO uint32_t  ICSR;                      /*!< (@ 0xE000ED04) Interrupt Control and State Register */
  __IO uint32_t  VTOR;                      /*!< (@ 0xE000ED08) Vector Table Offset Register */
  __IO uint32_t  AIRCR;                     /*!< (@ 0xE000ED0C) Application Interrupt and Reset Control Register */
  __IO uint32_t  SCR;                       /*!< (@ 0xE000ED10) System Control Register */
  __IO uint32_t  CCR;                       /*!< (@ 0xE000ED14) Configuration and Control Register */
  __IO uint32_t  SHPR1;                     /*!< (@ 0xE000ED18) System Handler Priority Register 1 */
  __IO uint32_t  SHPR2;                     /*!< (@ 0xE000ED1C) System Handler Priority Register 2 */
  __IO uint32_t  SHPR3;                     /*!< (@ 0xE000ED20) System Handler Priority Register 3 */
  __IO uint32_t  SHCSR;                     /*!< (@ 0xE000ED24) System Handler Control and State Register */
  __IO uint32_t  CFSR;                      /*!< (@ 0xE000ED28) Configurable Fault Status Register */
  __IO uint32_t  HFSR;                      /*!< (@ 0xE000ED2C) HardFault Status Register */
  __I  uint32_t  RESERVED9;
  __IO uint32_t  MMFAR;                     /*!< (@ 0xE000ED34) MemManage Fault Address Register */
  __IO uint32_t  BFAR;                      /*!< (@ 0xE000ED38) BusFault Address Register */
  __IO uint32_t  AFSR;                      /*!< (@ 0xE000ED3C) Auxiliary Fault Status Register */
  __I  uint32_t  RESERVED10[18];
  __IO uint32_t  CPACR;                     /*!< (@ 0xE000ED88) Coprocessor Access Control Register */
  __I  uint32_t  RESERVED11;
  __I  uint32_t  MPU_TYPE;                  /*!< (@ 0xE000ED90) MPU Type Register      */
  __IO uint32_t  MPU_CTRL;                  /*!< (@ 0xE000ED94) MPU Control Register   */
  __IO uint32_t  MPU_RNR;                   /*!< (@ 0xE000ED98) MPU Region Number Register */
  __IO uint32_t  MPU_RBAR;                  /*!< (@ 0xE000ED9C) MPU Region Base Address Register */
  __IO uint32_t  MPU_RASR;                  /*!< (@ 0xE000EDA0) MPU Region Attribute and Size Register */
  __IO uint32_t  MPU_RBAR_A1;               /*!< (@ 0xE000EDA4) MPU Region Base Address Register A1 */
  __IO uint32_t  MPU_RASR_A1;               /*!< (@ 0xE000EDA8) MPU Region Attribute and Size Register A1 */
  __IO uint32_t  MPU_RBAR_A2;               /*!< (@ 0xE000EDAC) MPU Region Base Address Register A2 */
  __IO uint32_t  MPU_RASR_A2;               /*!< (@ 0xE000EDB0) MPU Region Attribute and Size Register A2 */
  __IO uint32_t  MPU_RBAR_A3;               /*!< (@ 0xE000EDB4) MPU Region Base Address Register A3 */
  __IO uint32_t  MPU_RASR_A3;               /*!< (@ 0xE000EDB8) MPU Region Attribute and Size Register A3 */
  __I  uint32_t  RESERVED12[81];
  __O  uint32_t  STIR;                      /*!< (@ 0xE000EF00) Software Trigger Interrupt Register */
  __I  uint32_t  RESERVED13[12];
  __IO uint32_t  FPCCR;                     /*!< (@ 0xE000EF34) Floating-point Context Control Register */
  __IO uint32_t  FPCAR;                     /*!< (@ 0xE000EF38) Floating-point Context Address Register */
  __IO uint32_t  FPDSCR;                    /*!< (@ 0xE000EF3C) Floating-point Default Status Control Register */
} PPB_Type;

#define PPB_BASE                  0xE000E000
#define PPB                       ((PPB_Type                *) PPB_BASE)


/***************************************************************************/
/*                     PREF                          */
/***************************************************************************/

  /* 
  *Prefetch
  */
typedef struct {
__IO uint32_t	PCON; /*!<Prefetch Configuration Register Offset  0x0000*/
}PREF_GLOBAL_TypeDef;

#define   PREF_BASE			 ((uint32_t)0x58004000U)
#define   PREF_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   PREF		 ((PREF_GLOBAL_TypeDef*)(PREF_BASE + PREF_GLOBAL_BASE))

/***************************************************************************/
/*                     RTC                          */
/***************************************************************************/

  /* 
  *RTC Kernel registers
  */
typedef struct {
__I uint32_t	ID; /*!<RTC ID Register Offset  0x0000*/
__IO uint32_t	CTR; /*!<RTC Control Register Offset  0x0004*/
__I uint32_t	RAWSTAT; /*!<RTC Raw Service Request Register Offset  0x0008*/
__I uint32_t	STSSR; /*!<RTC Service Request Status Register Offset  0x000C*/
__IO uint32_t	MSKSR; /*!<RTC Service Request Mask Register Offset  0x0010*/
__O uint32_t	CLRSR; /*!<RTC Clear Service Request Register Offset  0x0014*/
__IO uint32_t	ATIM0; /*!<RTC Alarm Time Register 0 Offset  0x0018*/
__IO uint32_t	ATIM1; /*!<RTC Alarm Time Register 1 Offset  0x001C*/
__IO uint32_t	TIM0; /*!<RTC Time Register 0 Offset  0x0020*/
__IO uint32_t	TIM1; /*!<RTC Time Register 1 Offset  0x0024*/
}RTC_GLOBAL_TypeDef;

#define   RTC_BASE			 ((uint32_t)0x50004A00U)
#define   RTC_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   RTC		 ((RTC_GLOBAL_TypeDef*)(RTC_BASE + RTC_GLOBAL_BASE))

/***************************************************************************/
/*                     SCU                          */
/***************************************************************************/

  /* 
  *SCU CCU PLL registers
  */
typedef struct {
__I uint32_t	PLLSTAT; /*!<PLL Status Register Offset  0x0710*/
__IO uint32_t	PLLCON0; /*!<PLL Configuration 0 Register Offset  0x0714*/
__IO uint32_t	PLLCON1; /*!<PLL Configuration 1 Register Offset  0x0718*/
__IO uint32_t	PLLCON2; /*!<PLL Configuration 2 Register Offset  0x071C*/
__I uint32_t	USBPLLSTAT; /*!<USB PLL Status Register Offset  0x0720*/
__IO uint32_t	USBPLLCON; /*!<USB PLL Configuration Register Offset  0x0724*/
__I  uint32_t	RESERVED1[4];
__I uint32_t	CLKMXSTAT; /*!<Clock Multiplexing Status Register Offset  0x0738*/
}SCU_PLL_TypeDef;

  /* 
  *SCU CCU OSC registers
  */
typedef struct {
__I uint32_t	OSCHPSTAT; /*!<OSC_HP Status Register Offset  0x0700*/
__IO uint32_t	OSCHPCTRL; /*!<OSC_HP Control Register Offset  0x0704*/
}SCU_OSC_TypeDef;

  /* 
  *SCU CCU Clock Control registers
  */
typedef struct {
__I uint32_t	CLKSTAT; /*!<Clock Status Register Offset  0x0600*/
__O uint32_t	CLKSET; /*!<CLK Set Register Offset  0x0604*/
__O uint32_t	CLKCLR; /*!<CLK Clear Register Offset  0x0608*/
__IO uint32_t	SYSCLKCR; /*!<System Clock Control Register Offset  0x060C*/
__IO uint32_t	CPUCLKCR; /*!<CPU Clock Control Register Offset  0x0610*/
__IO uint32_t	PBCLKCR; /*!<Peripheral Bus Clock Control Register Offset  0x0614*/
__IO uint32_t	USBCLKCR; /*!<USB Clock Control Register Offset  0x0618*/
__IO uint32_t	EBUCLKCR; /*!<EBU Clock Control Register Offset  0x061C*/
__IO uint32_t	CCUCLKCR; /*!<CCU Clock Control Register Offset  0x0620*/
__IO uint32_t	WDTCLKCR; /*!<WDT Clock Control Register Offset  0x0624*/
__IO uint32_t	EXTCLKCR; /*!<External Clock Control Offset  0x0628*/
__I  uint32_t	RESERVED1;
__IO uint32_t	SLEEPCR; /*!<Sleep Control Register Offset  0x0630*/
__IO uint32_t	DSLEEPCR; /*!<Deep Sleep Control Register Offset  0x0634*/
}SCU_CLK_TypeDef;

#define   SCU_BASE			 ((uint32_t)0x50004000U)
#define   SCU_PLL_BASE		 (0x0710U)
#define   SCU_OSC_BASE		 (0x0700U)
#define   SCU_CLK_BASE		 (0x0600U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   SCU_PLL		 ((SCU_PLL_TypeDef*)(SCU_BASE + SCU_PLL_BASE))
#define   SCU_OSC		 ((SCU_OSC_TypeDef*)(SCU_BASE + SCU_OSC_BASE))
#define   SCU_CLK		 ((SCU_CLK_TypeDef*)(SCU_BASE + SCU_CLK_BASE))

/***************************************************************************/
/*                     SCU                          */
/***************************************************************************/

  /* 
  *SCU GCU CONTROL registers
  */
typedef struct {
__I uint32_t	ID; /*!<SCU Module ID Register Offset  0x0000*/
__I uint32_t	IDCHIP; /*!<Chip ID Register Offset  0x0004*/
__I uint32_t	IDMANUF; /*!<Manufactory ID Register Offset  0x0008*/
__I  uint32_t	RESERVED1;
__IO uint32_t	STCON; /*!<Startup Configuration Register Offset  0x0010*/
__I  uint32_t	RESERVED2[6];
__IO uint32_t GPR[2];  /*!< General Purpose Register 0  Offset  0x002C*/
__I  uint32_t	RESERVED3[6];
__IO uint32_t	CCUCON; /*!<CCU Control Register Offset  0x004C*/
__I  uint32_t	RESERVED4[15];
__IO uint32_t	DTSCON; /*!<Die Temperature Sensor Control Register Offset  0x008C*/
__I uint32_t	DTSSTAT; /*!<Die Temperature Sensor Status Register Offset  0x0090*/
__I  uint32_t	RESERVED5[2];
__IO uint32_t	SDMMCDEL; /*!<SD-MMC Delay Control Register Offset  0x009C*/
__IO uint32_t	GORCEN[2]; /*!<Out of Range Comparator Enable Register 0 Offset  0x00A0*/
__I  uint32_t	RESERVED6[7];
__I uint32_t	MIRRSTS; /*!<Mirror Update Status Register Offset  0x00C4*/
__IO uint32_t	RMACR; /*!<Retention Memory Access Control Register Offset  0x00C8*/
__IO uint32_t	RMDATA; /*!<Retention Memory Access Data Register Offset  0x00CC*/
}SCU_GENERAL_TypeDef;

  /* 
  *SCU GCU PARITY registers
  */
typedef struct {
__IO uint32_t	PEEN; /*!<Parity Error Enable Register Offset  0x013C*/
__IO uint32_t	MCHKCON; /*!<Memory Checking Control Register Offset  0x0140*/
__IO uint32_t	PETE; /*!<Parity Error Trap Enable Register Offset  0x0144*/
__IO uint32_t	PERSTEN; /*!<Parity Error Reset Enable Register Offset  0x0148*/
__I  uint32_t	RESERVED1;
__IO uint32_t	PEFLAG; /*!<Parity Error Flag Register Offset  0x0150*/
__IO uint32_t	PMTPR; /*!<Parity Memory Test Pattern Register Offset  0x0154*/
__IO uint32_t	PMTSR; /*!<Parity Memory Test Select Register Offset  0x0158*/
}SCU_PARITY_TypeDef;

  /* 
  *SCU GCU INTERRUPT registers
  */
typedef struct {
__I uint32_t	SRSTAT; /*!<SCU Service Request Status Offset  0x0074*/
__I uint32_t	SRRAW; /*!<SCU Raw Service Request Status Offset  0x0078*/
__IO uint32_t	SRMSK; /*!<SCU Service Request Mask Offset  0x007C*/
__O uint32_t	SRCLR; /*!<SCU Service Request Clear Offset  0x0080*/
__O uint32_t	SRSET; /*!<SCU Service Request Set Offset  0x0084*/
__IO uint32_t	NMIREQEN; /*!<SCU Service Request Mask Offset  0x0088*/
}SCU_INTERRUPT_TypeDef;

  /* 
  *SCU TRAP registers
  */
typedef struct {
__I uint32_t	TRAPSTAT; /*!<Trap Status Register Offset  0x0160*/
__I uint32_t	TRAPRAW; /*!<Trap Raw Status Register Offset  0x0164*/
__IO uint32_t	TRAPDIS; /*!<Trap Disable Register Offset  0x0168*/
__O uint32_t	TRAPCLR; /*!<Trap Clear Register Offset  0x016C*/
__O uint32_t	TRAPSET; /*!<Trap Set Register Offset  0x0170*/
}SCU_TRAP_TypeDef;

#define   SCU_BASE			 ((uint32_t)0x50004000U)
#define   SCU_GENERAL_BASE		 (0x0000U)
#define   SCU_PARITY_BASE		 (0x013CU)
#define   SCU_INTERRUPT_BASE		 (0x0074U)
#define   SCU_TRAP_BASE		 (0x0160U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   SCU_GENERAL		 ((SCU_GENERAL_TypeDef*)(SCU_BASE + SCU_GENERAL_BASE))
#define   SCU_PARITY		 ((SCU_PARITY_TypeDef*)(SCU_BASE + SCU_PARITY_BASE))
#define   SCU_INTERRUPT		 ((SCU_INTERRUPT_TypeDef*)(SCU_BASE + SCU_INTERRUPT_BASE))
#define   SCU_TRAP		 ((SCU_TRAP_TypeDef*)(SCU_BASE + SCU_TRAP_BASE))

/***************************************************************************/
/*                     SCU                          */
/***************************************************************************/

  /* 
  *SCU HIBERNATE registers
  */
typedef struct {
__I uint32_t	HDSTAT; /*!<Hibernate Domain Control & Status Register Offset  0x0300*/
__O uint32_t	HDCLR; /*!<Hibernate Domain Status Clear Register Offset  0x0304*/
__O uint32_t	HDSET; /*!<Hibernate Domain Status Set Register Offset  0x0308*/
__IO uint32_t	HDCR; /*!<Hibernate Domain Configuration Register Offset  0x030C*/
__I  uint32_t	RESERVED1;
__IO uint32_t	OSCSICTRL; /*!<OSC_SI Control Register Offset  0x0314*/
__I uint32_t	OSCULSTAT; /*!<OSC_ULP Status Register Offset  0x0318*/
__IO uint32_t	OSCULCTRL; /*!<OSC_ULP Control Register Offset  0x031C*/
}SCU_HIBERNATE_TypeDef;

#define   SCU_BASE			 ((uint32_t)0x50004000U)
#define   SCU_HIBERNATE_BASE		 (0x0300U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   SCU_HIBERNATE		 ((SCU_HIBERNATE_TypeDef*)(SCU_BASE + SCU_HIBERNATE_BASE))

/***************************************************************************/
/*                     SCU                          */
/***************************************************************************/

  /* 
  *SCU PWRCTRL registers
  */
typedef struct {
__I  uint32_t	PWRSTAT; /*!<PCU Status Register Offset  0x0200*/
__O  uint32_t	PWRSET; /*!<PCU Set Control Register Offset  0x0204*/
__O  uint32_t	PWRCLR; /*!<PCU Clear Control Register Offset  0x0208*/
__I  uint32_t	RESERVED1;
__I  uint32_t	EVRSTAT; /*!<EVR Status Register Offset  0x0210*/
__I  uint32_t	EVRVADCSTAT; /*!<EVR VADC Status Register Offset  0x0214*/
__I  uint32_t RESERVED2[5];
__IO uint32_t PWRMON; /*!< Power Monitor Control  Offset  0x022C*/
}SCU_POWER_TypeDef;

#define   SCU_BASE			 ((uint32_t)0x50004000U)
#define   SCU_POWER_BASE		 (0x0200U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   SCU_POWER		 ((SCU_POWER_TypeDef*)(SCU_BASE + SCU_POWER_BASE))

/***************************************************************************/
/*                     SCU                          */
/***************************************************************************/

  /* 
  *SCU RESET CONTROL registers
  */
typedef struct {
__I uint32_t	RSTSTAT; /*!<RCU Reset Status Offset  0x0400*/
__O uint32_t	RSTSET; /*!<RCU Reset Set Register Offset  0x0404*/
__O uint32_t	RSTCLR; /*!<RCU Reset Clear Register Offset  0x0408*/
__I uint32_t	PRSTAT0; /*!<RCU Peripheral 0 Reset Status Offset  0x040C*/
__O uint32_t	PRSET0; /*!<RCU Peripheral 0 Reset Set Offset  0x0410*/
__O uint32_t	PRCLR0; /*!<RCU Peripheral 0 Reset Clear Offset  0x0414*/
__I uint32_t	PRSTAT1; /*!<RCU Peripheral 1 Reset Status Offset  0x0418*/
__O uint32_t	PRSET1; /*!<RCU Peripheral 1 Reset Set Offset  0x041C*/
__O uint32_t	PRCLR1; /*!<RCU Peripheral 1 Reset Clear Offset  0x0420*/
__I uint32_t	PRSTAT2; /*!<RCU Peripheral 2 Reset Status Offset  0x0424*/
__O uint32_t	PRSET2; /*!<RCU Peripheral 2 Reset Set Offset  0x0428*/
__O uint32_t	PRCLR2; /*!<RCU Peripheral 2 Reset Clear Offset  0x042C*/
__I uint32_t	PRSTAT3; /*!<RCU Peripheral 3 Reset Status Offset  0x0430*/
__O uint32_t	PRSET3; /*!<RCU Peripheral 3 Reset Set Offset  0x0434*/
__O uint32_t	PRCLR3; /*!<RCU Peripheral 3 Reset Clear Offset  0x0438*/
}SCU_RESET_TypeDef;

#define   SCU_BASE			 ((uint32_t)0x50004000U)
#define   SCU_RESET_BASE		 (0x0400U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   SCU_RESET		 ((SCU_RESET_TypeDef*)(SCU_BASE + SCU_RESET_BASE))

/***************************************************************************/
/*                     SDMMC                          */
/***************************************************************************/

  /* 
  *SDMMC registers
  */
typedef struct {
__IO uint16_t	BLOCK_SIZE; /*!<Block Size Register Offset  0x0004*/
__IO uint16_t	BLOCK_COUNT; /*!<Block Count Register Offset  0x0006*/
__IO uint32_t	ARGUMENT1; /*!<Argument1 Register Offset  0x0008*/
__IO uint16_t	TRANSFER_MODE; /*!<Transfer Mode Register Offset  0x000C*/
__IO uint16_t	COMMAND; /*!<Command Register Offset  0x000E*/
__I uint32_t	RESPONSE0; /*!<Response 0 Register Offset  0x0010*/
__I uint32_t	RESPONSE2; /*!<Response 2 Register Offset  0x0014*/
__I uint32_t	RESPONSE4; /*!<Response 4 Register Offset  0x0018*/
__I uint32_t	RESPONSE6; /*!<Response 6 Register Offset  0x001C*/
__IO uint32_t	DATA_BUFFER; /*!<Data Buffer Register Offset  0x0020*/
__I uint32_t	PRESENT_STATE; /*!<Present State Register Offset  0x0024*/
__IO uint8_t	HOST_CTRL; /*!<Host Control Register Offset  0x0028*/
__IO uint8_t	POWER_CTRL; /*!<Power Control Register Offset  0x0029*/
__IO uint8_t	BLOCK_GAP_CTRL; /*!<Block Gap Control Register Offset  0x002A*/
__IO uint8_t	WAKEUP_CTRL; /*!<Wake-up Control Register Offset  0x002B*/
__IO uint16_t	CLOCK_CTRL; /*!<Clock Control Register Offset  0x002C*/
__IO uint8_t	TIMEOUT_CTRL; /*!<Timeout Control Register Offset  0x002E*/
__IO uint8_t	SW_RESET; /*!<Software Reset Register Offset  0x002F*/
__IO uint16_t	INT_STATUS_NORM; /*!<Normal Interrupt Status Register Offset  0x0030*/
__IO uint16_t	INT_STATUS_ERR; /*!<Error Interrupt Status Register Offset  0x0032*/
__IO uint16_t	EN_INT_STATUS_NORM; /*!<Normal Interrupt Status Enable Register Offset  0x0034*/
__IO uint16_t	EN_INT_STATUS_ERR; /*!<Error Interrupt Status Enable Register Offset  0x0036*/
__IO uint16_t	EN_INT_SIGNAL_NORM; /*!<Normal Interrupt Signal Enable Register Offset  0x0038*/
__IO uint16_t	EN_INT_SIGNAL_ERR; /*!<Error Interrupt Signal Enable Register Offset  0x003A*/
__I uint16_t	ACMD_ERR_STATUS; /*!<Auto CMD Error Status Register Offset  0x003C*/
__I  uint16_t	RESERVED0;
__I  uint32_t	RESERVED1[4];
__O uint16_t	FORCE_EVENT_ACMD_ERR_STATUS; /*!<Force Event Register for Auto CMD Error Status Offset  0x0050*/
__O uint16_t	FORCE_EVENT_ERR_STATUS; /*!<Force Event Register for Error Interrupt Status Offset  0x0052*/
__I  uint32_t	RESERVED2;
__I  uint32_t	RESERVED3[7];
__O uint32_t	DEBUG_SEL; /*!<Debug Selection Register Offset  0x0074*/
__I  uint32_t	RESERVED4[30];
__IO uint32_t	SPI; /*!<SPI Interrupt Support Register Offset  0x00F0*/
__I  uint32_t	RESERVED5[2];
__I uint16_t	SLOT_INT_STATUS; /*!<Slot Interrupt Status Register Offset  0x00FC*/
}SDMMC_GLOBAL_TypeDef;

#define   SDMMC_BASE			 ((uint32_t)0x4801C000U)
#define   SDMMC_GLOBAL_BASE		 (0x0004U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   SDMMC		 ((SDMMC_GLOBAL_TypeDef*)(SDMMC_BASE + SDMMC_GLOBAL_BASE))

/***************************************************************************/
/*                     TPIU                          */
/***************************************************************************/

  /* 
  *Cortex M4 - Trace Port Interface Unit
  */
typedef struct {                            /*!< TPIU Structure                        */
  __I  uint32_t  TPIU_SSPSR;                /*!< The TPIU_SSPSR characteristics are:Purpose Indicates the supported parallel trace port sizes.If TPIU_TYPE.PTINVALID is RAO, this register is UNK. For more information see TPIU Type register, TPIU_TYPE on page C1-902. */
  __IO uint32_t  TPIU_CSPSR;                /*!< The TPIU_CSPSR characteristics are:Purpose Defines the width of the current parallel trace port.Usage constraints Has the same format as the TPIU_SSPSR, but: only one bit is set to 1, all others must be zero the effect of writing a value with more than one bit set to 1 is UNPREDICTABLE the effect of a write to an unsupported bit is UNPREDICTABLE.If TPIU_TYPE.PTINVALID is RAO, this register is UNK/SBZP. For more information see TPIU Type register, TPIU_TYPE on page C1-902.Attributes See Table C1-40 on page  */
  __I  uint32_t  RESERVED0[2];
  __IO uint32_t  TPIU_ACPR;                 /*!< The TPIU_ACPR characteristics are:Purpose Scales the baud rate of the asynchronous output. */
  __I  uint32_t  RESERVED1[55];
  __IO uint32_t  TPIU_SPPR;                 /*!< The TPIU_SPPR characteristics are:Purpose Selects the protocol used for trace output.Usage constraints If a debugger changes the register value while the TPIU is transmitting data, the effect on the output stream is UNPREDICTABLE and the required recovery process is IMPLEMENTATION DEFINED. Bits [11:9] of the TPIU_TYPE register define the trace output protocols supported by the implementation, see TPIU Type register, TPIU_TYPE on page C1-902. The effect of selecting an unsupported mode is UNPREDICTABLE. */
  __I  uint32_t  RESERVED2[131];
  __I  uint32_t  TPIU_FFSR;                 /*!< The TPIU_FFSR characteristics are:Purpose Indicates the status of the TPIU formatter. */
  __IO uint32_t  TPIU_FFCR;                 /*!< The TPIU_FFCR characteristics are:Purpose Controls the TPIU formatter.When one of the two SWO modes is selected, bit [1] of TPIU_FFCR enables the formatter to be bypassed. If the formatter is bypassed, only the ITM and DWT trace source passes through.The TPIU accepts and discards data from the ETM. This function is can be used to connect a device containing an ETM to a trace capture device that is only able to capture SWO data.Enabling or disabling the formatter causes momentary data corruption.NoteIf TPIU */
  __I  uint32_t  TPIU_FSCR;                 /*!< The global synchronization trigger is generated by the Program Counter (PC) Sampler block. This means that there is no synchronization counter in the TPIU */
  __I  uint32_t  RESERVED3[759];
  __I  uint32_t  TRIGGER;                   /*!< The TRIGGER characteristics are:Purpose Integration test of the TRIGGER input. */
  __I  uint32_t  FIFO_Data0;                /*!< The Integration ETM Data characteristics are:Purpose Trace data integration testing.Usage constraints You must set bit [1] of TPIU_ITCTRL to use this register. See Integration Mode Control, TPIU_ITCTRL on page 11-11. */
  __I  uint32_t  ITATBCTR2;                 /*!< The ITATBCTR2 characteristics are:Purpose Integration test.Usage constraints You must set bit [0] of TPIU_ITCTRL to use this register. See Integration Mode Control, TPIU_ITCTRL on page 11-11. */
  __I  uint32_t  RESERVED4;
  __I  uint32_t  ITATBCTR0;                 /*!< The ITATBCTR0 characteristics are:Purpose Integration test. */
  __I  uint32_t  FIFO_Data1;                /*!< The Integration ITM Data characteristics are:Purpose Trace data integration testing.Usage constraints You must set bit [1] of TPIU_ITCTRL to use this register. See Integration Mode Control, TPIU_ITCTRL on page 11-11. */
  __IO uint32_t  ITCTRL;                    /*!< The TPIU_ITCTRL characteristics are:Purpose Specifies normal or integration mode for the TPIU. */
  __I  uint32_t  RESERVED5[39];
  __IO uint32_t  CLAIMSET;                  /*!< This register forms one half of the Claim Tag value. This location enables individual bits to be set, write, and returns the number of bits that can be set, read.Read Each bit is considered separately:0 = this claim tag bit is not implemented1 = this claim tag bit is implemented.Write Each bit is considered separately:0 = no effect1 = set this bit in the claim tag. */
  __IO uint32_t  CLAIMCLR;                  /*!< This register forms one half of the Claim Tag value. This location enables individual bits to be cleared, write, and returns the current Claim Tag value, read.The width (n) of this register can be determined from reading the Claim Tag Set Register.Read Current Claim Tag ValueWrite Each bit is considered separately:0 = no effect1 = clear this bit in the claim tag. */
  __I  uint32_t  RESERVED6[8];
  __I  uint32_t  DEVID;                     /*!< The TPIU_DEVID characteristics are:Purpose Indicates the functions provided by the TPIU for use in topology detection. */
  __I  uint32_t  DEVTYPE;                   /*!< The Device Type Identifier Register is read-only. It provides a debugger with information about the component when the Part Number field is not recognized. The debugger can then report this information.The TPIU_DEVTYPE characteristics are:Purpose Indicates the type of functionality the component supports.Attributes The Device Type reads as 0x11 and indicates this device is a trace sink and specifically a TPIU */
  __I  uint32_t  TPIUPID4;                  /*!< Peripheral identification register4   */
  __I  uint32_t  TPIUPID5;                  /*!< Peripheral identification register5   */
  __I  uint32_t  TPIUPID6;                  /*!< Peripheral identification register6   */
  __I  uint32_t  TPIUPID7;                  /*!< Peripheral identification register7   */
  __I  uint32_t  TPIUPID0;                  /*!< Peripheral identification register0   */
  __I  uint32_t  TPIUPID1;                  /*!< Peripheral identification register1   */
  __I  uint32_t  TPIUPID2;                  /*!< Peripheral identification register2   */
  __I  uint32_t  TPIUPID3;                  /*!< Peripheral identification register3   */
  __I  uint32_t  TPIUCID0;                  /*!< Component identification register0    */
  __I  uint32_t  TPIUCID1;                  /*!< Component identification register1    */
  __I  uint32_t  TPIUCID2;                  /*!< Component identification register2    */
  __I  uint32_t  TPIUCID3;                  /*!< Component identification register3    */
} TPIU_Type;

#define   TPIU_BASE			 ((uint32_t)0xE0040000U)
#define   TPIU_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   TPIU		 ((TPIU_Type *)(TPIU_BASE + TPIU_GLOBAL_BASE))

/***************************************************************************/
/*                     USB0                          */
/***************************************************************************/

  /* 
  *CONTROL registers
  */
typedef struct {
__IO uint32_t	GOTGCTL; /*!<Control and Status Register Offset  0x0000*/
__IO uint32_t	GOTGINT; /*!<OTG Interrupt Register Offset  0x0004*/
__IO uint32_t	GAHBCFG; /*!<AHB Configuration Register Offset  0x0008*/
__IO uint32_t	GUSBCFG; /*!<USB Configuration Register Offset  0x000C*/
__IO uint32_t	GRSTCTL; /*!<Reset Register Offset  0x0010*/
union
 {
__IO uint32_t	GINTSTS_HOSTMODE; /*!<Interrupt Register [Host Mode] Offset  0x0014*/
__IO uint32_t	GINTSTS_DEVICEMODE; /*!<Interrupt Register [Device Mode] Offset  0x0014*/
 };
union
 {
__IO uint32_t	GINTMSK_HOSTMODE; /*!<Interrupt Mask Register [Host Mode] Offset  0x0018*/
__IO uint32_t	GINTMSK_DEVICEMODE; /*!<Interrupt Mask Register [Device Mode] Offset  0x0018*/
 };
union
 {
__I uint32_t	GRXSTSR_HOSTMODE; /*!<Receive Status Debug Read Register [Host Mode] Offset  0x001C*/
__I uint32_t	GRXSTSR_DEVICEMODE; /*!<Receive Status Debug Read Register [Device Mode] Offset  0x001C*/
 };
union
 {
__I uint32_t	GRXSTSP_HOSTMODE; /*!<Receive Status Read and Pop Register [Host Mode] Offset  0x0020*/
__I uint32_t	GRXSTSP_DEVICEMODE; /*!<Receive Status Read and Pop Register [Device Mode] Offset  0x0020*/
 };
__IO uint32_t	GRXFSIZ; /*!<Receive FIFO Size Register Offset  0x0024*/
union
 {
__IO uint32_t	GNPTXFSIZ_HOSTMODE; /*!<Non-Periodic Transmit FIFO Size Register [Host Mode] Offset  0x0028*/
__IO uint32_t	GNPTXFSIZ_DEVICEMODE; /*!<Non-Periodic Transmit FIFO Size Register [Device Mode] Offset  0x0028*/
 };
__I uint32_t	GNPTXSTS; /*!<Non-Periodic Transmit FIFO/Queue Status Register Offset  0x002C*/
__I  uint32_t	RESERVED1[3];
__IO uint32_t	GUID; /*!<User ID Register Offset  0x003C*/
__I  uint32_t	RESERVED2[7];
__IO uint32_t	GDFIFOCFG; /*!<Global DFIFO Software Config Register Offset  0x005C*/
__I  uint32_t	RESERVED3[40];
__IO uint32_t	HPTXFSIZ; /*!<Host Periodic Transmit FIFO Size Register Offset  0x0100*/
__IO uint32_t	DIEPTXF1; /*!<Device IN Endpoint Transmit FIFO Size Register Offset  0x0104*/
__IO uint32_t	DIEPTXF2; /*!<Device IN Endpoint Transmit FIFO Size Register Offset  0x0108*/
__IO uint32_t	DIEPTXF3; /*!<Device IN Endpoint Transmit FIFO Size Register Offset  0x010C*/
__IO uint32_t	DIEPTXF4; /*!<Device IN Endpoint Transmit FIFO Size Register Offset  0x0110*/
__IO uint32_t	DIEPTXF5; /*!<Device IN Endpoint Transmit FIFO Size Register Offset  0x0114*/
__IO uint32_t	DIEPTXF6; /*!<Device IN Endpoint Transmit FIFO Size Register Offset  0x0118*/
__I  uint32_t	RESERVED4[185];
__IO uint32_t	HCFG; /*!<Host Configuration Register Offset  0x0400*/
__IO uint32_t	HFIR; /*!<Host Frame Interval Register Offset  0x0404*/
__IO uint32_t	HFNUM; /*!<Host Frame Number/Frame Time Remaining Register Offset  0x0408*/
__I  uint32_t	RESERVED5;
__IO uint32_t	HPTXSTS; /*!<Host Periodic Transmit FIFO/ Queue Status Register Offset  0x0410*/
__I uint32_t	HAINT; /*!<Host All Channels Interrupt Register Offset  0x0414*/
__IO uint32_t	HAINTMSK; /*!<Host All Channels Interrupt Mask Register Offset  0x0418*/
__IO uint32_t	HFLBADDR; /*!<Host Frame List Base Address Register Offset  0x041C*/
__I  uint32_t	RESERVED6[8];
__IO uint32_t	HPRT; /*!<Host Port Control and Status Register Offset  0x0440*/
__I  uint32_t	RESERVED7[239];
__IO uint32_t	DCFG; /*!<Device Configuration Register Offset  0x0800*/
__IO uint32_t	DCTL; /*!<Device Control Register Offset  0x0804*/
__I uint32_t	DSTS; /*!<Device Status Register Offset  0x0808*/
__I  uint32_t	RESERVED8;
__IO uint32_t	DIEPMSK; /*!<Device IN Endpoint Common Interrupt Mask Register Offset  0x0810*/
__IO uint32_t	DOEPMSK; /*!<Device OUT Endpoint Common Interrupt Mask Register Offset  0x0814*/
__I uint32_t	DAINT; /*!<Device All Endpoints Interrupt Register Offset  0x0818*/
__IO uint32_t	DAINTMSK; /*!<Device All Endpoints Interrupt Mask Register Offset  0x081C*/
__I  uint32_t	RESERVED9[2];
__IO uint32_t	DVBUSDIS; /*!<Device VBUS Discharge Time Register Offset  0x0828*/
__IO uint32_t	DVBUSPULSE; /*!<Device VBUS Pulsing Time Register Offset  0x082C*/
__I  uint32_t	RESERVED10;
__IO uint32_t	DIEPEMPMSK; /*!<Device IN Endpoint FIFO Empty Interrupt Mask Register Offset  0x0834*/
__I  uint32_t	RESERVED11[370];
__IO uint32_t	PCGCCTL; /*!<Power and Clock Gating Control Register Offset  0x0E00*/
}USB0_GLOBAL_TypeDef;

  /* 
  *ENDPOINT-0 registers
  */
typedef struct {
__IO uint32_t	DIEPCTL0; /*!<Device Control IN Endpoint 0 Control Register Offset  0x0900*/
__I  uint32_t	RESERVED1;
__IO uint32_t	DIEPINT0; /*!<Device Endpoint-0 Interrupt Register Offset  0x0908*/
__I  uint32_t	RESERVED2;
__IO uint32_t	DIEPTSIZ0; /*!<Device IN Endpoint 0 Transfer Size Register Offset  0x0910*/
__IO uint32_t	DIEPDMA0; /*!<Device Endpoint-0 DMA Address Register Offset  0x0914*/
__I uint32_t	DTXFSTS0; /*!<Device IN Endpoint Transmit FIFO Status Register Offset  0x0918*/
__I uint32_t	DIEPDMAB0; /*!<Device Endpoint-0 DMA Buffer Address Register Offset  0x091C*/
__I  uint32_t	RESERVED3[120];
__IO uint32_t	DOEPCTL0; /*!<Device Control OUT Endpoint 0 Control Register Offset  0x0B00*/
__I  uint32_t	RESERVED4;
__IO uint32_t	DOEPINT0; /*!<Device Endpoint-0 Interrupt Register Offset  0x0B08*/
__I  uint32_t	RESERVED5;
__IO uint32_t	DOEPTSIZ0; /*!<Device OUT Endpoint 0 Transfer Size Register Offset  0x0B10*/
__IO uint32_t	DOEPDMA0; /*!<Device Endpoint-0 DMA Address Register Offset  0x0B14*/
__I  uint32_t	RESERVED6;
__I uint32_t	DOEPDMAB0; /*!<Device Endpoint-0 DMA Buffer Address Register Offset  0x0B1C*/
}USB0_EP0_TypeDef;

  /* 
  *ENDPOINTx registers
  */
typedef struct {
union
 {
__IO uint32_t	DIEPCTL_INTBULK; /*!<Device Endpoint-1 Control Register [Interrupt/Bulk Endpoint] Offset  0x0920*/
__IO uint32_t	DIEPCTL_ISOCONT; /*!<Device Endpoint-1 Control Register [Isochronous/Control Endpoint] Offset  0x0920*/
 };
__I  uint32_t	RESERVED1;
__IO uint32_t	DIEPINT; /*!<Device Endpoint-1 Interrupt Register Offset  0x0928*/
__I  uint32_t	RESERVED2;
__IO uint32_t	DIEPTSIZ; /*!<Device Endpoint-1 Transfer Size Register Offset  0x0930*/
__IO uint32_t	DIEPDMA; /*!<Device Endpoint-1 DMA Address Register Offset  0x0934*/
__I uint32_t	DTXFSTS; /*!<Device IN Endpoint Transmit FIFO Status Register Offset  0x0938*/
__I uint32_t	DIEPDMAB; /*!<Device Endpoint-1 DMA Buffer Address Register Offset  0x093C*/
__I  uint32_t	RESERVED3[120];
union
 {
__IO uint32_t	DOEPCTL_INTBULK; /*!<Device Endpoint-1 Control Register [Interrupt/Bulk Endpoint] Offset  0x0B20*/
__IO uint32_t	DOEPCTL_ISOCONT; /*!<Device Endpoint-1 Control Register [Isochronous/Control Endpoint] Offset  0x0B20*/
 };
__I  uint32_t	RESERVED4;
__IO uint32_t	DOEPINT; /*!<Device Endpoint-1 Interrupt Register Offset  0x0B28*/
__I  uint32_t	RESERVED5;
union
 {
__IO uint32_t	DOEPTSIZ_ISO; /*!<Device Endpoint-1 Transfer Size Register [Isochronous OUT Endpoint] Offset  0x0B30*/
__IO uint32_t	DOEPTSIZ_CONTROL; /*!<Device Endpoint-1 Transfer Size Register [Control OUT Endpoint] Offset  0x0B30*/
 };
__IO uint32_t	DOEPDMA; /*!<Device Endpoint-1 DMA Address Register Offset  0x0B34*/
__I  uint32_t	RESERVED6;
__I uint32_t	DOEPDMAB; /*!<Device Endpoint-1 DMA Buffer Address Register Offset  0x0B3C*/
}USB0_EP_TypeDef;

  /* 
  *CHANNEL registers
  */
typedef struct {
__IO uint32_t	HCCHAR; /*!<Host Channel-0 Characteristics Register Offset  0x0500*/
__I  uint32_t	RESERVED1;
__IO uint32_t	HCINT; /*!<Host Channel-0 Interrupt Register Offset  0x0508*/
__IO uint32_t	HCINTMSK; /*!<Host Channel-0 Interrupt Mask Register Offset  0x050C*/
union
 {
__IO uint32_t	HCTSIZ_SCATGATHER; /*!<Host Channel-0 Transfer Size Register [Scatter/Gather DMA Mode] Offset  0x0510*/
__IO uint32_t	HCTSIZ_BUFFERMODE; /*!<Host Channel-0 Transfer Size Register [Buffer DMA Mode] Offset  0x0510*/
 };
union
 {
__IO uint32_t	HCDMA_SCATGATHER; /*!<Host Channel-0 DMA Address Register [Scatter/Gather DMA Mode] Offset  0x0514*/
__IO uint32_t	HCDMA_BUFFERMODE; /*!<Host Channel-0 DMA Address Register [Buffer DMA Mode] Offset  0x0514*/
 };
__I  uint32_t	RESERVED2;
__I uint32_t	HCDMAB; /*!<Host Channel-0 DMA Buffer Address Register Offset  0x051C*/
}USB0_CH_TypeDef;

#define   USB0_BASE			 ((uint32_t)0x50040000U)
#define   USB0_GLOBAL_BASE		 (0x0000U)
#define   USB0_EP0_BASE		 (0x0900U)
#define   USB0_EP1_BASE		 (0x0920U)
#define   USB0_EP2_BASE		 (0x0940U)
#define   USB0_EP3_BASE		 (0x0960U)
#define   USB0_EP4_BASE		 (0x0980U)
#define   USB0_EP5_BASE		 (0x09A0U)
#define   USB0_EP6_BASE		 (0x09C0U)
#define   USB0_CH0_BASE		 (0x0500U)
#define   USB0_CH1_BASE		 (0x0520U)
#define   USB0_CH2_BASE		 (0x0540U)
#define   USB0_CH3_BASE		 (0x0560U)
#define   USB0_CH4_BASE		 (0x0580U)
#define   USB0_CH5_BASE		 (0x05A0U)
#define   USB0_CH6_BASE		 (0x05C0U)
#define   USB0_CH7_BASE		 (0x05E0U)
#define   USB0_CH8_BASE		 (0x0600U)
#define   USB0_CH9_BASE		 (0x0620U)
#define   USB0_CH10_BASE		 (0x0640U)
#define   USB0_CH11_BASE		 (0x0660U)
#define   USB0_CH12_BASE		 (0x0680U)
#define   USB0_CH13_BASE		 (0x06A0U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   USB0		 ((USB0_GLOBAL_TypeDef*)(USB0_BASE + USB0_GLOBAL_BASE))
#define   USB0_EP0		 ((USB0_EP0_TypeDef*)(USB0_BASE + USB0_EP0_BASE))
#define   USB0_EP1		 ((USB0_EP_TypeDef*)(USB0_BASE + USB0_EP1_BASE))
#define   USB0_EP2		 ((USB0_EP_TypeDef*)(USB0_BASE + USB0_EP2_BASE))
#define   USB0_EP3		 ((USB0_EP_TypeDef*)(USB0_BASE + USB0_EP3_BASE))
#define   USB0_EP4		 ((USB0_EP_TypeDef*)(USB0_BASE + USB0_EP4_BASE))
#define   USB0_EP5		 ((USB0_EP_TypeDef*)(USB0_BASE + USB0_EP5_BASE))
#define   USB0_EP6		 ((USB0_EP_TypeDef*)(USB0_BASE + USB0_EP6_BASE))
#define   USB0_CH0		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH0_BASE))
#define   USB0_CH1		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH1_BASE))
#define   USB0_CH2		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH2_BASE))
#define   USB0_CH3		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH3_BASE))
#define   USB0_CH4		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH4_BASE))
#define   USB0_CH5		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH5_BASE))
#define   USB0_CH6		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH6_BASE))
#define   USB0_CH7		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH7_BASE))
#define   USB0_CH8		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH8_BASE))
#define   USB0_CH9		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH9_BASE))
#define   USB0_CH10		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH10_BASE))
#define   USB0_CH11		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH11_BASE))
#define   USB0_CH12		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH12_BASE))
#define   USB0_CH13		 ((USB0_CH_TypeDef*)(USB0_BASE + USB0_CH13_BASE))

/***************************************************************************/
/*                     USICx                          */
/***************************************************************************/

  /* 
  *Module wide registers
  */
typedef struct {
__I  uint32_t	RESERVED1[2];
__I uint32_t	ID; /*!<Module Identification Register Offset  0x0008*/
}USIC_GLOBAL_TypeDef;

  /* 
  *Channel Related registers
  */
typedef struct {
__I uint32_t	CCFG; /*!<Channel Configuration Register Offset  0x0004*/
__I  uint32_t	RESERVED1;
__IO uint16_t	KSCFG; /*!<Kernel State Configuration Register Offset  0x000C*/
__I  uint16_t	RESERVED2;
__IO uint32_t	FDR; /*!<Fractional Divider Register Offset  0x0010*/
__IO uint32_t	BRG; /*!<Baud Rate Generator Register Offset  0x0014*/
__IO uint32_t	INPR; /*!<Interrupt Node Pointer Register Offset  0x0018*/
__IO uint32_t	DX0CR; /*!<Input Control Register 0 Offset  0x001C*/
__IO uint32_t	DX1CR; /*!<Input Control Register 1 Offset  0x0020*/
__IO uint32_t	DX2CR; /*!<Input Control Register 2 Offset  0x0024*/
__IO uint32_t	DX3CR; /*!<Input Control Register 3 Offset  0x0028*/
__IO uint32_t	DX4CR; /*!<Input Control Register 4 Offset  0x002C*/
__IO uint32_t	DX5CR; /*!<Input Control Register 5 Offset  0x0030*/
__IO uint32_t	SCTR; /*!<Shift Control Register Offset  0x0034*/
__IO uint32_t	TCSR; /*!<Transmit Control/Status Register Offset  0x0038*/
union
 {
__IO uint32_t	PCR_ASCMode; /*!<Protocol Control Register [ASC Mode] Offset  0x003C*/
__IO uint32_t	PCR_SSCMode; /*!<Protocol Control Register [SSC Mode] Offset  0x003C*/
__IO uint32_t	PCR_IISMode; /*!<Protocol Control Register [IIS Mode] Offset  0x003C*/
__IO uint32_t	PCR_IICMode; /*!<Protocol Control Register [IIC Mode] Offset  0x003C*/
 };
__IO uint32_t	CCR; /*!<Channel Control Register Offset  0x0040*/
__IO uint32_t	CMTR; /*!<Capture Mode Timer Register Offset  0x0044*/
union
 {
__IO uint32_t	PSR_ASCMode; /*!<Protocol Status Register [ASC Mode] Offset  0x0048*/
__IO uint32_t	PSR_SSCMode; /*!<Protocol Status Register [SSC Mode] Offset  0x0048*/
__IO uint32_t	PSR_IISMode; /*!<Protocol Status Register [IIS Mode] Offset  0x0048*/
__IO uint32_t	PSR_IICMode; /*!<Protocol Status Register [IIC Mode] Offset  0x0048*/
 };
__O uint32_t	PSCR; /*!<Protocol Status Clear Register Offset  0x004C*/
__I uint32_t	RBUFSR; /*!<Receiver Buffer Status Register Offset  0x0050*/
__I uint32_t	RBUF; /*!<Receiver Buffer Register Offset  0x0054*/
__I uint32_t	RBUFD; /*!<Receiver Buffer Register for Debugger Offset  0x0058*/
__I uint32_t	RBUF0; /*!<Receiver Buffer Register 0 Offset  0x005C*/
__I uint32_t	RBUF1; /*!<Receiver Buffer Register 1 Offset  0x0060*/
__I uint32_t	RBUF01SR; /*!<Receiver Buffer 01 Status Register Offset  0x0064*/
__O uint32_t	FMR; /*!<Flag Modification Register Offset  0x0068*/
__I  uint32_t	RESERVED3[5];
__IO uint32_t	TBUF[32]; /*!<Transmit Buffer Input Location 00 Offset  0x0080*/
__IO uint32_t	BYP; /*!<Bypass Data Register Offset  0x0100*/
__IO uint32_t	BYPCR; /*!<Bypass Control Register Offset  0x0104*/
__IO uint32_t	TBCTR; /*!<Transmitter Buffer Control Register Offset  0x0108*/
__IO uint32_t	RBCTR; /*!<Receiver Buffer Control Register Offset  0x010C*/
__I uint32_t	TRBPTR; /*!<Transmit/Receive Buffer Pointer Register Offset  0x0110*/
__IO uint32_t	TRBSR; /*!<Transmit/Receive Buffer Status Register Offset  0x0114*/
__O uint32_t	TRBSCR; /*!<Transmit/Receive Buffer Status Clear Register Offset  0x0118*/
__I uint32_t	OUTR; /*!<Receiver Buffer Output Register Offset  0x011C*/
__I uint32_t	OUTDR; /*!<Receiver Buffer Output Register L for Debugger Offset  0x0120*/
__I  uint32_t	RESERVED4[23];
__O uint32_t	IN[32]; /*!<Transmit FIFO Buffer Input Location 00 Offset  0x0180*/
}USIC_CH_TypeDef;

#define   USIC0_BASE			 ((uint32_t)0x40030000U)
#define   USIC1_BASE			 ((uint32_t)0x48020000U)
#define   USIC2_BASE			 ((uint32_t)0x48024000U)
#define   USIC_GLOBAL_BASE		 (0x0000U)
#define   USIC0_CH0_BASE		 (0x0004U)
#define   USIC0_CH1_BASE		 (0x0204U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   USIC0		 ((USIC_GLOBAL_TypeDef*)(USIC0_BASE + USIC_GLOBAL_BASE))
#define   USIC0_CH0		 ((USIC_CH_TypeDef*)(USIC0_BASE + USIC0_CH0_BASE))
#define   USIC0_CH1		 ((USIC_CH_TypeDef*)(USIC0_BASE + USIC0_CH1_BASE))
#define   USIC1		 ((USIC_GLOBAL_TypeDef*)(USIC1_BASE + USIC_GLOBAL_BASE))
#define   USIC1_CH0		 ((USIC_CH_TypeDef*)(USIC1_BASE + USIC0_CH0_BASE))
#define   USIC1_CH1		 ((USIC_CH_TypeDef*)(USIC1_BASE + USIC0_CH1_BASE))
#define   USIC2		 ((USIC_GLOBAL_TypeDef*)(USIC2_BASE + USIC_GLOBAL_BASE))
#define   USIC2_CH0		 ((USIC_CH_TypeDef*)(USIC2_BASE + USIC0_CH0_BASE))
#define   USIC2_CH1		 ((USIC_CH_TypeDef*)(USIC2_BASE + USIC0_CH1_BASE))

/***************************************************************************/
/*                     VADC                          */
/***************************************************************************/

  /* 
  *ADC Group registers
  */
typedef struct {
__IO uint32_t	ARBCFG; /*!<Arbitration Configuration Register, Group 0 Offset  0x0480*/
__IO uint32_t	ARBPR; /*!<Arbitration Priority Register, Group 0 Offset  0x0484*/
__IO uint32_t	CHASS; /*!<Channel Assignment Register, Group 0 Offset  0x0488*/
__I  uint32_t	RESERVED1[5];
__IO uint32_t	ICLASS[2]; /*!<Input Class Register , Group 0 Offset  0x04A0*/
__I  uint32_t	RESERVED2[2];
__IO uint32_t	ALIAS; /*!<Alias Register, Group 0 Offset  0x04B0*/
__I  uint32_t	RESERVED3;
__IO uint32_t	BOUND; /*!<Boundary Select Register, Group 0 Offset  0x04B8*/
__I  uint32_t	RESERVED4;
__IO uint32_t	SYNCTR; /*!<Synchronization Control Register, Group 0 Offset  0x04C0*/
__I  uint32_t	RESERVED5;
__IO uint32_t	BFL; /*!<Boundary Flag Register, Group 0 Offset  0x04C8*/
__I  uint32_t	RESERVED6[13];
__IO uint32_t	QCTRL0; /*!<Queue 0 Source Control Register, Group 0 Offset  0x0500*/
__IO uint32_t	QMR0; /*!<Queue 0 Mode Register, Group 0 Offset  0x0504*/
__I uint32_t	QSR0; /*!<Queue 0 Status Register, Group 0 Offset  0x0508*/
__I uint32_t	Q0R0; /*!<Queue 0 Register 0, Group 0 Offset  0x050C*/
union
 {
__I uint32_t	QBUR0; /*!<Queue 0 Backup Register, Group 0 Offset  0x0510*/
__O uint32_t	QINR0; /*!<Queue 0 Input Register, Group 0 Offset  0x0510*/
 };
__I  uint32_t	RESERVED7[3];
__IO uint32_t	ASCTRL; /*!<Autoscan Source Control Register, Group 0 Offset  0x0520*/
__IO uint32_t	ASMR; /*!<Autoscan Source Mode Register, Group 0 Offset  0x0524*/
__IO uint32_t	ASSEL; /*!<Autoscan Source Channel Select Register, Group 0 Offset  0x0528*/
__IO uint32_t	ASPND; /*!<Autoscan Source Pending Register, Group 0 Offset  0x052C*/
__I  uint32_t	RESERVED8[20];
__IO uint32_t	CEFLAG; /*!<Channel Event Flag Register, Group 0 Offset  0x0580*/
__IO uint32_t	REFLAG; /*!<Result Event Flag Register, Group 0 Offset  0x0584*/
__IO uint32_t	SEFLAG; /*!<Source Event Flag Register, Group 0 Offset  0x0588*/
__I  uint32_t	RESERVED9;
__O uint32_t	CEFCLR; /*!<Channel Event Flag Clear Register, Group 0 Offset  0x0590*/
__O uint32_t	REFCLR; /*!<Result Event Flag Clear Register, Group 0 Offset  0x0594*/
__O uint32_t	SEFCLR; /*!<Source Event Flag Clear Register, Group 0 Offset  0x0598*/
__I  uint32_t	RESERVED10;
__IO uint32_t	CEVNP0; /*!<Channel Event Node Pointer Register 0, Group 0 Offset  0x05A0*/
__I  uint32_t	RESERVED11[3];
__IO uint32_t	REVNP0; /*!<Result Event Node Pointer Register 0, Group 0 Offset  0x05B0*/
__IO uint32_t	REVNP1; /*!<Result Event Node Pointer Register 1, Group 0 Offset  0x05B4*/
__I  uint32_t	RESERVED12[2];
__IO uint32_t	SEVNP; /*!<Source Event Node Pointer Register, Group 0 Offset  0x05C0*/
__I  uint32_t	RESERVED13;
__O uint32_t	SRACT; /*!<Service Request Software Activation Trigger, Group 0 Offset  0x05C8*/
__I  uint32_t	RESERVED14[9];
__IO uint32_t	EMUXCTR; /*!<E0ternal Multiplexer Control Register, Group x Offset  0x05F0*/
__I  uint32_t	RESERVED15;
__IO uint32_t	VFR; /*!<Valid Flag Register, Group 0 Offset  0x05F8*/
__I  uint32_t	RESERVED16;
__IO uint32_t	CHCTR[8]; /*!<Group 0, Channel 0 Ctrl. Reg. Offset  0x0600*/
__I  uint32_t	RESERVED17[24];
__IO uint32_t	RCR[16]; /*!<Group 0 Result Control Reg. 0 Offset  0x0680*/
__I  uint32_t	RESERVED18[16];
__IO uint32_t	RES[16]; /*!<Group 0 Result Register 0 Offset  0x0700*/
__I  uint32_t	RESERVED19[16];
__I uint32_t	RESD[16]; /*!<Group 0 Result Reg. 0, Debug Offset  0x0780*/
}VADC_G_TypeDef;

  /* 
  *ADC global registers
  */
typedef struct {
__IO uint32_t	CLC; /*!<Clock Control Register Offset  0x0000*/
__I  uint32_t	RESERVED1;
__I uint32_t	ID; /*!<Module Identification Register Offset  0x0008*/
__I  uint32_t	RESERVED2[7];
__IO uint32_t	OCS; /*!<OCDS Control and Status Register Offset  0x0028*/
__I  uint32_t	RESERVED3[21];
__IO uint32_t	GLOBCFG; /*!<Global Configuration Register Offset  0x0080*/
__I  uint32_t	RESERVED4[7];
__IO uint32_t	GLOBICLASS[2]; /*!<Input Class Register 0, Global Offset  0x00A0*/
__I  uint32_t	RESERVED5[4];
__IO uint32_t	GLOBBOUND; /*!<Global Boundary Select Register Offset  0x00B8*/
__I  uint32_t	RESERVED6[9];
__IO uint32_t	GLOBEFLAG; /*!<Global Event Flag Register Offset  0x00E0*/
__I  uint32_t	RESERVED7[23];
__IO uint32_t	GLOBEVNP; /*!<Global Event Node Pointer Register Offset  0x0140*/
__I  uint32_t	RESERVED8[7];
__IO uint32_t	GLOBTF; /*!<Global Test Functions Register Offset  0x0160*/
__I  uint32_t	RESERVED9[7];
__IO uint32_t	BRSSEL[4]; /*!<Background Request Source Channel Select Register, Group 0 Offset  0x0180*/
__I  uint32_t	RESERVED10[12];
__IO uint32_t	BRSPND[4]; /*!<Background Request Source Pending Register, Group 0 Offset  0x01C0*/
__I  uint32_t	RESERVED11[12];
__IO uint32_t	BRSCTRL; /*!<Background Request Source Control Register Offset  0x0200*/
__IO uint32_t	BRSMR; /*!<Background Request Source Mode Register Offset  0x0204*/
__I  uint32_t	RESERVED12[30];
__IO uint32_t	GLOBRCR; /*!<Global Result Control Register Offset  0x0280*/
__I  uint32_t	RESERVED13[31];
__IO uint32_t	GLOBRES; /*!<Global Result Register Offset  0x0300*/
__I  uint32_t	RESERVED14[31];
__IO uint32_t	GLOBRESD; /*!<Global Result Register, Debug Offset  0x0380*/
__I  uint32_t	RESERVED15[27];
__IO uint32_t	EMUXSEL; /*!<External Multiplexer Select Register Offset  0x03F0*/
}VADC_GLOBAL_TypeDef;

#define   VADC_BASE			 ((uint32_t)0x40004000U)
#define   VADC_G0_BASE		 (0x0480U)
#define   VADC_G1_BASE		 (0x0880U)
#define   VADC_G2_BASE		 (0x0C80U)
#define   VADC_G3_BASE		 (0x1080U)
#define   VADC_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   VADC_G0		 ((VADC_G_TypeDef*)(VADC_BASE + VADC_G0_BASE))
#define   VADC_G1		 ((VADC_G_TypeDef*)(VADC_BASE + VADC_G1_BASE))
#define   VADC_G2		 ((VADC_G_TypeDef*)(VADC_BASE + VADC_G2_BASE))
#define   VADC_G3		 ((VADC_G_TypeDef*)(VADC_BASE + VADC_G3_BASE))
#define   VADC		 ((VADC_GLOBAL_TypeDef*)(VADC_BASE + VADC_GLOBAL_BASE))

/***************************************************************************/
/*                     WDT                          */
/***************************************************************************/

  /* 
  *WDT Kernel registers
  */
typedef struct {
__I uint32_t	ID; /*!<WDT ID Register Offset  0x0000*/
__IO uint32_t	CTR; /*!<WDT Control Register Offset  0x0004*/
__O uint32_t	SRV; /*!<WDT Service Register Offset  0x0008*/
__I uint32_t	TIM; /*!<WDT Timer Register Offset  0x000C*/
__IO uint32_t	WLB; /*!<WDT Window Lower Bound Register Offset  0x0010*/
__IO uint32_t	WUB; /*!<WDT Window Upper Bound Register Offset  0x0014*/
__I uint32_t	WDTSTS; /*!<WDT Status Register Offset  0x0018*/
__O uint32_t	WDTCLR; /*!<WDT Clear Register Offset  0x001C*/
}WDT_GLOBAL_TypeDef;

#define   WDT_BASE			 ((uint32_t)0x50008000U)
#define   WDT_GLOBAL_BASE		 (0x0000U)
/***************************************************************************/
/* Peripheral Declaration                                                  */
/***************************************************************************/
#define   WDT		 ((WDT_GLOBAL_TypeDef*)(WDT_BASE + WDT_GLOBAL_BASE))


/************************** Bit Fileds *************************************/
/***************************************************************************/


/***   CAN Bit Fileds *******************/
/***************************************************************************/


/* CAN_CLC  =  CAN Clock Control Register*/
#define   CAN_CLC_DISR_Pos  	(0U)
#define   CAN_CLC_DISR_Msk  	(0x00000001U  << CAN_CLC_DISR_Pos)

#define   CAN_CLC_DISS_Pos  	(1U)
#define   CAN_CLC_DISS_Msk  	(0x00000001U  << CAN_CLC_DISS_Pos)

#define   CAN_CLC_EDIS_Pos  	(3U)
#define   CAN_CLC_EDIS_Msk  	(0x00000001U  << CAN_CLC_EDIS_Pos)

#define   CAN_CLC_SBWE_Pos  	(4U)
#define   CAN_CLC_SBWE_Msk  	(0x00000001U  << CAN_CLC_SBWE_Pos)

/* CAN_ID  =  Module Identification Register*/
#define   CAN_ID_MOD_REV_Pos  	(0U)
#define   CAN_ID_MOD_REV_Msk  	(0x000000FFU  << CAN_ID_MOD_REV_Pos)

#define   CAN_ID_MOD_TYPE_Pos  	(8U)
#define   CAN_ID_MOD_TYPE_Msk  	(0x000000FFU  << CAN_ID_MOD_TYPE_Pos)

#define   CAN_ID_MOD_NUMBER_Pos  	(16U)
#define   CAN_ID_MOD_NUMBER_Msk  	(0x0000FFFFU  << CAN_ID_MOD_NUMBER_Pos)

/* CAN_FDR  =  CAN Fractional Divider Register*/
#define   CAN_FDR_STEP_Pos  	(0U)
#define   CAN_FDR_STEP_Msk  	(0x000003FFU  << CAN_FDR_STEP_Pos)

#define   CAN_FDR_SM_Pos  	(11U)
#define   CAN_FDR_SM_Msk  	(0x00000001U  << CAN_FDR_SM_Pos)

#define   CAN_FDR_SC_Pos  	(12U)
#define   CAN_FDR_SC_Msk  	(0x00000003U  << CAN_FDR_SC_Pos)

#define   CAN_FDR_DM_Pos  	(14U)
#define   CAN_FDR_DM_Msk  	(0x00000003U  << CAN_FDR_DM_Pos)

#define   CAN_FDR_RESULT_Pos  	(16U)
#define   CAN_FDR_RESULT_Msk  	(0x000003FFU  << CAN_FDR_RESULT_Pos)

#define   CAN_FDR_SUSACK_Pos  	(28U)
#define   CAN_FDR_SUSACK_Msk  	(0x00000001U  << CAN_FDR_SUSACK_Pos)

#define   CAN_FDR_SUSREQ_Pos  	(29U)
#define   CAN_FDR_SUSREQ_Msk  	(0x00000001U  << CAN_FDR_SUSREQ_Pos)

#define   CAN_FDR_ENHW_Pos  	(30U)
#define   CAN_FDR_ENHW_Msk  	(0x00000001U  << CAN_FDR_ENHW_Pos)

#define   CAN_FDR_DISCLK_Pos  	(31U)
#define   CAN_FDR_DISCLK_Msk  	(0x00000001U  << CAN_FDR_DISCLK_Pos)

/* CAN_LIST  =  List Register 0*/
#define   CAN_LIST_BEGIN_Pos  	(0U)
#define   CAN_LIST_BEGIN_Msk  	(0x000000FFU  << CAN_LIST_BEGIN_Pos)

#define   CAN_LIST_END_Pos  	(8U)
#define   CAN_LIST_END_Msk  	(0x000000FFU  << CAN_LIST_END_Pos)

#define   CAN_LIST_SIZE_Pos  	(16U)
#define   CAN_LIST_SIZE_Msk  	(0x000000FFU  << CAN_LIST_SIZE_Pos)

#define   CAN_LIST_EMPTY_Pos  	(24U)
#define   CAN_LIST_EMPTY_Msk  	(0x00000001U  << CAN_LIST_EMPTY_Pos)

/* CAN_MSPND  =  Message Pending Register 0*/
#define   CAN_MSPND_PND_Pos  	(0U)
#define   CAN_MSPND_PND_Msk  	(0xFFFFFFFFU  << CAN_MSPND_PND_Pos)

/* CAN_MSID  =  Message Index Register 0*/
#define   CAN_MSID_INDEX_Pos  	(0U)
#define   CAN_MSID_INDEX_Msk  	(0x0000003FU  << CAN_MSID_INDEX_Pos)

/* CAN_MSIMASK  =  Message Index Mask Register*/
#define   CAN_MSIMASK_IM_Pos  	(0U)
#define   CAN_MSIMASK_IM_Msk  	(0xFFFFFFFFU  << CAN_MSIMASK_IM_Pos)

/* CAN_PANCTR  =  Panel Control Register*/
#define   CAN_PANCTR_PANCMD_Pos  	(0U)
#define   CAN_PANCTR_PANCMD_Msk  	(0x000000FFU  << CAN_PANCTR_PANCMD_Pos)

#define   CAN_PANCTR_BUSY_Pos  	(8U)
#define   CAN_PANCTR_BUSY_Msk  	(0x00000001U  << CAN_PANCTR_BUSY_Pos)

#define   CAN_PANCTR_RBUSY_Pos  	(9U)
#define   CAN_PANCTR_RBUSY_Msk  	(0x00000001U  << CAN_PANCTR_RBUSY_Pos)

#define   CAN_PANCTR_PANAR1_Pos  	(16U)
#define   CAN_PANCTR_PANAR1_Msk  	(0x000000FFU  << CAN_PANCTR_PANAR1_Pos)

#define   CAN_PANCTR_PANAR2_Pos  	(24U)
#define   CAN_PANCTR_PANAR2_Msk  	(0x000000FFU  << CAN_PANCTR_PANAR2_Pos)

/* CAN_MCR  =  Module Control Register*/
#define   CAN_MCR_MPSEL_Pos  	(12U)
#define   CAN_MCR_MPSEL_Msk  	(0x0000000FU  << CAN_MCR_MPSEL_Pos)

/* CAN_MITR  =  Module Interrupt Trigger Register*/
#define   CAN_MITR_IT_Pos  	(0U)
#define   CAN_MITR_IT_Msk  	(0x000000FFU  << CAN_MITR_IT_Pos)

/* CAN_NODE_NCRx  =  Node x Control Register*/
#define   CAN_NODE_NCR_INIT_Pos  	(0U)
#define   CAN_NODE_NCR_INIT_Msk  	(0x00000001U  << CAN_NODE_NCR_INIT_Pos)

#define   CAN_NODE_NCR_TRIE_Pos  	(1U)
#define   CAN_NODE_NCR_TRIE_Msk  	(0x00000001U  << CAN_NODE_NCR_TRIE_Pos)

#define   CAN_NODE_NCR_LECIE_Pos  	(2U)
#define   CAN_NODE_NCR_LECIE_Msk  	(0x00000001U  << CAN_NODE_NCR_LECIE_Pos)

#define   CAN_NODE_NCR_ALIE_Pos  	(3U)
#define   CAN_NODE_NCR_ALIE_Msk  	(0x00000001U  << CAN_NODE_NCR_ALIE_Pos)

#define   CAN_NODE_NCR_CANDIS_Pos  	(4U)
#define   CAN_NODE_NCR_CANDIS_Msk  	(0x00000001U  << CAN_NODE_NCR_CANDIS_Pos)

#define   CAN_NODE_NCR_CCE_Pos  	(6U)
#define   CAN_NODE_NCR_CCE_Msk  	(0x00000001U  << CAN_NODE_NCR_CCE_Pos)

#define   CAN_NODE_NCR_CALM_Pos  	(7U)
#define   CAN_NODE_NCR_CALM_Msk  	(0x00000001U  << CAN_NODE_NCR_CALM_Pos)

#define   CAN_NODE_NCR_SUSEN_Pos  	(8U)
#define   CAN_NODE_NCR_SUSEN_Msk  	(0x00000001U  << CAN_NODE_NCR_SUSEN_Pos)

/* CAN_NODE_NSRx  =  Node x Status Register*/
#define   CAN_NODE_NSR_LEC_Pos  	(0U)
#define   CAN_NODE_NSR_LEC_Msk  	(0x00000007U  << CAN_NODE_NSR_LEC_Pos)

#define   CAN_NODE_NSR_TXOK_Pos  	(3U)
#define   CAN_NODE_NSR_TXOK_Msk  	(0x00000001U  << CAN_NODE_NSR_TXOK_Pos)

#define   CAN_NODE_NSR_RXOK_Pos  	(4U)
#define   CAN_NODE_NSR_RXOK_Msk  	(0x00000001U  << CAN_NODE_NSR_RXOK_Pos)

#define   CAN_NODE_NSR_ALERT_Pos  	(5U)
#define   CAN_NODE_NSR_ALERT_Msk  	(0x00000001U  << CAN_NODE_NSR_ALERT_Pos)

#define   CAN_NODE_NSR_EWRN_Pos  	(6U)
#define   CAN_NODE_NSR_EWRN_Msk  	(0x00000001U  << CAN_NODE_NSR_EWRN_Pos)

#define   CAN_NODE_NSR_BOFF_Pos  	(7U)
#define   CAN_NODE_NSR_BOFF_Msk  	(0x00000001U  << CAN_NODE_NSR_BOFF_Pos)

#define   CAN_NODE_NSR_LLE_Pos  	(8U)
#define   CAN_NODE_NSR_LLE_Msk  	(0x00000001U  << CAN_NODE_NSR_LLE_Pos)

#define   CAN_NODE_NSR_LOE_Pos  	(9U)
#define   CAN_NODE_NSR_LOE_Msk  	(0x00000001U  << CAN_NODE_NSR_LOE_Pos)

#define   CAN_NODE_NSR_SUSACK_Pos  	(10U)
#define   CAN_NODE_NSR_SUSACK_Msk  	(0x00000001U  << CAN_NODE_NSR_SUSACK_Pos)

/* CAN_NODE_NIPRx  =  Node x Interrupt Pointer Register*/
#define   CAN_NODE_NIPR_ALINP_Pos  	(0U)
#define   CAN_NODE_NIPR_ALINP_Msk  	(0x00000007U  << CAN_NODE_NIPR_ALINP_Pos)

#define   CAN_NODE_NIPR_LECINP_Pos  	(4U)
#define   CAN_NODE_NIPR_LECINP_Msk  	(0x00000007U  << CAN_NODE_NIPR_LECINP_Pos)

#define   CAN_NODE_NIPR_TRINP_Pos  	(8U)
#define   CAN_NODE_NIPR_TRINP_Msk  	(0x00000007U  << CAN_NODE_NIPR_TRINP_Pos)

#define   CAN_NODE_NIPR_CFCINP_Pos  	(12U)
#define   CAN_NODE_NIPR_CFCINP_Msk  	(0x00000007U  << CAN_NODE_NIPR_CFCINP_Pos)

/* CAN_NODE_NPCRx  =  Node x Port Control Register*/
#define   CAN_NODE_NPCR_RXSEL_Pos  	(0U)
#define   CAN_NODE_NPCR_RXSEL_Msk  	(0x00000007U  << CAN_NODE_NPCR_RXSEL_Pos)

#define   CAN_NODE_NPCR_LBM_Pos  	(8U)
#define   CAN_NODE_NPCR_LBM_Msk  	(0x00000001U  << CAN_NODE_NPCR_LBM_Pos)

/* CAN_NODE_NBTRx  =  Node x Bit Timing Register*/
#define   CAN_NODE_NBTR_BRP_Pos  	(0U)
#define   CAN_NODE_NBTR_BRP_Msk  	(0x0000003FU  << CAN_NODE_NBTR_BRP_Pos)

#define   CAN_NODE_NBTR_SJW_Pos  	(6U)
#define   CAN_NODE_NBTR_SJW_Msk  	(0x00000003U  << CAN_NODE_NBTR_SJW_Pos)

#define   CAN_NODE_NBTR_TSEG1_Pos  	(8U)
#define   CAN_NODE_NBTR_TSEG1_Msk  	(0x0000000FU  << CAN_NODE_NBTR_TSEG1_Pos)

#define   CAN_NODE_NBTR_TSEG2_Pos  	(12U)
#define   CAN_NODE_NBTR_TSEG2_Msk  	(0x00000007U  << CAN_NODE_NBTR_TSEG2_Pos)

#define   CAN_NODE_NBTR_DIV8_Pos  	(15U)
#define   CAN_NODE_NBTR_DIV8_Msk  	(0x00000001U  << CAN_NODE_NBTR_DIV8_Pos)

/* CAN_NODE_NECNTx  =  Node x Error Counter Register*/
#define   CAN_NODE_NECNT_REC_Pos  	(0U)
#define   CAN_NODE_NECNT_REC_Msk  	(0x000000FFU  << CAN_NODE_NECNT_REC_Pos)

#define   CAN_NODE_NECNT_TEC_Pos  	(8U)
#define   CAN_NODE_NECNT_TEC_Msk  	(0x000000FFU  << CAN_NODE_NECNT_TEC_Pos)

#define   CAN_NODE_NECNT_EWRNLVL_Pos  	(16U)
#define   CAN_NODE_NECNT_EWRNLVL_Msk  	(0x000000FFU  << CAN_NODE_NECNT_EWRNLVL_Pos)

#define   CAN_NODE_NECNT_LETD_Pos  	(24U)
#define   CAN_NODE_NECNT_LETD_Msk  	(0x00000001U  << CAN_NODE_NECNT_LETD_Pos)

#define   CAN_NODE_NECNT_LEINC_Pos  	(25U)
#define   CAN_NODE_NECNT_LEINC_Msk  	(0x00000001U  << CAN_NODE_NECNT_LEINC_Pos)

/* CAN_NODE_NFCRx  =  Node x Frame Counter Register*/
#define   CAN_NODE_NFCR_CFC_Pos  	(0U)
#define   CAN_NODE_NFCR_CFC_Msk  	(0x0000FFFFU  << CAN_NODE_NFCR_CFC_Pos)

#define   CAN_NODE_NFCR_CFSEL_Pos  	(16U)
#define   CAN_NODE_NFCR_CFSEL_Msk  	(0x00000007U  << CAN_NODE_NFCR_CFSEL_Pos)

#define   CAN_NODE_NFCR_CFMOD_Pos  	(19U)
#define   CAN_NODE_NFCR_CFMOD_Msk  	(0x00000003U  << CAN_NODE_NFCR_CFMOD_Pos)

#define   CAN_NODE_NFCR_CFCIE_Pos  	(22U)
#define   CAN_NODE_NFCR_CFCIE_Msk  	(0x00000001U  << CAN_NODE_NFCR_CFCIE_Pos)

#define   CAN_NODE_NFCR_CFCOV_Pos  	(23U)
#define   CAN_NODE_NFCR_CFCOV_Msk  	(0x00000001U  << CAN_NODE_NFCR_CFCOV_Pos)

/* CAN_MO_MOFCRx  =  Message Object x Function Control Register*/
#define   CAN_MO_MOFCR_MMC_Pos  	(0U)
#define   CAN_MO_MOFCR_MMC_Msk  	(0x0000000FU  << CAN_MO_MOFCR_MMC_Pos)

#define   CAN_MO_MOFCR_GDFS_Pos  	(8U)
#define   CAN_MO_MOFCR_GDFS_Msk  	(0x00000001U  << CAN_MO_MOFCR_GDFS_Pos)

#define   CAN_MO_MOFCR_IDC_Pos  	(9U)
#define   CAN_MO_MOFCR_IDC_Msk  	(0x00000001U  << CAN_MO_MOFCR_IDC_Pos)

#define   CAN_MO_MOFCR_DLCC_Pos  	(10U)
#define   CAN_MO_MOFCR_DLCC_Msk  	(0x00000001U  << CAN_MO_MOFCR_DLCC_Pos)

#define   CAN_MO_MOFCR_DATC_Pos  	(11U)
#define   CAN_MO_MOFCR_DATC_Msk  	(0x00000001U  << CAN_MO_MOFCR_DATC_Pos)

#define   CAN_MO_MOFCR_RXIE_Pos  	(16U)
#define   CAN_MO_MOFCR_RXIE_Msk  	(0x00000001U  << CAN_MO_MOFCR_RXIE_Pos)

#define   CAN_MO_MOFCR_TXIE_Pos  	(17U)
#define   CAN_MO_MOFCR_TXIE_Msk  	(0x00000001U  << CAN_MO_MOFCR_TXIE_Pos)

#define   CAN_MO_MOFCR_OVIE_Pos  	(18U)
#define   CAN_MO_MOFCR_OVIE_Msk  	(0x00000001U  << CAN_MO_MOFCR_OVIE_Pos)

#define   CAN_MO_MOFCR_FRREN_Pos  	(20U)
#define   CAN_MO_MOFCR_FRREN_Msk  	(0x00000001U  << CAN_MO_MOFCR_FRREN_Pos)

#define   CAN_MO_MOFCR_RMM_Pos  	(21U)
#define   CAN_MO_MOFCR_RMM_Msk  	(0x00000001U  << CAN_MO_MOFCR_RMM_Pos)

#define   CAN_MO_MOFCR_SDT_Pos  	(22U)
#define   CAN_MO_MOFCR_SDT_Msk  	(0x00000001U  << CAN_MO_MOFCR_SDT_Pos)

#define   CAN_MO_MOFCR_STT_Pos  	(23U)
#define   CAN_MO_MOFCR_STT_Msk  	(0x00000001U  << CAN_MO_MOFCR_STT_Pos)

#define   CAN_MO_MOFCR_DLC_Pos  	(24U)
#define   CAN_MO_MOFCR_DLC_Msk  	(0x0000000FU  << CAN_MO_MOFCR_DLC_Pos)

/* CAN_MO_MOFGPRx  =  Message Object x FIFO/Gateway Pointer Register*/
#define   CAN_MO_MOFGPR_BOT_Pos  	(0U)
#define   CAN_MO_MOFGPR_BOT_Msk  	(0x000000FFU  << CAN_MO_MOFGPR_BOT_Pos)

#define   CAN_MO_MOFGPR_TOP_Pos  	(8U)
#define   CAN_MO_MOFGPR_TOP_Msk  	(0x000000FFU  << CAN_MO_MOFGPR_TOP_Pos)

#define   CAN_MO_MOFGPR_CUR_Pos  	(16U)
#define   CAN_MO_MOFGPR_CUR_Msk  	(0x000000FFU  << CAN_MO_MOFGPR_CUR_Pos)

#define   CAN_MO_MOFGPR_SEL_Pos  	(24U)
#define   CAN_MO_MOFGPR_SEL_Msk  	(0x000000FFU  << CAN_MO_MOFGPR_SEL_Pos)

/* CAN_MO_MOIPRx  =  Message Object x Interrupt Pointer Register*/
#define   CAN_MO_MOIPR_RXINP_Pos  	(0U)
#define   CAN_MO_MOIPR_RXINP_Msk  	(0x00000007U  << CAN_MO_MOIPR_RXINP_Pos)

#define   CAN_MO_MOIPR_TXINP_Pos  	(4U)
#define   CAN_MO_MOIPR_TXINP_Msk  	(0x00000007U  << CAN_MO_MOIPR_TXINP_Pos)

#define   CAN_MO_MOIPR_MPN_Pos  	(8U)
#define   CAN_MO_MOIPR_MPN_Msk  	(0x000000FFU  << CAN_MO_MOIPR_MPN_Pos)

#define   CAN_MO_MOIPR_CFCVAL_Pos  	(16U)
#define   CAN_MO_MOIPR_CFCVAL_Msk  	(0x0000FFFFU  << CAN_MO_MOIPR_CFCVAL_Pos)

/* CAN_MO_MOAMRx  =  Message Object x Acceptance Mask Register*/
#define   CAN_MO_MOAMR_AM_Pos  	(0U)
#define   CAN_MO_MOAMR_AM_Msk  	(0x1FFFFFFFU  << CAN_MO_MOAMR_AM_Pos)

#define   CAN_MO_MOAMR_MIDE_Pos  	(29U)
#define   CAN_MO_MOAMR_MIDE_Msk  	(0x00000001U  << CAN_MO_MOAMR_MIDE_Pos)

/* CAN_MO_MODATALx  =  Message Object x Data Register Low*/
#define   CAN_MO_MODATAL_DB0_Pos  	(0U)
#define   CAN_MO_MODATAL_DB0_Msk  	(0x000000FFU  << CAN_MO_MODATAL_DB0_Pos)

#define   CAN_MO_MODATAL_DB1_Pos  	(8U)
#define   CAN_MO_MODATAL_DB1_Msk  	(0x000000FFU  << CAN_MO_MODATAL_DB1_Pos)

#define   CAN_MO_MODATAL_DB2_Pos  	(16U)
#define   CAN_MO_MODATAL_DB2_Msk  	(0x000000FFU  << CAN_MO_MODATAL_DB2_Pos)

#define   CAN_MO_MODATAL_DB3_Pos  	(24U)
#define   CAN_MO_MODATAL_DB3_Msk  	(0x000000FFU  << CAN_MO_MODATAL_DB3_Pos)

/* CAN_MO_MODATAHx  =  Message Object x Data Register High*/
#define   CAN_MO_MODATAH_DB4_Pos  	(0U)
#define   CAN_MO_MODATAH_DB4_Msk  	(0x000000FFU  << CAN_MO_MODATAH_DB4_Pos)

#define   CAN_MO_MODATAH_DB5_Pos  	(8U)
#define   CAN_MO_MODATAH_DB5_Msk  	(0x000000FFU  << CAN_MO_MODATAH_DB5_Pos)

#define   CAN_MO_MODATAH_DB6_Pos  	(16U)
#define   CAN_MO_MODATAH_DB6_Msk  	(0x000000FFU  << CAN_MO_MODATAH_DB6_Pos)

#define   CAN_MO_MODATAH_DB7_Pos  	(24U)
#define   CAN_MO_MODATAH_DB7_Msk  	(0x000000FFU  << CAN_MO_MODATAH_DB7_Pos)

/* CAN_MO_MOARx  =  Message Object x Arbitration Register*/
#define   CAN_MO_MOAR_ID_Pos  	(0U)
#define   CAN_MO_MOAR_ID_Msk  	(0x1FFFFFFFU  << CAN_MO_MOAR_ID_Pos)

#define   CAN_MO_MOAR_IDE_Pos  	(29U)
#define   CAN_MO_MOAR_IDE_Msk  	(0x00000001U  << CAN_MO_MOAR_IDE_Pos)

#define   CAN_MO_MOAR_PRI_Pos  	(30U)
#define   CAN_MO_MOAR_PRI_Msk  	(0x00000003U  << CAN_MO_MOAR_PRI_Pos)

/* CAN_MO_MOCTRx  =  Message Object x Control Register*/
#define   CAN_MO_MOCTR_RESRXPND_Pos  	(0U)
#define   CAN_MO_MOCTR_RESRXPND_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESRXPND_Pos)

#define   CAN_MO_MOCTR_SETRXPND_Pos  	(16U)
#define   CAN_MO_MOCTR_SETRXPND_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETRXPND_Pos)

#define   CAN_MO_MOCTR_RESTXPND_Pos  	(1U)
#define   CAN_MO_MOCTR_RESTXPND_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESTXPND_Pos)

#define   CAN_MO_MOCTR_SETTXPND_Pos  	(17U)
#define   CAN_MO_MOCTR_SETTXPND_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETTXPND_Pos)

#define   CAN_MO_MOCTR_RESRXUPD_Pos  	(2U)
#define   CAN_MO_MOCTR_RESRXUPD_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESRXUPD_Pos)

#define   CAN_MO_MOCTR_SETRXUPD_Pos  	(18U)
#define   CAN_MO_MOCTR_SETRXUPD_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETRXUPD_Pos)

#define   CAN_MO_MOCTR_RESNEWDAT_Pos  	(3U)
#define   CAN_MO_MOCTR_RESNEWDAT_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESNEWDAT_Pos)

#define   CAN_MO_MOCTR_SETNEWDAT_Pos  	(19U)
#define   CAN_MO_MOCTR_SETNEWDAT_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETNEWDAT_Pos)

#define   CAN_MO_MOCTR_RESMSGLST_Pos  	(4U)
#define   CAN_MO_MOCTR_RESMSGLST_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESMSGLST_Pos)

#define   CAN_MO_MOCTR_SETMSGLST_Pos  	(20U)
#define   CAN_MO_MOCTR_SETMSGLST_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETMSGLST_Pos)

#define   CAN_MO_MOCTR_RESMSGVAL_Pos  	(5U)
#define   CAN_MO_MOCTR_RESMSGVAL_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESMSGVAL_Pos)

#define   CAN_MO_MOCTR_SETMSGVAL_Pos  	(21U)
#define   CAN_MO_MOCTR_SETMSGVAL_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETMSGVAL_Pos)

#define   CAN_MO_MOCTR_RESRTSEL_Pos  	(6U)
#define   CAN_MO_MOCTR_RESRTSEL_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESRTSEL_Pos)

#define   CAN_MO_MOCTR_SETRTSEL_Pos  	(22U)
#define   CAN_MO_MOCTR_SETRTSEL_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETRTSEL_Pos)

#define   CAN_MO_MOCTR_RESRXEN_Pos  	(7U)
#define   CAN_MO_MOCTR_RESRXEN_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESRXEN_Pos)

#define   CAN_MO_MOCTR_SETRXEN_Pos  	(23U)
#define   CAN_MO_MOCTR_SETRXEN_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETRXEN_Pos)

#define   CAN_MO_MOCTR_RESTXRQ_Pos  	(8U)
#define   CAN_MO_MOCTR_RESTXRQ_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESTXRQ_Pos)

#define   CAN_MO_MOCTR_SETTXRQ_Pos  	(24U)
#define   CAN_MO_MOCTR_SETTXRQ_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETTXRQ_Pos)

#define   CAN_MO_MOCTR_RESTXEN0_Pos  	(9U)
#define   CAN_MO_MOCTR_RESTXEN0_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESTXEN0_Pos)

#define   CAN_MO_MOCTR_SETTXEN0_Pos  	(25U)
#define   CAN_MO_MOCTR_SETTXEN0_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETTXEN0_Pos)

#define   CAN_MO_MOCTR_RESTXEN1_Pos  	(10U)
#define   CAN_MO_MOCTR_RESTXEN1_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESTXEN1_Pos)

#define   CAN_MO_MOCTR_SETTXEN1_Pos  	(26U)
#define   CAN_MO_MOCTR_SETTXEN1_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETTXEN1_Pos)

#define   CAN_MO_MOCTR_RESDIR_Pos  	(11U)
#define   CAN_MO_MOCTR_RESDIR_Msk  	(0x00000001U  << CAN_MO_MOCTR_RESDIR_Pos)

#define   CAN_MO_MOCTR_SETDIR_Pos  	(27U)
#define   CAN_MO_MOCTR_SETDIR_Msk  	(0x00000001U  << CAN_MO_MOCTR_SETDIR_Pos)

/* CAN_MO_MOSTATx  =  Message Object x Status Register*/
#define   CAN_MO_MOSTAT_RXPND_Pos  	(0U)
#define   CAN_MO_MOSTAT_RXPND_Msk  	(0x00000001U  << CAN_MO_MOSTAT_RXPND_Pos)

#define   CAN_MO_MOSTAT_TXPND_Pos  	(1U)
#define   CAN_MO_MOSTAT_TXPND_Msk  	(0x00000001U  << CAN_MO_MOSTAT_TXPND_Pos)

#define   CAN_MO_MOSTAT_RXUPD_Pos  	(2U)
#define   CAN_MO_MOSTAT_RXUPD_Msk  	(0x00000001U  << CAN_MO_MOSTAT_RXUPD_Pos)

#define   CAN_MO_MOSTAT_NEWDAT_Pos  	(3U)
#define   CAN_MO_MOSTAT_NEWDAT_Msk  	(0x00000001U  << CAN_MO_MOSTAT_NEWDAT_Pos)

#define   CAN_MO_MOSTAT_MSGLST_Pos  	(4U)
#define   CAN_MO_MOSTAT_MSGLST_Msk  	(0x00000001U  << CAN_MO_MOSTAT_MSGLST_Pos)

#define   CAN_MO_MOSTAT_MSGVAL_Pos  	(5U)
#define   CAN_MO_MOSTAT_MSGVAL_Msk  	(0x00000001U  << CAN_MO_MOSTAT_MSGVAL_Pos)

#define   CAN_MO_MOSTAT_RTSEL_Pos  	(6U)
#define   CAN_MO_MOSTAT_RTSEL_Msk  	(0x00000001U  << CAN_MO_MOSTAT_RTSEL_Pos)

#define   CAN_MO_MOSTAT_RXEN_Pos  	(7U)
#define   CAN_MO_MOSTAT_RXEN_Msk  	(0x00000001U  << CAN_MO_MOSTAT_RXEN_Pos)

#define   CAN_MO_MOSTAT_TXRQ_Pos  	(8U)
#define   CAN_MO_MOSTAT_TXRQ_Msk  	(0x00000001U  << CAN_MO_MOSTAT_TXRQ_Pos)

#define   CAN_MO_MOSTAT_TXEN0_Pos  	(9U)
#define   CAN_MO_MOSTAT_TXEN0_Msk  	(0x00000001U  << CAN_MO_MOSTAT_TXEN0_Pos)

#define   CAN_MO_MOSTAT_TXEN1_Pos  	(10U)
#define   CAN_MO_MOSTAT_TXEN1_Msk  	(0x00000001U  << CAN_MO_MOSTAT_TXEN1_Pos)

#define   CAN_MO_MOSTAT_DIR_Pos  	(11U)
#define   CAN_MO_MOSTAT_DIR_Msk  	(0x00000001U  << CAN_MO_MOSTAT_DIR_Pos)

#define   CAN_MO_MOSTAT_LIST_Pos  	(12U)
#define   CAN_MO_MOSTAT_LIST_Msk  	(0x0000000FU  << CAN_MO_MOSTAT_LIST_Pos)

#define   CAN_MO_MOSTAT_PPREV_Pos  	(16U)
#define   CAN_MO_MOSTAT_PPREV_Msk  	(0x000000FFU  << CAN_MO_MOSTAT_PPREV_Pos)

#define   CAN_MO_MOSTAT_PNEXT_Pos  	(24U)
#define   CAN_MO_MOSTAT_PNEXT_Msk  	(0x000000FFU  << CAN_MO_MOSTAT_PNEXT_Pos)



/***   CCU4x Bit Fileds *******************/
/***************************************************************************/


/* CCU4_GCTRL  =  Global Control Register*/
#define   CCU4_GCTRL_PRBC_Pos  	(0U)
#define   CCU4_GCTRL_PRBC_Msk  	(0x00000007U  << CCU4_GCTRL_PRBC_Pos)

#define   CCU4_GCTRL_PCIS_Pos  	(4U)
#define   CCU4_GCTRL_PCIS_Msk  	(0x00000003U  << CCU4_GCTRL_PCIS_Pos)

#define   CCU4_GCTRL_SUSCFG_Pos  	(8U)
#define   CCU4_GCTRL_SUSCFG_Msk  	(0x00000003U  << CCU4_GCTRL_SUSCFG_Pos)

#define   CCU4_GCTRL_MSE0_Pos  	(10U)
#define   CCU4_GCTRL_MSE0_Msk  	(0x00000001U  << CCU4_GCTRL_MSE0_Pos)

#define   CCU4_GCTRL_MSE1_Pos  	(11U)
#define   CCU4_GCTRL_MSE1_Msk  	(0x00000001U  << CCU4_GCTRL_MSE1_Pos)

#define   CCU4_GCTRL_MSE2_Pos  	(12U)
#define   CCU4_GCTRL_MSE2_Msk  	(0x00000001U  << CCU4_GCTRL_MSE2_Pos)

#define   CCU4_GCTRL_MSE3_Pos  	(13U)
#define   CCU4_GCTRL_MSE3_Msk  	(0x00000001U  << CCU4_GCTRL_MSE3_Pos)

#define   CCU4_GCTRL_MSDE_Pos  	(14U)
#define   CCU4_GCTRL_MSDE_Msk  	(0x00000003U  << CCU4_GCTRL_MSDE_Pos)

/* CCU4_GSTAT  =  Global Status Register*/
#define   CCU4_GSTAT_S0I_Pos  	(0U)
#define   CCU4_GSTAT_S0I_Msk  	(0x00000001U  << CCU4_GSTAT_S0I_Pos)

#define   CCU4_GSTAT_S1I_Pos  	(1U)
#define   CCU4_GSTAT_S1I_Msk  	(0x00000001U  << CCU4_GSTAT_S1I_Pos)

#define   CCU4_GSTAT_S2I_Pos  	(2U)
#define   CCU4_GSTAT_S2I_Msk  	(0x00000001U  << CCU4_GSTAT_S2I_Pos)

#define   CCU4_GSTAT_S3I_Pos  	(3U)
#define   CCU4_GSTAT_S3I_Msk  	(0x00000001U  << CCU4_GSTAT_S3I_Pos)

#define   CCU4_GSTAT_PRB_Pos  	(8U)
#define   CCU4_GSTAT_PRB_Msk  	(0x00000001U  << CCU4_GSTAT_PRB_Pos)

/* CCU4_GIDLS  =  Global Idle Set*/
#define   CCU4_GIDLS_SS0I_Pos  	(0U)
#define   CCU4_GIDLS_SS0I_Msk  	(0x00000001U  << CCU4_GIDLS_SS0I_Pos)

#define   CCU4_GIDLS_SS1I_Pos  	(1U)
#define   CCU4_GIDLS_SS1I_Msk  	(0x00000001U  << CCU4_GIDLS_SS1I_Pos)

#define   CCU4_GIDLS_SS2I_Pos  	(2U)
#define   CCU4_GIDLS_SS2I_Msk  	(0x00000001U  << CCU4_GIDLS_SS2I_Pos)

#define   CCU4_GIDLS_SS3I_Pos  	(3U)
#define   CCU4_GIDLS_SS3I_Msk  	(0x00000001U  << CCU4_GIDLS_SS3I_Pos)

#define   CCU4_GIDLS_CPRB_Pos  	(8U)
#define   CCU4_GIDLS_CPRB_Msk  	(0x00000001U  << CCU4_GIDLS_CPRB_Pos)

#define   CCU4_GIDLS_PSIC_Pos  	(9U)
#define   CCU4_GIDLS_PSIC_Msk  	(0x00000001U  << CCU4_GIDLS_PSIC_Pos)

/* CCU4_GIDLC  =  Global Idle Clear*/
#define   CCU4_GIDLC_CS0I_Pos  	(0U)
#define   CCU4_GIDLC_CS0I_Msk  	(0x00000001U  << CCU4_GIDLC_CS0I_Pos)

#define   CCU4_GIDLC_CS1I_Pos  	(1U)
#define   CCU4_GIDLC_CS1I_Msk  	(0x00000001U  << CCU4_GIDLC_CS1I_Pos)

#define   CCU4_GIDLC_CS2I_Pos  	(2U)
#define   CCU4_GIDLC_CS2I_Msk  	(0x00000001U  << CCU4_GIDLC_CS2I_Pos)

#define   CCU4_GIDLC_CS3I_Pos  	(3U)
#define   CCU4_GIDLC_CS3I_Msk  	(0x00000001U  << CCU4_GIDLC_CS3I_Pos)

#define   CCU4_GIDLC_SPRB_Pos  	(8U)
#define   CCU4_GIDLC_SPRB_Msk  	(0x00000001U  << CCU4_GIDLC_SPRB_Pos)

/* CCU4_GCSS  =  Global Channel Set*/
#define   CCU4_GCSS_S0SE_Pos  	(0U)
#define   CCU4_GCSS_S0SE_Msk  	(0x00000001U  << CCU4_GCSS_S0SE_Pos)

#define   CCU4_GCSS_S0DSE_Pos  	(1U)
#define   CCU4_GCSS_S0DSE_Msk  	(0x00000001U  << CCU4_GCSS_S0DSE_Pos)

#define   CCU4_GCSS_S0PSE_Pos  	(2U)
#define   CCU4_GCSS_S0PSE_Msk  	(0x00000001U  << CCU4_GCSS_S0PSE_Pos)

#define   CCU4_GCSS_S1SE_Pos  	(4U)
#define   CCU4_GCSS_S1SE_Msk  	(0x00000001U  << CCU4_GCSS_S1SE_Pos)

#define   CCU4_GCSS_S1DSE_Pos  	(5U)
#define   CCU4_GCSS_S1DSE_Msk  	(0x00000001U  << CCU4_GCSS_S1DSE_Pos)

#define   CCU4_GCSS_S1PSE_Pos  	(6U)
#define   CCU4_GCSS_S1PSE_Msk  	(0x00000001U  << CCU4_GCSS_S1PSE_Pos)

#define   CCU4_GCSS_S2SE_Pos  	(8U)
#define   CCU4_GCSS_S2SE_Msk  	(0x00000001U  << CCU4_GCSS_S2SE_Pos)

#define   CCU4_GCSS_S2DSE_Pos  	(9U)
#define   CCU4_GCSS_S2DSE_Msk  	(0x00000001U  << CCU4_GCSS_S2DSE_Pos)

#define   CCU4_GCSS_S2PSE_Pos  	(10U)
#define   CCU4_GCSS_S2PSE_Msk  	(0x00000001U  << CCU4_GCSS_S2PSE_Pos)

#define   CCU4_GCSS_S3SE_Pos  	(12U)
#define   CCU4_GCSS_S3SE_Msk  	(0x00000001U  << CCU4_GCSS_S3SE_Pos)

#define   CCU4_GCSS_S3DSE_Pos  	(13U)
#define   CCU4_GCSS_S3DSE_Msk  	(0x00000001U  << CCU4_GCSS_S3DSE_Pos)

#define   CCU4_GCSS_S3PSE_Pos  	(14U)
#define   CCU4_GCSS_S3PSE_Msk  	(0x00000001U  << CCU4_GCSS_S3PSE_Pos)

#define   CCU4_GCSS_S0STS_Pos  	(16U)
#define   CCU4_GCSS_S0STS_Msk  	(0x00000001U  << CCU4_GCSS_S0STS_Pos)

#define   CCU4_GCSS_S1STS_Pos  	(17U)
#define   CCU4_GCSS_S1STS_Msk  	(0x00000001U  << CCU4_GCSS_S1STS_Pos)

#define   CCU4_GCSS_S2STS_Pos  	(18U)
#define   CCU4_GCSS_S2STS_Msk  	(0x00000001U  << CCU4_GCSS_S2STS_Pos)

#define   CCU4_GCSS_S3STS_Pos  	(19U)
#define   CCU4_GCSS_S3STS_Msk  	(0x00000001U  << CCU4_GCSS_S3STS_Pos)

/* CCU4_GCSC  =  Global Channel Clear*/
#define   CCU4_GCSC_S0SC_Pos  	(0U)
#define   CCU4_GCSC_S0SC_Msk  	(0x00000001U  << CCU4_GCSC_S0SC_Pos)

#define   CCU4_GCSC_S0DSC_Pos  	(1U)
#define   CCU4_GCSC_S0DSC_Msk  	(0x00000001U  << CCU4_GCSC_S0DSC_Pos)

#define   CCU4_GCSC_S0PSC_Pos  	(2U)
#define   CCU4_GCSC_S0PSC_Msk  	(0x00000001U  << CCU4_GCSC_S0PSC_Pos)

#define   CCU4_GCSC_S1SC_Pos  	(4U)
#define   CCU4_GCSC_S1SC_Msk  	(0x00000001U  << CCU4_GCSC_S1SC_Pos)

#define   CCU4_GCSC_S1DSC_Pos  	(5U)
#define   CCU4_GCSC_S1DSC_Msk  	(0x00000001U  << CCU4_GCSC_S1DSC_Pos)

#define   CCU4_GCSC_S1PSC_Pos  	(6U)
#define   CCU4_GCSC_S1PSC_Msk  	(0x00000001U  << CCU4_GCSC_S1PSC_Pos)

#define   CCU4_GCSC_S2SC_Pos  	(8U)
#define   CCU4_GCSC_S2SC_Msk  	(0x00000001U  << CCU4_GCSC_S2SC_Pos)

#define   CCU4_GCSC_S2DSC_Pos  	(9U)
#define   CCU4_GCSC_S2DSC_Msk  	(0x00000001U  << CCU4_GCSC_S2DSC_Pos)

#define   CCU4_GCSC_S2PSC_Pos  	(10U)
#define   CCU4_GCSC_S2PSC_Msk  	(0x00000001U  << CCU4_GCSC_S2PSC_Pos)

#define   CCU4_GCSC_S3SC_Pos  	(12U)
#define   CCU4_GCSC_S3SC_Msk  	(0x00000001U  << CCU4_GCSC_S3SC_Pos)

#define   CCU4_GCSC_S3DSC_Pos  	(13U)
#define   CCU4_GCSC_S3DSC_Msk  	(0x00000001U  << CCU4_GCSC_S3DSC_Pos)

#define   CCU4_GCSC_S3PSC_Pos  	(14U)
#define   CCU4_GCSC_S3PSC_Msk  	(0x00000001U  << CCU4_GCSC_S3PSC_Pos)

#define   CCU4_GCSC_S0STC_Pos  	(16U)
#define   CCU4_GCSC_S0STC_Msk  	(0x00000001U  << CCU4_GCSC_S0STC_Pos)

#define   CCU4_GCSC_S1STC_Pos  	(17U)
#define   CCU4_GCSC_S1STC_Msk  	(0x00000001U  << CCU4_GCSC_S1STC_Pos)

#define   CCU4_GCSC_S2STC_Pos  	(18U)
#define   CCU4_GCSC_S2STC_Msk  	(0x00000001U  << CCU4_GCSC_S2STC_Pos)

#define   CCU4_GCSC_S3STC_Pos  	(19U)
#define   CCU4_GCSC_S3STC_Msk  	(0x00000001U  << CCU4_GCSC_S3STC_Pos)

/* CCU4_GCST  =  Global Channel Status*/
#define   CCU4_GCST_S0SS_Pos  	(0U)
#define   CCU4_GCST_S0SS_Msk  	(0x00000001U  << CCU4_GCST_S0SS_Pos)

#define   CCU4_GCST_S0DSS_Pos  	(1U)
#define   CCU4_GCST_S0DSS_Msk  	(0x00000001U  << CCU4_GCST_S0DSS_Pos)

#define   CCU4_GCST_S0PSS_Pos  	(2U)
#define   CCU4_GCST_S0PSS_Msk  	(0x00000001U  << CCU4_GCST_S0PSS_Pos)

#define   CCU4_GCST_S1SS_Pos  	(4U)
#define   CCU4_GCST_S1SS_Msk  	(0x00000001U  << CCU4_GCST_S1SS_Pos)

#define   CCU4_GCST_S1DSS_Pos  	(5U)
#define   CCU4_GCST_S1DSS_Msk  	(0x00000001U  << CCU4_GCST_S1DSS_Pos)

#define   CCU4_GCST_S1PSS_Pos  	(6U)
#define   CCU4_GCST_S1PSS_Msk  	(0x00000001U  << CCU4_GCST_S1PSS_Pos)

#define   CCU4_GCST_S2SS_Pos  	(8U)
#define   CCU4_GCST_S2SS_Msk  	(0x00000001U  << CCU4_GCST_S2SS_Pos)

#define   CCU4_GCST_S2DSS_Pos  	(9U)
#define   CCU4_GCST_S2DSS_Msk  	(0x00000001U  << CCU4_GCST_S2DSS_Pos)

#define   CCU4_GCST_S2PSS_Pos  	(10U)
#define   CCU4_GCST_S2PSS_Msk  	(0x00000001U  << CCU4_GCST_S2PSS_Pos)

#define   CCU4_GCST_S3SS_Pos  	(12U)
#define   CCU4_GCST_S3SS_Msk  	(0x00000001U  << CCU4_GCST_S3SS_Pos)

#define   CCU4_GCST_S3DSS_Pos  	(13U)
#define   CCU4_GCST_S3DSS_Msk  	(0x00000001U  << CCU4_GCST_S3DSS_Pos)

#define   CCU4_GCST_S3PSS_Pos  	(14U)
#define   CCU4_GCST_S3PSS_Msk  	(0x00000001U  << CCU4_GCST_S3PSS_Pos)

#define   CCU4_GCST_CC40ST_Pos  	(16U)
#define   CCU4_GCST_CC40ST_Msk  	(0x00000001U  << CCU4_GCST_CC40ST_Pos)

#define   CCU4_GCST_CC41ST_Pos  	(17U)
#define   CCU4_GCST_CC41ST_Msk  	(0x00000001U  << CCU4_GCST_CC41ST_Pos)

#define   CCU4_GCST_CC42ST_Pos  	(18U)
#define   CCU4_GCST_CC42ST_Msk  	(0x00000001U  << CCU4_GCST_CC42ST_Pos)

#define   CCU4_GCST_CC43ST_Pos  	(19U)
#define   CCU4_GCST_CC43ST_Msk  	(0x00000001U  << CCU4_GCST_CC43ST_Pos)

/* CCU4_ECRD  =  Extended Capture Mode Read*/
#define   CCU4_ECRD_CAPV_Pos  	(0U)
#define   CCU4_ECRD_CAPV_Msk  	(0x0000FFFFU  << CCU4_ECRD_CAPV_Pos)

#define   CCU4_ECRD_FPCV_Pos  	(16U)
#define   CCU4_ECRD_FPCV_Msk  	(0x0000000FU  << CCU4_ECRD_FPCV_Pos)

#define   CCU4_ECRD_SPTR_Pos  	(20U)
#define   CCU4_ECRD_SPTR_Msk  	(0x00000003U  << CCU4_ECRD_SPTR_Pos)

#define   CCU4_ECRD_VPTR_Pos  	(22U)
#define   CCU4_ECRD_VPTR_Msk  	(0x00000003U  << CCU4_ECRD_VPTR_Pos)

#define   CCU4_ECRD_FFL_Pos  	(24U)
#define   CCU4_ECRD_FFL_Msk  	(0x00000001U  << CCU4_ECRD_FFL_Pos)

/* CCU4_MIDR  =  Module Identification*/
#define   CCU4_MIDR_MODR_Pos  	(0U)
#define   CCU4_MIDR_MODR_Msk  	(0x000000FFU  << CCU4_MIDR_MODR_Pos)

#define   CCU4_MIDR_MODT_Pos  	(8U)
#define   CCU4_MIDR_MODT_Msk  	(0x000000FFU  << CCU4_MIDR_MODT_Pos)

#define   CCU4_MIDR_MODN_Pos  	(16U)
#define   CCU4_MIDR_MODN_Msk  	(0x0000FFFFU  << CCU4_MIDR_MODN_Pos)

/* CCU4_CC4_INS  =  Input Selector Configuration*/
#define   CCU4_CC4_INS_EV0IS_Pos  	(0U)
#define   CCU4_CC4_INS_EV0IS_Msk  	(0x0000000FU  << CCU4_CC4_INS_EV0IS_Pos)

#define   CCU4_CC4_INS_EV1IS_Pos  	(4U)
#define   CCU4_CC4_INS_EV1IS_Msk  	(0x0000000FU  << CCU4_CC4_INS_EV1IS_Pos)

#define   CCU4_CC4_INS_EV2S_Pos  	(8U)
#define   CCU4_CC4_INS_EV2S_Msk  	(0x0000000FU  << CCU4_CC4_INS_EV2S_Pos)

#define   CCU4_CC4_INS_EV0EM_Pos  	(16U)
#define   CCU4_CC4_INS_EV0EM_Msk  	(0x00000003U  << CCU4_CC4_INS_EV0EM_Pos)

#define   CCU4_CC4_INS_EV1EM_Pos  	(18U)
#define   CCU4_CC4_INS_EV1EM_Msk  	(0x00000003U  << CCU4_CC4_INS_EV1EM_Pos)

#define   CCU4_CC4_INS_EV2EM_Pos  	(20U)
#define   CCU4_CC4_INS_EV2EM_Msk  	(0x00000003U  << CCU4_CC4_INS_EV2EM_Pos)

#define   CCU4_CC4_INS_EV0LM_Pos  	(22U)
#define   CCU4_CC4_INS_EV0LM_Msk  	(0x00000001U  << CCU4_CC4_INS_EV0LM_Pos)

#define   CCU4_CC4_INS_EV1LM_Pos  	(23U)
#define   CCU4_CC4_INS_EV1LM_Msk  	(0x00000001U  << CCU4_CC4_INS_EV1LM_Pos)

#define   CCU4_CC4_INS_EV2LM_Pos  	(24U)
#define   CCU4_CC4_INS_EV2LM_Msk  	(0x00000001U  << CCU4_CC4_INS_EV2LM_Pos)

#define   CCU4_CC4_INS_LPF0M_Pos  	(25U)
#define   CCU4_CC4_INS_LPF0M_Msk  	(0x00000003U  << CCU4_CC4_INS_LPF0M_Pos)

#define   CCU4_CC4_INS_LPF1M_Pos  	(27U)
#define   CCU4_CC4_INS_LPF1M_Msk  	(0x00000003U  << CCU4_CC4_INS_LPF1M_Pos)

#define   CCU4_CC4_INS_LPF2M_Pos  	(29U)
#define   CCU4_CC4_INS_LPF2M_Msk  	(0x00000003U  << CCU4_CC4_INS_LPF2M_Pos)

/* CCU4_CC4_CMC  =  Connection Matrix Control*/
#define   CCU4_CC4_CMC_STRTS_Pos  	(0U)
#define   CCU4_CC4_CMC_STRTS_Msk  	(0x00000003U  << CCU4_CC4_CMC_STRTS_Pos)

#define   CCU4_CC4_CMC_ENDS_Pos  	(2U)
#define   CCU4_CC4_CMC_ENDS_Msk  	(0x00000003U  << CCU4_CC4_CMC_ENDS_Pos)

#define   CCU4_CC4_CMC_CAP0S_Pos  	(4U)
#define   CCU4_CC4_CMC_CAP0S_Msk  	(0x00000003U  << CCU4_CC4_CMC_CAP0S_Pos)

#define   CCU4_CC4_CMC_CAP1S_Pos  	(6U)
#define   CCU4_CC4_CMC_CAP1S_Msk  	(0x00000003U  << CCU4_CC4_CMC_CAP1S_Pos)

#define   CCU4_CC4_CMC_GATES_Pos  	(8U)
#define   CCU4_CC4_CMC_GATES_Msk  	(0x00000003U  << CCU4_CC4_CMC_GATES_Pos)

#define   CCU4_CC4_CMC_UDS_Pos  	(10U)
#define   CCU4_CC4_CMC_UDS_Msk  	(0x00000003U  << CCU4_CC4_CMC_UDS_Pos)

#define   CCU4_CC4_CMC_LDS_Pos  	(12U)
#define   CCU4_CC4_CMC_LDS_Msk  	(0x00000003U  << CCU4_CC4_CMC_LDS_Pos)

#define   CCU4_CC4_CMC_CNTS_Pos  	(14U)
#define   CCU4_CC4_CMC_CNTS_Msk  	(0x00000003U  << CCU4_CC4_CMC_CNTS_Pos)

#define   CCU4_CC4_CMC_OFS_Pos  	(16U)
#define   CCU4_CC4_CMC_OFS_Msk  	(0x00000001U  << CCU4_CC4_CMC_OFS_Pos)

#define   CCU4_CC4_CMC_TS_Pos  	(17U)
#define   CCU4_CC4_CMC_TS_Msk  	(0x00000001U  << CCU4_CC4_CMC_TS_Pos)

#define   CCU4_CC4_CMC_MOS_Pos  	(18U)
#define   CCU4_CC4_CMC_MOS_Msk  	(0x00000003U  << CCU4_CC4_CMC_MOS_Pos)

#define   CCU4_CC4_CMC_TCE_Pos  	(20U)
#define   CCU4_CC4_CMC_TCE_Msk  	(0x00000001U  << CCU4_CC4_CMC_TCE_Pos)

/* CCU4_CC4_TCST  =  Slice Timer Status*/
#define   CCU4_CC4_TCST_TRB_Pos  	(0U)
#define   CCU4_CC4_TCST_TRB_Msk  	(0x00000001U  << CCU4_CC4_TCST_TRB_Pos)

#define   CCU4_CC4_TCST_CDIR_Pos  	(1U)
#define   CCU4_CC4_TCST_CDIR_Msk  	(0x00000001U  << CCU4_CC4_TCST_CDIR_Pos)

/* CCU4_CC4_TCSET  =  Slice Timer Run Set*/
#define   CCU4_CC4_TCSET_TRBS_Pos  	(0U)
#define   CCU4_CC4_TCSET_TRBS_Msk  	(0x00000001U  << CCU4_CC4_TCSET_TRBS_Pos)

/* CCU4_CC4_TCCLR  =  Slice Timer Clear*/
#define   CCU4_CC4_TCCLR_TRBC_Pos  	(0U)
#define   CCU4_CC4_TCCLR_TRBC_Msk  	(0x00000001U  << CCU4_CC4_TCCLR_TRBC_Pos)

#define   CCU4_CC4_TCCLR_TCC_Pos  	(1U)
#define   CCU4_CC4_TCCLR_TCC_Msk  	(0x00000001U  << CCU4_CC4_TCCLR_TCC_Pos)

#define   CCU4_CC4_TCCLR_DITC_Pos  	(2U)
#define   CCU4_CC4_TCCLR_DITC_Msk  	(0x00000001U  << CCU4_CC4_TCCLR_DITC_Pos)

/* CCU4_CC4_TC  =  Slice Timer Control*/
#define   CCU4_CC4_TC_TCM_Pos  	(0U)
#define   CCU4_CC4_TC_TCM_Msk  	(0x00000001U  << CCU4_CC4_TC_TCM_Pos)

#define   CCU4_CC4_TC_TSSM_Pos  	(1U)
#define   CCU4_CC4_TC_TSSM_Msk  	(0x00000001U  << CCU4_CC4_TC_TSSM_Pos)

#define   CCU4_CC4_TC_CLST_Pos  	(2U)
#define   CCU4_CC4_TC_CLST_Msk  	(0x00000001U  << CCU4_CC4_TC_CLST_Pos)

#define   CCU4_CC4_TC_CMOD_Pos  	(3U)
#define   CCU4_CC4_TC_CMOD_Msk  	(0x00000001U  << CCU4_CC4_TC_CMOD_Pos)

#define   CCU4_CC4_TC_ECM_Pos  	(4U)
#define   CCU4_CC4_TC_ECM_Msk  	(0x00000001U  << CCU4_CC4_TC_ECM_Pos)

#define   CCU4_CC4_TC_CAPC_Pos  	(5U)
#define   CCU4_CC4_TC_CAPC_Msk  	(0x00000003U  << CCU4_CC4_TC_CAPC_Pos)

#define   CCU4_CC4_TC_ENDM_Pos  	(8U)
#define   CCU4_CC4_TC_ENDM_Msk  	(0x00000003U  << CCU4_CC4_TC_ENDM_Pos)

#define   CCU4_CC4_TC_STRM_Pos  	(10U)
#define   CCU4_CC4_TC_STRM_Msk  	(0x00000001U  << CCU4_CC4_TC_STRM_Pos)

#define   CCU4_CC4_TC_SCE_Pos  	(11U)
#define   CCU4_CC4_TC_SCE_Msk  	(0x00000001U  << CCU4_CC4_TC_SCE_Pos)

#define   CCU4_CC4_TC_CCS_Pos  	(12U)
#define   CCU4_CC4_TC_CCS_Msk  	(0x00000001U  << CCU4_CC4_TC_CCS_Pos)

#define   CCU4_CC4_TC_DITHE_Pos  	(13U)
#define   CCU4_CC4_TC_DITHE_Msk  	(0x00000003U  << CCU4_CC4_TC_DITHE_Pos)

#define   CCU4_CC4_TC_DIM_Pos  	(15U)
#define   CCU4_CC4_TC_DIM_Msk  	(0x00000001U  << CCU4_CC4_TC_DIM_Pos)

#define   CCU4_CC4_TC_FPE_Pos  	(16U)
#define   CCU4_CC4_TC_FPE_Msk  	(0x00000001U  << CCU4_CC4_TC_FPE_Pos)

#define   CCU4_CC4_TC_TRAPE_Pos  	(17U)
#define   CCU4_CC4_TC_TRAPE_Msk  	(0x00000001U  << CCU4_CC4_TC_TRAPE_Pos)

#define   CCU4_CC4_TC_TRPSE_Pos  	(21U)
#define   CCU4_CC4_TC_TRPSE_Msk  	(0x00000001U  << CCU4_CC4_TC_TRPSE_Pos)

#define   CCU4_CC4_TC_TRPSW_Pos  	(22U)
#define   CCU4_CC4_TC_TRPSW_Msk  	(0x00000001U  << CCU4_CC4_TC_TRPSW_Pos)

#define   CCU4_CC4_TC_EMS_Pos  	(23U)
#define   CCU4_CC4_TC_EMS_Msk  	(0x00000001U  << CCU4_CC4_TC_EMS_Pos)

#define   CCU4_CC4_TC_EMT_Pos  	(24U)
#define   CCU4_CC4_TC_EMT_Msk  	(0x00000001U  << CCU4_CC4_TC_EMT_Pos)

#define   CCU4_CC4_TC_MCME_Pos  	(25U)
#define   CCU4_CC4_TC_MCME_Msk  	(0x00000001U  << CCU4_CC4_TC_MCME_Pos)

/* CCU4_CC4_PSL  =  Passive Level Config*/
#define   CCU4_CC4_PSL_PSL_Pos  	(0U)
#define   CCU4_CC4_PSL_PSL_Msk  	(0x00000001U  << CCU4_CC4_PSL_PSL_Pos)

/* CCU4_CC4_DIT  =  Dither Config*/
#define   CCU4_CC4_DIT_DCV_Pos  	(0U)
#define   CCU4_CC4_DIT_DCV_Msk  	(0x0000000FU  << CCU4_CC4_DIT_DCV_Pos)

#define   CCU4_CC4_DIT_DCNT_Pos  	(8U)
#define   CCU4_CC4_DIT_DCNT_Msk  	(0x0000000FU  << CCU4_CC4_DIT_DCNT_Pos)

/* CCU4_CC4_DITS  =  Dither Shadow Register*/
#define   CCU4_CC4_DITS_DCVS_Pos  	(0U)
#define   CCU4_CC4_DITS_DCVS_Msk  	(0x0000000FU  << CCU4_CC4_DITS_DCVS_Pos)

/* CCU4_CC4_PSC  =  Prescaler Control*/
#define   CCU4_CC4_PSC_PSIV_Pos  	(0U)
#define   CCU4_CC4_PSC_PSIV_Msk  	(0x0000000FU  << CCU4_CC4_PSC_PSIV_Pos)

/* CCU4_CC4_FPC  =  Floating Prescaler Control*/
#define   CCU4_CC4_FPC_PCMP_Pos  	(0U)
#define   CCU4_CC4_FPC_PCMP_Msk  	(0x0000000FU  << CCU4_CC4_FPC_PCMP_Pos)

#define   CCU4_CC4_FPC_PVAL_Pos  	(8U)
#define   CCU4_CC4_FPC_PVAL_Msk  	(0x0000000FU  << CCU4_CC4_FPC_PVAL_Pos)

/* CCU4_CC4_FPCS  =  Floating Prescaler Shadow*/
#define   CCU4_CC4_FPCS_PCMP_Pos  	(0U)
#define   CCU4_CC4_FPCS_PCMP_Msk  	(0x0000000FU  << CCU4_CC4_FPCS_PCMP_Pos)

/* CCU4_CC4_PR  =  Timer Period Value*/
#define   CCU4_CC4_PR_PR_Pos  	(0U)
#define   CCU4_CC4_PR_PR_Msk  	(0x0000FFFFU  << CCU4_CC4_PR_PR_Pos)

/* CCU4_CC4_PRS  =  Timer Shadow Period Value*/
#define   CCU4_CC4_PRS_PRS_Pos  	(0U)
#define   CCU4_CC4_PRS_PRS_Msk  	(0x0000FFFFU  << CCU4_CC4_PRS_PRS_Pos)

/* CCU4_CC4_CR  =  Timer Compare Value*/
#define   CCU4_CC4_CR_CR_Pos  	(0U)
#define   CCU4_CC4_CR_CR_Msk  	(0x0000FFFFU  << CCU4_CC4_CR_CR_Pos)

/* CCU4_CC4_CRS  =  Timer Shadow Compare Value*/
#define   CCU4_CC4_CRS_CRS_Pos  	(0U)
#define   CCU4_CC4_CRS_CRS_Msk  	(0x0000FFFFU  << CCU4_CC4_CRS_CRS_Pos)

/* CCU4_CC4_TIMER  =  Timer Value*/
#define   CCU4_CC4_TIMER_TVAL_Pos  	(0U)
#define   CCU4_CC4_TIMER_TVAL_Msk  	(0x0000FFFFU  << CCU4_CC4_TIMER_TVAL_Pos)

/* CCU4_CC4_C0V  =  Capture Register 0*/
#define   CCU4_CC4_CV_CAPTV_Pos  	(0U)
#define   CCU4_CC4_CV_CAPTV_Msk  	(0x0000FFFFU  << CCU4_CC4_CV_CAPTV_Pos)

#define   CCU4_CC4_CV_FPCV_Pos  	(16U)
#define   CCU4_CC4_CV_FPCV_Msk  	(0x0000000FU  << CCU4_CC4_CV_FPCV_Pos)

#define   CCU4_CC4_CV_FFL_Pos  	(20U)
#define   CCU4_CC4_CV_FFL_Msk  	(0x00000001U  << CCU4_CC4_CV_FFL_Pos)

/* CCU4_CC4_INTS  =  Interrupt Status*/
#define   CCU4_CC4_INTS_PMUS_Pos  	(0U)
#define   CCU4_CC4_INTS_PMUS_Msk  	(0x00000001U  << CCU4_CC4_INTS_PMUS_Pos)

#define   CCU4_CC4_INTS_OMDS_Pos  	(1U)
#define   CCU4_CC4_INTS_OMDS_Msk  	(0x00000001U  << CCU4_CC4_INTS_OMDS_Pos)

#define   CCU4_CC4_INTS_CMUS_Pos  	(2U)
#define   CCU4_CC4_INTS_CMUS_Msk  	(0x00000001U  << CCU4_CC4_INTS_CMUS_Pos)

#define   CCU4_CC4_INTS_CMDS_Pos  	(3U)
#define   CCU4_CC4_INTS_CMDS_Msk  	(0x00000001U  << CCU4_CC4_INTS_CMDS_Pos)

#define   CCU4_CC4_INTS_E0AS_Pos  	(8U)
#define   CCU4_CC4_INTS_E0AS_Msk  	(0x00000001U  << CCU4_CC4_INTS_E0AS_Pos)

#define   CCU4_CC4_INTS_E1AS_Pos  	(9U)
#define   CCU4_CC4_INTS_E1AS_Msk  	(0x00000001U  << CCU4_CC4_INTS_E1AS_Pos)

#define   CCU4_CC4_INTS_E2AS_Pos  	(10U)
#define   CCU4_CC4_INTS_E2AS_Msk  	(0x00000001U  << CCU4_CC4_INTS_E2AS_Pos)

#define   CCU4_CC4_INTS_TRPF_Pos  	(11U)
#define   CCU4_CC4_INTS_TRPF_Msk  	(0x00000001U  << CCU4_CC4_INTS_TRPF_Pos)

/* CCU4_CC4_INTE  =  Interrupt Enable Control*/
#define   CCU4_CC4_INTE_PME_Pos  	(0U)
#define   CCU4_CC4_INTE_PME_Msk  	(0x00000001U  << CCU4_CC4_INTE_PME_Pos)

#define   CCU4_CC4_INTE_OME_Pos  	(1U)
#define   CCU4_CC4_INTE_OME_Msk  	(0x00000001U  << CCU4_CC4_INTE_OME_Pos)

#define   CCU4_CC4_INTE_CMUE_Pos  	(2U)
#define   CCU4_CC4_INTE_CMUE_Msk  	(0x00000001U  << CCU4_CC4_INTE_CMUE_Pos)

#define   CCU4_CC4_INTE_CMDE_Pos  	(3U)
#define   CCU4_CC4_INTE_CMDE_Msk  	(0x00000001U  << CCU4_CC4_INTE_CMDE_Pos)

#define   CCU4_CC4_INTE_E0AE_Pos  	(8U)
#define   CCU4_CC4_INTE_E0AE_Msk  	(0x00000001U  << CCU4_CC4_INTE_E0AE_Pos)

#define   CCU4_CC4_INTE_E1AE_Pos  	(9U)
#define   CCU4_CC4_INTE_E1AE_Msk  	(0x00000001U  << CCU4_CC4_INTE_E1AE_Pos)

#define   CCU4_CC4_INTE_E2AE_Pos  	(10U)
#define   CCU4_CC4_INTE_E2AE_Msk  	(0x00000001U  << CCU4_CC4_INTE_E2AE_Pos)

/* CCU4_CC4_SRS  =  Service Request Selector*/
#define   CCU4_CC4_SRS_POSR_Pos  	(0U)
#define   CCU4_CC4_SRS_POSR_Msk  	(0x00000003U  << CCU4_CC4_SRS_POSR_Pos)

#define   CCU4_CC4_SRS_CMSR_Pos  	(2U)
#define   CCU4_CC4_SRS_CMSR_Msk  	(0x00000003U  << CCU4_CC4_SRS_CMSR_Pos)

#define   CCU4_CC4_SRS_E0SR_Pos  	(8U)
#define   CCU4_CC4_SRS_E0SR_Msk  	(0x00000003U  << CCU4_CC4_SRS_E0SR_Pos)

#define   CCU4_CC4_SRS_E1SR_Pos  	(10U)
#define   CCU4_CC4_SRS_E1SR_Msk  	(0x00000003U  << CCU4_CC4_SRS_E1SR_Pos)

#define   CCU4_CC4_SRS_E2SR_Pos  	(12U)
#define   CCU4_CC4_SRS_E2SR_Msk  	(0x00000003U  << CCU4_CC4_SRS_E2SR_Pos)

/* CCU4_CC4_SWS  =  Interrupt Status Set*/
#define   CCU4_CC4_SWS_SPM_Pos  	(0U)
#define   CCU4_CC4_SWS_SPM_Msk  	(0x00000001U  << CCU4_CC4_SWS_SPM_Pos)

#define   CCU4_CC4_SWS_SOM_Pos  	(1U)
#define   CCU4_CC4_SWS_SOM_Msk  	(0x00000001U  << CCU4_CC4_SWS_SOM_Pos)

#define   CCU4_CC4_SWS_SCMU_Pos  	(2U)
#define   CCU4_CC4_SWS_SCMU_Msk  	(0x00000001U  << CCU4_CC4_SWS_SCMU_Pos)

#define   CCU4_CC4_SWS_SCMD_Pos  	(3U)
#define   CCU4_CC4_SWS_SCMD_Msk  	(0x00000001U  << CCU4_CC4_SWS_SCMD_Pos)

#define   CCU4_CC4_SWS_SE0A_Pos  	(8U)
#define   CCU4_CC4_SWS_SE0A_Msk  	(0x00000001U  << CCU4_CC4_SWS_SE0A_Pos)

#define   CCU4_CC4_SWS_SE1A_Pos  	(9U)
#define   CCU4_CC4_SWS_SE1A_Msk  	(0x00000001U  << CCU4_CC4_SWS_SE1A_Pos)

#define   CCU4_CC4_SWS_SE2A_Pos  	(10U)
#define   CCU4_CC4_SWS_SE2A_Msk  	(0x00000001U  << CCU4_CC4_SWS_SE2A_Pos)

#define   CCU4_CC4_SWS_STRPF_Pos  	(11U)
#define   CCU4_CC4_SWS_STRPF_Msk  	(0x00000001U  << CCU4_CC4_SWS_STRPF_Pos)

/* CCU4_CC4_SWR  =  Interrupt Status Clear*/
#define   CCU4_CC4_SWR_RPM_Pos  	(0U)
#define   CCU4_CC4_SWR_RPM_Msk  	(0x00000001U  << CCU4_CC4_SWR_RPM_Pos)

#define   CCU4_CC4_SWR_ROM_Pos  	(1U)
#define   CCU4_CC4_SWR_ROM_Msk  	(0x00000001U  << CCU4_CC4_SWR_ROM_Pos)

#define   CCU4_CC4_SWR_RCMU_Pos  	(2U)
#define   CCU4_CC4_SWR_RCMU_Msk  	(0x00000001U  << CCU4_CC4_SWR_RCMU_Pos)

#define   CCU4_CC4_SWR_RCMD_Pos  	(3U)
#define   CCU4_CC4_SWR_RCMD_Msk  	(0x00000001U  << CCU4_CC4_SWR_RCMD_Pos)

#define   CCU4_CC4_SWR_RE0A_Pos  	(8U)
#define   CCU4_CC4_SWR_RE0A_Msk  	(0x00000001U  << CCU4_CC4_SWR_RE0A_Pos)

#define   CCU4_CC4_SWR_RE1A_Pos  	(9U)
#define   CCU4_CC4_SWR_RE1A_Msk  	(0x00000001U  << CCU4_CC4_SWR_RE1A_Pos)

#define   CCU4_CC4_SWR_RE2A_Pos  	(10U)
#define   CCU4_CC4_SWR_RE2A_Msk  	(0x00000001U  << CCU4_CC4_SWR_RE2A_Pos)

#define   CCU4_CC4_SWR_RTRPF_Pos  	(11U)
#define   CCU4_CC4_SWR_RTRPF_Msk  	(0x00000001U  << CCU4_CC4_SWR_RTRPF_Pos)



/***   CCU8x Bit Fileds *******************/
/***************************************************************************/


/* CCU8_GCTRL  =  Global Control Register*/
#define   CCU8_GCTRL_PRBC_Pos  	(0U)
#define   CCU8_GCTRL_PRBC_Msk  	(0x00000007U  << CCU8_GCTRL_PRBC_Pos)

#define   CCU8_GCTRL_PCIS_Pos  	(4U)
#define   CCU8_GCTRL_PCIS_Msk  	(0x00000003U  << CCU8_GCTRL_PCIS_Pos)

#define   CCU8_GCTRL_SUSCFG_Pos  	(8U)
#define   CCU8_GCTRL_SUSCFG_Msk  	(0x00000003U  << CCU8_GCTRL_SUSCFG_Pos)

#define   CCU8_GCTRL_MSE0_Pos  	(10U)
#define   CCU8_GCTRL_MSE0_Msk  	(0x00000001U  << CCU8_GCTRL_MSE0_Pos)

#define   CCU8_GCTRL_MSE1_Pos  	(11U)
#define   CCU8_GCTRL_MSE1_Msk  	(0x00000001U  << CCU8_GCTRL_MSE1_Pos)

#define   CCU8_GCTRL_MSE2_Pos  	(12U)
#define   CCU8_GCTRL_MSE2_Msk  	(0x00000001U  << CCU8_GCTRL_MSE2_Pos)

#define   CCU8_GCTRL_MSE3_Pos  	(13U)
#define   CCU8_GCTRL_MSE3_Msk  	(0x00000001U  << CCU8_GCTRL_MSE3_Pos)

#define   CCU8_GCTRL_MSDE_Pos  	(14U)
#define   CCU8_GCTRL_MSDE_Msk  	(0x00000003U  << CCU8_GCTRL_MSDE_Pos)

/* CCU8_GSTAT  =  Global Status Register*/
#define   CCU8_GSTAT_S0I_Pos  	(0U)
#define   CCU8_GSTAT_S0I_Msk  	(0x00000001U  << CCU8_GSTAT_S0I_Pos)

#define   CCU8_GSTAT_S1I_Pos  	(1U)
#define   CCU8_GSTAT_S1I_Msk  	(0x00000001U  << CCU8_GSTAT_S1I_Pos)

#define   CCU8_GSTAT_S2I_Pos  	(2U)
#define   CCU8_GSTAT_S2I_Msk  	(0x00000001U  << CCU8_GSTAT_S2I_Pos)

#define   CCU8_GSTAT_S3I_Pos  	(3U)
#define   CCU8_GSTAT_S3I_Msk  	(0x00000001U  << CCU8_GSTAT_S3I_Pos)

#define   CCU8_GSTAT_PRB_Pos  	(8U)
#define   CCU8_GSTAT_PRB_Msk  	(0x00000001U  << CCU8_GSTAT_PRB_Pos)

#define   CCU8_GSTAT_PCRB_Pos  	(10U)
#define   CCU8_GSTAT_PCRB_Msk  	(0x00000001U  << CCU8_GSTAT_PCRB_Pos)

/* CCU8_GIDLS  =  Global Idle Set*/
#define   CCU8_GIDLS_SS0I_Pos  	(0U)
#define   CCU8_GIDLS_SS0I_Msk  	(0x00000001U  << CCU8_GIDLS_SS0I_Pos)

#define   CCU8_GIDLS_SS1I_Pos  	(1U)
#define   CCU8_GIDLS_SS1I_Msk  	(0x00000001U  << CCU8_GIDLS_SS1I_Pos)

#define   CCU8_GIDLS_SS2I_Pos  	(2U)
#define   CCU8_GIDLS_SS2I_Msk  	(0x00000001U  << CCU8_GIDLS_SS2I_Pos)

#define   CCU8_GIDLS_SS3I_Pos  	(3U)
#define   CCU8_GIDLS_SS3I_Msk  	(0x00000001U  << CCU8_GIDLS_SS3I_Pos)

#define   CCU8_GIDLS_CPRB_Pos  	(8U)
#define   CCU8_GIDLS_CPRB_Msk  	(0x00000001U  << CCU8_GIDLS_CPRB_Pos)

#define   CCU8_GIDLS_PSIC_Pos  	(9U)
#define   CCU8_GIDLS_PSIC_Msk  	(0x00000001U  << CCU8_GIDLS_PSIC_Pos)

#define   CCU8_GIDLS_CPCH_Pos  	(10U)
#define   CCU8_GIDLS_CPCH_Msk  	(0x00000001U  << CCU8_GIDLS_CPCH_Pos)

/* CCU8_GIDLC  =  Global Idle Clear*/
#define   CCU8_GIDLC_CS0I_Pos  	(0U)
#define   CCU8_GIDLC_CS0I_Msk  	(0x00000001U  << CCU8_GIDLC_CS0I_Pos)

#define   CCU8_GIDLC_CS1I_Pos  	(1U)
#define   CCU8_GIDLC_CS1I_Msk  	(0x00000001U  << CCU8_GIDLC_CS1I_Pos)

#define   CCU8_GIDLC_CS2I_Pos  	(2U)
#define   CCU8_GIDLC_CS2I_Msk  	(0x00000001U  << CCU8_GIDLC_CS2I_Pos)

#define   CCU8_GIDLC_CS3I_Pos  	(3U)
#define   CCU8_GIDLC_CS3I_Msk  	(0x00000001U  << CCU8_GIDLC_CS3I_Pos)

#define   CCU8_GIDLC_SPRB_Pos  	(8U)
#define   CCU8_GIDLC_SPRB_Msk  	(0x00000001U  << CCU8_GIDLC_SPRB_Pos)

#define   CCU8_GIDLC_SPCH_Pos  	(10U)
#define   CCU8_GIDLC_SPCH_Msk  	(0x00000001U  << CCU8_GIDLC_SPCH_Pos)

/* CCU8_GCSS  =  Global Channel Set*/
#define   CCU8_GCSS_S0SE_Pos  	(0U)
#define   CCU8_GCSS_S0SE_Msk  	(0x00000001U  << CCU8_GCSS_S0SE_Pos)

#define   CCU8_GCSS_S0DSE_Pos  	(1U)
#define   CCU8_GCSS_S0DSE_Msk  	(0x00000001U  << CCU8_GCSS_S0DSE_Pos)

#define   CCU8_GCSS_S0PSE_Pos  	(2U)
#define   CCU8_GCSS_S0PSE_Msk  	(0x00000001U  << CCU8_GCSS_S0PSE_Pos)

#define   CCU8_GCSS_S1SE_Pos  	(4U)
#define   CCU8_GCSS_S1SE_Msk  	(0x00000001U  << CCU8_GCSS_S1SE_Pos)

#define   CCU8_GCSS_S1DSE_Pos  	(5U)
#define   CCU8_GCSS_S1DSE_Msk  	(0x00000001U  << CCU8_GCSS_S1DSE_Pos)

#define   CCU8_GCSS_S1PSE_Pos  	(6U)
#define   CCU8_GCSS_S1PSE_Msk  	(0x00000001U  << CCU8_GCSS_S1PSE_Pos)

#define   CCU8_GCSS_S2SE_Pos  	(8U)
#define   CCU8_GCSS_S2SE_Msk  	(0x00000001U  << CCU8_GCSS_S2SE_Pos)

#define   CCU8_GCSS_S2DSE_Pos  	(9U)
#define   CCU8_GCSS_S2DSE_Msk  	(0x00000001U  << CCU8_GCSS_S2DSE_Pos)

#define   CCU8_GCSS_S2PSE_Pos  	(10U)
#define   CCU8_GCSS_S2PSE_Msk  	(0x00000001U  << CCU8_GCSS_S2PSE_Pos)

#define   CCU8_GCSS_S3SE_Pos  	(12U)
#define   CCU8_GCSS_S3SE_Msk  	(0x00000001U  << CCU8_GCSS_S3SE_Pos)

#define   CCU8_GCSS_S3DSE_Pos  	(13U)
#define   CCU8_GCSS_S3DSE_Msk  	(0x00000001U  << CCU8_GCSS_S3DSE_Pos)

#define   CCU8_GCSS_S3PSE_Pos  	(14U)
#define   CCU8_GCSS_S3PSE_Msk  	(0x00000001U  << CCU8_GCSS_S3PSE_Pos)

#define   CCU8_GCSS_S0ST1S_Pos  	(16U)
#define   CCU8_GCSS_S0ST1S_Msk  	(0x00000001U  << CCU8_GCSS_S0ST1S_Pos)

#define   CCU8_GCSS_S1ST1S_Pos  	(17U)
#define   CCU8_GCSS_S1ST1S_Msk  	(0x00000001U  << CCU8_GCSS_S1ST1S_Pos)

#define   CCU8_GCSS_S2ST1S_Pos  	(18U)
#define   CCU8_GCSS_S2ST1S_Msk  	(0x00000001U  << CCU8_GCSS_S2ST1S_Pos)

#define   CCU8_GCSS_S3ST1S_Pos  	(19U)
#define   CCU8_GCSS_S3ST1S_Msk  	(0x00000001U  << CCU8_GCSS_S3ST1S_Pos)

#define   CCU8_GCSS_S0ST2S_Pos  	(20U)
#define   CCU8_GCSS_S0ST2S_Msk  	(0x00000001U  << CCU8_GCSS_S0ST2S_Pos)

#define   CCU8_GCSS_S1ST2S_Pos  	(21U)
#define   CCU8_GCSS_S1ST2S_Msk  	(0x00000001U  << CCU8_GCSS_S1ST2S_Pos)

#define   CCU8_GCSS_S2ST2S_Pos  	(22U)
#define   CCU8_GCSS_S2ST2S_Msk  	(0x00000001U  << CCU8_GCSS_S2ST2S_Pos)

#define   CCU8_GCSS_S3ST2S_Pos  	(23U)
#define   CCU8_GCSS_S3ST2S_Msk  	(0x00000001U  << CCU8_GCSS_S3ST2S_Pos)

/* CCU8_GCSC  =  Global Channel Clear*/
#define   CCU8_GCSC_S0SC_Pos  	(0U)
#define   CCU8_GCSC_S0SC_Msk  	(0x00000001U  << CCU8_GCSC_S0SC_Pos)

#define   CCU8_GCSC_S0DSC_Pos  	(1U)
#define   CCU8_GCSC_S0DSC_Msk  	(0x00000001U  << CCU8_GCSC_S0DSC_Pos)

#define   CCU8_GCSC_S0PSC_Pos  	(2U)
#define   CCU8_GCSC_S0PSC_Msk  	(0x00000001U  << CCU8_GCSC_S0PSC_Pos)

#define   CCU8_GCSC_S1SC_Pos  	(4U)
#define   CCU8_GCSC_S1SC_Msk  	(0x00000001U  << CCU8_GCSC_S1SC_Pos)

#define   CCU8_GCSC_S1DSC_Pos  	(5U)
#define   CCU8_GCSC_S1DSC_Msk  	(0x00000001U  << CCU8_GCSC_S1DSC_Pos)

#define   CCU8_GCSC_S1PSC_Pos  	(6U)
#define   CCU8_GCSC_S1PSC_Msk  	(0x00000001U  << CCU8_GCSC_S1PSC_Pos)

#define   CCU8_GCSC_S2SC_Pos  	(8U)
#define   CCU8_GCSC_S2SC_Msk  	(0x00000001U  << CCU8_GCSC_S2SC_Pos)

#define   CCU8_GCSC_S2DSC_Pos  	(9U)
#define   CCU8_GCSC_S2DSC_Msk  	(0x00000001U  << CCU8_GCSC_S2DSC_Pos)

#define   CCU8_GCSC_S2PSC_Pos  	(10U)
#define   CCU8_GCSC_S2PSC_Msk  	(0x00000001U  << CCU8_GCSC_S2PSC_Pos)

#define   CCU8_GCSC_S3SC_Pos  	(12U)
#define   CCU8_GCSC_S3SC_Msk  	(0x00000001U  << CCU8_GCSC_S3SC_Pos)

#define   CCU8_GCSC_S3DSC_Pos  	(13U)
#define   CCU8_GCSC_S3DSC_Msk  	(0x00000001U  << CCU8_GCSC_S3DSC_Pos)

#define   CCU8_GCSC_S3PSC_Pos  	(14U)
#define   CCU8_GCSC_S3PSC_Msk  	(0x00000001U  << CCU8_GCSC_S3PSC_Pos)

#define   CCU8_GCSC_S0ST1C_Pos  	(16U)
#define   CCU8_GCSC_S0ST1C_Msk  	(0x00000001U  << CCU8_GCSC_S0ST1C_Pos)

#define   CCU8_GCSC_S1ST1C_Pos  	(17U)
#define   CCU8_GCSC_S1ST1C_Msk  	(0x00000001U  << CCU8_GCSC_S1ST1C_Pos)

#define   CCU8_GCSC_S2ST1C_Pos  	(18U)
#define   CCU8_GCSC_S2ST1C_Msk  	(0x00000001U  << CCU8_GCSC_S2ST1C_Pos)

#define   CCU8_GCSC_S3ST1C_Pos  	(19U)
#define   CCU8_GCSC_S3ST1C_Msk  	(0x00000001U  << CCU8_GCSC_S3ST1C_Pos)

#define   CCU8_GCSC_S0ST2C_Pos  	(20U)
#define   CCU8_GCSC_S0ST2C_Msk  	(0x00000001U  << CCU8_GCSC_S0ST2C_Pos)

#define   CCU8_GCSC_S1ST2C_Pos  	(21U)
#define   CCU8_GCSC_S1ST2C_Msk  	(0x00000001U  << CCU8_GCSC_S1ST2C_Pos)

#define   CCU8_GCSC_S2ST2C_Pos  	(22U)
#define   CCU8_GCSC_S2ST2C_Msk  	(0x00000001U  << CCU8_GCSC_S2ST2C_Pos)

#define   CCU8_GCSC_S3ST2C_Pos  	(23U)
#define   CCU8_GCSC_S3ST2C_Msk  	(0x00000001U  << CCU8_GCSC_S3ST2C_Pos)

/* CCU8_GCST  =  Global Channel status*/
#define   CCU8_GCST_S0SS_Pos  	(0U)
#define   CCU8_GCST_S0SS_Msk  	(0x00000001U  << CCU8_GCST_S0SS_Pos)

#define   CCU8_GCST_S0DSS_Pos  	(1U)
#define   CCU8_GCST_S0DSS_Msk  	(0x00000001U  << CCU8_GCST_S0DSS_Pos)

#define   CCU8_GCST_S0PSS_Pos  	(2U)
#define   CCU8_GCST_S0PSS_Msk  	(0x00000001U  << CCU8_GCST_S0PSS_Pos)

#define   CCU8_GCST_S1SS_Pos  	(4U)
#define   CCU8_GCST_S1SS_Msk  	(0x00000001U  << CCU8_GCST_S1SS_Pos)

#define   CCU8_GCST_S1DSS_Pos  	(5U)
#define   CCU8_GCST_S1DSS_Msk  	(0x00000001U  << CCU8_GCST_S1DSS_Pos)

#define   CCU8_GCST_S1PSS_Pos  	(6U)
#define   CCU8_GCST_S1PSS_Msk  	(0x00000001U  << CCU8_GCST_S1PSS_Pos)

#define   CCU8_GCST_S2SS_Pos  	(8U)
#define   CCU8_GCST_S2SS_Msk  	(0x00000001U  << CCU8_GCST_S2SS_Pos)

#define   CCU8_GCST_S2DSS_Pos  	(9U)
#define   CCU8_GCST_S2DSS_Msk  	(0x00000001U  << CCU8_GCST_S2DSS_Pos)

#define   CCU8_GCST_S2PSS_Pos  	(10U)
#define   CCU8_GCST_S2PSS_Msk  	(0x00000001U  << CCU8_GCST_S2PSS_Pos)

#define   CCU8_GCST_S3SS_Pos  	(12U)
#define   CCU8_GCST_S3SS_Msk  	(0x00000001U  << CCU8_GCST_S3SS_Pos)

#define   CCU8_GCST_S3DSS_Pos  	(13U)
#define   CCU8_GCST_S3DSS_Msk  	(0x00000001U  << CCU8_GCST_S3DSS_Pos)

#define   CCU8_GCST_S3PSS_Pos  	(14U)
#define   CCU8_GCST_S3PSS_Msk  	(0x00000001U  << CCU8_GCST_S3PSS_Pos)

#define   CCU8_GCST_CC80ST1_Pos  	(16U)
#define   CCU8_GCST_CC80ST1_Msk  	(0x00000001U  << CCU8_GCST_CC80ST1_Pos)

#define   CCU8_GCST_CC81ST1_Pos  	(17U)
#define   CCU8_GCST_CC81ST1_Msk  	(0x00000001U  << CCU8_GCST_CC81ST1_Pos)

#define   CCU8_GCST_CC82ST1_Pos  	(18U)
#define   CCU8_GCST_CC82ST1_Msk  	(0x00000001U  << CCU8_GCST_CC82ST1_Pos)

#define   CCU8_GCST_CC83ST1_Pos  	(19U)
#define   CCU8_GCST_CC83ST1_Msk  	(0x00000001U  << CCU8_GCST_CC83ST1_Pos)

#define   CCU8_GCST_CC80ST2_Pos  	(20U)
#define   CCU8_GCST_CC80ST2_Msk  	(0x00000001U  << CCU8_GCST_CC80ST2_Pos)

#define   CCU8_GCST_CC81ST2_Pos  	(21U)
#define   CCU8_GCST_CC81ST2_Msk  	(0x00000001U  << CCU8_GCST_CC81ST2_Pos)

#define   CCU8_GCST_CC82ST2_Pos  	(22U)
#define   CCU8_GCST_CC82ST2_Msk  	(0x00000001U  << CCU8_GCST_CC82ST2_Pos)

#define   CCU8_GCST_CC83ST2_Pos  	(23U)
#define   CCU8_GCST_CC83ST2_Msk  	(0x00000001U  << CCU8_GCST_CC83ST2_Pos)

/* CCU8_GPCHK  =  Parity Checker Configuration*/
#define   CCU8_GPCHK_PASE_Pos  	(0U)
#define   CCU8_GPCHK_PASE_Msk  	(0x00000001U  << CCU8_GPCHK_PASE_Pos)

#define   CCU8_GPCHK_PACS_Pos  	(1U)
#define   CCU8_GPCHK_PACS_Msk  	(0x00000003U  << CCU8_GPCHK_PACS_Pos)

#define   CCU8_GPCHK_PISEL_Pos  	(3U)
#define   CCU8_GPCHK_PISEL_Msk  	(0x00000003U  << CCU8_GPCHK_PISEL_Pos)

#define   CCU8_GPCHK_PCDS_Pos  	(5U)
#define   CCU8_GPCHK_PCDS_Msk  	(0x00000003U  << CCU8_GPCHK_PCDS_Pos)

#define   CCU8_GPCHK_PCTS_Pos  	(7U)
#define   CCU8_GPCHK_PCTS_Msk  	(0x00000001U  << CCU8_GPCHK_PCTS_Pos)

#define   CCU8_GPCHK_PCST_Pos  	(15U)
#define   CCU8_GPCHK_PCST_Msk  	(0x00000001U  << CCU8_GPCHK_PCST_Pos)

#define   CCU8_GPCHK_PCSEL0_Pos  	(16U)
#define   CCU8_GPCHK_PCSEL0_Msk  	(0x0000000FU  << CCU8_GPCHK_PCSEL0_Pos)

#define   CCU8_GPCHK_PCSEL1_Pos  	(20U)
#define   CCU8_GPCHK_PCSEL1_Msk  	(0x0000000FU  << CCU8_GPCHK_PCSEL1_Pos)

#define   CCU8_GPCHK_PCSEL2_Pos  	(24U)
#define   CCU8_GPCHK_PCSEL2_Msk  	(0x0000000FU  << CCU8_GPCHK_PCSEL2_Pos)

#define   CCU8_GPCHK_PCSEL3_Pos  	(28U)
#define   CCU8_GPCHK_PCSEL3_Msk  	(0x0000000FU  << CCU8_GPCHK_PCSEL3_Pos)

/* CCU8_ECRD  =  Extended Capture Mode Read*/
#define   CCU8_ECRD_CAPV_Pos  	(0U)
#define   CCU8_ECRD_CAPV_Msk  	(0x0000FFFFU  << CCU8_ECRD_CAPV_Pos)

#define   CCU8_ECRD_FPCV_Pos  	(16U)
#define   CCU8_ECRD_FPCV_Msk  	(0x0000000FU  << CCU8_ECRD_FPCV_Pos)

#define   CCU8_ECRD_SPTR_Pos  	(20U)
#define   CCU8_ECRD_SPTR_Msk  	(0x00000003U  << CCU8_ECRD_SPTR_Pos)

#define   CCU8_ECRD_VPTR_Pos  	(22U)
#define   CCU8_ECRD_VPTR_Msk  	(0x00000003U  << CCU8_ECRD_VPTR_Pos)

#define   CCU8_ECRD_FFL_Pos  	(24U)
#define   CCU8_ECRD_FFL_Msk  	(0x00000001U  << CCU8_ECRD_FFL_Pos)

/* CCU8_MIDR  =  Module Identification*/
#define   CCU8_MIDR_MODR_Pos  	(0U)
#define   CCU8_MIDR_MODR_Msk  	(0x000000FFU  << CCU8_MIDR_MODR_Pos)

#define   CCU8_MIDR_MODT_Pos  	(8U)
#define   CCU8_MIDR_MODT_Msk  	(0x000000FFU  << CCU8_MIDR_MODT_Pos)

#define   CCU8_MIDR_MODN_Pos  	(16U)
#define   CCU8_MIDR_MODN_Msk  	(0x0000FFFFU  << CCU8_MIDR_MODN_Pos)

/* CCU8_CC8_INS  =  Input Selector Configuration*/
#define   CCU8_CC8_INS_EV0IS_Pos  	(0U)
#define   CCU8_CC8_INS_EV0IS_Msk  	(0x0000000FU  << CCU8_CC8_INS_EV0IS_Pos)

#define   CCU8_CC8_INS_EV1IS_Pos  	(4U)
#define   CCU8_CC8_INS_EV1IS_Msk  	(0x0000000FU  << CCU8_CC8_INS_EV1IS_Pos)

#define   CCU8_CC8_INS_EV2S_Pos  	(8U)
#define   CCU8_CC8_INS_EV2S_Msk  	(0x0000000FU  << CCU8_CC8_INS_EV2S_Pos)

#define   CCU8_CC8_INS_EV0EM_Pos  	(16U)
#define   CCU8_CC8_INS_EV0EM_Msk  	(0x00000003U  << CCU8_CC8_INS_EV0EM_Pos)

#define   CCU8_CC8_INS_EV1EM_Pos  	(18U)
#define   CCU8_CC8_INS_EV1EM_Msk  	(0x00000003U  << CCU8_CC8_INS_EV1EM_Pos)

#define   CCU8_CC8_INS_EV2EM_Pos  	(20U)
#define   CCU8_CC8_INS_EV2EM_Msk  	(0x00000003U  << CCU8_CC8_INS_EV2EM_Pos)

#define   CCU8_CC8_INS_EV0LM_Pos  	(22U)
#define   CCU8_CC8_INS_EV0LM_Msk  	(0x00000001U  << CCU8_CC8_INS_EV0LM_Pos)

#define   CCU8_CC8_INS_EV1LM_Pos  	(23U)
#define   CCU8_CC8_INS_EV1LM_Msk  	(0x00000001U  << CCU8_CC8_INS_EV1LM_Pos)

#define   CCU8_CC8_INS_EV2LM_Pos  	(24U)
#define   CCU8_CC8_INS_EV2LM_Msk  	(0x00000001U  << CCU8_CC8_INS_EV2LM_Pos)

#define   CCU8_CC8_INS_LPF0M_Pos  	(25U)
#define   CCU8_CC8_INS_LPF0M_Msk  	(0x00000003U  << CCU8_CC8_INS_LPF0M_Pos)

#define   CCU8_CC8_INS_LPF1M_Pos  	(27U)
#define   CCU8_CC8_INS_LPF1M_Msk  	(0x00000003U  << CCU8_CC8_INS_LPF1M_Pos)

#define   CCU8_CC8_INS_LPF2M_Pos  	(29U)
#define   CCU8_CC8_INS_LPF2M_Msk  	(0x00000003U  << CCU8_CC8_INS_LPF2M_Pos)

/* CCU8_CC8_CMC  =  Connection Matrix Control*/
#define   CCU8_CC8_CMC_STRTS_Pos  	(0U)
#define   CCU8_CC8_CMC_STRTS_Msk  	(0x00000003U  << CCU8_CC8_CMC_STRTS_Pos)

#define   CCU8_CC8_CMC_ENDS_Pos  	(2U)
#define   CCU8_CC8_CMC_ENDS_Msk  	(0x00000003U  << CCU8_CC8_CMC_ENDS_Pos)

#define   CCU8_CC8_CMC_CAP0S_Pos  	(4U)
#define   CCU8_CC8_CMC_CAP0S_Msk  	(0x00000003U  << CCU8_CC8_CMC_CAP0S_Pos)

#define   CCU8_CC8_CMC_CAP1S_Pos  	(6U)
#define   CCU8_CC8_CMC_CAP1S_Msk  	(0x00000003U  << CCU8_CC8_CMC_CAP1S_Pos)

#define   CCU8_CC8_CMC_GATES_Pos  	(8U)
#define   CCU8_CC8_CMC_GATES_Msk  	(0x00000003U  << CCU8_CC8_CMC_GATES_Pos)

#define   CCU8_CC8_CMC_UDS_Pos  	(10U)
#define   CCU8_CC8_CMC_UDS_Msk  	(0x00000003U  << CCU8_CC8_CMC_UDS_Pos)

#define   CCU8_CC8_CMC_LDS_Pos  	(12U)
#define   CCU8_CC8_CMC_LDS_Msk  	(0x00000003U  << CCU8_CC8_CMC_LDS_Pos)

#define   CCU8_CC8_CMC_CNTS_Pos  	(14U)
#define   CCU8_CC8_CMC_CNTS_Msk  	(0x00000003U  << CCU8_CC8_CMC_CNTS_Pos)

#define   CCU8_CC8_CMC_OFS_Pos  	(16U)
#define   CCU8_CC8_CMC_OFS_Msk  	(0x00000001U  << CCU8_CC8_CMC_OFS_Pos)

#define   CCU8_CC8_CMC_TS_Pos  	(17U)
#define   CCU8_CC8_CMC_TS_Msk  	(0x00000001U  << CCU8_CC8_CMC_TS_Pos)

#define   CCU8_CC8_CMC_MOS_Pos  	(18U)
#define   CCU8_CC8_CMC_MOS_Msk  	(0x00000003U  << CCU8_CC8_CMC_MOS_Pos)

#define   CCU8_CC8_CMC_TCE_Pos  	(20U)
#define   CCU8_CC8_CMC_TCE_Msk  	(0x00000001U  << CCU8_CC8_CMC_TCE_Pos)

/* CCU8_CC8_TCST  =  Slice Timer Status*/
#define   CCU8_CC8_TCST_TRB_Pos  	(0U)
#define   CCU8_CC8_TCST_TRB_Msk  	(0x00000001U  << CCU8_CC8_TCST_TRB_Pos)

#define   CCU8_CC8_TCST_CDIR_Pos  	(1U)
#define   CCU8_CC8_TCST_CDIR_Msk  	(0x00000001U  << CCU8_CC8_TCST_CDIR_Pos)

#define   CCU8_CC8_TCST_DTR1_Pos  	(3U)
#define   CCU8_CC8_TCST_DTR1_Msk  	(0x00000001U  << CCU8_CC8_TCST_DTR1_Pos)

#define   CCU8_CC8_TCST_DTR2_Pos  	(4U)
#define   CCU8_CC8_TCST_DTR2_Msk  	(0x00000001U  << CCU8_CC8_TCST_DTR2_Pos)

/* CCU8_CC8_TCSET  =  Slice Timer Run Set*/
#define   CCU8_CC8_TCSET_TRBS_Pos  	(0U)
#define   CCU8_CC8_TCSET_TRBS_Msk  	(0x00000001U  << CCU8_CC8_TCSET_TRBS_Pos)

/* CCU8_CC8_TCCLR  =  Slice Timer Clear*/
#define   CCU8_CC8_TCCLR_TRBC_Pos  	(0U)
#define   CCU8_CC8_TCCLR_TRBC_Msk  	(0x00000001U  << CCU8_CC8_TCCLR_TRBC_Pos)

#define   CCU8_CC8_TCCLR_TCC_Pos  	(1U)
#define   CCU8_CC8_TCCLR_TCC_Msk  	(0x00000001U  << CCU8_CC8_TCCLR_TCC_Pos)

#define   CCU8_CC8_TCCLR_DITC_Pos  	(2U)
#define   CCU8_CC8_TCCLR_DITC_Msk  	(0x00000001U  << CCU8_CC8_TCCLR_DITC_Pos)

#define   CCU8_CC8_TCCLR_DTC1C_Pos  	(3U)
#define   CCU8_CC8_TCCLR_DTC1C_Msk  	(0x00000001U  << CCU8_CC8_TCCLR_DTC1C_Pos)

#define   CCU8_CC8_TCCLR_DTC2C_Pos  	(4U)
#define   CCU8_CC8_TCCLR_DTC2C_Msk  	(0x00000001U  << CCU8_CC8_TCCLR_DTC2C_Pos)

/* CCU8_CC8_TC  =  Slice Timer Control*/
#define   CCU8_CC8_TC_TCM_Pos  	(0U)
#define   CCU8_CC8_TC_TCM_Msk  	(0x00000001U  << CCU8_CC8_TC_TCM_Pos)

#define   CCU8_CC8_TC_TSSM_Pos  	(1U)
#define   CCU8_CC8_TC_TSSM_Msk  	(0x00000001U  << CCU8_CC8_TC_TSSM_Pos)

#define   CCU8_CC8_TC_CLST_Pos  	(2U)
#define   CCU8_CC8_TC_CLST_Msk  	(0x00000001U  << CCU8_CC8_TC_CLST_Pos)

#define   CCU8_CC8_TC_CMOD_Pos  	(3U)
#define   CCU8_CC8_TC_CMOD_Msk  	(0x00000001U  << CCU8_CC8_TC_CMOD_Pos)

#define   CCU8_CC8_TC_ECM_Pos  	(4U)
#define   CCU8_CC8_TC_ECM_Msk  	(0x00000001U  << CCU8_CC8_TC_ECM_Pos)

#define   CCU8_CC8_TC_CAPC_Pos  	(5U)
#define   CCU8_CC8_TC_CAPC_Msk  	(0x00000003U  << CCU8_CC8_TC_CAPC_Pos)

#define   CCU8_CC8_TC_TLS_Pos  	(7U)
#define   CCU8_CC8_TC_TLS_Msk  	(0x00000001U  << CCU8_CC8_TC_TLS_Pos)

#define   CCU8_CC8_TC_ENDM_Pos  	(8U)
#define   CCU8_CC8_TC_ENDM_Msk  	(0x00000003U  << CCU8_CC8_TC_ENDM_Pos)

#define   CCU8_CC8_TC_STRM_Pos  	(10U)
#define   CCU8_CC8_TC_STRM_Msk  	(0x00000001U  << CCU8_CC8_TC_STRM_Pos)

#define   CCU8_CC8_TC_SCE_Pos  	(11U)
#define   CCU8_CC8_TC_SCE_Msk  	(0x00000001U  << CCU8_CC8_TC_SCE_Pos)

#define   CCU8_CC8_TC_CCS_Pos  	(12U)
#define   CCU8_CC8_TC_CCS_Msk  	(0x00000001U  << CCU8_CC8_TC_CCS_Pos)

#define   CCU8_CC8_TC_DITHE_Pos  	(13U)
#define   CCU8_CC8_TC_DITHE_Msk  	(0x00000003U  << CCU8_CC8_TC_DITHE_Pos)

#define   CCU8_CC8_TC_DIM_Pos  	(15U)
#define   CCU8_CC8_TC_DIM_Msk  	(0x00000001U  << CCU8_CC8_TC_DIM_Pos)

#define   CCU8_CC8_TC_FPE_Pos  	(16U)
#define   CCU8_CC8_TC_FPE_Msk  	(0x00000001U  << CCU8_CC8_TC_FPE_Pos)

#define   CCU8_CC8_TC_TRAPE0_Pos  	(17U)
#define   CCU8_CC8_TC_TRAPE0_Msk  	(0x00000001U  << CCU8_CC8_TC_TRAPE0_Pos)

#define   CCU8_CC8_TC_TRAPE1_Pos  	(18U)
#define   CCU8_CC8_TC_TRAPE1_Msk  	(0x00000001U  << CCU8_CC8_TC_TRAPE1_Pos)

#define   CCU8_CC8_TC_TRAPE2_Pos  	(19U)
#define   CCU8_CC8_TC_TRAPE2_Msk  	(0x00000001U  << CCU8_CC8_TC_TRAPE2_Pos)

#define   CCU8_CC8_TC_TRAPE3_Pos  	(20U)
#define   CCU8_CC8_TC_TRAPE3_Msk  	(0x00000001U  << CCU8_CC8_TC_TRAPE3_Pos)

#define   CCU8_CC8_TC_TRPSE_Pos  	(21U)
#define   CCU8_CC8_TC_TRPSE_Msk  	(0x00000001U  << CCU8_CC8_TC_TRPSE_Pos)

#define   CCU8_CC8_TC_TRPSW_Pos  	(22U)
#define   CCU8_CC8_TC_TRPSW_Msk  	(0x00000001U  << CCU8_CC8_TC_TRPSW_Pos)

#define   CCU8_CC8_TC_EMS_Pos  	(23U)
#define   CCU8_CC8_TC_EMS_Msk  	(0x00000001U  << CCU8_CC8_TC_EMS_Pos)

#define   CCU8_CC8_TC_EMT_Pos  	(24U)
#define   CCU8_CC8_TC_EMT_Msk  	(0x00000001U  << CCU8_CC8_TC_EMT_Pos)

#define   CCU8_CC8_TC_MCME1_Pos  	(25U)
#define   CCU8_CC8_TC_MCME1_Msk  	(0x00000001U  << CCU8_CC8_TC_MCME1_Pos)

#define   CCU8_CC8_TC_MCME2_Pos  	(26U)
#define   CCU8_CC8_TC_MCME2_Msk  	(0x00000001U  << CCU8_CC8_TC_MCME2_Pos)

#define   CCU8_CC8_TC_EME_Pos  	(27U)
#define   CCU8_CC8_TC_EME_Msk  	(0x00000003U  << CCU8_CC8_TC_EME_Pos)

#define   CCU8_CC8_TC_STOS_Pos  	(29U)
#define   CCU8_CC8_TC_STOS_Msk  	(0x00000003U  << CCU8_CC8_TC_STOS_Pos)

/* CCU8_CC8_PSL  =  Passive Level Config*/
#define   CCU8_CC8_PSL_PSL11_Pos  	(0U)
#define   CCU8_CC8_PSL_PSL11_Msk  	(0x00000001U  << CCU8_CC8_PSL_PSL11_Pos)

#define   CCU8_CC8_PSL_PSL12_Pos  	(1U)
#define   CCU8_CC8_PSL_PSL12_Msk  	(0x00000001U  << CCU8_CC8_PSL_PSL12_Pos)

#define   CCU8_CC8_PSL_PSL21_Pos  	(2U)
#define   CCU8_CC8_PSL_PSL21_Msk  	(0x00000001U  << CCU8_CC8_PSL_PSL21_Pos)

#define   CCU8_CC8_PSL_PSL22_Pos  	(3U)
#define   CCU8_CC8_PSL_PSL22_Msk  	(0x00000001U  << CCU8_CC8_PSL_PSL22_Pos)

/* CCU8_CC8_DIT  =  Dither Config*/
#define   CCU8_CC8_DIT_DCV_Pos  	(0U)
#define   CCU8_CC8_DIT_DCV_Msk  	(0x0000000FU  << CCU8_CC8_DIT_DCV_Pos)

#define   CCU8_CC8_DIT_DCNT_Pos  	(8U)
#define   CCU8_CC8_DIT_DCNT_Msk  	(0x0000000FU  << CCU8_CC8_DIT_DCNT_Pos)

/* CCU8_CC8_DITS  =  Dither Shadow Register*/
#define   CCU8_CC8_DITS_DCVS_Pos  	(0U)
#define   CCU8_CC8_DITS_DCVS_Msk  	(0x0000000FU  << CCU8_CC8_DITS_DCVS_Pos)

/* CCU8_CC8_PSC  =  Prescaler Control*/
#define   CCU8_CC8_PSC_PSIV_Pos  	(0U)
#define   CCU8_CC8_PSC_PSIV_Msk  	(0x0000000FU  << CCU8_CC8_PSC_PSIV_Pos)

/* CCU8_CC8_FPC  =  Floating Prescaler Control*/
#define   CCU8_CC8_FPC_PCMP_Pos  	(0U)
#define   CCU8_CC8_FPC_PCMP_Msk  	(0x0000000FU  << CCU8_CC8_FPC_PCMP_Pos)

#define   CCU8_CC8_FPC_PVAL_Pos  	(8U)
#define   CCU8_CC8_FPC_PVAL_Msk  	(0x0000000FU  << CCU8_CC8_FPC_PVAL_Pos)

/* CCU8_CC8_FPCS  =  Floating Prescaler Shadow*/
#define   CCU8_CC8_FPCS_PCMP_Pos  	(0U)
#define   CCU8_CC8_FPCS_PCMP_Msk  	(0x0000000FU  << CCU8_CC8_FPCS_PCMP_Pos)

/* CCU8_CC8_PR  =  Timer Period Value*/
#define   CCU8_CC8_PR_PR_Pos  	(0U)
#define   CCU8_CC8_PR_PR_Msk  	(0x0000FFFFU  << CCU8_CC8_PR_PR_Pos)

/* CCU8_CC8_PRS  =  Timer Shadow Period Value*/
#define   CCU8_CC8_PRS_PRS_Pos  	(0U)
#define   CCU8_CC8_PRS_PRS_Msk  	(0x0000FFFFU  << CCU8_CC8_PRS_PRS_Pos)

/* CCU8_CC8_CR1  =  Channel 1 Compare Value*/
#define   CCU8_CC8_CR1_CR1_Pos  	(0U)
#define   CCU8_CC8_CR1_CR1_Msk  	(0x0000FFFFU  << CCU8_CC8_CR1_CR1_Pos)

/* CCU8_CC8_CR1S  =  Channel 1 Compare Shadow Value*/
#define   CCU8_CC8_CR1S_CR1S_Pos  	(0U)
#define   CCU8_CC8_CR1S_CR1S_Msk  	(0x0000FFFFU  << CCU8_CC8_CR1S_CR1S_Pos)

/* CCU8_CC8_CR2  =  Channel 2 Compare Value*/
#define   CCU8_CC8_CR2_CR2_Pos  	(0U)
#define   CCU8_CC8_CR2_CR2_Msk  	(0x0000FFFFU  << CCU8_CC8_CR2_CR2_Pos)

/* CCU8_CC8_CR2S  =  Channel 2 Compare Shadow Value*/
#define   CCU8_CC8_CR2S_CR2S_Pos  	(0U)
#define   CCU8_CC8_CR2S_CR2S_Msk  	(0x0000FFFFU  << CCU8_CC8_CR2S_CR2S_Pos)

/* CCU8_CC8_CHC  =  Channel Control*/
#define   CCU8_CC8_CHC_ASE_Pos  	(0U)
#define   CCU8_CC8_CHC_ASE_Msk  	(0x00000001U  << CCU8_CC8_CHC_ASE_Pos)

#define   CCU8_CC8_CHC_OCS1_Pos  	(1U)
#define   CCU8_CC8_CHC_OCS1_Msk  	(0x00000001U  << CCU8_CC8_CHC_OCS1_Pos)

#define   CCU8_CC8_CHC_OCS2_Pos  	(2U)
#define   CCU8_CC8_CHC_OCS2_Msk  	(0x00000001U  << CCU8_CC8_CHC_OCS2_Pos)

#define   CCU8_CC8_CHC_OCS3_Pos  	(3U)
#define   CCU8_CC8_CHC_OCS3_Msk  	(0x00000001U  << CCU8_CC8_CHC_OCS3_Pos)

#define   CCU8_CC8_CHC_OCS4_Pos  	(4U)
#define   CCU8_CC8_CHC_OCS4_Msk  	(0x00000001U  << CCU8_CC8_CHC_OCS4_Pos)

/* CCU8_CC8_DTC  =  Dead Time Control*/
#define   CCU8_CC8_DTC_DTE1_Pos  	(0U)
#define   CCU8_CC8_DTC_DTE1_Msk  	(0x00000001U  << CCU8_CC8_DTC_DTE1_Pos)

#define   CCU8_CC8_DTC_DTE2_Pos  	(1U)
#define   CCU8_CC8_DTC_DTE2_Msk  	(0x00000001U  << CCU8_CC8_DTC_DTE2_Pos)

#define   CCU8_CC8_DTC_DCEN1_Pos  	(2U)
#define   CCU8_CC8_DTC_DCEN1_Msk  	(0x00000001U  << CCU8_CC8_DTC_DCEN1_Pos)

#define   CCU8_CC8_DTC_DCEN2_Pos  	(3U)
#define   CCU8_CC8_DTC_DCEN2_Msk  	(0x00000001U  << CCU8_CC8_DTC_DCEN2_Pos)

#define   CCU8_CC8_DTC_DCEN3_Pos  	(4U)
#define   CCU8_CC8_DTC_DCEN3_Msk  	(0x00000001U  << CCU8_CC8_DTC_DCEN3_Pos)

#define   CCU8_CC8_DTC_DCEN4_Pos  	(5U)
#define   CCU8_CC8_DTC_DCEN4_Msk  	(0x00000001U  << CCU8_CC8_DTC_DCEN4_Pos)

#define   CCU8_CC8_DTC_DTCC_Pos  	(6U)
#define   CCU8_CC8_DTC_DTCC_Msk  	(0x00000003U  << CCU8_CC8_DTC_DTCC_Pos)

/* CCU8_CC8_DC1R  =  Channel 1 Dead Time Values*/
#define   CCU8_CC8_DC1R_DT1R_Pos  	(0U)
#define   CCU8_CC8_DC1R_DT1R_Msk  	(0x000000FFU  << CCU8_CC8_DC1R_DT1R_Pos)

#define   CCU8_CC8_DC1R_DT1F_Pos  	(8U)
#define   CCU8_CC8_DC1R_DT1F_Msk  	(0x000000FFU  << CCU8_CC8_DC1R_DT1F_Pos)

/* CCU8_CC8_DC2R  =  Channel 2 Dead Time Values*/
#define   CCU8_CC8_DC2R_DT2R_Pos  	(0U)
#define   CCU8_CC8_DC2R_DT2R_Msk  	(0x000000FFU  << CCU8_CC8_DC2R_DT2R_Pos)

#define   CCU8_CC8_DC2R_DT2F_Pos  	(8U)
#define   CCU8_CC8_DC2R_DT2F_Msk  	(0x000000FFU  << CCU8_CC8_DC2R_DT2F_Pos)

/* CCU8_CC8_TIMER  =  Timer Value*/
#define   CCU8_CC8_TIMER_TVAL_Pos  	(0U)
#define   CCU8_CC8_TIMER_TVAL_Msk  	(0x0000FFFFU  << CCU8_CC8_TIMER_TVAL_Pos)

/* CCU8_CC8_C0V  =  Capture Register 0*/
#define   CCU8_CC8_CV_CAPTV_Pos  	(0U)
#define   CCU8_CC8_CV_CAPTV_Msk  	(0x0000FFFFU  << CCU8_CC8_CV_CAPTV_Pos)

#define   CCU8_CC8_CV_FPCV_Pos  	(16U)
#define   CCU8_CC8_CV_FPCV_Msk  	(0x0000000FU  << CCU8_CC8_CV_FPCV_Pos)

#define   CCU8_CC8_CV_FFL_Pos  	(20U)
#define   CCU8_CC8_CV_FFL_Msk  	(0x00000001U  << CCU8_CC8_CV_FFL_Pos)

/* CCU8_CC8_INTS  =  Interrupt Status*/
#define   CCU8_CC8_INTS_PMUS_Pos  	(0U)
#define   CCU8_CC8_INTS_PMUS_Msk  	(0x00000001U  << CCU8_CC8_INTS_PMUS_Pos)

#define   CCU8_CC8_INTS_OMDS_Pos  	(1U)
#define   CCU8_CC8_INTS_OMDS_Msk  	(0x00000001U  << CCU8_CC8_INTS_OMDS_Pos)

#define   CCU8_CC8_INTS_CMU1S_Pos  	(2U)
#define   CCU8_CC8_INTS_CMU1S_Msk  	(0x00000001U  << CCU8_CC8_INTS_CMU1S_Pos)

#define   CCU8_CC8_INTS_CMD1S_Pos  	(3U)
#define   CCU8_CC8_INTS_CMD1S_Msk  	(0x00000001U  << CCU8_CC8_INTS_CMD1S_Pos)

#define   CCU8_CC8_INTS_CMU2S_Pos  	(4U)
#define   CCU8_CC8_INTS_CMU2S_Msk  	(0x00000001U  << CCU8_CC8_INTS_CMU2S_Pos)

#define   CCU8_CC8_INTS_CMD2S_Pos  	(5U)
#define   CCU8_CC8_INTS_CMD2S_Msk  	(0x00000001U  << CCU8_CC8_INTS_CMD2S_Pos)

#define   CCU8_CC8_INTS_E0AS_Pos  	(8U)
#define   CCU8_CC8_INTS_E0AS_Msk  	(0x00000001U  << CCU8_CC8_INTS_E0AS_Pos)

#define   CCU8_CC8_INTS_E1AS_Pos  	(9U)
#define   CCU8_CC8_INTS_E1AS_Msk  	(0x00000001U  << CCU8_CC8_INTS_E1AS_Pos)

#define   CCU8_CC8_INTS_E2AS_Pos  	(10U)
#define   CCU8_CC8_INTS_E2AS_Msk  	(0x00000001U  << CCU8_CC8_INTS_E2AS_Pos)

#define   CCU8_CC8_INTS_TRPF_Pos  	(11U)
#define   CCU8_CC8_INTS_TRPF_Msk  	(0x00000001U  << CCU8_CC8_INTS_TRPF_Pos)

/* CCU8_CC8_INTE  =  Interrupt Enable Control*/
#define   CCU8_CC8_INTE_PME_Pos  	(0U)
#define   CCU8_CC8_INTE_PME_Msk  	(0x00000001U  << CCU8_CC8_INTE_PME_Pos)

#define   CCU8_CC8_INTE_OME_Pos  	(1U)
#define   CCU8_CC8_INTE_OME_Msk  	(0x00000001U  << CCU8_CC8_INTE_OME_Pos)

#define   CCU8_CC8_INTE_CMU1E_Pos  	(2U)
#define   CCU8_CC8_INTE_CMU1E_Msk  	(0x00000001U  << CCU8_CC8_INTE_CMU1E_Pos)

#define   CCU8_CC8_INTE_CMD1E_Pos  	(3U)
#define   CCU8_CC8_INTE_CMD1E_Msk  	(0x00000001U  << CCU8_CC8_INTE_CMD1E_Pos)

#define   CCU8_CC8_INTE_CMU2E_Pos  	(4U)
#define   CCU8_CC8_INTE_CMU2E_Msk  	(0x00000001U  << CCU8_CC8_INTE_CMU2E_Pos)

#define   CCU8_CC8_INTE_CMD2E_Pos  	(5U)
#define   CCU8_CC8_INTE_CMD2E_Msk  	(0x00000001U  << CCU8_CC8_INTE_CMD2E_Pos)

#define   CCU8_CC8_INTE_E0AE_Pos  	(8U)
#define   CCU8_CC8_INTE_E0AE_Msk  	(0x00000001U  << CCU8_CC8_INTE_E0AE_Pos)

#define   CCU8_CC8_INTE_E1AE_Pos  	(9U)
#define   CCU8_CC8_INTE_E1AE_Msk  	(0x00000001U  << CCU8_CC8_INTE_E1AE_Pos)

#define   CCU8_CC8_INTE_E2AE_Pos  	(10U)
#define   CCU8_CC8_INTE_E2AE_Msk  	(0x00000001U  << CCU8_CC8_INTE_E2AE_Pos)

/* CCU8_CC8_SRS  =  Service Request Selector*/
#define   CCU8_CC8_SRS_POSR_Pos  	(0U)
#define   CCU8_CC8_SRS_POSR_Msk  	(0x00000003U  << CCU8_CC8_SRS_POSR_Pos)

#define   CCU8_CC8_SRS_CM1SR_Pos  	(2U)
#define   CCU8_CC8_SRS_CM1SR_Msk  	(0x00000003U  << CCU8_CC8_SRS_CM1SR_Pos)

#define   CCU8_CC8_SRS_CM2SR_Pos  	(4U)
#define   CCU8_CC8_SRS_CM2SR_Msk  	(0x00000003U  << CCU8_CC8_SRS_CM2SR_Pos)

#define   CCU8_CC8_SRS_E0SR_Pos  	(8U)
#define   CCU8_CC8_SRS_E0SR_Msk  	(0x00000003U  << CCU8_CC8_SRS_E0SR_Pos)

#define   CCU8_CC8_SRS_E1SR_Pos  	(10U)
#define   CCU8_CC8_SRS_E1SR_Msk  	(0x00000003U  << CCU8_CC8_SRS_E1SR_Pos)

#define   CCU8_CC8_SRS_E2SR_Pos  	(12U)
#define   CCU8_CC8_SRS_E2SR_Msk  	(0x00000003U  << CCU8_CC8_SRS_E2SR_Pos)

/* CCU8_CC8_SWS  =  Interrupt Status Set*/
#define   CCU8_CC8_SWS_SPM_Pos  	(0U)
#define   CCU8_CC8_SWS_SPM_Msk  	(0x00000001U  << CCU8_CC8_SWS_SPM_Pos)

#define   CCU8_CC8_SWS_SOM_Pos  	(1U)
#define   CCU8_CC8_SWS_SOM_Msk  	(0x00000001U  << CCU8_CC8_SWS_SOM_Pos)

#define   CCU8_CC8_SWS_SCM1U_Pos  	(2U)
#define   CCU8_CC8_SWS_SCM1U_Msk  	(0x00000001U  << CCU8_CC8_SWS_SCM1U_Pos)

#define   CCU8_CC8_SWS_SCM1D_Pos  	(3U)
#define   CCU8_CC8_SWS_SCM1D_Msk  	(0x00000001U  << CCU8_CC8_SWS_SCM1D_Pos)

#define   CCU8_CC8_SWS_SCM2U_Pos  	(4U)
#define   CCU8_CC8_SWS_SCM2U_Msk  	(0x00000001U  << CCU8_CC8_SWS_SCM2U_Pos)

#define   CCU8_CC8_SWS_SCM2D_Pos  	(5U)
#define   CCU8_CC8_SWS_SCM2D_Msk  	(0x00000001U  << CCU8_CC8_SWS_SCM2D_Pos)

#define   CCU8_CC8_SWS_SE0A_Pos  	(8U)
#define   CCU8_CC8_SWS_SE0A_Msk  	(0x00000001U  << CCU8_CC8_SWS_SE0A_Pos)

#define   CCU8_CC8_SWS_SE1A_Pos  	(9U)
#define   CCU8_CC8_SWS_SE1A_Msk  	(0x00000001U  << CCU8_CC8_SWS_SE1A_Pos)

#define   CCU8_CC8_SWS_SE2A_Pos  	(10U)
#define   CCU8_CC8_SWS_SE2A_Msk  	(0x00000001U  << CCU8_CC8_SWS_SE2A_Pos)

#define   CCU8_CC8_SWS_STRPF_Pos  	(11U)
#define   CCU8_CC8_SWS_STRPF_Msk  	(0x00000001U  << CCU8_CC8_SWS_STRPF_Pos)

/* CCU8_CC8_SWR  =  Interrupt Status Clear*/
#define   CCU8_CC8_SWR_RPM_Pos  	(0U)
#define   CCU8_CC8_SWR_RPM_Msk  	(0x00000001U  << CCU8_CC8_SWR_RPM_Pos)

#define   CCU8_CC8_SWR_ROM_Pos  	(1U)
#define   CCU8_CC8_SWR_ROM_Msk  	(0x00000001U  << CCU8_CC8_SWR_ROM_Pos)

#define   CCU8_CC8_SWR_RCM1U_Pos  	(2U)
#define   CCU8_CC8_SWR_RCM1U_Msk  	(0x00000001U  << CCU8_CC8_SWR_RCM1U_Pos)

#define   CCU8_CC8_SWR_RCM1D_Pos  	(3U)
#define   CCU8_CC8_SWR_RCM1D_Msk  	(0x00000001U  << CCU8_CC8_SWR_RCM1D_Pos)

#define   CCU8_CC8_SWR_RCM2U_Pos  	(4U)
#define   CCU8_CC8_SWR_RCM2U_Msk  	(0x00000001U  << CCU8_CC8_SWR_RCM2U_Pos)

#define   CCU8_CC8_SWR_RCM2D_Pos  	(5U)
#define   CCU8_CC8_SWR_RCM2D_Msk  	(0x00000001U  << CCU8_CC8_SWR_RCM2D_Pos)

#define   CCU8_CC8_SWR_RE0A_Pos  	(8U)
#define   CCU8_CC8_SWR_RE0A_Msk  	(0x00000001U  << CCU8_CC8_SWR_RE0A_Pos)

#define   CCU8_CC8_SWR_RE1A_Pos  	(9U)
#define   CCU8_CC8_SWR_RE1A_Msk  	(0x00000001U  << CCU8_CC8_SWR_RE1A_Pos)

#define   CCU8_CC8_SWR_RE2A_Pos  	(10U)
#define   CCU8_CC8_SWR_RE2A_Msk  	(0x00000001U  << CCU8_CC8_SWR_RE2A_Pos)

#define   CCU8_CC8_SWR_RTRPF_Pos  	(11U)
#define   CCU8_CC8_SWR_RTRPF_Msk  	(0x00000001U  << CCU8_CC8_SWR_RTRPF_Pos)



/***   DAC Bit Fileds *******************/
/***************************************************************************/


/* DAC_ID  =  Module Identification Register*/
#define   DAC_ID_MODR_Pos  	(0U)
#define   DAC_ID_MODR_Msk  	(0x000000FFU  << DAC_ID_MODR_Pos)

#define   DAC_ID_MODT_Pos  	(8U)
#define   DAC_ID_MODT_Msk  	(0x000000FFU  << DAC_ID_MODT_Pos)

#define   DAC_ID_MODN_Pos  	(16U)
#define   DAC_ID_MODN_Msk  	(0x0000FFFFU  << DAC_ID_MODN_Pos)

/* DAC_DAC0CFG0  =  DAC0 Configuration Register 0*/
#define   DAC_DAC0CFG0_FREQ_Pos  	(0U)
#define   DAC_DAC0CFG0_FREQ_Msk  	(0x000FFFFFU  << DAC_DAC0CFG0_FREQ_Pos)

#define   DAC_DAC0CFG0_MODE_Pos  	(20U)
#define   DAC_DAC0CFG0_MODE_Msk  	(0x00000007U  << DAC_DAC0CFG0_MODE_Pos)

#define   DAC_DAC0CFG0_SIGN_Pos  	(23U)
#define   DAC_DAC0CFG0_SIGN_Msk  	(0x00000001U  << DAC_DAC0CFG0_SIGN_Pos)

#define   DAC_DAC0CFG0_FIFOIND_Pos  	(24U)
#define   DAC_DAC0CFG0_FIFOIND_Msk  	(0x00000003U  << DAC_DAC0CFG0_FIFOIND_Pos)

#define   DAC_DAC0CFG0_FIFOEMP_Pos  	(26U)
#define   DAC_DAC0CFG0_FIFOEMP_Msk  	(0x00000001U  << DAC_DAC0CFG0_FIFOEMP_Pos)

#define   DAC_DAC0CFG0_FIFOFUL_Pos  	(27U)
#define   DAC_DAC0CFG0_FIFOFUL_Msk  	(0x00000001U  << DAC_DAC0CFG0_FIFOFUL_Pos)

#define   DAC_DAC0CFG0_SIGNEN_Pos  	(29U)
#define   DAC_DAC0CFG0_SIGNEN_Msk  	(0x00000001U  << DAC_DAC0CFG0_SIGNEN_Pos)

#define   DAC_DAC0CFG0_SREN_Pos  	(30U)
#define   DAC_DAC0CFG0_SREN_Msk  	(0x00000001U  << DAC_DAC0CFG0_SREN_Pos)

#define   DAC_DAC0CFG0_RUN_Pos  	(31U)
#define   DAC_DAC0CFG0_RUN_Msk  	(0x00000001U  << DAC_DAC0CFG0_RUN_Pos)

/* DAC_DAC0CFG1  =  DAC0 Configuration Register 1*/
#define   DAC_DAC0CFG1_SCALE_Pos  	(0U)
#define   DAC_DAC0CFG1_SCALE_Msk  	(0x00000007U  << DAC_DAC0CFG1_SCALE_Pos)

#define   DAC_DAC0CFG1_MULDIV_Pos  	(3U)
#define   DAC_DAC0CFG1_MULDIV_Msk  	(0x00000001U  << DAC_DAC0CFG1_MULDIV_Pos)

#define   DAC_DAC0CFG1_OFFS_Pos  	(4U)
#define   DAC_DAC0CFG1_OFFS_Msk  	(0x000000FFU  << DAC_DAC0CFG1_OFFS_Pos)

#define   DAC_DAC0CFG1_TRIGSEL_Pos  	(12U)
#define   DAC_DAC0CFG1_TRIGSEL_Msk  	(0x00000007U  << DAC_DAC0CFG1_TRIGSEL_Pos)

#define   DAC_DAC0CFG1_DATMOD_Pos  	(15U)
#define   DAC_DAC0CFG1_DATMOD_Msk  	(0x00000001U  << DAC_DAC0CFG1_DATMOD_Pos)

#define   DAC_DAC0CFG1_SWTRIG_Pos  	(16U)
#define   DAC_DAC0CFG1_SWTRIG_Msk  	(0x00000001U  << DAC_DAC0CFG1_SWTRIG_Pos)

#define   DAC_DAC0CFG1_TRIGMOD_Pos  	(17U)
#define   DAC_DAC0CFG1_TRIGMOD_Msk  	(0x00000003U  << DAC_DAC0CFG1_TRIGMOD_Pos)

#define   DAC_DAC0CFG1_ANACFG_Pos  	(19U)
#define   DAC_DAC0CFG1_ANACFG_Msk  	(0x0000001FU  << DAC_DAC0CFG1_ANACFG_Pos)

#define   DAC_DAC0CFG1_ANAEN_Pos  	(24U)
#define   DAC_DAC0CFG1_ANAEN_Msk  	(0x00000001U  << DAC_DAC0CFG1_ANAEN_Pos)

#define   DAC_DAC0CFG1_REFCFGL_Pos  	(28U)
#define   DAC_DAC0CFG1_REFCFGL_Msk  	(0x0000000FU  << DAC_DAC0CFG1_REFCFGL_Pos)

/* DAC_DAC1CFG0  =  DAC1 Configuration Register 0*/
#define   DAC_DAC1CFG0_FREQ_Pos  	(0U)
#define   DAC_DAC1CFG0_FREQ_Msk  	(0x000FFFFFU  << DAC_DAC1CFG0_FREQ_Pos)

#define   DAC_DAC1CFG0_MODE_Pos  	(20U)
#define   DAC_DAC1CFG0_MODE_Msk  	(0x00000007U  << DAC_DAC1CFG0_MODE_Pos)

#define   DAC_DAC1CFG0_SIGN_Pos  	(23U)
#define   DAC_DAC1CFG0_SIGN_Msk  	(0x00000001U  << DAC_DAC1CFG0_SIGN_Pos)

#define   DAC_DAC1CFG0_FIFOIND_Pos  	(24U)
#define   DAC_DAC1CFG0_FIFOIND_Msk  	(0x00000003U  << DAC_DAC1CFG0_FIFOIND_Pos)

#define   DAC_DAC1CFG0_FIFOEMP_Pos  	(26U)
#define   DAC_DAC1CFG0_FIFOEMP_Msk  	(0x00000001U  << DAC_DAC1CFG0_FIFOEMP_Pos)

#define   DAC_DAC1CFG0_FIFOFUL_Pos  	(27U)
#define   DAC_DAC1CFG0_FIFOFUL_Msk  	(0x00000001U  << DAC_DAC1CFG0_FIFOFUL_Pos)

#define   DAC_DAC1CFG0_SIGNEN_Pos  	(29U)
#define   DAC_DAC1CFG0_SIGNEN_Msk  	(0x00000001U  << DAC_DAC1CFG0_SIGNEN_Pos)

#define   DAC_DAC1CFG0_SREN_Pos  	(30U)
#define   DAC_DAC1CFG0_SREN_Msk  	(0x00000001U  << DAC_DAC1CFG0_SREN_Pos)

#define   DAC_DAC1CFG0_RUN_Pos  	(31U)
#define   DAC_DAC1CFG0_RUN_Msk  	(0x00000001U  << DAC_DAC1CFG0_RUN_Pos)

/* DAC_DAC1CFG1  =  DAC1 Configuration Register 1*/
#define   DAC_DAC1CFG1_SCALE_Pos  	(0U)
#define   DAC_DAC1CFG1_SCALE_Msk  	(0x00000007U  << DAC_DAC1CFG1_SCALE_Pos)

#define   DAC_DAC1CFG1_MULDIV_Pos  	(3U)
#define   DAC_DAC1CFG1_MULDIV_Msk  	(0x00000001U  << DAC_DAC1CFG1_MULDIV_Pos)

#define   DAC_DAC1CFG1_OFFS_Pos  	(4U)
#define   DAC_DAC1CFG1_OFFS_Msk  	(0x000000FFU  << DAC_DAC1CFG1_OFFS_Pos)

#define   DAC_DAC1CFG1_TRIGSEL_Pos  	(12U)
#define   DAC_DAC1CFG1_TRIGSEL_Msk  	(0x00000007U  << DAC_DAC1CFG1_TRIGSEL_Pos)

#define   DAC_DAC1CFG1_SWTRIG_Pos  	(16U)
#define   DAC_DAC1CFG1_SWTRIG_Msk  	(0x00000001U  << DAC_DAC1CFG1_SWTRIG_Pos)

#define   DAC_DAC1CFG1_TRIGMOD_Pos  	(17U)
#define   DAC_DAC1CFG1_TRIGMOD_Msk  	(0x00000003U  << DAC_DAC1CFG1_TRIGMOD_Pos)

#define   DAC_DAC1CFG1_ANACFG_Pos  	(19U)
#define   DAC_DAC1CFG1_ANACFG_Msk  	(0x0000001FU  << DAC_DAC1CFG1_ANACFG_Pos)

#define   DAC_DAC1CFG1_ANAEN_Pos  	(24U)
#define   DAC_DAC1CFG1_ANAEN_Msk  	(0x00000001U  << DAC_DAC1CFG1_ANAEN_Pos)

#define   DAC_DAC1CFG1_REFCFGH_Pos  	(28U)
#define   DAC_DAC1CFG1_REFCFGH_Msk  	(0x0000000FU  << DAC_DAC1CFG1_REFCFGH_Pos)

/* DAC_DAC0DATA  =  DAC0 Data Register*/
#define   DAC_DAC0DATA_DATA0_Pos  	(0U)
#define   DAC_DAC0DATA_DATA0_Msk  	(0x00000FFFU  << DAC_DAC0DATA_DATA0_Pos)

/* DAC_DAC1DATA  =  DAC1 Data Register*/
#define   DAC_DAC1DATA_DATA1_Pos  	(0U)
#define   DAC_DAC1DATA_DATA1_Msk  	(0x00000FFFU  << DAC_DAC1DATA_DATA1_Pos)

/* DAC_DAC01DATA  =  DAC01 Data Register*/
#define   DAC_DAC01DATA_DATA0_Pos  	(0U)
#define   DAC_DAC01DATA_DATA0_Msk  	(0x00000FFFU  << DAC_DAC01DATA_DATA0_Pos)

#define   DAC_DAC01DATA_DATA1_Pos  	(16U)
#define   DAC_DAC01DATA_DATA1_Msk  	(0x00000FFFU  << DAC_DAC01DATA_DATA1_Pos)

/* DAC_DAC0PATL  =  DAC0 Lower Pattern Register*/
#define   DAC_DAC0PATL_PAT0_Pos  	(0U)
#define   DAC_DAC0PATL_PAT0_Msk  	(0x0000001FU  << DAC_DAC0PATL_PAT0_Pos)

#define   DAC_DAC0PATL_PAT1_Pos  	(5U)
#define   DAC_DAC0PATL_PAT1_Msk  	(0x0000001FU  << DAC_DAC0PATL_PAT1_Pos)

#define   DAC_DAC0PATL_PAT2_Pos  	(10U)
#define   DAC_DAC0PATL_PAT2_Msk  	(0x0000001FU  << DAC_DAC0PATL_PAT2_Pos)

#define   DAC_DAC0PATL_PAT3_Pos  	(15U)
#define   DAC_DAC0PATL_PAT3_Msk  	(0x0000001FU  << DAC_DAC0PATL_PAT3_Pos)

#define   DAC_DAC0PATL_PAT4_Pos  	(20U)
#define   DAC_DAC0PATL_PAT4_Msk  	(0x0000001FU  << DAC_DAC0PATL_PAT4_Pos)

#define   DAC_DAC0PATL_PAT5_Pos  	(25U)
#define   DAC_DAC0PATL_PAT5_Msk  	(0x0000001FU  << DAC_DAC0PATL_PAT5_Pos)

/* DAC_DAC0PATH  =  DAC0 Higher Pattern Register*/
#define   DAC_DAC0PATH_PAT6_Pos  	(0U)
#define   DAC_DAC0PATH_PAT6_Msk  	(0x0000001FU  << DAC_DAC0PATH_PAT6_Pos)

#define   DAC_DAC0PATH_PAT7_Pos  	(5U)
#define   DAC_DAC0PATH_PAT7_Msk  	(0x0000001FU  << DAC_DAC0PATH_PAT7_Pos)

#define   DAC_DAC0PATH_PAT8_Pos  	(10U)
#define   DAC_DAC0PATH_PAT8_Msk  	(0x0000001FU  << DAC_DAC0PATH_PAT8_Pos)

/* DAC_DAC1PATL  =  DAC1 Lower Pattern Register*/
#define   DAC_DAC1PATL_PAT0_Pos  	(0U)
#define   DAC_DAC1PATL_PAT0_Msk  	(0x0000001FU  << DAC_DAC1PATL_PAT0_Pos)

#define   DAC_DAC1PATL_PAT1_Pos  	(5U)
#define   DAC_DAC1PATL_PAT1_Msk  	(0x0000001FU  << DAC_DAC1PATL_PAT1_Pos)

#define   DAC_DAC1PATL_PAT2_Pos  	(10U)
#define   DAC_DAC1PATL_PAT2_Msk  	(0x0000001FU  << DAC_DAC1PATL_PAT2_Pos)

#define   DAC_DAC1PATL_PAT3_Pos  	(15U)
#define   DAC_DAC1PATL_PAT3_Msk  	(0x0000001FU  << DAC_DAC1PATL_PAT3_Pos)

#define   DAC_DAC1PATL_PAT4_Pos  	(20U)
#define   DAC_DAC1PATL_PAT4_Msk  	(0x0000001FU  << DAC_DAC1PATL_PAT4_Pos)

#define   DAC_DAC1PATL_PAT5_Pos  	(25U)
#define   DAC_DAC1PATL_PAT5_Msk  	(0x0000001FU  << DAC_DAC1PATL_PAT5_Pos)

/* DAC_DAC1PATH  =  DAC1 Higher Pattern Register*/
#define   DAC_DAC1PATH_PAT6_Pos  	(0U)
#define   DAC_DAC1PATH_PAT6_Msk  	(0x0000001FU  << DAC_DAC1PATH_PAT6_Pos)

#define   DAC_DAC1PATH_PAT7_Pos  	(5U)
#define   DAC_DAC1PATH_PAT7_Msk  	(0x0000001FU  << DAC_DAC1PATH_PAT7_Pos)

#define   DAC_DAC1PATH_PAT8_Pos  	(10U)
#define   DAC_DAC1PATH_PAT8_Msk  	(0x0000001FU  << DAC_DAC1PATH_PAT8_Pos)



/***   DLR Bit Fileds *******************/
/***************************************************************************/


/* DLR_OVRSTAT  =  GPDMA Overrun Status*/
#define   DLR_OVRSTAT_LN0_Pos  	(0U)
#define   DLR_OVRSTAT_LN0_Msk  	(0x00000001U  << DLR_OVRSTAT_LN0_Pos)

#define   DLR_OVRSTAT_LN1_Pos  	(1U)
#define   DLR_OVRSTAT_LN1_Msk  	(0x00000001U  << DLR_OVRSTAT_LN1_Pos)

#define   DLR_OVRSTAT_LN2_Pos  	(2U)
#define   DLR_OVRSTAT_LN2_Msk  	(0x00000001U  << DLR_OVRSTAT_LN2_Pos)

#define   DLR_OVRSTAT_LN3_Pos  	(3U)
#define   DLR_OVRSTAT_LN3_Msk  	(0x00000001U  << DLR_OVRSTAT_LN3_Pos)

#define   DLR_OVRSTAT_LN4_Pos  	(4U)
#define   DLR_OVRSTAT_LN4_Msk  	(0x00000001U  << DLR_OVRSTAT_LN4_Pos)

#define   DLR_OVRSTAT_LN5_Pos  	(5U)
#define   DLR_OVRSTAT_LN5_Msk  	(0x00000001U  << DLR_OVRSTAT_LN5_Pos)

#define   DLR_OVRSTAT_LN6_Pos  	(6U)
#define   DLR_OVRSTAT_LN6_Msk  	(0x00000001U  << DLR_OVRSTAT_LN6_Pos)

#define   DLR_OVRSTAT_LN7_Pos  	(7U)
#define   DLR_OVRSTAT_LN7_Msk  	(0x00000001U  << DLR_OVRSTAT_LN7_Pos)

#define   DLR_OVRSTAT_LN8_Pos  	(8U)
#define   DLR_OVRSTAT_LN8_Msk  	(0x00000001U  << DLR_OVRSTAT_LN8_Pos)

#define   DLR_OVRSTAT_LN9_Pos  	(9U)
#define   DLR_OVRSTAT_LN9_Msk  	(0x00000001U  << DLR_OVRSTAT_LN9_Pos)

#define   DLR_OVRSTAT_LN10_Pos  	(10U)
#define   DLR_OVRSTAT_LN10_Msk  	(0x00000001U  << DLR_OVRSTAT_LN10_Pos)

#define   DLR_OVRSTAT_LN11_Pos  	(11U)
#define   DLR_OVRSTAT_LN11_Msk  	(0x00000001U  << DLR_OVRSTAT_LN11_Pos)

/* DLR_OVRCLR  =  GPDMA Overrun Clear*/
#define   DLR_OVRCLR_LN0_Pos  	(0U)
#define   DLR_OVRCLR_LN0_Msk  	(0x00000001U  << DLR_OVRCLR_LN0_Pos)

#define   DLR_OVRCLR_LN1_Pos  	(1U)
#define   DLR_OVRCLR_LN1_Msk  	(0x00000001U  << DLR_OVRCLR_LN1_Pos)

#define   DLR_OVRCLR_LN2_Pos  	(2U)
#define   DLR_OVRCLR_LN2_Msk  	(0x00000001U  << DLR_OVRCLR_LN2_Pos)

#define   DLR_OVRCLR_LN3_Pos  	(3U)
#define   DLR_OVRCLR_LN3_Msk  	(0x00000001U  << DLR_OVRCLR_LN3_Pos)

#define   DLR_OVRCLR_LN4_Pos  	(4U)
#define   DLR_OVRCLR_LN4_Msk  	(0x00000001U  << DLR_OVRCLR_LN4_Pos)

#define   DLR_OVRCLR_LN5_Pos  	(5U)
#define   DLR_OVRCLR_LN5_Msk  	(0x00000001U  << DLR_OVRCLR_LN5_Pos)

#define   DLR_OVRCLR_LN6_Pos  	(6U)
#define   DLR_OVRCLR_LN6_Msk  	(0x00000001U  << DLR_OVRCLR_LN6_Pos)

#define   DLR_OVRCLR_LN7_Pos  	(7U)
#define   DLR_OVRCLR_LN7_Msk  	(0x00000001U  << DLR_OVRCLR_LN7_Pos)

#define   DLR_OVRCLR_LN8_Pos  	(8U)
#define   DLR_OVRCLR_LN8_Msk  	(0x00000001U  << DLR_OVRCLR_LN8_Pos)

#define   DLR_OVRCLR_LN9_Pos  	(9U)
#define   DLR_OVRCLR_LN9_Msk  	(0x00000001U  << DLR_OVRCLR_LN9_Pos)

#define   DLR_OVRCLR_LN10_Pos  	(10U)
#define   DLR_OVRCLR_LN10_Msk  	(0x00000001U  << DLR_OVRCLR_LN10_Pos)

#define   DLR_OVRCLR_LN11_Pos  	(11U)
#define   DLR_OVRCLR_LN11_Msk  	(0x00000001U  << DLR_OVRCLR_LN11_Pos)

/* DLR_SRSEL0  =  GPDMA Service Request Selection 0*/
#define   DLR_SRSEL0_RS0_Pos  	(0U)
#define   DLR_SRSEL0_RS0_Msk  	(0x0000000FU  << DLR_SRSEL0_RS0_Pos)

#define   DLR_SRSEL0_RS1_Pos  	(4U)
#define   DLR_SRSEL0_RS1_Msk  	(0x0000000FU  << DLR_SRSEL0_RS1_Pos)

#define   DLR_SRSEL0_RS2_Pos  	(8U)
#define   DLR_SRSEL0_RS2_Msk  	(0x0000000FU  << DLR_SRSEL0_RS2_Pos)

#define   DLR_SRSEL0_RS3_Pos  	(12U)
#define   DLR_SRSEL0_RS3_Msk  	(0x0000000FU  << DLR_SRSEL0_RS3_Pos)

#define   DLR_SRSEL0_RS4_Pos  	(16U)
#define   DLR_SRSEL0_RS4_Msk  	(0x0000000FU  << DLR_SRSEL0_RS4_Pos)

#define   DLR_SRSEL0_RS5_Pos  	(20U)
#define   DLR_SRSEL0_RS5_Msk  	(0x0000000FU  << DLR_SRSEL0_RS5_Pos)

#define   DLR_SRSEL0_RS6_Pos  	(24U)
#define   DLR_SRSEL0_RS6_Msk  	(0x0000000FU  << DLR_SRSEL0_RS6_Pos)

#define   DLR_SRSEL0_RS7_Pos  	(28U)
#define   DLR_SRSEL0_RS7_Msk  	(0x0000000FU  << DLR_SRSEL0_RS7_Pos)

/* DLR_SRSEL1  =  GPDMA Service Request Selection 1*/
#define   DLR_SRSEL1_RS8_Pos  	(0U)
#define   DLR_SRSEL1_RS8_Msk  	(0x0000000FU  << DLR_SRSEL1_RS8_Pos)

#define   DLR_SRSEL1_RS9_Pos  	(4U)
#define   DLR_SRSEL1_RS9_Msk  	(0x0000000FU  << DLR_SRSEL1_RS9_Pos)

#define   DLR_SRSEL1_RS10_Pos  	(8U)
#define   DLR_SRSEL1_RS10_Msk  	(0x0000000FU  << DLR_SRSEL1_RS10_Pos)

#define   DLR_SRSEL1_RS11_Pos  	(12U)
#define   DLR_SRSEL1_RS11_Msk  	(0x0000000FU  << DLR_SRSEL1_RS11_Pos)

/* DLR_LNEN  =  GPDMA Line Enable*/
#define   DLR_LNEN_LN0_Pos  	(0U)
#define   DLR_LNEN_LN0_Msk  	(0x00000001U  << DLR_LNEN_LN0_Pos)

#define   DLR_LNEN_LN1_Pos  	(1U)
#define   DLR_LNEN_LN1_Msk  	(0x00000001U  << DLR_LNEN_LN1_Pos)

#define   DLR_LNEN_LN2_Pos  	(2U)
#define   DLR_LNEN_LN2_Msk  	(0x00000001U  << DLR_LNEN_LN2_Pos)

#define   DLR_LNEN_LN3_Pos  	(3U)
#define   DLR_LNEN_LN3_Msk  	(0x00000001U  << DLR_LNEN_LN3_Pos)

#define   DLR_LNEN_LN4_Pos  	(4U)
#define   DLR_LNEN_LN4_Msk  	(0x00000001U  << DLR_LNEN_LN4_Pos)

#define   DLR_LNEN_LN5_Pos  	(5U)
#define   DLR_LNEN_LN5_Msk  	(0x00000001U  << DLR_LNEN_LN5_Pos)

#define   DLR_LNEN_LN6_Pos  	(6U)
#define   DLR_LNEN_LN6_Msk  	(0x00000001U  << DLR_LNEN_LN6_Pos)

#define   DLR_LNEN_LN7_Pos  	(7U)
#define   DLR_LNEN_LN7_Msk  	(0x00000001U  << DLR_LNEN_LN7_Pos)

#define   DLR_LNEN_LN8_Pos  	(8U)
#define   DLR_LNEN_LN8_Msk  	(0x00000001U  << DLR_LNEN_LN8_Pos)

#define   DLR_LNEN_LN9_Pos  	(9U)
#define   DLR_LNEN_LN9_Msk  	(0x00000001U  << DLR_LNEN_LN9_Pos)

#define   DLR_LNEN_LN10_Pos  	(10U)
#define   DLR_LNEN_LN10_Msk  	(0x00000001U  << DLR_LNEN_LN10_Pos)

#define   DLR_LNEN_LN11_Pos  	(11U)
#define   DLR_LNEN_LN11_Msk  	(0x00000001U  << DLR_LNEN_LN11_Pos)



/***   GPDMA1 Bit Fileds *******************/
/***************************************************************************/


/* GPDMA1_SARx  =  Source Address Register for Channel 0*/
#define   GPDMA1_SAR_SAR_Pos  	(0U)
#define   GPDMA1_SAR_SAR_Msk  	(0xFFFFFFFFU  << GPDMA1_SAR_SAR_Pos)

/* GPDMA1_DARx  =  Destination Address Register for Channel 0*/
#define   GPDMA1_DAR_DAR_Pos  	(0U)
#define   GPDMA1_DAR_DAR_Msk  	(0xFFFFFFFFU  << GPDMA1_DAR_DAR_Pos)

/* GPDMA1_CH_CTLL  =  Control Register for Channel x Low Word*/
#define   GPDMA1_CH_CTLL_TT_FC_Pos  	(20U)
#define   GPDMA1_CH_CTLL_TT_FC_Msk  	(0x00000007U  << GPDMA1_CH_CTLL_TT_FC_Pos)

#define   GPDMA1_CH_CTLL_SRC_MSIZE_Pos  	(14U)
#define   GPDMA1_CH_CTLL_SRC_MSIZE_Msk  	(0x00000007U  << GPDMA1_CH_CTLL_SRC_MSIZE_Pos)

#define   GPDMA1_CH_CTLL_DEST_MSIZE_Pos  	(11U)
#define   GPDMA1_CH_CTLL_DEST_MSIZE_Msk  	(0x00000007U  << GPDMA1_CH_CTLL_DEST_MSIZE_Pos)

#define   GPDMA1_CH_CTLL_SINC_Pos  	(9U)
#define   GPDMA1_CH_CTLL_SINC_Msk  	(0x00000003U  << GPDMA1_CH_CTLL_SINC_Pos)

#define   GPDMA1_CH_CTLL_DINC_Pos  	(7U)
#define   GPDMA1_CH_CTLL_DINC_Msk  	(0x00000003U  << GPDMA1_CH_CTLL_DINC_Pos)

#define   GPDMA1_CH_CTLL_SRC_TR_WIDTH_Pos  	(4U)
#define   GPDMA1_CH_CTLL_SRC_TR_WIDTH_Msk  	(0x00000007U  << GPDMA1_CH_CTLL_SRC_TR_WIDTH_Pos)

#define   GPDMA1_CH_CTLL_DST_TR_WIDTH_Pos  	(1U)
#define   GPDMA1_CH_CTLL_DST_TR_WIDTH_Msk  	(0x00000007U  << GPDMA1_CH_CTLL_DST_TR_WIDTH_Pos)

#define   GPDMA1_CH_CTLL_INT_EN_Pos  	(0U)
#define   GPDMA1_CH_CTLL_INT_EN_Msk  	(0x00000001U  << GPDMA1_CH_CTLL_INT_EN_Pos)

/* GPDMA1_CH_CTLH  =  Control Register for Channel x High Word*/
#define   GPDMA1_CH_CTLH_DONE_Pos  	(12U)
#define   GPDMA1_CH_CTLH_DONE_Msk  	(0x00000001U  << GPDMA1_CH_CTLH_DONE_Pos)

#define   GPDMA1_CH_CTLH_BLOCK_TS_Pos  	(0U)
#define   GPDMA1_CH_CTLH_BLOCK_TS_Msk  	(0x00000FFFU  << GPDMA1_CH_CTLH_BLOCK_TS_Pos)

/* GPDMA1_CH_CFGL  =  Configuration Register for Channel x Low Word*/
#define   GPDMA1_CH_CFGL_MAX_ABRST_Pos  	(20U)
#define   GPDMA1_CH_CFGL_MAX_ABRST_Msk  	(0x000003FFU  << GPDMA1_CH_CFGL_MAX_ABRST_Pos)

#define   GPDMA1_CH_CFGL_SRC_HS_POL_Pos  	(19U)
#define   GPDMA1_CH_CFGL_SRC_HS_POL_Msk  	(0x00000001U  << GPDMA1_CH_CFGL_SRC_HS_POL_Pos)

#define   GPDMA1_CH_CFGL_DST_HS_POL_Pos  	(18U)
#define   GPDMA1_CH_CFGL_DST_HS_POL_Msk  	(0x00000001U  << GPDMA1_CH_CFGL_DST_HS_POL_Pos)

#define   GPDMA1_CH_CFGL_LOCK_B_Pos  	(17U)
#define   GPDMA1_CH_CFGL_LOCK_B_Msk  	(0x00000001U  << GPDMA1_CH_CFGL_LOCK_B_Pos)

#define   GPDMA1_CH_CFGL_LOCK_CH_Pos  	(16U)
#define   GPDMA1_CH_CFGL_LOCK_CH_Msk  	(0x00000001U  << GPDMA1_CH_CFGL_LOCK_CH_Pos)

#define   GPDMA1_CH_CFGL_LOCK_B_L_Pos  	(14U)
#define   GPDMA1_CH_CFGL_LOCK_B_L_Msk  	(0x00000003U  << GPDMA1_CH_CFGL_LOCK_B_L_Pos)

#define   GPDMA1_CH_CFGL_LOCK_CH_L_Pos  	(12U)
#define   GPDMA1_CH_CFGL_LOCK_CH_L_Msk  	(0x00000003U  << GPDMA1_CH_CFGL_LOCK_CH_L_Pos)

#define   GPDMA1_CH_CFGL_HS_SEL_SRC_Pos  	(11U)
#define   GPDMA1_CH_CFGL_HS_SEL_SRC_Msk  	(0x00000001U  << GPDMA1_CH_CFGL_HS_SEL_SRC_Pos)

#define   GPDMA1_CH_CFGL_HS_SEL_DST_Pos  	(10U)
#define   GPDMA1_CH_CFGL_HS_SEL_DST_Msk  	(0x00000001U  << GPDMA1_CH_CFGL_HS_SEL_DST_Pos)

#define   GPDMA1_CH_CFGL_FIFO_EMPTY_Pos  	(9U)
#define   GPDMA1_CH_CFGL_FIFO_EMPTY_Msk  	(0x00000001U  << GPDMA1_CH_CFGL_FIFO_EMPTY_Pos)

#define   GPDMA1_CH_CFGL_CH_SUSP_Pos  	(8U)
#define   GPDMA1_CH_CFGL_CH_SUSP_Msk  	(0x00000001U  << GPDMA1_CH_CFGL_CH_SUSP_Pos)

#define   GPDMA1_CH_CFGL_CH_PRIOR_Pos  	(5U)
#define   GPDMA1_CH_CFGL_CH_PRIOR_Msk  	(0x00000007U  << GPDMA1_CH_CFGL_CH_PRIOR_Pos)

/* GPDMA1_CH_CFGH  =  Configuration Register for Channel x High Word*/
#define   GPDMA1_CH_CFGH_DEST_PER_Pos  	(11U)
#define   GPDMA1_CH_CFGH_DEST_PER_Msk  	(0x0000000FU  << GPDMA1_CH_CFGH_DEST_PER_Pos)

#define   GPDMA1_CH_CFGH_SRC_PER_Pos  	(7U)
#define   GPDMA1_CH_CFGH_SRC_PER_Msk  	(0x0000000FU  << GPDMA1_CH_CFGH_SRC_PER_Pos)

#define   GPDMA1_CH_CFGH_PROTCTL_Pos  	(2U)
#define   GPDMA1_CH_CFGH_PROTCTL_Msk  	(0x00000007U  << GPDMA1_CH_CFGH_PROTCTL_Pos)

#define   GPDMA1_CH_CFGH_FIFO_MODE_Pos  	(1U)
#define   GPDMA1_CH_CFGH_FIFO_MODE_Msk  	(0x00000001U  << GPDMA1_CH_CFGH_FIFO_MODE_Pos)

#define   GPDMA1_CH_CFGH_FCMODE_Pos  	(0U)
#define   GPDMA1_CH_CFGH_FCMODE_Msk  	(0x00000001U  << GPDMA1_CH_CFGH_FCMODE_Pos)

/* GPDMA1_RAWTFR  =  Raw IntTfr Status*/
#define   GPDMA1_RAWTFR_CH0_Pos  	(0U)
#define   GPDMA1_RAWTFR_CH0_Msk  	(0x00000001U  << GPDMA1_RAWTFR_CH0_Pos)

#define   GPDMA1_RAWTFR_CH1_Pos  	(1U)
#define   GPDMA1_RAWTFR_CH1_Msk  	(0x00000001U  << GPDMA1_RAWTFR_CH1_Pos)

#define   GPDMA1_RAWTFR_CH2_Pos  	(2U)
#define   GPDMA1_RAWTFR_CH2_Msk  	(0x00000001U  << GPDMA1_RAWTFR_CH2_Pos)

#define   GPDMA1_RAWTFR_CH3_Pos  	(3U)
#define   GPDMA1_RAWTFR_CH3_Msk  	(0x00000001U  << GPDMA1_RAWTFR_CH3_Pos)

/* GPDMA1_RAWBLOCK  =  Raw IntBlock Status*/
#define   GPDMA1_RAWBLOCK_CH0_Pos  	(0U)
#define   GPDMA1_RAWBLOCK_CH0_Msk  	(0x00000001U  << GPDMA1_RAWBLOCK_CH0_Pos)

#define   GPDMA1_RAWBLOCK_CH1_Pos  	(1U)
#define   GPDMA1_RAWBLOCK_CH1_Msk  	(0x00000001U  << GPDMA1_RAWBLOCK_CH1_Pos)

#define   GPDMA1_RAWBLOCK_CH2_Pos  	(2U)
#define   GPDMA1_RAWBLOCK_CH2_Msk  	(0x00000001U  << GPDMA1_RAWBLOCK_CH2_Pos)

#define   GPDMA1_RAWBLOCK_CH3_Pos  	(3U)
#define   GPDMA1_RAWBLOCK_CH3_Msk  	(0x00000001U  << GPDMA1_RAWBLOCK_CH3_Pos)

/* GPDMA1_RAWSRCTRAN  =  Raw IntSrcTran Status*/
#define   GPDMA1_RAWSRCTRAN_CH0_Pos  	(0U)
#define   GPDMA1_RAWSRCTRAN_CH0_Msk  	(0x00000001U  << GPDMA1_RAWSRCTRAN_CH0_Pos)

#define   GPDMA1_RAWSRCTRAN_CH1_Pos  	(1U)
#define   GPDMA1_RAWSRCTRAN_CH1_Msk  	(0x00000001U  << GPDMA1_RAWSRCTRAN_CH1_Pos)

#define   GPDMA1_RAWSRCTRAN_CH2_Pos  	(2U)
#define   GPDMA1_RAWSRCTRAN_CH2_Msk  	(0x00000001U  << GPDMA1_RAWSRCTRAN_CH2_Pos)

#define   GPDMA1_RAWSRCTRAN_CH3_Pos  	(3U)
#define   GPDMA1_RAWSRCTRAN_CH3_Msk  	(0x00000001U  << GPDMA1_RAWSRCTRAN_CH3_Pos)

/* GPDMA1_RAWDSTTRAN  =  Raw IntBlock Status*/
#define   GPDMA1_RAWDSTTRAN_CH0_Pos  	(0U)
#define   GPDMA1_RAWDSTTRAN_CH0_Msk  	(0x00000001U  << GPDMA1_RAWDSTTRAN_CH0_Pos)

#define   GPDMA1_RAWDSTTRAN_CH1_Pos  	(1U)
#define   GPDMA1_RAWDSTTRAN_CH1_Msk  	(0x00000001U  << GPDMA1_RAWDSTTRAN_CH1_Pos)

#define   GPDMA1_RAWDSTTRAN_CH2_Pos  	(2U)
#define   GPDMA1_RAWDSTTRAN_CH2_Msk  	(0x00000001U  << GPDMA1_RAWDSTTRAN_CH2_Pos)

#define   GPDMA1_RAWDSTTRAN_CH3_Pos  	(3U)
#define   GPDMA1_RAWDSTTRAN_CH3_Msk  	(0x00000001U  << GPDMA1_RAWDSTTRAN_CH3_Pos)

/* GPDMA1_RAWERR  =  Raw IntErr Status*/
#define   GPDMA1_RAWERR_CH0_Pos  	(0U)
#define   GPDMA1_RAWERR_CH0_Msk  	(0x00000001U  << GPDMA1_RAWERR_CH0_Pos)

#define   GPDMA1_RAWERR_CH1_Pos  	(1U)
#define   GPDMA1_RAWERR_CH1_Msk  	(0x00000001U  << GPDMA1_RAWERR_CH1_Pos)

#define   GPDMA1_RAWERR_CH2_Pos  	(2U)
#define   GPDMA1_RAWERR_CH2_Msk  	(0x00000001U  << GPDMA1_RAWERR_CH2_Pos)

#define   GPDMA1_RAWERR_CH3_Pos  	(3U)
#define   GPDMA1_RAWERR_CH3_Msk  	(0x00000001U  << GPDMA1_RAWERR_CH3_Pos)

/* GPDMA1_STATUSTFR  =  IntTfr Status*/
#define   GPDMA1_STATUSTFR_CH0_Pos  	(0U)
#define   GPDMA1_STATUSTFR_CH0_Msk  	(0x00000001U  << GPDMA1_STATUSTFR_CH0_Pos)

#define   GPDMA1_STATUSTFR_CH1_Pos  	(1U)
#define   GPDMA1_STATUSTFR_CH1_Msk  	(0x00000001U  << GPDMA1_STATUSTFR_CH1_Pos)

#define   GPDMA1_STATUSTFR_CH2_Pos  	(2U)
#define   GPDMA1_STATUSTFR_CH2_Msk  	(0x00000001U  << GPDMA1_STATUSTFR_CH2_Pos)

#define   GPDMA1_STATUSTFR_CH3_Pos  	(3U)
#define   GPDMA1_STATUSTFR_CH3_Msk  	(0x00000001U  << GPDMA1_STATUSTFR_CH3_Pos)

/* GPDMA1_STATUSBLOCK  =  IntBlock Status*/
#define   GPDMA1_STATUSBLOCK_CH0_Pos  	(0U)
#define   GPDMA1_STATUSBLOCK_CH0_Msk  	(0x00000001U  << GPDMA1_STATUSBLOCK_CH0_Pos)

#define   GPDMA1_STATUSBLOCK_CH1_Pos  	(1U)
#define   GPDMA1_STATUSBLOCK_CH1_Msk  	(0x00000001U  << GPDMA1_STATUSBLOCK_CH1_Pos)

#define   GPDMA1_STATUSBLOCK_CH2_Pos  	(2U)
#define   GPDMA1_STATUSBLOCK_CH2_Msk  	(0x00000001U  << GPDMA1_STATUSBLOCK_CH2_Pos)

#define   GPDMA1_STATUSBLOCK_CH3_Pos  	(3U)
#define   GPDMA1_STATUSBLOCK_CH3_Msk  	(0x00000001U  << GPDMA1_STATUSBLOCK_CH3_Pos)

/* GPDMA1_STATUSSRCTRAN  =  IntSrcTran Status*/
#define   GPDMA1_STATUSSRCTRAN_CH0_Pos  	(0U)
#define   GPDMA1_STATUSSRCTRAN_CH0_Msk  	(0x00000001U  << GPDMA1_STATUSSRCTRAN_CH0_Pos)

#define   GPDMA1_STATUSSRCTRAN_CH1_Pos  	(1U)
#define   GPDMA1_STATUSSRCTRAN_CH1_Msk  	(0x00000001U  << GPDMA1_STATUSSRCTRAN_CH1_Pos)

#define   GPDMA1_STATUSSRCTRAN_CH2_Pos  	(2U)
#define   GPDMA1_STATUSSRCTRAN_CH2_Msk  	(0x00000001U  << GPDMA1_STATUSSRCTRAN_CH2_Pos)

#define   GPDMA1_STATUSSRCTRAN_CH3_Pos  	(3U)
#define   GPDMA1_STATUSSRCTRAN_CH3_Msk  	(0x00000001U  << GPDMA1_STATUSSRCTRAN_CH3_Pos)

/* GPDMA1_STATUSDSTTRAN  =  IntBlock Status*/
#define   GPDMA1_STATUSDSTTRAN_CH0_Pos  	(0U)
#define   GPDMA1_STATUSDSTTRAN_CH0_Msk  	(0x00000001U  << GPDMA1_STATUSDSTTRAN_CH0_Pos)

#define   GPDMA1_STATUSDSTTRAN_CH1_Pos  	(1U)
#define   GPDMA1_STATUSDSTTRAN_CH1_Msk  	(0x00000001U  << GPDMA1_STATUSDSTTRAN_CH1_Pos)

#define   GPDMA1_STATUSDSTTRAN_CH2_Pos  	(2U)
#define   GPDMA1_STATUSDSTTRAN_CH2_Msk  	(0x00000001U  << GPDMA1_STATUSDSTTRAN_CH2_Pos)

#define   GPDMA1_STATUSDSTTRAN_CH3_Pos  	(3U)
#define   GPDMA1_STATUSDSTTRAN_CH3_Msk  	(0x00000001U  << GPDMA1_STATUSDSTTRAN_CH3_Pos)

/* GPDMA1_STATUSERR  =  IntErr Status*/
#define   GPDMA1_STATUSERR_CH0_Pos  	(0U)
#define   GPDMA1_STATUSERR_CH0_Msk  	(0x00000001U  << GPDMA1_STATUSERR_CH0_Pos)

#define   GPDMA1_STATUSERR_CH1_Pos  	(1U)
#define   GPDMA1_STATUSERR_CH1_Msk  	(0x00000001U  << GPDMA1_STATUSERR_CH1_Pos)

#define   GPDMA1_STATUSERR_CH2_Pos  	(2U)
#define   GPDMA1_STATUSERR_CH2_Msk  	(0x00000001U  << GPDMA1_STATUSERR_CH2_Pos)

#define   GPDMA1_STATUSERR_CH3_Pos  	(3U)
#define   GPDMA1_STATUSERR_CH3_Msk  	(0x00000001U  << GPDMA1_STATUSERR_CH3_Pos)

/* GPDMA1_MASKTFR  =  Mask for Raw IntTfr Status*/
#define   GPDMA1_MASKTFR_WE_CH0_Pos  	(8U)
#define   GPDMA1_MASKTFR_WE_CH0_Msk  	(0x00000001U  << GPDMA1_MASKTFR_WE_CH0_Pos)

#define   GPDMA1_MASKTFR_WE_CH1_Pos  	(9U)
#define   GPDMA1_MASKTFR_WE_CH1_Msk  	(0x00000001U  << GPDMA1_MASKTFR_WE_CH1_Pos)

#define   GPDMA1_MASKTFR_WE_CH2_Pos  	(10U)
#define   GPDMA1_MASKTFR_WE_CH2_Msk  	(0x00000001U  << GPDMA1_MASKTFR_WE_CH2_Pos)

#define   GPDMA1_MASKTFR_WE_CH3_Pos  	(11U)
#define   GPDMA1_MASKTFR_WE_CH3_Msk  	(0x00000001U  << GPDMA1_MASKTFR_WE_CH3_Pos)

#define   GPDMA1_MASKTFR_CH0_Pos  	(0U)
#define   GPDMA1_MASKTFR_CH0_Msk  	(0x00000001U  << GPDMA1_MASKTFR_CH0_Pos)

#define   GPDMA1_MASKTFR_CH1_Pos  	(1U)
#define   GPDMA1_MASKTFR_CH1_Msk  	(0x00000001U  << GPDMA1_MASKTFR_CH1_Pos)

#define   GPDMA1_MASKTFR_CH2_Pos  	(2U)
#define   GPDMA1_MASKTFR_CH2_Msk  	(0x00000001U  << GPDMA1_MASKTFR_CH2_Pos)

#define   GPDMA1_MASKTFR_CH3_Pos  	(3U)
#define   GPDMA1_MASKTFR_CH3_Msk  	(0x00000001U  << GPDMA1_MASKTFR_CH3_Pos)

/* GPDMA1_MASKBLOCK  =  Mask for Raw IntBlock Status*/
#define   GPDMA1_MASKBLOCK_WE_CH0_Pos  	(8U)
#define   GPDMA1_MASKBLOCK_WE_CH0_Msk  	(0x00000001U  << GPDMA1_MASKBLOCK_WE_CH0_Pos)

#define   GPDMA1_MASKBLOCK_WE_CH1_Pos  	(9U)
#define   GPDMA1_MASKBLOCK_WE_CH1_Msk  	(0x00000001U  << GPDMA1_MASKBLOCK_WE_CH1_Pos)

#define   GPDMA1_MASKBLOCK_WE_CH2_Pos  	(10U)
#define   GPDMA1_MASKBLOCK_WE_CH2_Msk  	(0x00000001U  << GPDMA1_MASKBLOCK_WE_CH2_Pos)

#define   GPDMA1_MASKBLOCK_WE_CH3_Pos  	(11U)
#define   GPDMA1_MASKBLOCK_WE_CH3_Msk  	(0x00000001U  << GPDMA1_MASKBLOCK_WE_CH3_Pos)

#define   GPDMA1_MASKBLOCK_CH0_Pos  	(0U)
#define   GPDMA1_MASKBLOCK_CH0_Msk  	(0x00000001U  << GPDMA1_MASKBLOCK_CH0_Pos)

#define   GPDMA1_MASKBLOCK_CH1_Pos  	(1U)
#define   GPDMA1_MASKBLOCK_CH1_Msk  	(0x00000001U  << GPDMA1_MASKBLOCK_CH1_Pos)

#define   GPDMA1_MASKBLOCK_CH2_Pos  	(2U)
#define   GPDMA1_MASKBLOCK_CH2_Msk  	(0x00000001U  << GPDMA1_MASKBLOCK_CH2_Pos)

#define   GPDMA1_MASKBLOCK_CH3_Pos  	(3U)
#define   GPDMA1_MASKBLOCK_CH3_Msk  	(0x00000001U  << GPDMA1_MASKBLOCK_CH3_Pos)

/* GPDMA1_MASKSRCTRAN  =  Mask for Raw IntSrcTran Status*/
#define   GPDMA1_MASKSRCTRAN_WE_CH0_Pos  	(8U)
#define   GPDMA1_MASKSRCTRAN_WE_CH0_Msk  	(0x00000001U  << GPDMA1_MASKSRCTRAN_WE_CH0_Pos)

#define   GPDMA1_MASKSRCTRAN_WE_CH1_Pos  	(9U)
#define   GPDMA1_MASKSRCTRAN_WE_CH1_Msk  	(0x00000001U  << GPDMA1_MASKSRCTRAN_WE_CH1_Pos)

#define   GPDMA1_MASKSRCTRAN_WE_CH2_Pos  	(10U)
#define   GPDMA1_MASKSRCTRAN_WE_CH2_Msk  	(0x00000001U  << GPDMA1_MASKSRCTRAN_WE_CH2_Pos)

#define   GPDMA1_MASKSRCTRAN_WE_CH3_Pos  	(11U)
#define   GPDMA1_MASKSRCTRAN_WE_CH3_Msk  	(0x00000001U  << GPDMA1_MASKSRCTRAN_WE_CH3_Pos)

#define   GPDMA1_MASKSRCTRAN_CH0_Pos  	(0U)
#define   GPDMA1_MASKSRCTRAN_CH0_Msk  	(0x00000001U  << GPDMA1_MASKSRCTRAN_CH0_Pos)

#define   GPDMA1_MASKSRCTRAN_CH1_Pos  	(1U)
#define   GPDMA1_MASKSRCTRAN_CH1_Msk  	(0x00000001U  << GPDMA1_MASKSRCTRAN_CH1_Pos)

#define   GPDMA1_MASKSRCTRAN_CH2_Pos  	(2U)
#define   GPDMA1_MASKSRCTRAN_CH2_Msk  	(0x00000001U  << GPDMA1_MASKSRCTRAN_CH2_Pos)

#define   GPDMA1_MASKSRCTRAN_CH3_Pos  	(3U)
#define   GPDMA1_MASKSRCTRAN_CH3_Msk  	(0x00000001U  << GPDMA1_MASKSRCTRAN_CH3_Pos)

/* GPDMA1_MASKDSTTRAN  =  Mask for Raw IntBlock Status*/
#define   GPDMA1_MASKDSTTRAN_WE_CH0_Pos  	(8U)
#define   GPDMA1_MASKDSTTRAN_WE_CH0_Msk  	(0x00000001U  << GPDMA1_MASKDSTTRAN_WE_CH0_Pos)

#define   GPDMA1_MASKDSTTRAN_WE_CH1_Pos  	(9U)
#define   GPDMA1_MASKDSTTRAN_WE_CH1_Msk  	(0x00000001U  << GPDMA1_MASKDSTTRAN_WE_CH1_Pos)

#define   GPDMA1_MASKDSTTRAN_WE_CH2_Pos  	(10U)
#define   GPDMA1_MASKDSTTRAN_WE_CH2_Msk  	(0x00000001U  << GPDMA1_MASKDSTTRAN_WE_CH2_Pos)

#define   GPDMA1_MASKDSTTRAN_WE_CH3_Pos  	(11U)
#define   GPDMA1_MASKDSTTRAN_WE_CH3_Msk  	(0x00000001U  << GPDMA1_MASKDSTTRAN_WE_CH3_Pos)

#define   GPDMA1_MASKDSTTRAN_CH0_Pos  	(0U)
#define   GPDMA1_MASKDSTTRAN_CH0_Msk  	(0x00000001U  << GPDMA1_MASKDSTTRAN_CH0_Pos)

#define   GPDMA1_MASKDSTTRAN_CH1_Pos  	(1U)
#define   GPDMA1_MASKDSTTRAN_CH1_Msk  	(0x00000001U  << GPDMA1_MASKDSTTRAN_CH1_Pos)

#define   GPDMA1_MASKDSTTRAN_CH2_Pos  	(2U)
#define   GPDMA1_MASKDSTTRAN_CH2_Msk  	(0x00000001U  << GPDMA1_MASKDSTTRAN_CH2_Pos)

#define   GPDMA1_MASKDSTTRAN_CH3_Pos  	(3U)
#define   GPDMA1_MASKDSTTRAN_CH3_Msk  	(0x00000001U  << GPDMA1_MASKDSTTRAN_CH3_Pos)

/* GPDMA1_MASKERR  =  Mask for Raw IntErr Status*/
#define   GPDMA1_MASKERR_WE_CH0_Pos  	(8U)
#define   GPDMA1_MASKERR_WE_CH0_Msk  	(0x00000001U  << GPDMA1_MASKERR_WE_CH0_Pos)

#define   GPDMA1_MASKERR_WE_CH1_Pos  	(9U)
#define   GPDMA1_MASKERR_WE_CH1_Msk  	(0x00000001U  << GPDMA1_MASKERR_WE_CH1_Pos)

#define   GPDMA1_MASKERR_WE_CH2_Pos  	(10U)
#define   GPDMA1_MASKERR_WE_CH2_Msk  	(0x00000001U  << GPDMA1_MASKERR_WE_CH2_Pos)

#define   GPDMA1_MASKERR_WE_CH3_Pos  	(11U)
#define   GPDMA1_MASKERR_WE_CH3_Msk  	(0x00000001U  << GPDMA1_MASKERR_WE_CH3_Pos)

#define   GPDMA1_MASKERR_CH0_Pos  	(0U)
#define   GPDMA1_MASKERR_CH0_Msk  	(0x00000001U  << GPDMA1_MASKERR_CH0_Pos)

#define   GPDMA1_MASKERR_CH1_Pos  	(1U)
#define   GPDMA1_MASKERR_CH1_Msk  	(0x00000001U  << GPDMA1_MASKERR_CH1_Pos)

#define   GPDMA1_MASKERR_CH2_Pos  	(2U)
#define   GPDMA1_MASKERR_CH2_Msk  	(0x00000001U  << GPDMA1_MASKERR_CH2_Pos)

#define   GPDMA1_MASKERR_CH3_Pos  	(3U)
#define   GPDMA1_MASKERR_CH3_Msk  	(0x00000001U  << GPDMA1_MASKERR_CH3_Pos)

/* GPDMA1_CLEARTFR  =  IntTfr Status*/
#define   GPDMA1_CLEARTFR_CH0_Pos  	(0U)
#define   GPDMA1_CLEARTFR_CH0_Msk  	(0x00000001U  << GPDMA1_CLEARTFR_CH0_Pos)

#define   GPDMA1_CLEARTFR_CH1_Pos  	(1U)
#define   GPDMA1_CLEARTFR_CH1_Msk  	(0x00000001U  << GPDMA1_CLEARTFR_CH1_Pos)

#define   GPDMA1_CLEARTFR_CH2_Pos  	(2U)
#define   GPDMA1_CLEARTFR_CH2_Msk  	(0x00000001U  << GPDMA1_CLEARTFR_CH2_Pos)

#define   GPDMA1_CLEARTFR_CH3_Pos  	(3U)
#define   GPDMA1_CLEARTFR_CH3_Msk  	(0x00000001U  << GPDMA1_CLEARTFR_CH3_Pos)

/* GPDMA1_CLEARBLOCK  =  IntBlock Status*/
#define   GPDMA1_CLEARBLOCK_CH0_Pos  	(0U)
#define   GPDMA1_CLEARBLOCK_CH0_Msk  	(0x00000001U  << GPDMA1_CLEARBLOCK_CH0_Pos)

#define   GPDMA1_CLEARBLOCK_CH1_Pos  	(1U)
#define   GPDMA1_CLEARBLOCK_CH1_Msk  	(0x00000001U  << GPDMA1_CLEARBLOCK_CH1_Pos)

#define   GPDMA1_CLEARBLOCK_CH2_Pos  	(2U)
#define   GPDMA1_CLEARBLOCK_CH2_Msk  	(0x00000001U  << GPDMA1_CLEARBLOCK_CH2_Pos)

#define   GPDMA1_CLEARBLOCK_CH3_Pos  	(3U)
#define   GPDMA1_CLEARBLOCK_CH3_Msk  	(0x00000001U  << GPDMA1_CLEARBLOCK_CH3_Pos)

/* GPDMA1_CLEARSRCTRAN  =  IntSrcTran Status*/
#define   GPDMA1_CLEARSRCTRAN_CH0_Pos  	(0U)
#define   GPDMA1_CLEARSRCTRAN_CH0_Msk  	(0x00000001U  << GPDMA1_CLEARSRCTRAN_CH0_Pos)

#define   GPDMA1_CLEARSRCTRAN_CH1_Pos  	(1U)
#define   GPDMA1_CLEARSRCTRAN_CH1_Msk  	(0x00000001U  << GPDMA1_CLEARSRCTRAN_CH1_Pos)

#define   GPDMA1_CLEARSRCTRAN_CH2_Pos  	(2U)
#define   GPDMA1_CLEARSRCTRAN_CH2_Msk  	(0x00000001U  << GPDMA1_CLEARSRCTRAN_CH2_Pos)

#define   GPDMA1_CLEARSRCTRAN_CH3_Pos  	(3U)
#define   GPDMA1_CLEARSRCTRAN_CH3_Msk  	(0x00000001U  << GPDMA1_CLEARSRCTRAN_CH3_Pos)

/* GPDMA1_CLEARDSTTRAN  =  IntBlock Status*/
#define   GPDMA1_CLEARDSTTRAN_CH0_Pos  	(0U)
#define   GPDMA1_CLEARDSTTRAN_CH0_Msk  	(0x00000001U  << GPDMA1_CLEARDSTTRAN_CH0_Pos)

#define   GPDMA1_CLEARDSTTRAN_CH1_Pos  	(1U)
#define   GPDMA1_CLEARDSTTRAN_CH1_Msk  	(0x00000001U  << GPDMA1_CLEARDSTTRAN_CH1_Pos)

#define   GPDMA1_CLEARDSTTRAN_CH2_Pos  	(2U)
#define   GPDMA1_CLEARDSTTRAN_CH2_Msk  	(0x00000001U  << GPDMA1_CLEARDSTTRAN_CH2_Pos)

#define   GPDMA1_CLEARDSTTRAN_CH3_Pos  	(3U)
#define   GPDMA1_CLEARDSTTRAN_CH3_Msk  	(0x00000001U  << GPDMA1_CLEARDSTTRAN_CH3_Pos)

/* GPDMA1_CLEARERR  =  IntErr Status*/
#define   GPDMA1_CLEARERR_CH0_Pos  	(0U)
#define   GPDMA1_CLEARERR_CH0_Msk  	(0x00000001U  << GPDMA1_CLEARERR_CH0_Pos)

#define   GPDMA1_CLEARERR_CH1_Pos  	(1U)
#define   GPDMA1_CLEARERR_CH1_Msk  	(0x00000001U  << GPDMA1_CLEARERR_CH1_Pos)

#define   GPDMA1_CLEARERR_CH2_Pos  	(2U)
#define   GPDMA1_CLEARERR_CH2_Msk  	(0x00000001U  << GPDMA1_CLEARERR_CH2_Pos)

#define   GPDMA1_CLEARERR_CH3_Pos  	(3U)
#define   GPDMA1_CLEARERR_CH3_Msk  	(0x00000001U  << GPDMA1_CLEARERR_CH3_Pos)

/* GPDMA1_STATUSINT  =  Combined Interrupt Status Register*/
#define   GPDMA1_STATUSINT_ERR_Pos  	(4U)
#define   GPDMA1_STATUSINT_ERR_Msk  	(0x00000001U  << GPDMA1_STATUSINT_ERR_Pos)

#define   GPDMA1_STATUSINT_DSTT_Pos  	(3U)
#define   GPDMA1_STATUSINT_DSTT_Msk  	(0x00000001U  << GPDMA1_STATUSINT_DSTT_Pos)

#define   GPDMA1_STATUSINT_SRCT_Pos  	(2U)
#define   GPDMA1_STATUSINT_SRCT_Msk  	(0x00000001U  << GPDMA1_STATUSINT_SRCT_Pos)

#define   GPDMA1_STATUSINT_BLOCK_Pos  	(1U)
#define   GPDMA1_STATUSINT_BLOCK_Msk  	(0x00000001U  << GPDMA1_STATUSINT_BLOCK_Pos)

#define   GPDMA1_STATUSINT_TFR_Pos  	(0U)
#define   GPDMA1_STATUSINT_TFR_Msk  	(0x00000001U  << GPDMA1_STATUSINT_TFR_Pos)

/* GPDMA1_REQSRCREG  =  Source Software Transaction Request Register*/
#define   GPDMA1_REQSRCREG_WE_CH0_Pos  	(8U)
#define   GPDMA1_REQSRCREG_WE_CH0_Msk  	(0x00000001U  << GPDMA1_REQSRCREG_WE_CH0_Pos)

#define   GPDMA1_REQSRCREG_WE_CH1_Pos  	(9U)
#define   GPDMA1_REQSRCREG_WE_CH1_Msk  	(0x00000001U  << GPDMA1_REQSRCREG_WE_CH1_Pos)

#define   GPDMA1_REQSRCREG_WE_CH2_Pos  	(10U)
#define   GPDMA1_REQSRCREG_WE_CH2_Msk  	(0x00000001U  << GPDMA1_REQSRCREG_WE_CH2_Pos)

#define   GPDMA1_REQSRCREG_WE_CH3_Pos  	(11U)
#define   GPDMA1_REQSRCREG_WE_CH3_Msk  	(0x00000001U  << GPDMA1_REQSRCREG_WE_CH3_Pos)

#define   GPDMA1_REQSRCREG_CH0_Pos  	(0U)
#define   GPDMA1_REQSRCREG_CH0_Msk  	(0x00000001U  << GPDMA1_REQSRCREG_CH0_Pos)

#define   GPDMA1_REQSRCREG_CH1_Pos  	(1U)
#define   GPDMA1_REQSRCREG_CH1_Msk  	(0x00000001U  << GPDMA1_REQSRCREG_CH1_Pos)

#define   GPDMA1_REQSRCREG_CH2_Pos  	(2U)
#define   GPDMA1_REQSRCREG_CH2_Msk  	(0x00000001U  << GPDMA1_REQSRCREG_CH2_Pos)

#define   GPDMA1_REQSRCREG_CH3_Pos  	(3U)
#define   GPDMA1_REQSRCREG_CH3_Msk  	(0x00000001U  << GPDMA1_REQSRCREG_CH3_Pos)

/* GPDMA1_REQDSTREG  =  Destination Software Transaction Request Register*/
#define   GPDMA1_REQDSTREG_WE_CH0_Pos  	(8U)
#define   GPDMA1_REQDSTREG_WE_CH0_Msk  	(0x00000001U  << GPDMA1_REQDSTREG_WE_CH0_Pos)

#define   GPDMA1_REQDSTREG_WE_CH1_Pos  	(9U)
#define   GPDMA1_REQDSTREG_WE_CH1_Msk  	(0x00000001U  << GPDMA1_REQDSTREG_WE_CH1_Pos)

#define   GPDMA1_REQDSTREG_WE_CH2_Pos  	(10U)
#define   GPDMA1_REQDSTREG_WE_CH2_Msk  	(0x00000001U  << GPDMA1_REQDSTREG_WE_CH2_Pos)

#define   GPDMA1_REQDSTREG_WE_CH3_Pos  	(11U)
#define   GPDMA1_REQDSTREG_WE_CH3_Msk  	(0x00000001U  << GPDMA1_REQDSTREG_WE_CH3_Pos)

#define   GPDMA1_REQDSTREG_CH0_Pos  	(0U)
#define   GPDMA1_REQDSTREG_CH0_Msk  	(0x00000001U  << GPDMA1_REQDSTREG_CH0_Pos)

#define   GPDMA1_REQDSTREG_CH1_Pos  	(1U)
#define   GPDMA1_REQDSTREG_CH1_Msk  	(0x00000001U  << GPDMA1_REQDSTREG_CH1_Pos)

#define   GPDMA1_REQDSTREG_CH2_Pos  	(2U)
#define   GPDMA1_REQDSTREG_CH2_Msk  	(0x00000001U  << GPDMA1_REQDSTREG_CH2_Pos)

#define   GPDMA1_REQDSTREG_CH3_Pos  	(3U)
#define   GPDMA1_REQDSTREG_CH3_Msk  	(0x00000001U  << GPDMA1_REQDSTREG_CH3_Pos)

/* GPDMA1_SGLREQSRCREG  =  Single Source Transaction Request Register*/
#define   GPDMA1_SGLREQSRCREG_WE_CH0_Pos  	(8U)
#define   GPDMA1_SGLREQSRCREG_WE_CH0_Msk  	(0x00000001U  << GPDMA1_SGLREQSRCREG_WE_CH0_Pos)

#define   GPDMA1_SGLREQSRCREG_WE_CH1_Pos  	(9U)
#define   GPDMA1_SGLREQSRCREG_WE_CH1_Msk  	(0x00000001U  << GPDMA1_SGLREQSRCREG_WE_CH1_Pos)

#define   GPDMA1_SGLREQSRCREG_WE_CH2_Pos  	(10U)
#define   GPDMA1_SGLREQSRCREG_WE_CH2_Msk  	(0x00000001U  << GPDMA1_SGLREQSRCREG_WE_CH2_Pos)

#define   GPDMA1_SGLREQSRCREG_WE_CH3_Pos  	(11U)
#define   GPDMA1_SGLREQSRCREG_WE_CH3_Msk  	(0x00000001U  << GPDMA1_SGLREQSRCREG_WE_CH3_Pos)

#define   GPDMA1_SGLREQSRCREG_CH0_Pos  	(0U)
#define   GPDMA1_SGLREQSRCREG_CH0_Msk  	(0x00000001U  << GPDMA1_SGLREQSRCREG_CH0_Pos)

#define   GPDMA1_SGLREQSRCREG_CH1_Pos  	(1U)
#define   GPDMA1_SGLREQSRCREG_CH1_Msk  	(0x00000001U  << GPDMA1_SGLREQSRCREG_CH1_Pos)

#define   GPDMA1_SGLREQSRCREG_CH2_Pos  	(2U)
#define   GPDMA1_SGLREQSRCREG_CH2_Msk  	(0x00000001U  << GPDMA1_SGLREQSRCREG_CH2_Pos)

#define   GPDMA1_SGLREQSRCREG_CH3_Pos  	(3U)
#define   GPDMA1_SGLREQSRCREG_CH3_Msk  	(0x00000001U  << GPDMA1_SGLREQSRCREG_CH3_Pos)

/* GPDMA1_SGLREQDSTREG  =  Single Destination Transaction Request Register*/
#define   GPDMA1_SGLREQDSTREG_WE_CH0_Pos  	(8U)
#define   GPDMA1_SGLREQDSTREG_WE_CH0_Msk  	(0x00000001U  << GPDMA1_SGLREQDSTREG_WE_CH0_Pos)

#define   GPDMA1_SGLREQDSTREG_WE_CH1_Pos  	(9U)
#define   GPDMA1_SGLREQDSTREG_WE_CH1_Msk  	(0x00000001U  << GPDMA1_SGLREQDSTREG_WE_CH1_Pos)

#define   GPDMA1_SGLREQDSTREG_WE_CH2_Pos  	(10U)
#define   GPDMA1_SGLREQDSTREG_WE_CH2_Msk  	(0x00000001U  << GPDMA1_SGLREQDSTREG_WE_CH2_Pos)

#define   GPDMA1_SGLREQDSTREG_WE_CH3_Pos  	(11U)
#define   GPDMA1_SGLREQDSTREG_WE_CH3_Msk  	(0x00000001U  << GPDMA1_SGLREQDSTREG_WE_CH3_Pos)

#define   GPDMA1_SGLREQDSTREG_CH0_Pos  	(0U)
#define   GPDMA1_SGLREQDSTREG_CH0_Msk  	(0x00000001U  << GPDMA1_SGLREQDSTREG_CH0_Pos)

#define   GPDMA1_SGLREQDSTREG_CH1_Pos  	(1U)
#define   GPDMA1_SGLREQDSTREG_CH1_Msk  	(0x00000001U  << GPDMA1_SGLREQDSTREG_CH1_Pos)

#define   GPDMA1_SGLREQDSTREG_CH2_Pos  	(2U)
#define   GPDMA1_SGLREQDSTREG_CH2_Msk  	(0x00000001U  << GPDMA1_SGLREQDSTREG_CH2_Pos)

#define   GPDMA1_SGLREQDSTREG_CH3_Pos  	(3U)
#define   GPDMA1_SGLREQDSTREG_CH3_Msk  	(0x00000001U  << GPDMA1_SGLREQDSTREG_CH3_Pos)

/* GPDMA1_LSTSRCREG  =  Last Source Transaction Request Register*/
#define   GPDMA1_LSTSRCREG_WE_CH0_Pos  	(8U)
#define   GPDMA1_LSTSRCREG_WE_CH0_Msk  	(0x00000001U  << GPDMA1_LSTSRCREG_WE_CH0_Pos)

#define   GPDMA1_LSTSRCREG_WE_CH1_Pos  	(9U)
#define   GPDMA1_LSTSRCREG_WE_CH1_Msk  	(0x00000001U  << GPDMA1_LSTSRCREG_WE_CH1_Pos)

#define   GPDMA1_LSTSRCREG_WE_CH2_Pos  	(10U)
#define   GPDMA1_LSTSRCREG_WE_CH2_Msk  	(0x00000001U  << GPDMA1_LSTSRCREG_WE_CH2_Pos)

#define   GPDMA1_LSTSRCREG_WE_CH3_Pos  	(11U)
#define   GPDMA1_LSTSRCREG_WE_CH3_Msk  	(0x00000001U  << GPDMA1_LSTSRCREG_WE_CH3_Pos)

#define   GPDMA1_LSTSRCREG_CH0_Pos  	(0U)
#define   GPDMA1_LSTSRCREG_CH0_Msk  	(0x00000001U  << GPDMA1_LSTSRCREG_CH0_Pos)

#define   GPDMA1_LSTSRCREG_CH1_Pos  	(1U)
#define   GPDMA1_LSTSRCREG_CH1_Msk  	(0x00000001U  << GPDMA1_LSTSRCREG_CH1_Pos)

#define   GPDMA1_LSTSRCREG_CH2_Pos  	(2U)
#define   GPDMA1_LSTSRCREG_CH2_Msk  	(0x00000001U  << GPDMA1_LSTSRCREG_CH2_Pos)

#define   GPDMA1_LSTSRCREG_CH3_Pos  	(3U)
#define   GPDMA1_LSTSRCREG_CH3_Msk  	(0x00000001U  << GPDMA1_LSTSRCREG_CH3_Pos)

/* GPDMA1_LSTDSTREG  =  Last Destination Transaction Request Register*/
#define   GPDMA1_LSTDSTREG_WE_CH0_Pos  	(8U)
#define   GPDMA1_LSTDSTREG_WE_CH0_Msk  	(0x00000001U  << GPDMA1_LSTDSTREG_WE_CH0_Pos)

#define   GPDMA1_LSTDSTREG_WE_CH1_Pos  	(9U)
#define   GPDMA1_LSTDSTREG_WE_CH1_Msk  	(0x00000001U  << GPDMA1_LSTDSTREG_WE_CH1_Pos)

#define   GPDMA1_LSTDSTREG_WE_CH2_Pos  	(10U)
#define   GPDMA1_LSTDSTREG_WE_CH2_Msk  	(0x00000001U  << GPDMA1_LSTDSTREG_WE_CH2_Pos)

#define   GPDMA1_LSTDSTREG_WE_CH3_Pos  	(11U)
#define   GPDMA1_LSTDSTREG_WE_CH3_Msk  	(0x00000001U  << GPDMA1_LSTDSTREG_WE_CH3_Pos)

#define   GPDMA1_LSTDSTREG_CH0_Pos  	(0U)
#define   GPDMA1_LSTDSTREG_CH0_Msk  	(0x00000001U  << GPDMA1_LSTDSTREG_CH0_Pos)

#define   GPDMA1_LSTDSTREG_CH1_Pos  	(1U)
#define   GPDMA1_LSTDSTREG_CH1_Msk  	(0x00000001U  << GPDMA1_LSTDSTREG_CH1_Pos)

#define   GPDMA1_LSTDSTREG_CH2_Pos  	(2U)
#define   GPDMA1_LSTDSTREG_CH2_Msk  	(0x00000001U  << GPDMA1_LSTDSTREG_CH2_Pos)

#define   GPDMA1_LSTDSTREG_CH3_Pos  	(3U)
#define   GPDMA1_LSTDSTREG_CH3_Msk  	(0x00000001U  << GPDMA1_LSTDSTREG_CH3_Pos)

/* GPDMA1_DMACFGREG  =  GPDMA Configuration Register*/
#define   GPDMA1_DMACFGREG_DMA_EN_Pos  	(0U)
#define   GPDMA1_DMACFGREG_DMA_EN_Msk  	(0x00000001U  << GPDMA1_DMACFGREG_DMA_EN_Pos)

/* GPDMA1_CHENREG  =  GPDMA Channel Enable Register*/
#define   GPDMA1_CHENREG_WE_CH_Pos  	(8U)
#define   GPDMA1_CHENREG_WE_CH_Msk  	(0x0000000FU  << GPDMA1_CHENREG_WE_CH_Pos)

#define   GPDMA1_CHENREG_CH_Pos  	(0U)
#define   GPDMA1_CHENREG_CH_Msk  	(0x0000000FU  << GPDMA1_CHENREG_CH_Pos)

/* GPDMA1_ID  =  GPDMA1 ID Register*/
#define   GPDMA1_ID_VALUE_Pos  	(0U)
#define   GPDMA1_ID_VALUE_Msk  	(0xFFFFFFFFU  << GPDMA1_ID_VALUE_Pos)

/* GPDMA1_TYPE  =  GPDMA Component Type*/
#define   GPDMA1_TYPE_VALUE_Pos  	(0U)
#define   GPDMA1_TYPE_VALUE_Msk  	(0xFFFFFFFFU  << GPDMA1_TYPE_VALUE_Pos)

/* GPDMA1_VERSION  =  DMA Component Version*/
#define   GPDMA1_VERSION_VALUE_Pos  	(0U)
#define   GPDMA1_VERSION_VALUE_Msk  	(0xFFFFFFFFU  << GPDMA1_VERSION_VALUE_Pos)



/***   GPDMA0 Bit Fileds *******************/
/***************************************************************************/


/* GPDMA0_CH_SARx  =  Source Address Register for Channel 0*/
#define   GPDMA0_CH_SAR_SAR_Pos  	(0U)
#define   GPDMA0_CH_SAR_SAR_Msk  	(0xFFFFFFFFU  << GPDMA0_CH_SAR_SAR_Pos)

/* GPDMA0_CH_DARx  =  Destination Address Register for Channel 0*/
#define   GPDMA0_CH_DAR_DAR_Pos  	(0U)
#define   GPDMA0_CH_DAR_DAR_Msk  	(0xFFFFFFFFU  << GPDMA0_CH_DAR_DAR_Pos)

/* GPDMA0_CH_LLPx  =  Linked List Pointer Register for Channel 0*/
#define   GPDMA0_CH_LLP_LOC_Pos  	(2U)
#define   GPDMA0_CH_LLP_LOC_Msk  	(0x3FFFFFFFU  << GPDMA0_CH_LLP_LOC_Pos)

/* GPDMA0_CH_CTLL  =  Control Register for Channel x Low Word*/
#define   GPDMA0_CH_CTLL_LLP_SRC_EN_Pos  	(28U)
#define   GPDMA0_CH_CTLL_LLP_SRC_EN_Msk  	(0x00000001U  << GPDMA0_CH_CTLL_LLP_SRC_EN_Pos)

#define   GPDMA0_CH_CTLL_LLP_DST_EN_Pos  	(27U)
#define   GPDMA0_CH_CTLL_LLP_DST_EN_Msk  	(0x00000001U  << GPDMA0_CH_CTLL_LLP_DST_EN_Pos)

#define   GPDMA0_CH_CTLL_TT_FC_Pos  	(20U)
#define   GPDMA0_CH_CTLL_TT_FC_Msk  	(0x00000007U  << GPDMA0_CH_CTLL_TT_FC_Pos)

#define   GPDMA0_CH_CTLL_DST_SCATTER_EN_Pos  	(18U)
#define   GPDMA0_CH_CTLL_DST_SCATTER_EN_Msk  	(0x00000001U  << GPDMA0_CH_CTLL_DST_SCATTER_EN_Pos)

#define   GPDMA0_CH_CTLL_SRC_GATHER_EN_Pos  	(17U)
#define   GPDMA0_CH_CTLL_SRC_GATHER_EN_Msk  	(0x00000001U  << GPDMA0_CH_CTLL_SRC_GATHER_EN_Pos)

#define   GPDMA0_CH_CTLL_SRC_MSIZE_Pos  	(14U)
#define   GPDMA0_CH_CTLL_SRC_MSIZE_Msk  	(0x00000007U  << GPDMA0_CH_CTLL_SRC_MSIZE_Pos)

#define   GPDMA0_CH_CTLL_DEST_MSIZE_Pos  	(11U)
#define   GPDMA0_CH_CTLL_DEST_MSIZE_Msk  	(0x00000007U  << GPDMA0_CH_CTLL_DEST_MSIZE_Pos)

#define   GPDMA0_CH_CTLL_SINC_Pos  	(9U)
#define   GPDMA0_CH_CTLL_SINC_Msk  	(0x00000003U  << GPDMA0_CH_CTLL_SINC_Pos)

#define   GPDMA0_CH_CTLL_DINC_Pos  	(7U)
#define   GPDMA0_CH_CTLL_DINC_Msk  	(0x00000003U  << GPDMA0_CH_CTLL_DINC_Pos)

#define   GPDMA0_CH_CTLL_SRC_TR_WIDTH_Pos  	(4U)
#define   GPDMA0_CH_CTLL_SRC_TR_WIDTH_Msk  	(0x00000007U  << GPDMA0_CH_CTLL_SRC_TR_WIDTH_Pos)

#define   GPDMA0_CH_CTLL_DST_TR_WIDTH_Pos  	(1U)
#define   GPDMA0_CH_CTLL_DST_TR_WIDTH_Msk  	(0x00000007U  << GPDMA0_CH_CTLL_DST_TR_WIDTH_Pos)

#define   GPDMA0_CH_CTLL_INT_EN_Pos  	(0U)
#define   GPDMA0_CH_CTLL_INT_EN_Msk  	(0x00000001U  << GPDMA0_CH_CTLL_INT_EN_Pos)

/* GPDMA0_CH_CTLH  =  Control Register for Channel x High Word*/
#define   GPDMA0_CH_CTLH_DONE_Pos  	(12U)
#define   GPDMA0_CH_CTLH_DONE_Msk  	(0x00000001U  << GPDMA0_CH_CTLH_DONE_Pos)

#define   GPDMA0_CH_CTLH_BLOCK_TS_Pos  	(0U)
#define   GPDMA0_CH_CTLH_BLOCK_TS_Msk  	(0x00000FFFU  << GPDMA0_CH_CTLH_BLOCK_TS_Pos)

/* GPDMA0_CH_SSTATx  =  Source Status Register for Channel 0*/
#define   GPDMA0_CH_SSTAT_SSTAT_Pos  	(0U)
#define   GPDMA0_CH_SSTAT_SSTAT_Msk  	(0xFFFFFFFFU  << GPDMA0_CH_SSTAT_SSTAT_Pos)

/* GPDMA0_CH_DSTATx  =  Destination Status Register for Channel 0*/
#define   GPDMA0_CH_DSTAT_DSTAT_Pos  	(0U)
#define   GPDMA0_CH_DSTAT_DSTAT_Msk  	(0xFFFFFFFFU  << GPDMA0_CH_DSTAT_DSTAT_Pos)

/* GPDMA0_CH_SSTATARx  =  Source Status Address Register for Channel 0*/
#define   GPDMA0_CH_SSTATAR_SSTATAR_Pos  	(0U)
#define   GPDMA0_CH_SSTATAR_SSTATAR_Msk  	(0xFFFFFFFFU  << GPDMA0_CH_SSTATAR_SSTATAR_Pos)

/* GPDMA0_CH_DSTATARx  =  Destination Status Address Register for Channel 0*/
#define   GPDMA0_CH_DSTATAR_DSTATAR_Pos  	(0U)
#define   GPDMA0_CH_DSTATAR_DSTATAR_Msk  	(0xFFFFFFFFU  << GPDMA0_CH_DSTATAR_DSTATAR_Pos)

/* GPDMA0_CH_CFGL  =  Configuration Register for Channel x Low Word*/
#define   GPDMA0_CH_CFGL_RELOAD_DST_Pos  	(31U)
#define   GPDMA0_CH_CFGL_RELOAD_DST_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_RELOAD_DST_Pos)

#define   GPDMA0_CH_CFGL_RELOAD_SRC_Pos  	(30U)
#define   GPDMA0_CH_CFGL_RELOAD_SRC_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_RELOAD_SRC_Pos)

#define   GPDMA0_CH_CFGL_MAX_ABRST_Pos  	(20U)
#define   GPDMA0_CH_CFGL_MAX_ABRST_Msk  	(0x000003FFU  << GPDMA0_CH_CFGL_MAX_ABRST_Pos)

#define   GPDMA0_CH_CFGL_SRC_HS_POL_Pos  	(19U)
#define   GPDMA0_CH_CFGL_SRC_HS_POL_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_SRC_HS_POL_Pos)

#define   GPDMA0_CH_CFGL_DST_HS_POL_Pos  	(18U)
#define   GPDMA0_CH_CFGL_DST_HS_POL_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_DST_HS_POL_Pos)

#define   GPDMA0_CH_CFGL_LOCK_B_Pos  	(17U)
#define   GPDMA0_CH_CFGL_LOCK_B_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_LOCK_B_Pos)

#define   GPDMA0_CH_CFGL_LOCK_CH_Pos  	(16U)
#define   GPDMA0_CH_CFGL_LOCK_CH_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_LOCK_CH_Pos)

#define   GPDMA0_CH_CFGL_LOCK_B_L_Pos  	(14U)
#define   GPDMA0_CH_CFGL_LOCK_B_L_Msk  	(0x00000003U  << GPDMA0_CH_CFGL_LOCK_B_L_Pos)

#define   GPDMA0_CH_CFGL_LOCK_CH_L_Pos  	(12U)
#define   GPDMA0_CH_CFGL_LOCK_CH_L_Msk  	(0x00000003U  << GPDMA0_CH_CFGL_LOCK_CH_L_Pos)

#define   GPDMA0_CH_CFGL_HS_SEL_SRC_Pos  	(11U)
#define   GPDMA0_CH_CFGL_HS_SEL_SRC_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_HS_SEL_SRC_Pos)

#define   GPDMA0_CH_CFGL_HS_SEL_DST_Pos  	(10U)
#define   GPDMA0_CH_CFGL_HS_SEL_DST_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_HS_SEL_DST_Pos)

#define   GPDMA0_CH_CFGL_FIFO_EMPTY_Pos  	(9U)
#define   GPDMA0_CH_CFGL_FIFO_EMPTY_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_FIFO_EMPTY_Pos)

#define   GPDMA0_CH_CFGL_CH_SUSP_Pos  	(8U)
#define   GPDMA0_CH_CFGL_CH_SUSP_Msk  	(0x00000001U  << GPDMA0_CH_CFGL_CH_SUSP_Pos)

#define   GPDMA0_CH_CFGL_CH_PRIOR_Pos  	(5U)
#define   GPDMA0_CH_CFGL_CH_PRIOR_Msk  	(0x00000007U  << GPDMA0_CH_CFGL_CH_PRIOR_Pos)

/* GPDMA0_CH_CFGH  =  Configuration Register for Channel x High Word*/
#define   GPDMA0_CH_CFGH_DEST_PER_Pos  	(11U)
#define   GPDMA0_CH_CFGH_DEST_PER_Msk  	(0x0000000FU  << GPDMA0_CH_CFGH_DEST_PER_Pos)

#define   GPDMA0_CH_CFGH_SRC_PER_Pos  	(7U)
#define   GPDMA0_CH_CFGH_SRC_PER_Msk  	(0x0000000FU  << GPDMA0_CH_CFGH_SRC_PER_Pos)

#define   GPDMA0_CH_CFGH_SS_UPD_EN_Pos  	(6U)
#define   GPDMA0_CH_CFGH_SS_UPD_EN_Msk  	(0x00000001U  << GPDMA0_CH_CFGH_SS_UPD_EN_Pos)

#define   GPDMA0_CH_CFGH_DS_UPD_EN_Pos  	(5U)
#define   GPDMA0_CH_CFGH_DS_UPD_EN_Msk  	(0x00000001U  << GPDMA0_CH_CFGH_DS_UPD_EN_Pos)

#define   GPDMA0_CH_CFGH_PROTCTL_Pos  	(2U)
#define   GPDMA0_CH_CFGH_PROTCTL_Msk  	(0x00000007U  << GPDMA0_CH_CFGH_PROTCTL_Pos)

#define   GPDMA0_CH_CFGH_FIFO_MODE_Pos  	(1U)
#define   GPDMA0_CH_CFGH_FIFO_MODE_Msk  	(0x00000001U  << GPDMA0_CH_CFGH_FIFO_MODE_Pos)

#define   GPDMA0_CH_CFGH_FCMODE_Pos  	(0U)
#define   GPDMA0_CH_CFGH_FCMODE_Msk  	(0x00000001U  << GPDMA0_CH_CFGH_FCMODE_Pos)

/* GPDMA0_CH_SGRx  =  Source Gather Register for Channel 0*/
#define   GPDMA0_CH_SGR_SGC_Pos  	(20U)
#define   GPDMA0_CH_SGR_SGC_Msk  	(0x00000FFFU  << GPDMA0_CH_SGR_SGC_Pos)

#define   GPDMA0_CH_SGR_SGI_Pos  	(0U)
#define   GPDMA0_CH_SGR_SGI_Msk  	(0x000FFFFFU  << GPDMA0_CH_SGR_SGI_Pos)

/* GPDMA0_CH_DSRx  =  Destination Scatter Register for Channel 0*/
#define   GPDMA0_CH_DSR_DSC_Pos  	(20U)
#define   GPDMA0_CH_DSR_DSC_Msk  	(0x00000FFFU  << GPDMA0_CH_DSR_DSC_Pos)

#define   GPDMA0_CH_DSR_DSI_Pos  	(0U)
#define   GPDMA0_CH_DSR_DSI_Msk  	(0x000FFFFFU  << GPDMA0_CH_DSR_DSI_Pos)

/* GPDMA0_RAWTFR  =  Raw IntTfr Status*/
#define   GPDMA0_RAWTFR_CH0_Pos  	(0U)
#define   GPDMA0_RAWTFR_CH0_Msk  	(0x00000001U  << GPDMA0_RAWTFR_CH0_Pos)

#define   GPDMA0_RAWTFR_CH1_Pos  	(1U)
#define   GPDMA0_RAWTFR_CH1_Msk  	(0x00000001U  << GPDMA0_RAWTFR_CH1_Pos)

#define   GPDMA0_RAWTFR_CH2_Pos  	(2U)
#define   GPDMA0_RAWTFR_CH2_Msk  	(0x00000001U  << GPDMA0_RAWTFR_CH2_Pos)

#define   GPDMA0_RAWTFR_CH3_Pos  	(3U)
#define   GPDMA0_RAWTFR_CH3_Msk  	(0x00000001U  << GPDMA0_RAWTFR_CH3_Pos)

#define   GPDMA0_RAWTFR_CH4_Pos  	(4U)
#define   GPDMA0_RAWTFR_CH4_Msk  	(0x00000001U  << GPDMA0_RAWTFR_CH4_Pos)

#define   GPDMA0_RAWTFR_CH5_Pos  	(5U)
#define   GPDMA0_RAWTFR_CH5_Msk  	(0x00000001U  << GPDMA0_RAWTFR_CH5_Pos)

#define   GPDMA0_RAWTFR_CH6_Pos  	(6U)
#define   GPDMA0_RAWTFR_CH6_Msk  	(0x00000001U  << GPDMA0_RAWTFR_CH6_Pos)

#define   GPDMA0_RAWTFR_CH7_Pos  	(7U)
#define   GPDMA0_RAWTFR_CH7_Msk  	(0x00000001U  << GPDMA0_RAWTFR_CH7_Pos)

/* GPDMA0_RAWBLOCK  =  Raw IntBlock Status*/
#define   GPDMA0_RAWBLOCK_CH0_Pos  	(0U)
#define   GPDMA0_RAWBLOCK_CH0_Msk  	(0x00000001U  << GPDMA0_RAWBLOCK_CH0_Pos)

#define   GPDMA0_RAWBLOCK_CH1_Pos  	(1U)
#define   GPDMA0_RAWBLOCK_CH1_Msk  	(0x00000001U  << GPDMA0_RAWBLOCK_CH1_Pos)

#define   GPDMA0_RAWBLOCK_CH2_Pos  	(2U)
#define   GPDMA0_RAWBLOCK_CH2_Msk  	(0x00000001U  << GPDMA0_RAWBLOCK_CH2_Pos)

#define   GPDMA0_RAWBLOCK_CH3_Pos  	(3U)
#define   GPDMA0_RAWBLOCK_CH3_Msk  	(0x00000001U  << GPDMA0_RAWBLOCK_CH3_Pos)

#define   GPDMA0_RAWBLOCK_CH4_Pos  	(4U)
#define   GPDMA0_RAWBLOCK_CH4_Msk  	(0x00000001U  << GPDMA0_RAWBLOCK_CH4_Pos)

#define   GPDMA0_RAWBLOCK_CH5_Pos  	(5U)
#define   GPDMA0_RAWBLOCK_CH5_Msk  	(0x00000001U  << GPDMA0_RAWBLOCK_CH5_Pos)

#define   GPDMA0_RAWBLOCK_CH6_Pos  	(6U)
#define   GPDMA0_RAWBLOCK_CH6_Msk  	(0x00000001U  << GPDMA0_RAWBLOCK_CH6_Pos)

#define   GPDMA0_RAWBLOCK_CH7_Pos  	(7U)
#define   GPDMA0_RAWBLOCK_CH7_Msk  	(0x00000001U  << GPDMA0_RAWBLOCK_CH7_Pos)

/* GPDMA0_RAWSRCTRAN  =  Raw IntSrcTran Status*/
#define   GPDMA0_RAWSRCTRAN_CH0_Pos  	(0U)
#define   GPDMA0_RAWSRCTRAN_CH0_Msk  	(0x00000001U  << GPDMA0_RAWSRCTRAN_CH0_Pos)

#define   GPDMA0_RAWSRCTRAN_CH1_Pos  	(1U)
#define   GPDMA0_RAWSRCTRAN_CH1_Msk  	(0x00000001U  << GPDMA0_RAWSRCTRAN_CH1_Pos)

#define   GPDMA0_RAWSRCTRAN_CH2_Pos  	(2U)
#define   GPDMA0_RAWSRCTRAN_CH2_Msk  	(0x00000001U  << GPDMA0_RAWSRCTRAN_CH2_Pos)

#define   GPDMA0_RAWSRCTRAN_CH3_Pos  	(3U)
#define   GPDMA0_RAWSRCTRAN_CH3_Msk  	(0x00000001U  << GPDMA0_RAWSRCTRAN_CH3_Pos)

#define   GPDMA0_RAWSRCTRAN_CH4_Pos  	(4U)
#define   GPDMA0_RAWSRCTRAN_CH4_Msk  	(0x00000001U  << GPDMA0_RAWSRCTRAN_CH4_Pos)

#define   GPDMA0_RAWSRCTRAN_CH5_Pos  	(5U)
#define   GPDMA0_RAWSRCTRAN_CH5_Msk  	(0x00000001U  << GPDMA0_RAWSRCTRAN_CH5_Pos)

#define   GPDMA0_RAWSRCTRAN_CH6_Pos  	(6U)
#define   GPDMA0_RAWSRCTRAN_CH6_Msk  	(0x00000001U  << GPDMA0_RAWSRCTRAN_CH6_Pos)

#define   GPDMA0_RAWSRCTRAN_CH7_Pos  	(7U)
#define   GPDMA0_RAWSRCTRAN_CH7_Msk  	(0x00000001U  << GPDMA0_RAWSRCTRAN_CH7_Pos)

/* GPDMA0_RAWDSTTRAN  =  Raw IntBlock Status*/
#define   GPDMA0_RAWDSTTRAN_CH0_Pos  	(0U)
#define   GPDMA0_RAWDSTTRAN_CH0_Msk  	(0x00000001U  << GPDMA0_RAWDSTTRAN_CH0_Pos)

#define   GPDMA0_RAWDSTTRAN_CH1_Pos  	(1U)
#define   GPDMA0_RAWDSTTRAN_CH1_Msk  	(0x00000001U  << GPDMA0_RAWDSTTRAN_CH1_Pos)

#define   GPDMA0_RAWDSTTRAN_CH2_Pos  	(2U)
#define   GPDMA0_RAWDSTTRAN_CH2_Msk  	(0x00000001U  << GPDMA0_RAWDSTTRAN_CH2_Pos)

#define   GPDMA0_RAWDSTTRAN_CH3_Pos  	(3U)
#define   GPDMA0_RAWDSTTRAN_CH3_Msk  	(0x00000001U  << GPDMA0_RAWDSTTRAN_CH3_Pos)

#define   GPDMA0_RAWDSTTRAN_CH4_Pos  	(4U)
#define   GPDMA0_RAWDSTTRAN_CH4_Msk  	(0x00000001U  << GPDMA0_RAWDSTTRAN_CH4_Pos)

#define   GPDMA0_RAWDSTTRAN_CH5_Pos  	(5U)
#define   GPDMA0_RAWDSTTRAN_CH5_Msk  	(0x00000001U  << GPDMA0_RAWDSTTRAN_CH5_Pos)

#define   GPDMA0_RAWDSTTRAN_CH6_Pos  	(6U)
#define   GPDMA0_RAWDSTTRAN_CH6_Msk  	(0x00000001U  << GPDMA0_RAWDSTTRAN_CH6_Pos)

#define   GPDMA0_RAWDSTTRAN_CH7_Pos  	(7U)
#define   GPDMA0_RAWDSTTRAN_CH7_Msk  	(0x00000001U  << GPDMA0_RAWDSTTRAN_CH7_Pos)

/* GPDMA0_RAWERR  =  Raw IntErr Status*/
#define   GPDMA0_RAWERR_CH0_Pos  	(0U)
#define   GPDMA0_RAWERR_CH0_Msk  	(0x00000001U  << GPDMA0_RAWERR_CH0_Pos)

#define   GPDMA0_RAWERR_CH1_Pos  	(1U)
#define   GPDMA0_RAWERR_CH1_Msk  	(0x00000001U  << GPDMA0_RAWERR_CH1_Pos)

#define   GPDMA0_RAWERR_CH2_Pos  	(2U)
#define   GPDMA0_RAWERR_CH2_Msk  	(0x00000001U  << GPDMA0_RAWERR_CH2_Pos)

#define   GPDMA0_RAWERR_CH3_Pos  	(3U)
#define   GPDMA0_RAWERR_CH3_Msk  	(0x00000001U  << GPDMA0_RAWERR_CH3_Pos)

#define   GPDMA0_RAWERR_CH4_Pos  	(4U)
#define   GPDMA0_RAWERR_CH4_Msk  	(0x00000001U  << GPDMA0_RAWERR_CH4_Pos)

#define   GPDMA0_RAWERR_CH5_Pos  	(5U)
#define   GPDMA0_RAWERR_CH5_Msk  	(0x00000001U  << GPDMA0_RAWERR_CH5_Pos)

#define   GPDMA0_RAWERR_CH6_Pos  	(6U)
#define   GPDMA0_RAWERR_CH6_Msk  	(0x00000001U  << GPDMA0_RAWERR_CH6_Pos)

#define   GPDMA0_RAWERR_CH7_Pos  	(7U)
#define   GPDMA0_RAWERR_CH7_Msk  	(0x00000001U  << GPDMA0_RAWERR_CH7_Pos)

/* GPDMA0_STATUSTFR  =  IntTfr Status*/
#define   GPDMA0_STATUSTFR_CH0_Pos  	(0U)
#define   GPDMA0_STATUSTFR_CH0_Msk  	(0x00000001U  << GPDMA0_STATUSTFR_CH0_Pos)

#define   GPDMA0_STATUSTFR_CH1_Pos  	(1U)
#define   GPDMA0_STATUSTFR_CH1_Msk  	(0x00000001U  << GPDMA0_STATUSTFR_CH1_Pos)

#define   GPDMA0_STATUSTFR_CH2_Pos  	(2U)
#define   GPDMA0_STATUSTFR_CH2_Msk  	(0x00000001U  << GPDMA0_STATUSTFR_CH2_Pos)

#define   GPDMA0_STATUSTFR_CH3_Pos  	(3U)
#define   GPDMA0_STATUSTFR_CH3_Msk  	(0x00000001U  << GPDMA0_STATUSTFR_CH3_Pos)

#define   GPDMA0_STATUSTFR_CH4_Pos  	(4U)
#define   GPDMA0_STATUSTFR_CH4_Msk  	(0x00000001U  << GPDMA0_STATUSTFR_CH4_Pos)

#define   GPDMA0_STATUSTFR_CH5_Pos  	(5U)
#define   GPDMA0_STATUSTFR_CH5_Msk  	(0x00000001U  << GPDMA0_STATUSTFR_CH5_Pos)

#define   GPDMA0_STATUSTFR_CH6_Pos  	(6U)
#define   GPDMA0_STATUSTFR_CH6_Msk  	(0x00000001U  << GPDMA0_STATUSTFR_CH6_Pos)

#define   GPDMA0_STATUSTFR_CH7_Pos  	(7U)
#define   GPDMA0_STATUSTFR_CH7_Msk  	(0x00000001U  << GPDMA0_STATUSTFR_CH7_Pos)

/* GPDMA0_STATUSBLOCK  =  IntBlock Status*/
#define   GPDMA0_STATUSBLOCK_CH0_Pos  	(0U)
#define   GPDMA0_STATUSBLOCK_CH0_Msk  	(0x00000001U  << GPDMA0_STATUSBLOCK_CH0_Pos)

#define   GPDMA0_STATUSBLOCK_CH1_Pos  	(1U)
#define   GPDMA0_STATUSBLOCK_CH1_Msk  	(0x00000001U  << GPDMA0_STATUSBLOCK_CH1_Pos)

#define   GPDMA0_STATUSBLOCK_CH2_Pos  	(2U)
#define   GPDMA0_STATUSBLOCK_CH2_Msk  	(0x00000001U  << GPDMA0_STATUSBLOCK_CH2_Pos)

#define   GPDMA0_STATUSBLOCK_CH3_Pos  	(3U)
#define   GPDMA0_STATUSBLOCK_CH3_Msk  	(0x00000001U  << GPDMA0_STATUSBLOCK_CH3_Pos)

#define   GPDMA0_STATUSBLOCK_CH4_Pos  	(4U)
#define   GPDMA0_STATUSBLOCK_CH4_Msk  	(0x00000001U  << GPDMA0_STATUSBLOCK_CH4_Pos)

#define   GPDMA0_STATUSBLOCK_CH5_Pos  	(5U)
#define   GPDMA0_STATUSBLOCK_CH5_Msk  	(0x00000001U  << GPDMA0_STATUSBLOCK_CH5_Pos)

#define   GPDMA0_STATUSBLOCK_CH6_Pos  	(6U)
#define   GPDMA0_STATUSBLOCK_CH6_Msk  	(0x00000001U  << GPDMA0_STATUSBLOCK_CH6_Pos)

#define   GPDMA0_STATUSBLOCK_CH7_Pos  	(7U)
#define   GPDMA0_STATUSBLOCK_CH7_Msk  	(0x00000001U  << GPDMA0_STATUSBLOCK_CH7_Pos)

/* GPDMA0_STATUSSRCTRAN  =  IntSrcTran Status*/
#define   GPDMA0_STATUSSRCTRAN_CH0_Pos  	(0U)
#define   GPDMA0_STATUSSRCTRAN_CH0_Msk  	(0x00000001U  << GPDMA0_STATUSSRCTRAN_CH0_Pos)

#define   GPDMA0_STATUSSRCTRAN_CH1_Pos  	(1U)
#define   GPDMA0_STATUSSRCTRAN_CH1_Msk  	(0x00000001U  << GPDMA0_STATUSSRCTRAN_CH1_Pos)

#define   GPDMA0_STATUSSRCTRAN_CH2_Pos  	(2U)
#define   GPDMA0_STATUSSRCTRAN_CH2_Msk  	(0x00000001U  << GPDMA0_STATUSSRCTRAN_CH2_Pos)

#define   GPDMA0_STATUSSRCTRAN_CH3_Pos  	(3U)
#define   GPDMA0_STATUSSRCTRAN_CH3_Msk  	(0x00000001U  << GPDMA0_STATUSSRCTRAN_CH3_Pos)

#define   GPDMA0_STATUSSRCTRAN_CH4_Pos  	(4U)
#define   GPDMA0_STATUSSRCTRAN_CH4_Msk  	(0x00000001U  << GPDMA0_STATUSSRCTRAN_CH4_Pos)

#define   GPDMA0_STATUSSRCTRAN_CH5_Pos  	(5U)
#define   GPDMA0_STATUSSRCTRAN_CH5_Msk  	(0x00000001U  << GPDMA0_STATUSSRCTRAN_CH5_Pos)

#define   GPDMA0_STATUSSRCTRAN_CH6_Pos  	(6U)
#define   GPDMA0_STATUSSRCTRAN_CH6_Msk  	(0x00000001U  << GPDMA0_STATUSSRCTRAN_CH6_Pos)

#define   GPDMA0_STATUSSRCTRAN_CH7_Pos  	(7U)
#define   GPDMA0_STATUSSRCTRAN_CH7_Msk  	(0x00000001U  << GPDMA0_STATUSSRCTRAN_CH7_Pos)

/* GPDMA0_STATUSDSTTRAN  =  IntBlock Status*/
#define   GPDMA0_STATUSDSTTRAN_CH0_Pos  	(0U)
#define   GPDMA0_STATUSDSTTRAN_CH0_Msk  	(0x00000001U  << GPDMA0_STATUSDSTTRAN_CH0_Pos)

#define   GPDMA0_STATUSDSTTRAN_CH1_Pos  	(1U)
#define   GPDMA0_STATUSDSTTRAN_CH1_Msk  	(0x00000001U  << GPDMA0_STATUSDSTTRAN_CH1_Pos)

#define   GPDMA0_STATUSDSTTRAN_CH2_Pos  	(2U)
#define   GPDMA0_STATUSDSTTRAN_CH2_Msk  	(0x00000001U  << GPDMA0_STATUSDSTTRAN_CH2_Pos)

#define   GPDMA0_STATUSDSTTRAN_CH3_Pos  	(3U)
#define   GPDMA0_STATUSDSTTRAN_CH3_Msk  	(0x00000001U  << GPDMA0_STATUSDSTTRAN_CH3_Pos)

#define   GPDMA0_STATUSDSTTRAN_CH4_Pos  	(4U)
#define   GPDMA0_STATUSDSTTRAN_CH4_Msk  	(0x00000001U  << GPDMA0_STATUSDSTTRAN_CH4_Pos)

#define   GPDMA0_STATUSDSTTRAN_CH5_Pos  	(5U)
#define   GPDMA0_STATUSDSTTRAN_CH5_Msk  	(0x00000001U  << GPDMA0_STATUSDSTTRAN_CH5_Pos)

#define   GPDMA0_STATUSDSTTRAN_CH6_Pos  	(6U)
#define   GPDMA0_STATUSDSTTRAN_CH6_Msk  	(0x00000001U  << GPDMA0_STATUSDSTTRAN_CH6_Pos)

#define   GPDMA0_STATUSDSTTRAN_CH7_Pos  	(7U)
#define   GPDMA0_STATUSDSTTRAN_CH7_Msk  	(0x00000001U  << GPDMA0_STATUSDSTTRAN_CH7_Pos)

/* GPDMA0_STATUSERR  =  IntErr Status*/
#define   GPDMA0_STATUSERR_CH0_Pos  	(0U)
#define   GPDMA0_STATUSERR_CH0_Msk  	(0x00000001U  << GPDMA0_STATUSERR_CH0_Pos)

#define   GPDMA0_STATUSERR_CH1_Pos  	(1U)
#define   GPDMA0_STATUSERR_CH1_Msk  	(0x00000001U  << GPDMA0_STATUSERR_CH1_Pos)

#define   GPDMA0_STATUSERR_CH2_Pos  	(2U)
#define   GPDMA0_STATUSERR_CH2_Msk  	(0x00000001U  << GPDMA0_STATUSERR_CH2_Pos)

#define   GPDMA0_STATUSERR_CH3_Pos  	(3U)
#define   GPDMA0_STATUSERR_CH3_Msk  	(0x00000001U  << GPDMA0_STATUSERR_CH3_Pos)

#define   GPDMA0_STATUSERR_CH4_Pos  	(4U)
#define   GPDMA0_STATUSERR_CH4_Msk  	(0x00000001U  << GPDMA0_STATUSERR_CH4_Pos)

#define   GPDMA0_STATUSERR_CH5_Pos  	(5U)
#define   GPDMA0_STATUSERR_CH5_Msk  	(0x00000001U  << GPDMA0_STATUSERR_CH5_Pos)

#define   GPDMA0_STATUSERR_CH6_Pos  	(6U)
#define   GPDMA0_STATUSERR_CH6_Msk  	(0x00000001U  << GPDMA0_STATUSERR_CH6_Pos)

#define   GPDMA0_STATUSERR_CH7_Pos  	(7U)
#define   GPDMA0_STATUSERR_CH7_Msk  	(0x00000001U  << GPDMA0_STATUSERR_CH7_Pos)

/* GPDMA0_MASKTFR  =  Mask for Raw IntTfr Status*/
#define   GPDMA0_MASKTFR_WE_CH0_Pos  	(8U)
#define   GPDMA0_MASKTFR_WE_CH0_Msk  	(0x00000001U  << GPDMA0_MASKTFR_WE_CH0_Pos)

#define   GPDMA0_MASKTFR_WE_CH1_Pos  	(9U)
#define   GPDMA0_MASKTFR_WE_CH1_Msk  	(0x00000001U  << GPDMA0_MASKTFR_WE_CH1_Pos)

#define   GPDMA0_MASKTFR_WE_CH2_Pos  	(10U)
#define   GPDMA0_MASKTFR_WE_CH2_Msk  	(0x00000001U  << GPDMA0_MASKTFR_WE_CH2_Pos)

#define   GPDMA0_MASKTFR_WE_CH3_Pos  	(11U)
#define   GPDMA0_MASKTFR_WE_CH3_Msk  	(0x00000001U  << GPDMA0_MASKTFR_WE_CH3_Pos)

#define   GPDMA0_MASKTFR_WE_CH4_Pos  	(12U)
#define   GPDMA0_MASKTFR_WE_CH4_Msk  	(0x00000001U  << GPDMA0_MASKTFR_WE_CH4_Pos)

#define   GPDMA0_MASKTFR_WE_CH5_Pos  	(13U)
#define   GPDMA0_MASKTFR_WE_CH5_Msk  	(0x00000001U  << GPDMA0_MASKTFR_WE_CH5_Pos)

#define   GPDMA0_MASKTFR_WE_CH6_Pos  	(14U)
#define   GPDMA0_MASKTFR_WE_CH6_Msk  	(0x00000001U  << GPDMA0_MASKTFR_WE_CH6_Pos)

#define   GPDMA0_MASKTFR_WE_CH7_Pos  	(15U)
#define   GPDMA0_MASKTFR_WE_CH7_Msk  	(0x00000001U  << GPDMA0_MASKTFR_WE_CH7_Pos)

#define   GPDMA0_MASKTFR_CH0_Pos  	(0U)
#define   GPDMA0_MASKTFR_CH0_Msk  	(0x00000001U  << GPDMA0_MASKTFR_CH0_Pos)

#define   GPDMA0_MASKTFR_CH1_Pos  	(1U)
#define   GPDMA0_MASKTFR_CH1_Msk  	(0x00000001U  << GPDMA0_MASKTFR_CH1_Pos)

#define   GPDMA0_MASKTFR_CH2_Pos  	(2U)
#define   GPDMA0_MASKTFR_CH2_Msk  	(0x00000001U  << GPDMA0_MASKTFR_CH2_Pos)

#define   GPDMA0_MASKTFR_CH3_Pos  	(3U)
#define   GPDMA0_MASKTFR_CH3_Msk  	(0x00000001U  << GPDMA0_MASKTFR_CH3_Pos)

#define   GPDMA0_MASKTFR_CH4_Pos  	(4U)
#define   GPDMA0_MASKTFR_CH4_Msk  	(0x00000001U  << GPDMA0_MASKTFR_CH4_Pos)

#define   GPDMA0_MASKTFR_CH5_Pos  	(5U)
#define   GPDMA0_MASKTFR_CH5_Msk  	(0x00000001U  << GPDMA0_MASKTFR_CH5_Pos)

#define   GPDMA0_MASKTFR_CH6_Pos  	(6U)
#define   GPDMA0_MASKTFR_CH6_Msk  	(0x00000001U  << GPDMA0_MASKTFR_CH6_Pos)

#define   GPDMA0_MASKTFR_CH7_Pos  	(7U)
#define   GPDMA0_MASKTFR_CH7_Msk  	(0x00000001U  << GPDMA0_MASKTFR_CH7_Pos)

/* GPDMA0_MASKBLOCK  =  Mask for Raw IntBlock Status*/
#define   GPDMA0_MASKBLOCK_WE_CH0_Pos  	(8U)
#define   GPDMA0_MASKBLOCK_WE_CH0_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_WE_CH0_Pos)

#define   GPDMA0_MASKBLOCK_WE_CH1_Pos  	(9U)
#define   GPDMA0_MASKBLOCK_WE_CH1_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_WE_CH1_Pos)

#define   GPDMA0_MASKBLOCK_WE_CH2_Pos  	(10U)
#define   GPDMA0_MASKBLOCK_WE_CH2_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_WE_CH2_Pos)

#define   GPDMA0_MASKBLOCK_WE_CH3_Pos  	(11U)
#define   GPDMA0_MASKBLOCK_WE_CH3_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_WE_CH3_Pos)

#define   GPDMA0_MASKBLOCK_WE_CH4_Pos  	(12U)
#define   GPDMA0_MASKBLOCK_WE_CH4_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_WE_CH4_Pos)

#define   GPDMA0_MASKBLOCK_WE_CH5_Pos  	(13U)
#define   GPDMA0_MASKBLOCK_WE_CH5_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_WE_CH5_Pos)

#define   GPDMA0_MASKBLOCK_WE_CH6_Pos  	(14U)
#define   GPDMA0_MASKBLOCK_WE_CH6_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_WE_CH6_Pos)

#define   GPDMA0_MASKBLOCK_WE_CH7_Pos  	(15U)
#define   GPDMA0_MASKBLOCK_WE_CH7_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_WE_CH7_Pos)

#define   GPDMA0_MASKBLOCK_CH0_Pos  	(0U)
#define   GPDMA0_MASKBLOCK_CH0_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_CH0_Pos)

#define   GPDMA0_MASKBLOCK_CH1_Pos  	(1U)
#define   GPDMA0_MASKBLOCK_CH1_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_CH1_Pos)

#define   GPDMA0_MASKBLOCK_CH2_Pos  	(2U)
#define   GPDMA0_MASKBLOCK_CH2_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_CH2_Pos)

#define   GPDMA0_MASKBLOCK_CH3_Pos  	(3U)
#define   GPDMA0_MASKBLOCK_CH3_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_CH3_Pos)

#define   GPDMA0_MASKBLOCK_CH4_Pos  	(4U)
#define   GPDMA0_MASKBLOCK_CH4_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_CH4_Pos)

#define   GPDMA0_MASKBLOCK_CH5_Pos  	(5U)
#define   GPDMA0_MASKBLOCK_CH5_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_CH5_Pos)

#define   GPDMA0_MASKBLOCK_CH6_Pos  	(6U)
#define   GPDMA0_MASKBLOCK_CH6_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_CH6_Pos)

#define   GPDMA0_MASKBLOCK_CH7_Pos  	(7U)
#define   GPDMA0_MASKBLOCK_CH7_Msk  	(0x00000001U  << GPDMA0_MASKBLOCK_CH7_Pos)

/* GPDMA0_MASKSRCTRAN  =  Mask for Raw IntSrcTran Status*/
#define   GPDMA0_MASKSRCTRAN_WE_CH0_Pos  	(8U)
#define   GPDMA0_MASKSRCTRAN_WE_CH0_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_WE_CH0_Pos)

#define   GPDMA0_MASKSRCTRAN_WE_CH1_Pos  	(9U)
#define   GPDMA0_MASKSRCTRAN_WE_CH1_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_WE_CH1_Pos)

#define   GPDMA0_MASKSRCTRAN_WE_CH2_Pos  	(10U)
#define   GPDMA0_MASKSRCTRAN_WE_CH2_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_WE_CH2_Pos)

#define   GPDMA0_MASKSRCTRAN_WE_CH3_Pos  	(11U)
#define   GPDMA0_MASKSRCTRAN_WE_CH3_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_WE_CH3_Pos)

#define   GPDMA0_MASKSRCTRAN_WE_CH4_Pos  	(12U)
#define   GPDMA0_MASKSRCTRAN_WE_CH4_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_WE_CH4_Pos)

#define   GPDMA0_MASKSRCTRAN_WE_CH5_Pos  	(13U)
#define   GPDMA0_MASKSRCTRAN_WE_CH5_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_WE_CH5_Pos)

#define   GPDMA0_MASKSRCTRAN_WE_CH6_Pos  	(14U)
#define   GPDMA0_MASKSRCTRAN_WE_CH6_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_WE_CH6_Pos)

#define   GPDMA0_MASKSRCTRAN_WE_CH7_Pos  	(15U)
#define   GPDMA0_MASKSRCTRAN_WE_CH7_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_WE_CH7_Pos)

#define   GPDMA0_MASKSRCTRAN_CH0_Pos  	(0U)
#define   GPDMA0_MASKSRCTRAN_CH0_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_CH0_Pos)

#define   GPDMA0_MASKSRCTRAN_CH1_Pos  	(1U)
#define   GPDMA0_MASKSRCTRAN_CH1_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_CH1_Pos)

#define   GPDMA0_MASKSRCTRAN_CH2_Pos  	(2U)
#define   GPDMA0_MASKSRCTRAN_CH2_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_CH2_Pos)

#define   GPDMA0_MASKSRCTRAN_CH3_Pos  	(3U)
#define   GPDMA0_MASKSRCTRAN_CH3_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_CH3_Pos)

#define   GPDMA0_MASKSRCTRAN_CH4_Pos  	(4U)
#define   GPDMA0_MASKSRCTRAN_CH4_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_CH4_Pos)

#define   GPDMA0_MASKSRCTRAN_CH5_Pos  	(5U)
#define   GPDMA0_MASKSRCTRAN_CH5_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_CH5_Pos)

#define   GPDMA0_MASKSRCTRAN_CH6_Pos  	(6U)
#define   GPDMA0_MASKSRCTRAN_CH6_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_CH6_Pos)

#define   GPDMA0_MASKSRCTRAN_CH7_Pos  	(7U)
#define   GPDMA0_MASKSRCTRAN_CH7_Msk  	(0x00000001U  << GPDMA0_MASKSRCTRAN_CH7_Pos)

/* GPDMA0_MASKDSTTRAN  =  Mask for Raw IntBlock Status*/
#define   GPDMA0_MASKDSTTRAN_WE_CH0_Pos  	(8U)
#define   GPDMA0_MASKDSTTRAN_WE_CH0_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_WE_CH0_Pos)

#define   GPDMA0_MASKDSTTRAN_WE_CH1_Pos  	(9U)
#define   GPDMA0_MASKDSTTRAN_WE_CH1_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_WE_CH1_Pos)

#define   GPDMA0_MASKDSTTRAN_WE_CH2_Pos  	(10U)
#define   GPDMA0_MASKDSTTRAN_WE_CH2_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_WE_CH2_Pos)

#define   GPDMA0_MASKDSTTRAN_WE_CH3_Pos  	(11U)
#define   GPDMA0_MASKDSTTRAN_WE_CH3_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_WE_CH3_Pos)

#define   GPDMA0_MASKDSTTRAN_WE_CH4_Pos  	(12U)
#define   GPDMA0_MASKDSTTRAN_WE_CH4_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_WE_CH4_Pos)

#define   GPDMA0_MASKDSTTRAN_WE_CH5_Pos  	(13U)
#define   GPDMA0_MASKDSTTRAN_WE_CH5_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_WE_CH5_Pos)

#define   GPDMA0_MASKDSTTRAN_WE_CH6_Pos  	(14U)
#define   GPDMA0_MASKDSTTRAN_WE_CH6_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_WE_CH6_Pos)

#define   GPDMA0_MASKDSTTRAN_WE_CH7_Pos  	(15U)
#define   GPDMA0_MASKDSTTRAN_WE_CH7_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_WE_CH7_Pos)

#define   GPDMA0_MASKDSTTRAN_CH0_Pos  	(0U)
#define   GPDMA0_MASKDSTTRAN_CH0_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_CH0_Pos)

#define   GPDMA0_MASKDSTTRAN_CH1_Pos  	(1U)
#define   GPDMA0_MASKDSTTRAN_CH1_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_CH1_Pos)

#define   GPDMA0_MASKDSTTRAN_CH2_Pos  	(2U)
#define   GPDMA0_MASKDSTTRAN_CH2_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_CH2_Pos)

#define   GPDMA0_MASKDSTTRAN_CH3_Pos  	(3U)
#define   GPDMA0_MASKDSTTRAN_CH3_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_CH3_Pos)

#define   GPDMA0_MASKDSTTRAN_CH4_Pos  	(4U)
#define   GPDMA0_MASKDSTTRAN_CH4_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_CH4_Pos)

#define   GPDMA0_MASKDSTTRAN_CH5_Pos  	(5U)
#define   GPDMA0_MASKDSTTRAN_CH5_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_CH5_Pos)

#define   GPDMA0_MASKDSTTRAN_CH6_Pos  	(6U)
#define   GPDMA0_MASKDSTTRAN_CH6_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_CH6_Pos)

#define   GPDMA0_MASKDSTTRAN_CH7_Pos  	(7U)
#define   GPDMA0_MASKDSTTRAN_CH7_Msk  	(0x00000001U  << GPDMA0_MASKDSTTRAN_CH7_Pos)

/* GPDMA0_MASKERR  =  Mask for Raw IntErr Status*/
#define   GPDMA0_MASKERR_WE_CH0_Pos  	(8U)
#define   GPDMA0_MASKERR_WE_CH0_Msk  	(0x00000001U  << GPDMA0_MASKERR_WE_CH0_Pos)

#define   GPDMA0_MASKERR_WE_CH1_Pos  	(9U)
#define   GPDMA0_MASKERR_WE_CH1_Msk  	(0x00000001U  << GPDMA0_MASKERR_WE_CH1_Pos)

#define   GPDMA0_MASKERR_WE_CH2_Pos  	(10U)
#define   GPDMA0_MASKERR_WE_CH2_Msk  	(0x00000001U  << GPDMA0_MASKERR_WE_CH2_Pos)

#define   GPDMA0_MASKERR_WE_CH3_Pos  	(11U)
#define   GPDMA0_MASKERR_WE_CH3_Msk  	(0x00000001U  << GPDMA0_MASKERR_WE_CH3_Pos)

#define   GPDMA0_MASKERR_WE_CH4_Pos  	(12U)
#define   GPDMA0_MASKERR_WE_CH4_Msk  	(0x00000001U  << GPDMA0_MASKERR_WE_CH4_Pos)

#define   GPDMA0_MASKERR_WE_CH5_Pos  	(13U)
#define   GPDMA0_MASKERR_WE_CH5_Msk  	(0x00000001U  << GPDMA0_MASKERR_WE_CH5_Pos)

#define   GPDMA0_MASKERR_WE_CH6_Pos  	(14U)
#define   GPDMA0_MASKERR_WE_CH6_Msk  	(0x00000001U  << GPDMA0_MASKERR_WE_CH6_Pos)

#define   GPDMA0_MASKERR_WE_CH7_Pos  	(15U)
#define   GPDMA0_MASKERR_WE_CH7_Msk  	(0x00000001U  << GPDMA0_MASKERR_WE_CH7_Pos)

#define   GPDMA0_MASKERR_CH0_Pos  	(0U)
#define   GPDMA0_MASKERR_CH0_Msk  	(0x00000001U  << GPDMA0_MASKERR_CH0_Pos)

#define   GPDMA0_MASKERR_CH1_Pos  	(1U)
#define   GPDMA0_MASKERR_CH1_Msk  	(0x00000001U  << GPDMA0_MASKERR_CH1_Pos)

#define   GPDMA0_MASKERR_CH2_Pos  	(2U)
#define   GPDMA0_MASKERR_CH2_Msk  	(0x00000001U  << GPDMA0_MASKERR_CH2_Pos)

#define   GPDMA0_MASKERR_CH3_Pos  	(3U)
#define   GPDMA0_MASKERR_CH3_Msk  	(0x00000001U  << GPDMA0_MASKERR_CH3_Pos)

#define   GPDMA0_MASKERR_CH4_Pos  	(4U)
#define   GPDMA0_MASKERR_CH4_Msk  	(0x00000001U  << GPDMA0_MASKERR_CH4_Pos)

#define   GPDMA0_MASKERR_CH5_Pos  	(5U)
#define   GPDMA0_MASKERR_CH5_Msk  	(0x00000001U  << GPDMA0_MASKERR_CH5_Pos)

#define   GPDMA0_MASKERR_CH6_Pos  	(6U)
#define   GPDMA0_MASKERR_CH6_Msk  	(0x00000001U  << GPDMA0_MASKERR_CH6_Pos)

#define   GPDMA0_MASKERR_CH7_Pos  	(7U)
#define   GPDMA0_MASKERR_CH7_Msk  	(0x00000001U  << GPDMA0_MASKERR_CH7_Pos)

/* GPDMA0_CLEARTFR  =  IntTfr Status*/
#define   GPDMA0_CLEARTFR_CH0_Pos  	(0U)
#define   GPDMA0_CLEARTFR_CH0_Msk  	(0x00000001U  << GPDMA0_CLEARTFR_CH0_Pos)

#define   GPDMA0_CLEARTFR_CH1_Pos  	(1U)
#define   GPDMA0_CLEARTFR_CH1_Msk  	(0x00000001U  << GPDMA0_CLEARTFR_CH1_Pos)

#define   GPDMA0_CLEARTFR_CH2_Pos  	(2U)
#define   GPDMA0_CLEARTFR_CH2_Msk  	(0x00000001U  << GPDMA0_CLEARTFR_CH2_Pos)

#define   GPDMA0_CLEARTFR_CH3_Pos  	(3U)
#define   GPDMA0_CLEARTFR_CH3_Msk  	(0x00000001U  << GPDMA0_CLEARTFR_CH3_Pos)

#define   GPDMA0_CLEARTFR_CH4_Pos  	(4U)
#define   GPDMA0_CLEARTFR_CH4_Msk  	(0x00000001U  << GPDMA0_CLEARTFR_CH4_Pos)

#define   GPDMA0_CLEARTFR_CH5_Pos  	(5U)
#define   GPDMA0_CLEARTFR_CH5_Msk  	(0x00000001U  << GPDMA0_CLEARTFR_CH5_Pos)

#define   GPDMA0_CLEARTFR_CH6_Pos  	(6U)
#define   GPDMA0_CLEARTFR_CH6_Msk  	(0x00000001U  << GPDMA0_CLEARTFR_CH6_Pos)

#define   GPDMA0_CLEARTFR_CH7_Pos  	(7U)
#define   GPDMA0_CLEARTFR_CH7_Msk  	(0x00000001U  << GPDMA0_CLEARTFR_CH7_Pos)

/* GPDMA0_CLEARBLOCK  =  IntBlock Status*/
#define   GPDMA0_CLEARBLOCK_CH0_Pos  	(0U)
#define   GPDMA0_CLEARBLOCK_CH0_Msk  	(0x00000001U  << GPDMA0_CLEARBLOCK_CH0_Pos)

#define   GPDMA0_CLEARBLOCK_CH1_Pos  	(1U)
#define   GPDMA0_CLEARBLOCK_CH1_Msk  	(0x00000001U  << GPDMA0_CLEARBLOCK_CH1_Pos)

#define   GPDMA0_CLEARBLOCK_CH2_Pos  	(2U)
#define   GPDMA0_CLEARBLOCK_CH2_Msk  	(0x00000001U  << GPDMA0_CLEARBLOCK_CH2_Pos)

#define   GPDMA0_CLEARBLOCK_CH3_Pos  	(3U)
#define   GPDMA0_CLEARBLOCK_CH3_Msk  	(0x00000001U  << GPDMA0_CLEARBLOCK_CH3_Pos)

#define   GPDMA0_CLEARBLOCK_CH4_Pos  	(4U)
#define   GPDMA0_CLEARBLOCK_CH4_Msk  	(0x00000001U  << GPDMA0_CLEARBLOCK_CH4_Pos)

#define   GPDMA0_CLEARBLOCK_CH5_Pos  	(5U)
#define   GPDMA0_CLEARBLOCK_CH5_Msk  	(0x00000001U  << GPDMA0_CLEARBLOCK_CH5_Pos)

#define   GPDMA0_CLEARBLOCK_CH6_Pos  	(6U)
#define   GPDMA0_CLEARBLOCK_CH6_Msk  	(0x00000001U  << GPDMA0_CLEARBLOCK_CH6_Pos)

#define   GPDMA0_CLEARBLOCK_CH7_Pos  	(7U)
#define   GPDMA0_CLEARBLOCK_CH7_Msk  	(0x00000001U  << GPDMA0_CLEARBLOCK_CH7_Pos)

/* GPDMA0_CLEARSRCTRAN  =  IntSrcTran Status*/
#define   GPDMA0_CLEARSRCTRAN_CH0_Pos  	(0U)
#define   GPDMA0_CLEARSRCTRAN_CH0_Msk  	(0x00000001U  << GPDMA0_CLEARSRCTRAN_CH0_Pos)

#define   GPDMA0_CLEARSRCTRAN_CH1_Pos  	(1U)
#define   GPDMA0_CLEARSRCTRAN_CH1_Msk  	(0x00000001U  << GPDMA0_CLEARSRCTRAN_CH1_Pos)

#define   GPDMA0_CLEARSRCTRAN_CH2_Pos  	(2U)
#define   GPDMA0_CLEARSRCTRAN_CH2_Msk  	(0x00000001U  << GPDMA0_CLEARSRCTRAN_CH2_Pos)

#define   GPDMA0_CLEARSRCTRAN_CH3_Pos  	(3U)
#define   GPDMA0_CLEARSRCTRAN_CH3_Msk  	(0x00000001U  << GPDMA0_CLEARSRCTRAN_CH3_Pos)

#define   GPDMA0_CLEARSRCTRAN_CH4_Pos  	(4U)
#define   GPDMA0_CLEARSRCTRAN_CH4_Msk  	(0x00000001U  << GPDMA0_CLEARSRCTRAN_CH4_Pos)

#define   GPDMA0_CLEARSRCTRAN_CH5_Pos  	(5U)
#define   GPDMA0_CLEARSRCTRAN_CH5_Msk  	(0x00000001U  << GPDMA0_CLEARSRCTRAN_CH5_Pos)

#define   GPDMA0_CLEARSRCTRAN_CH6_Pos  	(6U)
#define   GPDMA0_CLEARSRCTRAN_CH6_Msk  	(0x00000001U  << GPDMA0_CLEARSRCTRAN_CH6_Pos)

#define   GPDMA0_CLEARSRCTRAN_CH7_Pos  	(7U)
#define   GPDMA0_CLEARSRCTRAN_CH7_Msk  	(0x00000001U  << GPDMA0_CLEARSRCTRAN_CH7_Pos)

/* GPDMA0_CLEARDSTTRAN  =  IntBlock Status*/
#define   GPDMA0_CLEARDSTTRAN_CH0_Pos  	(0U)
#define   GPDMA0_CLEARDSTTRAN_CH0_Msk  	(0x00000001U  << GPDMA0_CLEARDSTTRAN_CH0_Pos)

#define   GPDMA0_CLEARDSTTRAN_CH1_Pos  	(1U)
#define   GPDMA0_CLEARDSTTRAN_CH1_Msk  	(0x00000001U  << GPDMA0_CLEARDSTTRAN_CH1_Pos)

#define   GPDMA0_CLEARDSTTRAN_CH2_Pos  	(2U)
#define   GPDMA0_CLEARDSTTRAN_CH2_Msk  	(0x00000001U  << GPDMA0_CLEARDSTTRAN_CH2_Pos)

#define   GPDMA0_CLEARDSTTRAN_CH3_Pos  	(3U)
#define   GPDMA0_CLEARDSTTRAN_CH3_Msk  	(0x00000001U  << GPDMA0_CLEARDSTTRAN_CH3_Pos)

#define   GPDMA0_CLEARDSTTRAN_CH4_Pos  	(4U)
#define   GPDMA0_CLEARDSTTRAN_CH4_Msk  	(0x00000001U  << GPDMA0_CLEARDSTTRAN_CH4_Pos)

#define   GPDMA0_CLEARDSTTRAN_CH5_Pos  	(5U)
#define   GPDMA0_CLEARDSTTRAN_CH5_Msk  	(0x00000001U  << GPDMA0_CLEARDSTTRAN_CH5_Pos)

#define   GPDMA0_CLEARDSTTRAN_CH6_Pos  	(6U)
#define   GPDMA0_CLEARDSTTRAN_CH6_Msk  	(0x00000001U  << GPDMA0_CLEARDSTTRAN_CH6_Pos)

#define   GPDMA0_CLEARDSTTRAN_CH7_Pos  	(7U)
#define   GPDMA0_CLEARDSTTRAN_CH7_Msk  	(0x00000001U  << GPDMA0_CLEARDSTTRAN_CH7_Pos)

/* GPDMA0_CLEARERR  =  IntErr Status*/
#define   GPDMA0_CLEARERR_CH0_Pos  	(0U)
#define   GPDMA0_CLEARERR_CH0_Msk  	(0x00000001U  << GPDMA0_CLEARERR_CH0_Pos)

#define   GPDMA0_CLEARERR_CH1_Pos  	(1U)
#define   GPDMA0_CLEARERR_CH1_Msk  	(0x00000001U  << GPDMA0_CLEARERR_CH1_Pos)

#define   GPDMA0_CLEARERR_CH2_Pos  	(2U)
#define   GPDMA0_CLEARERR_CH2_Msk  	(0x00000001U  << GPDMA0_CLEARERR_CH2_Pos)

#define   GPDMA0_CLEARERR_CH3_Pos  	(3U)
#define   GPDMA0_CLEARERR_CH3_Msk  	(0x00000001U  << GPDMA0_CLEARERR_CH3_Pos)

#define   GPDMA0_CLEARERR_CH4_Pos  	(4U)
#define   GPDMA0_CLEARERR_CH4_Msk  	(0x00000001U  << GPDMA0_CLEARERR_CH4_Pos)

#define   GPDMA0_CLEARERR_CH5_Pos  	(5U)
#define   GPDMA0_CLEARERR_CH5_Msk  	(0x00000001U  << GPDMA0_CLEARERR_CH5_Pos)

#define   GPDMA0_CLEARERR_CH6_Pos  	(6U)
#define   GPDMA0_CLEARERR_CH6_Msk  	(0x00000001U  << GPDMA0_CLEARERR_CH6_Pos)

#define   GPDMA0_CLEARERR_CH7_Pos  	(7U)
#define   GPDMA0_CLEARERR_CH7_Msk  	(0x00000001U  << GPDMA0_CLEARERR_CH7_Pos)

/* GPDMA0_STATUSINT  =  Combined Interrupt Status Register*/
#define   GPDMA0_STATUSINT_ERR_Pos  	(4U)
#define   GPDMA0_STATUSINT_ERR_Msk  	(0x00000001U  << GPDMA0_STATUSINT_ERR_Pos)

#define   GPDMA0_STATUSINT_DSTT_Pos  	(3U)
#define   GPDMA0_STATUSINT_DSTT_Msk  	(0x00000001U  << GPDMA0_STATUSINT_DSTT_Pos)

#define   GPDMA0_STATUSINT_SRCT_Pos  	(2U)
#define   GPDMA0_STATUSINT_SRCT_Msk  	(0x00000001U  << GPDMA0_STATUSINT_SRCT_Pos)

#define   GPDMA0_STATUSINT_BLOCK_Pos  	(1U)
#define   GPDMA0_STATUSINT_BLOCK_Msk  	(0x00000001U  << GPDMA0_STATUSINT_BLOCK_Pos)

#define   GPDMA0_STATUSINT_TFR_Pos  	(0U)
#define   GPDMA0_STATUSINT_TFR_Msk  	(0x00000001U  << GPDMA0_STATUSINT_TFR_Pos)

/* GPDMA0_REQSRCREG  =  Source Software Transaction Request Register*/
#define   GPDMA0_REQSRCREG_WE_CH0_Pos  	(8U)
#define   GPDMA0_REQSRCREG_WE_CH0_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_WE_CH0_Pos)

#define   GPDMA0_REQSRCREG_WE_CH1_Pos  	(9U)
#define   GPDMA0_REQSRCREG_WE_CH1_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_WE_CH1_Pos)

#define   GPDMA0_REQSRCREG_WE_CH2_Pos  	(10U)
#define   GPDMA0_REQSRCREG_WE_CH2_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_WE_CH2_Pos)

#define   GPDMA0_REQSRCREG_WE_CH3_Pos  	(11U)
#define   GPDMA0_REQSRCREG_WE_CH3_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_WE_CH3_Pos)

#define   GPDMA0_REQSRCREG_WE_CH4_Pos  	(12U)
#define   GPDMA0_REQSRCREG_WE_CH4_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_WE_CH4_Pos)

#define   GPDMA0_REQSRCREG_WE_CH5_Pos  	(13U)
#define   GPDMA0_REQSRCREG_WE_CH5_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_WE_CH5_Pos)

#define   GPDMA0_REQSRCREG_WE_CH6_Pos  	(14U)
#define   GPDMA0_REQSRCREG_WE_CH6_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_WE_CH6_Pos)

#define   GPDMA0_REQSRCREG_WE_CH7_Pos  	(15U)
#define   GPDMA0_REQSRCREG_WE_CH7_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_WE_CH7_Pos)

#define   GPDMA0_REQSRCREG_CH0_Pos  	(0U)
#define   GPDMA0_REQSRCREG_CH0_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_CH0_Pos)

#define   GPDMA0_REQSRCREG_CH1_Pos  	(1U)
#define   GPDMA0_REQSRCREG_CH1_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_CH1_Pos)

#define   GPDMA0_REQSRCREG_CH2_Pos  	(2U)
#define   GPDMA0_REQSRCREG_CH2_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_CH2_Pos)

#define   GPDMA0_REQSRCREG_CH3_Pos  	(3U)
#define   GPDMA0_REQSRCREG_CH3_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_CH3_Pos)

#define   GPDMA0_REQSRCREG_CH4_Pos  	(4U)
#define   GPDMA0_REQSRCREG_CH4_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_CH4_Pos)

#define   GPDMA0_REQSRCREG_CH5_Pos  	(5U)
#define   GPDMA0_REQSRCREG_CH5_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_CH5_Pos)

#define   GPDMA0_REQSRCREG_CH6_Pos  	(6U)
#define   GPDMA0_REQSRCREG_CH6_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_CH6_Pos)

#define   GPDMA0_REQSRCREG_CH7_Pos  	(7U)
#define   GPDMA0_REQSRCREG_CH7_Msk  	(0x00000001U  << GPDMA0_REQSRCREG_CH7_Pos)

/* GPDMA0_REQDSTREG  =  Destination Software Transaction Request Register*/
#define   GPDMA0_REQDSTREG_WE_CH0_Pos  	(8U)
#define   GPDMA0_REQDSTREG_WE_CH0_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_WE_CH0_Pos)

#define   GPDMA0_REQDSTREG_WE_CH1_Pos  	(9U)
#define   GPDMA0_REQDSTREG_WE_CH1_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_WE_CH1_Pos)

#define   GPDMA0_REQDSTREG_WE_CH2_Pos  	(10U)
#define   GPDMA0_REQDSTREG_WE_CH2_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_WE_CH2_Pos)

#define   GPDMA0_REQDSTREG_WE_CH3_Pos  	(11U)
#define   GPDMA0_REQDSTREG_WE_CH3_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_WE_CH3_Pos)

#define   GPDMA0_REQDSTREG_WE_CH4_Pos  	(12U)
#define   GPDMA0_REQDSTREG_WE_CH4_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_WE_CH4_Pos)

#define   GPDMA0_REQDSTREG_WE_CH5_Pos  	(13U)
#define   GPDMA0_REQDSTREG_WE_CH5_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_WE_CH5_Pos)

#define   GPDMA0_REQDSTREG_WE_CH6_Pos  	(14U)
#define   GPDMA0_REQDSTREG_WE_CH6_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_WE_CH6_Pos)

#define   GPDMA0_REQDSTREG_WE_CH7_Pos  	(15U)
#define   GPDMA0_REQDSTREG_WE_CH7_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_WE_CH7_Pos)

#define   GPDMA0_REQDSTREG_CH0_Pos  	(0U)
#define   GPDMA0_REQDSTREG_CH0_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_CH0_Pos)

#define   GPDMA0_REQDSTREG_CH1_Pos  	(1U)
#define   GPDMA0_REQDSTREG_CH1_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_CH1_Pos)

#define   GPDMA0_REQDSTREG_CH2_Pos  	(2U)
#define   GPDMA0_REQDSTREG_CH2_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_CH2_Pos)

#define   GPDMA0_REQDSTREG_CH3_Pos  	(3U)
#define   GPDMA0_REQDSTREG_CH3_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_CH3_Pos)

#define   GPDMA0_REQDSTREG_CH4_Pos  	(4U)
#define   GPDMA0_REQDSTREG_CH4_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_CH4_Pos)

#define   GPDMA0_REQDSTREG_CH5_Pos  	(5U)
#define   GPDMA0_REQDSTREG_CH5_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_CH5_Pos)

#define   GPDMA0_REQDSTREG_CH6_Pos  	(6U)
#define   GPDMA0_REQDSTREG_CH6_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_CH6_Pos)

#define   GPDMA0_REQDSTREG_CH7_Pos  	(7U)
#define   GPDMA0_REQDSTREG_CH7_Msk  	(0x00000001U  << GPDMA0_REQDSTREG_CH7_Pos)

/* GPDMA0_SGLREQSRCREG  =  Single Source Transaction Request Register*/
#define   GPDMA0_SGLREQSRCREG_WE_CH0_Pos  	(8U)
#define   GPDMA0_SGLREQSRCREG_WE_CH0_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_WE_CH0_Pos)

#define   GPDMA0_SGLREQSRCREG_WE_CH1_Pos  	(9U)
#define   GPDMA0_SGLREQSRCREG_WE_CH1_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_WE_CH1_Pos)

#define   GPDMA0_SGLREQSRCREG_WE_CH2_Pos  	(10U)
#define   GPDMA0_SGLREQSRCREG_WE_CH2_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_WE_CH2_Pos)

#define   GPDMA0_SGLREQSRCREG_WE_CH3_Pos  	(11U)
#define   GPDMA0_SGLREQSRCREG_WE_CH3_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_WE_CH3_Pos)

#define   GPDMA0_SGLREQSRCREG_WE_CH4_Pos  	(12U)
#define   GPDMA0_SGLREQSRCREG_WE_CH4_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_WE_CH4_Pos)

#define   GPDMA0_SGLREQSRCREG_WE_CH5_Pos  	(13U)
#define   GPDMA0_SGLREQSRCREG_WE_CH5_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_WE_CH5_Pos)

#define   GPDMA0_SGLREQSRCREG_WE_CH6_Pos  	(14U)
#define   GPDMA0_SGLREQSRCREG_WE_CH6_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_WE_CH6_Pos)

#define   GPDMA0_SGLREQSRCREG_WE_CH7_Pos  	(15U)
#define   GPDMA0_SGLREQSRCREG_WE_CH7_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_WE_CH7_Pos)

#define   GPDMA0_SGLREQSRCREG_CH0_Pos  	(0U)
#define   GPDMA0_SGLREQSRCREG_CH0_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_CH0_Pos)

#define   GPDMA0_SGLREQSRCREG_CH1_Pos  	(1U)
#define   GPDMA0_SGLREQSRCREG_CH1_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_CH1_Pos)

#define   GPDMA0_SGLREQSRCREG_CH2_Pos  	(2U)
#define   GPDMA0_SGLREQSRCREG_CH2_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_CH2_Pos)

#define   GPDMA0_SGLREQSRCREG_CH3_Pos  	(3U)
#define   GPDMA0_SGLREQSRCREG_CH3_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_CH3_Pos)

#define   GPDMA0_SGLREQSRCREG_CH4_Pos  	(4U)
#define   GPDMA0_SGLREQSRCREG_CH4_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_CH4_Pos)

#define   GPDMA0_SGLREQSRCREG_CH5_Pos  	(5U)
#define   GPDMA0_SGLREQSRCREG_CH5_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_CH5_Pos)

#define   GPDMA0_SGLREQSRCREG_CH6_Pos  	(6U)
#define   GPDMA0_SGLREQSRCREG_CH6_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_CH6_Pos)

#define   GPDMA0_SGLREQSRCREG_CH7_Pos  	(7U)
#define   GPDMA0_SGLREQSRCREG_CH7_Msk  	(0x00000001U  << GPDMA0_SGLREQSRCREG_CH7_Pos)

/* GPDMA0_SGLREQDSTREG  =  Single Destination Transaction Request Register*/
#define   GPDMA0_SGLREQDSTREG_WE_CH0_Pos  	(8U)
#define   GPDMA0_SGLREQDSTREG_WE_CH0_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_WE_CH0_Pos)

#define   GPDMA0_SGLREQDSTREG_WE_CH1_Pos  	(9U)
#define   GPDMA0_SGLREQDSTREG_WE_CH1_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_WE_CH1_Pos)

#define   GPDMA0_SGLREQDSTREG_WE_CH2_Pos  	(10U)
#define   GPDMA0_SGLREQDSTREG_WE_CH2_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_WE_CH2_Pos)

#define   GPDMA0_SGLREQDSTREG_WE_CH3_Pos  	(11U)
#define   GPDMA0_SGLREQDSTREG_WE_CH3_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_WE_CH3_Pos)

#define   GPDMA0_SGLREQDSTREG_WE_CH4_Pos  	(12U)
#define   GPDMA0_SGLREQDSTREG_WE_CH4_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_WE_CH4_Pos)

#define   GPDMA0_SGLREQDSTREG_WE_CH5_Pos  	(13U)
#define   GPDMA0_SGLREQDSTREG_WE_CH5_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_WE_CH5_Pos)

#define   GPDMA0_SGLREQDSTREG_WE_CH6_Pos  	(14U)
#define   GPDMA0_SGLREQDSTREG_WE_CH6_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_WE_CH6_Pos)

#define   GPDMA0_SGLREQDSTREG_WE_CH7_Pos  	(15U)
#define   GPDMA0_SGLREQDSTREG_WE_CH7_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_WE_CH7_Pos)

#define   GPDMA0_SGLREQDSTREG_CH0_Pos  	(0U)
#define   GPDMA0_SGLREQDSTREG_CH0_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_CH0_Pos)

#define   GPDMA0_SGLREQDSTREG_CH1_Pos  	(1U)
#define   GPDMA0_SGLREQDSTREG_CH1_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_CH1_Pos)

#define   GPDMA0_SGLREQDSTREG_CH2_Pos  	(2U)
#define   GPDMA0_SGLREQDSTREG_CH2_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_CH2_Pos)

#define   GPDMA0_SGLREQDSTREG_CH3_Pos  	(3U)
#define   GPDMA0_SGLREQDSTREG_CH3_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_CH3_Pos)

#define   GPDMA0_SGLREQDSTREG_CH4_Pos  	(4U)
#define   GPDMA0_SGLREQDSTREG_CH4_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_CH4_Pos)

#define   GPDMA0_SGLREQDSTREG_CH5_Pos  	(5U)
#define   GPDMA0_SGLREQDSTREG_CH5_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_CH5_Pos)

#define   GPDMA0_SGLREQDSTREG_CH6_Pos  	(6U)
#define   GPDMA0_SGLREQDSTREG_CH6_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_CH6_Pos)

#define   GPDMA0_SGLREQDSTREG_CH7_Pos  	(7U)
#define   GPDMA0_SGLREQDSTREG_CH7_Msk  	(0x00000001U  << GPDMA0_SGLREQDSTREG_CH7_Pos)

/* GPDMA0_LSTSRCREG  =  Last Source Transaction Request Register*/
#define   GPDMA0_LSTSRCREG_WE_CH0_Pos  	(8U)
#define   GPDMA0_LSTSRCREG_WE_CH0_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_WE_CH0_Pos)

#define   GPDMA0_LSTSRCREG_WE_CH1_Pos  	(9U)
#define   GPDMA0_LSTSRCREG_WE_CH1_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_WE_CH1_Pos)

#define   GPDMA0_LSTSRCREG_WE_CH2_Pos  	(10U)
#define   GPDMA0_LSTSRCREG_WE_CH2_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_WE_CH2_Pos)

#define   GPDMA0_LSTSRCREG_WE_CH3_Pos  	(11U)
#define   GPDMA0_LSTSRCREG_WE_CH3_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_WE_CH3_Pos)

#define   GPDMA0_LSTSRCREG_WE_CH4_Pos  	(12U)
#define   GPDMA0_LSTSRCREG_WE_CH4_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_WE_CH4_Pos)

#define   GPDMA0_LSTSRCREG_WE_CH5_Pos  	(13U)
#define   GPDMA0_LSTSRCREG_WE_CH5_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_WE_CH5_Pos)

#define   GPDMA0_LSTSRCREG_WE_CH6_Pos  	(14U)
#define   GPDMA0_LSTSRCREG_WE_CH6_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_WE_CH6_Pos)

#define   GPDMA0_LSTSRCREG_WE_CH7_Pos  	(15U)
#define   GPDMA0_LSTSRCREG_WE_CH7_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_WE_CH7_Pos)

#define   GPDMA0_LSTSRCREG_CH0_Pos  	(0U)
#define   GPDMA0_LSTSRCREG_CH0_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_CH0_Pos)

#define   GPDMA0_LSTSRCREG_CH1_Pos  	(1U)
#define   GPDMA0_LSTSRCREG_CH1_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_CH1_Pos)

#define   GPDMA0_LSTSRCREG_CH2_Pos  	(2U)
#define   GPDMA0_LSTSRCREG_CH2_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_CH2_Pos)

#define   GPDMA0_LSTSRCREG_CH3_Pos  	(3U)
#define   GPDMA0_LSTSRCREG_CH3_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_CH3_Pos)

#define   GPDMA0_LSTSRCREG_CH4_Pos  	(4U)
#define   GPDMA0_LSTSRCREG_CH4_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_CH4_Pos)

#define   GPDMA0_LSTSRCREG_CH5_Pos  	(5U)
#define   GPDMA0_LSTSRCREG_CH5_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_CH5_Pos)

#define   GPDMA0_LSTSRCREG_CH6_Pos  	(6U)
#define   GPDMA0_LSTSRCREG_CH6_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_CH6_Pos)

#define   GPDMA0_LSTSRCREG_CH7_Pos  	(7U)
#define   GPDMA0_LSTSRCREG_CH7_Msk  	(0x00000001U  << GPDMA0_LSTSRCREG_CH7_Pos)

/* GPDMA0_LSTDSTREG  =  Last Destination Transaction Request Register*/
#define   GPDMA0_LSTDSTREG_WE_CH0_Pos  	(8U)
#define   GPDMA0_LSTDSTREG_WE_CH0_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_WE_CH0_Pos)

#define   GPDMA0_LSTDSTREG_WE_CH1_Pos  	(9U)
#define   GPDMA0_LSTDSTREG_WE_CH1_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_WE_CH1_Pos)

#define   GPDMA0_LSTDSTREG_WE_CH2_Pos  	(10U)
#define   GPDMA0_LSTDSTREG_WE_CH2_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_WE_CH2_Pos)

#define   GPDMA0_LSTDSTREG_WE_CH3_Pos  	(11U)
#define   GPDMA0_LSTDSTREG_WE_CH3_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_WE_CH3_Pos)

#define   GPDMA0_LSTDSTREG_WE_CH4_Pos  	(12U)
#define   GPDMA0_LSTDSTREG_WE_CH4_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_WE_CH4_Pos)

#define   GPDMA0_LSTDSTREG_WE_CH5_Pos  	(13U)
#define   GPDMA0_LSTDSTREG_WE_CH5_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_WE_CH5_Pos)

#define   GPDMA0_LSTDSTREG_WE_CH6_Pos  	(14U)
#define   GPDMA0_LSTDSTREG_WE_CH6_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_WE_CH6_Pos)

#define   GPDMA0_LSTDSTREG_WE_CH7_Pos  	(15U)
#define   GPDMA0_LSTDSTREG_WE_CH7_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_WE_CH7_Pos)

#define   GPDMA0_LSTDSTREG_CH0_Pos  	(0U)
#define   GPDMA0_LSTDSTREG_CH0_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_CH0_Pos)

#define   GPDMA0_LSTDSTREG_CH1_Pos  	(1U)
#define   GPDMA0_LSTDSTREG_CH1_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_CH1_Pos)

#define   GPDMA0_LSTDSTREG_CH2_Pos  	(2U)
#define   GPDMA0_LSTDSTREG_CH2_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_CH2_Pos)

#define   GPDMA0_LSTDSTREG_CH3_Pos  	(3U)
#define   GPDMA0_LSTDSTREG_CH3_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_CH3_Pos)

#define   GPDMA0_LSTDSTREG_CH4_Pos  	(4U)
#define   GPDMA0_LSTDSTREG_CH4_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_CH4_Pos)

#define   GPDMA0_LSTDSTREG_CH5_Pos  	(5U)
#define   GPDMA0_LSTDSTREG_CH5_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_CH5_Pos)

#define   GPDMA0_LSTDSTREG_CH6_Pos  	(6U)
#define   GPDMA0_LSTDSTREG_CH6_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_CH6_Pos)

#define   GPDMA0_LSTDSTREG_CH7_Pos  	(7U)
#define   GPDMA0_LSTDSTREG_CH7_Msk  	(0x00000001U  << GPDMA0_LSTDSTREG_CH7_Pos)

/* GPDMA0_DMACFGREG  =  GPDMA Configuration Register*/
#define   GPDMA0_DMACFGREG_DMA_EN_Pos  	(0U)
#define   GPDMA0_DMACFGREG_DMA_EN_Msk  	(0x00000001U  << GPDMA0_DMACFGREG_DMA_EN_Pos)

/* GPDMA0_CHENREG  =  GPDMA Channel Enable Register*/
#define   GPDMA0_CHENREG_WE_CH_Pos  	(8U)
#define   GPDMA0_CHENREG_WE_CH_Msk  	(0x000000FFU  << GPDMA0_CHENREG_WE_CH_Pos)

#define   GPDMA0_CHENREG_CH_Pos  	(0U)
#define   GPDMA0_CHENREG_CH_Msk  	(0x000000FFU  << GPDMA0_CHENREG_CH_Pos)

/* GPDMA0_ID  =  GPDMA0 ID Register*/
#define   GPDMA0_ID_VALUE_Pos  	(0U)
#define   GPDMA0_ID_VALUE_Msk  	(0xFFFFFFFFU  << GPDMA0_ID_VALUE_Pos)

/* GPDMA0_TYPE  =  GPDMA Component Type*/
#define   GPDMA0_TYPE_VALUE_Pos  	(0U)
#define   GPDMA0_TYPE_VALUE_Msk  	(0xFFFFFFFFU  << GPDMA0_TYPE_VALUE_Pos)

/* GPDMA0_VERSION  =  DMA Component Version*/
#define   GPDMA0_VERSION_VALUE_Pos  	(0U)
#define   GPDMA0_VERSION_VALUE_Msk  	(0xFFFFFFFFU  << GPDMA0_VERSION_VALUE_Pos)



/***   DSD Bit Fileds *******************/
/***************************************************************************/


/* DSD_CLC  =  Clock Control Register*/
#define   DSD_CLC_DISR_Pos  	(0U)
#define   DSD_CLC_DISR_Msk  	(0x00000001U  << DSD_CLC_DISR_Pos)

#define   DSD_CLC_DISS_Pos  	(1U)
#define   DSD_CLC_DISS_Msk  	(0x00000001U  << DSD_CLC_DISS_Pos)

#define   DSD_CLC_EDIS_Pos  	(3U)
#define   DSD_CLC_EDIS_Msk  	(0x00000001U  << DSD_CLC_EDIS_Pos)

/* DSD_ID  =  Module Identification Register*/
#define   DSD_ID_MOD_REV_Pos  	(0U)
#define   DSD_ID_MOD_REV_Msk  	(0x000000FFU  << DSD_ID_MOD_REV_Pos)

#define   DSD_ID_MOD_TYPE_Pos  	(8U)
#define   DSD_ID_MOD_TYPE_Msk  	(0x000000FFU  << DSD_ID_MOD_TYPE_Pos)

#define   DSD_ID_MOD_NUMBER_Pos  	(16U)
#define   DSD_ID_MOD_NUMBER_Msk  	(0x0000FFFFU  << DSD_ID_MOD_NUMBER_Pos)

/* DSD_OCS  =  OCDS Control and Status Register*/
#define   DSD_OCS_SUS_Pos  	(24U)
#define   DSD_OCS_SUS_Msk  	(0x0000000FU  << DSD_OCS_SUS_Pos)

#define   DSD_OCS_SUS_P_Pos  	(28U)
#define   DSD_OCS_SUS_P_Msk  	(0x00000001U  << DSD_OCS_SUS_P_Pos)

#define   DSD_OCS_SUSSTA_Pos  	(29U)
#define   DSD_OCS_SUSSTA_Msk  	(0x00000001U  << DSD_OCS_SUSSTA_Pos)

/* DSD_GLOBCFG  =  Global Configuration Register*/
#define   DSD_GLOBCFG_MCSEL_Pos  	(0U)
#define   DSD_GLOBCFG_MCSEL_Msk  	(0x00000007U  << DSD_GLOBCFG_MCSEL_Pos)

/* DSD_GLOBRC  =  Global Run Control Register*/
#define   DSD_GLOBRC_CHRUN_Pos  	(0U)
#define   DSD_GLOBRC_CHRUN_Msk  	(0x0000000FU  << DSD_GLOBRC_CHRUN_Pos)

/* DSD_CGCFG  =  Carrier Generator Configuration Register*/
#define   DSD_CGCFG_CGMOD_Pos  	(0U)
#define   DSD_CGCFG_CGMOD_Msk  	(0x00000003U  << DSD_CGCFG_CGMOD_Pos)

#define   DSD_CGCFG_BREV_Pos  	(2U)
#define   DSD_CGCFG_BREV_Msk  	(0x00000001U  << DSD_CGCFG_BREV_Pos)

#define   DSD_CGCFG_SIGPOL_Pos  	(3U)
#define   DSD_CGCFG_SIGPOL_Msk  	(0x00000001U  << DSD_CGCFG_SIGPOL_Pos)

#define   DSD_CGCFG_DIVCG_Pos  	(4U)
#define   DSD_CGCFG_DIVCG_Msk  	(0x0000000FU  << DSD_CGCFG_DIVCG_Pos)

#define   DSD_CGCFG_RUN_Pos  	(15U)
#define   DSD_CGCFG_RUN_Msk  	(0x00000001U  << DSD_CGCFG_RUN_Pos)

#define   DSD_CGCFG_BITCOUNT_Pos  	(16U)
#define   DSD_CGCFG_BITCOUNT_Msk  	(0x0000001FU  << DSD_CGCFG_BITCOUNT_Pos)

#define   DSD_CGCFG_STEPCOUNT_Pos  	(24U)
#define   DSD_CGCFG_STEPCOUNT_Msk  	(0x0000000FU  << DSD_CGCFG_STEPCOUNT_Pos)

#define   DSD_CGCFG_STEPS_Pos  	(28U)
#define   DSD_CGCFG_STEPS_Msk  	(0x00000001U  << DSD_CGCFG_STEPS_Pos)

#define   DSD_CGCFG_STEPD_Pos  	(29U)
#define   DSD_CGCFG_STEPD_Msk  	(0x00000001U  << DSD_CGCFG_STEPD_Pos)

#define   DSD_CGCFG_SGNCG_Pos  	(30U)
#define   DSD_CGCFG_SGNCG_Msk  	(0x00000001U  << DSD_CGCFG_SGNCG_Pos)

/* DSD_EVFLAG  =  Event Flag Register*/
#define   DSD_EVFLAG_RESEV0_Pos  	(0U)
#define   DSD_EVFLAG_RESEV0_Msk  	(0x00000001U  << DSD_EVFLAG_RESEV0_Pos)

#define   DSD_EVFLAG_RESEV1_Pos  	(1U)
#define   DSD_EVFLAG_RESEV1_Msk  	(0x00000001U  << DSD_EVFLAG_RESEV1_Pos)

#define   DSD_EVFLAG_RESEV2_Pos  	(2U)
#define   DSD_EVFLAG_RESEV2_Msk  	(0x00000001U  << DSD_EVFLAG_RESEV2_Pos)

#define   DSD_EVFLAG_RESEV3_Pos  	(3U)
#define   DSD_EVFLAG_RESEV3_Msk  	(0x00000001U  << DSD_EVFLAG_RESEV3_Pos)

#define   DSD_EVFLAG_ALEV0_Pos  	(16U)
#define   DSD_EVFLAG_ALEV0_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV0_Pos)

#define   DSD_EVFLAG_ALEV1_Pos  	(17U)
#define   DSD_EVFLAG_ALEV1_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV1_Pos)

#define   DSD_EVFLAG_ALEV2_Pos  	(18U)
#define   DSD_EVFLAG_ALEV2_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV2_Pos)

#define   DSD_EVFLAG_ALEV3_Pos  	(19U)
#define   DSD_EVFLAG_ALEV3_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV3_Pos)

#define   DSD_EVFLAG_ALEV4_Pos  	(20U)
#define   DSD_EVFLAG_ALEV4_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV4_Pos)

#define   DSD_EVFLAG_ALEV5_Pos  	(21U)
#define   DSD_EVFLAG_ALEV5_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV5_Pos)

#define   DSD_EVFLAG_ALEV6_Pos  	(22U)
#define   DSD_EVFLAG_ALEV6_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV6_Pos)

#define   DSD_EVFLAG_ALEV7_Pos  	(23U)
#define   DSD_EVFLAG_ALEV7_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV7_Pos)

#define   DSD_EVFLAG_ALEV8_Pos  	(24U)
#define   DSD_EVFLAG_ALEV8_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV8_Pos)

#define   DSD_EVFLAG_ALEV9_Pos  	(25U)
#define   DSD_EVFLAG_ALEV9_Msk  	(0x00000001U  << DSD_EVFLAG_ALEV9_Pos)

/* DSD_EVFLAGCLR  =  Event Flag Clear Register*/
#define   DSD_EVFLAGCLR_RESEC0_Pos  	(0U)
#define   DSD_EVFLAGCLR_RESEC0_Msk  	(0x00000001U  << DSD_EVFLAGCLR_RESEC0_Pos)

#define   DSD_EVFLAGCLR_RESEC1_Pos  	(1U)
#define   DSD_EVFLAGCLR_RESEC1_Msk  	(0x00000001U  << DSD_EVFLAGCLR_RESEC1_Pos)

#define   DSD_EVFLAGCLR_RESEC2_Pos  	(2U)
#define   DSD_EVFLAGCLR_RESEC2_Msk  	(0x00000001U  << DSD_EVFLAGCLR_RESEC2_Pos)

#define   DSD_EVFLAGCLR_RESEC3_Pos  	(3U)
#define   DSD_EVFLAGCLR_RESEC3_Msk  	(0x00000001U  << DSD_EVFLAGCLR_RESEC3_Pos)

#define   DSD_EVFLAGCLR_ALEC0_Pos  	(16U)
#define   DSD_EVFLAGCLR_ALEC0_Msk  	(0x00000001U  << DSD_EVFLAGCLR_ALEC0_Pos)

#define   DSD_EVFLAGCLR_ALEC1_Pos  	(17U)
#define   DSD_EVFLAGCLR_ALEC1_Msk  	(0x00000001U  << DSD_EVFLAGCLR_ALEC1_Pos)

#define   DSD_EVFLAGCLR_ALEC2_Pos  	(18U)
#define   DSD_EVFLAGCLR_ALEC2_Msk  	(0x00000001U  << DSD_EVFLAGCLR_ALEC2_Pos)

#define   DSD_EVFLAGCLR_ALEC3_Pos  	(19U)
#define   DSD_EVFLAGCLR_ALEC3_Msk  	(0x00000001U  << DSD_EVFLAGCLR_ALEC3_Pos)

/* DSD_CH_MODCFGx  =  Modulator Configuration Register 0*/
#define   DSD_CH_MODCFG_DIVM_Pos  	(16U)
#define   DSD_CH_MODCFG_DIVM_Msk  	(0x0000000FU  << DSD_CH_MODCFG_DIVM_Pos)

#define   DSD_CH_MODCFG_DWC_Pos  	(23U)
#define   DSD_CH_MODCFG_DWC_Msk  	(0x00000001U  << DSD_CH_MODCFG_DWC_Pos)

/* DSD_CH_DICFGx  =  Demodulator Input Configuration Register 0*/
#define   DSD_CH_DICFG_DSRC_Pos  	(0U)
#define   DSD_CH_DICFG_DSRC_Msk  	(0x0000000FU  << DSD_CH_DICFG_DSRC_Pos)

#define   DSD_CH_DICFG_DSWC_Pos  	(7U)
#define   DSD_CH_DICFG_DSWC_Msk  	(0x00000001U  << DSD_CH_DICFG_DSWC_Pos)

#define   DSD_CH_DICFG_ITRMODE_Pos  	(8U)
#define   DSD_CH_DICFG_ITRMODE_Msk  	(0x00000003U  << DSD_CH_DICFG_ITRMODE_Pos)

#define   DSD_CH_DICFG_TSTRMODE_Pos  	(10U)
#define   DSD_CH_DICFG_TSTRMODE_Msk  	(0x00000003U  << DSD_CH_DICFG_TSTRMODE_Pos)

#define   DSD_CH_DICFG_TRSEL_Pos  	(12U)
#define   DSD_CH_DICFG_TRSEL_Msk  	(0x00000007U  << DSD_CH_DICFG_TRSEL_Pos)

#define   DSD_CH_DICFG_TRWC_Pos  	(15U)
#define   DSD_CH_DICFG_TRWC_Msk  	(0x00000001U  << DSD_CH_DICFG_TRWC_Pos)

#define   DSD_CH_DICFG_CSRC_Pos  	(16U)
#define   DSD_CH_DICFG_CSRC_Msk  	(0x0000000FU  << DSD_CH_DICFG_CSRC_Pos)

#define   DSD_CH_DICFG_STROBE_Pos  	(20U)
#define   DSD_CH_DICFG_STROBE_Msk  	(0x0000000FU  << DSD_CH_DICFG_STROBE_Pos)

#define   DSD_CH_DICFG_SCWC_Pos  	(31U)
#define   DSD_CH_DICFG_SCWC_Msk  	(0x00000001U  << DSD_CH_DICFG_SCWC_Pos)

/* DSD_CH_FCFGCx  =  Filter Configuration Register 0, Main Comb Filter*/
#define   DSD_CH_FCFGC_CFMDF_Pos  	(0U)
#define   DSD_CH_FCFGC_CFMDF_Msk  	(0x000000FFU  << DSD_CH_FCFGC_CFMDF_Pos)

#define   DSD_CH_FCFGC_CFMC_Pos  	(8U)
#define   DSD_CH_FCFGC_CFMC_Msk  	(0x00000003U  << DSD_CH_FCFGC_CFMC_Pos)

#define   DSD_CH_FCFGC_CFEN_Pos  	(10U)
#define   DSD_CH_FCFGC_CFEN_Msk  	(0x00000001U  << DSD_CH_FCFGC_CFEN_Pos)

#define   DSD_CH_FCFGC_SRGM_Pos  	(14U)
#define   DSD_CH_FCFGC_SRGM_Msk  	(0x00000003U  << DSD_CH_FCFGC_SRGM_Pos)

#define   DSD_CH_FCFGC_CFMSV_Pos  	(16U)
#define   DSD_CH_FCFGC_CFMSV_Msk  	(0x000000FFU  << DSD_CH_FCFGC_CFMSV_Pos)

#define   DSD_CH_FCFGC_CFMDCNT_Pos  	(24U)
#define   DSD_CH_FCFGC_CFMDCNT_Msk  	(0x000000FFU  << DSD_CH_FCFGC_CFMDCNT_Pos)

/* DSD_CH_FCFGAx  =  Filter Configuration Register 0, Auxiliary Filter*/
#define   DSD_CH_FCFGA_CFADF_Pos  	(0U)
#define   DSD_CH_FCFGA_CFADF_Msk  	(0x000000FFU  << DSD_CH_FCFGA_CFADF_Pos)

#define   DSD_CH_FCFGA_CFAC_Pos  	(8U)
#define   DSD_CH_FCFGA_CFAC_Msk  	(0x00000003U  << DSD_CH_FCFGA_CFAC_Pos)

#define   DSD_CH_FCFGA_SRGA_Pos  	(10U)
#define   DSD_CH_FCFGA_SRGA_Msk  	(0x00000003U  << DSD_CH_FCFGA_SRGA_Pos)

#define   DSD_CH_FCFGA_ESEL_Pos  	(12U)
#define   DSD_CH_FCFGA_ESEL_Msk  	(0x00000003U  << DSD_CH_FCFGA_ESEL_Pos)

#define   DSD_CH_FCFGA_EGT_Pos  	(14U)
#define   DSD_CH_FCFGA_EGT_Msk  	(0x00000001U  << DSD_CH_FCFGA_EGT_Pos)

#define   DSD_CH_FCFGA_CFADCNT_Pos  	(24U)
#define   DSD_CH_FCFGA_CFADCNT_Msk  	(0x000000FFU  << DSD_CH_FCFGA_CFADCNT_Pos)

/* DSD_CH_IWCTRx  =  Integration Window Control Register 0*/
#define   DSD_CH_IWCTR_NVALCNT_Pos  	(0U)
#define   DSD_CH_IWCTR_NVALCNT_Msk  	(0x0000003FU  << DSD_CH_IWCTR_NVALCNT_Pos)

#define   DSD_CH_IWCTR_INTEN_Pos  	(7U)
#define   DSD_CH_IWCTR_INTEN_Msk  	(0x00000001U  << DSD_CH_IWCTR_INTEN_Pos)

#define   DSD_CH_IWCTR_REPCNT_Pos  	(8U)
#define   DSD_CH_IWCTR_REPCNT_Msk  	(0x0000000FU  << DSD_CH_IWCTR_REPCNT_Pos)

#define   DSD_CH_IWCTR_REPVAL_Pos  	(12U)
#define   DSD_CH_IWCTR_REPVAL_Msk  	(0x0000000FU  << DSD_CH_IWCTR_REPVAL_Pos)

#define   DSD_CH_IWCTR_NVALDIS_Pos  	(16U)
#define   DSD_CH_IWCTR_NVALDIS_Msk  	(0x0000003FU  << DSD_CH_IWCTR_NVALDIS_Pos)

#define   DSD_CH_IWCTR_IWS_Pos  	(23U)
#define   DSD_CH_IWCTR_IWS_Msk  	(0x00000001U  << DSD_CH_IWCTR_IWS_Pos)

#define   DSD_CH_IWCTR_NVALINT_Pos  	(24U)
#define   DSD_CH_IWCTR_NVALINT_Msk  	(0x0000003FU  << DSD_CH_IWCTR_NVALINT_Pos)

/* DSD_CH_BOUNDSELx  =  Boundary Select Register 0*/
#define   DSD_CH_BOUNDSEL_BOUNDARYL_Pos  	(0U)
#define   DSD_CH_BOUNDSEL_BOUNDARYL_Msk  	(0x0000FFFFU  << DSD_CH_BOUNDSEL_BOUNDARYL_Pos)

#define   DSD_CH_BOUNDSEL_BOUNDARYU_Pos  	(16U)
#define   DSD_CH_BOUNDSEL_BOUNDARYU_Msk  	(0x0000FFFFU  << DSD_CH_BOUNDSEL_BOUNDARYU_Pos)

/* DSD_CH_RESMx  =  Result Register x Main Filter*/
#define   DSD_CH_RESM_RESULT_Pos  	(0U)
#define   DSD_CH_RESM_RESULT_Msk  	(0x0000FFFFU  << DSD_CH_RESM_RESULT_Pos)

/* DSD_CH_OFFMx  =  Offset Register x Main Filter*/
#define   DSD_CH_OFFM_OFFSET_Pos  	(0U)
#define   DSD_CH_OFFM_OFFSET_Msk  	(0x0000FFFFU  << DSD_CH_OFFM_OFFSET_Pos)

/* DSD_CH_RESAx  =  Result Register x Auxiliary Filter*/
#define   DSD_CH_RESA_RESULT_Pos  	(0U)
#define   DSD_CH_RESA_RESULT_Msk  	(0x0000FFFFU  << DSD_CH_RESA_RESULT_Pos)

/* DSD_CH_TSTMPx  =  Time-Stamp Register 0*/
#define   DSD_CH_TSTMP_RESULT_Pos  	(0U)
#define   DSD_CH_TSTMP_RESULT_Msk  	(0x0000FFFFU  << DSD_CH_TSTMP_RESULT_Pos)

#define   DSD_CH_TSTMP_CFMDCNT_Pos  	(16U)
#define   DSD_CH_TSTMP_CFMDCNT_Msk  	(0x000000FFU  << DSD_CH_TSTMP_CFMDCNT_Pos)

#define   DSD_CH_TSTMP_NVALCNT_Pos  	(24U)
#define   DSD_CH_TSTMP_NVALCNT_Msk  	(0x0000003FU  << DSD_CH_TSTMP_NVALCNT_Pos)

/* DSD_CH_CGSYNCx  =  Carrier Generator Synchronization Register 0*/
#define   DSD_CH_CGSYNC_SDCOUNT_Pos  	(0U)
#define   DSD_CH_CGSYNC_SDCOUNT_Msk  	(0x000000FFU  << DSD_CH_CGSYNC_SDCOUNT_Pos)

#define   DSD_CH_CGSYNC_SDCAP_Pos  	(8U)
#define   DSD_CH_CGSYNC_SDCAP_Msk  	(0x000000FFU  << DSD_CH_CGSYNC_SDCAP_Pos)

#define   DSD_CH_CGSYNC_SDPOS_Pos  	(16U)
#define   DSD_CH_CGSYNC_SDPOS_Msk  	(0x000000FFU  << DSD_CH_CGSYNC_SDPOS_Pos)

#define   DSD_CH_CGSYNC_SDNEG_Pos  	(24U)
#define   DSD_CH_CGSYNC_SDNEG_Msk  	(0x000000FFU  << DSD_CH_CGSYNC_SDNEG_Pos)

/* DSD_CH_RECTCFGx  =  Rectification Configuration Register 0*/
#define   DSD_CH_RECTCFG_RFEN_Pos  	(0U)
#define   DSD_CH_RECTCFG_RFEN_Msk  	(0x00000001U  << DSD_CH_RECTCFG_RFEN_Pos)

#define   DSD_CH_RECTCFG_SSRC_Pos  	(4U)
#define   DSD_CH_RECTCFG_SSRC_Msk  	(0x00000003U  << DSD_CH_RECTCFG_SSRC_Pos)

#define   DSD_CH_RECTCFG_SDVAL_Pos  	(15U)
#define   DSD_CH_RECTCFG_SDVAL_Msk  	(0x00000001U  << DSD_CH_RECTCFG_SDVAL_Pos)

#define   DSD_CH_RECTCFG_SGNCS_Pos  	(30U)
#define   DSD_CH_RECTCFG_SGNCS_Msk  	(0x00000001U  << DSD_CH_RECTCFG_SGNCS_Pos)

#define   DSD_CH_RECTCFG_SGND_Pos  	(31U)
#define   DSD_CH_RECTCFG_SGND_Msk  	(0x00000001U  << DSD_CH_RECTCFG_SGND_Pos)


#if __CM4_CMSIS_VERSION_MAIN < 3
// --------------------------------------  DWT_DWT_CTRL  ------------------------------------------
#define DWT_DWT_CTRL_CYCCNTENA_Pos                            0                                                         /*!< DWT DWT_CTRL: CYCCNTENA Position    */
#define DWT_DWT_CTRL_CYCCNTENA_Msk                            (0x01UL << DWT_DWT_CTRL_CYCCNTENA_Pos)                    /*!< DWT DWT_CTRL: CYCCNTENA Mask        */
#define DWT_DWT_CTRL_POSTPRESET_Pos                           1                                                         /*!< DWT DWT_CTRL: POSTPRESET Position   */
#define DWT_DWT_CTRL_POSTPRESET_Msk                           (0x0fUL << DWT_DWT_CTRL_POSTPRESET_Pos)                   /*!< DWT DWT_CTRL: POSTPRESET Mask       */
#define DWT_DWT_CTRL_POSTINIT_Pos                             5                                                         /*!< DWT DWT_CTRL: POSTINIT Position     */
#define DWT_DWT_CTRL_POSTINIT_Msk                             (0x0fUL << DWT_DWT_CTRL_POSTINIT_Pos)                     /*!< DWT DWT_CTRL: POSTINIT Mask         */
#define DWT_DWT_CTRL_CYCTAP_Pos                               9                                                         /*!< DWT DWT_CTRL: CYCTAP Position       */
#define DWT_DWT_CTRL_CYCTAP_Msk                               (0x01UL << DWT_DWT_CTRL_CYCTAP_Pos)                       /*!< DWT DWT_CTRL: CYCTAP Mask           */
#define DWT_DWT_CTRL_SYNCTAP_Pos                              10                                                        /*!< DWT DWT_CTRL: SYNCTAP Position      */
#define DWT_DWT_CTRL_SYNCTAP_Msk                              (0x03UL << DWT_DWT_CTRL_SYNCTAP_Pos)                      /*!< DWT DWT_CTRL: SYNCTAP Mask          */
#define DWT_DWT_CTRL_PCSAMPLEENA_Pos                          12                                                        /*!< DWT DWT_CTRL: PCSAMPLEENA Position  */
#define DWT_DWT_CTRL_PCSAMPLEENA_Msk                          (0x01UL << DWT_DWT_CTRL_PCSAMPLEENA_Pos)                  /*!< DWT DWT_CTRL: PCSAMPLEENA Mask      */
#define DWT_DWT_CTRL_Reserved_Pos                             13                                                        /*!< DWT DWT_CTRL: Reserved Position     */
#define DWT_DWT_CTRL_Reserved_Msk                             (0x07UL << DWT_DWT_CTRL_Reserved_Pos)                     /*!< DWT DWT_CTRL: Reserved Mask         */
#define DWT_DWT_CTRL_EXCTRCENA_Pos                            16                                                        /*!< DWT DWT_CTRL: EXCTRCENA Position    */
#define DWT_DWT_CTRL_EXCTRCENA_Msk                            (0x01UL << DWT_DWT_CTRL_EXCTRCENA_Pos)                    /*!< DWT DWT_CTRL: EXCTRCENA Mask        */
#define DWT_DWT_CTRL_CPIEVTENA_Pos                            17                                                        /*!< DWT DWT_CTRL: CPIEVTENA Position    */
#define DWT_DWT_CTRL_CPIEVTENA_Msk                            (0x01UL << DWT_DWT_CTRL_CPIEVTENA_Pos)                    /*!< DWT DWT_CTRL: CPIEVTENA Mask        */
#define DWT_DWT_CTRL_EXCEVTENA_Pos                            18                                                        /*!< DWT DWT_CTRL: EXCEVTENA Position    */
#define DWT_DWT_CTRL_EXCEVTENA_Msk                            (0x01UL << DWT_DWT_CTRL_EXCEVTENA_Pos)                    /*!< DWT DWT_CTRL: EXCEVTENA Mask        */
#define DWT_DWT_CTRL_SLEEPEVTENA_Pos                          19                                                        /*!< DWT DWT_CTRL: SLEEPEVTENA Position  */
#define DWT_DWT_CTRL_SLEEPEVTENA_Msk                          (0x01UL << DWT_DWT_CTRL_SLEEPEVTENA_Pos)                  /*!< DWT DWT_CTRL: SLEEPEVTENA Mask      */
#define DWT_DWT_CTRL_LSUEVTENA_Pos                            20                                                        /*!< DWT DWT_CTRL: LSUEVTENA Position    */
#define DWT_DWT_CTRL_LSUEVTENA_Msk                            (0x01UL << DWT_DWT_CTRL_LSUEVTENA_Pos)                    /*!< DWT DWT_CTRL: LSUEVTENA Mask        */
#define DWT_DWT_CTRL_FOLDEVTENA_Pos                           21                                                        /*!< DWT DWT_CTRL: FOLDEVTENA Position   */
#define DWT_DWT_CTRL_FOLDEVTENA_Msk                           (0x01UL << DWT_DWT_CTRL_FOLDEVTENA_Pos)                   /*!< DWT DWT_CTRL: FOLDEVTENA Mask       */
#define DWT_DWT_CTRL_CYCEVTENA_Pos                            22                                                        /*!< DWT DWT_CTRL: CYCEVTENA Position    */
#define DWT_DWT_CTRL_CYCEVTENA_Msk                            (0x01UL << DWT_DWT_CTRL_CYCEVTENA_Pos)                    /*!< DWT DWT_CTRL: CYCEVTENA Mask        */
#define DWT_DWT_CTRL_Reserved1_Pos                            23                                                        /*!< DWT DWT_CTRL: Reserved1 Position    */
#define DWT_DWT_CTRL_Reserved1_Msk                            (0x01UL << DWT_DWT_CTRL_Reserved1_Pos)                    /*!< DWT DWT_CTRL: Reserved1 Mask        */
#define DWT_DWT_CTRL_NOPRFCNT_Pos                             24                                                        /*!< DWT DWT_CTRL: NOPRFCNT Position     */
#define DWT_DWT_CTRL_NOPRFCNT_Msk                             (0x01UL << DWT_DWT_CTRL_NOPRFCNT_Pos)                     /*!< DWT DWT_CTRL: NOPRFCNT Mask         */
#define DWT_DWT_CTRL_NOCYCCNT_Pos                             25                                                        /*!< DWT DWT_CTRL: NOCYCCNT Position     */
#define DWT_DWT_CTRL_NOCYCCNT_Msk                             (0x01UL << DWT_DWT_CTRL_NOCYCCNT_Pos)                     /*!< DWT DWT_CTRL: NOCYCCNT Mask         */
#define DWT_DWT_CTRL_NOEXTTRIG_Pos                            26                                                        /*!< DWT DWT_CTRL: NOEXTTRIG Position    */
#define DWT_DWT_CTRL_NOEXTTRIG_Msk                            (0x01UL << DWT_DWT_CTRL_NOEXTTRIG_Pos)                    /*!< DWT DWT_CTRL: NOEXTTRIG Mask        */
#define DWT_DWT_CTRL_NOTRCPKT_Pos                             27                                                        /*!< DWT DWT_CTRL: NOTRCPKT Position     */
#define DWT_DWT_CTRL_NOTRCPKT_Msk                             (0x01UL << DWT_DWT_CTRL_NOTRCPKT_Pos)                     /*!< DWT DWT_CTRL: NOTRCPKT Mask         */
#define DWT_DWT_CTRL_NUMCOMP_Pos                              28                                                        /*!< DWT DWT_CTRL: NUMCOMP Position      */
#define DWT_DWT_CTRL_NUMCOMP_Msk                              (0x0fUL << DWT_DWT_CTRL_NUMCOMP_Pos)                      /*!< DWT DWT_CTRL: NUMCOMP Mask          */

// -------------------------------------  DWT_DWT_CYCCNT  -----------------------------------------
#define DWT_DWT_CYCCNT_CYCCNT_Pos                             0                                                         /*!< DWT DWT_CYCCNT: CYCCNT Position     */
#define DWT_DWT_CYCCNT_CYCCNT_Msk                             (0xffffffffUL << DWT_DWT_CYCCNT_CYCCNT_Pos)               /*!< DWT DWT_CYCCNT: CYCCNT Mask         */

// -------------------------------------  DWT_DWT_CPICNT  -----------------------------------------
#define DWT_DWT_CPICNT_CPICNT_Pos                             0                                                         /*!< DWT DWT_CPICNT: CPICNT Position     */
#define DWT_DWT_CPICNT_CPICNT_Msk                             (0x000000ffUL << DWT_DWT_CPICNT_CPICNT_Pos)               /*!< DWT DWT_CPICNT: CPICNT Mask         */
#define DWT_DWT_CPICNT_Reserved_Pos                           8                                                         /*!< DWT DWT_CPICNT: Reserved Position   */
#define DWT_DWT_CPICNT_Reserved_Msk                           (0x00ffffffUL << DWT_DWT_CPICNT_Reserved_Pos)             /*!< DWT DWT_CPICNT: Reserved Mask       */

// -------------------------------------  DWT_DWT_EXCCNT  -----------------------------------------
#define DWT_DWT_EXCCNT_EXCCNT_Pos                             0                                                         /*!< DWT DWT_EXCCNT: EXCCNT Position     */
#define DWT_DWT_EXCCNT_EXCCNT_Msk                             (0x000000ffUL << DWT_DWT_EXCCNT_EXCCNT_Pos)               /*!< DWT DWT_EXCCNT: EXCCNT Mask         */
#define DWT_DWT_EXCCNT_Reserved_Pos                           8                                                         /*!< DWT DWT_EXCCNT: Reserved Position   */
#define DWT_DWT_EXCCNT_Reserved_Msk                           (0x00ffffffUL << DWT_DWT_EXCCNT_Reserved_Pos)             /*!< DWT DWT_EXCCNT: Reserved Mask       */

// ------------------------------------  DWT_DWT_SLEEPCNT  ----------------------------------------
#define DWT_DWT_SLEEPCNT_SLEEPCNT_Pos                         0                                                         /*!< DWT DWT_SLEEPCNT: SLEEPCNT Position */
#define DWT_DWT_SLEEPCNT_SLEEPCNT_Msk                         (0x000000ffUL << DWT_DWT_SLEEPCNT_SLEEPCNT_Pos)           /*!< DWT DWT_SLEEPCNT: SLEEPCNT Mask     */
#define DWT_DWT_SLEEPCNT_Reserved_Pos                         8                                                         /*!< DWT DWT_SLEEPCNT: Reserved Position */
#define DWT_DWT_SLEEPCNT_Reserved_Msk                         (0x00ffffffUL << DWT_DWT_SLEEPCNT_Reserved_Pos)           /*!< DWT DWT_SLEEPCNT: Reserved Mask     */

// -------------------------------------  DWT_DWT_LSUCNT  -----------------------------------------
#define DWT_DWT_LSUCNT_LSUCNT_Pos                             0                                                         /*!< DWT DWT_LSUCNT: LSUCNT Position     */
#define DWT_DWT_LSUCNT_LSUCNT_Msk                             (0x000000ffUL << DWT_DWT_LSUCNT_LSUCNT_Pos)               /*!< DWT DWT_LSUCNT: LSUCNT Mask         */
#define DWT_DWT_LSUCNT_Reserved_Pos                           8                                                         /*!< DWT DWT_LSUCNT: Reserved Position   */
#define DWT_DWT_LSUCNT_Reserved_Msk                           (0x00ffffffUL << DWT_DWT_LSUCNT_Reserved_Pos)             /*!< DWT DWT_LSUCNT: Reserved Mask       */

// -------------------------------------  DWT_DWT_FOLDCNT  ----------------------------------------
#define DWT_DWT_FOLDCNT_FOLDCNT_Pos                           0                                                         /*!< DWT DWT_FOLDCNT: FOLDCNT Position   */
#define DWT_DWT_FOLDCNT_FOLDCNT_Msk                           (0x000000ffUL << DWT_DWT_FOLDCNT_FOLDCNT_Pos)             /*!< DWT DWT_FOLDCNT: FOLDCNT Mask       */
#define DWT_DWT_FOLDCNT_Reserved_Pos                          8                                                         /*!< DWT DWT_FOLDCNT: Reserved Position  */
#define DWT_DWT_FOLDCNT_Reserved_Msk                          (0x00ffffffUL << DWT_DWT_FOLDCNT_Reserved_Pos)            /*!< DWT DWT_FOLDCNT: Reserved Mask      */

// --------------------------------------  DWT_DWT_PCSR  ------------------------------------------
#define DWT_DWT_PCSR_EIASAMPLE_Pos                            0                                                         /*!< DWT DWT_PCSR: EIASAMPLE Position    */
#define DWT_DWT_PCSR_EIASAMPLE_Msk                            (0xffffffffUL << DWT_DWT_PCSR_EIASAMPLE_Pos)              /*!< DWT DWT_PCSR: EIASAMPLE Mask        */

// --------------------------------------  DWT_DWT_COMP0  -----------------------------------------
#define DWT_DWT_COMP0_COMP_Pos                                0                                                         /*!< DWT DWT_COMP0: COMP Position        */
#define DWT_DWT_COMP0_COMP_Msk                                (0xffffffffUL << DWT_DWT_COMP0_COMP_Pos)                  /*!< DWT DWT_COMP0: COMP Mask            */

// --------------------------------------  DWT_DWT_MASK0  -----------------------------------------
#define DWT_DWT_MASK0_MASK_Pos                                0                                                         /*!< DWT DWT_MASK0: MASK Position        */
#define DWT_DWT_MASK0_MASK_Msk                                (0x1fUL << DWT_DWT_MASK0_MASK_Pos)                        /*!< DWT DWT_MASK0: MASK Mask            */
#define DWT_DWT_MASK0_Reserved_Pos                            5                                                         /*!< DWT DWT_MASK0: Reserved Position    */
#define DWT_DWT_MASK0_Reserved_Msk                            (0x07ffffffUL << DWT_DWT_MASK0_Reserved_Pos)              /*!< DWT DWT_MASK0: Reserved Mask        */

// ------------------------------------  DWT_DWT_FUNCTION0  ---------------------------------------
#define DWT_DWT_FUNCTION0_FUNCTION_Pos                        0                                                         /*!< DWT DWT_FUNCTION0: FUNCTION Position */
#define DWT_DWT_FUNCTION0_FUNCTION_Msk                        (0x0fUL << DWT_DWT_FUNCTION0_FUNCTION_Pos)                /*!< DWT DWT_FUNCTION0: FUNCTION Mask    */
#define DWT_DWT_FUNCTION0_Reserved_Pos                        4                                                         /*!< DWT DWT_FUNCTION0: Reserved Position */
#define DWT_DWT_FUNCTION0_Reserved_Msk                        (0x01UL << DWT_DWT_FUNCTION0_Reserved_Pos)                /*!< DWT DWT_FUNCTION0: Reserved Mask    */
#define DWT_DWT_FUNCTION0_EMITRANGE_Pos                       5                                                         /*!< DWT DWT_FUNCTION0: EMITRANGE Position */
#define DWT_DWT_FUNCTION0_EMITRANGE_Msk                       (0x01UL << DWT_DWT_FUNCTION0_EMITRANGE_Pos)               /*!< DWT DWT_FUNCTION0: EMITRANGE Mask   */
#define DWT_DWT_FUNCTION0_Reserved1_Pos                       6                                                         /*!< DWT DWT_FUNCTION0: Reserved1 Position */
#define DWT_DWT_FUNCTION0_Reserved1_Msk                       (0x01UL << DWT_DWT_FUNCTION0_Reserved1_Pos)               /*!< DWT DWT_FUNCTION0: Reserved1 Mask   */
#define DWT_DWT_FUNCTION0_CYCMATCH_Pos                        7                                                         /*!< DWT DWT_FUNCTION0: CYCMATCH Position */
#define DWT_DWT_FUNCTION0_CYCMATCH_Msk                        (0x01UL << DWT_DWT_FUNCTION0_CYCMATCH_Pos)                /*!< DWT DWT_FUNCTION0: CYCMATCH Mask    */
#define DWT_DWT_FUNCTION0_DATAVMATCH_Pos                      8                                                         /*!< DWT DWT_FUNCTION0: DATAVMATCH Position */
#define DWT_DWT_FUNCTION0_DATAVMATCH_Msk                      (0x01UL << DWT_DWT_FUNCTION0_DATAVMATCH_Pos)              /*!< DWT DWT_FUNCTION0: DATAVMATCH Mask  */
#define DWT_DWT_FUNCTION0_LNK1ENA_Pos                         9                                                         /*!< DWT DWT_FUNCTION0: LNK1ENA Position */
#define DWT_DWT_FUNCTION0_LNK1ENA_Msk                         (0x01UL << DWT_DWT_FUNCTION0_LNK1ENA_Pos)                 /*!< DWT DWT_FUNCTION0: LNK1ENA Mask     */
#define DWT_DWT_FUNCTION0_DATAVSIZE_Pos                       10                                                        /*!< DWT DWT_FUNCTION0: DATAVSIZE Position */
#define DWT_DWT_FUNCTION0_DATAVSIZE_Msk                       (0x03UL << DWT_DWT_FUNCTION0_DATAVSIZE_Pos)               /*!< DWT DWT_FUNCTION0: DATAVSIZE Mask   */
#define DWT_DWT_FUNCTION0_DATAVADDR0_Pos                      12                                                        /*!< DWT DWT_FUNCTION0: DATAVADDR0 Position */
#define DWT_DWT_FUNCTION0_DATAVADDR0_Msk                      (0x0fUL << DWT_DWT_FUNCTION0_DATAVADDR0_Pos)              /*!< DWT DWT_FUNCTION0: DATAVADDR0 Mask  */
#define DWT_DWT_FUNCTION0_DATAVADDR1_Pos                      16                                                        /*!< DWT DWT_FUNCTION0: DATAVADDR1 Position */
#define DWT_DWT_FUNCTION0_DATAVADDR1_Msk                      (0x0fUL << DWT_DWT_FUNCTION0_DATAVADDR1_Pos)              /*!< DWT DWT_FUNCTION0: DATAVADDR1 Mask  */
#define DWT_DWT_FUNCTION0_Reserved2_Pos                       20                                                        /*!< DWT DWT_FUNCTION0: Reserved2 Position */
#define DWT_DWT_FUNCTION0_Reserved2_Msk                       (0x0fUL << DWT_DWT_FUNCTION0_Reserved2_Pos)               /*!< DWT DWT_FUNCTION0: Reserved2 Mask   */
#define DWT_DWT_FUNCTION0_MATCHED_Pos                         24                                                        /*!< DWT DWT_FUNCTION0: MATCHED Position */
#define DWT_DWT_FUNCTION0_MATCHED_Msk                         (0x01UL << DWT_DWT_FUNCTION0_MATCHED_Pos)                 /*!< DWT DWT_FUNCTION0: MATCHED Mask     */
#define DWT_DWT_FUNCTION0_Reserved3_Pos                       25                                                        /*!< DWT DWT_FUNCTION0: Reserved3 Position */
#define DWT_DWT_FUNCTION0_Reserved3_Msk                       (0x7fUL << DWT_DWT_FUNCTION0_Reserved3_Pos)               /*!< DWT DWT_FUNCTION0: Reserved3 Mask   */

// --------------------------------------  DWT_DWT_COMP1  -----------------------------------------
#define DWT_DWT_COMP1_COMP_Pos                                0                                                         /*!< DWT DWT_COMP1: COMP Position        */
#define DWT_DWT_COMP1_COMP_Msk                                (0xffffffffUL << DWT_DWT_COMP1_COMP_Pos)                  /*!< DWT DWT_COMP1: COMP Mask            */

// --------------------------------------  DWT_DWT_MASK1  -----------------------------------------
#define DWT_DWT_MASK1_MASK_Pos                                0                                                         /*!< DWT DWT_MASK1: MASK Position        */
#define DWT_DWT_MASK1_MASK_Msk                                (0x1fUL << DWT_DWT_MASK1_MASK_Pos)                        /*!< DWT DWT_MASK1: MASK Mask            */
#define DWT_DWT_MASK1_Reserved_Pos                            5                                                         /*!< DWT DWT_MASK1: Reserved Position    */
#define DWT_DWT_MASK1_Reserved_Msk                            (0x07ffffffUL << DWT_DWT_MASK1_Reserved_Pos)              /*!< DWT DWT_MASK1: Reserved Mask        */

// ------------------------------------  DWT_DWT_FUNCTION1  ---------------------------------------
#define DWT_DWT_FUNCTION1_FUNCTION_Pos                        0                                                         /*!< DWT DWT_FUNCTION1: FUNCTION Position */
#define DWT_DWT_FUNCTION1_FUNCTION_Msk                        (0x0fUL << DWT_DWT_FUNCTION1_FUNCTION_Pos)                /*!< DWT DWT_FUNCTION1: FUNCTION Mask    */
#define DWT_DWT_FUNCTION1_Reserved_Pos                        4                                                         /*!< DWT DWT_FUNCTION1: Reserved Position */
#define DWT_DWT_FUNCTION1_Reserved_Msk                        (0x01UL << DWT_DWT_FUNCTION1_Reserved_Pos)                /*!< DWT DWT_FUNCTION1: Reserved Mask    */
#define DWT_DWT_FUNCTION1_EMITRANGE_Pos                       5                                                         /*!< DWT DWT_FUNCTION1: EMITRANGE Position */
#define DWT_DWT_FUNCTION1_EMITRANGE_Msk                       (0x01UL << DWT_DWT_FUNCTION1_EMITRANGE_Pos)               /*!< DWT DWT_FUNCTION1: EMITRANGE Mask   */
#define DWT_DWT_FUNCTION1_Reserved1_Pos                       6                                                         /*!< DWT DWT_FUNCTION1: Reserved1 Position */
#define DWT_DWT_FUNCTION1_Reserved1_Msk                       (0x03UL << DWT_DWT_FUNCTION1_Reserved1_Pos)               /*!< DWT DWT_FUNCTION1: Reserved1 Mask   */
#define DWT_DWT_FUNCTION1_DATAVMATCH_Pos                      8                                                         /*!< DWT DWT_FUNCTION1: DATAVMATCH Position */
#define DWT_DWT_FUNCTION1_DATAVMATCH_Msk                      (0x01UL << DWT_DWT_FUNCTION1_DATAVMATCH_Pos)              /*!< DWT DWT_FUNCTION1: DATAVMATCH Mask  */
#define DWT_DWT_FUNCTION1_LNK1ENA_Pos                         9                                                         /*!< DWT DWT_FUNCTION1: LNK1ENA Position */
#define DWT_DWT_FUNCTION1_LNK1ENA_Msk                         (0x01UL << DWT_DWT_FUNCTION1_LNK1ENA_Pos)                 /*!< DWT DWT_FUNCTION1: LNK1ENA Mask     */
#define DWT_DWT_FUNCTION1_DATAVSIZE_Pos                       10                                                        /*!< DWT DWT_FUNCTION1: DATAVSIZE Position */
#define DWT_DWT_FUNCTION1_DATAVSIZE_Msk                       (0x03UL << DWT_DWT_FUNCTION1_DATAVSIZE_Pos)               /*!< DWT DWT_FUNCTION1: DATAVSIZE Mask   */
#define DWT_DWT_FUNCTION1_DATAVADDR0_Pos                      12                                                        /*!< DWT DWT_FUNCTION1: DATAVADDR0 Position */
#define DWT_DWT_FUNCTION1_DATAVADDR0_Msk                      (0x0fUL << DWT_DWT_FUNCTION1_DATAVADDR0_Pos)              /*!< DWT DWT_FUNCTION1: DATAVADDR0 Mask  */
#define DWT_DWT_FUNCTION1_DATAVADDR1_Pos                      16                                                        /*!< DWT DWT_FUNCTION1: DATAVADDR1 Position */
#define DWT_DWT_FUNCTION1_DATAVADDR1_Msk                      (0x0fUL << DWT_DWT_FUNCTION1_DATAVADDR1_Pos)              /*!< DWT DWT_FUNCTION1: DATAVADDR1 Mask  */
#define DWT_DWT_FUNCTION1_Reserved2_Pos                       20                                                        /*!< DWT DWT_FUNCTION1: Reserved2 Position */
#define DWT_DWT_FUNCTION1_Reserved2_Msk                       (0x0fUL << DWT_DWT_FUNCTION1_Reserved2_Pos)               /*!< DWT DWT_FUNCTION1: Reserved2 Mask   */
#define DWT_DWT_FUNCTION1_MATCHED_Pos                         24                                                        /*!< DWT DWT_FUNCTION1: MATCHED Position */
#define DWT_DWT_FUNCTION1_MATCHED_Msk                         (0x01UL << DWT_DWT_FUNCTION1_MATCHED_Pos)                 /*!< DWT DWT_FUNCTION1: MATCHED Mask     */
#define DWT_DWT_FUNCTION1_Reserved3_Pos                       25                                                        /*!< DWT DWT_FUNCTION1: Reserved3 Position */
#define DWT_DWT_FUNCTION1_Reserved3_Msk                       (0x7fUL << DWT_DWT_FUNCTION1_Reserved3_Pos)               /*!< DWT DWT_FUNCTION1: Reserved3 Mask   */

// --------------------------------------  DWT_DWT_COMP2  -----------------------------------------
#define DWT_DWT_COMP2_COMP_Pos                                0                                                         /*!< DWT DWT_COMP2: COMP Position        */
#define DWT_DWT_COMP2_COMP_Msk                                (0xffffffffUL << DWT_DWT_COMP2_COMP_Pos)                  /*!< DWT DWT_COMP2: COMP Mask            */

// --------------------------------------  DWT_DWT_MASK2  -----------------------------------------
#define DWT_DWT_MASK2_MASK_Pos                                0                                                         /*!< DWT DWT_MASK2: MASK Position        */
#define DWT_DWT_MASK2_MASK_Msk                                (0x1fUL << DWT_DWT_MASK2_MASK_Pos)                        /*!< DWT DWT_MASK2: MASK Mask            */
#define DWT_DWT_MASK2_Reserved_Pos                            5                                                         /*!< DWT DWT_MASK2: Reserved Position    */
#define DWT_DWT_MASK2_Reserved_Msk                            (0x07ffffffUL << DWT_DWT_MASK2_Reserved_Pos)              /*!< DWT DWT_MASK2: Reserved Mask        */

// ------------------------------------  DWT_DWT_FUNCTION2  ---------------------------------------
#define DWT_DWT_FUNCTION2_FUNCTION_Pos                        0                                                         /*!< DWT DWT_FUNCTION2: FUNCTION Position */
#define DWT_DWT_FUNCTION2_FUNCTION_Msk                        (0x0fUL << DWT_DWT_FUNCTION2_FUNCTION_Pos)                /*!< DWT DWT_FUNCTION2: FUNCTION Mask    */
#define DWT_DWT_FUNCTION2_Reserved_Pos                        4                                                         /*!< DWT DWT_FUNCTION2: Reserved Position */
#define DWT_DWT_FUNCTION2_Reserved_Msk                        (0x01UL << DWT_DWT_FUNCTION2_Reserved_Pos)                /*!< DWT DWT_FUNCTION2: Reserved Mask    */
#define DWT_DWT_FUNCTION2_EMITRANGE_Pos                       5                                                         /*!< DWT DWT_FUNCTION2: EMITRANGE Position */
#define DWT_DWT_FUNCTION2_EMITRANGE_Msk                       (0x01UL << DWT_DWT_FUNCTION2_EMITRANGE_Pos)               /*!< DWT DWT_FUNCTION2: EMITRANGE Mask   */
#define DWT_DWT_FUNCTION2_Reserved1_Pos                       6                                                         /*!< DWT DWT_FUNCTION2: Reserved1 Position */
#define DWT_DWT_FUNCTION2_Reserved1_Msk                       (0x03UL << DWT_DWT_FUNCTION2_Reserved1_Pos)               /*!< DWT DWT_FUNCTION2: Reserved1 Mask   */
#define DWT_DWT_FUNCTION2_DATAVMATCH_Pos                      8                                                         /*!< DWT DWT_FUNCTION2: DATAVMATCH Position */
#define DWT_DWT_FUNCTION2_DATAVMATCH_Msk                      (0x01UL << DWT_DWT_FUNCTION2_DATAVMATCH_Pos)              /*!< DWT DWT_FUNCTION2: DATAVMATCH Mask  */
#define DWT_DWT_FUNCTION2_LNK1ENA_Pos                         9                                                         /*!< DWT DWT_FUNCTION2: LNK1ENA Position */
#define DWT_DWT_FUNCTION2_LNK1ENA_Msk                         (0x01UL << DWT_DWT_FUNCTION2_LNK1ENA_Pos)                 /*!< DWT DWT_FUNCTION2: LNK1ENA Mask     */
#define DWT_DWT_FUNCTION2_DATAVSIZE_Pos                       10                                                        /*!< DWT DWT_FUNCTION2: DATAVSIZE Position */
#define DWT_DWT_FUNCTION2_DATAVSIZE_Msk                       (0x03UL << DWT_DWT_FUNCTION2_DATAVSIZE_Pos)               /*!< DWT DWT_FUNCTION2: DATAVSIZE Mask   */
#define DWT_DWT_FUNCTION2_DATAVADDR0_Pos                      12                                                        /*!< DWT DWT_FUNCTION2: DATAVADDR0 Position */
#define DWT_DWT_FUNCTION2_DATAVADDR0_Msk                      (0x0fUL << DWT_DWT_FUNCTION2_DATAVADDR0_Pos)              /*!< DWT DWT_FUNCTION2: DATAVADDR0 Mask  */
#define DWT_DWT_FUNCTION2_DATAVADDR1_Pos                      16                                                        /*!< DWT DWT_FUNCTION2: DATAVADDR1 Position */
#define DWT_DWT_FUNCTION2_DATAVADDR1_Msk                      (0x0fUL << DWT_DWT_FUNCTION2_DATAVADDR1_Pos)              /*!< DWT DWT_FUNCTION2: DATAVADDR1 Mask  */
#define DWT_DWT_FUNCTION2_Reserved2_Pos                       20                                                        /*!< DWT DWT_FUNCTION2: Reserved2 Position */
#define DWT_DWT_FUNCTION2_Reserved2_Msk                       (0x0fUL << DWT_DWT_FUNCTION2_Reserved2_Pos)               /*!< DWT DWT_FUNCTION2: Reserved2 Mask   */
#define DWT_DWT_FUNCTION2_MATCHED_Pos                         24                                                        /*!< DWT DWT_FUNCTION2: MATCHED Position */
#define DWT_DWT_FUNCTION2_MATCHED_Msk                         (0x01UL << DWT_DWT_FUNCTION2_MATCHED_Pos)                 /*!< DWT DWT_FUNCTION2: MATCHED Mask     */
#define DWT_DWT_FUNCTION2_Reserved3_Pos                       25                                                        /*!< DWT DWT_FUNCTION2: Reserved3 Position */
#define DWT_DWT_FUNCTION2_Reserved3_Msk                       (0x7fUL << DWT_DWT_FUNCTION2_Reserved3_Pos)               /*!< DWT DWT_FUNCTION2: Reserved3 Mask   */

// --------------------------------------  DWT_DWT_COMP3  -----------------------------------------
#define DWT_DWT_COMP3_COMP_Pos                                0                                                         /*!< DWT DWT_COMP3: COMP Position        */
#define DWT_DWT_COMP3_COMP_Msk                                (0xffffffffUL << DWT_DWT_COMP3_COMP_Pos)                  /*!< DWT DWT_COMP3: COMP Mask            */

// --------------------------------------  DWT_DWT_MASK3  -----------------------------------------
#define DWT_DWT_MASK3_MASK_Pos                                0                                                         /*!< DWT DWT_MASK3: MASK Position        */
#define DWT_DWT_MASK3_MASK_Msk                                (0x1fUL << DWT_DWT_MASK3_MASK_Pos)                        /*!< DWT DWT_MASK3: MASK Mask            */
#define DWT_DWT_MASK3_Reserved_Pos                            5                                                         /*!< DWT DWT_MASK3: Reserved Position    */
#define DWT_DWT_MASK3_Reserved_Msk                            (0x07ffffffUL << DWT_DWT_MASK3_Reserved_Pos)              /*!< DWT DWT_MASK3: Reserved Mask        */

// ------------------------------------  DWT_DWT_FUNCTION3  ---------------------------------------
#define DWT_DWT_FUNCTION3_FUNCTION_Pos                        0                                                         /*!< DWT DWT_FUNCTION3: FUNCTION Position */
#define DWT_DWT_FUNCTION3_FUNCTION_Msk                        (0x0fUL << DWT_DWT_FUNCTION3_FUNCTION_Pos)                /*!< DWT DWT_FUNCTION3: FUNCTION Mask    */
#define DWT_DWT_FUNCTION3_Reserved_Pos                        4                                                         /*!< DWT DWT_FUNCTION3: Reserved Position */
#define DWT_DWT_FUNCTION3_Reserved_Msk                        (0x01UL << DWT_DWT_FUNCTION3_Reserved_Pos)                /*!< DWT DWT_FUNCTION3: Reserved Mask    */
#define DWT_DWT_FUNCTION3_EMITRANGE_Pos                       5                                                         /*!< DWT DWT_FUNCTION3: EMITRANGE Position */
#define DWT_DWT_FUNCTION3_EMITRANGE_Msk                       (0x01UL << DWT_DWT_FUNCTION3_EMITRANGE_Pos)               /*!< DWT DWT_FUNCTION3: EMITRANGE Mask   */
#define DWT_DWT_FUNCTION3_Reserved1_Pos                       6                                                         /*!< DWT DWT_FUNCTION3: Reserved1 Position */
#define DWT_DWT_FUNCTION3_Reserved1_Msk                       (0x03UL << DWT_DWT_FUNCTION3_Reserved1_Pos)               /*!< DWT DWT_FUNCTION3: Reserved1 Mask   */
#define DWT_DWT_FUNCTION3_DATAVMATCH_Pos                      8                                                         /*!< DWT DWT_FUNCTION3: DATAVMATCH Position */
#define DWT_DWT_FUNCTION3_DATAVMATCH_Msk                      (0x01UL << DWT_DWT_FUNCTION3_DATAVMATCH_Pos)              /*!< DWT DWT_FUNCTION3: DATAVMATCH Mask  */
#define DWT_DWT_FUNCTION3_LNK1ENA_Pos                         9                                                         /*!< DWT DWT_FUNCTION3: LNK1ENA Position */
#define DWT_DWT_FUNCTION3_LNK1ENA_Msk                         (0x01UL << DWT_DWT_FUNCTION3_LNK1ENA_Pos)                 /*!< DWT DWT_FUNCTION3: LNK1ENA Mask     */
#define DWT_DWT_FUNCTION3_DATAVSIZE_Pos                       10                                                        /*!< DWT DWT_FUNCTION3: DATAVSIZE Position */
#define DWT_DWT_FUNCTION3_DATAVSIZE_Msk                       (0x03UL << DWT_DWT_FUNCTION3_DATAVSIZE_Pos)               /*!< DWT DWT_FUNCTION3: DATAVSIZE Mask   */
#define DWT_DWT_FUNCTION3_DATAVADDR0_Pos                      12                                                        /*!< DWT DWT_FUNCTION3: DATAVADDR0 Position */
#define DWT_DWT_FUNCTION3_DATAVADDR0_Msk                      (0x0fUL << DWT_DWT_FUNCTION3_DATAVADDR0_Pos)              /*!< DWT DWT_FUNCTION3: DATAVADDR0 Mask  */
#define DWT_DWT_FUNCTION3_DATAVADDR1_Pos                      16                                                        /*!< DWT DWT_FUNCTION3: DATAVADDR1 Position */
#define DWT_DWT_FUNCTION3_DATAVADDR1_Msk                      (0x0fUL << DWT_DWT_FUNCTION3_DATAVADDR1_Pos)              /*!< DWT DWT_FUNCTION3: DATAVADDR1 Mask  */
#define DWT_DWT_FUNCTION3_Reserved2_Pos                       20                                                        /*!< DWT DWT_FUNCTION3: Reserved2 Position */
#define DWT_DWT_FUNCTION3_Reserved2_Msk                       (0x0fUL << DWT_DWT_FUNCTION3_Reserved2_Pos)               /*!< DWT DWT_FUNCTION3: Reserved2 Mask   */
#define DWT_DWT_FUNCTION3_MATCHED_Pos                         24                                                        /*!< DWT DWT_FUNCTION3: MATCHED Position */
#define DWT_DWT_FUNCTION3_MATCHED_Msk                         (0x01UL << DWT_DWT_FUNCTION3_MATCHED_Pos)                 /*!< DWT DWT_FUNCTION3: MATCHED Mask     */
#define DWT_DWT_FUNCTION3_Reserved3_Pos                       25                                                        /*!< DWT DWT_FUNCTION3: Reserved3 Position */
#define DWT_DWT_FUNCTION3_Reserved3_Msk                       (0x7fUL << DWT_DWT_FUNCTION3_Reserved3_Pos)               /*!< DWT DWT_FUNCTION3: Reserved3 Mask   */

// ---------------------------------------  DWT_DWTPID4  ------------------------------------------
#define DWT_DWTPID4_PID4_Pos                                  0                                                         /*!< DWT DWTPID4: PID4 Position          */
#define DWT_DWTPID4_PID4_Msk                                  (0xffffffffUL << DWT_DWTPID4_PID4_Pos)                    /*!< DWT DWTPID4: PID4 Mask              */

// ---------------------------------------  DWT_DWTPID5  ------------------------------------------
#define DWT_DWTPID5_PID5_Pos                                  0                                                         /*!< DWT DWTPID5: PID5 Position          */
#define DWT_DWTPID5_PID5_Msk                                  (0xffffffffUL << DWT_DWTPID5_PID5_Pos)                    /*!< DWT DWTPID5: PID5 Mask              */

// ---------------------------------------  DWT_DWTPID6  ------------------------------------------
#define DWT_DWTPID6_PID6_Pos                                  0                                                         /*!< DWT DWTPID6: PID6 Position          */
#define DWT_DWTPID6_PID6_Msk                                  (0xffffffffUL << DWT_DWTPID6_PID6_Pos)                    /*!< DWT DWTPID6: PID6 Mask              */

// ---------------------------------------  DWT_DWTPID7  ------------------------------------------
#define DWT_DWTPID7_PID7_Pos                                  0                                                         /*!< DWT DWTPID7: PID7 Position          */
#define DWT_DWTPID7_PID7_Msk                                  (0xffffffffUL << DWT_DWTPID7_PID7_Pos)                    /*!< DWT DWTPID7: PID7 Mask              */

// ---------------------------------------  DWT_DWTPID0  ------------------------------------------
#define DWT_DWTPID0_PID0_Pos                                  0                                                         /*!< DWT DWTPID0: PID0 Position          */
#define DWT_DWTPID0_PID0_Msk                                  (0xffffffffUL << DWT_DWTPID0_PID0_Pos)                    /*!< DWT DWTPID0: PID0 Mask              */

// ---------------------------------------  DWT_DWTPID1  ------------------------------------------
#define DWT_DWTPID1_PID1_Pos                                  0                                                         /*!< DWT DWTPID1: PID1 Position          */
#define DWT_DWTPID1_PID1_Msk                                  (0xffffffffUL << DWT_DWTPID1_PID1_Pos)                    /*!< DWT DWTPID1: PID1 Mask              */

// ---------------------------------------  DWT_DWTPID2  ------------------------------------------
#define DWT_DWTPID2_PID2_Pos                                  0                                                         /*!< DWT DWTPID2: PID2 Position          */
#define DWT_DWTPID2_PID2_Msk                                  (0xffffffffUL << DWT_DWTPID2_PID2_Pos)                    /*!< DWT DWTPID2: PID2 Mask              */

// ---------------------------------------  DWT_DWTPID3  ------------------------------------------
#define DWT_DWTPID3_PID3_Pos                                  0                                                         /*!< DWT DWTPID3: PID3 Position          */
#define DWT_DWTPID3_PID3_Msk                                  (0xffffffffUL << DWT_DWTPID3_PID3_Pos)                    /*!< DWT DWTPID3: PID3 Mask              */

// ---------------------------------------  DWT_DWTCID0  ------------------------------------------
#define DWT_DWTCID0_CID0_Pos                                  0                                                         /*!< DWT DWTCID0: CID0 Position          */
#define DWT_DWTCID0_CID0_Msk                                  (0xffffffffUL << DWT_DWTCID0_CID0_Pos)                    /*!< DWT DWTCID0: CID0 Mask              */

// ---------------------------------------  DWT_DWTCID1  ------------------------------------------
#define DWT_DWTCID1_CID1_Pos                                  0                                                         /*!< DWT DWTCID1: CID1 Position          */
#define DWT_DWTCID1_CID1_Msk                                  (0xffffffffUL << DWT_DWTCID1_CID1_Pos)                    /*!< DWT DWTCID1: CID1 Mask              */

// ---------------------------------------  DWT_DWTCID2  ------------------------------------------
#define DWT_DWTCID2_CID2_Pos                                  0                                                         /*!< DWT DWTCID2: CID2 Position          */
#define DWT_DWTCID2_CID2_Msk                                  (0xffffffffUL << DWT_DWTCID2_CID2_Pos)                    /*!< DWT DWTCID2: CID2 Mask              */

// ---------------------------------------  DWT_DWTCID3  ------------------------------------------
#define DWT_DWTCID3_CID3_Pos                                  0                                                         /*!< DWT DWTCID3: CID3 Position          */
#define DWT_DWTCID3_CID3_Msk                                  (0xffffffffUL << DWT_DWTCID3_CID3_Pos)                    /*!< DWT DWTCID3: CID3 Mask              */

#endif

/***   EBU Bit Fileds *******************/
/***************************************************************************/


/* EBU_CLC  =  EBU Clock Control Register*/
#define   EBU_CLC_DISR_Pos  	(0U)
#define   EBU_CLC_DISR_Msk  	(0x00000001U  << EBU_CLC_DISR_Pos)

#define   EBU_CLC_DISS_Pos  	(1U)
#define   EBU_CLC_DISS_Msk  	(0x00000001U  << EBU_CLC_DISS_Pos)

#define   EBU_CLC_SYNC_Pos  	(16U)
#define   EBU_CLC_SYNC_Msk  	(0x00000001U  << EBU_CLC_SYNC_Pos)

#define   EBU_CLC_DIV2_Pos  	(17U)
#define   EBU_CLC_DIV2_Msk  	(0x00000001U  << EBU_CLC_DIV2_Pos)

#define   EBU_CLC_EBUDIV_Pos  	(18U)
#define   EBU_CLC_EBUDIV_Msk  	(0x00000003U  << EBU_CLC_EBUDIV_Pos)

#define   EBU_CLC_SYNCACK_Pos  	(20U)
#define   EBU_CLC_SYNCACK_Msk  	(0x00000001U  << EBU_CLC_SYNCACK_Pos)

#define   EBU_CLC_DIV2ACK_Pos  	(21U)
#define   EBU_CLC_DIV2ACK_Msk  	(0x00000001U  << EBU_CLC_DIV2ACK_Pos)

#define   EBU_CLC_EBUDIVACK_Pos  	(22U)
#define   EBU_CLC_EBUDIVACK_Msk  	(0x00000003U  << EBU_CLC_EBUDIVACK_Pos)

/* EBU_MODCON  =  EBU Configuration Register*/
#define   EBU_MODCON_STS_Pos  	(0U)
#define   EBU_MODCON_STS_Msk  	(0x00000001U  << EBU_MODCON_STS_Pos)

#define   EBU_MODCON_LCKABRT_Pos  	(1U)
#define   EBU_MODCON_LCKABRT_Msk  	(0x00000001U  << EBU_MODCON_LCKABRT_Pos)

#define   EBU_MODCON_SDTRI_Pos  	(2U)
#define   EBU_MODCON_SDTRI_Msk  	(0x00000001U  << EBU_MODCON_SDTRI_Pos)

#define   EBU_MODCON_EXTLOCK_Pos  	(4U)
#define   EBU_MODCON_EXTLOCK_Msk  	(0x00000001U  << EBU_MODCON_EXTLOCK_Pos)

#define   EBU_MODCON_ARBSYNC_Pos  	(5U)
#define   EBU_MODCON_ARBSYNC_Msk  	(0x00000001U  << EBU_MODCON_ARBSYNC_Pos)

#define   EBU_MODCON_ARBMODE_Pos  	(6U)
#define   EBU_MODCON_ARBMODE_Msk  	(0x00000003U  << EBU_MODCON_ARBMODE_Pos)

#define   EBU_MODCON_TIMEOUTC_Pos  	(8U)
#define   EBU_MODCON_TIMEOUTC_Msk  	(0x000000FFU  << EBU_MODCON_TIMEOUTC_Pos)

#define   EBU_MODCON_LOCKTIMEOUT_Pos  	(16U)
#define   EBU_MODCON_LOCKTIMEOUT_Msk  	(0x000000FFU  << EBU_MODCON_LOCKTIMEOUT_Pos)

#define   EBU_MODCON_GLOBALCS_Pos  	(24U)
#define   EBU_MODCON_GLOBALCS_Msk  	(0x0000000FU  << EBU_MODCON_GLOBALCS_Pos)

#define   EBU_MODCON_ACCSINH_Pos  	(28U)
#define   EBU_MODCON_ACCSINH_Msk  	(0x00000001U  << EBU_MODCON_ACCSINH_Pos)

#define   EBU_MODCON_ACCSINHACK_Pos  	(29U)
#define   EBU_MODCON_ACCSINHACK_Msk  	(0x00000001U  << EBU_MODCON_ACCSINHACK_Pos)

#define   EBU_MODCON_ALE_Pos  	(31U)
#define   EBU_MODCON_ALE_Msk  	(0x00000001U  << EBU_MODCON_ALE_Pos)

/* EBU_ID  =  EBU Module Identification Register*/
#define   EBU_ID_MOD_REV_Pos  	(0U)
#define   EBU_ID_MOD_REV_Msk  	(0x000000FFU  << EBU_ID_MOD_REV_Pos)

#define   EBU_ID_MOD_TYPE_Pos  	(8U)
#define   EBU_ID_MOD_TYPE_Msk  	(0x000000FFU  << EBU_ID_MOD_TYPE_Pos)

#define   EBU_ID_MOD_NUMBER_Pos  	(16U)
#define   EBU_ID_MOD_NUMBER_Msk  	(0x0000FFFFU  << EBU_ID_MOD_NUMBER_Pos)

/* EBU_USERCON  =  EBU Test/Control Configuration Register*/
#define   EBU_USERCON_DIP_Pos  	(0U)
#define   EBU_USERCON_DIP_Msk  	(0x00000001U  << EBU_USERCON_DIP_Pos)

#define   EBU_USERCON_ADDIO_Pos  	(16U)
#define   EBU_USERCON_ADDIO_Msk  	(0x000001FFU  << EBU_USERCON_ADDIO_Pos)

#define   EBU_USERCON_ADVIO_Pos  	(25U)
#define   EBU_USERCON_ADVIO_Msk  	(0x00000001U  << EBU_USERCON_ADVIO_Pos)

/* EBU_CS_ADDRSELx  =  EBU Address Select Register 0*/
#define   EBU_CS_ADDRSEL_REGENAB_Pos  	(0U)
#define   EBU_CS_ADDRSEL_REGENAB_Msk  	(0x00000001U  << EBU_CS_ADDRSEL_REGENAB_Pos)

#define   EBU_CS_ADDRSEL_ALTENAB_Pos  	(1U)
#define   EBU_CS_ADDRSEL_ALTENAB_Msk  	(0x00000001U  << EBU_CS_ADDRSEL_ALTENAB_Pos)

#define   EBU_CS_ADDRSEL_WPROT_Pos  	(2U)
#define   EBU_CS_ADDRSEL_WPROT_Msk  	(0x00000001U  << EBU_CS_ADDRSEL_WPROT_Pos)

/* EBU_CS_BUSRCONx  =  EBU Bus Configuration Register*/
#define   EBU_CS_BUSRCON_FETBLEN_Pos  	(0U)
#define   EBU_CS_BUSRCON_FETBLEN_Msk  	(0x00000007U  << EBU_CS_BUSRCON_FETBLEN_Pos)

#define   EBU_CS_BUSRCON_FBBMSEL_Pos  	(3U)
#define   EBU_CS_BUSRCON_FBBMSEL_Msk  	(0x00000001U  << EBU_CS_BUSRCON_FBBMSEL_Pos)

#define   EBU_CS_BUSRCON_BFSSS_Pos  	(4U)
#define   EBU_CS_BUSRCON_BFSSS_Msk  	(0x00000001U  << EBU_CS_BUSRCON_BFSSS_Pos)

#define   EBU_CS_BUSRCON_FDBKEN_Pos  	(5U)
#define   EBU_CS_BUSRCON_FDBKEN_Msk  	(0x00000001U  << EBU_CS_BUSRCON_FDBKEN_Pos)

#define   EBU_CS_BUSRCON_BFCMSEL_Pos  	(6U)
#define   EBU_CS_BUSRCON_BFCMSEL_Msk  	(0x00000001U  << EBU_CS_BUSRCON_BFCMSEL_Pos)

#define   EBU_CS_BUSRCON_NAA_Pos  	(7U)
#define   EBU_CS_BUSRCON_NAA_Msk  	(0x00000001U  << EBU_CS_BUSRCON_NAA_Pos)

#define   EBU_CS_BUSRCON_ECSE_Pos  	(16U)
#define   EBU_CS_BUSRCON_ECSE_Msk  	(0x00000001U  << EBU_CS_BUSRCON_ECSE_Pos)

#define   EBU_CS_BUSRCON_EBSE_Pos  	(17U)
#define   EBU_CS_BUSRCON_EBSE_Msk  	(0x00000001U  << EBU_CS_BUSRCON_EBSE_Pos)

#define   EBU_CS_BUSRCON_DBA_Pos  	(18U)
#define   EBU_CS_BUSRCON_DBA_Msk  	(0x00000001U  << EBU_CS_BUSRCON_DBA_Pos)

#define   EBU_CS_BUSRCON_WAITINV_Pos  	(19U)
#define   EBU_CS_BUSRCON_WAITINV_Msk  	(0x00000001U  << EBU_CS_BUSRCON_WAITINV_Pos)

#define   EBU_CS_BUSRCON_BCGEN_Pos  	(20U)
#define   EBU_CS_BUSRCON_BCGEN_Msk  	(0x00000003U  << EBU_CS_BUSRCON_BCGEN_Pos)

#define   EBU_CS_BUSRCON_PORTW_Pos  	(22U)
#define   EBU_CS_BUSRCON_PORTW_Msk  	(0x00000003U  << EBU_CS_BUSRCON_PORTW_Pos)

#define   EBU_CS_BUSRCON_WAIT_Pos  	(24U)
#define   EBU_CS_BUSRCON_WAIT_Msk  	(0x00000003U  << EBU_CS_BUSRCON_WAIT_Pos)

#define   EBU_CS_BUSRCON_AAP_Pos  	(26U)
#define   EBU_CS_BUSRCON_AAP_Msk  	(0x00000001U  << EBU_CS_BUSRCON_AAP_Pos)

#define   EBU_CS_BUSRCON_AGEN_Pos  	(28U)
#define   EBU_CS_BUSRCON_AGEN_Msk  	(0x0000000FU  << EBU_CS_BUSRCON_AGEN_Pos)

/* EBU_CS_BUSRAPx  =  EBU Bus Read Access Parameter Register*/
#define   EBU_CS_BUSRAP_RDDTACS_Pos  	(0U)
#define   EBU_CS_BUSRAP_RDDTACS_Msk  	(0x0000000FU  << EBU_CS_BUSRAP_RDDTACS_Pos)

#define   EBU_CS_BUSRAP_RDRECOVC_Pos  	(4U)
#define   EBU_CS_BUSRAP_RDRECOVC_Msk  	(0x00000007U  << EBU_CS_BUSRAP_RDRECOVC_Pos)

#define   EBU_CS_BUSRAP_WAITRDC_Pos  	(7U)
#define   EBU_CS_BUSRAP_WAITRDC_Msk  	(0x0000001FU  << EBU_CS_BUSRAP_WAITRDC_Pos)

#define   EBU_CS_BUSRAP_DATAC_Pos  	(12U)
#define   EBU_CS_BUSRAP_DATAC_Msk  	(0x0000000FU  << EBU_CS_BUSRAP_DATAC_Pos)

#define   EBU_CS_BUSRAP_EXTCLOCK_Pos  	(16U)
#define   EBU_CS_BUSRAP_EXTCLOCK_Msk  	(0x00000003U  << EBU_CS_BUSRAP_EXTCLOCK_Pos)

#define   EBU_CS_BUSRAP_EXTDATA_Pos  	(18U)
#define   EBU_CS_BUSRAP_EXTDATA_Msk  	(0x00000003U  << EBU_CS_BUSRAP_EXTDATA_Pos)

#define   EBU_CS_BUSRAP_CMDDELAY_Pos  	(20U)
#define   EBU_CS_BUSRAP_CMDDELAY_Msk  	(0x0000000FU  << EBU_CS_BUSRAP_CMDDELAY_Pos)

#define   EBU_CS_BUSRAP_AHOLDC_Pos  	(24U)
#define   EBU_CS_BUSRAP_AHOLDC_Msk  	(0x0000000FU  << EBU_CS_BUSRAP_AHOLDC_Pos)

#define   EBU_CS_BUSRAP_ADDRC_Pos  	(28U)
#define   EBU_CS_BUSRAP_ADDRC_Msk  	(0x0000000FU  << EBU_CS_BUSRAP_ADDRC_Pos)

/* EBU_CS_BUSWCONx  =  EBU Bus Write Configuration Register*/
#define   EBU_CS_BUSWCON_FETBLEN_Pos  	(0U)
#define   EBU_CS_BUSWCON_FETBLEN_Msk  	(0x00000007U  << EBU_CS_BUSWCON_FETBLEN_Pos)

#define   EBU_CS_BUSWCON_FBBMSEL_Pos  	(3U)
#define   EBU_CS_BUSWCON_FBBMSEL_Msk  	(0x00000001U  << EBU_CS_BUSWCON_FBBMSEL_Pos)

#define   EBU_CS_BUSWCON_NAA_Pos  	(7U)
#define   EBU_CS_BUSWCON_NAA_Msk  	(0x00000001U  << EBU_CS_BUSWCON_NAA_Pos)

#define   EBU_CS_BUSWCON_ECSE_Pos  	(16U)
#define   EBU_CS_BUSWCON_ECSE_Msk  	(0x00000001U  << EBU_CS_BUSWCON_ECSE_Pos)

#define   EBU_CS_BUSWCON_EBSE_Pos  	(17U)
#define   EBU_CS_BUSWCON_EBSE_Msk  	(0x00000001U  << EBU_CS_BUSWCON_EBSE_Pos)

#define   EBU_CS_BUSWCON_WAITINV_Pos  	(19U)
#define   EBU_CS_BUSWCON_WAITINV_Msk  	(0x00000001U  << EBU_CS_BUSWCON_WAITINV_Pos)

#define   EBU_CS_BUSWCON_BCGEN_Pos  	(20U)
#define   EBU_CS_BUSWCON_BCGEN_Msk  	(0x00000003U  << EBU_CS_BUSWCON_BCGEN_Pos)

#define   EBU_CS_BUSWCON_PORTW_Pos  	(22U)
#define   EBU_CS_BUSWCON_PORTW_Msk  	(0x00000003U  << EBU_CS_BUSWCON_PORTW_Pos)

#define   EBU_CS_BUSWCON_WAIT_Pos  	(24U)
#define   EBU_CS_BUSWCON_WAIT_Msk  	(0x00000003U  << EBU_CS_BUSWCON_WAIT_Pos)

#define   EBU_CS_BUSWCON_AAP_Pos  	(26U)
#define   EBU_CS_BUSWCON_AAP_Msk  	(0x00000001U  << EBU_CS_BUSWCON_AAP_Pos)

#define   EBU_CS_BUSWCON_LOCKCS_Pos  	(27U)
#define   EBU_CS_BUSWCON_LOCKCS_Msk  	(0x00000001U  << EBU_CS_BUSWCON_LOCKCS_Pos)

#define   EBU_CS_BUSWCON_AGEN_Pos  	(28U)
#define   EBU_CS_BUSWCON_AGEN_Msk  	(0x0000000FU  << EBU_CS_BUSWCON_AGEN_Pos)

/* EBU_CS_BUSWAPx  =  EBU Bus Write Access Parameter Register*/
#define   EBU_CS_BUSWAP_WRDTACS_Pos  	(0U)
#define   EBU_CS_BUSWAP_WRDTACS_Msk  	(0x0000000FU  << EBU_CS_BUSWAP_WRDTACS_Pos)

#define   EBU_CS_BUSWAP_WRRECOVC_Pos  	(4U)
#define   EBU_CS_BUSWAP_WRRECOVC_Msk  	(0x00000007U  << EBU_CS_BUSWAP_WRRECOVC_Pos)

#define   EBU_CS_BUSWAP_WAITWRC_Pos  	(7U)
#define   EBU_CS_BUSWAP_WAITWRC_Msk  	(0x0000001FU  << EBU_CS_BUSWAP_WAITWRC_Pos)

#define   EBU_CS_BUSWAP_DATAC_Pos  	(12U)
#define   EBU_CS_BUSWAP_DATAC_Msk  	(0x0000000FU  << EBU_CS_BUSWAP_DATAC_Pos)

#define   EBU_CS_BUSWAP_EXTCLOCK_Pos  	(16U)
#define   EBU_CS_BUSWAP_EXTCLOCK_Msk  	(0x00000003U  << EBU_CS_BUSWAP_EXTCLOCK_Pos)

#define   EBU_CS_BUSWAP_EXTDATA_Pos  	(18U)
#define   EBU_CS_BUSWAP_EXTDATA_Msk  	(0x00000003U  << EBU_CS_BUSWAP_EXTDATA_Pos)

#define   EBU_CS_BUSWAP_CMDDELAY_Pos  	(20U)
#define   EBU_CS_BUSWAP_CMDDELAY_Msk  	(0x0000000FU  << EBU_CS_BUSWAP_CMDDELAY_Pos)

#define   EBU_CS_BUSWAP_AHOLDC_Pos  	(24U)
#define   EBU_CS_BUSWAP_AHOLDC_Msk  	(0x0000000FU  << EBU_CS_BUSWAP_AHOLDC_Pos)

#define   EBU_CS_BUSWAP_ADDRC_Pos  	(28U)
#define   EBU_CS_BUSWAP_ADDRC_Msk  	(0x0000000FU  << EBU_CS_BUSWAP_ADDRC_Pos)

/* EBU_SDRAM_SDRMCON  =  EBU SDRAM Control Register*/
#define   EBU_SDRAM_SDRMCON_SDCMSEL_Pos  	(31U)
#define   EBU_SDRAM_SDRMCON_SDCMSEL_Msk  	(0x00000001U  << EBU_SDRAM_SDRMCON_SDCMSEL_Pos)

#define   EBU_SDRAM_SDRMCON_PWR_MODE_Pos  	(29U)
#define   EBU_SDRAM_SDRMCON_PWR_MODE_Msk  	(0x00000003U  << EBU_SDRAM_SDRMCON_PWR_MODE_Pos)

#define   EBU_SDRAM_SDRMCON_CLKDIS_Pos  	(28U)
#define   EBU_SDRAM_SDRMCON_CLKDIS_Msk  	(0x00000001U  << EBU_SDRAM_SDRMCON_CLKDIS_Pos)

#define   EBU_SDRAM_SDRMCON_CRCE_Pos  	(25U)
#define   EBU_SDRAM_SDRMCON_CRCE_Msk  	(0x00000007U  << EBU_SDRAM_SDRMCON_CRCE_Pos)

#define   EBU_SDRAM_SDRMCON_BANKM_Pos  	(22U)
#define   EBU_SDRAM_SDRMCON_BANKM_Msk  	(0x00000007U  << EBU_SDRAM_SDRMCON_BANKM_Pos)

#define   EBU_SDRAM_SDRMCON_ROWM_Pos  	(19U)
#define   EBU_SDRAM_SDRMCON_ROWM_Msk  	(0x00000007U  << EBU_SDRAM_SDRMCON_ROWM_Pos)

#define   EBU_SDRAM_SDRMCON_CRC_Pos  	(16U)
#define   EBU_SDRAM_SDRMCON_CRC_Msk  	(0x00000007U  << EBU_SDRAM_SDRMCON_CRC_Pos)

#define   EBU_SDRAM_SDRMCON_CRCD_Pos  	(14U)
#define   EBU_SDRAM_SDRMCON_CRCD_Msk  	(0x00000003U  << EBU_SDRAM_SDRMCON_CRCD_Pos)

#define   EBU_SDRAM_SDRMCON_AWIDTH_Pos  	(12U)
#define   EBU_SDRAM_SDRMCON_AWIDTH_Msk  	(0x00000003U  << EBU_SDRAM_SDRMCON_AWIDTH_Pos)

#define   EBU_SDRAM_SDRMCON_CRP_Pos  	(10U)
#define   EBU_SDRAM_SDRMCON_CRP_Msk  	(0x00000003U  << EBU_SDRAM_SDRMCON_CRP_Pos)

#define   EBU_SDRAM_SDRMCON_CRSC_Pos  	(8U)
#define   EBU_SDRAM_SDRMCON_CRSC_Msk  	(0x00000003U  << EBU_SDRAM_SDRMCON_CRSC_Pos)

#define   EBU_SDRAM_SDRMCON_CRFSH_Pos  	(4U)
#define   EBU_SDRAM_SDRMCON_CRFSH_Msk  	(0x0000000FU  << EBU_SDRAM_SDRMCON_CRFSH_Pos)

#define   EBU_SDRAM_SDRMCON_CRAS_Pos  	(0U)
#define   EBU_SDRAM_SDRMCON_CRAS_Msk  	(0x0000000FU  << EBU_SDRAM_SDRMCON_CRAS_Pos)

/* EBU_SDRAM_SDRMOD  =  EBU SDRAM Mode Register*/
#define   EBU_SDRAM_SDRMOD_XBA_Pos  	(28U)
#define   EBU_SDRAM_SDRMOD_XBA_Msk  	(0x0000000FU  << EBU_SDRAM_SDRMOD_XBA_Pos)

#define   EBU_SDRAM_SDRMOD_XOPM_Pos  	(16U)
#define   EBU_SDRAM_SDRMOD_XOPM_Msk  	(0x00000FFFU  << EBU_SDRAM_SDRMOD_XOPM_Pos)

#define   EBU_SDRAM_SDRMOD_COLDSTART_Pos  	(15U)
#define   EBU_SDRAM_SDRMOD_COLDSTART_Msk  	(0x00000001U  << EBU_SDRAM_SDRMOD_COLDSTART_Pos)

#define   EBU_SDRAM_SDRMOD_OPMODE_Pos  	(7U)
#define   EBU_SDRAM_SDRMOD_OPMODE_Msk  	(0x0000007FU  << EBU_SDRAM_SDRMOD_OPMODE_Pos)

#define   EBU_SDRAM_SDRMOD_CASLAT_Pos  	(4U)
#define   EBU_SDRAM_SDRMOD_CASLAT_Msk  	(0x00000007U  << EBU_SDRAM_SDRMOD_CASLAT_Pos)

#define   EBU_SDRAM_SDRMOD_BTYP_Pos  	(3U)
#define   EBU_SDRAM_SDRMOD_BTYP_Msk  	(0x00000001U  << EBU_SDRAM_SDRMOD_BTYP_Pos)

#define   EBU_SDRAM_SDRMOD_BURSTL_Pos  	(0U)
#define   EBU_SDRAM_SDRMOD_BURSTL_Msk  	(0x00000007U  << EBU_SDRAM_SDRMOD_BURSTL_Pos)

/* EBU_SDRAM_SDRMREF  =  EBU SDRAM Refresh Control Register*/
#define   EBU_SDRAM_SDRMREF_RES_DLY_Pos  	(25U)
#define   EBU_SDRAM_SDRMREF_RES_DLY_Msk  	(0x00000007U  << EBU_SDRAM_SDRMREF_RES_DLY_Pos)

#define   EBU_SDRAM_SDRMREF_ARFSH_Pos  	(24U)
#define   EBU_SDRAM_SDRMREF_ARFSH_Msk  	(0x00000001U  << EBU_SDRAM_SDRMREF_ARFSH_Pos)

#define   EBU_SDRAM_SDRMREF_SELFREX_DLY_Pos  	(16U)
#define   EBU_SDRAM_SDRMREF_SELFREX_DLY_Msk  	(0x000000FFU  << EBU_SDRAM_SDRMREF_SELFREX_DLY_Pos)

#define   EBU_SDRAM_SDRMREF_ERFSHC_Pos  	(14U)
#define   EBU_SDRAM_SDRMREF_ERFSHC_Msk  	(0x00000003U  << EBU_SDRAM_SDRMREF_ERFSHC_Pos)

#define   EBU_SDRAM_SDRMREF_AUTOSELFR_Pos  	(13U)
#define   EBU_SDRAM_SDRMREF_AUTOSELFR_Msk  	(0x00000001U  << EBU_SDRAM_SDRMREF_AUTOSELFR_Pos)

#define   EBU_SDRAM_SDRMREF_SELFREN_Pos  	(12U)
#define   EBU_SDRAM_SDRMREF_SELFREN_Msk  	(0x00000001U  << EBU_SDRAM_SDRMREF_SELFREN_Pos)

#define   EBU_SDRAM_SDRMREF_SELFRENST_Pos  	(11U)
#define   EBU_SDRAM_SDRMREF_SELFRENST_Msk  	(0x00000001U  << EBU_SDRAM_SDRMREF_SELFRENST_Pos)

#define   EBU_SDRAM_SDRMREF_SELFREX_Pos  	(10U)
#define   EBU_SDRAM_SDRMREF_SELFREX_Msk  	(0x00000001U  << EBU_SDRAM_SDRMREF_SELFREX_Pos)

#define   EBU_SDRAM_SDRMREF_SELFREXST_Pos  	(9U)
#define   EBU_SDRAM_SDRMREF_SELFREXST_Msk  	(0x00000001U  << EBU_SDRAM_SDRMREF_SELFREXST_Pos)

#define   EBU_SDRAM_SDRMREF_REFRESHR_Pos  	(6U)
#define   EBU_SDRAM_SDRMREF_REFRESHR_Msk  	(0x00000007U  << EBU_SDRAM_SDRMREF_REFRESHR_Pos)

#define   EBU_SDRAM_SDRMREF_REFRESHC_Pos  	(0U)
#define   EBU_SDRAM_SDRMREF_REFRESHC_Msk  	(0x0000003FU  << EBU_SDRAM_SDRMREF_REFRESHC_Pos)

/* EBU_SDRAM_SDRSTAT  =  EBU SDRAM Status Register*/
#define   EBU_SDRAM_SDRSTAT_SDERR_Pos  	(2U)
#define   EBU_SDRAM_SDRSTAT_SDERR_Msk  	(0x00000001U  << EBU_SDRAM_SDRSTAT_SDERR_Pos)

#define   EBU_SDRAM_SDRSTAT_SDRMBUSY_Pos  	(1U)
#define   EBU_SDRAM_SDRSTAT_SDRMBUSY_Msk  	(0x00000001U  << EBU_SDRAM_SDRSTAT_SDRMBUSY_Pos)

#define   EBU_SDRAM_SDRSTAT_REFERR_Pos  	(0U)
#define   EBU_SDRAM_SDRSTAT_REFERR_Msk  	(0x00000001U  << EBU_SDRAM_SDRSTAT_REFERR_Pos)



/***   ERUx Bit Fileds *******************/
/***************************************************************************/


/* ERU_EXISEL  =  Event Input Select Register 0*/
#define   ERU_EXISEL_EXS0A_Pos  	(0U)
#define   ERU_EXISEL_EXS0A_Msk  	(0x00000003U  << ERU_EXISEL_EXS0A_Pos)

#define   ERU_EXISEL_EXS0B_Pos  	(2U)
#define   ERU_EXISEL_EXS0B_Msk  	(0x00000003U  << ERU_EXISEL_EXS0B_Pos)

#define   ERU_EXISEL_EXS1A_Pos  	(4U)
#define   ERU_EXISEL_EXS1A_Msk  	(0x00000003U  << ERU_EXISEL_EXS1A_Pos)

#define   ERU_EXISEL_EXS1B_Pos  	(6U)
#define   ERU_EXISEL_EXS1B_Msk  	(0x00000003U  << ERU_EXISEL_EXS1B_Pos)

#define   ERU_EXISEL_EXS2A_Pos  	(8U)
#define   ERU_EXISEL_EXS2A_Msk  	(0x00000003U  << ERU_EXISEL_EXS2A_Pos)

#define   ERU_EXISEL_EXS2B_Pos  	(10U)
#define   ERU_EXISEL_EXS2B_Msk  	(0x00000003U  << ERU_EXISEL_EXS2B_Pos)

#define   ERU_EXISEL_EXS3A_Pos  	(12U)
#define   ERU_EXISEL_EXS3A_Msk  	(0x00000003U  << ERU_EXISEL_EXS3A_Pos)

#define   ERU_EXISEL_EXS3B_Pos  	(14U)
#define   ERU_EXISEL_EXS3B_Msk  	(0x00000003U  << ERU_EXISEL_EXS3B_Pos)

/* ERU_EXICON0  =  Event Input Control x Register*/
#define   ERU_EXICON_PE_Pos  	(0U)
#define   ERU_EXICON_PE_Msk  	(0x00000001U  << ERU_EXICON_PE_Pos)

#define   ERU_EXICON_LD_Pos  	(1U)
#define   ERU_EXICON_LD_Msk  	(0x00000001U  << ERU_EXICON_LD_Pos)

#define   ERU_EXICON_RE_Pos  	(2U)
#define   ERU_EXICON_RE_Msk  	(0x00000001U  << ERU_EXICON_RE_Pos)

#define   ERU_EXICON_FE_Pos  	(3U)
#define   ERU_EXICON_FE_Msk  	(0x00000001U  << ERU_EXICON_FE_Pos)

#define   ERU_EXICON_OCS_Pos  	(4U)
#define   ERU_EXICON_OCS_Msk  	(0x00000007U  << ERU_EXICON_OCS_Pos)

#define   ERU_EXICON_FL_Pos  	(7U)
#define   ERU_EXICON_FL_Msk  	(0x00000001U  << ERU_EXICON_FL_Pos)

#define   ERU_EXICON_SS_Pos  	(8U)
#define   ERU_EXICON_SS_Msk  	(0x00000003U  << ERU_EXICON_SS_Pos)

#define   ERU_EXICON_NA_Pos  	(10U)
#define   ERU_EXICON_NA_Msk  	(0x00000001U  << ERU_EXICON_NA_Pos)

#define   ERU_EXICON_NB_Pos  	(11U)
#define   ERU_EXICON_NB_Msk  	(0x00000001U  << ERU_EXICON_NB_Pos)


/* ERU_EXOCON0  =  Event Output Trigger Control x Register*/
#define   ERU_EXOCON_ISS_Pos  	(0U)
#define   ERU_EXOCON_ISS_Msk  	(0x00000003U  << ERU_EXOCON_ISS_Pos)

#define   ERU_EXOCON_GEEN_Pos  	(2U)
#define   ERU_EXOCON_GEEN_Msk  	(0x00000001U  << ERU_EXOCON_GEEN_Pos)

#define   ERU_EXOCON_PDR_Pos  	(3U)
#define   ERU_EXOCON_PDR_Msk  	(0x00000001U  << ERU_EXOCON_PDR_Pos)

#define   ERU_EXOCON_GP_Pos  	(4U)
#define   ERU_EXOCON_GP_Msk  	(0x00000003U  << ERU_EXOCON_GP_Pos)

#define   ERU_EXOCON_IPEN0_Pos  	(12U)
#define   ERU_EXOCON_IPEN0_Msk  	(0x00000001U  << ERU_EXOCON_IPEN0_Pos)

#define   ERU_EXOCON_IPEN1_Pos  	(13U)
#define   ERU_EXOCON_IPEN1_Msk  	(0x00000001U  << ERU_EXOCON_IPEN1_Pos)

#define   ERU_EXOCON_IPEN2_Pos  	(14U)
#define   ERU_EXOCON_IPEN2_Msk  	(0x00000001U  << ERU_EXOCON_IPEN2_Pos)

#define   ERU_EXOCON_IPEN3_Pos  	(15U)
#define   ERU_EXOCON_IPEN3_Msk  	(0x00000001U  << ERU_EXOCON_IPEN3_Pos)


/***   ETHx Bit Fileds *******************/
/***************************************************************************/


/* ETH_MAC_CONFIGURATION  =  Register x [MAC Configuration Register]*/
#define   ETH_MAC_CONFIGURATION_PRELEN_Pos  	(0U)
#define   ETH_MAC_CONFIGURATION_PRELEN_Msk  	(0x00000003U  << ETH_MAC_CONFIGURATION_PRELEN_Pos)

#define   ETH_MAC_CONFIGURATION_RE_Pos  	(2U)
#define   ETH_MAC_CONFIGURATION_RE_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_RE_Pos)

#define   ETH_MAC_CONFIGURATION_TE_Pos  	(3U)
#define   ETH_MAC_CONFIGURATION_TE_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_TE_Pos)

#define   ETH_MAC_CONFIGURATION_DC_Pos  	(4U)
#define   ETH_MAC_CONFIGURATION_DC_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_DC_Pos)

#define   ETH_MAC_CONFIGURATION_BL_Pos  	(5U)
#define   ETH_MAC_CONFIGURATION_BL_Msk  	(0x00000003U  << ETH_MAC_CONFIGURATION_BL_Pos)

#define   ETH_MAC_CONFIGURATION_ACS_Pos  	(7U)
#define   ETH_MAC_CONFIGURATION_ACS_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_ACS_Pos)

#define   ETH_MAC_CONFIGURATION_LUD_Pos  	(8U)
#define   ETH_MAC_CONFIGURATION_LUD_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_LUD_Pos)

#define   ETH_MAC_CONFIGURATION_DR_Pos  	(9U)
#define   ETH_MAC_CONFIGURATION_DR_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_DR_Pos)

#define   ETH_MAC_CONFIGURATION_IPC_Pos  	(10U)
#define   ETH_MAC_CONFIGURATION_IPC_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_IPC_Pos)

#define   ETH_MAC_CONFIGURATION_DM_Pos  	(11U)
#define   ETH_MAC_CONFIGURATION_DM_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_DM_Pos)

#define   ETH_MAC_CONFIGURATION_LM_Pos  	(12U)
#define   ETH_MAC_CONFIGURATION_LM_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_LM_Pos)

#define   ETH_MAC_CONFIGURATION_DO_Pos  	(13U)
#define   ETH_MAC_CONFIGURATION_DO_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_DO_Pos)

#define   ETH_MAC_CONFIGURATION_FES_Pos  	(14U)
#define   ETH_MAC_CONFIGURATION_FES_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_FES_Pos)

#define   ETH_MAC_CONFIGURATION_Reserved_Pos  	(15U)
#define   ETH_MAC_CONFIGURATION_Reserved_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_Reserved_Pos)

#define   ETH_MAC_CONFIGURATION_DCRS_Pos  	(16U)
#define   ETH_MAC_CONFIGURATION_DCRS_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_DCRS_Pos)

#define   ETH_MAC_CONFIGURATION_IFG_Pos  	(17U)
#define   ETH_MAC_CONFIGURATION_IFG_Msk  	(0x00000007U  << ETH_MAC_CONFIGURATION_IFG_Pos)

#define   ETH_MAC_CONFIGURATION_JE_Pos  	(20U)
#define   ETH_MAC_CONFIGURATION_JE_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_JE_Pos)

#define   ETH_MAC_CONFIGURATION_BE_Pos  	(21U)
#define   ETH_MAC_CONFIGURATION_BE_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_BE_Pos)

#define   ETH_MAC_CONFIGURATION_JD_Pos  	(22U)
#define   ETH_MAC_CONFIGURATION_JD_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_JD_Pos)

#define   ETH_MAC_CONFIGURATION_WD_Pos  	(23U)
#define   ETH_MAC_CONFIGURATION_WD_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_WD_Pos)

#define   ETH_MAC_CONFIGURATION_TC_Pos  	(24U)
#define   ETH_MAC_CONFIGURATION_TC_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_TC_Pos)

#define   ETH_MAC_CONFIGURATION_CST_Pos  	(25U)
#define   ETH_MAC_CONFIGURATION_CST_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_CST_Pos)

#define   ETH_MAC_CONFIGURATION_Reserved_26_Pos  	(26U)
#define   ETH_MAC_CONFIGURATION_Reserved_26_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_Reserved_26_Pos)

#define   ETH_MAC_CONFIGURATION_TWOKPE_Pos  	(27U)
#define   ETH_MAC_CONFIGURATION_TWOKPE_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_TWOKPE_Pos)

#define   ETH_MAC_CONFIGURATION_SARC_Pos  	(28U)
#define   ETH_MAC_CONFIGURATION_SARC_Msk  	(0x00000007U  << ETH_MAC_CONFIGURATION_SARC_Pos)

#define   ETH_MAC_CONFIGURATION_Reserved_31_Pos  	(31U)
#define   ETH_MAC_CONFIGURATION_Reserved_31_Msk  	(0x00000001U  << ETH_MAC_CONFIGURATION_Reserved_31_Pos)

/* ETH_MAC_FRAME_FILTER  =  Register 1 [MAC Frame Filter]*/
#define   ETH_MAC_FRAME_FILTER_PR_Pos  	(0U)
#define   ETH_MAC_FRAME_FILTER_PR_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_PR_Pos)

#define   ETH_MAC_FRAME_FILTER_HUC_Pos  	(1U)
#define   ETH_MAC_FRAME_FILTER_HUC_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_HUC_Pos)

#define   ETH_MAC_FRAME_FILTER_HMC_Pos  	(2U)
#define   ETH_MAC_FRAME_FILTER_HMC_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_HMC_Pos)

#define   ETH_MAC_FRAME_FILTER_DAIF_Pos  	(3U)
#define   ETH_MAC_FRAME_FILTER_DAIF_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_DAIF_Pos)

#define   ETH_MAC_FRAME_FILTER_PM_Pos  	(4U)
#define   ETH_MAC_FRAME_FILTER_PM_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_PM_Pos)

#define   ETH_MAC_FRAME_FILTER_DBF_Pos  	(5U)
#define   ETH_MAC_FRAME_FILTER_DBF_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_DBF_Pos)

#define   ETH_MAC_FRAME_FILTER_PCF_Pos  	(6U)
#define   ETH_MAC_FRAME_FILTER_PCF_Msk  	(0x00000003U  << ETH_MAC_FRAME_FILTER_PCF_Pos)

#define   ETH_MAC_FRAME_FILTER_SAIF_Pos  	(8U)
#define   ETH_MAC_FRAME_FILTER_SAIF_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_SAIF_Pos)

#define   ETH_MAC_FRAME_FILTER_SAF_Pos  	(9U)
#define   ETH_MAC_FRAME_FILTER_SAF_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_SAF_Pos)

#define   ETH_MAC_FRAME_FILTER_HPF_Pos  	(10U)
#define   ETH_MAC_FRAME_FILTER_HPF_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_HPF_Pos)

#define   ETH_MAC_FRAME_FILTER_Reserved_15_11_Pos  	(11U)
#define   ETH_MAC_FRAME_FILTER_Reserved_15_11_Msk  	(0x0000001FU  << ETH_MAC_FRAME_FILTER_Reserved_15_11_Pos)

#define   ETH_MAC_FRAME_FILTER_VTFE_Pos  	(16U)
#define   ETH_MAC_FRAME_FILTER_VTFE_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_VTFE_Pos)

#define   ETH_MAC_FRAME_FILTER_Reserved_19_17_Pos  	(17U)
#define   ETH_MAC_FRAME_FILTER_Reserved_19_17_Msk  	(0x00000007U  << ETH_MAC_FRAME_FILTER_Reserved_19_17_Pos)

#define   ETH_MAC_FRAME_FILTER_IPFE_Pos  	(20U)
#define   ETH_MAC_FRAME_FILTER_IPFE_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_IPFE_Pos)

#define   ETH_MAC_FRAME_FILTER_DNTU_Pos  	(21U)
#define   ETH_MAC_FRAME_FILTER_DNTU_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_DNTU_Pos)

#define   ETH_MAC_FRAME_FILTER_Reserved_30_22_Pos  	(22U)
#define   ETH_MAC_FRAME_FILTER_Reserved_30_22_Msk  	(0x000001FFU  << ETH_MAC_FRAME_FILTER_Reserved_30_22_Pos)

#define   ETH_MAC_FRAME_FILTER_RA_Pos  	(31U)
#define   ETH_MAC_FRAME_FILTER_RA_Msk  	(0x00000001U  << ETH_MAC_FRAME_FILTER_RA_Pos)

/* ETH_HASH_TABLE_HIGH  =  Register 2 [Hash Table High Register]*/
#define   ETH_HASH_TABLE_HIGH_HTH_Pos  	(0U)
#define   ETH_HASH_TABLE_HIGH_HTH_Msk  	(0xFFFFFFFFU  << ETH_HASH_TABLE_HIGH_HTH_Pos)

/* ETH_HASH_TABLE_LOW  =  Register 3 [Hash Table Low Register]*/
#define   ETH_HASH_TABLE_LOW_HTL_Pos  	(0U)
#define   ETH_HASH_TABLE_LOW_HTL_Msk  	(0xFFFFFFFFU  << ETH_HASH_TABLE_LOW_HTL_Pos)

/* ETH_GMII_ADDRESS  =  Register 4 [MII Address Register)*/
#define   ETH_GMII_ADDRESS_MB_Pos  	(0U)
#define   ETH_GMII_ADDRESS_MB_Msk  	(0x00000001U  << ETH_GMII_ADDRESS_MB_Pos)

#define   ETH_GMII_ADDRESS_MW_Pos  	(1U)
#define   ETH_GMII_ADDRESS_MW_Msk  	(0x00000001U  << ETH_GMII_ADDRESS_MW_Pos)

#define   ETH_GMII_ADDRESS_CR_Pos  	(2U)
#define   ETH_GMII_ADDRESS_CR_Msk  	(0x0000000FU  << ETH_GMII_ADDRESS_CR_Pos)

#define   ETH_GMII_ADDRESS_MR_Pos  	(6U)
#define   ETH_GMII_ADDRESS_MR_Msk  	(0x0000001FU  << ETH_GMII_ADDRESS_MR_Pos)

#define   ETH_GMII_ADDRESS_PA_Pos  	(11U)
#define   ETH_GMII_ADDRESS_PA_Msk  	(0x0000001FU  << ETH_GMII_ADDRESS_PA_Pos)

#define   ETH_GMII_ADDRESS_Reserved_31_16_Pos  	(16U)
#define   ETH_GMII_ADDRESS_Reserved_31_16_Msk  	(0x0000FFFFU  << ETH_GMII_ADDRESS_Reserved_31_16_Pos)

/* ETH_GMII_DATA  =  Register 5 [MII Data Register]*/
#define   ETH_GMII_DATA_MD_Pos  	(0U)
#define   ETH_GMII_DATA_MD_Msk  	(0x0000FFFFU  << ETH_GMII_DATA_MD_Pos)

#define   ETH_GMII_DATA_Reserved_31_16_Pos  	(16U)
#define   ETH_GMII_DATA_Reserved_31_16_Msk  	(0x0000FFFFU  << ETH_GMII_DATA_Reserved_31_16_Pos)

/* ETH_FLOW_CONTROL  =  Register 6 [Flow Control Register]*/
#define   ETH_FLOW_CONTROL_FCA_BPA_Pos  	(0U)
#define   ETH_FLOW_CONTROL_FCA_BPA_Msk  	(0x00000001U  << ETH_FLOW_CONTROL_FCA_BPA_Pos)

#define   ETH_FLOW_CONTROL_TFE_Pos  	(1U)
#define   ETH_FLOW_CONTROL_TFE_Msk  	(0x00000001U  << ETH_FLOW_CONTROL_TFE_Pos)

#define   ETH_FLOW_CONTROL_RFE_Pos  	(2U)
#define   ETH_FLOW_CONTROL_RFE_Msk  	(0x00000001U  << ETH_FLOW_CONTROL_RFE_Pos)

#define   ETH_FLOW_CONTROL_UP_Pos  	(3U)
#define   ETH_FLOW_CONTROL_UP_Msk  	(0x00000001U  << ETH_FLOW_CONTROL_UP_Pos)

#define   ETH_FLOW_CONTROL_PLT_Pos  	(4U)
#define   ETH_FLOW_CONTROL_PLT_Msk  	(0x00000003U  << ETH_FLOW_CONTROL_PLT_Pos)

#define   ETH_FLOW_CONTROL_Reserved_6_Pos  	(6U)
#define   ETH_FLOW_CONTROL_Reserved_6_Msk  	(0x00000001U  << ETH_FLOW_CONTROL_Reserved_6_Pos)

#define   ETH_FLOW_CONTROL_DZPQ_Pos  	(7U)
#define   ETH_FLOW_CONTROL_DZPQ_Msk  	(0x00000001U  << ETH_FLOW_CONTROL_DZPQ_Pos)

#define   ETH_FLOW_CONTROL_Reserved_15_8_Pos  	(8U)
#define   ETH_FLOW_CONTROL_Reserved_15_8_Msk  	(0x000000FFU  << ETH_FLOW_CONTROL_Reserved_15_8_Pos)

#define   ETH_FLOW_CONTROL_PT_Pos  	(16U)
#define   ETH_FLOW_CONTROL_PT_Msk  	(0x0000FFFFU  << ETH_FLOW_CONTROL_PT_Pos)

/* ETH_VLAN_TAG  =  Register 7 [VLAN Tag Register]*/
#define   ETH_VLAN_TAG_VL_Pos  	(0U)
#define   ETH_VLAN_TAG_VL_Msk  	(0x0000FFFFU  << ETH_VLAN_TAG_VL_Pos)

#define   ETH_VLAN_TAG_ETV_Pos  	(16U)
#define   ETH_VLAN_TAG_ETV_Msk  	(0x00000001U  << ETH_VLAN_TAG_ETV_Pos)

#define   ETH_VLAN_TAG_VTIM_Pos  	(17U)
#define   ETH_VLAN_TAG_VTIM_Msk  	(0x00000001U  << ETH_VLAN_TAG_VTIM_Pos)

#define   ETH_VLAN_TAG_ESVL_Pos  	(18U)
#define   ETH_VLAN_TAG_ESVL_Msk  	(0x00000001U  << ETH_VLAN_TAG_ESVL_Pos)

#define   ETH_VLAN_TAG_VTHM_Pos  	(19U)
#define   ETH_VLAN_TAG_VTHM_Msk  	(0x00000001U  << ETH_VLAN_TAG_VTHM_Pos)

#define   ETH_VLAN_TAG_Reserved_31_20_Pos  	(20U)
#define   ETH_VLAN_TAG_Reserved_31_20_Msk  	(0x00000FFFU  << ETH_VLAN_TAG_Reserved_31_20_Pos)

/* ETH_VERSION  =  Register 8 [Version Register]*/
#define   ETH_VERSION_SNPSVER_Pos  	(0U)
#define   ETH_VERSION_SNPSVER_Msk  	(0x000000FFU  << ETH_VERSION_SNPSVER_Pos)

#define   ETH_VERSION_USERVER_Pos  	(8U)
#define   ETH_VERSION_USERVER_Msk  	(0x000000FFU  << ETH_VERSION_USERVER_Pos)

#define   ETH_VERSION_Reserved_31_16_Pos  	(16U)
#define   ETH_VERSION_Reserved_31_16_Msk  	(0x0000FFFFU  << ETH_VERSION_Reserved_31_16_Pos)

/* ETH_DEBUG  =  Register 9 [Debug Register]*/
#define   ETH_DEBUG_RPESTS_Pos  	(0U)
#define   ETH_DEBUG_RPESTS_Msk  	(0x00000001U  << ETH_DEBUG_RPESTS_Pos)

#define   ETH_DEBUG_RFCFCSTS_Pos  	(1U)
#define   ETH_DEBUG_RFCFCSTS_Msk  	(0x00000003U  << ETH_DEBUG_RFCFCSTS_Pos)

#define   ETH_DEBUG_Reserved_3_Pos  	(3U)
#define   ETH_DEBUG_Reserved_3_Msk  	(0x00000001U  << ETH_DEBUG_Reserved_3_Pos)

#define   ETH_DEBUG_RWCSTS_Pos  	(4U)
#define   ETH_DEBUG_RWCSTS_Msk  	(0x00000001U  << ETH_DEBUG_RWCSTS_Pos)

#define   ETH_DEBUG_RRCSTS_Pos  	(5U)
#define   ETH_DEBUG_RRCSTS_Msk  	(0x00000003U  << ETH_DEBUG_RRCSTS_Pos)

#define   ETH_DEBUG_Reserved_7_Pos  	(7U)
#define   ETH_DEBUG_Reserved_7_Msk  	(0x00000001U  << ETH_DEBUG_Reserved_7_Pos)

#define   ETH_DEBUG_RXFSTS_Pos  	(8U)
#define   ETH_DEBUG_RXFSTS_Msk  	(0x00000003U  << ETH_DEBUG_RXFSTS_Pos)

#define   ETH_DEBUG_Reserved_15_10_Pos  	(10U)
#define   ETH_DEBUG_Reserved_15_10_Msk  	(0x0000003FU  << ETH_DEBUG_Reserved_15_10_Pos)

#define   ETH_DEBUG_TPESTS_Pos  	(16U)
#define   ETH_DEBUG_TPESTS_Msk  	(0x00000001U  << ETH_DEBUG_TPESTS_Pos)

#define   ETH_DEBUG_TFCSTS_Pos  	(17U)
#define   ETH_DEBUG_TFCSTS_Msk  	(0x00000003U  << ETH_DEBUG_TFCSTS_Pos)

#define   ETH_DEBUG_TXPAUSED_Pos  	(19U)
#define   ETH_DEBUG_TXPAUSED_Msk  	(0x00000001U  << ETH_DEBUG_TXPAUSED_Pos)

#define   ETH_DEBUG_TRCSTS_Pos  	(20U)
#define   ETH_DEBUG_TRCSTS_Msk  	(0x00000003U  << ETH_DEBUG_TRCSTS_Pos)

#define   ETH_DEBUG_TWCSTS_Pos  	(22U)
#define   ETH_DEBUG_TWCSTS_Msk  	(0x00000001U  << ETH_DEBUG_TWCSTS_Pos)

#define   ETH_DEBUG_Reserved_23_Pos  	(23U)
#define   ETH_DEBUG_Reserved_23_Msk  	(0x00000001U  << ETH_DEBUG_Reserved_23_Pos)

#define   ETH_DEBUG_TXFSTS_Pos  	(24U)
#define   ETH_DEBUG_TXFSTS_Msk  	(0x00000001U  << ETH_DEBUG_TXFSTS_Pos)

#define   ETH_DEBUG_TXSTSFSTS_Pos  	(25U)
#define   ETH_DEBUG_TXSTSFSTS_Msk  	(0x00000001U  << ETH_DEBUG_TXSTSFSTS_Pos)

#define   ETH_DEBUG_Reserved_31_26_Pos  	(26U)
#define   ETH_DEBUG_Reserved_31_26_Msk  	(0x0000003FU  << ETH_DEBUG_Reserved_31_26_Pos)

/* ETH_REMOTE_WAKE_UP_FRAME_FILTER  =  */
#define   ETH_REMOTE_WAKE_UP_FRAME_FILTER_WKUPFRMFTR_Pos  	(0U)
#define   ETH_REMOTE_WAKE_UP_FRAME_FILTER_WKUPFRMFTR_Msk  	(0xFFFFFFFFU  << ETH_REMOTE_WAKE_UP_FRAME_FILTER_WKUPFRMFTR_Pos)

/* ETH_PMT_CONTROL_STATUS  =  Register 11 [PMT Control and Status Register]*/
#define   ETH_PMT_CONTROL_STATUS_PWRDWN_Pos  	(0U)
#define   ETH_PMT_CONTROL_STATUS_PWRDWN_Msk  	(0x00000001U  << ETH_PMT_CONTROL_STATUS_PWRDWN_Pos)

#define   ETH_PMT_CONTROL_STATUS_MGKPKTEN_Pos  	(1U)
#define   ETH_PMT_CONTROL_STATUS_MGKPKTEN_Msk  	(0x00000001U  << ETH_PMT_CONTROL_STATUS_MGKPKTEN_Pos)

#define   ETH_PMT_CONTROL_STATUS_RWKPKTEN_Pos  	(2U)
#define   ETH_PMT_CONTROL_STATUS_RWKPKTEN_Msk  	(0x00000001U  << ETH_PMT_CONTROL_STATUS_RWKPKTEN_Pos)

#define   ETH_PMT_CONTROL_STATUS_Reserved_4_3_Pos  	(3U)
#define   ETH_PMT_CONTROL_STATUS_Reserved_4_3_Msk  	(0x00000003U  << ETH_PMT_CONTROL_STATUS_Reserved_4_3_Pos)

#define   ETH_PMT_CONTROL_STATUS_MGKPRCVD_Pos  	(5U)
#define   ETH_PMT_CONTROL_STATUS_MGKPRCVD_Msk  	(0x00000001U  << ETH_PMT_CONTROL_STATUS_MGKPRCVD_Pos)

#define   ETH_PMT_CONTROL_STATUS_RWKPRCVD_Pos  	(6U)
#define   ETH_PMT_CONTROL_STATUS_RWKPRCVD_Msk  	(0x00000001U  << ETH_PMT_CONTROL_STATUS_RWKPRCVD_Pos)

#define   ETH_PMT_CONTROL_STATUS_Reserved_8_7_Pos  	(7U)
#define   ETH_PMT_CONTROL_STATUS_Reserved_8_7_Msk  	(0x00000003U  << ETH_PMT_CONTROL_STATUS_Reserved_8_7_Pos)

#define   ETH_PMT_CONTROL_STATUS_GLBLUCAST_Pos  	(9U)
#define   ETH_PMT_CONTROL_STATUS_GLBLUCAST_Msk  	(0x00000001U  << ETH_PMT_CONTROL_STATUS_GLBLUCAST_Pos)

#define   ETH_PMT_CONTROL_STATUS_Reserved_30_10_Pos  	(10U)
#define   ETH_PMT_CONTROL_STATUS_Reserved_30_10_Msk  	(0x001FFFFFU  << ETH_PMT_CONTROL_STATUS_Reserved_30_10_Pos)

#define   ETH_PMT_CONTROL_STATUS_RWKFILTRST_Pos  	(31U)
#define   ETH_PMT_CONTROL_STATUS_RWKFILTRST_Msk  	(0x00000001U  << ETH_PMT_CONTROL_STATUS_RWKFILTRST_Pos)

/* ETH_INTERRUPT_STATUS  =  Register 14 [Interrupt Register]*/
#define   ETH_INTERRUPT_STATUS_RMIIIS_Pos  	(0U)
#define   ETH_INTERRUPT_STATUS_RMIIIS_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_RMIIIS_Pos)

#define   ETH_INTERRUPT_STATUS_Reserved_1_Pos  	(1U)
#define   ETH_INTERRUPT_STATUS_Reserved_1_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_Reserved_1_Pos)

#define   ETH_INTERRUPT_STATUS_Reserved_2_Pos  	(2U)
#define   ETH_INTERRUPT_STATUS_Reserved_2_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_Reserved_2_Pos)

#define   ETH_INTERRUPT_STATUS_PMTIS_Pos  	(3U)
#define   ETH_INTERRUPT_STATUS_PMTIS_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_PMTIS_Pos)

#define   ETH_INTERRUPT_STATUS_MMCIS_Pos  	(4U)
#define   ETH_INTERRUPT_STATUS_MMCIS_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_MMCIS_Pos)

#define   ETH_INTERRUPT_STATUS_MMCRXIS_Pos  	(5U)
#define   ETH_INTERRUPT_STATUS_MMCRXIS_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_MMCRXIS_Pos)

#define   ETH_INTERRUPT_STATUS_MMCTXIS_Pos  	(6U)
#define   ETH_INTERRUPT_STATUS_MMCTXIS_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_MMCTXIS_Pos)

#define   ETH_INTERRUPT_STATUS_MMCRXIPIS_Pos  	(7U)
#define   ETH_INTERRUPT_STATUS_MMCRXIPIS_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_MMCRXIPIS_Pos)

#define   ETH_INTERRUPT_STATUS_Reserved_8_Pos  	(8U)
#define   ETH_INTERRUPT_STATUS_Reserved_8_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_Reserved_8_Pos)

#define   ETH_INTERRUPT_STATUS_TSIS_Pos  	(9U)
#define   ETH_INTERRUPT_STATUS_TSIS_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_TSIS_Pos)

#define   ETH_INTERRUPT_STATUS_LPIIS_Pos  	(10U)
#define   ETH_INTERRUPT_STATUS_LPIIS_Msk  	(0x00000001U  << ETH_INTERRUPT_STATUS_LPIIS_Pos)

#define   ETH_INTERRUPT_STATUS_Reserved_31_11_Pos  	(11U)
#define   ETH_INTERRUPT_STATUS_Reserved_31_11_Msk  	(0x001FFFFFU  << ETH_INTERRUPT_STATUS_Reserved_31_11_Pos)

/* ETH_INTERRUPT_MASK  =  Register 15 [Interrupt Mask Register]*/
#define   ETH_INTERRUPT_MASK_RMIIIM_Pos  	(0U)
#define   ETH_INTERRUPT_MASK_RMIIIM_Msk  	(0x00000001U  << ETH_INTERRUPT_MASK_RMIIIM_Pos)

#define   ETH_INTERRUPT_MASK_Reserved_1_Pos  	(1U)
#define   ETH_INTERRUPT_MASK_Reserved_1_Msk  	(0x00000001U  << ETH_INTERRUPT_MASK_Reserved_1_Pos)

#define   ETH_INTERRUPT_MASK_Reserved_2_Pos  	(2U)
#define   ETH_INTERRUPT_MASK_Reserved_2_Msk  	(0x00000001U  << ETH_INTERRUPT_MASK_Reserved_2_Pos)

#define   ETH_INTERRUPT_MASK_PMTIM_Pos  	(3U)
#define   ETH_INTERRUPT_MASK_PMTIM_Msk  	(0x00000001U  << ETH_INTERRUPT_MASK_PMTIM_Pos)

#define   ETH_INTERRUPT_MASK_Reserved_8_4_Pos  	(4U)
#define   ETH_INTERRUPT_MASK_Reserved_8_4_Msk  	(0x0000001FU  << ETH_INTERRUPT_MASK_Reserved_8_4_Pos)

#define   ETH_INTERRUPT_MASK_TSIM_Pos  	(9U)
#define   ETH_INTERRUPT_MASK_TSIM_Msk  	(0x00000001U  << ETH_INTERRUPT_MASK_TSIM_Pos)

#define   ETH_INTERRUPT_MASK_LPIIM_Pos  	(10U)
#define   ETH_INTERRUPT_MASK_LPIIM_Msk  	(0x00000001U  << ETH_INTERRUPT_MASK_LPIIM_Pos)

#define   ETH_INTERRUPT_MASK_Reserved_31_11_Pos  	(11U)
#define   ETH_INTERRUPT_MASK_Reserved_31_11_Msk  	(0x001FFFFFU  << ETH_INTERRUPT_MASK_Reserved_31_11_Pos)

/* ETH_MAC_ADDRESS0_HIGH  =  Register 16 [MAC Address0 High Register]*/
#define   ETH_MAC_ADDRESS0_HIGH_ADDRHI_Pos  	(0U)
#define   ETH_MAC_ADDRESS0_HIGH_ADDRHI_Msk  	(0x0000FFFFU  << ETH_MAC_ADDRESS0_HIGH_ADDRHI_Pos)

#define   ETH_MAC_ADDRESS0_HIGH_Reserved_30_16_Pos  	(16U)
#define   ETH_MAC_ADDRESS0_HIGH_Reserved_30_16_Msk  	(0x00007FFFU  << ETH_MAC_ADDRESS0_HIGH_Reserved_30_16_Pos)

#define   ETH_MAC_ADDRESS0_HIGH_AE_Pos  	(31U)
#define   ETH_MAC_ADDRESS0_HIGH_AE_Msk  	(0x00000001U  << ETH_MAC_ADDRESS0_HIGH_AE_Pos)

/* ETH_MAC_ADDRESS0_LOW  =  Register 17 [MAC Address0 Low Register]*/
#define   ETH_MAC_ADDRESS0_LOW_ADDRLO_Pos  	(0U)
#define   ETH_MAC_ADDRESS0_LOW_ADDRLO_Msk  	(0xFFFFFFFFU  << ETH_MAC_ADDRESS0_LOW_ADDRLO_Pos)

/* ETH_MAC_ADDRESS1_HIGH  =  Register 18 [MAC Address1 High Register]*/
#define   ETH_MAC_ADDRESS1_HIGH_ADDRHI_Pos  	(0U)
#define   ETH_MAC_ADDRESS1_HIGH_ADDRHI_Msk  	(0x0000FFFFU  << ETH_MAC_ADDRESS1_HIGH_ADDRHI_Pos)

#define   ETH_MAC_ADDRESS1_HIGH_Reserved_23_16_Pos  	(16U)
#define   ETH_MAC_ADDRESS1_HIGH_Reserved_23_16_Msk  	(0x000000FFU  << ETH_MAC_ADDRESS1_HIGH_Reserved_23_16_Pos)

#define   ETH_MAC_ADDRESS1_HIGH_MBC_Pos  	(24U)
#define   ETH_MAC_ADDRESS1_HIGH_MBC_Msk  	(0x0000003FU  << ETH_MAC_ADDRESS1_HIGH_MBC_Pos)

#define   ETH_MAC_ADDRESS1_HIGH_SA_Pos  	(30U)
#define   ETH_MAC_ADDRESS1_HIGH_SA_Msk  	(0x00000001U  << ETH_MAC_ADDRESS1_HIGH_SA_Pos)

#define   ETH_MAC_ADDRESS1_HIGH_AE_Pos  	(31U)
#define   ETH_MAC_ADDRESS1_HIGH_AE_Msk  	(0x00000001U  << ETH_MAC_ADDRESS1_HIGH_AE_Pos)

/* ETH_MAC_ADDRESS1_LOW  =  Register 19 [MAC Address1 Low Register]*/
#define   ETH_MAC_ADDRESS1_LOW_ADDRLO_Pos  	(0U)
#define   ETH_MAC_ADDRESS1_LOW_ADDRLO_Msk  	(0xFFFFFFFFU  << ETH_MAC_ADDRESS1_LOW_ADDRLO_Pos)

/* ETH_MAC_ADDRESS2_HIGH  =  Register 20 [MAC Address2 High Register]*/
#define   ETH_MAC_ADDRESS2_HIGH_ADDRHI_Pos  	(0U)
#define   ETH_MAC_ADDRESS2_HIGH_ADDRHI_Msk  	(0x0000FFFFU  << ETH_MAC_ADDRESS2_HIGH_ADDRHI_Pos)

#define   ETH_MAC_ADDRESS2_HIGH_Reserved_23_16_Pos  	(16U)
#define   ETH_MAC_ADDRESS2_HIGH_Reserved_23_16_Msk  	(0x000000FFU  << ETH_MAC_ADDRESS2_HIGH_Reserved_23_16_Pos)

#define   ETH_MAC_ADDRESS2_HIGH_MBC_Pos  	(24U)
#define   ETH_MAC_ADDRESS2_HIGH_MBC_Msk  	(0x0000003FU  << ETH_MAC_ADDRESS2_HIGH_MBC_Pos)

#define   ETH_MAC_ADDRESS2_HIGH_SA_Pos  	(30U)
#define   ETH_MAC_ADDRESS2_HIGH_SA_Msk  	(0x00000001U  << ETH_MAC_ADDRESS2_HIGH_SA_Pos)

#define   ETH_MAC_ADDRESS2_HIGH_AE_Pos  	(31U)
#define   ETH_MAC_ADDRESS2_HIGH_AE_Msk  	(0x00000001U  << ETH_MAC_ADDRESS2_HIGH_AE_Pos)

/* ETH_MAC_ADDRESS2_LOW  =  Register 21 [MAC Address2 Low Register]*/
#define   ETH_MAC_ADDRESS2_LOW_ADDRLO_Pos  	(0U)
#define   ETH_MAC_ADDRESS2_LOW_ADDRLO_Msk  	(0xFFFFFFFFU  << ETH_MAC_ADDRESS2_LOW_ADDRLO_Pos)

/* ETH_MAC_ADDRESS3_HIGH  =  Register 22 [MAC Address3 High Register]*/
#define   ETH_MAC_ADDRESS3_HIGH_ADDRHI_Pos  	(0U)
#define   ETH_MAC_ADDRESS3_HIGH_ADDRHI_Msk  	(0x0000FFFFU  << ETH_MAC_ADDRESS3_HIGH_ADDRHI_Pos)

#define   ETH_MAC_ADDRESS3_HIGH_Reserved_23_16_Pos  	(16U)
#define   ETH_MAC_ADDRESS3_HIGH_Reserved_23_16_Msk  	(0x000000FFU  << ETH_MAC_ADDRESS3_HIGH_Reserved_23_16_Pos)

#define   ETH_MAC_ADDRESS3_HIGH_MBC_Pos  	(24U)
#define   ETH_MAC_ADDRESS3_HIGH_MBC_Msk  	(0x0000003FU  << ETH_MAC_ADDRESS3_HIGH_MBC_Pos)

#define   ETH_MAC_ADDRESS3_HIGH_SA_Pos  	(30U)
#define   ETH_MAC_ADDRESS3_HIGH_SA_Msk  	(0x00000001U  << ETH_MAC_ADDRESS3_HIGH_SA_Pos)

#define   ETH_MAC_ADDRESS3_HIGH_AE_Pos  	(31U)
#define   ETH_MAC_ADDRESS3_HIGH_AE_Msk  	(0x00000001U  << ETH_MAC_ADDRESS3_HIGH_AE_Pos)

/* ETH_MAC_ADDRESS3_LOW  =  Register 23 [MAC Address3 Low Register]*/
#define   ETH_MAC_ADDRESS3_LOW_ADDRLO_Pos  	(0U)
#define   ETH_MAC_ADDRESS3_LOW_ADDRLO_Msk  	(0xFFFFFFFFU  << ETH_MAC_ADDRESS3_LOW_ADDRLO_Pos)

/* ETH_MMC_CONTROL  =  Register 64 [MMC Control Register]*/
#define   ETH_MMC_CONTROL_CNTRST_Pos  	(0U)
#define   ETH_MMC_CONTROL_CNTRST_Msk  	(0x00000001U  << ETH_MMC_CONTROL_CNTRST_Pos)

#define   ETH_MMC_CONTROL_CNTSTOPRO_Pos  	(1U)
#define   ETH_MMC_CONTROL_CNTSTOPRO_Msk  	(0x00000001U  << ETH_MMC_CONTROL_CNTSTOPRO_Pos)

#define   ETH_MMC_CONTROL_RSTONRD_Pos  	(2U)
#define   ETH_MMC_CONTROL_RSTONRD_Msk  	(0x00000001U  << ETH_MMC_CONTROL_RSTONRD_Pos)

#define   ETH_MMC_CONTROL_CNTFREEZ_Pos  	(3U)
#define   ETH_MMC_CONTROL_CNTFREEZ_Msk  	(0x00000001U  << ETH_MMC_CONTROL_CNTFREEZ_Pos)

#define   ETH_MMC_CONTROL_CNTPRST_Pos  	(4U)
#define   ETH_MMC_CONTROL_CNTPRST_Msk  	(0x00000001U  << ETH_MMC_CONTROL_CNTPRST_Pos)

#define   ETH_MMC_CONTROL_CNTPRSTLVL_Pos  	(5U)
#define   ETH_MMC_CONTROL_CNTPRSTLVL_Msk  	(0x00000001U  << ETH_MMC_CONTROL_CNTPRSTLVL_Pos)

#define   ETH_MMC_CONTROL_Reserved_7_6_Pos  	(6U)
#define   ETH_MMC_CONTROL_Reserved_7_6_Msk  	(0x00000003U  << ETH_MMC_CONTROL_Reserved_7_6_Pos)

#define   ETH_MMC_CONTROL_UCDBC_Pos  	(8U)
#define   ETH_MMC_CONTROL_UCDBC_Msk  	(0x00000001U  << ETH_MMC_CONTROL_UCDBC_Pos)

#define   ETH_MMC_CONTROL_Reserved_31_9_Pos  	(9U)
#define   ETH_MMC_CONTROL_Reserved_31_9_Msk  	(0x007FFFFFU  << ETH_MMC_CONTROL_Reserved_31_9_Pos)

/* ETH_MMC_RECEIVE_INTERRUPT  =  Register 65 [MMC Receive Interrupt Register]*/
#define   ETH_MMC_RECEIVE_INTERRUPT_RXGBFRMIS_Pos  	(0U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXGBFRMIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXGBFRMIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXGBOCTIS_Pos  	(1U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXGBOCTIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXGBOCTIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXGOCTIS_Pos  	(2U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXGOCTIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXGOCTIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXBCGFIS_Pos  	(3U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXBCGFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXBCGFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXMCGFIS_Pos  	(4U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXMCGFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXMCGFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXCRCERFIS_Pos  	(5U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXCRCERFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXCRCERFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXALGNERFIS_Pos  	(6U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXALGNERFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXALGNERFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXRUNTFIS_Pos  	(7U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXRUNTFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXRUNTFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXJABERFIS_Pos  	(8U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXJABERFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXJABERFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXUSIZEGFIS_Pos  	(9U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXUSIZEGFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXUSIZEGFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXOSIZEGFIS_Pos  	(10U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXOSIZEGFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXOSIZEGFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RX64OCTGBFIS_Pos  	(11U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RX64OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RX64OCTGBFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RX65T127OCTGBFIS_Pos  	(12U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RX65T127OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RX65T127OCTGBFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RX128T255OCTGBFIS_Pos  	(13U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RX128T255OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RX128T255OCTGBFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RX256T511OCTGBFIS_Pos  	(14U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RX256T511OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RX256T511OCTGBFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RX512T1023OCTGBFIS_Pos  	(15U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RX512T1023OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RX512T1023OCTGBFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RX1024TMAXOCTGBFIS_Pos  	(16U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RX1024TMAXOCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RX1024TMAXOCTGBFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXUCGFIS_Pos  	(17U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXUCGFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXUCGFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXLENERFIS_Pos  	(18U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXLENERFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXLENERFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXORANGEFIS_Pos  	(19U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXORANGEFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXORANGEFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXPAUSFIS_Pos  	(20U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXPAUSFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXPAUSFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXFOVFIS_Pos  	(21U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXFOVFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXFOVFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXVLANGBFIS_Pos  	(22U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXVLANGBFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXVLANGBFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXWDOGFIS_Pos  	(23U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXWDOGFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXWDOGFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXRCVERRFIS_Pos  	(24U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXRCVERRFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXRCVERRFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_RXCTRLFIS_Pos  	(25U)
#define   ETH_MMC_RECEIVE_INTERRUPT_RXCTRLFIS_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_RXCTRLFIS_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_Reserved_31_26_Pos  	(26U)
#define   ETH_MMC_RECEIVE_INTERRUPT_Reserved_31_26_Msk  	(0x0000003FU  << ETH_MMC_RECEIVE_INTERRUPT_Reserved_31_26_Pos)

/* ETH_MMC_TRANSMIT_INTERRUPT  =  Register 66 [MMC Transmit Interrupt Register]*/
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXGBOCTIS_Pos  	(0U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXGBOCTIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXGBOCTIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXGBFRMIS_Pos  	(1U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXGBFRMIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXGBFRMIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXBCGFIS_Pos  	(2U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXBCGFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXBCGFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXMCGFIS_Pos  	(3U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXMCGFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXMCGFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TX64OCTGBFIS_Pos  	(4U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TX64OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TX64OCTGBFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TX65T127OCTGBFIS_Pos  	(5U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TX65T127OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TX65T127OCTGBFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TX128T255OCTGBFIS_Pos  	(6U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TX128T255OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TX128T255OCTGBFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TX256T511OCTGBFIS_Pos  	(7U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TX256T511OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TX256T511OCTGBFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TX512T1023OCTGBFIS_Pos  	(8U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TX512T1023OCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TX512T1023OCTGBFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TX1024TMAXOCTGBFIS_Pos  	(9U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TX1024TMAXOCTGBFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TX1024TMAXOCTGBFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXUCGBFIS_Pos  	(10U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXUCGBFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXUCGBFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXMCGBFIS_Pos  	(11U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXMCGBFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXMCGBFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXBCGBFIS_Pos  	(12U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXBCGBFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXBCGBFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXUFLOWERFIS_Pos  	(13U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXUFLOWERFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXUFLOWERFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXSCOLGFIS_Pos  	(14U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXSCOLGFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXSCOLGFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXMCOLGFIS_Pos  	(15U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXMCOLGFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXMCOLGFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXDEFFIS_Pos  	(16U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXDEFFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXDEFFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXLATCOLFIS_Pos  	(17U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXLATCOLFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXLATCOLFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXEXCOLFIS_Pos  	(18U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXEXCOLFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXEXCOLFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXCARERFIS_Pos  	(19U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXCARERFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXCARERFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXGOCTIS_Pos  	(20U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXGOCTIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXGOCTIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXGFRMIS_Pos  	(21U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXGFRMIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXGFRMIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXEXDEFFIS_Pos  	(22U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXEXDEFFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXEXDEFFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXPAUSFIS_Pos  	(23U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXPAUSFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXPAUSFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXVLANGFIS_Pos  	(24U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXVLANGFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXVLANGFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_TXOSIZEGFIS_Pos  	(25U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_TXOSIZEGFIS_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_TXOSIZEGFIS_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_Reserved_31_26_Pos  	(26U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_Reserved_31_26_Msk  	(0x0000003FU  << ETH_MMC_TRANSMIT_INTERRUPT_Reserved_31_26_Pos)

/* ETH_MMC_RECEIVE_INTERRUPT_MASK  =  */
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXGBFRMIM_Pos  	(0U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXGBFRMIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXGBFRMIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXGBOCTIM_Pos  	(1U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXGBOCTIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXGBOCTIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXGOCTIM_Pos  	(2U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXGOCTIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXGOCTIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXBCGFIM_Pos  	(3U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXBCGFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXBCGFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXMCGFIM_Pos  	(4U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXMCGFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXMCGFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXCRCERFIM_Pos  	(5U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXCRCERFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXCRCERFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXALGNERFIM_Pos  	(6U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXALGNERFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXALGNERFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXRUNTFIM_Pos  	(7U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXRUNTFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXRUNTFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXJABERFIM_Pos  	(8U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXJABERFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXJABERFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXUSIZEGFIM_Pos  	(9U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXUSIZEGFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXUSIZEGFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXOSIZEGFIM_Pos  	(10U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXOSIZEGFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXOSIZEGFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX64OCTGBFIM_Pos  	(11U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX64OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RX64OCTGBFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX65T127OCTGBFIM_Pos  	(12U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX65T127OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RX65T127OCTGBFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX128T255OCTGBFIM_Pos  	(13U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX128T255OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RX128T255OCTGBFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX256T511OCTGBFIM_Pos  	(14U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX256T511OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RX256T511OCTGBFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX512T1023OCTGBFIM_Pos  	(15U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX512T1023OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RX512T1023OCTGBFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX1024TMAXOCTGBFIM_Pos  	(16U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RX1024TMAXOCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RX1024TMAXOCTGBFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXUCGFIM_Pos  	(17U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXUCGFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXUCGFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXLENERFIM_Pos  	(18U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXLENERFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXLENERFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXORANGEFIM_Pos  	(19U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXORANGEFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXORANGEFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXPAUSFIM_Pos  	(20U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXPAUSFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXPAUSFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXFOVFIM_Pos  	(21U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXFOVFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXFOVFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXVLANGBFIM_Pos  	(22U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXVLANGBFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXVLANGBFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXWDOGFIM_Pos  	(23U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXWDOGFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXWDOGFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXRCVERRFIM_Pos  	(24U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXRCVERRFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXRCVERRFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXCTRLFIM_Pos  	(25U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_RXCTRLFIM_Msk  	(0x00000001U  << ETH_MMC_RECEIVE_INTERRUPT_MASK_RXCTRLFIM_Pos)

#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_Reserved_31_26_Pos  	(26U)
#define   ETH_MMC_RECEIVE_INTERRUPT_MASK_Reserved_31_26_Msk  	(0x0000003FU  << ETH_MMC_RECEIVE_INTERRUPT_MASK_Reserved_31_26_Pos)

/* ETH_MMC_TRANSMIT_INTERRUPT_MASK  =  */
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGBOCTIM_Pos  	(0U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGBOCTIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGBOCTIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGBFRMIM_Pos  	(1U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGBFRMIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGBFRMIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXBCGFIM_Pos  	(2U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXBCGFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXBCGFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXMCGFIM_Pos  	(3U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXMCGFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXMCGFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX64OCTGBFIM_Pos  	(4U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX64OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX64OCTGBFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX65T127OCTGBFIM_Pos  	(5U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX65T127OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX65T127OCTGBFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX128T255OCTGBFIM_Pos  	(6U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX128T255OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX128T255OCTGBFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX256T511OCTGBFIM_Pos  	(7U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX256T511OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX256T511OCTGBFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX512T1023OCTGBFIM_Pos  	(8U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX512T1023OCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX512T1023OCTGBFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX1024TMAXOCTGBFIM_Pos  	(9U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX1024TMAXOCTGBFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TX1024TMAXOCTGBFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXUCGBFIM_Pos  	(10U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXUCGBFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXUCGBFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXMCGBFIM_Pos  	(11U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXMCGBFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXMCGBFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXBCGBFIM_Pos  	(12U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXBCGBFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXBCGBFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXUFLOWERFIM_Pos  	(13U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXUFLOWERFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXUFLOWERFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXSCOLGFIM_Pos  	(14U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXSCOLGFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXSCOLGFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXMCOLGFIM_Pos  	(15U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXMCOLGFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXMCOLGFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXDEFFIM_Pos  	(16U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXDEFFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXDEFFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXLATCOLFIM_Pos  	(17U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXLATCOLFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXLATCOLFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXEXCOLFIM_Pos  	(18U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXEXCOLFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXEXCOLFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXCARERFIM_Pos  	(19U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXCARERFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXCARERFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGOCTIM_Pos  	(20U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGOCTIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGOCTIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGFRMIM_Pos  	(21U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGFRMIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXGFRMIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXEXDEFFIM_Pos  	(22U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXEXDEFFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXEXDEFFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXPAUSFIM_Pos  	(23U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXPAUSFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXPAUSFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXVLANGFIM_Pos  	(24U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXVLANGFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXVLANGFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXOSIZEGFIM_Pos  	(25U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXOSIZEGFIM_Msk  	(0x00000001U  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_TXOSIZEGFIM_Pos)

#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_Reserved_31_26_Pos  	(26U)
#define   ETH_MMC_TRANSMIT_INTERRUPT_MASK_Reserved_31_26_Msk  	(0x0000003FU  << ETH_MMC_TRANSMIT_INTERRUPT_MASK_Reserved_31_26_Pos)

/* ETH_TX_OCTET_COUNT_GOOD_BAD  =  Register 69 [Transmit Octet Count for Good and Bad Frames]*/
#define   ETH_TX_OCTET_COUNT_GOOD_BAD_TXOCTGB_Pos  	(0U)
#define   ETH_TX_OCTET_COUNT_GOOD_BAD_TXOCTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_OCTET_COUNT_GOOD_BAD_TXOCTGB_Pos)

/* ETH_TX_FRAME_COUNT_GOOD_BAD  =  Register 70 [Transmit Frame Count for Good and Bad Frames]*/
#define   ETH_TX_FRAME_COUNT_GOOD_BAD_TXFRMGB_Pos  	(0U)
#define   ETH_TX_FRAME_COUNT_GOOD_BAD_TXFRMGB_Msk  	(0xFFFFFFFFU  << ETH_TX_FRAME_COUNT_GOOD_BAD_TXFRMGB_Pos)

/* ETH_TX_BROADCAST_FRAMES_GOOD  =  Register 71 [Transmit Frame Count for Good Broadcast Frames]*/
#define   ETH_TX_BROADCAST_FRAMES_GOOD_TXBCASTG_Pos  	(0U)
#define   ETH_TX_BROADCAST_FRAMES_GOOD_TXBCASTG_Msk  	(0xFFFFFFFFU  << ETH_TX_BROADCAST_FRAMES_GOOD_TXBCASTG_Pos)

/* ETH_TX_MULTICAST_FRAMES_GOOD  =  Register 72 [Transmit Frame Count for Good Multicast Frames]*/
#define   ETH_TX_MULTICAST_FRAMES_GOOD_TXMCASTG_Pos  	(0U)
#define   ETH_TX_MULTICAST_FRAMES_GOOD_TXMCASTG_Msk  	(0xFFFFFFFFU  << ETH_TX_MULTICAST_FRAMES_GOOD_TXMCASTG_Pos)

/* ETH_TX_64OCTETS_FRAMES_GOOD_BAD  =  Register 73 [Transmit Octet Count for Good and Bad 64 Byte Frames]*/
#define   ETH_TX_64OCTETS_FRAMES_GOOD_BAD_TX64OCTGB_Pos  	(0U)
#define   ETH_TX_64OCTETS_FRAMES_GOOD_BAD_TX64OCTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_64OCTETS_FRAMES_GOOD_BAD_TX64OCTGB_Pos)

/* ETH_TX_65TO127OCTETS_FRAMES_GOOD_BAD  =  Register 74 [Transmit Octet Count for Good and Bad 65 to 127 Bytes Frames]*/
#define   ETH_TX_65TO127OCTETS_FRAMES_GOOD_BAD_TX65_127OCTGB_Pos  	(0U)
#define   ETH_TX_65TO127OCTETS_FRAMES_GOOD_BAD_TX65_127OCTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_65TO127OCTETS_FRAMES_GOOD_BAD_TX65_127OCTGB_Pos)

/* ETH_TX_128TO255OCTETS_FRAMES_GOOD_BAD  =  Register 75 [Transmit Octet Count for Good and Bad 128 to 255 Bytes Frames]*/
#define   ETH_TX_128TO255OCTETS_FRAMES_GOOD_BAD_TX128_255OCTGB_Pos  	(0U)
#define   ETH_TX_128TO255OCTETS_FRAMES_GOOD_BAD_TX128_255OCTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_128TO255OCTETS_FRAMES_GOOD_BAD_TX128_255OCTGB_Pos)

/* ETH_TX_256TO511OCTETS_FRAMES_GOOD_BAD  =  Register 76 [Transmit Octet Count for Good and Bad 256 to 511 Bytes Frames]*/
#define   ETH_TX_256TO511OCTETS_FRAMES_GOOD_BAD_TX256_511OCTGB_Pos  	(0U)
#define   ETH_TX_256TO511OCTETS_FRAMES_GOOD_BAD_TX256_511OCTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_256TO511OCTETS_FRAMES_GOOD_BAD_TX256_511OCTGB_Pos)

/* ETH_TX_512TO1023OCTETS_FRAMES_GOOD_BAD  =  Register 77 [Transmit Octet Count for Good and Bad 512 to 1023 Bytes Frames]*/
#define   ETH_TX_512TO1023OCTETS_FRAMES_GOOD_BAD_TX512_1023OCTGB_Pos  	(0U)
#define   ETH_TX_512TO1023OCTETS_FRAMES_GOOD_BAD_TX512_1023OCTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_512TO1023OCTETS_FRAMES_GOOD_BAD_TX512_1023OCTGB_Pos)

/* ETH_TX_1024TOMAXOCTETS_FRAMES_GOOD_BAD  =  Register 78 [Transmit Octet Count for Good and Bad 1024 to Maxsize Bytes Frames]*/
#define   ETH_TX_1024TOMAXOCTETS_FRAMES_GOOD_BAD_TX1024_MAXOCTGB_Pos  	(0U)
#define   ETH_TX_1024TOMAXOCTETS_FRAMES_GOOD_BAD_TX1024_MAXOCTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_1024TOMAXOCTETS_FRAMES_GOOD_BAD_TX1024_MAXOCTGB_Pos)

/* ETH_TX_UNICAST_FRAMES_GOOD_BAD  =  Register 79 [Transmit Frame Count for Good and Bad Unicast Frames]*/
#define   ETH_TX_UNICAST_FRAMES_GOOD_BAD_TXUCASTGB_Pos  	(0U)
#define   ETH_TX_UNICAST_FRAMES_GOOD_BAD_TXUCASTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_UNICAST_FRAMES_GOOD_BAD_TXUCASTGB_Pos)

/* ETH_TX_MULTICAST_FRAMES_GOOD_BAD  =  Register 80 [Transmit Frame Count for Good and Bad Multicast Frames]*/
#define   ETH_TX_MULTICAST_FRAMES_GOOD_BAD_TXMCASTGB_Pos  	(0U)
#define   ETH_TX_MULTICAST_FRAMES_GOOD_BAD_TXMCASTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_MULTICAST_FRAMES_GOOD_BAD_TXMCASTGB_Pos)

/* ETH_TX_BROADCAST_FRAMES_GOOD_BAD  =  Register 81 [Transmit Frame Count for Good and Bad Broadcast Frames]*/
#define   ETH_TX_BROADCAST_FRAMES_GOOD_BAD_TXBCASTGB_Pos  	(0U)
#define   ETH_TX_BROADCAST_FRAMES_GOOD_BAD_TXBCASTGB_Msk  	(0xFFFFFFFFU  << ETH_TX_BROADCAST_FRAMES_GOOD_BAD_TXBCASTGB_Pos)

/* ETH_TX_UNDERFLOW_ERROR_FRAMES  =  Register 82 [Transmit Frame Count for Underflow Error Frames]*/
#define   ETH_TX_UNDERFLOW_ERROR_FRAMES_TXUNDRFLW_Pos  	(0U)
#define   ETH_TX_UNDERFLOW_ERROR_FRAMES_TXUNDRFLW_Msk  	(0xFFFFFFFFU  << ETH_TX_UNDERFLOW_ERROR_FRAMES_TXUNDRFLW_Pos)

/* ETH_TX_SINGLE_COLLISION_GOOD_FRAMES  =  Register 83 [Transmit Frame Count for Frames Transmitted after Single Collision]*/
#define   ETH_TX_SINGLE_COLLISION_GOOD_FRAMES_TXSNGLCOLG_Pos  	(0U)
#define   ETH_TX_SINGLE_COLLISION_GOOD_FRAMES_TXSNGLCOLG_Msk  	(0xFFFFFFFFU  << ETH_TX_SINGLE_COLLISION_GOOD_FRAMES_TXSNGLCOLG_Pos)

/* ETH_TX_MULTIPLE_COLLISION_GOOD_FRAMES  =  Register 84 [Transmit Frame Count for Frames Transmitted after Multiple Collision]*/
#define   ETH_TX_MULTIPLE_COLLISION_GOOD_FRAMES_TXMULTCOLG_Pos  	(0U)
#define   ETH_TX_MULTIPLE_COLLISION_GOOD_FRAMES_TXMULTCOLG_Msk  	(0xFFFFFFFFU  << ETH_TX_MULTIPLE_COLLISION_GOOD_FRAMES_TXMULTCOLG_Pos)

/* ETH_TX_DEFERRED_FRAMES  =  */
#define   ETH_TX_DEFERRED_FRAMES_TXDEFRD_Pos  	(0U)
#define   ETH_TX_DEFERRED_FRAMES_TXDEFRD_Msk  	(0xFFFFFFFFU  << ETH_TX_DEFERRED_FRAMES_TXDEFRD_Pos)

/* ETH_TX_LATE_COLLISION_FRAMES  =  Register 86 [Transmit Frame Count for Late Collision Error Frames]*/
#define   ETH_TX_LATE_COLLISION_FRAMES_TXLATECOL_Pos  	(0U)
#define   ETH_TX_LATE_COLLISION_FRAMES_TXLATECOL_Msk  	(0xFFFFFFFFU  << ETH_TX_LATE_COLLISION_FRAMES_TXLATECOL_Pos)

/* ETH_TX_EXCESSIVE_COLLISION_FRAMES  =  Register 87 [Transmit Frame Count for Excessive Collision Error Frames]*/
#define   ETH_TX_EXCESSIVE_COLLISION_FRAMES_TXEXSCOL_Pos  	(0U)
#define   ETH_TX_EXCESSIVE_COLLISION_FRAMES_TXEXSCOL_Msk  	(0xFFFFFFFFU  << ETH_TX_EXCESSIVE_COLLISION_FRAMES_TXEXSCOL_Pos)

/* ETH_TX_CARRIER_ERROR_FRAMES  =  Register 88 [Transmit Frame Count for Carrier Sense Error Frames]*/
#define   ETH_TX_CARRIER_ERROR_FRAMES_TXCARR_Pos  	(0U)
#define   ETH_TX_CARRIER_ERROR_FRAMES_TXCARR_Msk  	(0xFFFFFFFFU  << ETH_TX_CARRIER_ERROR_FRAMES_TXCARR_Pos)

/* ETH_TX_OCTET_COUNT_GOOD  =  */
#define   ETH_TX_OCTET_COUNT_GOOD_TXOCTG_Pos  	(0U)
#define   ETH_TX_OCTET_COUNT_GOOD_TXOCTG_Msk  	(0xFFFFFFFFU  << ETH_TX_OCTET_COUNT_GOOD_TXOCTG_Pos)

/* ETH_TX_FRAME_COUNT_GOOD  =  */
#define   ETH_TX_FRAME_COUNT_GOOD_TXFRMG_Pos  	(0U)
#define   ETH_TX_FRAME_COUNT_GOOD_TXFRMG_Msk  	(0xFFFFFFFFU  << ETH_TX_FRAME_COUNT_GOOD_TXFRMG_Pos)

/* ETH_TX_EXCESSIVE_DEFERRAL_ERROR  =  Register 91 [Transmit Frame Count for Excessive Deferral Error Frames]*/
#define   ETH_TX_EXCESSIVE_DEFERRAL_ERROR_TXEXSDEF_Pos  	(0U)
#define   ETH_TX_EXCESSIVE_DEFERRAL_ERROR_TXEXSDEF_Msk  	(0xFFFFFFFFU  << ETH_TX_EXCESSIVE_DEFERRAL_ERROR_TXEXSDEF_Pos)

/* ETH_TX_PAUSE_FRAMES  =  Register 92 [Transmit Frame Count for Good PAUSE Frames]*/
#define   ETH_TX_PAUSE_FRAMES_TXPAUSE_Pos  	(0U)
#define   ETH_TX_PAUSE_FRAMES_TXPAUSE_Msk  	(0xFFFFFFFFU  << ETH_TX_PAUSE_FRAMES_TXPAUSE_Pos)

/* ETH_TX_VLAN_FRAMES_GOOD  =  Register 93 [Transmit Frame Count for Good VLAN Frames]*/
#define   ETH_TX_VLAN_FRAMES_GOOD_TXVLANG_Pos  	(0U)
#define   ETH_TX_VLAN_FRAMES_GOOD_TXVLANG_Msk  	(0xFFFFFFFFU  << ETH_TX_VLAN_FRAMES_GOOD_TXVLANG_Pos)

/* ETH_TX_OSIZE_FRAMES_GOOD  =  Register 94 [Transmit Frame Count for Good Oversize Frames]*/
#define   ETH_TX_OSIZE_FRAMES_GOOD_TXOSIZG_Pos  	(0U)
#define   ETH_TX_OSIZE_FRAMES_GOOD_TXOSIZG_Msk  	(0xFFFFFFFFU  << ETH_TX_OSIZE_FRAMES_GOOD_TXOSIZG_Pos)

/* ETH_RX_FRAMES_COUNT_GOOD_BAD  =  Register 96 [Receive Frame Count for Good and Bad Frames]*/
#define   ETH_RX_FRAMES_COUNT_GOOD_BAD_RXFRMGB_Pos  	(0U)
#define   ETH_RX_FRAMES_COUNT_GOOD_BAD_RXFRMGB_Msk  	(0xFFFFFFFFU  << ETH_RX_FRAMES_COUNT_GOOD_BAD_RXFRMGB_Pos)

/* ETH_RX_OCTET_COUNT_GOOD_BAD  =  Register 97 [Receive Octet Count for Good and Bad Frames]*/
#define   ETH_RX_OCTET_COUNT_GOOD_BAD_RXOCTGB_Pos  	(0U)
#define   ETH_RX_OCTET_COUNT_GOOD_BAD_RXOCTGB_Msk  	(0xFFFFFFFFU  << ETH_RX_OCTET_COUNT_GOOD_BAD_RXOCTGB_Pos)

/* ETH_RX_OCTET_COUNT_GOOD  =  */
#define   ETH_RX_OCTET_COUNT_GOOD_RXOCTG_Pos  	(0U)
#define   ETH_RX_OCTET_COUNT_GOOD_RXOCTG_Msk  	(0xFFFFFFFFU  << ETH_RX_OCTET_COUNT_GOOD_RXOCTG_Pos)

/* ETH_RX_BROADCAST_FRAMES_GOOD  =  Register 99 [Receive Frame Count for Good Broadcast Frames]*/
#define   ETH_RX_BROADCAST_FRAMES_GOOD_RXBCASTG_Pos  	(0U)
#define   ETH_RX_BROADCAST_FRAMES_GOOD_RXBCASTG_Msk  	(0xFFFFFFFFU  << ETH_RX_BROADCAST_FRAMES_GOOD_RXBCASTG_Pos)

/* ETH_RX_MULTICAST_FRAMES_GOOD  =  Register 100 [Receive Frame Count for Good Multicast Frames]*/
#define   ETH_RX_MULTICAST_FRAMES_GOOD_RXMCASTG_Pos  	(0U)
#define   ETH_RX_MULTICAST_FRAMES_GOOD_RXMCASTG_Msk  	(0xFFFFFFFFU  << ETH_RX_MULTICAST_FRAMES_GOOD_RXMCASTG_Pos)

/* ETH_RX_CRC_ERROR_FRAMES  =  Register 101 [Receive Frame Count for CRC Error Frames]*/
#define   ETH_RX_CRC_ERROR_FRAMES_RXCRCERR_Pos  	(0U)
#define   ETH_RX_CRC_ERROR_FRAMES_RXCRCERR_Msk  	(0xFFFFFFFFU  << ETH_RX_CRC_ERROR_FRAMES_RXCRCERR_Pos)

/* ETH_RX_ALIGNMENT_ERROR_FRAMES  =  Register 102 [Receive Frame Count for Alignment Error Frames]*/
#define   ETH_RX_ALIGNMENT_ERROR_FRAMES_RXALGNERR_Pos  	(0U)
#define   ETH_RX_ALIGNMENT_ERROR_FRAMES_RXALGNERR_Msk  	(0xFFFFFFFFU  << ETH_RX_ALIGNMENT_ERROR_FRAMES_RXALGNERR_Pos)

/* ETH_RX_RUNT_ERROR_FRAMES  =  Register 103 [Receive Frame Count for Runt Error Frames]*/
#define   ETH_RX_RUNT_ERROR_FRAMES_RXRUNTERR_Pos  	(0U)
#define   ETH_RX_RUNT_ERROR_FRAMES_RXRUNTERR_Msk  	(0xFFFFFFFFU  << ETH_RX_RUNT_ERROR_FRAMES_RXRUNTERR_Pos)

/* ETH_RX_JABBER_ERROR_FRAMES  =  Register 104 [Receive Frame Count for Jabber Error Frames]*/
#define   ETH_RX_JABBER_ERROR_FRAMES_RXJABERR_Pos  	(0U)
#define   ETH_RX_JABBER_ERROR_FRAMES_RXJABERR_Msk  	(0xFFFFFFFFU  << ETH_RX_JABBER_ERROR_FRAMES_RXJABERR_Pos)

/* ETH_RX_UNDERSIZE_FRAMES_GOOD  =  Register 105 [Receive Frame Count for Undersize Frames]*/
#define   ETH_RX_UNDERSIZE_FRAMES_GOOD_RXUNDERSZG_Pos  	(0U)
#define   ETH_RX_UNDERSIZE_FRAMES_GOOD_RXUNDERSZG_Msk  	(0xFFFFFFFFU  << ETH_RX_UNDERSIZE_FRAMES_GOOD_RXUNDERSZG_Pos)

/* ETH_RX_OVERSIZE_FRAMES_GOOD  =  */
#define   ETH_RX_OVERSIZE_FRAMES_GOOD_RXOVERSZG_Pos  	(0U)
#define   ETH_RX_OVERSIZE_FRAMES_GOOD_RXOVERSZG_Msk  	(0xFFFFFFFFU  << ETH_RX_OVERSIZE_FRAMES_GOOD_RXOVERSZG_Pos)

/* ETH_RX_64OCTETS_FRAMES_GOOD_BAD  =  Register 107 [Receive Frame Count for Good and Bad 64 Byte Frames]*/
#define   ETH_RX_64OCTETS_FRAMES_GOOD_BAD_RX64OCTGB_Pos  	(0U)
#define   ETH_RX_64OCTETS_FRAMES_GOOD_BAD_RX64OCTGB_Msk  	(0xFFFFFFFFU  << ETH_RX_64OCTETS_FRAMES_GOOD_BAD_RX64OCTGB_Pos)

/* ETH_RX_65TO127OCTETS_FRAMES_GOOD_BAD  =  Register 108 [Receive Frame Count for Good and Bad 65 to 127 Bytes Frames]*/
#define   ETH_RX_65TO127OCTETS_FRAMES_GOOD_BAD_RX65_127OCTGB_Pos  	(0U)
#define   ETH_RX_65TO127OCTETS_FRAMES_GOOD_BAD_RX65_127OCTGB_Msk  	(0xFFFFFFFFU  << ETH_RX_65TO127OCTETS_FRAMES_GOOD_BAD_RX65_127OCTGB_Pos)

/* ETH_RX_128TO255OCTETS_FRAMES_GOOD_BAD  =  Register 109 [Receive Frame Count for Good and Bad 128 to 255 Bytes Frames]*/
#define   ETH_RX_128TO255OCTETS_FRAMES_GOOD_BAD_RX128_255OCTGB_Pos  	(0U)
#define   ETH_RX_128TO255OCTETS_FRAMES_GOOD_BAD_RX128_255OCTGB_Msk  	(0xFFFFFFFFU  << ETH_RX_128TO255OCTETS_FRAMES_GOOD_BAD_RX128_255OCTGB_Pos)

/* ETH_RX_256TO511OCTETS_FRAMES_GOOD_BAD  =  Register 110 [Receive Frame Count for Good and Bad 256 to 511 Bytes Frames]*/
#define   ETH_RX_256TO511OCTETS_FRAMES_GOOD_BAD_RX256_511OCTGB_Pos  	(0U)
#define   ETH_RX_256TO511OCTETS_FRAMES_GOOD_BAD_RX256_511OCTGB_Msk  	(0xFFFFFFFFU  << ETH_RX_256TO511OCTETS_FRAMES_GOOD_BAD_RX256_511OCTGB_Pos)

/* ETH_RX_512TO1023OCTETS_FRAMES_GOOD_BAD  =  Register 111 [Receive Frame Count for Good and Bad 512 to 1,023 Bytes Frames]*/
#define   ETH_RX_512TO1023OCTETS_FRAMES_GOOD_BAD_RX512_1023OCTGB_Pos  	(0U)
#define   ETH_RX_512TO1023OCTETS_FRAMES_GOOD_BAD_RX512_1023OCTGB_Msk  	(0xFFFFFFFFU  << ETH_RX_512TO1023OCTETS_FRAMES_GOOD_BAD_RX512_1023OCTGB_Pos)

/* ETH_RX_1024TOMAXOCTETS_FRAMES_GOOD_BAD  =  Register 112 [Receive Frame Count for Good and Bad 1,024 to Maxsize Bytes Frames]*/
#define   ETH_RX_1024TOMAXOCTETS_FRAMES_GOOD_BAD_RX1024_MAXOCTGB_Pos  	(0U)
#define   ETH_RX_1024TOMAXOCTETS_FRAMES_GOOD_BAD_RX1024_MAXOCTGB_Msk  	(0xFFFFFFFFU  << ETH_RX_1024TOMAXOCTETS_FRAMES_GOOD_BAD_RX1024_MAXOCTGB_Pos)

/* ETH_RX_UNICAST_FRAMES_GOOD  =  Register 113 [Receive Frame Count for Good Unicast Frames]*/
#define   ETH_RX_UNICAST_FRAMES_GOOD_RXUCASTG_Pos  	(0U)
#define   ETH_RX_UNICAST_FRAMES_GOOD_RXUCASTG_Msk  	(0xFFFFFFFFU  << ETH_RX_UNICAST_FRAMES_GOOD_RXUCASTG_Pos)

/* ETH_RX_LENGTH_ERROR_FRAMES  =  Register 114 [Receive Frame Count for Length Error Frames]*/
#define   ETH_RX_LENGTH_ERROR_FRAMES_RXLENERR_Pos  	(0U)
#define   ETH_RX_LENGTH_ERROR_FRAMES_RXLENERR_Msk  	(0xFFFFFFFFU  << ETH_RX_LENGTH_ERROR_FRAMES_RXLENERR_Pos)

/* ETH_RX_OUT_OF_RANGE_TYPE_FRAMES  =  Register 115 [Receive Frame Count for Out of Range Frames]*/
#define   ETH_RX_OUT_OF_RANGE_TYPE_FRAMES_RXOUTOFRNG_Pos  	(0U)
#define   ETH_RX_OUT_OF_RANGE_TYPE_FRAMES_RXOUTOFRNG_Msk  	(0xFFFFFFFFU  << ETH_RX_OUT_OF_RANGE_TYPE_FRAMES_RXOUTOFRNG_Pos)

/* ETH_RX_PAUSE_FRAMES  =  Register 116 [Receive Frame Count for PAUSE Frames]*/
#define   ETH_RX_PAUSE_FRAMES_RXPAUSEFRM_Pos  	(0U)
#define   ETH_RX_PAUSE_FRAMES_RXPAUSEFRM_Msk  	(0xFFFFFFFFU  << ETH_RX_PAUSE_FRAMES_RXPAUSEFRM_Pos)

/* ETH_RX_FIFO_OVERFLOW_FRAMES  =  Register 117 [Receive Frame Count for FIFO Overflow Frames]*/
#define   ETH_RX_FIFO_OVERFLOW_FRAMES_RXFIFOOVFL_Pos  	(0U)
#define   ETH_RX_FIFO_OVERFLOW_FRAMES_RXFIFOOVFL_Msk  	(0xFFFFFFFFU  << ETH_RX_FIFO_OVERFLOW_FRAMES_RXFIFOOVFL_Pos)

/* ETH_RX_VLAN_FRAMES_GOOD_BAD  =  Register 118 [Receive Frame Count for Good and Bad VLAN Frames]*/
#define   ETH_RX_VLAN_FRAMES_GOOD_BAD_RXVLANFRGB_Pos  	(0U)
#define   ETH_RX_VLAN_FRAMES_GOOD_BAD_RXVLANFRGB_Msk  	(0xFFFFFFFFU  << ETH_RX_VLAN_FRAMES_GOOD_BAD_RXVLANFRGB_Pos)

/* ETH_RX_WATCHDOG_ERROR_FRAMES  =  Register 119 [Receive Frame Count for Watchdog Error Frames]*/
#define   ETH_RX_WATCHDOG_ERROR_FRAMES_RXWDGERR_Pos  	(0U)
#define   ETH_RX_WATCHDOG_ERROR_FRAMES_RXWDGERR_Msk  	(0xFFFFFFFFU  << ETH_RX_WATCHDOG_ERROR_FRAMES_RXWDGERR_Pos)

/* ETH_RX_RECEIVE_ERROR_FRAMES  =  Register 120 [Receive Frame Count for Receive Error Frames]*/
#define   ETH_RX_RECEIVE_ERROR_FRAMES_RXRCVERR_Pos  	(0U)
#define   ETH_RX_RECEIVE_ERROR_FRAMES_RXRCVERR_Msk  	(0xFFFFFFFFU  << ETH_RX_RECEIVE_ERROR_FRAMES_RXRCVERR_Pos)

/* ETH_RX_CONTROL_FRAMES_GOOD  =  Register 121 [Receive Frame Count for Good Control Frames Frames]*/
#define   ETH_RX_CONTROL_FRAMES_GOOD_RXCTRLG_Pos  	(0U)
#define   ETH_RX_CONTROL_FRAMES_GOOD_RXCTRLG_Msk  	(0xFFFFFFFFU  << ETH_RX_CONTROL_FRAMES_GOOD_RXCTRLG_Pos)

/* ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK  =  Register 128 [MMC Receive Checksum Offload Interrupt Mask Register]*/
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4GFIM_Pos  	(0U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4GFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4GFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4HERFIM_Pos  	(1U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4HERFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4HERFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4NOPAYFIM_Pos  	(2U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4NOPAYFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4NOPAYFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4FRAGFIM_Pos  	(3U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4FRAGFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4FRAGFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4UDSBLFIM_Pos  	(4U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4UDSBLFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4UDSBLFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6GFIM_Pos  	(5U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6GFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6GFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6HERFIM_Pos  	(6U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6HERFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6HERFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6NOPAYFIM_Pos  	(7U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6NOPAYFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6NOPAYFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPGFIM_Pos  	(8U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPGFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPGFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPERFIM_Pos  	(9U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPERFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPERFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPGFIM_Pos  	(10U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPGFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPGFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPERFIM_Pos  	(11U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPERFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPERFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPGFIM_Pos  	(12U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPGFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPGFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPERFIM_Pos  	(13U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPERFIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPERFIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_Reserved_15_14_Pos  	(14U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_Reserved_15_14_Msk  	(0x00000003U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_Reserved_15_14_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4GOIM_Pos  	(16U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4GOIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4GOIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4HEROIM_Pos  	(17U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4HEROIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4HEROIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4NOPAYOIM_Pos  	(18U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4NOPAYOIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4NOPAYOIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4FRAGOIM_Pos  	(19U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4FRAGOIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4FRAGOIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4UDSBLOIM_Pos  	(20U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4UDSBLOIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV4UDSBLOIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6GOIM_Pos  	(21U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6GOIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6GOIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6HEROIM_Pos  	(22U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6HEROIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6HEROIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6NOPAYOIM_Pos  	(23U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6NOPAYOIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXIPV6NOPAYOIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPGOIM_Pos  	(24U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPGOIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPGOIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPEROIM_Pos  	(25U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPEROIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXUDPEROIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPGOIM_Pos  	(26U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPGOIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPGOIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPEROIM_Pos  	(27U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPEROIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXTCPEROIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPGOIM_Pos  	(28U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPGOIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPGOIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPEROIM_Pos  	(29U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPEROIM_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_RXICMPEROIM_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_Reserved_31_30_Pos  	(30U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_Reserved_31_30_Msk  	(0x00000003U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_MASK_Reserved_31_30_Pos)

/* ETH_MMC_IPC_RECEIVE_INTERRUPT  =  Register 130 [MMC Receive Checksum Offload Interrupt Register]*/
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4GFIS_Pos  	(0U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4GFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4GFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4HERFIS_Pos  	(1U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4HERFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4HERFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4NOPAYFIS_Pos  	(2U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4NOPAYFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4NOPAYFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4FRAGFIS_Pos  	(3U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4FRAGFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4FRAGFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4UDSBLFIS_Pos  	(4U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4UDSBLFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4UDSBLFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6GFIS_Pos  	(5U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6GFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6GFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6HERFIS_Pos  	(6U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6HERFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6HERFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6NOPAYFIS_Pos  	(7U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6NOPAYFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6NOPAYFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPGFIS_Pos  	(8U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPGFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPGFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPERFIS_Pos  	(9U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPERFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPERFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPGFIS_Pos  	(10U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPGFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPGFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPERFIS_Pos  	(11U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPERFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPERFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPGFIS_Pos  	(12U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPGFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPGFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPERFIS_Pos  	(13U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPERFIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPERFIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_Reserved_15_14_Pos  	(14U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_Reserved_15_14_Msk  	(0x00000003U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_Reserved_15_14_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4GOIS_Pos  	(16U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4GOIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4GOIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4HEROIS_Pos  	(17U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4HEROIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4HEROIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4NOPAYOIS_Pos  	(18U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4NOPAYOIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4NOPAYOIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4FRAGOIS_Pos  	(19U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4FRAGOIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4FRAGOIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4UDSBLOIS_Pos  	(20U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4UDSBLOIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV4UDSBLOIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6GOIS_Pos  	(21U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6GOIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6GOIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6HEROIS_Pos  	(22U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6HEROIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6HEROIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6NOPAYOIS_Pos  	(23U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6NOPAYOIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXIPV6NOPAYOIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPGOIS_Pos  	(24U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPGOIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPGOIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPEROIS_Pos  	(25U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPEROIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXUDPEROIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPGOIS_Pos  	(26U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPGOIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPGOIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPEROIS_Pos  	(27U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPEROIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXTCPEROIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPGOIS_Pos  	(28U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPGOIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPGOIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPEROIS_Pos  	(29U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPEROIS_Msk  	(0x00000001U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_RXICMPEROIS_Pos)

#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_Reserved_31_30_Pos  	(30U)
#define   ETH_MMC_IPC_RECEIVE_INTERRUPT_Reserved_31_30_Msk  	(0x00000003U  << ETH_MMC_IPC_RECEIVE_INTERRUPT_Reserved_31_30_Pos)

/* ETH_RXIPV4_GOOD_FRAMES  =  */
#define   ETH_RXIPV4_GOOD_FRAMES_RXIPV4GDFRM_Pos  	(0U)
#define   ETH_RXIPV4_GOOD_FRAMES_RXIPV4GDFRM_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_GOOD_FRAMES_RXIPV4GDFRM_Pos)

/* ETH_RXIPV4_HEADER_ERROR_FRAMES  =  Register 133 [Receive IPV4 Header Error Frame Counter Register]*/
#define   ETH_RXIPV4_HEADER_ERROR_FRAMES_RXIPV4HDRERRFRM_Pos  	(0U)
#define   ETH_RXIPV4_HEADER_ERROR_FRAMES_RXIPV4HDRERRFRM_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_HEADER_ERROR_FRAMES_RXIPV4HDRERRFRM_Pos)

/* ETH_RXIPV4_NO_PAYLOAD_FRAMES  =  Register 134 [Receive IPV4 No Payload Frame Counter Register]*/
#define   ETH_RXIPV4_NO_PAYLOAD_FRAMES_RXIPV4NOPAYFRM_Pos  	(0U)
#define   ETH_RXIPV4_NO_PAYLOAD_FRAMES_RXIPV4NOPAYFRM_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_NO_PAYLOAD_FRAMES_RXIPV4NOPAYFRM_Pos)

/* ETH_RXIPV4_FRAGMENTED_FRAMES  =  Register 135 [Receive IPV4 Fragmented Frame Counter Register]*/
#define   ETH_RXIPV4_FRAGMENTED_FRAMES_RXIPV4FRAGFRM_Pos  	(0U)
#define   ETH_RXIPV4_FRAGMENTED_FRAMES_RXIPV4FRAGFRM_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_FRAGMENTED_FRAMES_RXIPV4FRAGFRM_Pos)

/* ETH_RXIPV4_UDPORT_CHECKSUM_DISABLED_FRAMES  =  Register 136 [Receive IPV4 UDP Checksum Disabled Frame Counter Register]*/
#define   ETH_RXIPV4_UDP_CHECKSUM_DISABLED_FRAMES_RXIPV4UDSBLFRM_Pos  	(0U)
#define   ETH_RXIPV4_UDP_CHECKSUM_DISABLED_FRAMES_RXIPV4UDSBLFRM_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_UDP_CHECKSUM_DISABLED_FRAMES_RXIPV4UDSBLFRM_Pos)

/* ETH_RXIPV6_GOOD_FRAMES  =  */
#define   ETH_RXIPV6_GOOD_FRAMES_RXIPV6GDFRM_Pos  	(0U)
#define   ETH_RXIPV6_GOOD_FRAMES_RXIPV6GDFRM_Msk  	(0xFFFFFFFFU  << ETH_RXIPV6_GOOD_FRAMES_RXIPV6GDFRM_Pos)

/* ETH_RXIPV6_HEADER_ERROR_FRAMES  =  Register 138 [Receive IPV6 Header Error Frame Counter Register]*/
#define   ETH_RXIPV6_HEADER_ERROR_FRAMES_RXIPV6HDRERRFRM_Pos  	(0U)
#define   ETH_RXIPV6_HEADER_ERROR_FRAMES_RXIPV6HDRERRFRM_Msk  	(0xFFFFFFFFU  << ETH_RXIPV6_HEADER_ERROR_FRAMES_RXIPV6HDRERRFRM_Pos)

/* ETH_RXIPV6_NO_PAYLOAD_FRAMES  =  Register 139 [Receive IPV6 No Payload Frame Counter Register]*/
#define   ETH_RXIPV6_NO_PAYLOAD_FRAMES_RXIPV6NOPAYFRM_Pos  	(0U)
#define   ETH_RXIPV6_NO_PAYLOAD_FRAMES_RXIPV6NOPAYFRM_Msk  	(0xFFFFFFFFU  << ETH_RXIPV6_NO_PAYLOAD_FRAMES_RXIPV6NOPAYFRM_Pos)

/* ETH_RXUDPORT_GOOD_FRAMES  =  */
#define   ETH_RXUDP_GOOD_FRAMES_RXUDPGDFRM_Pos  	(0U)
#define   ETH_RXUDP_GOOD_FRAMES_RXUDPGDFRM_Msk  	(0xFFFFFFFFU  << ETH_RXUDP_GOOD_FRAMES_RXUDPGDFRM_Pos)

/* ETH_RXUDPORT_ERROR_FRAMES  =  */
#define   ETH_RXUDP_ERROR_FRAMES_RXUDPERRFRM_Pos  	(0U)
#define   ETH_RXUDP_ERROR_FRAMES_RXUDPERRFRM_Msk  	(0xFFFFFFFFU  << ETH_RXUDP_ERROR_FRAMES_RXUDPERRFRM_Pos)

/* ETH_RXTCPORT_GOOD_FRAMES  =  */
#define   ETH_RXTCP_GOOD_FRAMES_RXTCPGDFRM_Pos  	(0U)
#define   ETH_RXTCP_GOOD_FRAMES_RXTCPGDFRM_Msk  	(0xFFFFFFFFU  << ETH_RXTCP_GOOD_FRAMES_RXTCPGDFRM_Pos)

/* ETH_RXTCPORT_ERROR_FRAMES  =  */
#define   ETH_RXTCP_ERROR_FRAMES_RXTCPERRFRM_Pos  	(0U)
#define   ETH_RXTCP_ERROR_FRAMES_RXTCPERRFRM_Msk  	(0xFFFFFFFFU  << ETH_RXTCP_ERROR_FRAMES_RXTCPERRFRM_Pos)

/* ETH_RXICMP_GOOD_FRAMES  =  */
#define   ETH_RXICMP_GOOD_FRAMES_RXICMPGDFRM_Pos  	(0U)
#define   ETH_RXICMP_GOOD_FRAMES_RXICMPGDFRM_Msk  	(0xFFFFFFFFU  << ETH_RXICMP_GOOD_FRAMES_RXICMPGDFRM_Pos)

/* ETH_RXICMP_ERROR_FRAMES  =  */
#define   ETH_RXICMP_ERROR_FRAMES_RXICMPERRFRM_Pos  	(0U)
#define   ETH_RXICMP_ERROR_FRAMES_RXICMPERRFRM_Msk  	(0xFFFFFFFFU  << ETH_RXICMP_ERROR_FRAMES_RXICMPERRFRM_Pos)

/* ETH_RXIPV4_GOOD_OCTETS  =  */
#define   ETH_RXIPV4_GOOD_OCTETS_RXIPV4GDOCT_Pos  	(0U)
#define   ETH_RXIPV4_GOOD_OCTETS_RXIPV4GDOCT_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_GOOD_OCTETS_RXIPV4GDOCT_Pos)

/* ETH_RXIPV4_HEADER_ERROR_OCTETS  =  Register 149 [Receive IPV4 Header Error Octet Counter Register]*/
#define   ETH_RXIPV4_HEADER_ERROR_OCTETS_RXIPV4HDRERROCT_Pos  	(0U)
#define   ETH_RXIPV4_HEADER_ERROR_OCTETS_RXIPV4HDRERROCT_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_HEADER_ERROR_OCTETS_RXIPV4HDRERROCT_Pos)

/* ETH_RXIPV4_NO_PAYLOAD_OCTETS  =  Register 150 [Receive IPV4 No Payload Octet Counter Register]*/
#define   ETH_RXIPV4_NO_PAYLOAD_OCTETS_RXIPV4NOPAYOCT_Pos  	(0U)
#define   ETH_RXIPV4_NO_PAYLOAD_OCTETS_RXIPV4NOPAYOCT_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_NO_PAYLOAD_OCTETS_RXIPV4NOPAYOCT_Pos)

/* ETH_RXIPV4_FRAGMENTED_OCTETS  =  Register 151 [Receive IPV4 Fragmented Octet Counter Register]*/
#define   ETH_RXIPV4_FRAGMENTED_OCTETS_RXIPV4FRAGOCT_Pos  	(0U)
#define   ETH_RXIPV4_FRAGMENTED_OCTETS_RXIPV4FRAGOCT_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_FRAGMENTED_OCTETS_RXIPV4FRAGOCT_Pos)

/* ETH_RXIPV4_UDPORT_CHECKSUM_DISABLE_OCTETS  =  Register 152 [Receive IPV4 Fragmented Octet Counter Register]*/
#define   ETH_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS_RXIPV4UDSBLOCT_Pos  	(0U)
#define   ETH_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS_RXIPV4UDSBLOCT_Msk  	(0xFFFFFFFFU  << ETH_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS_RXIPV4UDSBLOCT_Pos)

/* ETH_RXIPV6_GOOD_OCTETS  =  */
#define   ETH_RXIPV6_GOOD_OCTETS_RXIPV6GDOCT_Pos  	(0U)
#define   ETH_RXIPV6_GOOD_OCTETS_RXIPV6GDOCT_Msk  	(0xFFFFFFFFU  << ETH_RXIPV6_GOOD_OCTETS_RXIPV6GDOCT_Pos)

/* ETH_RXIPV6_HEADER_ERROR_OCTETS  =  Register 154 [Receive IPV6 Header Error Octet Counter Register]*/
#define   ETH_RXIPV6_HEADER_ERROR_OCTETS_RXIPV6HDRERROCT_Pos  	(0U)
#define   ETH_RXIPV6_HEADER_ERROR_OCTETS_RXIPV6HDRERROCT_Msk  	(0xFFFFFFFFU  << ETH_RXIPV6_HEADER_ERROR_OCTETS_RXIPV6HDRERROCT_Pos)

/* ETH_RXIPV6_NO_PAYLOAD_OCTETS  =  Register 155 [Receive IPV6 No Payload Octet Counter Register]*/
#define   ETH_RXIPV6_NO_PAYLOAD_OCTETS_RXIPV6NOPAYOCT_Pos  	(0U)
#define   ETH_RXIPV6_NO_PAYLOAD_OCTETS_RXIPV6NOPAYOCT_Msk  	(0xFFFFFFFFU  << ETH_RXIPV6_NO_PAYLOAD_OCTETS_RXIPV6NOPAYOCT_Pos)

/* ETH_RXUDPORT_GOOD_OCTETS  =  */
#define   ETH_RXUDP_GOOD_OCTETS_RXUDPGDOCT_Pos  	(0U)
#define   ETH_RXUDP_GOOD_OCTETS_RXUDPGDOCT_Msk  	(0xFFFFFFFFU  << ETH_RXUDP_GOOD_OCTETS_RXUDPGDOCT_Pos)

/* ETH_RXUDPORT_ERROR_OCTETS  =  */
#define   ETH_RXUDP_ERROR_OCTETS_RXUDPERROCT_Pos  	(0U)
#define   ETH_RXUDP_ERROR_OCTETS_RXUDPERROCT_Msk  	(0xFFFFFFFFU  << ETH_RXUDP_ERROR_OCTETS_RXUDPERROCT_Pos)

/* ETH_RXTCPORT_GOOD_OCTETS  =  */
#define   ETH_RXTCP_GOOD_OCTETS_RXTCPGDOCT_Pos  	(0U)
#define   ETH_RXTCP_GOOD_OCTETS_RXTCPGDOCT_Msk  	(0xFFFFFFFFU  << ETH_RXTCP_GOOD_OCTETS_RXTCPGDOCT_Pos)

/* ETH_RXTCPORT_ERROR_OCTETS  =  */
#define   ETH_RXTCP_ERROR_OCTETS_RXTCPERROCT_Pos  	(0U)
#define   ETH_RXTCP_ERROR_OCTETS_RXTCPERROCT_Msk  	(0xFFFFFFFFU  << ETH_RXTCP_ERROR_OCTETS_RXTCPERROCT_Pos)

/* ETH_RXICMP_GOOD_OCTETS  =  */
#define   ETH_RXICMP_GOOD_OCTETS_RXICMPGDOCT_Pos  	(0U)
#define   ETH_RXICMP_GOOD_OCTETS_RXICMPGDOCT_Msk  	(0xFFFFFFFFU  << ETH_RXICMP_GOOD_OCTETS_RXICMPGDOCT_Pos)

/* ETH_RXICMP_ERROR_OCTETS  =  */
#define   ETH_RXICMP_ERROR_OCTETS_RXICMPERROCT_Pos  	(0U)
#define   ETH_RXICMP_ERROR_OCTETS_RXICMPERROCT_Msk  	(0xFFFFFFFFU  << ETH_RXICMP_ERROR_OCTETS_RXICMPERROCT_Pos)

/* ETH_TIMESTAMP_CONTROL  =  Register 448 [Timestamp Control Register]*/
#define   ETH_TIMESTAMP_CONTROL_TSENA_Pos  	(0U)
#define   ETH_TIMESTAMP_CONTROL_TSENA_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSENA_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSCFUPDT_Pos  	(1U)
#define   ETH_TIMESTAMP_CONTROL_TSCFUPDT_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSCFUPDT_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSINIT_Pos  	(2U)
#define   ETH_TIMESTAMP_CONTROL_TSINIT_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSINIT_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSUPDT_Pos  	(3U)
#define   ETH_TIMESTAMP_CONTROL_TSUPDT_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSUPDT_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSTRIG_Pos  	(4U)
#define   ETH_TIMESTAMP_CONTROL_TSTRIG_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSTRIG_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSADDREG_Pos  	(5U)
#define   ETH_TIMESTAMP_CONTROL_TSADDREG_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSADDREG_Pos)

#define   ETH_TIMESTAMP_CONTROL_Reserved_7_6_Pos  	(6U)
#define   ETH_TIMESTAMP_CONTROL_Reserved_7_6_Msk  	(0x00000003U  << ETH_TIMESTAMP_CONTROL_Reserved_7_6_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSENALL_Pos  	(8U)
#define   ETH_TIMESTAMP_CONTROL_TSENALL_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSENALL_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSCTRLSSR_Pos  	(9U)
#define   ETH_TIMESTAMP_CONTROL_TSCTRLSSR_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSCTRLSSR_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSVER2ENA_Pos  	(10U)
#define   ETH_TIMESTAMP_CONTROL_TSVER2ENA_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSVER2ENA_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSIPENA_Pos  	(11U)
#define   ETH_TIMESTAMP_CONTROL_TSIPENA_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSIPENA_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSIPV6ENA_Pos  	(12U)
#define   ETH_TIMESTAMP_CONTROL_TSIPV6ENA_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSIPV6ENA_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSIPV4ENA_Pos  	(13U)
#define   ETH_TIMESTAMP_CONTROL_TSIPV4ENA_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSIPV4ENA_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSEVNTENA_Pos  	(14U)
#define   ETH_TIMESTAMP_CONTROL_TSEVNTENA_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSEVNTENA_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSMSTRENA_Pos  	(15U)
#define   ETH_TIMESTAMP_CONTROL_TSMSTRENA_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSMSTRENA_Pos)

#define   ETH_TIMESTAMP_CONTROL_SNAPTYPSEL_Pos  	(16U)
#define   ETH_TIMESTAMP_CONTROL_SNAPTYPSEL_Msk  	(0x00000003U  << ETH_TIMESTAMP_CONTROL_SNAPTYPSEL_Pos)

#define   ETH_TIMESTAMP_CONTROL_TSENMACADDR_Pos  	(18U)
#define   ETH_TIMESTAMP_CONTROL_TSENMACADDR_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_TSENMACADDR_Pos)

#define   ETH_TIMESTAMP_CONTROL_Reserved_23_19_Pos  	(19U)
#define   ETH_TIMESTAMP_CONTROL_Reserved_23_19_Msk  	(0x0000001FU  << ETH_TIMESTAMP_CONTROL_Reserved_23_19_Pos)

#define   ETH_TIMESTAMP_CONTROL_ATSFC_Pos  	(24U)
#define   ETH_TIMESTAMP_CONTROL_ATSFC_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_ATSFC_Pos)

#define   ETH_TIMESTAMP_CONTROL_ATSEN0_Pos  	(25U)
#define   ETH_TIMESTAMP_CONTROL_ATSEN0_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_ATSEN0_Pos)

#define   ETH_TIMESTAMP_CONTROL_ATSEN1_Pos  	(26U)
#define   ETH_TIMESTAMP_CONTROL_ATSEN1_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_ATSEN1_Pos)

#define   ETH_TIMESTAMP_CONTROL_ATSEN2_Pos  	(27U)
#define   ETH_TIMESTAMP_CONTROL_ATSEN2_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_ATSEN2_Pos)

#define   ETH_TIMESTAMP_CONTROL_ATSEN3_Pos  	(28U)
#define   ETH_TIMESTAMP_CONTROL_ATSEN3_Msk  	(0x00000001U  << ETH_TIMESTAMP_CONTROL_ATSEN3_Pos)

#define   ETH_TIMESTAMP_CONTROL_Reserved_31_29_Pos  	(29U)
#define   ETH_TIMESTAMP_CONTROL_Reserved_31_29_Msk  	(0x00000007U  << ETH_TIMESTAMP_CONTROL_Reserved_31_29_Pos)

/* ETH_SUB_SECOND_INCREMENT  =  Register 449 [Sub-Second Increment Register]*/
#define   ETH_SUB_SECOND_INCREMENT_SSINC_Pos  	(0U)
#define   ETH_SUB_SECOND_INCREMENT_SSINC_Msk  	(0x000000FFU  << ETH_SUB_SECOND_INCREMENT_SSINC_Pos)

#define   ETH_SUB_SECOND_INCREMENT_Reserved_31_8_Pos  	(8U)
#define   ETH_SUB_SECOND_INCREMENT_Reserved_31_8_Msk  	(0x00FFFFFFU  << ETH_SUB_SECOND_INCREMENT_Reserved_31_8_Pos)

/* ETH_SYSTEM_TIME_SECONDS  =  Register 450 [System Time - Seconds Register]*/
#define   ETH_SYSTEM_TIME_SECONDS_TSS_Pos  	(0U)
#define   ETH_SYSTEM_TIME_SECONDS_TSS_Msk  	(0xFFFFFFFFU  << ETH_SYSTEM_TIME_SECONDS_TSS_Pos)

/* ETH_SYSTEM_TIME_NANOSECONDS  =  */
#define   ETH_SYSTEM_TIME_NANOSECONDS_TSSS_Pos  	(0U)
#define   ETH_SYSTEM_TIME_NANOSECONDS_TSSS_Msk  	(0x7FFFFFFFU  << ETH_SYSTEM_TIME_NANOSECONDS_TSSS_Pos)

#define   ETH_SYSTEM_TIME_NANOSECONDS_Reserved_31_Pos  	(31U)
#define   ETH_SYSTEM_TIME_NANOSECONDS_Reserved_31_Msk  	(0x00000001U  << ETH_SYSTEM_TIME_NANOSECONDS_Reserved_31_Pos)

/* ETH_SYSTEM_TIME_SECONDS_UPDATE  =  Register 452 [System Time - Seconds Update Register]*/
#define   ETH_SYSTEM_TIME_SECONDS_UPDATE_TSS_Pos  	(0U)
#define   ETH_SYSTEM_TIME_SECONDS_UPDATE_TSS_Msk  	(0xFFFFFFFFU  << ETH_SYSTEM_TIME_SECONDS_UPDATE_TSS_Pos)

/* ETH_SYSTEM_TIME_NANOSECONDS_UPDATE  =  */
#define   ETH_SYSTEM_TIME_NANOSECONDS_UPDATE_TSSS_Pos  	(0U)
#define   ETH_SYSTEM_TIME_NANOSECONDS_UPDATE_TSSS_Msk  	(0x7FFFFFFFU  << ETH_SYSTEM_TIME_NANOSECONDS_UPDATE_TSSS_Pos)

#define   ETH_SYSTEM_TIME_NANOSECONDS_UPDATE_ADDSUB_Pos  	(31U)
#define   ETH_SYSTEM_TIME_NANOSECONDS_UPDATE_ADDSUB_Msk  	(0x00000001U  << ETH_SYSTEM_TIME_NANOSECONDS_UPDATE_ADDSUB_Pos)

/* ETH_TIMESTAMP_ADDEND  =  Register 454 [Timestamp Addend Register]*/
#define   ETH_TIMESTAMP_ADDEND_TSAR_Pos  	(0U)
#define   ETH_TIMESTAMP_ADDEND_TSAR_Msk  	(0xFFFFFFFFU  << ETH_TIMESTAMP_ADDEND_TSAR_Pos)

/* ETH_TARGET_TIME_SECONDS  =  Register 455 [Target Time Seconds Register]*/
#define   ETH_TARGET_TIME_SECONDS_TSTR_Pos  	(0U)
#define   ETH_TARGET_TIME_SECONDS_TSTR_Msk  	(0xFFFFFFFFU  << ETH_TARGET_TIME_SECONDS_TSTR_Pos)

/* ETH_TARGET_TIME_NANOSECONDS  =  */
#define   ETH_TARGET_TIME_NANOSECONDS_TTSLO_Pos  	(0U)
#define   ETH_TARGET_TIME_NANOSECONDS_TTSLO_Msk  	(0x7FFFFFFFU  << ETH_TARGET_TIME_NANOSECONDS_TTSLO_Pos)

#define   ETH_TARGET_TIME_NANOSECONDS_TRGTBUSY_Pos  	(31U)
#define   ETH_TARGET_TIME_NANOSECONDS_TRGTBUSY_Msk  	(0x00000001U  << ETH_TARGET_TIME_NANOSECONDS_TRGTBUSY_Pos)

/* ETH_SYSTEM_TIME_HIGHER_WORD_SECONDS  =  Register 457 [System Time - Higher Word Seconds Register]*/
#define   ETH_SYSTEM_TIME_HIGHER_WORD_SECONDS_TSHWR_Pos  	(0U)
#define   ETH_SYSTEM_TIME_HIGHER_WORD_SECONDS_TSHWR_Msk  	(0x0000FFFFU  << ETH_SYSTEM_TIME_HIGHER_WORD_SECONDS_TSHWR_Pos)

#define   ETH_SYSTEM_TIME_HIGHER_WORD_SECONDS_Reserved_31_16_Pos  	(16U)
#define   ETH_SYSTEM_TIME_HIGHER_WORD_SECONDS_Reserved_31_16_Msk  	(0x0000FFFFU  << ETH_SYSTEM_TIME_HIGHER_WORD_SECONDS_Reserved_31_16_Pos)

/* ETH_TIMESTAMP_STATUS  =  Register 458 [Timestamp Status Register]*/
#define   ETH_TIMESTAMP_STATUS_TSSOVF_Pos  	(0U)
#define   ETH_TIMESTAMP_STATUS_TSSOVF_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_TSSOVF_Pos)

#define   ETH_TIMESTAMP_STATUS_TSTARGT_Pos  	(1U)
#define   ETH_TIMESTAMP_STATUS_TSTARGT_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_TSTARGT_Pos)

#define   ETH_TIMESTAMP_STATUS_AUXTSTRIG_Pos  	(2U)
#define   ETH_TIMESTAMP_STATUS_AUXTSTRIG_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_AUXTSTRIG_Pos)

#define   ETH_TIMESTAMP_STATUS_TSTRGTERR_Pos  	(3U)
#define   ETH_TIMESTAMP_STATUS_TSTRGTERR_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_TSTRGTERR_Pos)

#define   ETH_TIMESTAMP_STATUS_TSTARGT1_Pos  	(4U)
#define   ETH_TIMESTAMP_STATUS_TSTARGT1_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_TSTARGT1_Pos)

#define   ETH_TIMESTAMP_STATUS_TSTRGTERR1_Pos  	(5U)
#define   ETH_TIMESTAMP_STATUS_TSTRGTERR1_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_TSTRGTERR1_Pos)

#define   ETH_TIMESTAMP_STATUS_TSTARGT2_Pos  	(6U)
#define   ETH_TIMESTAMP_STATUS_TSTARGT2_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_TSTARGT2_Pos)

#define   ETH_TIMESTAMP_STATUS_TSTRGTERR2_Pos  	(7U)
#define   ETH_TIMESTAMP_STATUS_TSTRGTERR2_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_TSTRGTERR2_Pos)

#define   ETH_TIMESTAMP_STATUS_TSTARGT3_Pos  	(8U)
#define   ETH_TIMESTAMP_STATUS_TSTARGT3_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_TSTARGT3_Pos)

#define   ETH_TIMESTAMP_STATUS_TSTRGTERR3_Pos  	(9U)
#define   ETH_TIMESTAMP_STATUS_TSTRGTERR3_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_TSTRGTERR3_Pos)

#define   ETH_TIMESTAMP_STATUS_Reserved_15_10_Pos  	(10U)
#define   ETH_TIMESTAMP_STATUS_Reserved_15_10_Msk  	(0x0000003FU  << ETH_TIMESTAMP_STATUS_Reserved_15_10_Pos)

#define   ETH_TIMESTAMP_STATUS_ATSSTN_Pos  	(16U)
#define   ETH_TIMESTAMP_STATUS_ATSSTN_Msk  	(0x0000000FU  << ETH_TIMESTAMP_STATUS_ATSSTN_Pos)

#define   ETH_TIMESTAMP_STATUS_Reserved_23_20_Pos  	(20U)
#define   ETH_TIMESTAMP_STATUS_Reserved_23_20_Msk  	(0x0000000FU  << ETH_TIMESTAMP_STATUS_Reserved_23_20_Pos)

#define   ETH_TIMESTAMP_STATUS_ATSSTM_Pos  	(24U)
#define   ETH_TIMESTAMP_STATUS_ATSSTM_Msk  	(0x00000001U  << ETH_TIMESTAMP_STATUS_ATSSTM_Pos)

#define   ETH_TIMESTAMP_STATUS_ATSNS_Pos  	(25U)
#define   ETH_TIMESTAMP_STATUS_ATSNS_Msk  	(0x0000001FU  << ETH_TIMESTAMP_STATUS_ATSNS_Pos)

#define   ETH_TIMESTAMP_STATUS_Reserved_31_30_Pos  	(30U)
#define   ETH_TIMESTAMP_STATUS_Reserved_31_30_Msk  	(0x00000003U  << ETH_TIMESTAMP_STATUS_Reserved_31_30_Pos)

/* ETH_PPS_CONTROL  =  Register 459 [PPS Control Register]*/
#define   ETH_PPS_CONTROL_PPSCTRL_PPSCMD_Pos  	(0U)
#define   ETH_PPS_CONTROL_PPSCTRL_PPSCMD_Msk  	(0x0000000FU  << ETH_PPS_CONTROL_PPSCTRL_PPSCMD_Pos)

#define   ETH_PPS_CONTROL_PPSEN0_Pos  	(4U)
#define   ETH_PPS_CONTROL_PPSEN0_Msk  	(0x00000001U  << ETH_PPS_CONTROL_PPSEN0_Pos)

#define   ETH_PPS_CONTROL_TRGTMODSEL0_Pos  	(5U)
#define   ETH_PPS_CONTROL_TRGTMODSEL0_Msk  	(0x00000003U  << ETH_PPS_CONTROL_TRGTMODSEL0_Pos)

#define   ETH_PPS_CONTROL_Reserved_7_Pos  	(7U)
#define   ETH_PPS_CONTROL_Reserved_7_Msk  	(0x00000001U  << ETH_PPS_CONTROL_Reserved_7_Pos)

#define   ETH_PPS_CONTROL_PPSCMD1_Pos  	(8U)
#define   ETH_PPS_CONTROL_PPSCMD1_Msk  	(0x00000007U  << ETH_PPS_CONTROL_PPSCMD1_Pos)

#define   ETH_PPS_CONTROL_Reserved_12_11_Pos  	(11U)
#define   ETH_PPS_CONTROL_Reserved_12_11_Msk  	(0x00000003U  << ETH_PPS_CONTROL_Reserved_12_11_Pos)

#define   ETH_PPS_CONTROL_TRGTMODSEL1_Pos  	(13U)
#define   ETH_PPS_CONTROL_TRGTMODSEL1_Msk  	(0x00000003U  << ETH_PPS_CONTROL_TRGTMODSEL1_Pos)

#define   ETH_PPS_CONTROL_Reserved_15_Pos  	(15U)
#define   ETH_PPS_CONTROL_Reserved_15_Msk  	(0x00000001U  << ETH_PPS_CONTROL_Reserved_15_Pos)

#define   ETH_PPS_CONTROL_PPSCMD2_Pos  	(16U)
#define   ETH_PPS_CONTROL_PPSCMD2_Msk  	(0x00000007U  << ETH_PPS_CONTROL_PPSCMD2_Pos)

#define   ETH_PPS_CONTROL_Reserved_20_19_Pos  	(19U)
#define   ETH_PPS_CONTROL_Reserved_20_19_Msk  	(0x00000003U  << ETH_PPS_CONTROL_Reserved_20_19_Pos)

#define   ETH_PPS_CONTROL_TRGTMODSEL2_Pos  	(21U)
#define   ETH_PPS_CONTROL_TRGTMODSEL2_Msk  	(0x00000003U  << ETH_PPS_CONTROL_TRGTMODSEL2_Pos)

#define   ETH_PPS_CONTROL_Reserved_23_Pos  	(23U)
#define   ETH_PPS_CONTROL_Reserved_23_Msk  	(0x00000001U  << ETH_PPS_CONTROL_Reserved_23_Pos)

#define   ETH_PPS_CONTROL_PPSCMD3_Pos  	(24U)
#define   ETH_PPS_CONTROL_PPSCMD3_Msk  	(0x00000007U  << ETH_PPS_CONTROL_PPSCMD3_Pos)

#define   ETH_PPS_CONTROL_Reserved_28_27_Pos  	(27U)
#define   ETH_PPS_CONTROL_Reserved_28_27_Msk  	(0x00000003U  << ETH_PPS_CONTROL_Reserved_28_27_Pos)

#define   ETH_PPS_CONTROL_TRGTMODSEL3_Pos  	(29U)
#define   ETH_PPS_CONTROL_TRGTMODSEL3_Msk  	(0x00000003U  << ETH_PPS_CONTROL_TRGTMODSEL3_Pos)

#define   ETH_PPS_CONTROL_Reserved_31_Pos  	(31U)
#define   ETH_PPS_CONTROL_Reserved_31_Msk  	(0x00000001U  << ETH_PPS_CONTROL_Reserved_31_Pos)

/* ETH_BUS_MODE  =  Register x [Bus Mode Register]*/
#define   ETH_BUS_MODE_SWR_Pos  	(0U)
#define   ETH_BUS_MODE_SWR_Msk  	(0x00000001U  << ETH_BUS_MODE_SWR_Pos)

#define   ETH_BUS_MODE_DA_Pos  	(1U)
#define   ETH_BUS_MODE_DA_Msk  	(0x00000001U  << ETH_BUS_MODE_DA_Pos)

#define   ETH_BUS_MODE_DSL_Pos  	(2U)
#define   ETH_BUS_MODE_DSL_Msk  	(0x0000001FU  << ETH_BUS_MODE_DSL_Pos)

#define   ETH_BUS_MODE_Reserved_7_Pos  	(7U)
#define   ETH_BUS_MODE_Reserved_7_Msk  	(0x00000001U  << ETH_BUS_MODE_Reserved_7_Pos)

#define   ETH_BUS_MODE_PBL_Pos  	(8U)
#define   ETH_BUS_MODE_PBL_Msk  	(0x0000003FU  << ETH_BUS_MODE_PBL_Pos)

#define   ETH_BUS_MODE_PR_Pos  	(14U)
#define   ETH_BUS_MODE_PR_Msk  	(0x00000003U  << ETH_BUS_MODE_PR_Pos)

#define   ETH_BUS_MODE_FB_Pos  	(16U)
#define   ETH_BUS_MODE_FB_Msk  	(0x00000001U  << ETH_BUS_MODE_FB_Pos)

#define   ETH_BUS_MODE_RPBL_Pos  	(17U)
#define   ETH_BUS_MODE_RPBL_Msk  	(0x0000003FU  << ETH_BUS_MODE_RPBL_Pos)

#define   ETH_BUS_MODE_USP_Pos  	(23U)
#define   ETH_BUS_MODE_USP_Msk  	(0x00000001U  << ETH_BUS_MODE_USP_Pos)

#define   ETH_BUS_MODE_EIGHTxPBL_Pos  	(24U)
#define   ETH_BUS_MODE_EIGHTxPBL_Msk  	(0x00000001U  << ETH_BUS_MODE_EIGHTxPBL_Pos)

#define   ETH_BUS_MODE_AAL_Pos  	(25U)
#define   ETH_BUS_MODE_AAL_Msk  	(0x00000001U  << ETH_BUS_MODE_AAL_Pos)

#define   ETH_BUS_MODE_MB_Pos  	(26U)
#define   ETH_BUS_MODE_MB_Msk  	(0x00000001U  << ETH_BUS_MODE_MB_Pos)

#define   ETH_BUS_MODE_TXPR_Pos  	(27U)
#define   ETH_BUS_MODE_TXPR_Msk  	(0x00000001U  << ETH_BUS_MODE_TXPR_Pos)

#define   ETH_BUS_MODE_PRWG_Pos  	(28U)
#define   ETH_BUS_MODE_PRWG_Msk  	(0x00000003U  << ETH_BUS_MODE_PRWG_Pos)

#define   ETH_BUS_MODE_Reserved_31_30_Pos  	(30U)
#define   ETH_BUS_MODE_Reserved_31_30_Msk  	(0x00000003U  << ETH_BUS_MODE_Reserved_31_30_Pos)

/* ETH_TRANSMIT_POLL_DEMAND  =  Register 1 [Transmit Poll Demand Register]*/
#define   ETH_TRANSMIT_POLL_DEMAND_TPD_Pos  	(0U)
#define   ETH_TRANSMIT_POLL_DEMAND_TPD_Msk  	(0xFFFFFFFFU  << ETH_TRANSMIT_POLL_DEMAND_TPD_Pos)

/* ETH_RECEIVE_POLL_DEMAND  =  */
#define   ETH_RECEIVE_POLL_DEMAND_RPD_Pos  	(0U)
#define   ETH_RECEIVE_POLL_DEMAND_RPD_Msk  	(0xFFFFFFFFU  << ETH_RECEIVE_POLL_DEMAND_RPD_Pos)

/* ETH_RECEIVE_DESCRIPTOR_LIST_ADDRESS  =  */
#define   ETH_RECEIVE_DESCRIPTOR_LIST_ADDRESS_Reserved_1_0_Pos  	(0U)
#define   ETH_RECEIVE_DESCRIPTOR_LIST_ADDRESS_Reserved_1_0_Msk  	(0x00000003U  << ETH_RECEIVE_DESCRIPTOR_LIST_ADDRESS_Reserved_1_0_Pos)

#define   ETH_RECEIVE_DESCRIPTOR_LIST_ADDRESS_RDESLA_32bit_Pos  	(2U)
#define   ETH_RECEIVE_DESCRIPTOR_LIST_ADDRESS_RDESLA_32bit_Msk  	(0x3FFFFFFFU  << ETH_RECEIVE_DESCRIPTOR_LIST_ADDRESS_RDESLA_32bit_Pos)

/* ETH_TRANSMIT_DESCRIPTOR_LIST_ADDRESS  =  */
#define   ETH_TRANSMIT_DESCRIPTOR_LIST_ADDRESS_Reserved_1_0_Pos  	(0U)
#define   ETH_TRANSMIT_DESCRIPTOR_LIST_ADDRESS_Reserved_1_0_Msk  	(0x00000003U  << ETH_TRANSMIT_DESCRIPTOR_LIST_ADDRESS_Reserved_1_0_Pos)

#define   ETH_TRANSMIT_DESCRIPTOR_LIST_ADDRESS_TDESLA_32bit_Pos  	(2U)
#define   ETH_TRANSMIT_DESCRIPTOR_LIST_ADDRESS_TDESLA_32bit_Msk  	(0x3FFFFFFFU  << ETH_TRANSMIT_DESCRIPTOR_LIST_ADDRESS_TDESLA_32bit_Pos)

/* ETH_STATUS  =  */
#define   ETH_STATUS_TI_Pos  	(0U)
#define   ETH_STATUS_TI_Msk  	(0x00000001U  << ETH_STATUS_TI_Pos)

#define   ETH_STATUS_TPS_Pos  	(1U)
#define   ETH_STATUS_TPS_Msk  	(0x00000001U  << ETH_STATUS_TPS_Pos)

#define   ETH_STATUS_TU_Pos  	(2U)
#define   ETH_STATUS_TU_Msk  	(0x00000001U  << ETH_STATUS_TU_Pos)

#define   ETH_STATUS_TJT_Pos  	(3U)
#define   ETH_STATUS_TJT_Msk  	(0x00000001U  << ETH_STATUS_TJT_Pos)

#define   ETH_STATUS_OVF_Pos  	(4U)
#define   ETH_STATUS_OVF_Msk  	(0x00000001U  << ETH_STATUS_OVF_Pos)

#define   ETH_STATUS_UNF_Pos  	(5U)
#define   ETH_STATUS_UNF_Msk  	(0x00000001U  << ETH_STATUS_UNF_Pos)

#define   ETH_STATUS_RI_Pos  	(6U)
#define   ETH_STATUS_RI_Msk  	(0x00000001U  << ETH_STATUS_RI_Pos)

#define   ETH_STATUS_RU_Pos  	(7U)
#define   ETH_STATUS_RU_Msk  	(0x00000001U  << ETH_STATUS_RU_Pos)

#define   ETH_STATUS_RPS_Pos  	(8U)
#define   ETH_STATUS_RPS_Msk  	(0x00000001U  << ETH_STATUS_RPS_Pos)

#define   ETH_STATUS_RWT_Pos  	(9U)
#define   ETH_STATUS_RWT_Msk  	(0x00000001U  << ETH_STATUS_RWT_Pos)

#define   ETH_STATUS_ETI_Pos  	(10U)
#define   ETH_STATUS_ETI_Msk  	(0x00000001U  << ETH_STATUS_ETI_Pos)

#define   ETH_STATUS_Reserved_12_11_Pos  	(11U)
#define   ETH_STATUS_Reserved_12_11_Msk  	(0x00000003U  << ETH_STATUS_Reserved_12_11_Pos)

#define   ETH_STATUS_FBI_Pos  	(13U)
#define   ETH_STATUS_FBI_Msk  	(0x00000001U  << ETH_STATUS_FBI_Pos)

#define   ETH_STATUS_ERI_Pos  	(14U)
#define   ETH_STATUS_ERI_Msk  	(0x00000001U  << ETH_STATUS_ERI_Pos)

#define   ETH_STATUS_AIS_Pos  	(15U)
#define   ETH_STATUS_AIS_Msk  	(0x00000001U  << ETH_STATUS_AIS_Pos)

#define   ETH_STATUS_NIS_Pos  	(16U)
#define   ETH_STATUS_NIS_Msk  	(0x00000001U  << ETH_STATUS_NIS_Pos)

#define   ETH_STATUS_RS_Pos  	(17U)
#define   ETH_STATUS_RS_Msk  	(0x00000007U  << ETH_STATUS_RS_Pos)

#define   ETH_STATUS_TS_Pos  	(20U)
#define   ETH_STATUS_TS_Msk  	(0x00000007U  << ETH_STATUS_TS_Pos)

#define   ETH_STATUS_EB_Pos  	(23U)
#define   ETH_STATUS_EB_Msk  	(0x00000007U  << ETH_STATUS_EB_Pos)

#define   ETH_STATUS_MLI_Pos  	(26U)
#define   ETH_STATUS_MLI_Msk  	(0x00000001U  << ETH_STATUS_MLI_Pos)

#define   ETH_STATUS_EMI_Pos  	(27U)
#define   ETH_STATUS_EMI_Msk  	(0x00000001U  << ETH_STATUS_EMI_Pos)

#define   ETH_STATUS_EPI_Pos  	(28U)
#define   ETH_STATUS_EPI_Msk  	(0x00000001U  << ETH_STATUS_EPI_Pos)

#define   ETH_STATUS_TTI_Pos  	(29U)
#define   ETH_STATUS_TTI_Msk  	(0x00000001U  << ETH_STATUS_TTI_Pos)

#define   ETH_STATUS_ELPII_Pos  	(30U)
#define   ETH_STATUS_ELPII_Msk  	(0x00000001U  << ETH_STATUS_ELPII_Pos)

#define   ETH_STATUS_Reserved_31_Pos  	(31U)
#define   ETH_STATUS_Reserved_31_Msk  	(0x00000001U  << ETH_STATUS_Reserved_31_Pos)

/* ETH_OPERATION_MODE  =  Register 6 [Operation Mode Register]*/
#define   ETH_OPERATION_MODE_Reserved_0_Pos  	(0U)
#define   ETH_OPERATION_MODE_Reserved_0_Msk  	(0x00000001U  << ETH_OPERATION_MODE_Reserved_0_Pos)

#define   ETH_OPERATION_MODE_SR_Pos  	(1U)
#define   ETH_OPERATION_MODE_SR_Msk  	(0x00000001U  << ETH_OPERATION_MODE_SR_Pos)

#define   ETH_OPERATION_MODE_OSF_Pos  	(2U)
#define   ETH_OPERATION_MODE_OSF_Msk  	(0x00000001U  << ETH_OPERATION_MODE_OSF_Pos)

#define   ETH_OPERATION_MODE_RTC_Pos  	(3U)
#define   ETH_OPERATION_MODE_RTC_Msk  	(0x00000003U  << ETH_OPERATION_MODE_RTC_Pos)

#define   ETH_OPERATION_MODE_Reserved_5_Pos  	(5U)
#define   ETH_OPERATION_MODE_Reserved_5_Msk  	(0x00000001U  << ETH_OPERATION_MODE_Reserved_5_Pos)

#define   ETH_OPERATION_MODE_FUF_Pos  	(6U)
#define   ETH_OPERATION_MODE_FUF_Msk  	(0x00000001U  << ETH_OPERATION_MODE_FUF_Pos)

#define   ETH_OPERATION_MODE_FEF_Pos  	(7U)
#define   ETH_OPERATION_MODE_FEF_Msk  	(0x00000001U  << ETH_OPERATION_MODE_FEF_Pos)

#define   ETH_OPERATION_MODE_EFC_Pos  	(8U)
#define   ETH_OPERATION_MODE_EFC_Msk  	(0x00000001U  << ETH_OPERATION_MODE_EFC_Pos)

#define   ETH_OPERATION_MODE_RFA_Pos  	(9U)
#define   ETH_OPERATION_MODE_RFA_Msk  	(0x00000003U  << ETH_OPERATION_MODE_RFA_Pos)

#define   ETH_OPERATION_MODE_RFD_Pos  	(11U)
#define   ETH_OPERATION_MODE_RFD_Msk  	(0x00000003U  << ETH_OPERATION_MODE_RFD_Pos)

#define   ETH_OPERATION_MODE_ST_Pos  	(13U)
#define   ETH_OPERATION_MODE_ST_Msk  	(0x00000001U  << ETH_OPERATION_MODE_ST_Pos)

#define   ETH_OPERATION_MODE_TTC_Pos  	(14U)
#define   ETH_OPERATION_MODE_TTC_Msk  	(0x00000007U  << ETH_OPERATION_MODE_TTC_Pos)

#define   ETH_OPERATION_MODE_Reserved_19_17_Pos  	(17U)
#define   ETH_OPERATION_MODE_Reserved_19_17_Msk  	(0x00000007U  << ETH_OPERATION_MODE_Reserved_19_17_Pos)

#define   ETH_OPERATION_MODE_FTF_Pos  	(20U)
#define   ETH_OPERATION_MODE_FTF_Msk  	(0x00000001U  << ETH_OPERATION_MODE_FTF_Pos)

#define   ETH_OPERATION_MODE_TSF_Pos  	(21U)
#define   ETH_OPERATION_MODE_TSF_Msk  	(0x00000001U  << ETH_OPERATION_MODE_TSF_Pos)

#define   ETH_OPERATION_MODE_RFD2_Pos  	(22U)
#define   ETH_OPERATION_MODE_RFD2_Msk  	(0x00000001U  << ETH_OPERATION_MODE_RFD2_Pos)

#define   ETH_OPERATION_MODE_RFA2_Pos  	(23U)
#define   ETH_OPERATION_MODE_RFA2_Msk  	(0x00000001U  << ETH_OPERATION_MODE_RFA2_Pos)

#define   ETH_OPERATION_MODE_DFF_Pos  	(24U)
#define   ETH_OPERATION_MODE_DFF_Msk  	(0x00000001U  << ETH_OPERATION_MODE_DFF_Pos)

#define   ETH_OPERATION_MODE_RSF_Pos  	(25U)
#define   ETH_OPERATION_MODE_RSF_Msk  	(0x00000001U  << ETH_OPERATION_MODE_RSF_Pos)

#define   ETH_OPERATION_MODE_DT_Pos  	(26U)
#define   ETH_OPERATION_MODE_DT_Msk  	(0x00000001U  << ETH_OPERATION_MODE_DT_Pos)

#define   ETH_OPERATION_MODE_Reserved_31_27_Pos  	(27U)
#define   ETH_OPERATION_MODE_Reserved_31_27_Msk  	(0x0000001FU  << ETH_OPERATION_MODE_Reserved_31_27_Pos)

/* ETH_INTERRUPT_ENABLE  =  Register 7 [Interrupt Enable Register]*/
#define   ETH_INTERRUPT_ENABLE_TIE_Pos  	(0U)
#define   ETH_INTERRUPT_ENABLE_TIE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_TIE_Pos)

#define   ETH_INTERRUPT_ENABLE_TSE_Pos  	(1U)
#define   ETH_INTERRUPT_ENABLE_TSE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_TSE_Pos)

#define   ETH_INTERRUPT_ENABLE_TUE_Pos  	(2U)
#define   ETH_INTERRUPT_ENABLE_TUE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_TUE_Pos)

#define   ETH_INTERRUPT_ENABLE_TJE_Pos  	(3U)
#define   ETH_INTERRUPT_ENABLE_TJE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_TJE_Pos)

#define   ETH_INTERRUPT_ENABLE_OVE_Pos  	(4U)
#define   ETH_INTERRUPT_ENABLE_OVE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_OVE_Pos)

#define   ETH_INTERRUPT_ENABLE_UNE_Pos  	(5U)
#define   ETH_INTERRUPT_ENABLE_UNE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_UNE_Pos)

#define   ETH_INTERRUPT_ENABLE_RIE_Pos  	(6U)
#define   ETH_INTERRUPT_ENABLE_RIE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_RIE_Pos)

#define   ETH_INTERRUPT_ENABLE_RUE_Pos  	(7U)
#define   ETH_INTERRUPT_ENABLE_RUE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_RUE_Pos)

#define   ETH_INTERRUPT_ENABLE_RSE_Pos  	(8U)
#define   ETH_INTERRUPT_ENABLE_RSE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_RSE_Pos)

#define   ETH_INTERRUPT_ENABLE_RWE_Pos  	(9U)
#define   ETH_INTERRUPT_ENABLE_RWE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_RWE_Pos)

#define   ETH_INTERRUPT_ENABLE_ETE_Pos  	(10U)
#define   ETH_INTERRUPT_ENABLE_ETE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_ETE_Pos)

#define   ETH_INTERRUPT_ENABLE_Reserved_12_11_Pos  	(11U)
#define   ETH_INTERRUPT_ENABLE_Reserved_12_11_Msk  	(0x00000003U  << ETH_INTERRUPT_ENABLE_Reserved_12_11_Pos)

#define   ETH_INTERRUPT_ENABLE_FBE_Pos  	(13U)
#define   ETH_INTERRUPT_ENABLE_FBE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_FBE_Pos)

#define   ETH_INTERRUPT_ENABLE_ERE_Pos  	(14U)
#define   ETH_INTERRUPT_ENABLE_ERE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_ERE_Pos)

#define   ETH_INTERRUPT_ENABLE_AIE_Pos  	(15U)
#define   ETH_INTERRUPT_ENABLE_AIE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_AIE_Pos)

#define   ETH_INTERRUPT_ENABLE_NIE_Pos  	(16U)
#define   ETH_INTERRUPT_ENABLE_NIE_Msk  	(0x00000001U  << ETH_INTERRUPT_ENABLE_NIE_Pos)

#define   ETH_INTERRUPT_ENABLE_Reserved_31_17_Pos  	(17U)
#define   ETH_INTERRUPT_ENABLE_Reserved_31_17_Msk  	(0x00007FFFU  << ETH_INTERRUPT_ENABLE_Reserved_31_17_Pos)

/* ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER  =  */
#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_MISFRMCNT_Pos  	(0U)
#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_MISFRMCNT_Msk  	(0x0000FFFFU  << ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_MISFRMCNT_Pos)

#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_MISCNTOVF_Pos  	(16U)
#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_MISCNTOVF_Msk  	(0x00000001U  << ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_MISCNTOVF_Pos)

#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_OVFFRMCNT_Pos  	(17U)
#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_OVFFRMCNT_Msk  	(0x000007FFU  << ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_OVFFRMCNT_Pos)

#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_OVFCNTOVF_Pos  	(28U)
#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_OVFCNTOVF_Msk  	(0x00000001U  << ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_OVFCNTOVF_Pos)

#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_Reserved_31_29_Pos  	(29U)
#define   ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_Reserved_31_29_Msk  	(0x00000007U  << ETH_MISSED_FRAME_AND_BUFFER_OVERFLOW_COUNTER_Reserved_31_29_Pos)

/* ETH_RECEIVE_INTERRUPT_WATCHDOG_TIMER  =  */
#define   ETH_RECEIVE_INTERRUPT_WATCHDOG_TIMER_RIWT_Pos  	(0U)
#define   ETH_RECEIVE_INTERRUPT_WATCHDOG_TIMER_RIWT_Msk  	(0x000000FFU  << ETH_RECEIVE_INTERRUPT_WATCHDOG_TIMER_RIWT_Pos)

#define   ETH_RECEIVE_INTERRUPT_WATCHDOG_TIMER_Reserved_31_8_Pos  	(8U)
#define   ETH_RECEIVE_INTERRUPT_WATCHDOG_TIMER_Reserved_31_8_Msk  	(0x00FFFFFFU  << ETH_RECEIVE_INTERRUPT_WATCHDOG_TIMER_Reserved_31_8_Pos)

/* ETH_AHB_STATUS  =  Register 11 [AHB Status Register]*/
#define   ETH_AHB_STATUS_AHBMS_Pos  	(0U)
#define   ETH_AHB_STATUS_AHBMS_Msk  	(0x00000001U  << ETH_AHB_STATUS_AHBMS_Pos)

#define   ETH_AHB_STATUS_Reserved_1_Pos  	(1U)
#define   ETH_AHB_STATUS_Reserved_1_Msk  	(0x00000001U  << ETH_AHB_STATUS_Reserved_1_Pos)

#define   ETH_AHB_STATUS_Reserved_31_2_Pos  	(2U)
#define   ETH_AHB_STATUS_Reserved_31_2_Msk  	(0x3FFFFFFFU  << ETH_AHB_STATUS_Reserved_31_2_Pos)

/* ETH_CURRENT_HOST_TRANSMIT_DESCRIPTOR  =  */
#define   ETH_CURRENT_HOST_TRANSMIT_DESCRIPTOR_CURTDESAPTR_Pos  	(0U)
#define   ETH_CURRENT_HOST_TRANSMIT_DESCRIPTOR_CURTDESAPTR_Msk  	(0xFFFFFFFFU  << ETH_CURRENT_HOST_TRANSMIT_DESCRIPTOR_CURTDESAPTR_Pos)

/* ETH_CURRENT_HOST_RECEIVE_DESCRIPTOR  =  */
#define   ETH_CURRENT_HOST_RECEIVE_DESCRIPTOR_CURRDESAPTR_Pos  	(0U)
#define   ETH_CURRENT_HOST_RECEIVE_DESCRIPTOR_CURRDESAPTR_Msk  	(0xFFFFFFFFU  << ETH_CURRENT_HOST_RECEIVE_DESCRIPTOR_CURRDESAPTR_Pos)

/* ETH_CURRENT_HOST_TRANSMIT_BUFFER_ADDRESS  =  Register 20 [Current Host Transmit Buffer Address Register]*/
#define   ETH_CURRENT_HOST_TRANSMIT_BUFFER_ADDRESS_CURTBUFAPTR_Pos  	(0U)
#define   ETH_CURRENT_HOST_TRANSMIT_BUFFER_ADDRESS_CURTBUFAPTR_Msk  	(0xFFFFFFFFU  << ETH_CURRENT_HOST_TRANSMIT_BUFFER_ADDRESS_CURTBUFAPTR_Pos)

/* ETH_CURRENT_HOST_RECEIVE_BUFFER_ADDRESS  =  Register 21 [Current Host Receive Buffer Address Register]*/
#define   ETH_CURRENT_HOST_RECEIVE_BUFFER_ADDRESS_CURRBUFAPTR_Pos  	(0U)
#define   ETH_CURRENT_HOST_RECEIVE_BUFFER_ADDRESS_CURRBUFAPTR_Msk  	(0xFFFFFFFFU  << ETH_CURRENT_HOST_RECEIVE_BUFFER_ADDRESS_CURRBUFAPTR_Pos)

/* ETH_HW_FEATURE  =  Register 22 [HW Feature Register]*/
#define   ETH_HW_FEATURE_MIISEL_Pos  	(0U)
#define   ETH_HW_FEATURE_MIISEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_MIISEL_Pos)

#define   ETH_HW_FEATURE_GMIISEL_Pos  	(1U)
#define   ETH_HW_FEATURE_GMIISEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_GMIISEL_Pos)

#define   ETH_HW_FEATURE_HDSEL_Pos  	(2U)
#define   ETH_HW_FEATURE_HDSEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_HDSEL_Pos)

#define   ETH_HW_FEATURE_EXTHASHEN_Pos  	(3U)
#define   ETH_HW_FEATURE_EXTHASHEN_Msk  	(0x00000001U  << ETH_HW_FEATURE_EXTHASHEN_Pos)

#define   ETH_HW_FEATURE_HASHSEL_Pos  	(4U)
#define   ETH_HW_FEATURE_HASHSEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_HASHSEL_Pos)

#define   ETH_HW_FEATURE_ADDMACADRSEL_Pos  	(5U)
#define   ETH_HW_FEATURE_ADDMACADRSEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_ADDMACADRSEL_Pos)

#define   ETH_HW_FEATURE_PCSSEL_Pos  	(6U)
#define   ETH_HW_FEATURE_PCSSEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_PCSSEL_Pos)

#define   ETH_HW_FEATURE_L3L4FLTREN_Pos  	(7U)
#define   ETH_HW_FEATURE_L3L4FLTREN_Msk  	(0x00000001U  << ETH_HW_FEATURE_L3L4FLTREN_Pos)

#define   ETH_HW_FEATURE_SMASEL_Pos  	(8U)
#define   ETH_HW_FEATURE_SMASEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_SMASEL_Pos)

#define   ETH_HW_FEATURE_RWKSEL_Pos  	(9U)
#define   ETH_HW_FEATURE_RWKSEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_RWKSEL_Pos)

#define   ETH_HW_FEATURE_MGKSEL_Pos  	(10U)
#define   ETH_HW_FEATURE_MGKSEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_MGKSEL_Pos)

#define   ETH_HW_FEATURE_MMCSEL_Pos  	(11U)
#define   ETH_HW_FEATURE_MMCSEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_MMCSEL_Pos)

#define   ETH_HW_FEATURE_TSVER1SEL_Pos  	(12U)
#define   ETH_HW_FEATURE_TSVER1SEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_TSVER1SEL_Pos)

#define   ETH_HW_FEATURE_TSVER2SEL_Pos  	(13U)
#define   ETH_HW_FEATURE_TSVER2SEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_TSVER2SEL_Pos)

#define   ETH_HW_FEATURE_EEESEL_Pos  	(14U)
#define   ETH_HW_FEATURE_EEESEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_EEESEL_Pos)

#define   ETH_HW_FEATURE_AVSEL_Pos  	(15U)
#define   ETH_HW_FEATURE_AVSEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_AVSEL_Pos)

#define   ETH_HW_FEATURE_TXCOESEL_Pos  	(16U)
#define   ETH_HW_FEATURE_TXCOESEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_TXCOESEL_Pos)

#define   ETH_HW_FEATURE_RXTYP1COE_Pos  	(17U)
#define   ETH_HW_FEATURE_RXTYP1COE_Msk  	(0x00000001U  << ETH_HW_FEATURE_RXTYP1COE_Pos)

#define   ETH_HW_FEATURE_RXTYP2COE_Pos  	(18U)
#define   ETH_HW_FEATURE_RXTYP2COE_Msk  	(0x00000001U  << ETH_HW_FEATURE_RXTYP2COE_Pos)

#define   ETH_HW_FEATURE_RXFIFOSIZE_Pos  	(19U)
#define   ETH_HW_FEATURE_RXFIFOSIZE_Msk  	(0x00000001U  << ETH_HW_FEATURE_RXFIFOSIZE_Pos)

#define   ETH_HW_FEATURE_RXCHCNT_Pos  	(20U)
#define   ETH_HW_FEATURE_RXCHCNT_Msk  	(0x00000003U  << ETH_HW_FEATURE_RXCHCNT_Pos)

#define   ETH_HW_FEATURE_TXCHCNT_Pos  	(22U)
#define   ETH_HW_FEATURE_TXCHCNT_Msk  	(0x00000003U  << ETH_HW_FEATURE_TXCHCNT_Pos)

#define   ETH_HW_FEATURE_ENHDESSEL_Pos  	(24U)
#define   ETH_HW_FEATURE_ENHDESSEL_Msk  	(0x00000001U  << ETH_HW_FEATURE_ENHDESSEL_Pos)

#define   ETH_HW_FEATURE_INTTSEN_Pos  	(25U)
#define   ETH_HW_FEATURE_INTTSEN_Msk  	(0x00000001U  << ETH_HW_FEATURE_INTTSEN_Pos)

#define   ETH_HW_FEATURE_FLEXIPPSEN_Pos  	(26U)
#define   ETH_HW_FEATURE_FLEXIPPSEN_Msk  	(0x00000001U  << ETH_HW_FEATURE_FLEXIPPSEN_Pos)

#define   ETH_HW_FEATURE_SAVLANINS_Pos  	(27U)
#define   ETH_HW_FEATURE_SAVLANINS_Msk  	(0x00000001U  << ETH_HW_FEATURE_SAVLANINS_Pos)

#define   ETH_HW_FEATURE_ACTPHYIF_Pos  	(28U)
#define   ETH_HW_FEATURE_ACTPHYIF_Msk  	(0x00000007U  << ETH_HW_FEATURE_ACTPHYIF_Pos)

#define   ETH_HW_FEATURE_Reserved_31_Pos  	(31U)
#define   ETH_HW_FEATURE_Reserved_31_Msk  	(0x00000001U  << ETH_HW_FEATURE_Reserved_31_Pos)



/***   ETH0_CON Bit Fileds *******************/
/***************************************************************************/


/* ETH0_CON  =  Ethernet x Port Control Register*/
#define   ETH_CON_RXD0_Pos  	(0U)
#define   ETH_CON_RXD0_Msk  	(0x00000003U  << ETH_CON_RXD0_Pos)

#define   ETH_CON_RXD1_Pos  	(2U)
#define   ETH_CON_RXD1_Msk  	(0x00000003U  << ETH_CON_RXD1_Pos)

#define   ETH_CON_RXD2_Pos  	(4U)
#define   ETH_CON_RXD2_Msk  	(0x00000003U  << ETH_CON_RXD2_Pos)

#define   ETH_CON_RXD3_Pos  	(6U)
#define   ETH_CON_RXD3_Msk  	(0x00000003U  << ETH_CON_RXD3_Pos)

#define   ETH_CON_CLK_RMII_Pos  	(8U)
#define   ETH_CON_CLK_RMII_Msk  	(0x00000003U  << ETH_CON_CLK_RMII_Pos)

#define   ETH_CON_CRS_DV_Pos  	(10U)
#define   ETH_CON_CRS_DV_Msk  	(0x00000003U  << ETH_CON_CRS_DV_Pos)

#define   ETH_CON_CRS_Pos  	(12U)
#define   ETH_CON_CRS_Msk  	(0x00000003U  << ETH_CON_CRS_Pos)

#define   ETH_CON_RXER_Pos  	(14U)
#define   ETH_CON_RXER_Msk  	(0x00000003U  << ETH_CON_RXER_Pos)

#define   ETH_CON_COL_Pos  	(16U)
#define   ETH_CON_COL_Msk  	(0x00000003U  << ETH_CON_COL_Pos)

#define   ETH_CON_CLK_TX_Pos  	(18U)
#define   ETH_CON_CLK_TX_Msk  	(0x00000003U  << ETH_CON_CLK_TX_Pos)

#define   ETH_CON_MDIO_Pos  	(22U)
#define   ETH_CON_MDIO_Msk  	(0x00000003U  << ETH_CON_MDIO_Pos)

#define   ETH_CON_INFSEL_Pos  	(26U)
#define   ETH_CON_INFSEL_Msk  	(0x00000001U  << ETH_CON_INFSEL_Pos)



/***   ETM Bit Fileds *******************/
/***************************************************************************/


// ----------------------------------------  ETM_ETMCR  -------------------------------------------
#define ETM_ETMCR_ETM_PowerDown_Pos                           0                                                         /*!< ETM ETMCR: ETM_PowerDown Position   */
#define ETM_ETMCR_ETM_PowerDown_Msk                           (0x01UL << ETM_ETMCR_ETM_PowerDown_Pos)                   /*!< ETM ETMCR: ETM_PowerDown Mask       */
#define ETM_ETMCR_Reserved_Pos                                1                                                         /*!< ETM ETMCR: Reserved Position        */
#define ETM_ETMCR_Reserved_Msk                                (0x07UL << ETM_ETMCR_Reserved_Pos)                        /*!< ETM ETMCR: Reserved Mask            */
#define ETM_ETMCR_PortSize_2_0_Pos                            4                                                         /*!< ETM ETMCR: PortSize_2_0 Position    */
#define ETM_ETMCR_PortSize_2_0_Msk                            (0x07UL << ETM_ETMCR_PortSize_2_0_Pos)                    /*!< ETM ETMCR: PortSize_2_0 Mask        */
#define ETM_ETMCR_StallProcessor_Pos                          7                                                         /*!< ETM ETMCR: StallProcessor Position  */
#define ETM_ETMCR_StallProcessor_Msk                          (0x01UL << ETM_ETMCR_StallProcessor_Pos)                  /*!< ETM ETMCR: StallProcessor Mask      */
#define ETM_ETMCR_BranchOutput_Pos                            8                                                         /*!< ETM ETMCR: BranchOutput Position    */
#define ETM_ETMCR_BranchOutput_Msk                            (0x01UL << ETM_ETMCR_BranchOutput_Pos)                    /*!< ETM ETMCR: BranchOutput Mask        */
#define ETM_ETMCR_DebugRequestControl_Pos                     9                                                         /*!< ETM ETMCR: DebugRequestControl Position */
#define ETM_ETMCR_DebugRequestControl_Msk                     (0x01UL << ETM_ETMCR_DebugRequestControl_Pos)             /*!< ETM ETMCR: DebugRequestControl Mask */
#define ETM_ETMCR_ETM_Programming_Pos                         10                                                        /*!< ETM ETMCR: ETM_Programming Position */
#define ETM_ETMCR_ETM_Programming_Msk                         (0x01UL << ETM_ETMCR_ETM_Programming_Pos)                 /*!< ETM ETMCR: ETM_Programming Mask     */
#define ETM_ETMCR_ETM_PortSelection_Pos                       11                                                        /*!< ETM ETMCR: ETM_PortSelection Position */
#define ETM_ETMCR_ETM_PortSelection_Msk                       (0x01UL << ETM_ETMCR_ETM_PortSelection_Pos)               /*!< ETM ETMCR: ETM_PortSelection Mask   */
#define ETM_ETMCR_Reserved1_Pos                               12                                                        /*!< ETM ETMCR: Reserved1 Position       */
#define ETM_ETMCR_Reserved1_Msk                               (0x01UL << ETM_ETMCR_Reserved1_Pos)                       /*!< ETM ETMCR: Reserved1 Mask           */
#define ETM_ETMCR_PortMode_2_Pos                              13                                                        /*!< ETM ETMCR: PortMode_2 Position      */
#define ETM_ETMCR_PortMode_2_Msk                              (0x01UL << ETM_ETMCR_PortMode_2_Pos)                      /*!< ETM ETMCR: PortMode_2 Mask          */
#define ETM_ETMCR_Reserved2_Pos                               14                                                        /*!< ETM ETMCR: Reserved2 Position       */
#define ETM_ETMCR_Reserved2_Msk                               (0x03UL << ETM_ETMCR_Reserved2_Pos)                       /*!< ETM ETMCR: Reserved2 Mask           */
#define ETM_ETMCR_PortMode_1_0_Pos                            16                                                        /*!< ETM ETMCR: PortMode_1_0 Position    */
#define ETM_ETMCR_PortMode_1_0_Msk                            (0x03UL << ETM_ETMCR_PortMode_1_0_Pos)                    /*!< ETM ETMCR: PortMode_1_0 Mask        */
#define ETM_ETMCR_Reserved3_Pos                               18                                                        /*!< ETM ETMCR: Reserved3 Position       */
#define ETM_ETMCR_Reserved3_Msk                               (0x07UL << ETM_ETMCR_Reserved3_Pos)                       /*!< ETM ETMCR: Reserved3 Mask           */
#define ETM_ETMCR_PortSize_3_Pos                              21                                                        /*!< ETM ETMCR: PortSize_3 Position      */
#define ETM_ETMCR_PortSize_3_Msk                              (0x01UL << ETM_ETMCR_PortSize_3_Pos)                      /*!< ETM ETMCR: PortSize_3 Mask          */
#define ETM_ETMCR_Reserved4_Pos                               22                                                        /*!< ETM ETMCR: Reserved4 Position       */
#define ETM_ETMCR_Reserved4_Msk                               (0x3fUL << ETM_ETMCR_Reserved4_Pos)                       /*!< ETM ETMCR: Reserved4 Mask           */
#define ETM_ETMCR_Timestamp_Pos                               28                                                        /*!< ETM ETMCR: Timestamp Position       */
#define ETM_ETMCR_Timestamp_Msk                               (0x01UL << ETM_ETMCR_Timestamp_Pos)                       /*!< ETM ETMCR: Timestamp Mask           */
#define ETM_ETMCR_Reserved5_Pos                               29                                                        /*!< ETM ETMCR: Reserved5 Position       */
#define ETM_ETMCR_Reserved5_Msk                               (0x07UL << ETM_ETMCR_Reserved5_Pos)                       /*!< ETM ETMCR: Reserved5 Mask           */

// ---------------------------------------  ETM_ETMCCR  -------------------------------------------
#define ETM_ETMCCR_NumberOfAddressComparatorPairs_Pos         0                                                         /*!< ETM ETMCCR: NumberOfAddressComparatorPairs Position */
#define ETM_ETMCCR_NumberOfAddressComparatorPairs_Msk         (0x0fUL << ETM_ETMCCR_NumberOfAddressComparatorPairs_Pos) /*!< ETM ETMCCR: NumberOfAddressComparatorPairs Mask */
#define ETM_ETMCCR_NumberOfDataValueComparators_Pos           4                                                         /*!< ETM ETMCCR: NumberOfDataValueComparators Position */
#define ETM_ETMCCR_NumberOfDataValueComparators_Msk           (0x0fUL << ETM_ETMCCR_NumberOfDataValueComparators_Pos)   /*!< ETM ETMCCR: NumberOfDataValueComparators Mask */
#define ETM_ETMCCR_NumberOfMemoryMapDecoders_Pos              8                                                         /*!< ETM ETMCCR: NumberOfMemoryMapDecoders Position */
#define ETM_ETMCCR_NumberOfMemoryMapDecoders_Msk              (0x1fUL << ETM_ETMCCR_NumberOfMemoryMapDecoders_Pos)      /*!< ETM ETMCCR: NumberOfMemoryMapDecoders Mask */
#define ETM_ETMCCR_NumberOfCounters_Pos                       13                                                        /*!< ETM ETMCCR: NumberOfCounters Position */
#define ETM_ETMCCR_NumberOfCounters_Msk                       (0x07UL << ETM_ETMCCR_NumberOfCounters_Pos)               /*!< ETM ETMCCR: NumberOfCounters Mask   */
#define ETM_ETMCCR_SequencerPresent_Pos                       16                                                        /*!< ETM ETMCCR: SequencerPresent Position */
#define ETM_ETMCCR_SequencerPresent_Msk                       (0x01UL << ETM_ETMCCR_SequencerPresent_Pos)               /*!< ETM ETMCCR: SequencerPresent Mask   */
#define ETM_ETMCCR_NumberOfExternalIinputs_Pos                17                                                        /*!< ETM ETMCCR: NumberOfExternalIinputs Position */
#define ETM_ETMCCR_NumberOfExternalIinputs_Msk                (0x07UL << ETM_ETMCCR_NumberOfExternalIinputs_Pos)        /*!< ETM ETMCCR: NumberOfExternalIinputs Mask */
#define ETM_ETMCCR_NumberOfExternalOutputs_Pos                20                                                        /*!< ETM ETMCCR: NumberOfExternalOutputs Position */
#define ETM_ETMCCR_NumberOfExternalOutputs_Msk                (0x07UL << ETM_ETMCCR_NumberOfExternalOutputs_Pos)        /*!< ETM ETMCCR: NumberOfExternalOutputs Mask */
#define ETM_ETMCCR_FIFOFULL_LogicPresent_Pos                  23                                                        /*!< ETM ETMCCR: FIFOFULL_LogicPresent Position */
#define ETM_ETMCCR_FIFOFULL_LogicPresent_Msk                  (0x01UL << ETM_ETMCCR_FIFOFULL_LogicPresent_Pos)          /*!< ETM ETMCCR: FIFOFULL_LogicPresent Mask */
#define ETM_ETMCCR_NumberOfContextID_Comparators_Pos          24                                                        /*!< ETM ETMCCR: NumberOfContextID_Comparators Position */
#define ETM_ETMCCR_NumberOfContextID_Comparators_Msk          (0x03UL << ETM_ETMCCR_NumberOfContextID_Comparators_Pos)  /*!< ETM ETMCCR: NumberOfContextID_Comparators Mask */
#define ETM_ETMCCR_TraceStartStopBlockPresent_Pos             26                                                        /*!< ETM ETMCCR: TraceStartStopBlockPresent Position */
#define ETM_ETMCCR_TraceStartStopBlockPresent_Msk             (0x01UL << ETM_ETMCCR_TraceStartStopBlockPresent_Pos)     /*!< ETM ETMCCR: TraceStartStopBlockPresent Mask */
#define ETM_ETMCCR_CoprocessorAndMemoryAccess_Pos             27                                                        /*!< ETM ETMCCR: CoprocessorAndMemoryAccess Position */
#define ETM_ETMCCR_CoprocessorAndMemoryAccess_Msk             (0x01UL << ETM_ETMCCR_CoprocessorAndMemoryAccess_Pos)     /*!< ETM ETMCCR: CoprocessorAndMemoryAccess Mask */
#define ETM_ETMCCR_Reserved_Pos                               28                                                        /*!< ETM ETMCCR: Reserved Position       */
#define ETM_ETMCCR_Reserved_Msk                               (0x07UL << ETM_ETMCCR_Reserved_Pos)                       /*!< ETM ETMCCR: Reserved Mask           */
#define ETM_ETMCCR_ETM_ID_RegisterPresent_Pos                 31                                                        /*!< ETM ETMCCR: ETM_ID_RegisterPresent Position */
#define ETM_ETMCCR_ETM_ID_RegisterPresent_Msk                 (0x01UL << ETM_ETMCCR_ETM_ID_RegisterPresent_Pos)         /*!< ETM ETMCCR: ETM_ID_RegisterPresent Mask */

// -------------------------------------  ETM_ETMTRIGGER  -----------------------------------------
#define ETM_ETMTRIGGER_ResourceA_Pos                          0                                                         /*!< ETM ETMTRIGGER: ResourceA Position  */
#define ETM_ETMTRIGGER_ResourceA_Msk                          (0x7fUL << ETM_ETMTRIGGER_ResourceA_Pos)                  /*!< ETM ETMTRIGGER: ResourceA Mask      */
#define ETM_ETMTRIGGER_ResourceB_Pos                          7                                                         /*!< ETM ETMTRIGGER: ResourceB Position  */
#define ETM_ETMTRIGGER_ResourceB_Msk                          (0x7fUL << ETM_ETMTRIGGER_ResourceB_Pos)                  /*!< ETM ETMTRIGGER: ResourceB Mask      */
#define ETM_ETMTRIGGER_Function_Pos                           14                                                        /*!< ETM ETMTRIGGER: Function Position   */
#define ETM_ETMTRIGGER_Function_Msk                           (0x07UL << ETM_ETMTRIGGER_Function_Pos)                   /*!< ETM ETMTRIGGER: Function Mask       */
#define ETM_ETMTRIGGER_Reserved_Pos                           17                                                        /*!< ETM ETMTRIGGER: Reserved Position   */
#define ETM_ETMTRIGGER_Reserved_Msk                           (0x00007fffUL << ETM_ETMTRIGGER_Reserved_Pos)             /*!< ETM ETMTRIGGER: Reserved Mask       */

// ----------------------------------------  ETM_ETMSR  -------------------------------------------
#define ETM_ETMSR_UntracedOverflowFlag_Pos                    0                                                         /*!< ETM ETMSR: UntracedOverflowFlag Position */
#define ETM_ETMSR_UntracedOverflowFlag_Msk                    (0x01UL << ETM_ETMSR_UntracedOverflowFlag_Pos)            /*!< ETM ETMSR: UntracedOverflowFlag Mask */
#define ETM_ETMSR_Progbit_Pos                                 1                                                         /*!< ETM ETMSR: Progbit Position         */
#define ETM_ETMSR_Progbit_Msk                                 (0x01UL << ETM_ETMSR_Progbit_Pos)                         /*!< ETM ETMSR: Progbit Mask             */
#define ETM_ETMSR_TraceStartStopResourceStatus_Pos            2                                                         /*!< ETM ETMSR: TraceStartStopResourceStatus Position */
#define ETM_ETMSR_TraceStartStopResourceStatus_Msk            (0x01UL << ETM_ETMSR_TraceStartStopResourceStatus_Pos)    /*!< ETM ETMSR: TraceStartStopResourceStatus Mask */
#define ETM_ETMSR_TriggerFlag_Pos                             3                                                         /*!< ETM ETMSR: TriggerFlag Position     */
#define ETM_ETMSR_TriggerFlag_Msk                             (0x01UL << ETM_ETMSR_TriggerFlag_Pos)                     /*!< ETM ETMSR: TriggerFlag Mask         */
#define ETM_ETMSR_Reserved_Pos                                4                                                         /*!< ETM ETMSR: Reserved Position        */
#define ETM_ETMSR_Reserved_Msk                                (0x0fffffffUL << ETM_ETMSR_Reserved_Pos)                  /*!< ETM ETMSR: Reserved Mask            */

// ---------------------------------------  ETM_ETMSCR  -------------------------------------------
#define ETM_ETMSCR_MaximumPortSize_2_0_Pos                    0                                                         /*!< ETM ETMSCR: MaximumPortSize_2_0 Position */
#define ETM_ETMSCR_MaximumPortSize_2_0_Msk                    (0x07UL << ETM_ETMSCR_MaximumPortSize_2_0_Pos)            /*!< ETM ETMSCR: MaximumPortSize_2_0 Mask */
#define ETM_ETMSCR_Reserved_Pos                               3                                                         /*!< ETM ETMSCR: Reserved Position       */
#define ETM_ETMSCR_Reserved_Msk                               (0x01UL << ETM_ETMSCR_Reserved_Pos)                       /*!< ETM ETMSCR: Reserved Mask           */
#define ETM_ETMSCR_Reserved1_Pos                              4                                                         /*!< ETM ETMSCR: Reserved1 Position      */
#define ETM_ETMSCR_Reserved1_Msk                              (0x0fUL << ETM_ETMSCR_Reserved1_Pos)                      /*!< ETM ETMSCR: Reserved1 Mask          */
#define ETM_ETMSCR_FIFOFULL_Supported_Pos                     8                                                         /*!< ETM ETMSCR: FIFOFULL_Supported Position */
#define ETM_ETMSCR_FIFOFULL_Supported_Msk                     (0x01UL << ETM_ETMSCR_FIFOFULL_Supported_Pos)             /*!< ETM ETMSCR: FIFOFULL_Supported Mask */
#define ETM_ETMSCR_MaximumPortSize_3_Pos                      9                                                         /*!< ETM ETMSCR: MaximumPortSize_3 Position */
#define ETM_ETMSCR_MaximumPortSize_3_Msk                      (0x01UL << ETM_ETMSCR_MaximumPortSize_3_Pos)              /*!< ETM ETMSCR: MaximumPortSize_3 Mask  */
#define ETM_ETMSCR_PortSizeSupported_Pos                      10                                                        /*!< ETM ETMSCR: PortSizeSupported Position */
#define ETM_ETMSCR_PortSizeSupported_Msk                      (0x01UL << ETM_ETMSCR_PortSizeSupported_Pos)              /*!< ETM ETMSCR: PortSizeSupported Mask  */
#define ETM_ETMSCR_PortModeSupported_Pos                      11                                                        /*!< ETM ETMSCR: PortModeSupported Position */
#define ETM_ETMSCR_PortModeSupported_Msk                      (0x01UL << ETM_ETMSCR_PortModeSupported_Pos)              /*!< ETM ETMSCR: PortModeSupported Mask  */
#define ETM_ETMSCR_N_Minus_1_Pos                              12                                                        /*!< ETM ETMSCR: N_Minus_1 Position      */
#define ETM_ETMSCR_N_Minus_1_Msk                              (0x07UL << ETM_ETMSCR_N_Minus_1_Pos)                      /*!< ETM ETMSCR: N_Minus_1 Mask          */
#define ETM_ETMSCR_Reserved2_Pos                              15                                                        /*!< ETM ETMSCR: Reserved2 Position      */
#define ETM_ETMSCR_Reserved2_Msk                              (0x03UL << ETM_ETMSCR_Reserved2_Pos)                      /*!< ETM ETMSCR: Reserved2 Mask          */
#define ETM_ETMSCR_NoFetchComparisons_Pos                     17                                                        /*!< ETM ETMSCR: NoFetchComparisons Position */
#define ETM_ETMSCR_NoFetchComparisons_Msk                     (0x01UL << ETM_ETMSCR_NoFetchComparisons_Pos)             /*!< ETM ETMSCR: NoFetchComparisons Mask */
#define ETM_ETMSCR_Reserved3_Pos                              18                                                        /*!< ETM ETMSCR: Reserved3 Position      */
#define ETM_ETMSCR_Reserved3_Msk                              (0x00003fffUL << ETM_ETMSCR_Reserved3_Pos)                /*!< ETM ETMSCR: Reserved3 Mask          */

// --------------------------------------  ETM_ETMTEEVR  ------------------------------------------
#define ETM_ETMTEEVR_ResourceA_Pos                            0                                                         /*!< ETM ETMTEEVR: ResourceA Position    */
#define ETM_ETMTEEVR_ResourceA_Msk                            (0x7fUL << ETM_ETMTEEVR_ResourceA_Pos)                    /*!< ETM ETMTEEVR: ResourceA Mask        */
#define ETM_ETMTEEVR_ResourceB_Pos                            7                                                         /*!< ETM ETMTEEVR: ResourceB Position    */
#define ETM_ETMTEEVR_ResourceB_Msk                            (0x7fUL << ETM_ETMTEEVR_ResourceB_Pos)                    /*!< ETM ETMTEEVR: ResourceB Mask        */
#define ETM_ETMTEEVR_Function_Pos                             14                                                        /*!< ETM ETMTEEVR: Function Position     */
#define ETM_ETMTEEVR_Function_Msk                             (0x07UL << ETM_ETMTEEVR_Function_Pos)                     /*!< ETM ETMTEEVR: Function Mask         */
#define ETM_ETMTEEVR_Reserved_Pos                             17                                                        /*!< ETM ETMTEEVR: Reserved Position     */
#define ETM_ETMTEEVR_Reserved_Msk                             (0x00007fffUL << ETM_ETMTEEVR_Reserved_Pos)               /*!< ETM ETMTEEVR: Reserved Mask         */

// --------------------------------------  ETM_ETMTECR1  ------------------------------------------
#define ETM_ETMTECR1_Reserved_Pos                             0                                                         /*!< ETM ETMTECR1: Reserved Position     */
#define ETM_ETMTECR1_Reserved_Msk                             (0x01ffffffUL << ETM_ETMTECR1_Reserved_Pos)               /*!< ETM ETMTECR1: Reserved Mask         */
#define ETM_ETMTECR1_TraceStartStopEnable_Pos                 25                                                        /*!< ETM ETMTECR1: TraceStartStopEnable Position */
#define ETM_ETMTECR1_TraceStartStopEnable_Msk                 (0x01UL << ETM_ETMTECR1_TraceStartStopEnable_Pos)         /*!< ETM ETMTECR1: TraceStartStopEnable Mask */
#define ETM_ETMTECR1_Reserved1_Pos                            26                                                        /*!< ETM ETMTECR1: Reserved1 Position    */
#define ETM_ETMTECR1_Reserved1_Msk                            (0x3fUL << ETM_ETMTECR1_Reserved1_Pos)                    /*!< ETM ETMTECR1: Reserved1 Mask        */

// ---------------------------------------  ETM_ETMFFLR  ------------------------------------------
#define ETM_ETMFFLR_FIFO_FullLevel_Pos                        0                                                         /*!< ETM ETMFFLR: FIFO_FullLevel Position */
#define ETM_ETMFFLR_FIFO_FullLevel_Msk                        (0x000000ffUL << ETM_ETMFFLR_FIFO_FullLevel_Pos)          /*!< ETM ETMFFLR: FIFO_FullLevel Mask    */
#define ETM_ETMFFLR_Reserved_Pos                              8                                                         /*!< ETM ETMFFLR: Reserved Position      */
#define ETM_ETMFFLR_Reserved_Msk                              (0x00ffffffUL << ETM_ETMFFLR_Reserved_Pos)                /*!< ETM ETMFFLR: Reserved Mask          */

// ------------------------------------  ETM_ETMCNTRLDVR1  ----------------------------------------
#define ETM_ETMCNTRLDVR1_InitialCount_Pos                     0                                                         /*!< ETM ETMCNTRLDVR1: InitialCount Position */
#define ETM_ETMCNTRLDVR1_InitialCount_Msk                     (0x0000ffffUL << ETM_ETMCNTRLDVR1_InitialCount_Pos)       /*!< ETM ETMCNTRLDVR1: InitialCount Mask */
#define ETM_ETMCNTRLDVR1_Reserved_Pos                         16                                                        /*!< ETM ETMCNTRLDVR1: Reserved Position */
#define ETM_ETMCNTRLDVR1_Reserved_Msk                         (0x0000ffffUL << ETM_ETMCNTRLDVR1_Reserved_Pos)           /*!< ETM ETMCNTRLDVR1: Reserved Mask     */

// --------------------------------------  ETM_ETMSYNCFR  -----------------------------------------
#define ETM_ETMSYNCFR_SynchronizationFrequency_Pos            0                                                         /*!< ETM ETMSYNCFR: SynchronizationFrequency Position */
#define ETM_ETMSYNCFR_SynchronizationFrequency_Msk            (0x00000fffUL << ETM_ETMSYNCFR_SynchronizationFrequency_Pos)/*!< ETM ETMSYNCFR: SynchronizationFrequency Mask */
#define ETM_ETMSYNCFR_Reserved_Pos                            12                                                        /*!< ETM ETMSYNCFR: Reserved Position    */
#define ETM_ETMSYNCFR_Reserved_Msk                            (0x000fffffUL << ETM_ETMSYNCFR_Reserved_Pos)              /*!< ETM ETMSYNCFR: Reserved Mask        */

// ---------------------------------------  ETM_ETMIDR  -------------------------------------------
#define ETM_ETMIDR_ImplementationRevision_Pos                 0                                                         /*!< ETM ETMIDR: ImplementationRevision Position */
#define ETM_ETMIDR_ImplementationRevision_Msk                 (0x0fUL << ETM_ETMIDR_ImplementationRevision_Pos)         /*!< ETM ETMIDR: ImplementationRevision Mask */
#define ETM_ETMIDR_MinorETM_ArchitectureVersion_Pos           4                                                         /*!< ETM ETMIDR: MinorETM_ArchitectureVersion Position */
#define ETM_ETMIDR_MinorETM_ArchitectureVersion_Msk           (0x0fUL << ETM_ETMIDR_MinorETM_ArchitectureVersion_Pos)   /*!< ETM ETMIDR: MinorETM_ArchitectureVersion Mask */
#define ETM_ETMIDR_MajorETM_ArchitectureVersion_Pos           8                                                         /*!< ETM ETMIDR: MajorETM_ArchitectureVersion Position */
#define ETM_ETMIDR_MajorETM_ArchitectureVersion_Msk           (0x0fUL << ETM_ETMIDR_MajorETM_ArchitectureVersion_Pos)   /*!< ETM ETMIDR: MajorETM_ArchitectureVersion Mask */
#define ETM_ETMIDR_ProcessorFamily_Pos                        12                                                        /*!< ETM ETMIDR: ProcessorFamily Position */
#define ETM_ETMIDR_ProcessorFamily_Msk                        (0x0fUL << ETM_ETMIDR_ProcessorFamily_Pos)                /*!< ETM ETMIDR: ProcessorFamily Mask    */
#define ETM_ETMIDR_LoadPC_First_Pos                           16                                                        /*!< ETM ETMIDR: LoadPC_First Position   */
#define ETM_ETMIDR_LoadPC_First_Msk                           (0x01UL << ETM_ETMIDR_LoadPC_First_Pos)                   /*!< ETM ETMIDR: LoadPC_First Mask       */
#define ETM_ETMIDR_Reserved_Pos                               17                                                        /*!< ETM ETMIDR: Reserved Position       */
#define ETM_ETMIDR_Reserved_Msk                               (0x01UL << ETM_ETMIDR_Reserved_Pos)                       /*!< ETM ETMIDR: Reserved Mask           */
#define ETM_ETMIDR_Bitfield_32bitThumbInstructionTracing_Pos  18                                                        /*!< ETM ETMIDR: Bitfield_32bitThumbInstructionTracing Position */
#define ETM_ETMIDR_Bitfield_32bitThumbInstructionTracing_Msk  (0x01UL << ETM_ETMIDR_Bitfield_32bitThumbInstructionTracing_Pos)/*!< ETM ETMIDR: Bitfield_32bitThumbInstructionTracing Mask */
#define ETM_ETMIDR_SecurityExtensionsSupport_Pos              19                                                        /*!< ETM ETMIDR: SecurityExtensionsSupport Position */
#define ETM_ETMIDR_SecurityExtensionsSupport_Msk              (0x01UL << ETM_ETMIDR_SecurityExtensionsSupport_Pos)      /*!< ETM ETMIDR: SecurityExtensionsSupport Mask */
#define ETM_ETMIDR_BranchPacketEncoding_Pos                   20                                                        /*!< ETM ETMIDR: BranchPacketEncoding Position */
#define ETM_ETMIDR_BranchPacketEncoding_Msk                   (0x01UL << ETM_ETMIDR_BranchPacketEncoding_Pos)           /*!< ETM ETMIDR: BranchPacketEncoding Mask */
#define ETM_ETMIDR_Reserved1_Pos                              21                                                        /*!< ETM ETMIDR: Reserved1 Position      */
#define ETM_ETMIDR_Reserved1_Msk                              (0x07UL << ETM_ETMIDR_Reserved1_Pos)                      /*!< ETM ETMIDR: Reserved1 Mask          */
#define ETM_ETMIDR_ImplementerCode_Pos                        24                                                        /*!< ETM ETMIDR: ImplementerCode Position */
#define ETM_ETMIDR_ImplementerCode_Msk                        (0x000000ffUL << ETM_ETMIDR_ImplementerCode_Pos)          /*!< ETM ETMIDR: ImplementerCode Mask    */

// ---------------------------------------  ETM_ETMCCER  ------------------------------------------
#define ETM_ETMCCER_ExtendedExternalInputSelectors_Pos        0                                                         /*!< ETM ETMCCER: ExtendedExternalInputSelectors Position */
#define ETM_ETMCCER_ExtendedExternalInputSelectors_Msk        (0x07UL << ETM_ETMCCER_ExtendedExternalInputSelectors_Pos)/*!< ETM ETMCCER: ExtendedExternalInputSelectors Mask */
#define ETM_ETMCCER_ExtendedExternalInputBus_Pos              3                                                         /*!< ETM ETMCCER: ExtendedExternalInputBus Position */
#define ETM_ETMCCER_ExtendedExternalInputBus_Msk              (0x000000ffUL << ETM_ETMCCER_ExtendedExternalInputBus_Pos)/*!< ETM ETMCCER: ExtendedExternalInputBus Mask */
#define ETM_ETMCCER_ReadableRegisters_Pos                     11                                                        /*!< ETM ETMCCER: ReadableRegisters Position */
#define ETM_ETMCCER_ReadableRegisters_Msk                     (0x01UL << ETM_ETMCCER_ReadableRegisters_Pos)             /*!< ETM ETMCCER: ReadableRegisters Mask */
#define ETM_ETMCCER_DataAddressComparisons_Pos                12                                                        /*!< ETM ETMCCER: DataAddressComparisons Position */
#define ETM_ETMCCER_DataAddressComparisons_Msk                (0x01UL << ETM_ETMCCER_DataAddressComparisons_Pos)        /*!< ETM ETMCCER: DataAddressComparisons Mask */
#define ETM_ETMCCER_InstrumentationResources_Pos              13                                                        /*!< ETM ETMCCER: InstrumentationResources Position */
#define ETM_ETMCCER_InstrumentationResources_Msk              (0x07UL << ETM_ETMCCER_InstrumentationResources_Pos)      /*!< ETM ETMCCER: InstrumentationResources Mask */
#define ETM_ETMCCER_EmbeddedICE_WatchpointInputs_Pos          16                                                        /*!< ETM ETMCCER: EmbeddedICE_WatchpointInputs Position */
#define ETM_ETMCCER_EmbeddedICE_WatchpointInputs_Msk          (0x0fUL << ETM_ETMCCER_EmbeddedICE_WatchpointInputs_Pos)  /*!< ETM ETMCCER: EmbeddedICE_WatchpointInputs Mask */
#define ETM_ETMCCER_TraceStartStopBlockUsesEmbeddediceWatchpointInputs_Pos20                                            /*!< ETM ETMCCER: TraceStartStopBlockUsesEmbeddediceWatchpointInputs Position */
#define ETM_ETMCCER_TraceStartStopBlockUsesEmbeddediceWatchpointInputs_Msk (0x01UL << ETM_ETMCCER_TraceStartStopBlockUsesEmbeddediceWatchpointInputs_Pos)/*!< ETM ETMCCER: TraceStartStopBlockUsesEmbeddediceWatchpointInputs Mask */
#define ETM_ETMCCER_EmbeddediceBehaviorControlImplemented_Pos 21                                                        /*!< ETM ETMCCER: EmbeddediceBehaviorControlImplemented Position */
#define ETM_ETMCCER_EmbeddediceBehaviorControlImplemented_Msk (0x01UL << ETM_ETMCCER_EmbeddediceBehaviorControlImplemented_Pos)/*!< ETM ETMCCER: EmbeddediceBehaviorControlImplemented Mask */
#define ETM_ETMCCER_TimestampingImplemented_Pos               22                                                        /*!< ETM ETMCCER: TimestampingImplemented Position */
#define ETM_ETMCCER_TimestampingImplemented_Msk               (0x01UL << ETM_ETMCCER_TimestampingImplemented_Pos)       /*!< ETM ETMCCER: TimestampingImplemented Mask */
#define ETM_ETMCCER_Reserved_Pos                              23                                                        /*!< ETM ETMCCER: Reserved Position      */
#define ETM_ETMCCER_Reserved_Msk                              (0x0fUL << ETM_ETMCCER_Reserved_Pos)                      /*!< ETM ETMCCER: Reserved Mask          */
#define ETM_ETMCCER_ReducedFunctionCounter_Pos                27                                                        /*!< ETM ETMCCER: ReducedFunctionCounter Position */
#define ETM_ETMCCER_ReducedFunctionCounter_Msk                (0x01UL << ETM_ETMCCER_ReducedFunctionCounter_Pos)        /*!< ETM ETMCCER: ReducedFunctionCounter Mask */
#define ETM_ETMCCER_TimestampEncoding_Pos                     28                                                        /*!< ETM ETMCCER: TimestampEncoding Position */
#define ETM_ETMCCER_TimestampEncoding_Msk                     (0x01UL << ETM_ETMCCER_TimestampEncoding_Pos)             /*!< ETM ETMCCER: TimestampEncoding Mask */
#define ETM_ETMCCER_TimestampSize_Pos                         29                                                        /*!< ETM ETMCCER: TimestampSize Position */
#define ETM_ETMCCER_TimestampSize_Msk                         (0x01UL << ETM_ETMCCER_TimestampSize_Pos)                 /*!< ETM ETMCCER: TimestampSize Mask     */
#define ETM_ETMCCER_Reserved1_Pos                             30                                                        /*!< ETM ETMCCER: Reserved1 Position     */
#define ETM_ETMCCER_Reserved1_Msk                             (0x03UL << ETM_ETMCCER_Reserved1_Pos)                     /*!< ETM ETMCCER: Reserved1 Mask         */

// -------------------------------------  ETM_ETMTESSEICR  ----------------------------------------
#define ETM_ETMTESSEICR_StartResourceSelect_Pos               0                                                         /*!< ETM ETMTESSEICR: StartResourceSelect Position */
#define ETM_ETMTESSEICR_StartResourceSelect_Msk               (0x000000ffUL << ETM_ETMTESSEICR_StartResourceSelect_Pos) /*!< ETM ETMTESSEICR: StartResourceSelect Mask */
#define ETM_ETMTESSEICR_Reserved_Pos                          8                                                         /*!< ETM ETMTESSEICR: Reserved Position  */
#define ETM_ETMTESSEICR_Reserved_Msk                          (0x000000ffUL << ETM_ETMTESSEICR_Reserved_Pos)            /*!< ETM ETMTESSEICR: Reserved Mask      */
#define ETM_ETMTESSEICR_StopResourceSelect_Pos                16                                                        /*!< ETM ETMTESSEICR: StopResourceSelect Position */
#define ETM_ETMTESSEICR_StopResourceSelect_Msk                (0x000000ffUL << ETM_ETMTESSEICR_StopResourceSelect_Pos)  /*!< ETM ETMTESSEICR: StopResourceSelect Mask */
#define ETM_ETMTESSEICR_Reserved1_Pos                         24                                                        /*!< ETM ETMTESSEICR: Reserved1 Position */
#define ETM_ETMTESSEICR_Reserved1_Msk                         (0x000000ffUL << ETM_ETMTESSEICR_Reserved1_Pos)           /*!< ETM ETMTESSEICR: Reserved1 Mask     */

// --------------------------------------  ETM_ETMTSEVR  ------------------------------------------
#define ETM_ETMTSEVR_ResourceA_Pos                            0                                                         /*!< ETM ETMTSEVR: ResourceA Position    */
#define ETM_ETMTSEVR_ResourceA_Msk                            (0x7fUL << ETM_ETMTSEVR_ResourceA_Pos)                    /*!< ETM ETMTSEVR: ResourceA Mask        */
#define ETM_ETMTSEVR_ResourceB_Pos                            7                                                         /*!< ETM ETMTSEVR: ResourceB Position    */
#define ETM_ETMTSEVR_ResourceB_Msk                            (0x7fUL << ETM_ETMTSEVR_ResourceB_Pos)                    /*!< ETM ETMTSEVR: ResourceB Mask        */
#define ETM_ETMTSEVR_Function_Pos                             14                                                        /*!< ETM ETMTSEVR: Function Position     */
#define ETM_ETMTSEVR_Function_Msk                             (0x07UL << ETM_ETMTSEVR_Function_Pos)                     /*!< ETM ETMTSEVR: Function Mask         */
#define ETM_ETMTSEVR_Reserved_Pos                             17                                                        /*!< ETM ETMTSEVR: Reserved Position     */
#define ETM_ETMTSEVR_Reserved_Msk                             (0x00007fffUL << ETM_ETMTSEVR_Reserved_Pos)               /*!< ETM ETMTSEVR: Reserved Mask         */

// -------------------------------------  ETM_ETMTRACEIDR  ----------------------------------------
#define ETM_ETMTRACEIDR_EmbeddedICE_WatchpointInputs_Pos      0                                                         /*!< ETM ETMTRACEIDR: EmbeddedICE_WatchpointInputs Position */
#define ETM_ETMTRACEIDR_EmbeddedICE_WatchpointInputs_Msk      (0x7fUL << ETM_ETMTRACEIDR_EmbeddedICE_WatchpointInputs_Pos)/*!< ETM ETMTRACEIDR: EmbeddedICE_WatchpointInputs Mask */
#define ETM_ETMTRACEIDR_Reserved_Pos                          7                                                         /*!< ETM ETMTRACEIDR: Reserved Position  */
#define ETM_ETMTRACEIDR_Reserved_Msk                          (0x01ffffffUL << ETM_ETMTRACEIDR_Reserved_Pos)            /*!< ETM ETMTRACEIDR: Reserved Mask      */

// ---------------------------------------  ETM_ETMIDR2  ------------------------------------------
#define ETM_ETMIDR2_RFE_TransferOrder_Pos                     0                                                         /*!< ETM ETMIDR2: RFE_TransferOrder Position */
#define ETM_ETMIDR2_RFE_TransferOrder_Msk                     (0x01UL << ETM_ETMIDR2_RFE_TransferOrder_Pos)             /*!< ETM ETMIDR2: RFE_TransferOrder Mask */
#define ETM_ETMIDR2_SWP_TransferOrder_Pos                     1                                                         /*!< ETM ETMIDR2: SWP_TransferOrder Position */
#define ETM_ETMIDR2_SWP_TransferOrder_Msk                     (0x01UL << ETM_ETMIDR2_SWP_TransferOrder_Pos)             /*!< ETM ETMIDR2: SWP_TransferOrder Mask */
#define ETM_ETMIDR2_Reserved_Pos                              2                                                         /*!< ETM ETMIDR2: Reserved Position      */
#define ETM_ETMIDR2_Reserved_Msk                              (0x3fffffffUL << ETM_ETMIDR2_Reserved_Pos)                /*!< ETM ETMIDR2: Reserved Mask          */

// ---------------------------------------  ETM_ETMPDSR  ------------------------------------------
#define ETM_ETMPDSR_ETM_PoweredUp_Pos                         0                                                         /*!< ETM ETMPDSR: ETM_PoweredUp Position */
#define ETM_ETMPDSR_ETM_PoweredUp_Msk                         (0x01UL << ETM_ETMPDSR_ETM_PoweredUp_Pos)                 /*!< ETM ETMPDSR: ETM_PoweredUp Mask     */
#define ETM_ETMPDSR_Reserved_Pos                              1                                                         /*!< ETM ETMPDSR: Reserved Position      */
#define ETM_ETMPDSR_Reserved_Msk                              (0x7fffffffUL << ETM_ETMPDSR_Reserved_Pos)                /*!< ETM ETMPDSR: Reserved Mask          */

// --------------------------------------  ETM_ITMISCIN  ------------------------------------------
#define ETM_ITMISCIN_EXTIN_1_0_Pos                            0                                                         /*!< ETM ITMISCIN: EXTIN_1_0 Position    */
#define ETM_ITMISCIN_EXTIN_1_0_Msk                            (0x03UL << ETM_ITMISCIN_EXTIN_1_0_Pos)                    /*!< ETM ITMISCIN: EXTIN_1_0 Mask        */
#define ETM_ITMISCIN_Reserved_Pos                             2                                                         /*!< ETM ITMISCIN: Reserved Position     */
#define ETM_ITMISCIN_Reserved_Msk                             (0x03UL << ETM_ITMISCIN_Reserved_Pos)                     /*!< ETM ITMISCIN: Reserved Mask         */
#define ETM_ITMISCIN_COREHALT_Pos                             4                                                         /*!< ETM ITMISCIN: COREHALT Position     */
#define ETM_ITMISCIN_COREHALT_Msk                             (0x01UL << ETM_ITMISCIN_COREHALT_Pos)                     /*!< ETM ITMISCIN: COREHALT Mask         */
#define ETM_ITMISCIN_Reserved1_Pos                            5                                                         /*!< ETM ITMISCIN: Reserved1 Position    */
#define ETM_ITMISCIN_Reserved1_Msk                            (0x07ffffffUL << ETM_ITMISCIN_Reserved1_Pos)              /*!< ETM ITMISCIN: Reserved1 Mask        */

// --------------------------------------  ETM_ITTRIGOUT  -----------------------------------------
#define ETM_ITTRIGOUT_TRIGGER_OutputValue_Pos                 0                                                         /*!< ETM ITTRIGOUT: TRIGGER_OutputValue Position */
#define ETM_ITTRIGOUT_TRIGGER_OutputValue_Msk                 (0x01UL << ETM_ITTRIGOUT_TRIGGER_OutputValue_Pos)         /*!< ETM ITTRIGOUT: TRIGGER_OutputValue Mask */
#define ETM_ITTRIGOUT_Reserved_Pos                            1                                                         /*!< ETM ITTRIGOUT: Reserved Position    */
#define ETM_ITTRIGOUT_Reserved_Msk                            (0x7fffffffUL << ETM_ITTRIGOUT_Reserved_Pos)              /*!< ETM ITTRIGOUT: Reserved Mask        */

// ------------------------------------  ETM_ETM_ITATBCTR2  ---------------------------------------
#define ETM_ETM_ITATBCTR2_ATREADYInputValue_Pos               0                                                         /*!< ETM ETM_ITATBCTR2: ATREADYInputValue Position */
#define ETM_ETM_ITATBCTR2_ATREADYInputValue_Msk               (0x01UL << ETM_ETM_ITATBCTR2_ATREADYInputValue_Pos)       /*!< ETM ETM_ITATBCTR2: ATREADYInputValue Mask */
#define ETM_ETM_ITATBCTR2_Reserved_Pos                        1                                                         /*!< ETM ETM_ITATBCTR2: Reserved Position */
#define ETM_ETM_ITATBCTR2_Reserved_Msk                        (0x7fffffffUL << ETM_ETM_ITATBCTR2_Reserved_Pos)          /*!< ETM ETM_ITATBCTR2: Reserved Mask    */

// ------------------------------------  ETM_ETM_ITATBCTR0  ---------------------------------------
#define ETM_ETM_ITATBCTR0_ATVALIDOutputValue_Pos              0                                                         /*!< ETM ETM_ITATBCTR0: ATVALIDOutputValue Position */
#define ETM_ETM_ITATBCTR0_ATVALIDOutputValue_Msk              (0x01UL << ETM_ETM_ITATBCTR0_ATVALIDOutputValue_Pos)      /*!< ETM ETM_ITATBCTR0: ATVALIDOutputValue Mask */
#define ETM_ETM_ITATBCTR0_Reserved_Pos                        1                                                         /*!< ETM ETM_ITATBCTR0: Reserved Position */
#define ETM_ETM_ITATBCTR0_Reserved_Msk                        (0x7fffffffUL << ETM_ETM_ITATBCTR0_Reserved_Pos)          /*!< ETM ETM_ITATBCTR0: Reserved Mask    */

// --------------------------------------  ETM_ETMITCTRL  -----------------------------------------
#define ETM_ETMITCTRL_EnableIntegrationMode_Pos               0                                                         /*!< ETM ETMITCTRL: EnableIntegrationMode Position */
#define ETM_ETMITCTRL_EnableIntegrationMode_Msk               (0x01UL << ETM_ETMITCTRL_EnableIntegrationMode_Pos)       /*!< ETM ETMITCTRL: EnableIntegrationMode Mask */
#define ETM_ETMITCTRL_Reserved_Pos                            1                                                         /*!< ETM ETMITCTRL: Reserved Position    */
#define ETM_ETMITCTRL_Reserved_Msk                            (0x7fffffffUL << ETM_ETMITCTRL_Reserved_Pos)              /*!< ETM ETMITCTRL: Reserved Mask        */

// -------------------------------------  ETM_ETMCLAIMSET  ----------------------------------------
#define ETM_ETMCLAIMSET_ClaimTag_Pos                          0                                                         /*!< ETM ETMCLAIMSET: ClaimTag Position  */
#define ETM_ETMCLAIMSET_ClaimTag_Msk                          (0x000000ffUL << ETM_ETMCLAIMSET_ClaimTag_Pos)            /*!< ETM ETMCLAIMSET: ClaimTag Mask      */
#define ETM_ETMCLAIMSET_Reserved_Pos                          8                                                         /*!< ETM ETMCLAIMSET: Reserved Position  */
#define ETM_ETMCLAIMSET_Reserved_Msk                          (0x00ffffffUL << ETM_ETMCLAIMSET_Reserved_Pos)            /*!< ETM ETMCLAIMSET: Reserved Mask      */

// -------------------------------------  ETM_ETMCLAIMCLR  ----------------------------------------
#define ETM_ETMCLAIMCLR_ClaimTag_Pos                          0                                                         /*!< ETM ETMCLAIMCLR: ClaimTag Position  */
#define ETM_ETMCLAIMCLR_ClaimTag_Msk                          (0x000000ffUL << ETM_ETMCLAIMCLR_ClaimTag_Pos)            /*!< ETM ETMCLAIMCLR: ClaimTag Mask      */
#define ETM_ETMCLAIMCLR_Reserved_Pos                          8                                                         /*!< ETM ETMCLAIMCLR: Reserved Position  */
#define ETM_ETMCLAIMCLR_Reserved_Msk                          (0x00ffffffUL << ETM_ETMCLAIMCLR_Reserved_Pos)            /*!< ETM ETMCLAIMCLR: Reserved Mask      */

// ---------------------------------------  ETM_ETMLAR  -------------------------------------------
#define ETM_ETMLAR_KeyValue_Pos                               0                                                         /*!< ETM ETMLAR: KeyValue Position       */
#define ETM_ETMLAR_KeyValue_Msk                               (0xffffffffUL << ETM_ETMLAR_KeyValue_Pos)                 /*!< ETM ETMLAR: KeyValue Mask           */

// ---------------------------------------  ETM_ETMLSR  -------------------------------------------
#define ETM_ETMLSR_ETM_LockingImplemented_Pos                 0                                                         /*!< ETM ETMLSR: ETM_LockingImplemented Position */
#define ETM_ETMLSR_ETM_LockingImplemented_Msk                 (0x01UL << ETM_ETMLSR_ETM_LockingImplemented_Pos)         /*!< ETM ETMLSR: ETM_LockingImplemented Mask */
#define ETM_ETMLSR_ETM_Locked_Pos                             1                                                         /*!< ETM ETMLSR: ETM_Locked Position     */
#define ETM_ETMLSR_ETM_Locked_Msk                             (0x01UL << ETM_ETMLSR_ETM_Locked_Pos)                     /*!< ETM ETMLSR: ETM_Locked Mask         */
#define ETM_ETMLSR_ReadsAsZero_Pos                            2                                                         /*!< ETM ETMLSR: ReadsAsZero Position    */
#define ETM_ETMLSR_ReadsAsZero_Msk                            (0x01UL << ETM_ETMLSR_ReadsAsZero_Pos)                    /*!< ETM ETMLSR: ReadsAsZero Mask        */
#define ETM_ETMLSR_Reserved_Pos                               3                                                         /*!< ETM ETMLSR: Reserved Position       */
#define ETM_ETMLSR_Reserved_Msk                               (0x1fffffffUL << ETM_ETMLSR_Reserved_Pos)                 /*!< ETM ETMLSR: Reserved Mask           */

// ------------------------------------  ETM_ETMAUTHSTATUS  ---------------------------------------
#define ETM_ETMAUTHSTATUS_NonsecureInvasiveDebug_Pos          0                                                         /*!< ETM ETMAUTHSTATUS: NonsecureInvasiveDebug Position */
#define ETM_ETMAUTHSTATUS_NonsecureInvasiveDebug_Msk          (0x03UL << ETM_ETMAUTHSTATUS_NonsecureInvasiveDebug_Pos)  /*!< ETM ETMAUTHSTATUS: NonsecureInvasiveDebug Mask */
#define ETM_ETMAUTHSTATUS_NonsecureNonInvasiveDebug_Pos       2                                                         /*!< ETM ETMAUTHSTATUS: NonsecureNonInvasiveDebug Position */
#define ETM_ETMAUTHSTATUS_NonsecureNonInvasiveDebug_Msk       (0x03UL << ETM_ETMAUTHSTATUS_NonsecureNonInvasiveDebug_Pos)/*!< ETM ETMAUTHSTATUS: NonsecureNonInvasiveDebug Mask */
#define ETM_ETMAUTHSTATUS_SecureInvasiveDebug_Pos             4                                                         /*!< ETM ETMAUTHSTATUS: SecureInvasiveDebug Position */
#define ETM_ETMAUTHSTATUS_SecureInvasiveDebug_Msk             (0x03UL << ETM_ETMAUTHSTATUS_SecureInvasiveDebug_Pos)     /*!< ETM ETMAUTHSTATUS: SecureInvasiveDebug Mask */
#define ETM_ETMAUTHSTATUS_SecureNonInvasiveDebug_Pos          6                                                         /*!< ETM ETMAUTHSTATUS: SecureNonInvasiveDebug Position */
#define ETM_ETMAUTHSTATUS_SecureNonInvasiveDebug_Msk          (0x03UL << ETM_ETMAUTHSTATUS_SecureNonInvasiveDebug_Pos)  /*!< ETM ETMAUTHSTATUS: SecureNonInvasiveDebug Mask */
#define ETM_ETMAUTHSTATUS_Reserved_Pos                        8                                                         /*!< ETM ETMAUTHSTATUS: Reserved Position */
#define ETM_ETMAUTHSTATUS_Reserved_Msk                        (0x00ffffffUL << ETM_ETMAUTHSTATUS_Reserved_Pos)          /*!< ETM ETMAUTHSTATUS: Reserved Mask    */

// -------------------------------------  ETM_ETMDEVTYPE  -----------------------------------------
#define ETM_ETMDEVTYPE_MainType_Pos                           0                                                         /*!< ETM ETMDEVTYPE: MainType Position   */
#define ETM_ETMDEVTYPE_MainType_Msk                           (0x0fUL << ETM_ETMDEVTYPE_MainType_Pos)                   /*!< ETM ETMDEVTYPE: MainType Mask       */
#define ETM_ETMDEVTYPE_SubType_Pos                            4                                                         /*!< ETM ETMDEVTYPE: SubType Position    */
#define ETM_ETMDEVTYPE_SubType_Msk                            (0x0fUL << ETM_ETMDEVTYPE_SubType_Pos)                    /*!< ETM ETMDEVTYPE: SubType Mask        */
#define ETM_ETMDEVTYPE_Reserved_Pos                           8                                                         /*!< ETM ETMDEVTYPE: Reserved Position   */
#define ETM_ETMDEVTYPE_Reserved_Msk                           (0x00ffffffUL << ETM_ETMDEVTYPE_Reserved_Pos)             /*!< ETM ETMDEVTYPE: Reserved Mask       */

// --------------------------------------  ETM_ETMPIDR4  ------------------------------------------
#define ETM_ETMPIDR4_JEP106ContinuationCode_Pos               0                                                         /*!< ETM ETMPIDR4: JEP106ContinuationCode Position */
#define ETM_ETMPIDR4_JEP106ContinuationCode_Msk               (0x0fUL << ETM_ETMPIDR4_JEP106ContinuationCode_Pos)       /*!< ETM ETMPIDR4: JEP106ContinuationCode Mask */
#define ETM_ETMPIDR4_Bitfield_4KBcount_Pos                    4                                                         /*!< ETM ETMPIDR4: Bitfield_4KBcount Position */
#define ETM_ETMPIDR4_Bitfield_4KBcount_Msk                    (0x0fUL << ETM_ETMPIDR4_Bitfield_4KBcount_Pos)            /*!< ETM ETMPIDR4: Bitfield_4KBcount Mask */
#define ETM_ETMPIDR4_Reserved_Pos                             8                                                         /*!< ETM ETMPIDR4: Reserved Position     */
#define ETM_ETMPIDR4_Reserved_Msk                             (0x00ffffffUL << ETM_ETMPIDR4_Reserved_Pos)               /*!< ETM ETMPIDR4: Reserved Mask         */

// --------------------------------------  ETM_ETMPIDR5  ------------------------------------------
#define ETM_ETMPIDR5_ReservedForFutureUse_Pos                 0                                                         /*!< ETM ETMPIDR5: ReservedForFutureUse Position */
#define ETM_ETMPIDR5_ReservedForFutureUse_Msk                 (0x000000ffUL << ETM_ETMPIDR5_ReservedForFutureUse_Pos)   /*!< ETM ETMPIDR5: ReservedForFutureUse Mask */
#define ETM_ETMPIDR5_Reserved_Pos                             8                                                         /*!< ETM ETMPIDR5: Reserved Position     */
#define ETM_ETMPIDR5_Reserved_Msk                             (0x00ffffffUL << ETM_ETMPIDR5_Reserved_Pos)               /*!< ETM ETMPIDR5: Reserved Mask         */

// --------------------------------------  ETM_ETMPIDR6  ------------------------------------------
#define ETM_ETMPIDR6_ReservedForFutureUse_Pos                 0                                                         /*!< ETM ETMPIDR6: ReservedForFutureUse Position */
#define ETM_ETMPIDR6_ReservedForFutureUse_Msk                 (0x000000ffUL << ETM_ETMPIDR6_ReservedForFutureUse_Pos)   /*!< ETM ETMPIDR6: ReservedForFutureUse Mask */
#define ETM_ETMPIDR6_Reserved_Pos                             8                                                         /*!< ETM ETMPIDR6: Reserved Position     */
#define ETM_ETMPIDR6_Reserved_Msk                             (0x00ffffffUL << ETM_ETMPIDR6_Reserved_Pos)               /*!< ETM ETMPIDR6: Reserved Mask         */

// --------------------------------------  ETM_ETMPIDR7  ------------------------------------------
#define ETM_ETMPIDR7_ReservedForFutureUse_Pos                 0                                                         /*!< ETM ETMPIDR7: ReservedForFutureUse Position */
#define ETM_ETMPIDR7_ReservedForFutureUse_Msk                 (0x000000ffUL << ETM_ETMPIDR7_ReservedForFutureUse_Pos)   /*!< ETM ETMPIDR7: ReservedForFutureUse Mask */
#define ETM_ETMPIDR7_Reserved_Pos                             8                                                         /*!< ETM ETMPIDR7: Reserved Position     */
#define ETM_ETMPIDR7_Reserved_Msk                             (0x00ffffffUL << ETM_ETMPIDR7_Reserved_Pos)               /*!< ETM ETMPIDR7: Reserved Mask         */

// --------------------------------------  ETM_ETMPIDR0  ------------------------------------------
#define ETM_ETMPIDR0_PartNumber_7_0_Pos                       0                                                         /*!< ETM ETMPIDR0: PartNumber_7_0 Position */
#define ETM_ETMPIDR0_PartNumber_7_0_Msk                       (0x000000ffUL << ETM_ETMPIDR0_PartNumber_7_0_Pos)         /*!< ETM ETMPIDR0: PartNumber_7_0 Mask   */
#define ETM_ETMPIDR0_Reserved_Pos                             8                                                         /*!< ETM ETMPIDR0: Reserved Position     */
#define ETM_ETMPIDR0_Reserved_Msk                             (0x00ffffffUL << ETM_ETMPIDR0_Reserved_Pos)               /*!< ETM ETMPIDR0: Reserved Mask         */

// --------------------------------------  ETM_ETMPIDR1  ------------------------------------------
#define ETM_ETMPIDR1_PartNumber_11_8_Pos                      0                                                         /*!< ETM ETMPIDR1: PartNumber_11_8 Position */
#define ETM_ETMPIDR1_PartNumber_11_8_Msk                      (0x0fUL << ETM_ETMPIDR1_PartNumber_11_8_Pos)              /*!< ETM ETMPIDR1: PartNumber_11_8 Mask  */
#define ETM_ETMPIDR1_JEP106IdentityCode_3_0_Pos               4                                                         /*!< ETM ETMPIDR1: JEP106IdentityCode_3_0 Position */
#define ETM_ETMPIDR1_JEP106IdentityCode_3_0_Msk               (0x0fUL << ETM_ETMPIDR1_JEP106IdentityCode_3_0_Pos)       /*!< ETM ETMPIDR1: JEP106IdentityCode_3_0 Mask */
#define ETM_ETMPIDR1_Reserved_Pos                             8                                                         /*!< ETM ETMPIDR1: Reserved Position     */
#define ETM_ETMPIDR1_Reserved_Msk                             (0x00ffffffUL << ETM_ETMPIDR1_Reserved_Pos)               /*!< ETM ETMPIDR1: Reserved Mask         */

// --------------------------------------  ETM_ETMPIDR2  ------------------------------------------
#define ETM_ETMPIDR2_JEP106IdentityCode_6_4_Pos               0                                                         /*!< ETM ETMPIDR2: JEP106IdentityCode_6_4 Position */
#define ETM_ETMPIDR2_JEP106IdentityCode_6_4_Msk               (0x0fUL << ETM_ETMPIDR2_JEP106IdentityCode_6_4_Pos)       /*!< ETM ETMPIDR2: JEP106IdentityCode_6_4 Mask */
#define ETM_ETMPIDR2_Always1_Pos                              4                                                         /*!< ETM ETMPIDR2: Always1 Position      */
#define ETM_ETMPIDR2_Always1_Msk                              (0x01UL << ETM_ETMPIDR2_Always1_Pos)                      /*!< ETM ETMPIDR2: Always1 Mask          */
#define ETM_ETMPIDR2_Revision_Pos                             5                                                         /*!< ETM ETMPIDR2: Revision Position     */
#define ETM_ETMPIDR2_Revision_Msk                             (0x07UL << ETM_ETMPIDR2_Revision_Pos)                     /*!< ETM ETMPIDR2: Revision Mask         */
#define ETM_ETMPIDR2_Reserved_Pos                             8                                                         /*!< ETM ETMPIDR2: Reserved Position     */
#define ETM_ETMPIDR2_Reserved_Msk                             (0x00ffffffUL << ETM_ETMPIDR2_Reserved_Pos)               /*!< ETM ETMPIDR2: Reserved Mask         */

// --------------------------------------  ETM_ETMPIDR3  ------------------------------------------
#define ETM_ETMPIDR3_CustomerModified_Pos                     0                                                         /*!< ETM ETMPIDR3: CustomerModified Position */
#define ETM_ETMPIDR3_CustomerModified_Msk                     (0x0fUL << ETM_ETMPIDR3_CustomerModified_Pos)             /*!< ETM ETMPIDR3: CustomerModified Mask */
#define ETM_ETMPIDR3_RevAnd_Pos                               4                                                         /*!< ETM ETMPIDR3: RevAnd Position       */
#define ETM_ETMPIDR3_RevAnd_Msk                               (0x0fUL << ETM_ETMPIDR3_RevAnd_Pos)                       /*!< ETM ETMPIDR3: RevAnd Mask           */
#define ETM_ETMPIDR3_Reserved_Pos                             8                                                         /*!< ETM ETMPIDR3: Reserved Position     */
#define ETM_ETMPIDR3_Reserved_Msk                             (0x00ffffffUL << ETM_ETMPIDR3_Reserved_Pos)               /*!< ETM ETMPIDR3: Reserved Mask         */

// --------------------------------------  ETM_ETMCIDR0  ------------------------------------------
#define ETM_ETMCIDR0_CID0_Pos                                 0                                                         /*!< ETM ETMCIDR0: CID0 Position         */
#define ETM_ETMCIDR0_CID0_Msk                                 (0x000000ffUL << ETM_ETMCIDR0_CID0_Pos)                   /*!< ETM ETMCIDR0: CID0 Mask             */
#define ETM_ETMCIDR0_Reserved_Pos                             8                                                         /*!< ETM ETMCIDR0: Reserved Position     */
#define ETM_ETMCIDR0_Reserved_Msk                             (0x00ffffffUL << ETM_ETMCIDR0_Reserved_Pos)               /*!< ETM ETMCIDR0: Reserved Mask         */

// --------------------------------------  ETM_ETMCIDR1  ------------------------------------------
#define ETM_ETMCIDR1_CID1_Pos                                 0                                                         /*!< ETM ETMCIDR1: CID1 Position         */
#define ETM_ETMCIDR1_CID1_Msk                                 (0x000000ffUL << ETM_ETMCIDR1_CID1_Pos)                   /*!< ETM ETMCIDR1: CID1 Mask             */
#define ETM_ETMCIDR1_Reserved_Pos                             8                                                         /*!< ETM ETMCIDR1: Reserved Position     */
#define ETM_ETMCIDR1_Reserved_Msk                             (0x00ffffffUL << ETM_ETMCIDR1_Reserved_Pos)               /*!< ETM ETMCIDR1: Reserved Mask         */

// --------------------------------------  ETM_ETMCIDR2  ------------------------------------------
#define ETM_ETMCIDR2_CID2_Pos                                 0                                                         /*!< ETM ETMCIDR2: CID2 Position         */
#define ETM_ETMCIDR2_CID2_Msk                                 (0x000000ffUL << ETM_ETMCIDR2_CID2_Pos)                   /*!< ETM ETMCIDR2: CID2 Mask             */
#define ETM_ETMCIDR2_Reserved_Pos                             8                                                         /*!< ETM ETMCIDR2: Reserved Position     */
#define ETM_ETMCIDR2_Reserved_Msk                             (0x00ffffffUL << ETM_ETMCIDR2_Reserved_Pos)               /*!< ETM ETMCIDR2: Reserved Mask         */

// --------------------------------------  ETM_ETMCIDR3  ------------------------------------------
#define ETM_ETMCIDR3_CID3_Pos                                 0                                                         /*!< ETM ETMCIDR3: CID3 Position         */
#define ETM_ETMCIDR3_CID3_Msk                                 (0x000000ffUL << ETM_ETMCIDR3_CID3_Pos)                   /*!< ETM ETMCIDR3: CID3 Mask             */
#define ETM_ETMCIDR3_Reserved_Pos                             8                                                         /*!< ETM ETMCIDR3: Reserved Position     */
#define ETM_ETMCIDR3_Reserved_Msk                             (0x00ffffffUL << ETM_ETMCIDR3_Reserved_Pos)               /*!< ETM ETMCIDR3: Reserved Mask         */



/***   FCE Bit Fileds *******************/
/***************************************************************************/


/* FCE_CLC  =  Clock Control Register*/
#define   FCE_CLC_DISR_Pos  	(0U)
#define   FCE_CLC_DISR_Msk  	(0x00000001U  << FCE_CLC_DISR_Pos)

#define   FCE_CLC_DISS_Pos  	(1U)
#define   FCE_CLC_DISS_Msk  	(0x00000001U  << FCE_CLC_DISS_Pos)

/* FCE_ID  =  Module Identification Register*/
#define   FCE_ID_MOD_REV_Pos  	(0U)
#define   FCE_ID_MOD_REV_Msk  	(0x000000FFU  << FCE_ID_MOD_REV_Pos)

#define   FCE_ID_MOD_TYPE_Pos  	(8U)
#define   FCE_ID_MOD_TYPE_Msk  	(0x000000FFU  << FCE_ID_MOD_TYPE_Pos)

#define   FCE_ID_MOD_NUMBER_Pos  	(16U)
#define   FCE_ID_MOD_NUMBER_Msk  	(0x0000FFFFU  << FCE_ID_MOD_NUMBER_Pos)

/* FCE_KE_IRx  =  Input Register 0*/
#define   FCE_KE_IR_IR_Pos  	(0U)
#define   FCE_KE_IR_IR_Msk  	(0xFFFFFFFFU  << FCE_KE_IR_IR_Pos)

/* FCE_KE_RESx  =  CRC Result Register 0*/
#define   FCE_KE_RES_RES_Pos  	(0U)
#define   FCE_KE_RES_RES_Msk  	(0xFFFFFFFFU  << FCE_KE_RES_RES_Pos)

/* FCE_KE_CFGx  =  CRC Configuration Register 0*/
#define   FCE_KE_CFG_CMI_Pos  	(0U)
#define   FCE_KE_CFG_CMI_Msk  	(0x00000001U  << FCE_KE_CFG_CMI_Pos)

#define   FCE_KE_CFG_CEI_Pos  	(1U)
#define   FCE_KE_CFG_CEI_Msk  	(0x00000001U  << FCE_KE_CFG_CEI_Pos)

#define   FCE_KE_CFG_LEI_Pos  	(2U)
#define   FCE_KE_CFG_LEI_Msk  	(0x00000001U  << FCE_KE_CFG_LEI_Pos)

#define   FCE_KE_CFG_BEI_Pos  	(3U)
#define   FCE_KE_CFG_BEI_Msk  	(0x00000001U  << FCE_KE_CFG_BEI_Pos)

#define   FCE_KE_CFG_CCE_Pos  	(4U)
#define   FCE_KE_CFG_CCE_Msk  	(0x00000001U  << FCE_KE_CFG_CCE_Pos)

#define   FCE_KE_CFG_ALR_Pos  	(5U)
#define   FCE_KE_CFG_ALR_Msk  	(0x00000001U  << FCE_KE_CFG_ALR_Pos)

#define   FCE_KE_CFG_REFIN_Pos  	(8U)
#define   FCE_KE_CFG_REFIN_Msk  	(0x00000001U  << FCE_KE_CFG_REFIN_Pos)

#define   FCE_KE_CFG_REFOUT_Pos  	(9U)
#define   FCE_KE_CFG_REFOUT_Msk  	(0x00000001U  << FCE_KE_CFG_REFOUT_Pos)

#define   FCE_KE_CFG_XSEL_Pos  	(10U)
#define   FCE_KE_CFG_XSEL_Msk  	(0x00000001U  << FCE_KE_CFG_XSEL_Pos)

/* FCE_KE_STSx  =  CRC Status Register 0*/
#define   FCE_KE_STS_CMF_Pos  	(0U)
#define   FCE_KE_STS_CMF_Msk  	(0x00000001U  << FCE_KE_STS_CMF_Pos)

#define   FCE_KE_STS_CEF_Pos  	(1U)
#define   FCE_KE_STS_CEF_Msk  	(0x00000001U  << FCE_KE_STS_CEF_Pos)

#define   FCE_KE_STS_LEF_Pos  	(2U)
#define   FCE_KE_STS_LEF_Msk  	(0x00000001U  << FCE_KE_STS_LEF_Pos)

#define   FCE_KE_STS_BEF_Pos  	(3U)
#define   FCE_KE_STS_BEF_Msk  	(0x00000001U  << FCE_KE_STS_BEF_Pos)

/* FCE_KE_LENGTHx  =  CRC Length Register 0*/
#define   FCE_KE_LENGTH_LENGTH_Pos  	(0U)
#define   FCE_KE_LENGTH_LENGTH_Msk  	(0x0000FFFFU  << FCE_KE_LENGTH_LENGTH_Pos)

/* FCE_KE_CHECKx  =  CRC Check Register 0*/
#define   FCE_KE_CHECK_CHECK_Pos  	(0U)
#define   FCE_KE_CHECK_CHECK_Msk  	(0xFFFFFFFFU  << FCE_KE_CHECK_CHECK_Pos)

/* FCE_KE_CRCx  =  CRC Register 0*/
#define   FCE_KE_CRC_CRC_Pos  	(0U)
#define   FCE_KE_CRC_CRC_Msk  	(0xFFFFFFFFU  << FCE_KE_CRC_CRC_Pos)

/* FCE_KE_CTRx  =  CRC Test Register 0*/
#define   FCE_KE_CTR_FCM_Pos  	(0U)
#define   FCE_KE_CTR_FCM_Msk  	(0x00000001U  << FCE_KE_CTR_FCM_Pos)

#define   FCE_KE_CTR_FRM_CFG_Pos  	(1U)
#define   FCE_KE_CTR_FRM_CFG_Msk  	(0x00000001U  << FCE_KE_CTR_FRM_CFG_Pos)

#define   FCE_KE_CTR_FRM_CHECK_Pos  	(2U)
#define   FCE_KE_CTR_FRM_CHECK_Msk  	(0x00000001U  << FCE_KE_CTR_FRM_CHECK_Pos)



/***   FLASH0 Bit Fileds *******************/
/***************************************************************************/


/* FLASH_ID  =  Flash Module Identification Register*/
#define   FLASH_ID_MOD_REV_Pos  	(0U)
#define   FLASH_ID_MOD_REV_Msk  	(0x000000FFU  << FLASH_ID_MOD_REV_Pos)

#define   FLASH_ID_MOD_TYPE_Pos  	(8U)
#define   FLASH_ID_MOD_TYPE_Msk  	(0x000000FFU  << FLASH_ID_MOD_TYPE_Pos)

#define   FLASH_ID_MOD_NUMBER_Pos  	(16U)
#define   FLASH_ID_MOD_NUMBER_Msk  	(0x0000FFFFU  << FLASH_ID_MOD_NUMBER_Pos)

/* FLASH_FSR  =  Flash Status Register*/
#define   FLASH_FSR_PBUSY_Pos  	(0U)
#define   FLASH_FSR_PBUSY_Msk  	(0x00000001U  << FLASH_FSR_PBUSY_Pos)

#define   FLASH_FSR_FABUSY_Pos  	(1U)
#define   FLASH_FSR_FABUSY_Msk  	(0x00000001U  << FLASH_FSR_FABUSY_Pos)

#define   FLASH_FSR_PROG_Pos  	(4U)
#define   FLASH_FSR_PROG_Msk  	(0x00000001U  << FLASH_FSR_PROG_Pos)

#define   FLASH_FSR_ERASE_Pos  	(5U)
#define   FLASH_FSR_ERASE_Msk  	(0x00000001U  << FLASH_FSR_ERASE_Pos)

#define   FLASH_FSR_PFPAGE_Pos  	(6U)
#define   FLASH_FSR_PFPAGE_Msk  	(0x00000001U  << FLASH_FSR_PFPAGE_Pos)

#define   FLASH_FSR_PFOPER_Pos  	(8U)
#define   FLASH_FSR_PFOPER_Msk  	(0x00000001U  << FLASH_FSR_PFOPER_Pos)

#define   FLASH_FSR_SQER_Pos  	(10U)
#define   FLASH_FSR_SQER_Msk  	(0x00000001U  << FLASH_FSR_SQER_Pos)

#define   FLASH_FSR_PROER_Pos  	(11U)
#define   FLASH_FSR_PROER_Msk  	(0x00000001U  << FLASH_FSR_PROER_Pos)

#define   FLASH_FSR_PFSBER_Pos  	(12U)
#define   FLASH_FSR_PFSBER_Msk  	(0x00000001U  << FLASH_FSR_PFSBER_Pos)

#define   FLASH_FSR_PFDBER_Pos  	(14U)
#define   FLASH_FSR_PFDBER_Msk  	(0x00000001U  << FLASH_FSR_PFDBER_Pos)

#define   FLASH_FSR_PROIN_Pos  	(16U)
#define   FLASH_FSR_PROIN_Msk  	(0x00000001U  << FLASH_FSR_PROIN_Pos)

#define   FLASH_FSR_RPROIN_Pos  	(18U)
#define   FLASH_FSR_RPROIN_Msk  	(0x00000001U  << FLASH_FSR_RPROIN_Pos)

#define   FLASH_FSR_RPRODIS_Pos  	(19U)
#define   FLASH_FSR_RPRODIS_Msk  	(0x00000001U  << FLASH_FSR_RPRODIS_Pos)

#define   FLASH_FSR_WPROIN0_Pos  	(21U)
#define   FLASH_FSR_WPROIN0_Msk  	(0x00000001U  << FLASH_FSR_WPROIN0_Pos)

#define   FLASH_FSR_WPROIN1_Pos  	(22U)
#define   FLASH_FSR_WPROIN1_Msk  	(0x00000001U  << FLASH_FSR_WPROIN1_Pos)

#define   FLASH_FSR_WPROIN2_Pos  	(23U)
#define   FLASH_FSR_WPROIN2_Msk  	(0x00000001U  << FLASH_FSR_WPROIN2_Pos)

#define   FLASH_FSR_WPRODIS0_Pos  	(25U)
#define   FLASH_FSR_WPRODIS0_Msk  	(0x00000001U  << FLASH_FSR_WPRODIS0_Pos)

#define   FLASH_FSR_WPRODIS1_Pos  	(26U)
#define   FLASH_FSR_WPRODIS1_Msk  	(0x00000001U  << FLASH_FSR_WPRODIS1_Pos)

#define   FLASH_FSR_SLM_Pos  	(28U)
#define   FLASH_FSR_SLM_Msk  	(0x00000001U  << FLASH_FSR_SLM_Pos)

#define   FLASH_FSR_X_Pos  	(30U)
#define   FLASH_FSR_X_Msk  	(0x00000001U  << FLASH_FSR_X_Pos)

#define   FLASH_FSR_VER_Pos  	(31U)
#define   FLASH_FSR_VER_Msk  	(0x00000001U  << FLASH_FSR_VER_Pos)

/* FLASH_FCON  =  Flash Configuration Register*/
#define   FLASH_FCON_WSPFLASH_Pos  	(0U)
#define   FLASH_FCON_WSPFLASH_Msk  	(0x0000000FU  << FLASH_FCON_WSPFLASH_Pos)

#define   FLASH_FCON_WSECPF_Pos  	(4U)
#define   FLASH_FCON_WSECPF_Msk  	(0x00000001U  << FLASH_FCON_WSECPF_Pos)

#define   FLASH_FCON_IDLE_Pos  	(13U)
#define   FLASH_FCON_IDLE_Msk  	(0x00000001U  << FLASH_FCON_IDLE_Pos)

#define   FLASH_FCON_ESLDIS_Pos  	(14U)
#define   FLASH_FCON_ESLDIS_Msk  	(0x00000001U  << FLASH_FCON_ESLDIS_Pos)

#define   FLASH_FCON_SLEEP_Pos  	(15U)
#define   FLASH_FCON_SLEEP_Msk  	(0x00000001U  << FLASH_FCON_SLEEP_Pos)

#define   FLASH_FCON_RPA_Pos  	(16U)
#define   FLASH_FCON_RPA_Msk  	(0x00000001U  << FLASH_FCON_RPA_Pos)

#define   FLASH_FCON_DCF_Pos  	(17U)
#define   FLASH_FCON_DCF_Msk  	(0x00000001U  << FLASH_FCON_DCF_Pos)

#define   FLASH_FCON_DDF_Pos  	(18U)
#define   FLASH_FCON_DDF_Msk  	(0x00000001U  << FLASH_FCON_DDF_Pos)

#define   FLASH_FCON_VOPERM_Pos  	(24U)
#define   FLASH_FCON_VOPERM_Msk  	(0x00000001U  << FLASH_FCON_VOPERM_Pos)

#define   FLASH_FCON_SQERM_Pos  	(25U)
#define   FLASH_FCON_SQERM_Msk  	(0x00000001U  << FLASH_FCON_SQERM_Pos)

#define   FLASH_FCON_PROERM_Pos  	(26U)
#define   FLASH_FCON_PROERM_Msk  	(0x00000001U  << FLASH_FCON_PROERM_Pos)

#define   FLASH_FCON_PFSBERM_Pos  	(27U)
#define   FLASH_FCON_PFSBERM_Msk  	(0x00000001U  << FLASH_FCON_PFSBERM_Pos)

#define   FLASH_FCON_PFDBERM_Pos  	(29U)
#define   FLASH_FCON_PFDBERM_Msk  	(0x00000001U  << FLASH_FCON_PFDBERM_Pos)

#define   FLASH_FCON_EOBM_Pos  	(31U)
#define   FLASH_FCON_EOBM_Msk  	(0x00000001U  << FLASH_FCON_EOBM_Pos)

/* FLASH_MARP  =  Margin Control Register PFLASH*/
#define   FLASH_MARP_MARGIN_Pos  	(0U)
#define   FLASH_MARP_MARGIN_Msk  	(0x0000000FU  << FLASH_MARP_MARGIN_Pos)

#define   FLASH_MARP_TRAPDIS_Pos  	(15U)
#define   FLASH_MARP_TRAPDIS_Msk  	(0x00000001U  << FLASH_MARP_TRAPDIS_Pos)

/* FLASH_PROCON0  =  Flash Protection Configuration Register User 0*/
#define   FLASH_PROCON0_S0L_Pos  	(0U)
#define   FLASH_PROCON0_S0L_Msk  	(0x00000001U  << FLASH_PROCON0_S0L_Pos)

#define   FLASH_PROCON0_S1L_Pos  	(1U)
#define   FLASH_PROCON0_S1L_Msk  	(0x00000001U  << FLASH_PROCON0_S1L_Pos)

#define   FLASH_PROCON0_S2L_Pos  	(2U)
#define   FLASH_PROCON0_S2L_Msk  	(0x00000001U  << FLASH_PROCON0_S2L_Pos)

#define   FLASH_PROCON0_S3L_Pos  	(3U)
#define   FLASH_PROCON0_S3L_Msk  	(0x00000001U  << FLASH_PROCON0_S3L_Pos)

#define   FLASH_PROCON0_S4L_Pos  	(4U)
#define   FLASH_PROCON0_S4L_Msk  	(0x00000001U  << FLASH_PROCON0_S4L_Pos)

#define   FLASH_PROCON0_S5L_Pos  	(5U)
#define   FLASH_PROCON0_S5L_Msk  	(0x00000001U  << FLASH_PROCON0_S5L_Pos)

#define   FLASH_PROCON0_S6L_Pos  	(6U)
#define   FLASH_PROCON0_S6L_Msk  	(0x00000001U  << FLASH_PROCON0_S6L_Pos)

#define   FLASH_PROCON0_S7L_Pos  	(7U)
#define   FLASH_PROCON0_S7L_Msk  	(0x00000001U  << FLASH_PROCON0_S7L_Pos)

#define   FLASH_PROCON0_S8L_Pos  	(8U)
#define   FLASH_PROCON0_S8L_Msk  	(0x00000001U  << FLASH_PROCON0_S8L_Pos)

#define   FLASH_PROCON0_S9L_Pos  	(9U)
#define   FLASH_PROCON0_S9L_Msk  	(0x00000001U  << FLASH_PROCON0_S9L_Pos)

#define   FLASH_PROCON0_S10_S11L_Pos  	(10U)
#define   FLASH_PROCON0_S10_S11L_Msk  	(0x00000001U  << FLASH_PROCON0_S10_S11L_Pos)

#define   FLASH_PROCON0_RPRO_Pos  	(15U)
#define   FLASH_PROCON0_RPRO_Msk  	(0x00000001U  << FLASH_PROCON0_RPRO_Pos)

/* FLASH_PROCON1  =  Flash Protection Configuration Register User 1*/
#define   FLASH_PROCON1_S0L_Pos  	(0U)
#define   FLASH_PROCON1_S0L_Msk  	(0x00000001U  << FLASH_PROCON1_S0L_Pos)

#define   FLASH_PROCON1_S1L_Pos  	(1U)
#define   FLASH_PROCON1_S1L_Msk  	(0x00000001U  << FLASH_PROCON1_S1L_Pos)

#define   FLASH_PROCON1_S2L_Pos  	(2U)
#define   FLASH_PROCON1_S2L_Msk  	(0x00000001U  << FLASH_PROCON1_S2L_Pos)

#define   FLASH_PROCON1_S3L_Pos  	(3U)
#define   FLASH_PROCON1_S3L_Msk  	(0x00000001U  << FLASH_PROCON1_S3L_Pos)

#define   FLASH_PROCON1_S4L_Pos  	(4U)
#define   FLASH_PROCON1_S4L_Msk  	(0x00000001U  << FLASH_PROCON1_S4L_Pos)

#define   FLASH_PROCON1_S5L_Pos  	(5U)
#define   FLASH_PROCON1_S5L_Msk  	(0x00000001U  << FLASH_PROCON1_S5L_Pos)

#define   FLASH_PROCON1_S6L_Pos  	(6U)
#define   FLASH_PROCON1_S6L_Msk  	(0x00000001U  << FLASH_PROCON1_S6L_Pos)

#define   FLASH_PROCON1_S7L_Pos  	(7U)
#define   FLASH_PROCON1_S7L_Msk  	(0x00000001U  << FLASH_PROCON1_S7L_Pos)

#define   FLASH_PROCON1_S8L_Pos  	(8U)
#define   FLASH_PROCON1_S8L_Msk  	(0x00000001U  << FLASH_PROCON1_S8L_Pos)

#define   FLASH_PROCON1_S9L_Pos  	(9U)
#define   FLASH_PROCON1_S9L_Msk  	(0x00000001U  << FLASH_PROCON1_S9L_Pos)

#define   FLASH_PROCON1_S10_S11L_Pos  	(10U)
#define   FLASH_PROCON1_S10_S11L_Msk  	(0x00000001U  << FLASH_PROCON1_S10_S11L_Pos)

/* FLASH_PROCON2  =  Flash Protection Configuration Register User 2*/
#define   FLASH_PROCON2_S0ROM_Pos  	(0U)
#define   FLASH_PROCON2_S0ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S0ROM_Pos)

#define   FLASH_PROCON2_S1ROM_Pos  	(1U)
#define   FLASH_PROCON2_S1ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S1ROM_Pos)

#define   FLASH_PROCON2_S2ROM_Pos  	(2U)
#define   FLASH_PROCON2_S2ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S2ROM_Pos)

#define   FLASH_PROCON2_S3ROM_Pos  	(3U)
#define   FLASH_PROCON2_S3ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S3ROM_Pos)

#define   FLASH_PROCON2_S4ROM_Pos  	(4U)
#define   FLASH_PROCON2_S4ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S4ROM_Pos)

#define   FLASH_PROCON2_S5ROM_Pos  	(5U)
#define   FLASH_PROCON2_S5ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S5ROM_Pos)

#define   FLASH_PROCON2_S6ROM_Pos  	(6U)
#define   FLASH_PROCON2_S6ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S6ROM_Pos)

#define   FLASH_PROCON2_S7ROM_Pos  	(7U)
#define   FLASH_PROCON2_S7ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S7ROM_Pos)

#define   FLASH_PROCON2_S8ROM_Pos  	(8U)
#define   FLASH_PROCON2_S8ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S8ROM_Pos)

#define   FLASH_PROCON2_S9ROM_Pos  	(9U)
#define   FLASH_PROCON2_S9ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S9ROM_Pos)

#define   FLASH_PROCON2_S10_S11ROM_Pos  	(10U)
#define   FLASH_PROCON2_S10_S11ROM_Msk  	(0x00000001U  << FLASH_PROCON2_S10_S11ROM_Pos)



/***   FPB Bit Fileds *******************/
/***************************************************************************/


// ---------------------------------------  FPB_FP_CTRL  ------------------------------------------
#define FPB_FP_CTRL_ENABLE_Pos                                0                                                         /*!< FPB FP_CTRL: ENABLE Position        */
#define FPB_FP_CTRL_ENABLE_Msk                                (0x01UL << FPB_FP_CTRL_ENABLE_Pos)                        /*!< FPB FP_CTRL: ENABLE Mask            */
#define FPB_FP_CTRL_KEY_Pos                                   1                                                         /*!< FPB FP_CTRL: KEY Position           */
#define FPB_FP_CTRL_KEY_Msk                                   (0x01UL << FPB_FP_CTRL_KEY_Pos)                           /*!< FPB FP_CTRL: KEY Mask               */
#define FPB_FP_CTRL_Reserved_Pos                              2                                                         /*!< FPB FP_CTRL: Reserved Position      */
#define FPB_FP_CTRL_Reserved_Msk                              (0x03UL << FPB_FP_CTRL_Reserved_Pos)                      /*!< FPB FP_CTRL: Reserved Mask          */
#define FPB_FP_CTRL_NUM_CODE_3_0_Pos                          4                                                         /*!< FPB FP_CTRL: NUM_CODE_3_0 Position  */
#define FPB_FP_CTRL_NUM_CODE_3_0_Msk                          (0x0fUL << FPB_FP_CTRL_NUM_CODE_3_0_Pos)                  /*!< FPB FP_CTRL: NUM_CODE_3_0 Mask      */
#define FPB_FP_CTRL_NUM_LIT_Pos                               8                                                         /*!< FPB FP_CTRL: NUM_LIT Position       */
#define FPB_FP_CTRL_NUM_LIT_Msk                               (0x0fUL << FPB_FP_CTRL_NUM_LIT_Pos)                       /*!< FPB FP_CTRL: NUM_LIT Mask           */
#define FPB_FP_CTRL_NUM_CODE_6_4_Pos                          12                                                        /*!< FPB FP_CTRL: NUM_CODE_6_4 Position  */
#define FPB_FP_CTRL_NUM_CODE_6_4_Msk                          (0x07UL << FPB_FP_CTRL_NUM_CODE_6_4_Pos)                  /*!< FPB FP_CTRL: NUM_CODE_6_4 Mask      */
#define FPB_FP_CTRL_Reserved1_Pos                             15                                                        /*!< FPB FP_CTRL: Reserved1 Position     */
#define FPB_FP_CTRL_Reserved1_Msk                             (0x0001ffffUL << FPB_FP_CTRL_Reserved1_Pos)               /*!< FPB FP_CTRL: Reserved1 Mask         */

// --------------------------------------  FPB_FP_REMAP  ------------------------------------------
#define FPB_FP_REMAP_Reserved_Pos                             0                                                         /*!< FPB FP_REMAP: Reserved Position     */
#define FPB_FP_REMAP_Reserved_Msk                             (0x1fUL << FPB_FP_REMAP_Reserved_Pos)                     /*!< FPB FP_REMAP: Reserved Mask         */
#define FPB_FP_REMAP_REMAP_Pos                                5                                                         /*!< FPB FP_REMAP: REMAP Position        */
#define FPB_FP_REMAP_REMAP_Msk                                (0x00ffffffUL << FPB_FP_REMAP_REMAP_Pos)                  /*!< FPB FP_REMAP: REMAP Mask            */
#define FPB_FP_REMAP_RMPSPT_Pos                               29                                                        /*!< FPB FP_REMAP: RMPSPT Position       */
#define FPB_FP_REMAP_RMPSPT_Msk                               (0x01UL << FPB_FP_REMAP_RMPSPT_Pos)                       /*!< FPB FP_REMAP: RMPSPT Mask           */
#define FPB_FP_REMAP_Reserved1_Pos                            30                                                        /*!< FPB FP_REMAP: Reserved1 Position    */
#define FPB_FP_REMAP_Reserved1_Msk                            (0x03UL << FPB_FP_REMAP_Reserved1_Pos)                    /*!< FPB FP_REMAP: Reserved1 Mask        */

// --------------------------------------  FPB_FP_COMP0  ------------------------------------------
#define FPB_FP_COMP0_ENABLE_Pos                               0                                                         /*!< FPB FP_COMP0: ENABLE Position       */
#define FPB_FP_COMP0_ENABLE_Msk                               (0x01UL << FPB_FP_COMP0_ENABLE_Pos)                       /*!< FPB FP_COMP0: ENABLE Mask           */
#define FPB_FP_COMP0_Reserved_Pos                             1                                                         /*!< FPB FP_COMP0: Reserved Position     */
#define FPB_FP_COMP0_Reserved_Msk                             (0x01UL << FPB_FP_COMP0_Reserved_Pos)                     /*!< FPB FP_COMP0: Reserved Mask         */
#define FPB_FP_COMP0_COMP_Pos                                 2                                                         /*!< FPB FP_COMP0: COMP Position         */
#define FPB_FP_COMP0_COMP_Msk                                 (0x07ffffffUL << FPB_FP_COMP0_COMP_Pos)                   /*!< FPB FP_COMP0: COMP Mask             */
#define FPB_FP_COMP0_Reserved1_Pos                            29                                                        /*!< FPB FP_COMP0: Reserved1 Position    */
#define FPB_FP_COMP0_Reserved1_Msk                            (0x01UL << FPB_FP_COMP0_Reserved1_Pos)                    /*!< FPB FP_COMP0: Reserved1 Mask        */
#define FPB_FP_COMP0_REPLACE_Pos                              30                                                        /*!< FPB FP_COMP0: REPLACE Position      */
#define FPB_FP_COMP0_REPLACE_Msk                              (0x03UL << FPB_FP_COMP0_REPLACE_Pos)                      /*!< FPB FP_COMP0: REPLACE Mask          */

// --------------------------------------  FPB_FP_COMP1  ------------------------------------------
#define FPB_FP_COMP1_ENABLE_Pos                               0                                                         /*!< FPB FP_COMP1: ENABLE Position       */
#define FPB_FP_COMP1_ENABLE_Msk                               (0x01UL << FPB_FP_COMP1_ENABLE_Pos)                       /*!< FPB FP_COMP1: ENABLE Mask           */
#define FPB_FP_COMP1_Reserved_Pos                             1                                                         /*!< FPB FP_COMP1: Reserved Position     */
#define FPB_FP_COMP1_Reserved_Msk                             (0x01UL << FPB_FP_COMP1_Reserved_Pos)                     /*!< FPB FP_COMP1: Reserved Mask         */
#define FPB_FP_COMP1_COMP_Pos                                 2                                                         /*!< FPB FP_COMP1: COMP Position         */
#define FPB_FP_COMP1_COMP_Msk                                 (0x07ffffffUL << FPB_FP_COMP1_COMP_Pos)                   /*!< FPB FP_COMP1: COMP Mask             */
#define FPB_FP_COMP1_Reserved1_Pos                            29                                                        /*!< FPB FP_COMP1: Reserved1 Position    */
#define FPB_FP_COMP1_Reserved1_Msk                            (0x01UL << FPB_FP_COMP1_Reserved1_Pos)                    /*!< FPB FP_COMP1: Reserved1 Mask        */
#define FPB_FP_COMP1_REPLACE_Pos                              30                                                        /*!< FPB FP_COMP1: REPLACE Position      */
#define FPB_FP_COMP1_REPLACE_Msk                              (0x03UL << FPB_FP_COMP1_REPLACE_Pos)                      /*!< FPB FP_COMP1: REPLACE Mask          */

// --------------------------------------  FPB_FP_COMP2  ------------------------------------------
#define FPB_FP_COMP2_ENABLE_Pos                               0                                                         /*!< FPB FP_COMP2: ENABLE Position       */
#define FPB_FP_COMP2_ENABLE_Msk                               (0x01UL << FPB_FP_COMP2_ENABLE_Pos)                       /*!< FPB FP_COMP2: ENABLE Mask           */
#define FPB_FP_COMP2_Reserved_Pos                             1                                                         /*!< FPB FP_COMP2: Reserved Position     */
#define FPB_FP_COMP2_Reserved_Msk                             (0x01UL << FPB_FP_COMP2_Reserved_Pos)                     /*!< FPB FP_COMP2: Reserved Mask         */
#define FPB_FP_COMP2_COMP_Pos                                 2                                                         /*!< FPB FP_COMP2: COMP Position         */
#define FPB_FP_COMP2_COMP_Msk                                 (0x07ffffffUL << FPB_FP_COMP2_COMP_Pos)                   /*!< FPB FP_COMP2: COMP Mask             */
#define FPB_FP_COMP2_Reserved1_Pos                            29                                                        /*!< FPB FP_COMP2: Reserved1 Position    */
#define FPB_FP_COMP2_Reserved1_Msk                            (0x01UL << FPB_FP_COMP2_Reserved1_Pos)                    /*!< FPB FP_COMP2: Reserved1 Mask        */
#define FPB_FP_COMP2_REPLACE_Pos                              30                                                        /*!< FPB FP_COMP2: REPLACE Position      */
#define FPB_FP_COMP2_REPLACE_Msk                              (0x03UL << FPB_FP_COMP2_REPLACE_Pos)                      /*!< FPB FP_COMP2: REPLACE Mask          */

// --------------------------------------  FPB_FP_COMP3  ------------------------------------------
#define FPB_FP_COMP3_ENABLE_Pos                               0                                                         /*!< FPB FP_COMP3: ENABLE Position       */
#define FPB_FP_COMP3_ENABLE_Msk                               (0x01UL << FPB_FP_COMP3_ENABLE_Pos)                       /*!< FPB FP_COMP3: ENABLE Mask           */
#define FPB_FP_COMP3_Reserved_Pos                             1                                                         /*!< FPB FP_COMP3: Reserved Position     */
#define FPB_FP_COMP3_Reserved_Msk                             (0x01UL << FPB_FP_COMP3_Reserved_Pos)                     /*!< FPB FP_COMP3: Reserved Mask         */
#define FPB_FP_COMP3_COMP_Pos                                 2                                                         /*!< FPB FP_COMP3: COMP Position         */
#define FPB_FP_COMP3_COMP_Msk                                 (0x07ffffffUL << FPB_FP_COMP3_COMP_Pos)                   /*!< FPB FP_COMP3: COMP Mask             */
#define FPB_FP_COMP3_Reserved1_Pos                            29                                                        /*!< FPB FP_COMP3: Reserved1 Position    */
#define FPB_FP_COMP3_Reserved1_Msk                            (0x01UL << FPB_FP_COMP3_Reserved1_Pos)                    /*!< FPB FP_COMP3: Reserved1 Mask        */
#define FPB_FP_COMP3_REPLACE_Pos                              30                                                        /*!< FPB FP_COMP3: REPLACE Position      */
#define FPB_FP_COMP3_REPLACE_Msk                              (0x03UL << FPB_FP_COMP3_REPLACE_Pos)                      /*!< FPB FP_COMP3: REPLACE Mask          */

// --------------------------------------  FPB_FP_COMP4  ------------------------------------------
#define FPB_FP_COMP4_ENABLE_Pos                               0                                                         /*!< FPB FP_COMP4: ENABLE Position       */
#define FPB_FP_COMP4_ENABLE_Msk                               (0x01UL << FPB_FP_COMP4_ENABLE_Pos)                       /*!< FPB FP_COMP4: ENABLE Mask           */
#define FPB_FP_COMP4_Reserved_Pos                             1                                                         /*!< FPB FP_COMP4: Reserved Position     */
#define FPB_FP_COMP4_Reserved_Msk                             (0x01UL << FPB_FP_COMP4_Reserved_Pos)                     /*!< FPB FP_COMP4: Reserved Mask         */
#define FPB_FP_COMP4_COMP_Pos                                 2                                                         /*!< FPB FP_COMP4: COMP Position         */
#define FPB_FP_COMP4_COMP_Msk                                 (0x07ffffffUL << FPB_FP_COMP4_COMP_Pos)                   /*!< FPB FP_COMP4: COMP Mask             */
#define FPB_FP_COMP4_Reserved1_Pos                            29                                                        /*!< FPB FP_COMP4: Reserved1 Position    */
#define FPB_FP_COMP4_Reserved1_Msk                            (0x01UL << FPB_FP_COMP4_Reserved1_Pos)                    /*!< FPB FP_COMP4: Reserved1 Mask        */
#define FPB_FP_COMP4_REPLACE_Pos                              30                                                        /*!< FPB FP_COMP4: REPLACE Position      */
#define FPB_FP_COMP4_REPLACE_Msk                              (0x03UL << FPB_FP_COMP4_REPLACE_Pos)                      /*!< FPB FP_COMP4: REPLACE Mask          */

// --------------------------------------  FPB_FP_COMP5  ------------------------------------------
#define FPB_FP_COMP5_ENABLE_Pos                               0                                                         /*!< FPB FP_COMP5: ENABLE Position       */
#define FPB_FP_COMP5_ENABLE_Msk                               (0x01UL << FPB_FP_COMP5_ENABLE_Pos)                       /*!< FPB FP_COMP5: ENABLE Mask           */
#define FPB_FP_COMP5_Reserved_Pos                             1                                                         /*!< FPB FP_COMP5: Reserved Position     */
#define FPB_FP_COMP5_Reserved_Msk                             (0x01UL << FPB_FP_COMP5_Reserved_Pos)                     /*!< FPB FP_COMP5: Reserved Mask         */
#define FPB_FP_COMP5_COMP_Pos                                 2                                                         /*!< FPB FP_COMP5: COMP Position         */
#define FPB_FP_COMP5_COMP_Msk                                 (0x07ffffffUL << FPB_FP_COMP5_COMP_Pos)                   /*!< FPB FP_COMP5: COMP Mask             */
#define FPB_FP_COMP5_Reserved1_Pos                            29                                                        /*!< FPB FP_COMP5: Reserved1 Position    */
#define FPB_FP_COMP5_Reserved1_Msk                            (0x01UL << FPB_FP_COMP5_Reserved1_Pos)                    /*!< FPB FP_COMP5: Reserved1 Mask        */
#define FPB_FP_COMP5_REPLACE_Pos                              30                                                        /*!< FPB FP_COMP5: REPLACE Position      */
#define FPB_FP_COMP5_REPLACE_Msk                              (0x03UL << FPB_FP_COMP5_REPLACE_Pos)                      /*!< FPB FP_COMP5: REPLACE Mask          */

// --------------------------------------  FPB_FP_COMP6  ------------------------------------------
#define FPB_FP_COMP6_ENABLE_Pos                               0                                                         /*!< FPB FP_COMP6: ENABLE Position       */
#define FPB_FP_COMP6_ENABLE_Msk                               (0x01UL << FPB_FP_COMP6_ENABLE_Pos)                       /*!< FPB FP_COMP6: ENABLE Mask           */
#define FPB_FP_COMP6_Reserved_Pos                             1                                                         /*!< FPB FP_COMP6: Reserved Position     */
#define FPB_FP_COMP6_Reserved_Msk                             (0x01UL << FPB_FP_COMP6_Reserved_Pos)                     /*!< FPB FP_COMP6: Reserved Mask         */
#define FPB_FP_COMP6_COMP_Pos                                 2                                                         /*!< FPB FP_COMP6: COMP Position         */
#define FPB_FP_COMP6_COMP_Msk                                 (0x07ffffffUL << FPB_FP_COMP6_COMP_Pos)                   /*!< FPB FP_COMP6: COMP Mask             */
#define FPB_FP_COMP6_Reserved1_Pos                            29                                                        /*!< FPB FP_COMP6: Reserved1 Position    */
#define FPB_FP_COMP6_Reserved1_Msk                            (0x01UL << FPB_FP_COMP6_Reserved1_Pos)                    /*!< FPB FP_COMP6: Reserved1 Mask        */
#define FPB_FP_COMP6_REPLACE_Pos                              30                                                        /*!< FPB FP_COMP6: REPLACE Position      */
#define FPB_FP_COMP6_REPLACE_Msk                              (0x03UL << FPB_FP_COMP6_REPLACE_Pos)                      /*!< FPB FP_COMP6: REPLACE Mask          */

// --------------------------------------  FPB_FP_COMP7  ------------------------------------------
#define FPB_FP_COMP7_ENABLE_Pos                               0                                                         /*!< FPB FP_COMP7: ENABLE Position       */
#define FPB_FP_COMP7_ENABLE_Msk                               (0x01UL << FPB_FP_COMP7_ENABLE_Pos)                       /*!< FPB FP_COMP7: ENABLE Mask           */
#define FPB_FP_COMP7_Reserved_Pos                             1                                                         /*!< FPB FP_COMP7: Reserved Position     */
#define FPB_FP_COMP7_Reserved_Msk                             (0x01UL << FPB_FP_COMP7_Reserved_Pos)                     /*!< FPB FP_COMP7: Reserved Mask         */
#define FPB_FP_COMP7_COMP_Pos                                 2                                                         /*!< FPB FP_COMP7: COMP Position         */
#define FPB_FP_COMP7_COMP_Msk                                 (0x07ffffffUL << FPB_FP_COMP7_COMP_Pos)                   /*!< FPB FP_COMP7: COMP Mask             */
#define FPB_FP_COMP7_Reserved1_Pos                            29                                                        /*!< FPB FP_COMP7: Reserved1 Position    */
#define FPB_FP_COMP7_Reserved1_Msk                            (0x01UL << FPB_FP_COMP7_Reserved1_Pos)                    /*!< FPB FP_COMP7: Reserved1 Mask        */
#define FPB_FP_COMP7_REPLACE_Pos                              30                                                        /*!< FPB FP_COMP7: REPLACE Position      */
#define FPB_FP_COMP7_REPLACE_Msk                              (0x03UL << FPB_FP_COMP7_REPLACE_Pos)                      /*!< FPB FP_COMP7: REPLACE Mask          */

// ---------------------------------------  FPB_FPBPID4  ------------------------------------------
#define FPB_FPBPID4_PID4_Pos                                  0                                                         /*!< FPB FPBPID4: PID4 Position          */
#define FPB_FPBPID4_PID4_Msk                                  (0xffffffffUL << FPB_FPBPID4_PID4_Pos)                    /*!< FPB FPBPID4: PID4 Mask              */

// ---------------------------------------  FPB_FPBPID5  ------------------------------------------
#define FPB_FPBPID5_PID5_Pos                                  0                                                         /*!< FPB FPBPID5: PID5 Position          */
#define FPB_FPBPID5_PID5_Msk                                  (0xffffffffUL << FPB_FPBPID5_PID5_Pos)                    /*!< FPB FPBPID5: PID5 Mask              */

// ---------------------------------------  FPB_FPBPID6  ------------------------------------------
#define FPB_FPBPID6_PID6_Pos                                  0                                                         /*!< FPB FPBPID6: PID6 Position          */
#define FPB_FPBPID6_PID6_Msk                                  (0xffffffffUL << FPB_FPBPID6_PID6_Pos)                    /*!< FPB FPBPID6: PID6 Mask              */

// ---------------------------------------  FPB_FPBPID7  ------------------------------------------
#define FPB_FPBPID7_PID7_Pos                                  0                                                         /*!< FPB FPBPID7: PID7 Position          */
#define FPB_FPBPID7_PID7_Msk                                  (0xffffffffUL << FPB_FPBPID7_PID7_Pos)                    /*!< FPB FPBPID7: PID7 Mask              */

// ---------------------------------------  FPB_FPBPID0  ------------------------------------------
#define FPB_FPBPID0_PID0_Pos                                  0                                                         /*!< FPB FPBPID0: PID0 Position          */
#define FPB_FPBPID0_PID0_Msk                                  (0xffffffffUL << FPB_FPBPID0_PID0_Pos)                    /*!< FPB FPBPID0: PID0 Mask              */

// ---------------------------------------  FPB_FPBPID1  ------------------------------------------
#define FPB_FPBPID1_PID1_Pos                                  0                                                         /*!< FPB FPBPID1: PID1 Position          */
#define FPB_FPBPID1_PID1_Msk                                  (0xffffffffUL << FPB_FPBPID1_PID1_Pos)                    /*!< FPB FPBPID1: PID1 Mask              */

// ---------------------------------------  FPB_FPBPID2  ------------------------------------------
#define FPB_FPBPID2_PID2_Pos                                  0                                                         /*!< FPB FPBPID2: PID2 Position          */
#define FPB_FPBPID2_PID2_Msk                                  (0xffffffffUL << FPB_FPBPID2_PID2_Pos)                    /*!< FPB FPBPID2: PID2 Mask              */

// ---------------------------------------  FPB_FPBPID3  ------------------------------------------
#define FPB_FPBPID3_PID3_Pos                                  0                                                         /*!< FPB FPBPID3: PID3 Position          */
#define FPB_FPBPID3_PID3_Msk                                  (0xffffffffUL << FPB_FPBPID3_PID3_Pos)                    /*!< FPB FPBPID3: PID3 Mask              */

// ---------------------------------------  FPB_FPBCID0  ------------------------------------------
#define FPB_FPBCID0_CID0_Pos                                  0                                                         /*!< FPB FPBCID0: CID0 Position          */
#define FPB_FPBCID0_CID0_Msk                                  (0xffffffffUL << FPB_FPBCID0_CID0_Pos)                    /*!< FPB FPBCID0: CID0 Mask              */

// ---------------------------------------  FPB_FPBCID1  ------------------------------------------
#define FPB_FPBCID1_CID1_Pos                                  0                                                         /*!< FPB FPBCID1: CID1 Position          */
#define FPB_FPBCID1_CID1_Msk                                  (0xffffffffUL << FPB_FPBCID1_CID1_Pos)                    /*!< FPB FPBCID1: CID1 Mask              */

// ---------------------------------------  FPB_FPBCID2  ------------------------------------------
#define FPB_FPBCID2_CID2_Pos                                  0                                                         /*!< FPB FPBCID2: CID2 Position          */
#define FPB_FPBCID2_CID2_Msk                                  (0xffffffffUL << FPB_FPBCID2_CID2_Pos)                    /*!< FPB FPBCID2: CID2 Mask              */

// ---------------------------------------  FPB_FPBCID3  ------------------------------------------
#define FPB_FPBCID3_CID3_Pos                                  0                                                         /*!< FPB FPBCID3: CID3 Position          */
#define FPB_FPBCID3_CID3_Msk                                  (0xffffffffUL << FPB_FPBCID3_CID3_Pos)                    /*!< FPB FPBCID3: CID3 Mask              */



/***   LEDTS0 Bit Fileds *******************/
/***************************************************************************/


/* LEDTS_ID  =  Module Identification Register*/
#define   LEDTS_ID_MOD_REV_Pos  	(0U)
#define   LEDTS_ID_MOD_REV_Msk  	(0x000000FFU  << LEDTS_ID_MOD_REV_Pos)

#define   LEDTS_ID_MOD_TYPE_Pos  	(8U)
#define   LEDTS_ID_MOD_TYPE_Msk  	(0x000000FFU  << LEDTS_ID_MOD_TYPE_Pos)

#define   LEDTS_ID_MOD_NUMBER_Pos  	(16U)
#define   LEDTS_ID_MOD_NUMBER_Msk  	(0x0000FFFFU  << LEDTS_ID_MOD_NUMBER_Pos)

/* LEDTS_GLOBCTL  =  Global Control Register*/
#define   LEDTS_GLOBCTL_TS_EN_Pos  	(0U)
#define   LEDTS_GLOBCTL_TS_EN_Msk  	(0x00000001U  << LEDTS_GLOBCTL_TS_EN_Pos)

#define   LEDTS_GLOBCTL_LD_EN_Pos  	(1U)
#define   LEDTS_GLOBCTL_LD_EN_Msk  	(0x00000001U  << LEDTS_GLOBCTL_LD_EN_Pos)

#define   LEDTS_GLOBCTL_CMTR_Pos  	(2U)
#define   LEDTS_GLOBCTL_CMTR_Msk  	(0x00000001U  << LEDTS_GLOBCTL_CMTR_Pos)

#define   LEDTS_GLOBCTL_ENSYNC_Pos  	(3U)
#define   LEDTS_GLOBCTL_ENSYNC_Msk  	(0x00000001U  << LEDTS_GLOBCTL_ENSYNC_Pos)

#define   LEDTS_GLOBCTL_SUSCFG_Pos  	(8U)
#define   LEDTS_GLOBCTL_SUSCFG_Msk  	(0x00000001U  << LEDTS_GLOBCTL_SUSCFG_Pos)

#define   LEDTS_GLOBCTL_MASKVAL_Pos  	(9U)
#define   LEDTS_GLOBCTL_MASKVAL_Msk  	(0x00000007U  << LEDTS_GLOBCTL_MASKVAL_Pos)

#define   LEDTS_GLOBCTL_FENVAL_Pos  	(12U)
#define   LEDTS_GLOBCTL_FENVAL_Msk  	(0x00000001U  << LEDTS_GLOBCTL_FENVAL_Pos)

#define   LEDTS_GLOBCTL_ITS_EN_Pos  	(13U)
#define   LEDTS_GLOBCTL_ITS_EN_Msk  	(0x00000001U  << LEDTS_GLOBCTL_ITS_EN_Pos)

#define   LEDTS_GLOBCTL_ITF_EN_Pos  	(14U)
#define   LEDTS_GLOBCTL_ITF_EN_Msk  	(0x00000001U  << LEDTS_GLOBCTL_ITF_EN_Pos)

#define   LEDTS_GLOBCTL_ITP_EN_Pos  	(15U)
#define   LEDTS_GLOBCTL_ITP_EN_Msk  	(0x00000001U  << LEDTS_GLOBCTL_ITP_EN_Pos)

#define   LEDTS_GLOBCTL_CLK_PS_Pos  	(16U)
#define   LEDTS_GLOBCTL_CLK_PS_Msk  	(0x0000FFFFU  << LEDTS_GLOBCTL_CLK_PS_Pos)

/* LEDTS_FNCTL  =  Function Control Register*/
#define   LEDTS_FNCTL_PADT_Pos  	(0U)
#define   LEDTS_FNCTL_PADT_Msk  	(0x00000007U  << LEDTS_FNCTL_PADT_Pos)

#define   LEDTS_FNCTL_PADTSW_Pos  	(3U)
#define   LEDTS_FNCTL_PADTSW_Msk  	(0x00000001U  << LEDTS_FNCTL_PADTSW_Pos)

#define   LEDTS_FNCTL_EPULL_Pos  	(4U)
#define   LEDTS_FNCTL_EPULL_Msk  	(0x00000001U  << LEDTS_FNCTL_EPULL_Pos)

#define   LEDTS_FNCTL_FNCOL_Pos  	(5U)
#define   LEDTS_FNCTL_FNCOL_Msk  	(0x00000007U  << LEDTS_FNCTL_FNCOL_Pos)

#define   LEDTS_FNCTL_ACCCNT_Pos  	(16U)
#define   LEDTS_FNCTL_ACCCNT_Msk  	(0x0000000FU  << LEDTS_FNCTL_ACCCNT_Pos)

#define   LEDTS_FNCTL_TSCCMP_Pos  	(20U)
#define   LEDTS_FNCTL_TSCCMP_Msk  	(0x00000001U  << LEDTS_FNCTL_TSCCMP_Pos)

#define   LEDTS_FNCTL_TSOEXT_Pos  	(21U)
#define   LEDTS_FNCTL_TSOEXT_Msk  	(0x00000003U  << LEDTS_FNCTL_TSOEXT_Pos)

#define   LEDTS_FNCTL_TSCTRR_Pos  	(23U)
#define   LEDTS_FNCTL_TSCTRR_Msk  	(0x00000001U  << LEDTS_FNCTL_TSCTRR_Pos)

#define   LEDTS_FNCTL_TSCTRSAT_Pos  	(24U)
#define   LEDTS_FNCTL_TSCTRSAT_Msk  	(0x00000001U  << LEDTS_FNCTL_TSCTRSAT_Pos)

#define   LEDTS_FNCTL_NR_TSIN_Pos  	(25U)
#define   LEDTS_FNCTL_NR_TSIN_Msk  	(0x00000007U  << LEDTS_FNCTL_NR_TSIN_Pos)

#define   LEDTS_FNCTL_COLLEV_Pos  	(28U)
#define   LEDTS_FNCTL_COLLEV_Msk  	(0x00000001U  << LEDTS_FNCTL_COLLEV_Pos)

#define   LEDTS_FNCTL_NR_LEDCOL_Pos  	(29U)
#define   LEDTS_FNCTL_NR_LEDCOL_Msk  	(0x00000007U  << LEDTS_FNCTL_NR_LEDCOL_Pos)

/* LEDTS_EVFR  =  Event Flag Register*/
#define   LEDTS_EVFR_TSF_Pos  	(0U)
#define   LEDTS_EVFR_TSF_Msk  	(0x00000001U  << LEDTS_EVFR_TSF_Pos)

#define   LEDTS_EVFR_TFF_Pos  	(1U)
#define   LEDTS_EVFR_TFF_Msk  	(0x00000001U  << LEDTS_EVFR_TFF_Pos)

#define   LEDTS_EVFR_TPF_Pos  	(2U)
#define   LEDTS_EVFR_TPF_Msk  	(0x00000001U  << LEDTS_EVFR_TPF_Pos)

#define   LEDTS_EVFR_TSCTROVF_Pos  	(3U)
#define   LEDTS_EVFR_TSCTROVF_Msk  	(0x00000001U  << LEDTS_EVFR_TSCTROVF_Pos)

#define   LEDTS_EVFR_CTSF_Pos  	(16U)
#define   LEDTS_EVFR_CTSF_Msk  	(0x00000001U  << LEDTS_EVFR_CTSF_Pos)

#define   LEDTS_EVFR_CTFF_Pos  	(17U)
#define   LEDTS_EVFR_CTFF_Msk  	(0x00000001U  << LEDTS_EVFR_CTFF_Pos)

#define   LEDTS_EVFR_CTPF_Pos  	(18U)
#define   LEDTS_EVFR_CTPF_Msk  	(0x00000001U  << LEDTS_EVFR_CTPF_Pos)

/* LEDTS_TSVAL  =  Touch-sense TS-Counter Value*/
#define   LEDTS_TSVAL_TSCTRVALR_Pos  	(0U)
#define   LEDTS_TSVAL_TSCTRVALR_Msk  	(0x0000FFFFU  << LEDTS_TSVAL_TSCTRVALR_Pos)

#define   LEDTS_TSVAL_TSCTRVAL_Pos  	(16U)
#define   LEDTS_TSVAL_TSCTRVAL_Msk  	(0x0000FFFFU  << LEDTS_TSVAL_TSCTRVAL_Pos)

/* LEDTS_LINE0  =  Line Pattern Register 0*/
#define   LEDTS_LINE0_LINE_0_Pos  	(0U)
#define   LEDTS_LINE0_LINE_0_Msk  	(0x000000FFU  << LEDTS_LINE0_LINE_0_Pos)

#define   LEDTS_LINE0_LINE_1_Pos  	(8U)
#define   LEDTS_LINE0_LINE_1_Msk  	(0x000000FFU  << LEDTS_LINE0_LINE_1_Pos)

#define   LEDTS_LINE0_LINE_2_Pos  	(16U)
#define   LEDTS_LINE0_LINE_2_Msk  	(0x000000FFU  << LEDTS_LINE0_LINE_2_Pos)

#define   LEDTS_LINE0_LINE_3_Pos  	(24U)
#define   LEDTS_LINE0_LINE_3_Msk  	(0x000000FFU  << LEDTS_LINE0_LINE_3_Pos)

/* LEDTS_LINE1  =  Line Pattern Register 1*/
#define   LEDTS_LINE1_LINE_4_Pos  	(0U)
#define   LEDTS_LINE1_LINE_4_Msk  	(0x000000FFU  << LEDTS_LINE1_LINE_4_Pos)

#define   LEDTS_LINE1_LINE_5_Pos  	(8U)
#define   LEDTS_LINE1_LINE_5_Msk  	(0x000000FFU  << LEDTS_LINE1_LINE_5_Pos)

#define   LEDTS_LINE1_LINE_6_Pos  	(16U)
#define   LEDTS_LINE1_LINE_6_Msk  	(0x000000FFU  << LEDTS_LINE1_LINE_6_Pos)

#define   LEDTS_LINE1_LINE_A_Pos  	(24U)
#define   LEDTS_LINE1_LINE_A_Msk  	(0x000000FFU  << LEDTS_LINE1_LINE_A_Pos)

/* LEDTS_LDCMP0  =  LED Compare Register 0*/
#define   LEDTS_LDCMP0_CMP_LD0_Pos  	(0U)
#define   LEDTS_LDCMP0_CMP_LD0_Msk  	(0x000000FFU  << LEDTS_LDCMP0_CMP_LD0_Pos)

#define   LEDTS_LDCMP0_CMP_LD1_Pos  	(8U)
#define   LEDTS_LDCMP0_CMP_LD1_Msk  	(0x000000FFU  << LEDTS_LDCMP0_CMP_LD1_Pos)

#define   LEDTS_LDCMP0_CMP_LD2_Pos  	(16U)
#define   LEDTS_LDCMP0_CMP_LD2_Msk  	(0x000000FFU  << LEDTS_LDCMP0_CMP_LD2_Pos)

#define   LEDTS_LDCMP0_CMP_LD3_Pos  	(24U)
#define   LEDTS_LDCMP0_CMP_LD3_Msk  	(0x000000FFU  << LEDTS_LDCMP0_CMP_LD3_Pos)

/* LEDTS_LDCMP1  =  LED Compare Register 1*/
#define   LEDTS_LDCMP1_CMP_LD4_Pos  	(0U)
#define   LEDTS_LDCMP1_CMP_LD4_Msk  	(0x000000FFU  << LEDTS_LDCMP1_CMP_LD4_Pos)

#define   LEDTS_LDCMP1_CMP_LD5_Pos  	(8U)
#define   LEDTS_LDCMP1_CMP_LD5_Msk  	(0x000000FFU  << LEDTS_LDCMP1_CMP_LD5_Pos)

#define   LEDTS_LDCMP1_CMP_LD6_Pos  	(16U)
#define   LEDTS_LDCMP1_CMP_LD6_Msk  	(0x000000FFU  << LEDTS_LDCMP1_CMP_LD6_Pos)

#define   LEDTS_LDCMP1_CMP_LDATSCOM_Pos  	(24U)
#define   LEDTS_LDCMP1_CMP_LDATSCOM_Msk  	(0x000000FFU  << LEDTS_LDCMP1_CMP_LDATSCOM_Pos)

/* LEDTS_TSCMP0  =  Touch-sense Compare Register 0*/
#define   LEDTS_TSCMP0_CMP_TS0_Pos  	(0U)
#define   LEDTS_TSCMP0_CMP_TS0_Msk  	(0x000000FFU  << LEDTS_TSCMP0_CMP_TS0_Pos)

#define   LEDTS_TSCMP0_CMP_TS1_Pos  	(8U)
#define   LEDTS_TSCMP0_CMP_TS1_Msk  	(0x000000FFU  << LEDTS_TSCMP0_CMP_TS1_Pos)

#define   LEDTS_TSCMP0_CMP_TS2_Pos  	(16U)
#define   LEDTS_TSCMP0_CMP_TS2_Msk  	(0x000000FFU  << LEDTS_TSCMP0_CMP_TS2_Pos)

#define   LEDTS_TSCMP0_CMP_TS3_Pos  	(24U)
#define   LEDTS_TSCMP0_CMP_TS3_Msk  	(0x000000FFU  << LEDTS_TSCMP0_CMP_TS3_Pos)

/* LEDTS_TSCMP1  =  Touch-sense Compare Register 1*/
#define   LEDTS_TSCMP1_CMP_TS4_Pos  	(0U)
#define   LEDTS_TSCMP1_CMP_TS4_Msk  	(0x000000FFU  << LEDTS_TSCMP1_CMP_TS4_Pos)

#define   LEDTS_TSCMP1_CMP_TS5_Pos  	(8U)
#define   LEDTS_TSCMP1_CMP_TS5_Msk  	(0x000000FFU  << LEDTS_TSCMP1_CMP_TS5_Pos)

#define   LEDTS_TSCMP1_CMP_TS6_Pos  	(16U)
#define   LEDTS_TSCMP1_CMP_TS6_Msk  	(0x000000FFU  << LEDTS_TSCMP1_CMP_TS6_Pos)

#define   LEDTS_TSCMP1_CMP_TS7_Pos  	(24U)
#define   LEDTS_TSCMP1_CMP_TS7_Msk  	(0x000000FFU  << LEDTS_TSCMP1_CMP_TS7_Pos)



/***   PBAx Bit Fileds *******************/
/***************************************************************************/


/* PBA_STS  =  Peripheral Bridge Status Register*/
#define   PBA_STS_WERR_Pos  	(0U)
#define   PBA_STS_WERR_Msk  	(0x00000001U  << PBA_STS_WERR_Pos)

/* PBA_WADDR  =  PBA Write Error Address Register*/
#define   PBA_WADDR_WADDR_Pos  	(0U)
#define   PBA_WADDR_WADDR_Msk  	(0xFFFFFFFFU  << PBA_WADDR_WADDR_Pos)



/***   PMU0 Bit Fileds *******************/
/***************************************************************************/


/* PMU_ID  =  PMU0 Identification Register*/
#define   PMU_ID_MOD_REV_Pos  	(0U)
#define   PMU_ID_MOD_REV_Msk  	(0x000000FFU  << PMU_ID_MOD_REV_Pos)

#define   PMU_ID_MOD_TYPE_Pos  	(8U)
#define   PMU_ID_MOD_TYPE_Msk  	(0x000000FFU  << PMU_ID_MOD_TYPE_Pos)

#define   PMU_ID_MOD_NUMBER_Pos  	(16U)
#define   PMU_ID_MOD_NUMBER_Msk  	(0x0000FFFFU  << PMU_ID_MOD_NUMBER_Pos)



// ------------------------------------------------------------------------------------------------
// -----                                 PORT0 Position & Mask                                -----
// ------------------------------------------------------------------------------------------------


// ----------------------------------------  PORT0_OUT  -------------------------------------------
#define PORT0_OUT_P0_Pos                                      0                                                         /*!< PORT0 OUT: P0 Position              */
#define PORT0_OUT_P0_Msk                                      (0x01UL << PORT0_OUT_P0_Pos)                              /*!< PORT0 OUT: P0 Mask                  */
#define PORT0_OUT_P1_Pos                                      1                                                         /*!< PORT0 OUT: P1 Position              */
#define PORT0_OUT_P1_Msk                                      (0x01UL << PORT0_OUT_P1_Pos)                              /*!< PORT0 OUT: P1 Mask                  */
#define PORT0_OUT_P2_Pos                                      2                                                         /*!< PORT0 OUT: P2 Position              */
#define PORT0_OUT_P2_Msk                                      (0x01UL << PORT0_OUT_P2_Pos)                              /*!< PORT0 OUT: P2 Mask                  */
#define PORT0_OUT_P3_Pos                                      3                                                         /*!< PORT0 OUT: P3 Position              */
#define PORT0_OUT_P3_Msk                                      (0x01UL << PORT0_OUT_P3_Pos)                              /*!< PORT0 OUT: P3 Mask                  */
#define PORT0_OUT_P4_Pos                                      4                                                         /*!< PORT0 OUT: P4 Position              */
#define PORT0_OUT_P4_Msk                                      (0x01UL << PORT0_OUT_P4_Pos)                              /*!< PORT0 OUT: P4 Mask                  */
#define PORT0_OUT_P5_Pos                                      5                                                         /*!< PORT0 OUT: P5 Position              */
#define PORT0_OUT_P5_Msk                                      (0x01UL << PORT0_OUT_P5_Pos)                              /*!< PORT0 OUT: P5 Mask                  */
#define PORT0_OUT_P6_Pos                                      6                                                         /*!< PORT0 OUT: P6 Position              */
#define PORT0_OUT_P6_Msk                                      (0x01UL << PORT0_OUT_P6_Pos)                              /*!< PORT0 OUT: P6 Mask                  */
#define PORT0_OUT_P7_Pos                                      7                                                         /*!< PORT0 OUT: P7 Position              */
#define PORT0_OUT_P7_Msk                                      (0x01UL << PORT0_OUT_P7_Pos)                              /*!< PORT0 OUT: P7 Mask                  */
#define PORT0_OUT_P8_Pos                                      8                                                         /*!< PORT0 OUT: P8 Position              */
#define PORT0_OUT_P8_Msk                                      (0x01UL << PORT0_OUT_P8_Pos)                              /*!< PORT0 OUT: P8 Mask                  */
#define PORT0_OUT_P9_Pos                                      9                                                         /*!< PORT0 OUT: P9 Position              */
#define PORT0_OUT_P9_Msk                                      (0x01UL << PORT0_OUT_P9_Pos)                              /*!< PORT0 OUT: P9 Mask                  */
#define PORT0_OUT_P10_Pos                                     10                                                        /*!< PORT0 OUT: P10 Position             */
#define PORT0_OUT_P10_Msk                                     (0x01UL << PORT0_OUT_P10_Pos)                             /*!< PORT0 OUT: P10 Mask                 */
#define PORT0_OUT_P11_Pos                                     11                                                        /*!< PORT0 OUT: P11 Position             */
#define PORT0_OUT_P11_Msk                                     (0x01UL << PORT0_OUT_P11_Pos)                             /*!< PORT0 OUT: P11 Mask                 */
#define PORT0_OUT_P12_Pos                                     12                                                        /*!< PORT0 OUT: P12 Position             */
#define PORT0_OUT_P12_Msk                                     (0x01UL << PORT0_OUT_P12_Pos)                             /*!< PORT0 OUT: P12 Mask                 */
#define PORT0_OUT_P13_Pos                                     13                                                        /*!< PORT0 OUT: P13 Position             */
#define PORT0_OUT_P13_Msk                                     (0x01UL << PORT0_OUT_P13_Pos)                             /*!< PORT0 OUT: P13 Mask                 */
#define PORT0_OUT_P14_Pos                                     14                                                        /*!< PORT0 OUT: P14 Position             */
#define PORT0_OUT_P14_Msk                                     (0x01UL << PORT0_OUT_P14_Pos)                             /*!< PORT0 OUT: P14 Mask                 */
#define PORT0_OUT_P15_Pos                                     15                                                        /*!< PORT0 OUT: P15 Position             */
#define PORT0_OUT_P15_Msk                                     (0x01UL << PORT0_OUT_P15_Pos)                             /*!< PORT0 OUT: P15 Mask                 */

// ----------------------------------------  PORT0_OMR  -------------------------------------------
#define PORT0_OMR_PS0_Pos                                     0                                                         /*!< PORT0 OMR: PS0 Position             */
#define PORT0_OMR_PS0_Msk                                     (0x01UL << PORT0_OMR_PS0_Pos)                             /*!< PORT0 OMR: PS0 Mask                 */
#define PORT0_OMR_PS1_Pos                                     1                                                         /*!< PORT0 OMR: PS1 Position             */
#define PORT0_OMR_PS1_Msk                                     (0x01UL << PORT0_OMR_PS1_Pos)                             /*!< PORT0 OMR: PS1 Mask                 */
#define PORT0_OMR_PS2_Pos                                     2                                                         /*!< PORT0 OMR: PS2 Position             */
#define PORT0_OMR_PS2_Msk                                     (0x01UL << PORT0_OMR_PS2_Pos)                             /*!< PORT0 OMR: PS2 Mask                 */
#define PORT0_OMR_PS3_Pos                                     3                                                         /*!< PORT0 OMR: PS3 Position             */
#define PORT0_OMR_PS3_Msk                                     (0x01UL << PORT0_OMR_PS3_Pos)                             /*!< PORT0 OMR: PS3 Mask                 */
#define PORT0_OMR_PS4_Pos                                     4                                                         /*!< PORT0 OMR: PS4 Position             */
#define PORT0_OMR_PS4_Msk                                     (0x01UL << PORT0_OMR_PS4_Pos)                             /*!< PORT0 OMR: PS4 Mask                 */
#define PORT0_OMR_PS5_Pos                                     5                                                         /*!< PORT0 OMR: PS5 Position             */
#define PORT0_OMR_PS5_Msk                                     (0x01UL << PORT0_OMR_PS5_Pos)                             /*!< PORT0 OMR: PS5 Mask                 */
#define PORT0_OMR_PS6_Pos                                     6                                                         /*!< PORT0 OMR: PS6 Position             */
#define PORT0_OMR_PS6_Msk                                     (0x01UL << PORT0_OMR_PS6_Pos)                             /*!< PORT0 OMR: PS6 Mask                 */
#define PORT0_OMR_PS7_Pos                                     7                                                         /*!< PORT0 OMR: PS7 Position             */
#define PORT0_OMR_PS7_Msk                                     (0x01UL << PORT0_OMR_PS7_Pos)                             /*!< PORT0 OMR: PS7 Mask                 */
#define PORT0_OMR_PS8_Pos                                     8                                                         /*!< PORT0 OMR: PS8 Position             */
#define PORT0_OMR_PS8_Msk                                     (0x01UL << PORT0_OMR_PS8_Pos)                             /*!< PORT0 OMR: PS8 Mask                 */
#define PORT0_OMR_PS9_Pos                                     9                                                         /*!< PORT0 OMR: PS9 Position             */
#define PORT0_OMR_PS9_Msk                                     (0x01UL << PORT0_OMR_PS9_Pos)                             /*!< PORT0 OMR: PS9 Mask                 */
#define PORT0_OMR_PS10_Pos                                    10                                                        /*!< PORT0 OMR: PS10 Position            */
#define PORT0_OMR_PS10_Msk                                    (0x01UL << PORT0_OMR_PS10_Pos)                            /*!< PORT0 OMR: PS10 Mask                */
#define PORT0_OMR_PS11_Pos                                    11                                                        /*!< PORT0 OMR: PS11 Position            */
#define PORT0_OMR_PS11_Msk                                    (0x01UL << PORT0_OMR_PS11_Pos)                            /*!< PORT0 OMR: PS11 Mask                */
#define PORT0_OMR_PS12_Pos                                    12                                                        /*!< PORT0 OMR: PS12 Position            */
#define PORT0_OMR_PS12_Msk                                    (0x01UL << PORT0_OMR_PS12_Pos)                            /*!< PORT0 OMR: PS12 Mask                */
#define PORT0_OMR_PS13_Pos                                    13                                                        /*!< PORT0 OMR: PS13 Position            */
#define PORT0_OMR_PS13_Msk                                    (0x01UL << PORT0_OMR_PS13_Pos)                            /*!< PORT0 OMR: PS13 Mask                */
#define PORT0_OMR_PS14_Pos                                    14                                                        /*!< PORT0 OMR: PS14 Position            */
#define PORT0_OMR_PS14_Msk                                    (0x01UL << PORT0_OMR_PS14_Pos)                            /*!< PORT0 OMR: PS14 Mask                */
#define PORT0_OMR_PS15_Pos                                    15                                                        /*!< PORT0 OMR: PS15 Position            */
#define PORT0_OMR_PS15_Msk                                    (0x01UL << PORT0_OMR_PS15_Pos)                            /*!< PORT0 OMR: PS15 Mask                */
#define PORT0_OMR_PR0_Pos                                     16                                                        /*!< PORT0 OMR: PR0 Position             */
#define PORT0_OMR_PR0_Msk                                     (0x01UL << PORT0_OMR_PR0_Pos)                             /*!< PORT0 OMR: PR0 Mask                 */
#define PORT0_OMR_PR1_Pos                                     17                                                        /*!< PORT0 OMR: PR1 Position             */
#define PORT0_OMR_PR1_Msk                                     (0x01UL << PORT0_OMR_PR1_Pos)                             /*!< PORT0 OMR: PR1 Mask                 */
#define PORT0_OMR_PR2_Pos                                     18                                                        /*!< PORT0 OMR: PR2 Position             */
#define PORT0_OMR_PR2_Msk                                     (0x01UL << PORT0_OMR_PR2_Pos)                             /*!< PORT0 OMR: PR2 Mask                 */
#define PORT0_OMR_PR3_Pos                                     19                                                        /*!< PORT0 OMR: PR3 Position             */
#define PORT0_OMR_PR3_Msk                                     (0x01UL << PORT0_OMR_PR3_Pos)                             /*!< PORT0 OMR: PR3 Mask                 */
#define PORT0_OMR_PR4_Pos                                     20                                                        /*!< PORT0 OMR: PR4 Position             */
#define PORT0_OMR_PR4_Msk                                     (0x01UL << PORT0_OMR_PR4_Pos)                             /*!< PORT0 OMR: PR4 Mask                 */
#define PORT0_OMR_PR5_Pos                                     21                                                        /*!< PORT0 OMR: PR5 Position             */
#define PORT0_OMR_PR5_Msk                                     (0x01UL << PORT0_OMR_PR5_Pos)                             /*!< PORT0 OMR: PR5 Mask                 */
#define PORT0_OMR_PR6_Pos                                     22                                                        /*!< PORT0 OMR: PR6 Position             */
#define PORT0_OMR_PR6_Msk                                     (0x01UL << PORT0_OMR_PR6_Pos)                             /*!< PORT0 OMR: PR6 Mask                 */
#define PORT0_OMR_PR7_Pos                                     23                                                        /*!< PORT0 OMR: PR7 Position             */
#define PORT0_OMR_PR7_Msk                                     (0x01UL << PORT0_OMR_PR7_Pos)                             /*!< PORT0 OMR: PR7 Mask                 */
#define PORT0_OMR_PR8_Pos                                     24                                                        /*!< PORT0 OMR: PR8 Position             */
#define PORT0_OMR_PR8_Msk                                     (0x01UL << PORT0_OMR_PR8_Pos)                             /*!< PORT0 OMR: PR8 Mask                 */
#define PORT0_OMR_PR9_Pos                                     25                                                        /*!< PORT0 OMR: PR9 Position             */
#define PORT0_OMR_PR9_Msk                                     (0x01UL << PORT0_OMR_PR9_Pos)                             /*!< PORT0 OMR: PR9 Mask                 */
#define PORT0_OMR_PR10_Pos                                    26                                                        /*!< PORT0 OMR: PR10 Position            */
#define PORT0_OMR_PR10_Msk                                    (0x01UL << PORT0_OMR_PR10_Pos)                            /*!< PORT0 OMR: PR10 Mask                */
#define PORT0_OMR_PR11_Pos                                    27                                                        /*!< PORT0 OMR: PR11 Position            */
#define PORT0_OMR_PR11_Msk                                    (0x01UL << PORT0_OMR_PR11_Pos)                            /*!< PORT0 OMR: PR11 Mask                */
#define PORT0_OMR_PR12_Pos                                    28                                                        /*!< PORT0 OMR: PR12 Position            */
#define PORT0_OMR_PR12_Msk                                    (0x01UL << PORT0_OMR_PR12_Pos)                            /*!< PORT0 OMR: PR12 Mask                */
#define PORT0_OMR_PR13_Pos                                    29                                                        /*!< PORT0 OMR: PR13 Position            */
#define PORT0_OMR_PR13_Msk                                    (0x01UL << PORT0_OMR_PR13_Pos)                            /*!< PORT0 OMR: PR13 Mask                */
#define PORT0_OMR_PR14_Pos                                    30                                                        /*!< PORT0 OMR: PR14 Position            */
#define PORT0_OMR_PR14_Msk                                    (0x01UL << PORT0_OMR_PR14_Pos)                            /*!< PORT0 OMR: PR14 Mask                */
#define PORT0_OMR_PR15_Pos                                    31                                                        /*!< PORT0 OMR: PR15 Position            */
#define PORT0_OMR_PR15_Msk                                    (0x01UL << PORT0_OMR_PR15_Pos)                            /*!< PORT0 OMR: PR15 Mask                */

// ---------------------------------------  PORT0_IOCR0  ------------------------------------------
#define PORT0_IOCR0_PC0_Pos                                   3                                                         /*!< PORT0 IOCR0: PC0 Position           */
#define PORT0_IOCR0_PC0_Msk                                   (0x1fUL << PORT0_IOCR0_PC0_Pos)                           /*!< PORT0 IOCR0: PC0 Mask               */
#define PORT0_IOCR0_PC1_Pos                                   11                                                        /*!< PORT0 IOCR0: PC1 Position           */
#define PORT0_IOCR0_PC1_Msk                                   (0x1fUL << PORT0_IOCR0_PC1_Pos)                           /*!< PORT0 IOCR0: PC1 Mask               */
#define PORT0_IOCR0_PC2_Pos                                   19                                                        /*!< PORT0 IOCR0: PC2 Position           */
#define PORT0_IOCR0_PC2_Msk                                   (0x1fUL << PORT0_IOCR0_PC2_Pos)                           /*!< PORT0 IOCR0: PC2 Mask               */
#define PORT0_IOCR0_PC3_Pos                                   27                                                        /*!< PORT0 IOCR0: PC3 Position           */
#define PORT0_IOCR0_PC3_Msk                                   (0x1fUL << PORT0_IOCR0_PC3_Pos)                           /*!< PORT0 IOCR0: PC3 Mask               */

// ---------------------------------------  PORT0_IOCR4  ------------------------------------------
#define PORT0_IOCR4_PC4_Pos                                   3                                                         /*!< PORT0 IOCR4: PC4 Position           */
#define PORT0_IOCR4_PC4_Msk                                   (0x1fUL << PORT0_IOCR4_PC4_Pos)                           /*!< PORT0 IOCR4: PC4 Mask               */
#define PORT0_IOCR4_PC5_Pos                                   11                                                        /*!< PORT0 IOCR4: PC5 Position           */
#define PORT0_IOCR4_PC5_Msk                                   (0x1fUL << PORT0_IOCR4_PC5_Pos)                           /*!< PORT0 IOCR4: PC5 Mask               */
#define PORT0_IOCR4_PC6_Pos                                   19                                                        /*!< PORT0 IOCR4: PC6 Position           */
#define PORT0_IOCR4_PC6_Msk                                   (0x1fUL << PORT0_IOCR4_PC6_Pos)                           /*!< PORT0 IOCR4: PC6 Mask               */
#define PORT0_IOCR4_PC7_Pos                                   27                                                        /*!< PORT0 IOCR4: PC7 Position           */
#define PORT0_IOCR4_PC7_Msk                                   (0x1fUL << PORT0_IOCR4_PC7_Pos)                           /*!< PORT0 IOCR4: PC7 Mask               */

// ---------------------------------------  PORT0_IOCR8  ------------------------------------------
#define PORT0_IOCR8_PC8_Pos                                   3                                                         /*!< PORT0 IOCR8: PC8 Position           */
#define PORT0_IOCR8_PC8_Msk                                   (0x1fUL << PORT0_IOCR8_PC8_Pos)                           /*!< PORT0 IOCR8: PC8 Mask               */
#define PORT0_IOCR8_PC9_Pos                                   11                                                        /*!< PORT0 IOCR8: PC9 Position           */
#define PORT0_IOCR8_PC9_Msk                                   (0x1fUL << PORT0_IOCR8_PC9_Pos)                           /*!< PORT0 IOCR8: PC9 Mask               */
#define PORT0_IOCR8_PC10_Pos                                  19                                                        /*!< PORT0 IOCR8: PC10 Position          */
#define PORT0_IOCR8_PC10_Msk                                  (0x1fUL << PORT0_IOCR8_PC10_Pos)                          /*!< PORT0 IOCR8: PC10 Mask              */
#define PORT0_IOCR8_PC11_Pos                                  27                                                        /*!< PORT0 IOCR8: PC11 Position          */
#define PORT0_IOCR8_PC11_Msk                                  (0x1fUL << PORT0_IOCR8_PC11_Pos)                          /*!< PORT0 IOCR8: PC11 Mask              */

// --------------------------------------  PORT0_IOCR12  ------------------------------------------
#define PORT0_IOCR12_PC12_Pos                                 3                                                         /*!< PORT0 IOCR12: PC12 Position         */
#define PORT0_IOCR12_PC12_Msk                                 (0x1fUL << PORT0_IOCR12_PC12_Pos)                         /*!< PORT0 IOCR12: PC12 Mask             */
#define PORT0_IOCR12_PC13_Pos                                 11                                                        /*!< PORT0 IOCR12: PC13 Position         */
#define PORT0_IOCR12_PC13_Msk                                 (0x1fUL << PORT0_IOCR12_PC13_Pos)                         /*!< PORT0 IOCR12: PC13 Mask             */
#define PORT0_IOCR12_PC14_Pos                                 19                                                        /*!< PORT0 IOCR12: PC14 Position         */
#define PORT0_IOCR12_PC14_Msk                                 (0x1fUL << PORT0_IOCR12_PC14_Pos)                         /*!< PORT0 IOCR12: PC14 Mask             */
#define PORT0_IOCR12_PC15_Pos                                 27                                                        /*!< PORT0 IOCR12: PC15 Position         */
#define PORT0_IOCR12_PC15_Msk                                 (0x1fUL << PORT0_IOCR12_PC15_Pos)                         /*!< PORT0 IOCR12: PC15 Mask             */

// ----------------------------------------  PORT0_IN  --------------------------------------------
#define PORT0_IN_P0_Pos                                       0                                                         /*!< PORT0 IN: P0 Position               */
#define PORT0_IN_P0_Msk                                       (0x01UL << PORT0_IN_P0_Pos)                               /*!< PORT0 IN: P0 Mask                   */
#define PORT0_IN_P1_Pos                                       1                                                         /*!< PORT0 IN: P1 Position               */
#define PORT0_IN_P1_Msk                                       (0x01UL << PORT0_IN_P1_Pos)                               /*!< PORT0 IN: P1 Mask                   */
#define PORT0_IN_P2_Pos                                       2                                                         /*!< PORT0 IN: P2 Position               */
#define PORT0_IN_P2_Msk                                       (0x01UL << PORT0_IN_P2_Pos)                               /*!< PORT0 IN: P2 Mask                   */
#define PORT0_IN_P3_Pos                                       3                                                         /*!< PORT0 IN: P3 Position               */
#define PORT0_IN_P3_Msk                                       (0x01UL << PORT0_IN_P3_Pos)                               /*!< PORT0 IN: P3 Mask                   */
#define PORT0_IN_P4_Pos                                       4                                                         /*!< PORT0 IN: P4 Position               */
#define PORT0_IN_P4_Msk                                       (0x01UL << PORT0_IN_P4_Pos)                               /*!< PORT0 IN: P4 Mask                   */
#define PORT0_IN_P5_Pos                                       5                                                         /*!< PORT0 IN: P5 Position               */
#define PORT0_IN_P5_Msk                                       (0x01UL << PORT0_IN_P5_Pos)                               /*!< PORT0 IN: P5 Mask                   */
#define PORT0_IN_P6_Pos                                       6                                                         /*!< PORT0 IN: P6 Position               */
#define PORT0_IN_P6_Msk                                       (0x01UL << PORT0_IN_P6_Pos)                               /*!< PORT0 IN: P6 Mask                   */
#define PORT0_IN_P7_Pos                                       7                                                         /*!< PORT0 IN: P7 Position               */
#define PORT0_IN_P7_Msk                                       (0x01UL << PORT0_IN_P7_Pos)                               /*!< PORT0 IN: P7 Mask                   */
#define PORT0_IN_P8_Pos                                       8                                                         /*!< PORT0 IN: P8 Position               */
#define PORT0_IN_P8_Msk                                       (0x01UL << PORT0_IN_P8_Pos)                               /*!< PORT0 IN: P8 Mask                   */
#define PORT0_IN_P9_Pos                                       9                                                         /*!< PORT0 IN: P9 Position               */
#define PORT0_IN_P9_Msk                                       (0x01UL << PORT0_IN_P9_Pos)                               /*!< PORT0 IN: P9 Mask                   */
#define PORT0_IN_P10_Pos                                      10                                                        /*!< PORT0 IN: P10 Position              */
#define PORT0_IN_P10_Msk                                      (0x01UL << PORT0_IN_P10_Pos)                              /*!< PORT0 IN: P10 Mask                  */
#define PORT0_IN_P11_Pos                                      11                                                        /*!< PORT0 IN: P11 Position              */
#define PORT0_IN_P11_Msk                                      (0x01UL << PORT0_IN_P11_Pos)                              /*!< PORT0 IN: P11 Mask                  */
#define PORT0_IN_P12_Pos                                      12                                                        /*!< PORT0 IN: P12 Position              */
#define PORT0_IN_P12_Msk                                      (0x01UL << PORT0_IN_P12_Pos)                              /*!< PORT0 IN: P12 Mask                  */
#define PORT0_IN_P13_Pos                                      13                                                        /*!< PORT0 IN: P13 Position              */
#define PORT0_IN_P13_Msk                                      (0x01UL << PORT0_IN_P13_Pos)                              /*!< PORT0 IN: P13 Mask                  */
#define PORT0_IN_P14_Pos                                      14                                                        /*!< PORT0 IN: P14 Position              */
#define PORT0_IN_P14_Msk                                      (0x01UL << PORT0_IN_P14_Pos)                              /*!< PORT0 IN: P14 Mask                  */
#define PORT0_IN_P15_Pos                                      15                                                        /*!< PORT0 IN: P15 Position              */
#define PORT0_IN_P15_Msk                                      (0x01UL << PORT0_IN_P15_Pos)                              /*!< PORT0 IN: P15 Mask                  */

// ---------------------------------------  PORT0_PDR0  -------------------------------------------
#define PORT0_PDR0_PD0_Pos                                    0                                                         /*!< PORT0 PDR0: PD0 Position            */
#define PORT0_PDR0_PD0_Msk                                    (0x07UL << PORT0_PDR0_PD0_Pos)                            /*!< PORT0 PDR0: PD0 Mask                */
#define PORT0_PDR0_PD1_Pos                                    4                                                         /*!< PORT0 PDR0: PD1 Position            */
#define PORT0_PDR0_PD1_Msk                                    (0x07UL << PORT0_PDR0_PD1_Pos)                            /*!< PORT0 PDR0: PD1 Mask                */
#define PORT0_PDR0_PD2_Pos                                    8                                                         /*!< PORT0 PDR0: PD2 Position            */
#define PORT0_PDR0_PD2_Msk                                    (0x07UL << PORT0_PDR0_PD2_Pos)                            /*!< PORT0 PDR0: PD2 Mask                */
#define PORT0_PDR0_PD3_Pos                                    12                                                        /*!< PORT0 PDR0: PD3 Position            */
#define PORT0_PDR0_PD3_Msk                                    (0x07UL << PORT0_PDR0_PD3_Pos)                            /*!< PORT0 PDR0: PD3 Mask                */
#define PORT0_PDR0_PD4_Pos                                    16                                                        /*!< PORT0 PDR0: PD4 Position            */
#define PORT0_PDR0_PD4_Msk                                    (0x07UL << PORT0_PDR0_PD4_Pos)                            /*!< PORT0 PDR0: PD4 Mask                */
#define PORT0_PDR0_PD5_Pos                                    20                                                        /*!< PORT0 PDR0: PD5 Position            */
#define PORT0_PDR0_PD5_Msk                                    (0x07UL << PORT0_PDR0_PD5_Pos)                            /*!< PORT0 PDR0: PD5 Mask                */
#define PORT0_PDR0_PD6_Pos                                    24                                                        /*!< PORT0 PDR0: PD6 Position            */
#define PORT0_PDR0_PD6_Msk                                    (0x07UL << PORT0_PDR0_PD6_Pos)                            /*!< PORT0 PDR0: PD6 Mask                */
#define PORT0_PDR0_PD7_Pos                                    28                                                        /*!< PORT0 PDR0: PD7 Position            */
#define PORT0_PDR0_PD7_Msk                                    (0x07UL << PORT0_PDR0_PD7_Pos)                            /*!< PORT0 PDR0: PD7 Mask                */

// ---------------------------------------  PORT0_PDR1  -------------------------------------------
#define PORT0_PDR1_PD8_Pos                                    0                                                         /*!< PORT0 PDR1: PD8 Position            */
#define PORT0_PDR1_PD8_Msk                                    (0x07UL << PORT0_PDR1_PD8_Pos)                            /*!< PORT0 PDR1: PD8 Mask                */
#define PORT0_PDR1_PD9_Pos                                    4                                                         /*!< PORT0 PDR1: PD9 Position            */
#define PORT0_PDR1_PD9_Msk                                    (0x07UL << PORT0_PDR1_PD9_Pos)                            /*!< PORT0 PDR1: PD9 Mask                */
#define PORT0_PDR1_PD10_Pos                                   8                                                         /*!< PORT0 PDR1: PD10 Position           */
#define PORT0_PDR1_PD10_Msk                                   (0x07UL << PORT0_PDR1_PD10_Pos)                           /*!< PORT0 PDR1: PD10 Mask               */
#define PORT0_PDR1_PD11_Pos                                   12                                                        /*!< PORT0 PDR1: PD11 Position           */
#define PORT0_PDR1_PD11_Msk                                   (0x07UL << PORT0_PDR1_PD11_Pos)                           /*!< PORT0 PDR1: PD11 Mask               */
#define PORT0_PDR1_PD12_Pos                                   16                                                        /*!< PORT0 PDR1: PD12 Position           */
#define PORT0_PDR1_PD12_Msk                                   (0x07UL << PORT0_PDR1_PD12_Pos)                           /*!< PORT0 PDR1: PD12 Mask               */
#define PORT0_PDR1_PD13_Pos                                   20                                                        /*!< PORT0 PDR1: PD13 Position           */
#define PORT0_PDR1_PD13_Msk                                   (0x07UL << PORT0_PDR1_PD13_Pos)                           /*!< PORT0 PDR1: PD13 Mask               */
#define PORT0_PDR1_PD14_Pos                                   24                                                        /*!< PORT0 PDR1: PD14 Position           */
#define PORT0_PDR1_PD14_Msk                                   (0x07UL << PORT0_PDR1_PD14_Pos)                           /*!< PORT0 PDR1: PD14 Mask               */
#define PORT0_PDR1_PD15_Pos                                   28                                                        /*!< PORT0 PDR1: PD15 Position           */
#define PORT0_PDR1_PD15_Msk                                   (0x07UL << PORT0_PDR1_PD15_Pos)                           /*!< PORT0 PDR1: PD15 Mask               */

// ---------------------------------------  PORT0_PDISC  ------------------------------------------
#define PORT0_PDISC_PDIS0_Pos                                 0                                                         /*!< PORT0 PDISC: PDIS0 Position         */
#define PORT0_PDISC_PDIS0_Msk                                 (0x01UL << PORT0_PDISC_PDIS0_Pos)                         /*!< PORT0 PDISC: PDIS0 Mask             */
#define PORT0_PDISC_PDIS1_Pos                                 1                                                         /*!< PORT0 PDISC: PDIS1 Position         */
#define PORT0_PDISC_PDIS1_Msk                                 (0x01UL << PORT0_PDISC_PDIS1_Pos)                         /*!< PORT0 PDISC: PDIS1 Mask             */
#define PORT0_PDISC_PDIS2_Pos                                 2                                                         /*!< PORT0 PDISC: PDIS2 Position         */
#define PORT0_PDISC_PDIS2_Msk                                 (0x01UL << PORT0_PDISC_PDIS2_Pos)                         /*!< PORT0 PDISC: PDIS2 Mask             */
#define PORT0_PDISC_PDIS3_Pos                                 3                                                         /*!< PORT0 PDISC: PDIS3 Position         */
#define PORT0_PDISC_PDIS3_Msk                                 (0x01UL << PORT0_PDISC_PDIS3_Pos)                         /*!< PORT0 PDISC: PDIS3 Mask             */
#define PORT0_PDISC_PDIS4_Pos                                 4                                                         /*!< PORT0 PDISC: PDIS4 Position         */
#define PORT0_PDISC_PDIS4_Msk                                 (0x01UL << PORT0_PDISC_PDIS4_Pos)                         /*!< PORT0 PDISC: PDIS4 Mask             */
#define PORT0_PDISC_PDIS5_Pos                                 5                                                         /*!< PORT0 PDISC: PDIS5 Position         */
#define PORT0_PDISC_PDIS5_Msk                                 (0x01UL << PORT0_PDISC_PDIS5_Pos)                         /*!< PORT0 PDISC: PDIS5 Mask             */
#define PORT0_PDISC_PDIS6_Pos                                 6                                                         /*!< PORT0 PDISC: PDIS6 Position         */
#define PORT0_PDISC_PDIS6_Msk                                 (0x01UL << PORT0_PDISC_PDIS6_Pos)                         /*!< PORT0 PDISC: PDIS6 Mask             */
#define PORT0_PDISC_PDIS7_Pos                                 7                                                         /*!< PORT0 PDISC: PDIS7 Position         */
#define PORT0_PDISC_PDIS7_Msk                                 (0x01UL << PORT0_PDISC_PDIS7_Pos)                         /*!< PORT0 PDISC: PDIS7 Mask             */
#define PORT0_PDISC_PDIS8_Pos                                 8                                                         /*!< PORT0 PDISC: PDIS8 Position         */
#define PORT0_PDISC_PDIS8_Msk                                 (0x01UL << PORT0_PDISC_PDIS8_Pos)                         /*!< PORT0 PDISC: PDIS8 Mask             */
#define PORT0_PDISC_PDIS9_Pos                                 9                                                         /*!< PORT0 PDISC: PDIS9 Position         */
#define PORT0_PDISC_PDIS9_Msk                                 (0x01UL << PORT0_PDISC_PDIS9_Pos)                         /*!< PORT0 PDISC: PDIS9 Mask             */
#define PORT0_PDISC_PDIS10_Pos                                10                                                        /*!< PORT0 PDISC: PDIS10 Position        */
#define PORT0_PDISC_PDIS10_Msk                                (0x01UL << PORT0_PDISC_PDIS10_Pos)                        /*!< PORT0 PDISC: PDIS10 Mask            */
#define PORT0_PDISC_PDIS11_Pos                                11                                                        /*!< PORT0 PDISC: PDIS11 Position        */
#define PORT0_PDISC_PDIS11_Msk                                (0x01UL << PORT0_PDISC_PDIS11_Pos)                        /*!< PORT0 PDISC: PDIS11 Mask            */
#define PORT0_PDISC_PDIS12_Pos                                12                                                        /*!< PORT0 PDISC: PDIS12 Position        */
#define PORT0_PDISC_PDIS12_Msk                                (0x01UL << PORT0_PDISC_PDIS12_Pos)                        /*!< PORT0 PDISC: PDIS12 Mask            */
#define PORT0_PDISC_PDIS13_Pos                                13                                                        /*!< PORT0 PDISC: PDIS13 Position        */
#define PORT0_PDISC_PDIS13_Msk                                (0x01UL << PORT0_PDISC_PDIS13_Pos)                        /*!< PORT0 PDISC: PDIS13 Mask            */
#define PORT0_PDISC_PDIS14_Pos                                14                                                        /*!< PORT0 PDISC: PDIS14 Position        */
#define PORT0_PDISC_PDIS14_Msk                                (0x01UL << PORT0_PDISC_PDIS14_Pos)                        /*!< PORT0 PDISC: PDIS14 Mask            */
#define PORT0_PDISC_PDIS15_Pos                                15                                                        /*!< PORT0 PDISC: PDIS15 Position        */
#define PORT0_PDISC_PDIS15_Msk                                (0x01UL << PORT0_PDISC_PDIS15_Pos)                        /*!< PORT0 PDISC: PDIS15 Mask            */

// ----------------------------------------  PORT0_PPS  -------------------------------------------
#define PORT0_PPS_PPS0_Pos                                    0                                                         /*!< PORT0 PPS: PPS0 Position            */
#define PORT0_PPS_PPS0_Msk                                    (0x01UL << PORT0_PPS_PPS0_Pos)                            /*!< PORT0 PPS: PPS0 Mask                */
#define PORT0_PPS_PPS1_Pos                                    1                                                         /*!< PORT0 PPS: PPS1 Position            */
#define PORT0_PPS_PPS1_Msk                                    (0x01UL << PORT0_PPS_PPS1_Pos)                            /*!< PORT0 PPS: PPS1 Mask                */
#define PORT0_PPS_PPS2_Pos                                    2                                                         /*!< PORT0 PPS: PPS2 Position            */
#define PORT0_PPS_PPS2_Msk                                    (0x01UL << PORT0_PPS_PPS2_Pos)                            /*!< PORT0 PPS: PPS2 Mask                */
#define PORT0_PPS_PPS3_Pos                                    3                                                         /*!< PORT0 PPS: PPS3 Position            */
#define PORT0_PPS_PPS3_Msk                                    (0x01UL << PORT0_PPS_PPS3_Pos)                            /*!< PORT0 PPS: PPS3 Mask                */
#define PORT0_PPS_PPS4_Pos                                    4                                                         /*!< PORT0 PPS: PPS4 Position            */
#define PORT0_PPS_PPS4_Msk                                    (0x01UL << PORT0_PPS_PPS4_Pos)                            /*!< PORT0 PPS: PPS4 Mask                */
#define PORT0_PPS_PPS5_Pos                                    5                                                         /*!< PORT0 PPS: PPS5 Position            */
#define PORT0_PPS_PPS5_Msk                                    (0x01UL << PORT0_PPS_PPS5_Pos)                            /*!< PORT0 PPS: PPS5 Mask                */
#define PORT0_PPS_PPS6_Pos                                    6                                                         /*!< PORT0 PPS: PPS6 Position            */
#define PORT0_PPS_PPS6_Msk                                    (0x01UL << PORT0_PPS_PPS6_Pos)                            /*!< PORT0 PPS: PPS6 Mask                */
#define PORT0_PPS_PPS7_Pos                                    7                                                         /*!< PORT0 PPS: PPS7 Position            */
#define PORT0_PPS_PPS7_Msk                                    (0x01UL << PORT0_PPS_PPS7_Pos)                            /*!< PORT0 PPS: PPS7 Mask                */
#define PORT0_PPS_PPS8_Pos                                    8                                                         /*!< PORT0 PPS: PPS8 Position            */
#define PORT0_PPS_PPS8_Msk                                    (0x01UL << PORT0_PPS_PPS8_Pos)                            /*!< PORT0 PPS: PPS8 Mask                */
#define PORT0_PPS_PPS9_Pos                                    9                                                         /*!< PORT0 PPS: PPS9 Position            */
#define PORT0_PPS_PPS9_Msk                                    (0x01UL << PORT0_PPS_PPS9_Pos)                            /*!< PORT0 PPS: PPS9 Mask                */
#define PORT0_PPS_PPS10_Pos                                   10                                                        /*!< PORT0 PPS: PPS10 Position           */
#define PORT0_PPS_PPS10_Msk                                   (0x01UL << PORT0_PPS_PPS10_Pos)                           /*!< PORT0 PPS: PPS10 Mask               */
#define PORT0_PPS_PPS11_Pos                                   11                                                        /*!< PORT0 PPS: PPS11 Position           */
#define PORT0_PPS_PPS11_Msk                                   (0x01UL << PORT0_PPS_PPS11_Pos)                           /*!< PORT0 PPS: PPS11 Mask               */
#define PORT0_PPS_PPS12_Pos                                   12                                                        /*!< PORT0 PPS: PPS12 Position           */
#define PORT0_PPS_PPS12_Msk                                   (0x01UL << PORT0_PPS_PPS12_Pos)                           /*!< PORT0 PPS: PPS12 Mask               */
#define PORT0_PPS_PPS13_Pos                                   13                                                        /*!< PORT0 PPS: PPS13 Position           */
#define PORT0_PPS_PPS13_Msk                                   (0x01UL << PORT0_PPS_PPS13_Pos)                           /*!< PORT0 PPS: PPS13 Mask               */
#define PORT0_PPS_PPS14_Pos                                   14                                                        /*!< PORT0 PPS: PPS14 Position           */
#define PORT0_PPS_PPS14_Msk                                   (0x01UL << PORT0_PPS_PPS14_Pos)                           /*!< PORT0 PPS: PPS14 Mask               */
#define PORT0_PPS_PPS15_Pos                                   15                                                        /*!< PORT0 PPS: PPS15 Position           */
#define PORT0_PPS_PPS15_Msk                                   (0x01UL << PORT0_PPS_PPS15_Pos)                           /*!< PORT0 PPS: PPS15 Mask               */

// ---------------------------------------  PORT0_HWSEL  ------------------------------------------
#define PORT0_HWSEL_HW0_Pos                                   0                                                         /*!< PORT0 HWSEL: HW0 Position           */
#define PORT0_HWSEL_HW0_Msk                                   (0x03UL << PORT0_HWSEL_HW0_Pos)                           /*!< PORT0 HWSEL: HW0 Mask               */
#define PORT0_HWSEL_HW1_Pos                                   2                                                         /*!< PORT0 HWSEL: HW1 Position           */
#define PORT0_HWSEL_HW1_Msk                                   (0x03UL << PORT0_HWSEL_HW1_Pos)                           /*!< PORT0 HWSEL: HW1 Mask               */
#define PORT0_HWSEL_HW2_Pos                                   4                                                         /*!< PORT0 HWSEL: HW2 Position           */
#define PORT0_HWSEL_HW2_Msk                                   (0x03UL << PORT0_HWSEL_HW2_Pos)                           /*!< PORT0 HWSEL: HW2 Mask               */
#define PORT0_HWSEL_HW3_Pos                                   6                                                         /*!< PORT0 HWSEL: HW3 Position           */
#define PORT0_HWSEL_HW3_Msk                                   (0x03UL << PORT0_HWSEL_HW3_Pos)                           /*!< PORT0 HWSEL: HW3 Mask               */
#define PORT0_HWSEL_HW4_Pos                                   8                                                         /*!< PORT0 HWSEL: HW4 Position           */
#define PORT0_HWSEL_HW4_Msk                                   (0x03UL << PORT0_HWSEL_HW4_Pos)                           /*!< PORT0 HWSEL: HW4 Mask               */
#define PORT0_HWSEL_HW5_Pos                                   10                                                        /*!< PORT0 HWSEL: HW5 Position           */
#define PORT0_HWSEL_HW5_Msk                                   (0x03UL << PORT0_HWSEL_HW5_Pos)                           /*!< PORT0 HWSEL: HW5 Mask               */
#define PORT0_HWSEL_HW6_Pos                                   12                                                        /*!< PORT0 HWSEL: HW6 Position           */
#define PORT0_HWSEL_HW6_Msk                                   (0x03UL << PORT0_HWSEL_HW6_Pos)                           /*!< PORT0 HWSEL: HW6 Mask               */
#define PORT0_HWSEL_HW7_Pos                                   14                                                        /*!< PORT0 HWSEL: HW7 Position           */
#define PORT0_HWSEL_HW7_Msk                                   (0x03UL << PORT0_HWSEL_HW7_Pos)                           /*!< PORT0 HWSEL: HW7 Mask               */
#define PORT0_HWSEL_HW8_Pos                                   16                                                        /*!< PORT0 HWSEL: HW8 Position           */
#define PORT0_HWSEL_HW8_Msk                                   (0x03UL << PORT0_HWSEL_HW8_Pos)                           /*!< PORT0 HWSEL: HW8 Mask               */
#define PORT0_HWSEL_HW9_Pos                                   18                                                        /*!< PORT0 HWSEL: HW9 Position           */
#define PORT0_HWSEL_HW9_Msk                                   (0x03UL << PORT0_HWSEL_HW9_Pos)                           /*!< PORT0 HWSEL: HW9 Mask               */
#define PORT0_HWSEL_HW10_Pos                                  20                                                        /*!< PORT0 HWSEL: HW10 Position          */
#define PORT0_HWSEL_HW10_Msk                                  (0x03UL << PORT0_HWSEL_HW10_Pos)                          /*!< PORT0 HWSEL: HW10 Mask              */
#define PORT0_HWSEL_HW11_Pos                                  22                                                        /*!< PORT0 HWSEL: HW11 Position          */
#define PORT0_HWSEL_HW11_Msk                                  (0x03UL << PORT0_HWSEL_HW11_Pos)                          /*!< PORT0 HWSEL: HW11 Mask              */
#define PORT0_HWSEL_HW12_Pos                                  24                                                        /*!< PORT0 HWSEL: HW12 Position          */
#define PORT0_HWSEL_HW12_Msk                                  (0x03UL << PORT0_HWSEL_HW12_Pos)                          /*!< PORT0 HWSEL: HW12 Mask              */
#define PORT0_HWSEL_HW13_Pos                                  26                                                        /*!< PORT0 HWSEL: HW13 Position          */
#define PORT0_HWSEL_HW13_Msk                                  (0x03UL << PORT0_HWSEL_HW13_Pos)                          /*!< PORT0 HWSEL: HW13 Mask              */
#define PORT0_HWSEL_HW14_Pos                                  28                                                        /*!< PORT0 HWSEL: HW14 Position          */
#define PORT0_HWSEL_HW14_Msk                                  (0x03UL << PORT0_HWSEL_HW14_Pos)                          /*!< PORT0 HWSEL: HW14 Mask              */
#define PORT0_HWSEL_HW15_Pos                                  30                                                        /*!< PORT0 HWSEL: HW15 Position          */
#define PORT0_HWSEL_HW15_Msk                                  (0x03UL << PORT0_HWSEL_HW15_Pos)                          /*!< PORT0 HWSEL: HW15 Mask              */


// ------------------------------------------------------------------------------------------------
// -----                                 PORT1 Position & Mask                                -----
// ------------------------------------------------------------------------------------------------


// ----------------------------------------  PORT1_OUT  -------------------------------------------
#define PORT1_OUT_P0_Pos                                      0                                                         /*!< PORT1 OUT: P0 Position              */
#define PORT1_OUT_P0_Msk                                      (0x01UL << PORT1_OUT_P0_Pos)                              /*!< PORT1 OUT: P0 Mask                  */
#define PORT1_OUT_P1_Pos                                      1                                                         /*!< PORT1 OUT: P1 Position              */
#define PORT1_OUT_P1_Msk                                      (0x01UL << PORT1_OUT_P1_Pos)                              /*!< PORT1 OUT: P1 Mask                  */
#define PORT1_OUT_P2_Pos                                      2                                                         /*!< PORT1 OUT: P2 Position              */
#define PORT1_OUT_P2_Msk                                      (0x01UL << PORT1_OUT_P2_Pos)                              /*!< PORT1 OUT: P2 Mask                  */
#define PORT1_OUT_P3_Pos                                      3                                                         /*!< PORT1 OUT: P3 Position              */
#define PORT1_OUT_P3_Msk                                      (0x01UL << PORT1_OUT_P3_Pos)                              /*!< PORT1 OUT: P3 Mask                  */
#define PORT1_OUT_P4_Pos                                      4                                                         /*!< PORT1 OUT: P4 Position              */
#define PORT1_OUT_P4_Msk                                      (0x01UL << PORT1_OUT_P4_Pos)                              /*!< PORT1 OUT: P4 Mask                  */
#define PORT1_OUT_P5_Pos                                      5                                                         /*!< PORT1 OUT: P5 Position              */
#define PORT1_OUT_P5_Msk                                      (0x01UL << PORT1_OUT_P5_Pos)                              /*!< PORT1 OUT: P5 Mask                  */
#define PORT1_OUT_P6_Pos                                      6                                                         /*!< PORT1 OUT: P6 Position              */
#define PORT1_OUT_P6_Msk                                      (0x01UL << PORT1_OUT_P6_Pos)                              /*!< PORT1 OUT: P6 Mask                  */
#define PORT1_OUT_P7_Pos                                      7                                                         /*!< PORT1 OUT: P7 Position              */
#define PORT1_OUT_P7_Msk                                      (0x01UL << PORT1_OUT_P7_Pos)                              /*!< PORT1 OUT: P7 Mask                  */
#define PORT1_OUT_P8_Pos                                      8                                                         /*!< PORT1 OUT: P8 Position              */
#define PORT1_OUT_P8_Msk                                      (0x01UL << PORT1_OUT_P8_Pos)                              /*!< PORT1 OUT: P8 Mask                  */
#define PORT1_OUT_P9_Pos                                      9                                                         /*!< PORT1 OUT: P9 Position              */
#define PORT1_OUT_P9_Msk                                      (0x01UL << PORT1_OUT_P9_Pos)                              /*!< PORT1 OUT: P9 Mask                  */
#define PORT1_OUT_P10_Pos                                     10                                                        /*!< PORT1 OUT: P10 Position             */
#define PORT1_OUT_P10_Msk                                     (0x01UL << PORT1_OUT_P10_Pos)                             /*!< PORT1 OUT: P10 Mask                 */
#define PORT1_OUT_P11_Pos                                     11                                                        /*!< PORT1 OUT: P11 Position             */
#define PORT1_OUT_P11_Msk                                     (0x01UL << PORT1_OUT_P11_Pos)                             /*!< PORT1 OUT: P11 Mask                 */
#define PORT1_OUT_P12_Pos                                     12                                                        /*!< PORT1 OUT: P12 Position             */
#define PORT1_OUT_P12_Msk                                     (0x01UL << PORT1_OUT_P12_Pos)                             /*!< PORT1 OUT: P12 Mask                 */
#define PORT1_OUT_P13_Pos                                     13                                                        /*!< PORT1 OUT: P13 Position             */
#define PORT1_OUT_P13_Msk                                     (0x01UL << PORT1_OUT_P13_Pos)                             /*!< PORT1 OUT: P13 Mask                 */
#define PORT1_OUT_P14_Pos                                     14                                                        /*!< PORT1 OUT: P14 Position             */
#define PORT1_OUT_P14_Msk                                     (0x01UL << PORT1_OUT_P14_Pos)                             /*!< PORT1 OUT: P14 Mask                 */
#define PORT1_OUT_P15_Pos                                     15                                                        /*!< PORT1 OUT: P15 Position             */
#define PORT1_OUT_P15_Msk                                     (0x01UL << PORT1_OUT_P15_Pos)                             /*!< PORT1 OUT: P15 Mask                 */

// ----------------------------------------  PORT1_OMR  -------------------------------------------
#define PORT1_OMR_PS0_Pos                                     0                                                         /*!< PORT1 OMR: PS0 Position             */
#define PORT1_OMR_PS0_Msk                                     (0x01UL << PORT1_OMR_PS0_Pos)                             /*!< PORT1 OMR: PS0 Mask                 */
#define PORT1_OMR_PS1_Pos                                     1                                                         /*!< PORT1 OMR: PS1 Position             */
#define PORT1_OMR_PS1_Msk                                     (0x01UL << PORT1_OMR_PS1_Pos)                             /*!< PORT1 OMR: PS1 Mask                 */
#define PORT1_OMR_PS2_Pos                                     2                                                         /*!< PORT1 OMR: PS2 Position             */
#define PORT1_OMR_PS2_Msk                                     (0x01UL << PORT1_OMR_PS2_Pos)                             /*!< PORT1 OMR: PS2 Mask                 */
#define PORT1_OMR_PS3_Pos                                     3                                                         /*!< PORT1 OMR: PS3 Position             */
#define PORT1_OMR_PS3_Msk                                     (0x01UL << PORT1_OMR_PS3_Pos)                             /*!< PORT1 OMR: PS3 Mask                 */
#define PORT1_OMR_PS4_Pos                                     4                                                         /*!< PORT1 OMR: PS4 Position             */
#define PORT1_OMR_PS4_Msk                                     (0x01UL << PORT1_OMR_PS4_Pos)                             /*!< PORT1 OMR: PS4 Mask                 */
#define PORT1_OMR_PS5_Pos                                     5                                                         /*!< PORT1 OMR: PS5 Position             */
#define PORT1_OMR_PS5_Msk                                     (0x01UL << PORT1_OMR_PS5_Pos)                             /*!< PORT1 OMR: PS5 Mask                 */
#define PORT1_OMR_PS6_Pos                                     6                                                         /*!< PORT1 OMR: PS6 Position             */
#define PORT1_OMR_PS6_Msk                                     (0x01UL << PORT1_OMR_PS6_Pos)                             /*!< PORT1 OMR: PS6 Mask                 */
#define PORT1_OMR_PS7_Pos                                     7                                                         /*!< PORT1 OMR: PS7 Position             */
#define PORT1_OMR_PS7_Msk                                     (0x01UL << PORT1_OMR_PS7_Pos)                             /*!< PORT1 OMR: PS7 Mask                 */
#define PORT1_OMR_PS8_Pos                                     8                                                         /*!< PORT1 OMR: PS8 Position             */
#define PORT1_OMR_PS8_Msk                                     (0x01UL << PORT1_OMR_PS8_Pos)                             /*!< PORT1 OMR: PS8 Mask                 */
#define PORT1_OMR_PS9_Pos                                     9                                                         /*!< PORT1 OMR: PS9 Position             */
#define PORT1_OMR_PS9_Msk                                     (0x01UL << PORT1_OMR_PS9_Pos)                             /*!< PORT1 OMR: PS9 Mask                 */
#define PORT1_OMR_PS10_Pos                                    10                                                        /*!< PORT1 OMR: PS10 Position            */
#define PORT1_OMR_PS10_Msk                                    (0x01UL << PORT1_OMR_PS10_Pos)                            /*!< PORT1 OMR: PS10 Mask                */
#define PORT1_OMR_PS11_Pos                                    11                                                        /*!< PORT1 OMR: PS11 Position            */
#define PORT1_OMR_PS11_Msk                                    (0x01UL << PORT1_OMR_PS11_Pos)                            /*!< PORT1 OMR: PS11 Mask                */
#define PORT1_OMR_PS12_Pos                                    12                                                        /*!< PORT1 OMR: PS12 Position            */
#define PORT1_OMR_PS12_Msk                                    (0x01UL << PORT1_OMR_PS12_Pos)                            /*!< PORT1 OMR: PS12 Mask                */
#define PORT1_OMR_PS13_Pos                                    13                                                        /*!< PORT1 OMR: PS13 Position            */
#define PORT1_OMR_PS13_Msk                                    (0x01UL << PORT1_OMR_PS13_Pos)                            /*!< PORT1 OMR: PS13 Mask                */
#define PORT1_OMR_PS14_Pos                                    14                                                        /*!< PORT1 OMR: PS14 Position            */
#define PORT1_OMR_PS14_Msk                                    (0x01UL << PORT1_OMR_PS14_Pos)                            /*!< PORT1 OMR: PS14 Mask                */
#define PORT1_OMR_PS15_Pos                                    15                                                        /*!< PORT1 OMR: PS15 Position            */
#define PORT1_OMR_PS15_Msk                                    (0x01UL << PORT1_OMR_PS15_Pos)                            /*!< PORT1 OMR: PS15 Mask                */
#define PORT1_OMR_PR0_Pos                                     16                                                        /*!< PORT1 OMR: PR0 Position             */
#define PORT1_OMR_PR0_Msk                                     (0x01UL << PORT1_OMR_PR0_Pos)                             /*!< PORT1 OMR: PR0 Mask                 */
#define PORT1_OMR_PR1_Pos                                     17                                                        /*!< PORT1 OMR: PR1 Position             */
#define PORT1_OMR_PR1_Msk                                     (0x01UL << PORT1_OMR_PR1_Pos)                             /*!< PORT1 OMR: PR1 Mask                 */
#define PORT1_OMR_PR2_Pos                                     18                                                        /*!< PORT1 OMR: PR2 Position             */
#define PORT1_OMR_PR2_Msk                                     (0x01UL << PORT1_OMR_PR2_Pos)                             /*!< PORT1 OMR: PR2 Mask                 */
#define PORT1_OMR_PR3_Pos                                     19                                                        /*!< PORT1 OMR: PR3 Position             */
#define PORT1_OMR_PR3_Msk                                     (0x01UL << PORT1_OMR_PR3_Pos)                             /*!< PORT1 OMR: PR3 Mask                 */
#define PORT1_OMR_PR4_Pos                                     20                                                        /*!< PORT1 OMR: PR4 Position             */
#define PORT1_OMR_PR4_Msk                                     (0x01UL << PORT1_OMR_PR4_Pos)                             /*!< PORT1 OMR: PR4 Mask                 */
#define PORT1_OMR_PR5_Pos                                     21                                                        /*!< PORT1 OMR: PR5 Position             */
#define PORT1_OMR_PR5_Msk                                     (0x01UL << PORT1_OMR_PR5_Pos)                             /*!< PORT1 OMR: PR5 Mask                 */
#define PORT1_OMR_PR6_Pos                                     22                                                        /*!< PORT1 OMR: PR6 Position             */
#define PORT1_OMR_PR6_Msk                                     (0x01UL << PORT1_OMR_PR6_Pos)                             /*!< PORT1 OMR: PR6 Mask                 */
#define PORT1_OMR_PR7_Pos                                     23                                                        /*!< PORT1 OMR: PR7 Position             */
#define PORT1_OMR_PR7_Msk                                     (0x01UL << PORT1_OMR_PR7_Pos)                             /*!< PORT1 OMR: PR7 Mask                 */
#define PORT1_OMR_PR8_Pos                                     24                                                        /*!< PORT1 OMR: PR8 Position             */
#define PORT1_OMR_PR8_Msk                                     (0x01UL << PORT1_OMR_PR8_Pos)                             /*!< PORT1 OMR: PR8 Mask                 */
#define PORT1_OMR_PR9_Pos                                     25                                                        /*!< PORT1 OMR: PR9 Position             */
#define PORT1_OMR_PR9_Msk                                     (0x01UL << PORT1_OMR_PR9_Pos)                             /*!< PORT1 OMR: PR9 Mask                 */
#define PORT1_OMR_PR10_Pos                                    26                                                        /*!< PORT1 OMR: PR10 Position            */
#define PORT1_OMR_PR10_Msk                                    (0x01UL << PORT1_OMR_PR10_Pos)                            /*!< PORT1 OMR: PR10 Mask                */
#define PORT1_OMR_PR11_Pos                                    27                                                        /*!< PORT1 OMR: PR11 Position            */
#define PORT1_OMR_PR11_Msk                                    (0x01UL << PORT1_OMR_PR11_Pos)                            /*!< PORT1 OMR: PR11 Mask                */
#define PORT1_OMR_PR12_Pos                                    28                                                        /*!< PORT1 OMR: PR12 Position            */
#define PORT1_OMR_PR12_Msk                                    (0x01UL << PORT1_OMR_PR12_Pos)                            /*!< PORT1 OMR: PR12 Mask                */
#define PORT1_OMR_PR13_Pos                                    29                                                        /*!< PORT1 OMR: PR13 Position            */
#define PORT1_OMR_PR13_Msk                                    (0x01UL << PORT1_OMR_PR13_Pos)                            /*!< PORT1 OMR: PR13 Mask                */
#define PORT1_OMR_PR14_Pos                                    30                                                        /*!< PORT1 OMR: PR14 Position            */
#define PORT1_OMR_PR14_Msk                                    (0x01UL << PORT1_OMR_PR14_Pos)                            /*!< PORT1 OMR: PR14 Mask                */
#define PORT1_OMR_PR15_Pos                                    31                                                        /*!< PORT1 OMR: PR15 Position            */
#define PORT1_OMR_PR15_Msk                                    (0x01UL << PORT1_OMR_PR15_Pos)                            /*!< PORT1 OMR: PR15 Mask                */

// ---------------------------------------  PORT1_IOCR0  ------------------------------------------
#define PORT1_IOCR0_PC0_Pos                                   3                                                         /*!< PORT1 IOCR0: PC0 Position           */
#define PORT1_IOCR0_PC0_Msk                                   (0x1fUL << PORT1_IOCR0_PC0_Pos)                           /*!< PORT1 IOCR0: PC0 Mask               */
#define PORT1_IOCR0_PC1_Pos                                   11                                                        /*!< PORT1 IOCR0: PC1 Position           */
#define PORT1_IOCR0_PC1_Msk                                   (0x1fUL << PORT1_IOCR0_PC1_Pos)                           /*!< PORT1 IOCR0: PC1 Mask               */
#define PORT1_IOCR0_PC2_Pos                                   19                                                        /*!< PORT1 IOCR0: PC2 Position           */
#define PORT1_IOCR0_PC2_Msk                                   (0x1fUL << PORT1_IOCR0_PC2_Pos)                           /*!< PORT1 IOCR0: PC2 Mask               */
#define PORT1_IOCR0_PC3_Pos                                   27                                                        /*!< PORT1 IOCR0: PC3 Position           */
#define PORT1_IOCR0_PC3_Msk                                   (0x1fUL << PORT1_IOCR0_PC3_Pos)                           /*!< PORT1 IOCR0: PC3 Mask               */

// ---------------------------------------  PORT1_IOCR4  ------------------------------------------
#define PORT1_IOCR4_PC4_Pos                                   3                                                         /*!< PORT1 IOCR4: PC4 Position           */
#define PORT1_IOCR4_PC4_Msk                                   (0x1fUL << PORT1_IOCR4_PC4_Pos)                           /*!< PORT1 IOCR4: PC4 Mask               */
#define PORT1_IOCR4_PC5_Pos                                   11                                                        /*!< PORT1 IOCR4: PC5 Position           */
#define PORT1_IOCR4_PC5_Msk                                   (0x1fUL << PORT1_IOCR4_PC5_Pos)                           /*!< PORT1 IOCR4: PC5 Mask               */
#define PORT1_IOCR4_PC6_Pos                                   19                                                        /*!< PORT1 IOCR4: PC6 Position           */
#define PORT1_IOCR4_PC6_Msk                                   (0x1fUL << PORT1_IOCR4_PC6_Pos)                           /*!< PORT1 IOCR4: PC6 Mask               */
#define PORT1_IOCR4_PC7_Pos                                   27                                                        /*!< PORT1 IOCR4: PC7 Position           */
#define PORT1_IOCR4_PC7_Msk                                   (0x1fUL << PORT1_IOCR4_PC7_Pos)                           /*!< PORT1 IOCR4: PC7 Mask               */

// ---------------------------------------  PORT1_IOCR8  ------------------------------------------
#define PORT1_IOCR8_PC8_Pos                                   3                                                         /*!< PORT1 IOCR8: PC8 Position           */
#define PORT1_IOCR8_PC8_Msk                                   (0x1fUL << PORT1_IOCR8_PC8_Pos)                           /*!< PORT1 IOCR8: PC8 Mask               */
#define PORT1_IOCR8_PC9_Pos                                   11                                                        /*!< PORT1 IOCR8: PC9 Position           */
#define PORT1_IOCR8_PC9_Msk                                   (0x1fUL << PORT1_IOCR8_PC9_Pos)                           /*!< PORT1 IOCR8: PC9 Mask               */
#define PORT1_IOCR8_PC10_Pos                                  19                                                        /*!< PORT1 IOCR8: PC10 Position          */
#define PORT1_IOCR8_PC10_Msk                                  (0x1fUL << PORT1_IOCR8_PC10_Pos)                          /*!< PORT1 IOCR8: PC10 Mask              */
#define PORT1_IOCR8_PC11_Pos                                  27                                                        /*!< PORT1 IOCR8: PC11 Position          */
#define PORT1_IOCR8_PC11_Msk                                  (0x1fUL << PORT1_IOCR8_PC11_Pos)                          /*!< PORT1 IOCR8: PC11 Mask              */

// --------------------------------------  PORT1_IOCR12  ------------------------------------------
#define PORT1_IOCR12_PC12_Pos                                 3                                                         /*!< PORT1 IOCR12: PC12 Position         */
#define PORT1_IOCR12_PC12_Msk                                 (0x1fUL << PORT1_IOCR12_PC12_Pos)                         /*!< PORT1 IOCR12: PC12 Mask             */
#define PORT1_IOCR12_PC13_Pos                                 11                                                        /*!< PORT1 IOCR12: PC13 Position         */
#define PORT1_IOCR12_PC13_Msk                                 (0x1fUL << PORT1_IOCR12_PC13_Pos)                         /*!< PORT1 IOCR12: PC13 Mask             */
#define PORT1_IOCR12_PC14_Pos                                 19                                                        /*!< PORT1 IOCR12: PC14 Position         */
#define PORT1_IOCR12_PC14_Msk                                 (0x1fUL << PORT1_IOCR12_PC14_Pos)                         /*!< PORT1 IOCR12: PC14 Mask             */
#define PORT1_IOCR12_PC15_Pos                                 27                                                        /*!< PORT1 IOCR12: PC15 Position         */
#define PORT1_IOCR12_PC15_Msk                                 (0x1fUL << PORT1_IOCR12_PC15_Pos)                         /*!< PORT1 IOCR12: PC15 Mask             */

// ----------------------------------------  PORT1_IN  --------------------------------------------
#define PORT1_IN_P0_Pos                                       0                                                         /*!< PORT1 IN: P0 Position               */
#define PORT1_IN_P0_Msk                                       (0x01UL << PORT1_IN_P0_Pos)                               /*!< PORT1 IN: P0 Mask                   */
#define PORT1_IN_P1_Pos                                       1                                                         /*!< PORT1 IN: P1 Position               */
#define PORT1_IN_P1_Msk                                       (0x01UL << PORT1_IN_P1_Pos)                               /*!< PORT1 IN: P1 Mask                   */
#define PORT1_IN_P2_Pos                                       2                                                         /*!< PORT1 IN: P2 Position               */
#define PORT1_IN_P2_Msk                                       (0x01UL << PORT1_IN_P2_Pos)                               /*!< PORT1 IN: P2 Mask                   */
#define PORT1_IN_P3_Pos                                       3                                                         /*!< PORT1 IN: P3 Position               */
#define PORT1_IN_P3_Msk                                       (0x01UL << PORT1_IN_P3_Pos)                               /*!< PORT1 IN: P3 Mask                   */
#define PORT1_IN_P4_Pos                                       4                                                         /*!< PORT1 IN: P4 Position               */
#define PORT1_IN_P4_Msk                                       (0x01UL << PORT1_IN_P4_Pos)                               /*!< PORT1 IN: P4 Mask                   */
#define PORT1_IN_P5_Pos                                       5                                                         /*!< PORT1 IN: P5 Position               */
#define PORT1_IN_P5_Msk                                       (0x01UL << PORT1_IN_P5_Pos)                               /*!< PORT1 IN: P5 Mask                   */
#define PORT1_IN_P6_Pos                                       6                                                         /*!< PORT1 IN: P6 Position               */
#define PORT1_IN_P6_Msk                                       (0x01UL << PORT1_IN_P6_Pos)                               /*!< PORT1 IN: P6 Mask                   */
#define PORT1_IN_P7_Pos                                       7                                                         /*!< PORT1 IN: P7 Position               */
#define PORT1_IN_P7_Msk                                       (0x01UL << PORT1_IN_P7_Pos)                               /*!< PORT1 IN: P7 Mask                   */
#define PORT1_IN_P8_Pos                                       8                                                         /*!< PORT1 IN: P8 Position               */
#define PORT1_IN_P8_Msk                                       (0x01UL << PORT1_IN_P8_Pos)                               /*!< PORT1 IN: P8 Mask                   */
#define PORT1_IN_P9_Pos                                       9                                                         /*!< PORT1 IN: P9 Position               */
#define PORT1_IN_P9_Msk                                       (0x01UL << PORT1_IN_P9_Pos)                               /*!< PORT1 IN: P9 Mask                   */
#define PORT1_IN_P10_Pos                                      10                                                        /*!< PORT1 IN: P10 Position              */
#define PORT1_IN_P10_Msk                                      (0x01UL << PORT1_IN_P10_Pos)                              /*!< PORT1 IN: P10 Mask                  */
#define PORT1_IN_P11_Pos                                      11                                                        /*!< PORT1 IN: P11 Position              */
#define PORT1_IN_P11_Msk                                      (0x01UL << PORT1_IN_P11_Pos)                              /*!< PORT1 IN: P11 Mask                  */
#define PORT1_IN_P12_Pos                                      12                                                        /*!< PORT1 IN: P12 Position              */
#define PORT1_IN_P12_Msk                                      (0x01UL << PORT1_IN_P12_Pos)                              /*!< PORT1 IN: P12 Mask                  */
#define PORT1_IN_P13_Pos                                      13                                                        /*!< PORT1 IN: P13 Position              */
#define PORT1_IN_P13_Msk                                      (0x01UL << PORT1_IN_P13_Pos)                              /*!< PORT1 IN: P13 Mask                  */
#define PORT1_IN_P14_Pos                                      14                                                        /*!< PORT1 IN: P14 Position              */
#define PORT1_IN_P14_Msk                                      (0x01UL << PORT1_IN_P14_Pos)                              /*!< PORT1 IN: P14 Mask                  */
#define PORT1_IN_P15_Pos                                      15                                                        /*!< PORT1 IN: P15 Position              */
#define PORT1_IN_P15_Msk                                      (0x01UL << PORT1_IN_P15_Pos)                              /*!< PORT1 IN: P15 Mask                  */

// ---------------------------------------  PORT1_PDR0  -------------------------------------------
#define PORT1_PDR0_PD0_Pos                                    0                                                         /*!< PORT1 PDR0: PD0 Position            */
#define PORT1_PDR0_PD0_Msk                                    (0x07UL << PORT1_PDR0_PD0_Pos)                            /*!< PORT1 PDR0: PD0 Mask                */
#define PORT1_PDR0_PD1_Pos                                    4                                                         /*!< PORT1 PDR0: PD1 Position            */
#define PORT1_PDR0_PD1_Msk                                    (0x07UL << PORT1_PDR0_PD1_Pos)                            /*!< PORT1 PDR0: PD1 Mask                */
#define PORT1_PDR0_PD2_Pos                                    8                                                         /*!< PORT1 PDR0: PD2 Position            */
#define PORT1_PDR0_PD2_Msk                                    (0x07UL << PORT1_PDR0_PD2_Pos)                            /*!< PORT1 PDR0: PD2 Mask                */
#define PORT1_PDR0_PD3_Pos                                    12                                                        /*!< PORT1 PDR0: PD3 Position            */
#define PORT1_PDR0_PD3_Msk                                    (0x07UL << PORT1_PDR0_PD3_Pos)                            /*!< PORT1 PDR0: PD3 Mask                */
#define PORT1_PDR0_PD4_Pos                                    16                                                        /*!< PORT1 PDR0: PD4 Position            */
#define PORT1_PDR0_PD4_Msk                                    (0x07UL << PORT1_PDR0_PD4_Pos)                            /*!< PORT1 PDR0: PD4 Mask                */
#define PORT1_PDR0_PD5_Pos                                    20                                                        /*!< PORT1 PDR0: PD5 Position            */
#define PORT1_PDR0_PD5_Msk                                    (0x07UL << PORT1_PDR0_PD5_Pos)                            /*!< PORT1 PDR0: PD5 Mask                */
#define PORT1_PDR0_PD6_Pos                                    24                                                        /*!< PORT1 PDR0: PD6 Position            */
#define PORT1_PDR0_PD6_Msk                                    (0x07UL << PORT1_PDR0_PD6_Pos)                            /*!< PORT1 PDR0: PD6 Mask                */
#define PORT1_PDR0_PD7_Pos                                    28                                                        /*!< PORT1 PDR0: PD7 Position            */
#define PORT1_PDR0_PD7_Msk                                    (0x07UL << PORT1_PDR0_PD7_Pos)                            /*!< PORT1 PDR0: PD7 Mask                */

// ---------------------------------------  PORT1_PDR1  -------------------------------------------
#define PORT1_PDR1_PD8_Pos                                    0                                                         /*!< PORT1 PDR1: PD8 Position            */
#define PORT1_PDR1_PD8_Msk                                    (0x07UL << PORT1_PDR1_PD8_Pos)                            /*!< PORT1 PDR1: PD8 Mask                */
#define PORT1_PDR1_PD9_Pos                                    4                                                         /*!< PORT1 PDR1: PD9 Position            */
#define PORT1_PDR1_PD9_Msk                                    (0x07UL << PORT1_PDR1_PD9_Pos)                            /*!< PORT1 PDR1: PD9 Mask                */
#define PORT1_PDR1_PD10_Pos                                   8                                                         /*!< PORT1 PDR1: PD10 Position           */
#define PORT1_PDR1_PD10_Msk                                   (0x07UL << PORT1_PDR1_PD10_Pos)                           /*!< PORT1 PDR1: PD10 Mask               */
#define PORT1_PDR1_PD11_Pos                                   12                                                        /*!< PORT1 PDR1: PD11 Position           */
#define PORT1_PDR1_PD11_Msk                                   (0x07UL << PORT1_PDR1_PD11_Pos)                           /*!< PORT1 PDR1: PD11 Mask               */
#define PORT1_PDR1_PD12_Pos                                   16                                                        /*!< PORT1 PDR1: PD12 Position           */
#define PORT1_PDR1_PD12_Msk                                   (0x07UL << PORT1_PDR1_PD12_Pos)                           /*!< PORT1 PDR1: PD12 Mask               */
#define PORT1_PDR1_PD13_Pos                                   20                                                        /*!< PORT1 PDR1: PD13 Position           */
#define PORT1_PDR1_PD13_Msk                                   (0x07UL << PORT1_PDR1_PD13_Pos)                           /*!< PORT1 PDR1: PD13 Mask               */
#define PORT1_PDR1_PD14_Pos                                   24                                                        /*!< PORT1 PDR1: PD14 Position           */
#define PORT1_PDR1_PD14_Msk                                   (0x07UL << PORT1_PDR1_PD14_Pos)                           /*!< PORT1 PDR1: PD14 Mask               */
#define PORT1_PDR1_PD15_Pos                                   28                                                        /*!< PORT1 PDR1: PD15 Position           */
#define PORT1_PDR1_PD15_Msk                                   (0x07UL << PORT1_PDR1_PD15_Pos)                           /*!< PORT1 PDR1: PD15 Mask               */

// ---------------------------------------  PORT1_PDISC  ------------------------------------------
#define PORT1_PDISC_PDIS0_Pos                                 0                                                         /*!< PORT1 PDISC: PDIS0 Position         */
#define PORT1_PDISC_PDIS0_Msk                                 (0x01UL << PORT1_PDISC_PDIS0_Pos)                         /*!< PORT1 PDISC: PDIS0 Mask             */
#define PORT1_PDISC_PDIS1_Pos                                 1                                                         /*!< PORT1 PDISC: PDIS1 Position         */
#define PORT1_PDISC_PDIS1_Msk                                 (0x01UL << PORT1_PDISC_PDIS1_Pos)                         /*!< PORT1 PDISC: PDIS1 Mask             */
#define PORT1_PDISC_PDIS2_Pos                                 2                                                         /*!< PORT1 PDISC: PDIS2 Position         */
#define PORT1_PDISC_PDIS2_Msk                                 (0x01UL << PORT1_PDISC_PDIS2_Pos)                         /*!< PORT1 PDISC: PDIS2 Mask             */
#define PORT1_PDISC_PDIS3_Pos                                 3                                                         /*!< PORT1 PDISC: PDIS3 Position         */
#define PORT1_PDISC_PDIS3_Msk                                 (0x01UL << PORT1_PDISC_PDIS3_Pos)                         /*!< PORT1 PDISC: PDIS3 Mask             */
#define PORT1_PDISC_PDIS4_Pos                                 4                                                         /*!< PORT1 PDISC: PDIS4 Position         */
#define PORT1_PDISC_PDIS4_Msk                                 (0x01UL << PORT1_PDISC_PDIS4_Pos)                         /*!< PORT1 PDISC: PDIS4 Mask             */
#define PORT1_PDISC_PDIS5_Pos                                 5                                                         /*!< PORT1 PDISC: PDIS5 Position         */
#define PORT1_PDISC_PDIS5_Msk                                 (0x01UL << PORT1_PDISC_PDIS5_Pos)                         /*!< PORT1 PDISC: PDIS5 Mask             */
#define PORT1_PDISC_PDIS6_Pos                                 6                                                         /*!< PORT1 PDISC: PDIS6 Position         */
#define PORT1_PDISC_PDIS6_Msk                                 (0x01UL << PORT1_PDISC_PDIS6_Pos)                         /*!< PORT1 PDISC: PDIS6 Mask             */
#define PORT1_PDISC_PDIS7_Pos                                 7                                                         /*!< PORT1 PDISC: PDIS7 Position         */
#define PORT1_PDISC_PDIS7_Msk                                 (0x01UL << PORT1_PDISC_PDIS7_Pos)                         /*!< PORT1 PDISC: PDIS7 Mask             */
#define PORT1_PDISC_PDIS8_Pos                                 8                                                         /*!< PORT1 PDISC: PDIS8 Position         */
#define PORT1_PDISC_PDIS8_Msk                                 (0x01UL << PORT1_PDISC_PDIS8_Pos)                         /*!< PORT1 PDISC: PDIS8 Mask             */
#define PORT1_PDISC_PDIS9_Pos                                 9                                                         /*!< PORT1 PDISC: PDIS9 Position         */
#define PORT1_PDISC_PDIS9_Msk                                 (0x01UL << PORT1_PDISC_PDIS9_Pos)                         /*!< PORT1 PDISC: PDIS9 Mask             */
#define PORT1_PDISC_PDIS10_Pos                                10                                                        /*!< PORT1 PDISC: PDIS10 Position        */
#define PORT1_PDISC_PDIS10_Msk                                (0x01UL << PORT1_PDISC_PDIS10_Pos)                        /*!< PORT1 PDISC: PDIS10 Mask            */
#define PORT1_PDISC_PDIS11_Pos                                11                                                        /*!< PORT1 PDISC: PDIS11 Position        */
#define PORT1_PDISC_PDIS11_Msk                                (0x01UL << PORT1_PDISC_PDIS11_Pos)                        /*!< PORT1 PDISC: PDIS11 Mask            */
#define PORT1_PDISC_PDIS12_Pos                                12                                                        /*!< PORT1 PDISC: PDIS12 Position        */
#define PORT1_PDISC_PDIS12_Msk                                (0x01UL << PORT1_PDISC_PDIS12_Pos)                        /*!< PORT1 PDISC: PDIS12 Mask            */
#define PORT1_PDISC_PDIS13_Pos                                13                                                        /*!< PORT1 PDISC: PDIS13 Position        */
#define PORT1_PDISC_PDIS13_Msk                                (0x01UL << PORT1_PDISC_PDIS13_Pos)                        /*!< PORT1 PDISC: PDIS13 Mask            */
#define PORT1_PDISC_PDIS14_Pos                                14                                                        /*!< PORT1 PDISC: PDIS14 Position        */
#define PORT1_PDISC_PDIS14_Msk                                (0x01UL << PORT1_PDISC_PDIS14_Pos)                        /*!< PORT1 PDISC: PDIS14 Mask            */
#define PORT1_PDISC_PDIS15_Pos                                15                                                        /*!< PORT1 PDISC: PDIS15 Position        */
#define PORT1_PDISC_PDIS15_Msk                                (0x01UL << PORT1_PDISC_PDIS15_Pos)                        /*!< PORT1 PDISC: PDIS15 Mask            */

// ----------------------------------------  PORT1_PPS  -------------------------------------------
#define PORT1_PPS_PPS0_Pos                                    0                                                         /*!< PORT1 PPS: PPS0 Position            */
#define PORT1_PPS_PPS0_Msk                                    (0x01UL << PORT1_PPS_PPS0_Pos)                            /*!< PORT1 PPS: PPS0 Mask                */
#define PORT1_PPS_PPS1_Pos                                    1                                                         /*!< PORT1 PPS: PPS1 Position            */
#define PORT1_PPS_PPS1_Msk                                    (0x01UL << PORT1_PPS_PPS1_Pos)                            /*!< PORT1 PPS: PPS1 Mask                */
#define PORT1_PPS_PPS2_Pos                                    2                                                         /*!< PORT1 PPS: PPS2 Position            */
#define PORT1_PPS_PPS2_Msk                                    (0x01UL << PORT1_PPS_PPS2_Pos)                            /*!< PORT1 PPS: PPS2 Mask                */
#define PORT1_PPS_PPS3_Pos                                    3                                                         /*!< PORT1 PPS: PPS3 Position            */
#define PORT1_PPS_PPS3_Msk                                    (0x01UL << PORT1_PPS_PPS3_Pos)                            /*!< PORT1 PPS: PPS3 Mask                */
#define PORT1_PPS_PPS4_Pos                                    4                                                         /*!< PORT1 PPS: PPS4 Position            */
#define PORT1_PPS_PPS4_Msk                                    (0x01UL << PORT1_PPS_PPS4_Pos)                            /*!< PORT1 PPS: PPS4 Mask                */
#define PORT1_PPS_PPS5_Pos                                    5                                                         /*!< PORT1 PPS: PPS5 Position            */
#define PORT1_PPS_PPS5_Msk                                    (0x01UL << PORT1_PPS_PPS5_Pos)                            /*!< PORT1 PPS: PPS5 Mask                */
#define PORT1_PPS_PPS6_Pos                                    6                                                         /*!< PORT1 PPS: PPS6 Position            */
#define PORT1_PPS_PPS6_Msk                                    (0x01UL << PORT1_PPS_PPS6_Pos)                            /*!< PORT1 PPS: PPS6 Mask                */
#define PORT1_PPS_PPS7_Pos                                    7                                                         /*!< PORT1 PPS: PPS7 Position            */
#define PORT1_PPS_PPS7_Msk                                    (0x01UL << PORT1_PPS_PPS7_Pos)                            /*!< PORT1 PPS: PPS7 Mask                */
#define PORT1_PPS_PPS8_Pos                                    8                                                         /*!< PORT1 PPS: PPS8 Position            */
#define PORT1_PPS_PPS8_Msk                                    (0x01UL << PORT1_PPS_PPS8_Pos)                            /*!< PORT1 PPS: PPS8 Mask                */
#define PORT1_PPS_PPS9_Pos                                    9                                                         /*!< PORT1 PPS: PPS9 Position            */
#define PORT1_PPS_PPS9_Msk                                    (0x01UL << PORT1_PPS_PPS9_Pos)                            /*!< PORT1 PPS: PPS9 Mask                */
#define PORT1_PPS_PPS10_Pos                                   10                                                        /*!< PORT1 PPS: PPS10 Position           */
#define PORT1_PPS_PPS10_Msk                                   (0x01UL << PORT1_PPS_PPS10_Pos)                           /*!< PORT1 PPS: PPS10 Mask               */
#define PORT1_PPS_PPS11_Pos                                   11                                                        /*!< PORT1 PPS: PPS11 Position           */
#define PORT1_PPS_PPS11_Msk                                   (0x01UL << PORT1_PPS_PPS11_Pos)                           /*!< PORT1 PPS: PPS11 Mask               */
#define PORT1_PPS_PPS12_Pos                                   12                                                        /*!< PORT1 PPS: PPS12 Position           */
#define PORT1_PPS_PPS12_Msk                                   (0x01UL << PORT1_PPS_PPS12_Pos)                           /*!< PORT1 PPS: PPS12 Mask               */
#define PORT1_PPS_PPS13_Pos                                   13                                                        /*!< PORT1 PPS: PPS13 Position           */
#define PORT1_PPS_PPS13_Msk                                   (0x01UL << PORT1_PPS_PPS13_Pos)                           /*!< PORT1 PPS: PPS13 Mask               */
#define PORT1_PPS_PPS14_Pos                                   14                                                        /*!< PORT1 PPS: PPS14 Position           */
#define PORT1_PPS_PPS14_Msk                                   (0x01UL << PORT1_PPS_PPS14_Pos)                           /*!< PORT1 PPS: PPS14 Mask               */
#define PORT1_PPS_PPS15_Pos                                   15                                                        /*!< PORT1 PPS: PPS15 Position           */
#define PORT1_PPS_PPS15_Msk                                   (0x01UL << PORT1_PPS_PPS15_Pos)                           /*!< PORT1 PPS: PPS15 Mask               */

// ---------------------------------------  PORT1_HWSEL  ------------------------------------------
#define PORT1_HWSEL_HW0_Pos                                   0                                                         /*!< PORT1 HWSEL: HW0 Position           */
#define PORT1_HWSEL_HW0_Msk                                   (0x03UL << PORT1_HWSEL_HW0_Pos)                           /*!< PORT1 HWSEL: HW0 Mask               */
#define PORT1_HWSEL_HW1_Pos                                   2                                                         /*!< PORT1 HWSEL: HW1 Position           */
#define PORT1_HWSEL_HW1_Msk                                   (0x03UL << PORT1_HWSEL_HW1_Pos)                           /*!< PORT1 HWSEL: HW1 Mask               */
#define PORT1_HWSEL_HW2_Pos                                   4                                                         /*!< PORT1 HWSEL: HW2 Position           */
#define PORT1_HWSEL_HW2_Msk                                   (0x03UL << PORT1_HWSEL_HW2_Pos)                           /*!< PORT1 HWSEL: HW2 Mask               */
#define PORT1_HWSEL_HW3_Pos                                   6                                                         /*!< PORT1 HWSEL: HW3 Position           */
#define PORT1_HWSEL_HW3_Msk                                   (0x03UL << PORT1_HWSEL_HW3_Pos)                           /*!< PORT1 HWSEL: HW3 Mask               */
#define PORT1_HWSEL_HW4_Pos                                   8                                                         /*!< PORT1 HWSEL: HW4 Position           */
#define PORT1_HWSEL_HW4_Msk                                   (0x03UL << PORT1_HWSEL_HW4_Pos)                           /*!< PORT1 HWSEL: HW4 Mask               */
#define PORT1_HWSEL_HW5_Pos                                   10                                                        /*!< PORT1 HWSEL: HW5 Position           */
#define PORT1_HWSEL_HW5_Msk                                   (0x03UL << PORT1_HWSEL_HW5_Pos)                           /*!< PORT1 HWSEL: HW5 Mask               */
#define PORT1_HWSEL_HW6_Pos                                   12                                                        /*!< PORT1 HWSEL: HW6 Position           */
#define PORT1_HWSEL_HW6_Msk                                   (0x03UL << PORT1_HWSEL_HW6_Pos)                           /*!< PORT1 HWSEL: HW6 Mask               */
#define PORT1_HWSEL_HW7_Pos                                   14                                                        /*!< PORT1 HWSEL: HW7 Position           */
#define PORT1_HWSEL_HW7_Msk                                   (0x03UL << PORT1_HWSEL_HW7_Pos)                           /*!< PORT1 HWSEL: HW7 Mask               */
#define PORT1_HWSEL_HW8_Pos                                   16                                                        /*!< PORT1 HWSEL: HW8 Position           */
#define PORT1_HWSEL_HW8_Msk                                   (0x03UL << PORT1_HWSEL_HW8_Pos)                           /*!< PORT1 HWSEL: HW8 Mask               */
#define PORT1_HWSEL_HW9_Pos                                   18                                                        /*!< PORT1 HWSEL: HW9 Position           */
#define PORT1_HWSEL_HW9_Msk                                   (0x03UL << PORT1_HWSEL_HW9_Pos)                           /*!< PORT1 HWSEL: HW9 Mask               */
#define PORT1_HWSEL_HW10_Pos                                  20                                                        /*!< PORT1 HWSEL: HW10 Position          */
#define PORT1_HWSEL_HW10_Msk                                  (0x03UL << PORT1_HWSEL_HW10_Pos)                          /*!< PORT1 HWSEL: HW10 Mask              */
#define PORT1_HWSEL_HW11_Pos                                  22                                                        /*!< PORT1 HWSEL: HW11 Position          */
#define PORT1_HWSEL_HW11_Msk                                  (0x03UL << PORT1_HWSEL_HW11_Pos)                          /*!< PORT1 HWSEL: HW11 Mask              */
#define PORT1_HWSEL_HW12_Pos                                  24                                                        /*!< PORT1 HWSEL: HW12 Position          */
#define PORT1_HWSEL_HW12_Msk                                  (0x03UL << PORT1_HWSEL_HW12_Pos)                          /*!< PORT1 HWSEL: HW12 Mask              */
#define PORT1_HWSEL_HW13_Pos                                  26                                                        /*!< PORT1 HWSEL: HW13 Position          */
#define PORT1_HWSEL_HW13_Msk                                  (0x03UL << PORT1_HWSEL_HW13_Pos)                          /*!< PORT1 HWSEL: HW13 Mask              */
#define PORT1_HWSEL_HW14_Pos                                  28                                                        /*!< PORT1 HWSEL: HW14 Position          */
#define PORT1_HWSEL_HW14_Msk                                  (0x03UL << PORT1_HWSEL_HW14_Pos)                          /*!< PORT1 HWSEL: HW14 Mask              */
#define PORT1_HWSEL_HW15_Pos                                  30                                                        /*!< PORT1 HWSEL: HW15 Position          */
#define PORT1_HWSEL_HW15_Msk                                  (0x03UL << PORT1_HWSEL_HW15_Pos)                          /*!< PORT1 HWSEL: HW15 Mask              */


// ------------------------------------------------------------------------------------------------
// -----                                 PORT2 Position & Mask                                -----
// ------------------------------------------------------------------------------------------------


// ----------------------------------------  PORT2_OUT  -------------------------------------------
#define PORT2_OUT_P0_Pos                                      0                                                         /*!< PORT2 OUT: P0 Position              */
#define PORT2_OUT_P0_Msk                                      (0x01UL << PORT2_OUT_P0_Pos)                              /*!< PORT2 OUT: P0 Mask                  */
#define PORT2_OUT_P1_Pos                                      1                                                         /*!< PORT2 OUT: P1 Position              */
#define PORT2_OUT_P1_Msk                                      (0x01UL << PORT2_OUT_P1_Pos)                              /*!< PORT2 OUT: P1 Mask                  */
#define PORT2_OUT_P2_Pos                                      2                                                         /*!< PORT2 OUT: P2 Position              */
#define PORT2_OUT_P2_Msk                                      (0x01UL << PORT2_OUT_P2_Pos)                              /*!< PORT2 OUT: P2 Mask                  */
#define PORT2_OUT_P3_Pos                                      3                                                         /*!< PORT2 OUT: P3 Position              */
#define PORT2_OUT_P3_Msk                                      (0x01UL << PORT2_OUT_P3_Pos)                              /*!< PORT2 OUT: P3 Mask                  */
#define PORT2_OUT_P4_Pos                                      4                                                         /*!< PORT2 OUT: P4 Position              */
#define PORT2_OUT_P4_Msk                                      (0x01UL << PORT2_OUT_P4_Pos)                              /*!< PORT2 OUT: P4 Mask                  */
#define PORT2_OUT_P5_Pos                                      5                                                         /*!< PORT2 OUT: P5 Position              */
#define PORT2_OUT_P5_Msk                                      (0x01UL << PORT2_OUT_P5_Pos)                              /*!< PORT2 OUT: P5 Mask                  */
#define PORT2_OUT_P6_Pos                                      6                                                         /*!< PORT2 OUT: P6 Position              */
#define PORT2_OUT_P6_Msk                                      (0x01UL << PORT2_OUT_P6_Pos)                              /*!< PORT2 OUT: P6 Mask                  */
#define PORT2_OUT_P7_Pos                                      7                                                         /*!< PORT2 OUT: P7 Position              */
#define PORT2_OUT_P7_Msk                                      (0x01UL << PORT2_OUT_P7_Pos)                              /*!< PORT2 OUT: P7 Mask                  */
#define PORT2_OUT_P8_Pos                                      8                                                         /*!< PORT2 OUT: P8 Position              */
#define PORT2_OUT_P8_Msk                                      (0x01UL << PORT2_OUT_P8_Pos)                              /*!< PORT2 OUT: P8 Mask                  */
#define PORT2_OUT_P9_Pos                                      9                                                         /*!< PORT2 OUT: P9 Position              */
#define PORT2_OUT_P9_Msk                                      (0x01UL << PORT2_OUT_P9_Pos)                              /*!< PORT2 OUT: P9 Mask                  */
#define PORT2_OUT_P10_Pos                                     10                                                        /*!< PORT2 OUT: P10 Position             */
#define PORT2_OUT_P10_Msk                                     (0x01UL << PORT2_OUT_P10_Pos)                             /*!< PORT2 OUT: P10 Mask                 */
#define PORT2_OUT_P11_Pos                                     11                                                        /*!< PORT2 OUT: P11 Position             */
#define PORT2_OUT_P11_Msk                                     (0x01UL << PORT2_OUT_P11_Pos)                             /*!< PORT2 OUT: P11 Mask                 */
#define PORT2_OUT_P12_Pos                                     12                                                        /*!< PORT2 OUT: P12 Position             */
#define PORT2_OUT_P12_Msk                                     (0x01UL << PORT2_OUT_P12_Pos)                             /*!< PORT2 OUT: P12 Mask                 */
#define PORT2_OUT_P13_Pos                                     13                                                        /*!< PORT2 OUT: P13 Position             */
#define PORT2_OUT_P13_Msk                                     (0x01UL << PORT2_OUT_P13_Pos)                             /*!< PORT2 OUT: P13 Mask                 */
#define PORT2_OUT_P14_Pos                                     14                                                        /*!< PORT2 OUT: P14 Position             */
#define PORT2_OUT_P14_Msk                                     (0x01UL << PORT2_OUT_P14_Pos)                             /*!< PORT2 OUT: P14 Mask                 */
#define PORT2_OUT_P15_Pos                                     15                                                        /*!< PORT2 OUT: P15 Position             */
#define PORT2_OUT_P15_Msk                                     (0x01UL << PORT2_OUT_P15_Pos)                             /*!< PORT2 OUT: P15 Mask                 */

// ----------------------------------------  PORT2_OMR  -------------------------------------------
#define PORT2_OMR_PS0_Pos                                     0                                                         /*!< PORT2 OMR: PS0 Position             */
#define PORT2_OMR_PS0_Msk                                     (0x01UL << PORT2_OMR_PS0_Pos)                             /*!< PORT2 OMR: PS0 Mask                 */
#define PORT2_OMR_PS1_Pos                                     1                                                         /*!< PORT2 OMR: PS1 Position             */
#define PORT2_OMR_PS1_Msk                                     (0x01UL << PORT2_OMR_PS1_Pos)                             /*!< PORT2 OMR: PS1 Mask                 */
#define PORT2_OMR_PS2_Pos                                     2                                                         /*!< PORT2 OMR: PS2 Position             */
#define PORT2_OMR_PS2_Msk                                     (0x01UL << PORT2_OMR_PS2_Pos)                             /*!< PORT2 OMR: PS2 Mask                 */
#define PORT2_OMR_PS3_Pos                                     3                                                         /*!< PORT2 OMR: PS3 Position             */
#define PORT2_OMR_PS3_Msk                                     (0x01UL << PORT2_OMR_PS3_Pos)                             /*!< PORT2 OMR: PS3 Mask                 */
#define PORT2_OMR_PS4_Pos                                     4                                                         /*!< PORT2 OMR: PS4 Position             */
#define PORT2_OMR_PS4_Msk                                     (0x01UL << PORT2_OMR_PS4_Pos)                             /*!< PORT2 OMR: PS4 Mask                 */
#define PORT2_OMR_PS5_Pos                                     5                                                         /*!< PORT2 OMR: PS5 Position             */
#define PORT2_OMR_PS5_Msk                                     (0x01UL << PORT2_OMR_PS5_Pos)                             /*!< PORT2 OMR: PS5 Mask                 */
#define PORT2_OMR_PS6_Pos                                     6                                                         /*!< PORT2 OMR: PS6 Position             */
#define PORT2_OMR_PS6_Msk                                     (0x01UL << PORT2_OMR_PS6_Pos)                             /*!< PORT2 OMR: PS6 Mask                 */
#define PORT2_OMR_PS7_Pos                                     7                                                         /*!< PORT2 OMR: PS7 Position             */
#define PORT2_OMR_PS7_Msk                                     (0x01UL << PORT2_OMR_PS7_Pos)                             /*!< PORT2 OMR: PS7 Mask                 */
#define PORT2_OMR_PS8_Pos                                     8                                                         /*!< PORT2 OMR: PS8 Position             */
#define PORT2_OMR_PS8_Msk                                     (0x01UL << PORT2_OMR_PS8_Pos)                             /*!< PORT2 OMR: PS8 Mask                 */
#define PORT2_OMR_PS9_Pos                                     9                                                         /*!< PORT2 OMR: PS9 Position             */
#define PORT2_OMR_PS9_Msk                                     (0x01UL << PORT2_OMR_PS9_Pos)                             /*!< PORT2 OMR: PS9 Mask                 */
#define PORT2_OMR_PS10_Pos                                    10                                                        /*!< PORT2 OMR: PS10 Position            */
#define PORT2_OMR_PS10_Msk                                    (0x01UL << PORT2_OMR_PS10_Pos)                            /*!< PORT2 OMR: PS10 Mask                */
#define PORT2_OMR_PS11_Pos                                    11                                                        /*!< PORT2 OMR: PS11 Position            */
#define PORT2_OMR_PS11_Msk                                    (0x01UL << PORT2_OMR_PS11_Pos)                            /*!< PORT2 OMR: PS11 Mask                */
#define PORT2_OMR_PS12_Pos                                    12                                                        /*!< PORT2 OMR: PS12 Position            */
#define PORT2_OMR_PS12_Msk                                    (0x01UL << PORT2_OMR_PS12_Pos)                            /*!< PORT2 OMR: PS12 Mask                */
#define PORT2_OMR_PS13_Pos                                    13                                                        /*!< PORT2 OMR: PS13 Position            */
#define PORT2_OMR_PS13_Msk                                    (0x01UL << PORT2_OMR_PS13_Pos)                            /*!< PORT2 OMR: PS13 Mask                */
#define PORT2_OMR_PS14_Pos                                    14                                                        /*!< PORT2 OMR: PS14 Position            */
#define PORT2_OMR_PS14_Msk                                    (0x01UL << PORT2_OMR_PS14_Pos)                            /*!< PORT2 OMR: PS14 Mask                */
#define PORT2_OMR_PS15_Pos                                    15                                                        /*!< PORT2 OMR: PS15 Position            */
#define PORT2_OMR_PS15_Msk                                    (0x01UL << PORT2_OMR_PS15_Pos)                            /*!< PORT2 OMR: PS15 Mask                */
#define PORT2_OMR_PR0_Pos                                     16                                                        /*!< PORT2 OMR: PR0 Position             */
#define PORT2_OMR_PR0_Msk                                     (0x01UL << PORT2_OMR_PR0_Pos)                             /*!< PORT2 OMR: PR0 Mask                 */
#define PORT2_OMR_PR1_Pos                                     17                                                        /*!< PORT2 OMR: PR1 Position             */
#define PORT2_OMR_PR1_Msk                                     (0x01UL << PORT2_OMR_PR1_Pos)                             /*!< PORT2 OMR: PR1 Mask                 */
#define PORT2_OMR_PR2_Pos                                     18                                                        /*!< PORT2 OMR: PR2 Position             */
#define PORT2_OMR_PR2_Msk                                     (0x01UL << PORT2_OMR_PR2_Pos)                             /*!< PORT2 OMR: PR2 Mask                 */
#define PORT2_OMR_PR3_Pos                                     19                                                        /*!< PORT2 OMR: PR3 Position             */
#define PORT2_OMR_PR3_Msk                                     (0x01UL << PORT2_OMR_PR3_Pos)                             /*!< PORT2 OMR: PR3 Mask                 */
#define PORT2_OMR_PR4_Pos                                     20                                                        /*!< PORT2 OMR: PR4 Position             */
#define PORT2_OMR_PR4_Msk                                     (0x01UL << PORT2_OMR_PR4_Pos)                             /*!< PORT2 OMR: PR4 Mask                 */
#define PORT2_OMR_PR5_Pos                                     21                                                        /*!< PORT2 OMR: PR5 Position             */
#define PORT2_OMR_PR5_Msk                                     (0x01UL << PORT2_OMR_PR5_Pos)                             /*!< PORT2 OMR: PR5 Mask                 */
#define PORT2_OMR_PR6_Pos                                     22                                                        /*!< PORT2 OMR: PR6 Position             */
#define PORT2_OMR_PR6_Msk                                     (0x01UL << PORT2_OMR_PR6_Pos)                             /*!< PORT2 OMR: PR6 Mask                 */
#define PORT2_OMR_PR7_Pos                                     23                                                        /*!< PORT2 OMR: PR7 Position             */
#define PORT2_OMR_PR7_Msk                                     (0x01UL << PORT2_OMR_PR7_Pos)                             /*!< PORT2 OMR: PR7 Mask                 */
#define PORT2_OMR_PR8_Pos                                     24                                                        /*!< PORT2 OMR: PR8 Position             */
#define PORT2_OMR_PR8_Msk                                     (0x01UL << PORT2_OMR_PR8_Pos)                             /*!< PORT2 OMR: PR8 Mask                 */
#define PORT2_OMR_PR9_Pos                                     25                                                        /*!< PORT2 OMR: PR9 Position             */
#define PORT2_OMR_PR9_Msk                                     (0x01UL << PORT2_OMR_PR9_Pos)                             /*!< PORT2 OMR: PR9 Mask                 */
#define PORT2_OMR_PR10_Pos                                    26                                                        /*!< PORT2 OMR: PR10 Position            */
#define PORT2_OMR_PR10_Msk                                    (0x01UL << PORT2_OMR_PR10_Pos)                            /*!< PORT2 OMR: PR10 Mask                */
#define PORT2_OMR_PR11_Pos                                    27                                                        /*!< PORT2 OMR: PR11 Position            */
#define PORT2_OMR_PR11_Msk                                    (0x01UL << PORT2_OMR_PR11_Pos)                            /*!< PORT2 OMR: PR11 Mask                */
#define PORT2_OMR_PR12_Pos                                    28                                                        /*!< PORT2 OMR: PR12 Position            */
#define PORT2_OMR_PR12_Msk                                    (0x01UL << PORT2_OMR_PR12_Pos)                            /*!< PORT2 OMR: PR12 Mask                */
#define PORT2_OMR_PR13_Pos                                    29                                                        /*!< PORT2 OMR: PR13 Position            */
#define PORT2_OMR_PR13_Msk                                    (0x01UL << PORT2_OMR_PR13_Pos)                            /*!< PORT2 OMR: PR13 Mask                */
#define PORT2_OMR_PR14_Pos                                    30                                                        /*!< PORT2 OMR: PR14 Position            */
#define PORT2_OMR_PR14_Msk                                    (0x01UL << PORT2_OMR_PR14_Pos)                            /*!< PORT2 OMR: PR14 Mask                */
#define PORT2_OMR_PR15_Pos                                    31                                                        /*!< PORT2 OMR: PR15 Position            */
#define PORT2_OMR_PR15_Msk                                    (0x01UL << PORT2_OMR_PR15_Pos)                            /*!< PORT2 OMR: PR15 Mask                */

// ---------------------------------------  PORT2_IOCR0  ------------------------------------------
#define PORT2_IOCR0_PC0_Pos                                   3                                                         /*!< PORT2 IOCR0: PC0 Position           */
#define PORT2_IOCR0_PC0_Msk                                   (0x1fUL << PORT2_IOCR0_PC0_Pos)                           /*!< PORT2 IOCR0: PC0 Mask               */
#define PORT2_IOCR0_PC1_Pos                                   11                                                        /*!< PORT2 IOCR0: PC1 Position           */
#define PORT2_IOCR0_PC1_Msk                                   (0x1fUL << PORT2_IOCR0_PC1_Pos)                           /*!< PORT2 IOCR0: PC1 Mask               */
#define PORT2_IOCR0_PC2_Pos                                   19                                                        /*!< PORT2 IOCR0: PC2 Position           */
#define PORT2_IOCR0_PC2_Msk                                   (0x1fUL << PORT2_IOCR0_PC2_Pos)                           /*!< PORT2 IOCR0: PC2 Mask               */
#define PORT2_IOCR0_PC3_Pos                                   27                                                        /*!< PORT2 IOCR0: PC3 Position           */
#define PORT2_IOCR0_PC3_Msk                                   (0x1fUL << PORT2_IOCR0_PC3_Pos)                           /*!< PORT2 IOCR0: PC3 Mask               */

// ---------------------------------------  PORT2_IOCR4  ------------------------------------------
#define PORT2_IOCR4_PC4_Pos                                   3                                                         /*!< PORT2 IOCR4: PC4 Position           */
#define PORT2_IOCR4_PC4_Msk                                   (0x1fUL << PORT2_IOCR4_PC4_Pos)                           /*!< PORT2 IOCR4: PC4 Mask               */
#define PORT2_IOCR4_PC5_Pos                                   11                                                        /*!< PORT2 IOCR4: PC5 Position           */
#define PORT2_IOCR4_PC5_Msk                                   (0x1fUL << PORT2_IOCR4_PC5_Pos)                           /*!< PORT2 IOCR4: PC5 Mask               */
#define PORT2_IOCR4_PC6_Pos                                   19                                                        /*!< PORT2 IOCR4: PC6 Position           */
#define PORT2_IOCR4_PC6_Msk                                   (0x1fUL << PORT2_IOCR4_PC6_Pos)                           /*!< PORT2 IOCR4: PC6 Mask               */
#define PORT2_IOCR4_PC7_Pos                                   27                                                        /*!< PORT2 IOCR4: PC7 Position           */
#define PORT2_IOCR4_PC7_Msk                                   (0x1fUL << PORT2_IOCR4_PC7_Pos)                           /*!< PORT2 IOCR4: PC7 Mask               */

// ---------------------------------------  PORT2_IOCR8  ------------------------------------------
#define PORT2_IOCR8_PC8_Pos                                   3                                                         /*!< PORT2 IOCR8: PC8 Position           */
#define PORT2_IOCR8_PC8_Msk                                   (0x1fUL << PORT2_IOCR8_PC8_Pos)                           /*!< PORT2 IOCR8: PC8 Mask               */
#define PORT2_IOCR8_PC9_Pos                                   11                                                        /*!< PORT2 IOCR8: PC9 Position           */
#define PORT2_IOCR8_PC9_Msk                                   (0x1fUL << PORT2_IOCR8_PC9_Pos)                           /*!< PORT2 IOCR8: PC9 Mask               */
#define PORT2_IOCR8_PC10_Pos                                  19                                                        /*!< PORT2 IOCR8: PC10 Position          */
#define PORT2_IOCR8_PC10_Msk                                  (0x1fUL << PORT2_IOCR8_PC10_Pos)                          /*!< PORT2 IOCR8: PC10 Mask              */
#define PORT2_IOCR8_PC11_Pos                                  27                                                        /*!< PORT2 IOCR8: PC11 Position          */
#define PORT2_IOCR8_PC11_Msk                                  (0x1fUL << PORT2_IOCR8_PC11_Pos)                          /*!< PORT2 IOCR8: PC11 Mask              */

// --------------------------------------  PORT2_IOCR12  ------------------------------------------
#define PORT2_IOCR12_PC12_Pos                                 3                                                         /*!< PORT2 IOCR12: PC12 Position         */
#define PORT2_IOCR12_PC12_Msk                                 (0x1fUL << PORT2_IOCR12_PC12_Pos)                         /*!< PORT2 IOCR12: PC12 Mask             */
#define PORT2_IOCR12_PC13_Pos                                 11                                                        /*!< PORT2 IOCR12: PC13 Position         */
#define PORT2_IOCR12_PC13_Msk                                 (0x1fUL << PORT2_IOCR12_PC13_Pos)                         /*!< PORT2 IOCR12: PC13 Mask             */
#define PORT2_IOCR12_PC14_Pos                                 19                                                        /*!< PORT2 IOCR12: PC14 Position         */
#define PORT2_IOCR12_PC14_Msk                                 (0x1fUL << PORT2_IOCR12_PC14_Pos)                         /*!< PORT2 IOCR12: PC14 Mask             */
#define PORT2_IOCR12_PC15_Pos                                 27                                                        /*!< PORT2 IOCR12: PC15 Position         */
#define PORT2_IOCR12_PC15_Msk                                 (0x1fUL << PORT2_IOCR12_PC15_Pos)                         /*!< PORT2 IOCR12: PC15 Mask             */

// ----------------------------------------  PORT2_IN  --------------------------------------------
#define PORT2_IN_P0_Pos                                       0                                                         /*!< PORT2 IN: P0 Position               */
#define PORT2_IN_P0_Msk                                       (0x01UL << PORT2_IN_P0_Pos)                               /*!< PORT2 IN: P0 Mask                   */
#define PORT2_IN_P1_Pos                                       1                                                         /*!< PORT2 IN: P1 Position               */
#define PORT2_IN_P1_Msk                                       (0x01UL << PORT2_IN_P1_Pos)                               /*!< PORT2 IN: P1 Mask                   */
#define PORT2_IN_P2_Pos                                       2                                                         /*!< PORT2 IN: P2 Position               */
#define PORT2_IN_P2_Msk                                       (0x01UL << PORT2_IN_P2_Pos)                               /*!< PORT2 IN: P2 Mask                   */
#define PORT2_IN_P3_Pos                                       3                                                         /*!< PORT2 IN: P3 Position               */
#define PORT2_IN_P3_Msk                                       (0x01UL << PORT2_IN_P3_Pos)                               /*!< PORT2 IN: P3 Mask                   */
#define PORT2_IN_P4_Pos                                       4                                                         /*!< PORT2 IN: P4 Position               */
#define PORT2_IN_P4_Msk                                       (0x01UL << PORT2_IN_P4_Pos)                               /*!< PORT2 IN: P4 Mask                   */
#define PORT2_IN_P5_Pos                                       5                                                         /*!< PORT2 IN: P5 Position               */
#define PORT2_IN_P5_Msk                                       (0x01UL << PORT2_IN_P5_Pos)                               /*!< PORT2 IN: P5 Mask                   */
#define PORT2_IN_P6_Pos                                       6                                                         /*!< PORT2 IN: P6 Position               */
#define PORT2_IN_P6_Msk                                       (0x01UL << PORT2_IN_P6_Pos)                               /*!< PORT2 IN: P6 Mask                   */
#define PORT2_IN_P7_Pos                                       7                                                         /*!< PORT2 IN: P7 Position               */
#define PORT2_IN_P7_Msk                                       (0x01UL << PORT2_IN_P7_Pos)                               /*!< PORT2 IN: P7 Mask                   */
#define PORT2_IN_P8_Pos                                       8                                                         /*!< PORT2 IN: P8 Position               */
#define PORT2_IN_P8_Msk                                       (0x01UL << PORT2_IN_P8_Pos)                               /*!< PORT2 IN: P8 Mask                   */
#define PORT2_IN_P9_Pos                                       9                                                         /*!< PORT2 IN: P9 Position               */
#define PORT2_IN_P9_Msk                                       (0x01UL << PORT2_IN_P9_Pos)                               /*!< PORT2 IN: P9 Mask                   */
#define PORT2_IN_P10_Pos                                      10                                                        /*!< PORT2 IN: P10 Position              */
#define PORT2_IN_P10_Msk                                      (0x01UL << PORT2_IN_P10_Pos)                              /*!< PORT2 IN: P10 Mask                  */
#define PORT2_IN_P11_Pos                                      11                                                        /*!< PORT2 IN: P11 Position              */
#define PORT2_IN_P11_Msk                                      (0x01UL << PORT2_IN_P11_Pos)                              /*!< PORT2 IN: P11 Mask                  */
#define PORT2_IN_P12_Pos                                      12                                                        /*!< PORT2 IN: P12 Position              */
#define PORT2_IN_P12_Msk                                      (0x01UL << PORT2_IN_P12_Pos)                              /*!< PORT2 IN: P12 Mask                  */
#define PORT2_IN_P13_Pos                                      13                                                        /*!< PORT2 IN: P13 Position              */
#define PORT2_IN_P13_Msk                                      (0x01UL << PORT2_IN_P13_Pos)                              /*!< PORT2 IN: P13 Mask                  */
#define PORT2_IN_P14_Pos                                      14                                                        /*!< PORT2 IN: P14 Position              */
#define PORT2_IN_P14_Msk                                      (0x01UL << PORT2_IN_P14_Pos)                              /*!< PORT2 IN: P14 Mask                  */
#define PORT2_IN_P15_Pos                                      15                                                        /*!< PORT2 IN: P15 Position              */
#define PORT2_IN_P15_Msk                                      (0x01UL << PORT2_IN_P15_Pos)                              /*!< PORT2 IN: P15 Mask                  */

// ---------------------------------------  PORT2_PDR0  -------------------------------------------
#define PORT2_PDR0_PD0_Pos                                    0                                                         /*!< PORT2 PDR0: PD0 Position            */
#define PORT2_PDR0_PD0_Msk                                    (0x07UL << PORT2_PDR0_PD0_Pos)                            /*!< PORT2 PDR0: PD0 Mask                */
#define PORT2_PDR0_PD1_Pos                                    4                                                         /*!< PORT2 PDR0: PD1 Position            */
#define PORT2_PDR0_PD1_Msk                                    (0x07UL << PORT2_PDR0_PD1_Pos)                            /*!< PORT2 PDR0: PD1 Mask                */
#define PORT2_PDR0_PD2_Pos                                    8                                                         /*!< PORT2 PDR0: PD2 Position            */
#define PORT2_PDR0_PD2_Msk                                    (0x07UL << PORT2_PDR0_PD2_Pos)                            /*!< PORT2 PDR0: PD2 Mask                */
#define PORT2_PDR0_PD3_Pos                                    12                                                        /*!< PORT2 PDR0: PD3 Position            */
#define PORT2_PDR0_PD3_Msk                                    (0x07UL << PORT2_PDR0_PD3_Pos)                            /*!< PORT2 PDR0: PD3 Mask                */
#define PORT2_PDR0_PD4_Pos                                    16                                                        /*!< PORT2 PDR0: PD4 Position            */
#define PORT2_PDR0_PD4_Msk                                    (0x07UL << PORT2_PDR0_PD4_Pos)                            /*!< PORT2 PDR0: PD4 Mask                */
#define PORT2_PDR0_PD5_Pos                                    20                                                        /*!< PORT2 PDR0: PD5 Position            */
#define PORT2_PDR0_PD5_Msk                                    (0x07UL << PORT2_PDR0_PD5_Pos)                            /*!< PORT2 PDR0: PD5 Mask                */
#define PORT2_PDR0_PD6_Pos                                    24                                                        /*!< PORT2 PDR0: PD6 Position            */
#define PORT2_PDR0_PD6_Msk                                    (0x07UL << PORT2_PDR0_PD6_Pos)                            /*!< PORT2 PDR0: PD6 Mask                */
#define PORT2_PDR0_PD7_Pos                                    28                                                        /*!< PORT2 PDR0: PD7 Position            */
#define PORT2_PDR0_PD7_Msk                                    (0x07UL << PORT2_PDR0_PD7_Pos)                            /*!< PORT2 PDR0: PD7 Mask                */

// ---------------------------------------  PORT2_PDR1  -------------------------------------------
#define PORT2_PDR1_PD8_Pos                                    0                                                         /*!< PORT2 PDR1: PD8 Position            */
#define PORT2_PDR1_PD8_Msk                                    (0x07UL << PORT2_PDR1_PD8_Pos)                            /*!< PORT2 PDR1: PD8 Mask                */
#define PORT2_PDR1_PD9_Pos                                    4                                                         /*!< PORT2 PDR1: PD9 Position            */
#define PORT2_PDR1_PD9_Msk                                    (0x07UL << PORT2_PDR1_PD9_Pos)                            /*!< PORT2 PDR1: PD9 Mask                */
#define PORT2_PDR1_PD10_Pos                                   8                                                         /*!< PORT2 PDR1: PD10 Position           */
#define PORT2_PDR1_PD10_Msk                                   (0x07UL << PORT2_PDR1_PD10_Pos)                           /*!< PORT2 PDR1: PD10 Mask               */
#define PORT2_PDR1_PD11_Pos                                   12                                                        /*!< PORT2 PDR1: PD11 Position           */
#define PORT2_PDR1_PD11_Msk                                   (0x07UL << PORT2_PDR1_PD11_Pos)                           /*!< PORT2 PDR1: PD11 Mask               */
#define PORT2_PDR1_PD12_Pos                                   16                                                        /*!< PORT2 PDR1: PD12 Position           */
#define PORT2_PDR1_PD12_Msk                                   (0x07UL << PORT2_PDR1_PD12_Pos)                           /*!< PORT2 PDR1: PD12 Mask               */
#define PORT2_PDR1_PD13_Pos                                   20                                                        /*!< PORT2 PDR1: PD13 Position           */
#define PORT2_PDR1_PD13_Msk                                   (0x07UL << PORT2_PDR1_PD13_Pos)                           /*!< PORT2 PDR1: PD13 Mask               */
#define PORT2_PDR1_PD14_Pos                                   24                                                        /*!< PORT2 PDR1: PD14 Position           */
#define PORT2_PDR1_PD14_Msk                                   (0x07UL << PORT2_PDR1_PD14_Pos)                           /*!< PORT2 PDR1: PD14 Mask               */
#define PORT2_PDR1_PD15_Pos                                   28                                                        /*!< PORT2 PDR1: PD15 Position           */
#define PORT2_PDR1_PD15_Msk                                   (0x07UL << PORT2_PDR1_PD15_Pos)                           /*!< PORT2 PDR1: PD15 Mask               */

// ---------------------------------------  PORT2_PDISC  ------------------------------------------
#define PORT2_PDISC_PDIS0_Pos                                 0                                                         /*!< PORT2 PDISC: PDIS0 Position         */
#define PORT2_PDISC_PDIS0_Msk                                 (0x01UL << PORT2_PDISC_PDIS0_Pos)                         /*!< PORT2 PDISC: PDIS0 Mask             */
#define PORT2_PDISC_PDIS1_Pos                                 1                                                         /*!< PORT2 PDISC: PDIS1 Position         */
#define PORT2_PDISC_PDIS1_Msk                                 (0x01UL << PORT2_PDISC_PDIS1_Pos)                         /*!< PORT2 PDISC: PDIS1 Mask             */
#define PORT2_PDISC_PDIS2_Pos                                 2                                                         /*!< PORT2 PDISC: PDIS2 Position         */
#define PORT2_PDISC_PDIS2_Msk                                 (0x01UL << PORT2_PDISC_PDIS2_Pos)                         /*!< PORT2 PDISC: PDIS2 Mask             */
#define PORT2_PDISC_PDIS3_Pos                                 3                                                         /*!< PORT2 PDISC: PDIS3 Position         */
#define PORT2_PDISC_PDIS3_Msk                                 (0x01UL << PORT2_PDISC_PDIS3_Pos)                         /*!< PORT2 PDISC: PDIS3 Mask             */
#define PORT2_PDISC_PDIS4_Pos                                 4                                                         /*!< PORT2 PDISC: PDIS4 Position         */
#define PORT2_PDISC_PDIS4_Msk                                 (0x01UL << PORT2_PDISC_PDIS4_Pos)                         /*!< PORT2 PDISC: PDIS4 Mask             */
#define PORT2_PDISC_PDIS5_Pos                                 5                                                         /*!< PORT2 PDISC: PDIS5 Position         */
#define PORT2_PDISC_PDIS5_Msk                                 (0x01UL << PORT2_PDISC_PDIS5_Pos)                         /*!< PORT2 PDISC: PDIS5 Mask             */
#define PORT2_PDISC_PDIS6_Pos                                 6                                                         /*!< PORT2 PDISC: PDIS6 Position         */
#define PORT2_PDISC_PDIS6_Msk                                 (0x01UL << PORT2_PDISC_PDIS6_Pos)                         /*!< PORT2 PDISC: PDIS6 Mask             */
#define PORT2_PDISC_PDIS7_Pos                                 7                                                         /*!< PORT2 PDISC: PDIS7 Position         */
#define PORT2_PDISC_PDIS7_Msk                                 (0x01UL << PORT2_PDISC_PDIS7_Pos)                         /*!< PORT2 PDISC: PDIS7 Mask             */
#define PORT2_PDISC_PDIS8_Pos                                 8                                                         /*!< PORT2 PDISC: PDIS8 Position         */
#define PORT2_PDISC_PDIS8_Msk                                 (0x01UL << PORT2_PDISC_PDIS8_Pos)                         /*!< PORT2 PDISC: PDIS8 Mask             */
#define PORT2_PDISC_PDIS9_Pos                                 9                                                         /*!< PORT2 PDISC: PDIS9 Position         */
#define PORT2_PDISC_PDIS9_Msk                                 (0x01UL << PORT2_PDISC_PDIS9_Pos)                         /*!< PORT2 PDISC: PDIS9 Mask             */
#define PORT2_PDISC_PDIS10_Pos                                10                                                        /*!< PORT2 PDISC: PDIS10 Position        */
#define PORT2_PDISC_PDIS10_Msk                                (0x01UL << PORT2_PDISC_PDIS10_Pos)                        /*!< PORT2 PDISC: PDIS10 Mask            */
#define PORT2_PDISC_PDIS11_Pos                                11                                                        /*!< PORT2 PDISC: PDIS11 Position        */
#define PORT2_PDISC_PDIS11_Msk                                (0x01UL << PORT2_PDISC_PDIS11_Pos)                        /*!< PORT2 PDISC: PDIS11 Mask            */
#define PORT2_PDISC_PDIS12_Pos                                12                                                        /*!< PORT2 PDISC: PDIS12 Position        */
#define PORT2_PDISC_PDIS12_Msk                                (0x01UL << PORT2_PDISC_PDIS12_Pos)                        /*!< PORT2 PDISC: PDIS12 Mask            */
#define PORT2_PDISC_PDIS13_Pos                                13                                                        /*!< PORT2 PDISC: PDIS13 Position        */
#define PORT2_PDISC_PDIS13_Msk                                (0x01UL << PORT2_PDISC_PDIS13_Pos)                        /*!< PORT2 PDISC: PDIS13 Mask            */
#define PORT2_PDISC_PDIS14_Pos                                14                                                        /*!< PORT2 PDISC: PDIS14 Position        */
#define PORT2_PDISC_PDIS14_Msk                                (0x01UL << PORT2_PDISC_PDIS14_Pos)                        /*!< PORT2 PDISC: PDIS14 Mask            */
#define PORT2_PDISC_PDIS15_Pos                                15                                                        /*!< PORT2 PDISC: PDIS15 Position        */
#define PORT2_PDISC_PDIS15_Msk                                (0x01UL << PORT2_PDISC_PDIS15_Pos)                        /*!< PORT2 PDISC: PDIS15 Mask            */

// ----------------------------------------  PORT2_PPS  -------------------------------------------
#define PORT2_PPS_PPS0_Pos                                    0                                                         /*!< PORT2 PPS: PPS0 Position            */
#define PORT2_PPS_PPS0_Msk                                    (0x01UL << PORT2_PPS_PPS0_Pos)                            /*!< PORT2 PPS: PPS0 Mask                */
#define PORT2_PPS_PPS1_Pos                                    1                                                         /*!< PORT2 PPS: PPS1 Position            */
#define PORT2_PPS_PPS1_Msk                                    (0x01UL << PORT2_PPS_PPS1_Pos)                            /*!< PORT2 PPS: PPS1 Mask                */
#define PORT2_PPS_PPS2_Pos                                    2                                                         /*!< PORT2 PPS: PPS2 Position            */
#define PORT2_PPS_PPS2_Msk                                    (0x01UL << PORT2_PPS_PPS2_Pos)                            /*!< PORT2 PPS: PPS2 Mask                */
#define PORT2_PPS_PPS3_Pos                                    3                                                         /*!< PORT2 PPS: PPS3 Position            */
#define PORT2_PPS_PPS3_Msk                                    (0x01UL << PORT2_PPS_PPS3_Pos)                            /*!< PORT2 PPS: PPS3 Mask                */
#define PORT2_PPS_PPS4_Pos                                    4                                                         /*!< PORT2 PPS: PPS4 Position            */
#define PORT2_PPS_PPS4_Msk                                    (0x01UL << PORT2_PPS_PPS4_Pos)                            /*!< PORT2 PPS: PPS4 Mask                */
#define PORT2_PPS_PPS5_Pos                                    5                                                         /*!< PORT2 PPS: PPS5 Position            */
#define PORT2_PPS_PPS5_Msk                                    (0x01UL << PORT2_PPS_PPS5_Pos)                            /*!< PORT2 PPS: PPS5 Mask                */
#define PORT2_PPS_PPS6_Pos                                    6                                                         /*!< PORT2 PPS: PPS6 Position            */
#define PORT2_PPS_PPS6_Msk                                    (0x01UL << PORT2_PPS_PPS6_Pos)                            /*!< PORT2 PPS: PPS6 Mask                */
#define PORT2_PPS_PPS7_Pos                                    7                                                         /*!< PORT2 PPS: PPS7 Position            */
#define PORT2_PPS_PPS7_Msk                                    (0x01UL << PORT2_PPS_PPS7_Pos)                            /*!< PORT2 PPS: PPS7 Mask                */
#define PORT2_PPS_PPS8_Pos                                    8                                                         /*!< PORT2 PPS: PPS8 Position            */
#define PORT2_PPS_PPS8_Msk                                    (0x01UL << PORT2_PPS_PPS8_Pos)                            /*!< PORT2 PPS: PPS8 Mask                */
#define PORT2_PPS_PPS9_Pos                                    9                                                         /*!< PORT2 PPS: PPS9 Position            */
#define PORT2_PPS_PPS9_Msk                                    (0x01UL << PORT2_PPS_PPS9_Pos)                            /*!< PORT2 PPS: PPS9 Mask                */
#define PORT2_PPS_PPS10_Pos                                   10                                                        /*!< PORT2 PPS: PPS10 Position           */
#define PORT2_PPS_PPS10_Msk                                   (0x01UL << PORT2_PPS_PPS10_Pos)                           /*!< PORT2 PPS: PPS10 Mask               */
#define PORT2_PPS_PPS11_Pos                                   11                                                        /*!< PORT2 PPS: PPS11 Position           */
#define PORT2_PPS_PPS11_Msk                                   (0x01UL << PORT2_PPS_PPS11_Pos)                           /*!< PORT2 PPS: PPS11 Mask               */
#define PORT2_PPS_PPS12_Pos                                   12                                                        /*!< PORT2 PPS: PPS12 Position           */
#define PORT2_PPS_PPS12_Msk                                   (0x01UL << PORT2_PPS_PPS12_Pos)                           /*!< PORT2 PPS: PPS12 Mask               */
#define PORT2_PPS_PPS13_Pos                                   13                                                        /*!< PORT2 PPS: PPS13 Position           */
#define PORT2_PPS_PPS13_Msk                                   (0x01UL << PORT2_PPS_PPS13_Pos)                           /*!< PORT2 PPS: PPS13 Mask               */
#define PORT2_PPS_PPS14_Pos                                   14                                                        /*!< PORT2 PPS: PPS14 Position           */
#define PORT2_PPS_PPS14_Msk                                   (0x01UL << PORT2_PPS_PPS14_Pos)                           /*!< PORT2 PPS: PPS14 Mask               */
#define PORT2_PPS_PPS15_Pos                                   15                                                        /*!< PORT2 PPS: PPS15 Position           */
#define PORT2_PPS_PPS15_Msk                                   (0x01UL << PORT2_PPS_PPS15_Pos)                           /*!< PORT2 PPS: PPS15 Mask               */

// ---------------------------------------  PORT2_HWSEL  ------------------------------------------
#define PORT2_HWSEL_HW0_Pos                                   0                                                         /*!< PORT2 HWSEL: HW0 Position           */
#define PORT2_HWSEL_HW0_Msk                                   (0x03UL << PORT2_HWSEL_HW0_Pos)                           /*!< PORT2 HWSEL: HW0 Mask               */
#define PORT2_HWSEL_HW1_Pos                                   2                                                         /*!< PORT2 HWSEL: HW1 Position           */
#define PORT2_HWSEL_HW1_Msk                                   (0x03UL << PORT2_HWSEL_HW1_Pos)                           /*!< PORT2 HWSEL: HW1 Mask               */
#define PORT2_HWSEL_HW2_Pos                                   4                                                         /*!< PORT2 HWSEL: HW2 Position           */
#define PORT2_HWSEL_HW2_Msk                                   (0x03UL << PORT2_HWSEL_HW2_Pos)                           /*!< PORT2 HWSEL: HW2 Mask               */
#define PORT2_HWSEL_HW3_Pos                                   6                                                         /*!< PORT2 HWSEL: HW3 Position           */
#define PORT2_HWSEL_HW3_Msk                                   (0x03UL << PORT2_HWSEL_HW3_Pos)                           /*!< PORT2 HWSEL: HW3 Mask               */
#define PORT2_HWSEL_HW4_Pos                                   8                                                         /*!< PORT2 HWSEL: HW4 Position           */
#define PORT2_HWSEL_HW4_Msk                                   (0x03UL << PORT2_HWSEL_HW4_Pos)                           /*!< PORT2 HWSEL: HW4 Mask               */
#define PORT2_HWSEL_HW5_Pos                                   10                                                        /*!< PORT2 HWSEL: HW5 Position           */
#define PORT2_HWSEL_HW5_Msk                                   (0x03UL << PORT2_HWSEL_HW5_Pos)                           /*!< PORT2 HWSEL: HW5 Mask               */
#define PORT2_HWSEL_HW6_Pos                                   12                                                        /*!< PORT2 HWSEL: HW6 Position           */
#define PORT2_HWSEL_HW6_Msk                                   (0x03UL << PORT2_HWSEL_HW6_Pos)                           /*!< PORT2 HWSEL: HW6 Mask               */
#define PORT2_HWSEL_HW7_Pos                                   14                                                        /*!< PORT2 HWSEL: HW7 Position           */
#define PORT2_HWSEL_HW7_Msk                                   (0x03UL << PORT2_HWSEL_HW7_Pos)                           /*!< PORT2 HWSEL: HW7 Mask               */
#define PORT2_HWSEL_HW8_Pos                                   16                                                        /*!< PORT2 HWSEL: HW8 Position           */
#define PORT2_HWSEL_HW8_Msk                                   (0x03UL << PORT2_HWSEL_HW8_Pos)                           /*!< PORT2 HWSEL: HW8 Mask               */
#define PORT2_HWSEL_HW9_Pos                                   18                                                        /*!< PORT2 HWSEL: HW9 Position           */
#define PORT2_HWSEL_HW9_Msk                                   (0x03UL << PORT2_HWSEL_HW9_Pos)                           /*!< PORT2 HWSEL: HW9 Mask               */
#define PORT2_HWSEL_HW10_Pos                                  20                                                        /*!< PORT2 HWSEL: HW10 Position          */
#define PORT2_HWSEL_HW10_Msk                                  (0x03UL << PORT2_HWSEL_HW10_Pos)                          /*!< PORT2 HWSEL: HW10 Mask              */
#define PORT2_HWSEL_HW11_Pos                                  22                                                        /*!< PORT2 HWSEL: HW11 Position          */
#define PORT2_HWSEL_HW11_Msk                                  (0x03UL << PORT2_HWSEL_HW11_Pos)                          /*!< PORT2 HWSEL: HW11 Mask              */
#define PORT2_HWSEL_HW12_Pos                                  24                                                        /*!< PORT2 HWSEL: HW12 Position          */
#define PORT2_HWSEL_HW12_Msk                                  (0x03UL << PORT2_HWSEL_HW12_Pos)                          /*!< PORT2 HWSEL: HW12 Mask              */
#define PORT2_HWSEL_HW13_Pos                                  26                                                        /*!< PORT2 HWSEL: HW13 Position          */
#define PORT2_HWSEL_HW13_Msk                                  (0x03UL << PORT2_HWSEL_HW13_Pos)                          /*!< PORT2 HWSEL: HW13 Mask              */
#define PORT2_HWSEL_HW14_Pos                                  28                                                        /*!< PORT2 HWSEL: HW14 Position          */
#define PORT2_HWSEL_HW14_Msk                                  (0x03UL << PORT2_HWSEL_HW14_Pos)                          /*!< PORT2 HWSEL: HW14 Mask              */
#define PORT2_HWSEL_HW15_Pos                                  30                                                        /*!< PORT2 HWSEL: HW15 Position          */
#define PORT2_HWSEL_HW15_Msk                                  (0x03UL << PORT2_HWSEL_HW15_Pos)                          /*!< PORT2 HWSEL: HW15 Mask              */


// ------------------------------------------------------------------------------------------------
// -----                                 PORT3 Position & Mask                                -----
// ------------------------------------------------------------------------------------------------


// ----------------------------------------  PORT3_OUT  -------------------------------------------
#define PORT3_OUT_P0_Pos                                      0                                                         /*!< PORT3 OUT: P0 Position              */
#define PORT3_OUT_P0_Msk                                      (0x01UL << PORT3_OUT_P0_Pos)                              /*!< PORT3 OUT: P0 Mask                  */
#define PORT3_OUT_P1_Pos                                      1                                                         /*!< PORT3 OUT: P1 Position              */
#define PORT3_OUT_P1_Msk                                      (0x01UL << PORT3_OUT_P1_Pos)                              /*!< PORT3 OUT: P1 Mask                  */
#define PORT3_OUT_P2_Pos                                      2                                                         /*!< PORT3 OUT: P2 Position              */
#define PORT3_OUT_P2_Msk                                      (0x01UL << PORT3_OUT_P2_Pos)                              /*!< PORT3 OUT: P2 Mask                  */
#define PORT3_OUT_P3_Pos                                      3                                                         /*!< PORT3 OUT: P3 Position              */
#define PORT3_OUT_P3_Msk                                      (0x01UL << PORT3_OUT_P3_Pos)                              /*!< PORT3 OUT: P3 Mask                  */
#define PORT3_OUT_P4_Pos                                      4                                                         /*!< PORT3 OUT: P4 Position              */
#define PORT3_OUT_P4_Msk                                      (0x01UL << PORT3_OUT_P4_Pos)                              /*!< PORT3 OUT: P4 Mask                  */
#define PORT3_OUT_P5_Pos                                      5                                                         /*!< PORT3 OUT: P5 Position              */
#define PORT3_OUT_P5_Msk                                      (0x01UL << PORT3_OUT_P5_Pos)                              /*!< PORT3 OUT: P5 Mask                  */
#define PORT3_OUT_P6_Pos                                      6                                                         /*!< PORT3 OUT: P6 Position              */
#define PORT3_OUT_P6_Msk                                      (0x01UL << PORT3_OUT_P6_Pos)                              /*!< PORT3 OUT: P6 Mask                  */
#define PORT3_OUT_P7_Pos                                      7                                                         /*!< PORT3 OUT: P7 Position              */
#define PORT3_OUT_P7_Msk                                      (0x01UL << PORT3_OUT_P7_Pos)                              /*!< PORT3 OUT: P7 Mask                  */
#define PORT3_OUT_P8_Pos                                      8                                                         /*!< PORT3 OUT: P8 Position              */
#define PORT3_OUT_P8_Msk                                      (0x01UL << PORT3_OUT_P8_Pos)                              /*!< PORT3 OUT: P8 Mask                  */
#define PORT3_OUT_P9_Pos                                      9                                                         /*!< PORT3 OUT: P9 Position              */
#define PORT3_OUT_P9_Msk                                      (0x01UL << PORT3_OUT_P9_Pos)                              /*!< PORT3 OUT: P9 Mask                  */
#define PORT3_OUT_P10_Pos                                     10                                                        /*!< PORT3 OUT: P10 Position             */
#define PORT3_OUT_P10_Msk                                     (0x01UL << PORT3_OUT_P10_Pos)                             /*!< PORT3 OUT: P10 Mask                 */
#define PORT3_OUT_P11_Pos                                     11                                                        /*!< PORT3 OUT: P11 Position             */
#define PORT3_OUT_P11_Msk                                     (0x01UL << PORT3_OUT_P11_Pos)                             /*!< PORT3 OUT: P11 Mask                 */
#define PORT3_OUT_P12_Pos                                     12                                                        /*!< PORT3 OUT: P12 Position             */
#define PORT3_OUT_P12_Msk                                     (0x01UL << PORT3_OUT_P12_Pos)                             /*!< PORT3 OUT: P12 Mask                 */
#define PORT3_OUT_P13_Pos                                     13                                                        /*!< PORT3 OUT: P13 Position             */
#define PORT3_OUT_P13_Msk                                     (0x01UL << PORT3_OUT_P13_Pos)                             /*!< PORT3 OUT: P13 Mask                 */
#define PORT3_OUT_P14_Pos                                     14                                                        /*!< PORT3 OUT: P14 Position             */
#define PORT3_OUT_P14_Msk                                     (0x01UL << PORT3_OUT_P14_Pos)                             /*!< PORT3 OUT: P14 Mask                 */
#define PORT3_OUT_P15_Pos                                     15                                                        /*!< PORT3 OUT: P15 Position             */
#define PORT3_OUT_P15_Msk                                     (0x01UL << PORT3_OUT_P15_Pos)                             /*!< PORT3 OUT: P15 Mask                 */

// ----------------------------------------  PORT3_OMR  -------------------------------------------
#define PORT3_OMR_PS0_Pos                                     0                                                         /*!< PORT3 OMR: PS0 Position             */
#define PORT3_OMR_PS0_Msk                                     (0x01UL << PORT3_OMR_PS0_Pos)                             /*!< PORT3 OMR: PS0 Mask                 */
#define PORT3_OMR_PS1_Pos                                     1                                                         /*!< PORT3 OMR: PS1 Position             */
#define PORT3_OMR_PS1_Msk                                     (0x01UL << PORT3_OMR_PS1_Pos)                             /*!< PORT3 OMR: PS1 Mask                 */
#define PORT3_OMR_PS2_Pos                                     2                                                         /*!< PORT3 OMR: PS2 Position             */
#define PORT3_OMR_PS2_Msk                                     (0x01UL << PORT3_OMR_PS2_Pos)                             /*!< PORT3 OMR: PS2 Mask                 */
#define PORT3_OMR_PS3_Pos                                     3                                                         /*!< PORT3 OMR: PS3 Position             */
#define PORT3_OMR_PS3_Msk                                     (0x01UL << PORT3_OMR_PS3_Pos)                             /*!< PORT3 OMR: PS3 Mask                 */
#define PORT3_OMR_PS4_Pos                                     4                                                         /*!< PORT3 OMR: PS4 Position             */
#define PORT3_OMR_PS4_Msk                                     (0x01UL << PORT3_OMR_PS4_Pos)                             /*!< PORT3 OMR: PS4 Mask                 */
#define PORT3_OMR_PS5_Pos                                     5                                                         /*!< PORT3 OMR: PS5 Position             */
#define PORT3_OMR_PS5_Msk                                     (0x01UL << PORT3_OMR_PS5_Pos)                             /*!< PORT3 OMR: PS5 Mask                 */
#define PORT3_OMR_PS6_Pos                                     6                                                         /*!< PORT3 OMR: PS6 Position             */
#define PORT3_OMR_PS6_Msk                                     (0x01UL << PORT3_OMR_PS6_Pos)                             /*!< PORT3 OMR: PS6 Mask                 */
#define PORT3_OMR_PS7_Pos                                     7                                                         /*!< PORT3 OMR: PS7 Position             */
#define PORT3_OMR_PS7_Msk                                     (0x01UL << PORT3_OMR_PS7_Pos)                             /*!< PORT3 OMR: PS7 Mask                 */
#define PORT3_OMR_PS8_Pos                                     8                                                         /*!< PORT3 OMR: PS8 Position             */
#define PORT3_OMR_PS8_Msk                                     (0x01UL << PORT3_OMR_PS8_Pos)                             /*!< PORT3 OMR: PS8 Mask                 */
#define PORT3_OMR_PS9_Pos                                     9                                                         /*!< PORT3 OMR: PS9 Position             */
#define PORT3_OMR_PS9_Msk                                     (0x01UL << PORT3_OMR_PS9_Pos)                             /*!< PORT3 OMR: PS9 Mask                 */
#define PORT3_OMR_PS10_Pos                                    10                                                        /*!< PORT3 OMR: PS10 Position            */
#define PORT3_OMR_PS10_Msk                                    (0x01UL << PORT3_OMR_PS10_Pos)                            /*!< PORT3 OMR: PS10 Mask                */
#define PORT3_OMR_PS11_Pos                                    11                                                        /*!< PORT3 OMR: PS11 Position            */
#define PORT3_OMR_PS11_Msk                                    (0x01UL << PORT3_OMR_PS11_Pos)                            /*!< PORT3 OMR: PS11 Mask                */
#define PORT3_OMR_PS12_Pos                                    12                                                        /*!< PORT3 OMR: PS12 Position            */
#define PORT3_OMR_PS12_Msk                                    (0x01UL << PORT3_OMR_PS12_Pos)                            /*!< PORT3 OMR: PS12 Mask                */
#define PORT3_OMR_PS13_Pos                                    13                                                        /*!< PORT3 OMR: PS13 Position            */
#define PORT3_OMR_PS13_Msk                                    (0x01UL << PORT3_OMR_PS13_Pos)                            /*!< PORT3 OMR: PS13 Mask                */
#define PORT3_OMR_PS14_Pos                                    14                                                        /*!< PORT3 OMR: PS14 Position            */
#define PORT3_OMR_PS14_Msk                                    (0x01UL << PORT3_OMR_PS14_Pos)                            /*!< PORT3 OMR: PS14 Mask                */
#define PORT3_OMR_PS15_Pos                                    15                                                        /*!< PORT3 OMR: PS15 Position            */
#define PORT3_OMR_PS15_Msk                                    (0x01UL << PORT3_OMR_PS15_Pos)                            /*!< PORT3 OMR: PS15 Mask                */
#define PORT3_OMR_PR0_Pos                                     16                                                        /*!< PORT3 OMR: PR0 Position             */
#define PORT3_OMR_PR0_Msk                                     (0x01UL << PORT3_OMR_PR0_Pos)                             /*!< PORT3 OMR: PR0 Mask                 */
#define PORT3_OMR_PR1_Pos                                     17                                                        /*!< PORT3 OMR: PR1 Position             */
#define PORT3_OMR_PR1_Msk                                     (0x01UL << PORT3_OMR_PR1_Pos)                             /*!< PORT3 OMR: PR1 Mask                 */
#define PORT3_OMR_PR2_Pos                                     18                                                        /*!< PORT3 OMR: PR2 Position             */
#define PORT3_OMR_PR2_Msk                                     (0x01UL << PORT3_OMR_PR2_Pos)                             /*!< PORT3 OMR: PR2 Mask                 */
#define PORT3_OMR_PR3_Pos                                     19                                                        /*!< PORT3 OMR: PR3 Position             */
#define PORT3_OMR_PR3_Msk                                     (0x01UL << PORT3_OMR_PR3_Pos)                             /*!< PORT3 OMR: PR3 Mask                 */
#define PORT3_OMR_PR4_Pos                                     20                                                        /*!< PORT3 OMR: PR4 Position             */
#define PORT3_OMR_PR4_Msk                                     (0x01UL << PORT3_OMR_PR4_Pos)                             /*!< PORT3 OMR: PR4 Mask                 */
#define PORT3_OMR_PR5_Pos                                     21                                                        /*!< PORT3 OMR: PR5 Position             */
#define PORT3_OMR_PR5_Msk                                     (0x01UL << PORT3_OMR_PR5_Pos)                             /*!< PORT3 OMR: PR5 Mask                 */
#define PORT3_OMR_PR6_Pos                                     22                                                        /*!< PORT3 OMR: PR6 Position             */
#define PORT3_OMR_PR6_Msk                                     (0x01UL << PORT3_OMR_PR6_Pos)                             /*!< PORT3 OMR: PR6 Mask                 */
#define PORT3_OMR_PR7_Pos                                     23                                                        /*!< PORT3 OMR: PR7 Position             */
#define PORT3_OMR_PR7_Msk                                     (0x01UL << PORT3_OMR_PR7_Pos)                             /*!< PORT3 OMR: PR7 Mask                 */
#define PORT3_OMR_PR8_Pos                                     24                                                        /*!< PORT3 OMR: PR8 Position             */
#define PORT3_OMR_PR8_Msk                                     (0x01UL << PORT3_OMR_PR8_Pos)                             /*!< PORT3 OMR: PR8 Mask                 */
#define PORT3_OMR_PR9_Pos                                     25                                                        /*!< PORT3 OMR: PR9 Position             */
#define PORT3_OMR_PR9_Msk                                     (0x01UL << PORT3_OMR_PR9_Pos)                             /*!< PORT3 OMR: PR9 Mask                 */
#define PORT3_OMR_PR10_Pos                                    26                                                        /*!< PORT3 OMR: PR10 Position            */
#define PORT3_OMR_PR10_Msk                                    (0x01UL << PORT3_OMR_PR10_Pos)                            /*!< PORT3 OMR: PR10 Mask                */
#define PORT3_OMR_PR11_Pos                                    27                                                        /*!< PORT3 OMR: PR11 Position            */
#define PORT3_OMR_PR11_Msk                                    (0x01UL << PORT3_OMR_PR11_Pos)                            /*!< PORT3 OMR: PR11 Mask                */
#define PORT3_OMR_PR12_Pos                                    28                                                        /*!< PORT3 OMR: PR12 Position            */
#define PORT3_OMR_PR12_Msk                                    (0x01UL << PORT3_OMR_PR12_Pos)                            /*!< PORT3 OMR: PR12 Mask                */
#define PORT3_OMR_PR13_Pos                                    29                                                        /*!< PORT3 OMR: PR13 Position            */
#define PORT3_OMR_PR13_Msk                                    (0x01UL << PORT3_OMR_PR13_Pos)                            /*!< PORT3 OMR: PR13 Mask                */
#define PORT3_OMR_PR14_Pos                                    30                                                        /*!< PORT3 OMR: PR14 Position            */
#define PORT3_OMR_PR14_Msk                                    (0x01UL << PORT3_OMR_PR14_Pos)                            /*!< PORT3 OMR: PR14 Mask                */
#define PORT3_OMR_PR15_Pos                                    31                                                        /*!< PORT3 OMR: PR15 Position            */
#define PORT3_OMR_PR15_Msk                                    (0x01UL << PORT3_OMR_PR15_Pos)                            /*!< PORT3 OMR: PR15 Mask                */

// ---------------------------------------  PORT3_IOCR0  ------------------------------------------
#define PORT3_IOCR0_PC0_Pos                                   3                                                         /*!< PORT3 IOCR0: PC0 Position           */
#define PORT3_IOCR0_PC0_Msk                                   (0x1fUL << PORT3_IOCR0_PC0_Pos)                           /*!< PORT3 IOCR0: PC0 Mask               */
#define PORT3_IOCR0_PC1_Pos                                   11                                                        /*!< PORT3 IOCR0: PC1 Position           */
#define PORT3_IOCR0_PC1_Msk                                   (0x1fUL << PORT3_IOCR0_PC1_Pos)                           /*!< PORT3 IOCR0: PC1 Mask               */
#define PORT3_IOCR0_PC2_Pos                                   19                                                        /*!< PORT3 IOCR0: PC2 Position           */
#define PORT3_IOCR0_PC2_Msk                                   (0x1fUL << PORT3_IOCR0_PC2_Pos)                           /*!< PORT3 IOCR0: PC2 Mask               */
#define PORT3_IOCR0_PC3_Pos                                   27                                                        /*!< PORT3 IOCR0: PC3 Position           */
#define PORT3_IOCR0_PC3_Msk                                   (0x1fUL << PORT3_IOCR0_PC3_Pos)                           /*!< PORT3 IOCR0: PC3 Mask               */

// ---------------------------------------  PORT3_IOCR4  ------------------------------------------
#define PORT3_IOCR4_PC4_Pos                                   3                                                         /*!< PORT3 IOCR4: PC4 Position           */
#define PORT3_IOCR4_PC4_Msk                                   (0x1fUL << PORT3_IOCR4_PC4_Pos)                           /*!< PORT3 IOCR4: PC4 Mask               */
#define PORT3_IOCR4_PC5_Pos                                   11                                                        /*!< PORT3 IOCR4: PC5 Position           */
#define PORT3_IOCR4_PC5_Msk                                   (0x1fUL << PORT3_IOCR4_PC5_Pos)                           /*!< PORT3 IOCR4: PC5 Mask               */
#define PORT3_IOCR4_PC6_Pos                                   19                                                        /*!< PORT3 IOCR4: PC6 Position           */
#define PORT3_IOCR4_PC6_Msk                                   (0x1fUL << PORT3_IOCR4_PC6_Pos)                           /*!< PORT3 IOCR4: PC6 Mask               */
#define PORT3_IOCR4_PC7_Pos                                   27                                                        /*!< PORT3 IOCR4: PC7 Position           */
#define PORT3_IOCR4_PC7_Msk                                   (0x1fUL << PORT3_IOCR4_PC7_Pos)                           /*!< PORT3 IOCR4: PC7 Mask               */

// ---------------------------------------  PORT3_IOCR8  ------------------------------------------
#define PORT3_IOCR8_PC8_Pos                                   3                                                         /*!< PORT3 IOCR8: PC8 Position           */
#define PORT3_IOCR8_PC8_Msk                                   (0x1fUL << PORT3_IOCR8_PC8_Pos)                           /*!< PORT3 IOCR8: PC8 Mask               */
#define PORT3_IOCR8_PC9_Pos                                   11                                                        /*!< PORT3 IOCR8: PC9 Position           */
#define PORT3_IOCR8_PC9_Msk                                   (0x1fUL << PORT3_IOCR8_PC9_Pos)                           /*!< PORT3 IOCR8: PC9 Mask               */
#define PORT3_IOCR8_PC10_Pos                                  19                                                        /*!< PORT3 IOCR8: PC10 Position          */
#define PORT3_IOCR8_PC10_Msk                                  (0x1fUL << PORT3_IOCR8_PC10_Pos)                          /*!< PORT3 IOCR8: PC10 Mask              */
#define PORT3_IOCR8_PC11_Pos                                  27                                                        /*!< PORT3 IOCR8: PC11 Position          */
#define PORT3_IOCR8_PC11_Msk                                  (0x1fUL << PORT3_IOCR8_PC11_Pos)                          /*!< PORT3 IOCR8: PC11 Mask              */

// --------------------------------------  PORT3_IOCR12  ------------------------------------------
#define PORT3_IOCR12_PC12_Pos                                 3                                                         /*!< PORT3 IOCR12: PC12 Position         */
#define PORT3_IOCR12_PC12_Msk                                 (0x1fUL << PORT3_IOCR12_PC12_Pos)                         /*!< PORT3 IOCR12: PC12 Mask             */
#define PORT3_IOCR12_PC13_Pos                                 11                                                        /*!< PORT3 IOCR12: PC13 Position         */
#define PORT3_IOCR12_PC13_Msk                                 (0x1fUL << PORT3_IOCR12_PC13_Pos)                         /*!< PORT3 IOCR12: PC13 Mask             */
#define PORT3_IOCR12_PC14_Pos                                 19                                                        /*!< PORT3 IOCR12: PC14 Position         */
#define PORT3_IOCR12_PC14_Msk                                 (0x1fUL << PORT3_IOCR12_PC14_Pos)                         /*!< PORT3 IOCR12: PC14 Mask             */
#define PORT3_IOCR12_PC15_Pos                                 27                                                        /*!< PORT3 IOCR12: PC15 Position         */
#define PORT3_IOCR12_PC15_Msk                                 (0x1fUL << PORT3_IOCR12_PC15_Pos)                         /*!< PORT3 IOCR12: PC15 Mask             */

// ----------------------------------------  PORT3_IN  --------------------------------------------
#define PORT3_IN_P0_Pos                                       0                                                         /*!< PORT3 IN: P0 Position               */
#define PORT3_IN_P0_Msk                                       (0x01UL << PORT3_IN_P0_Pos)                               /*!< PORT3 IN: P0 Mask                   */
#define PORT3_IN_P1_Pos                                       1                                                         /*!< PORT3 IN: P1 Position               */
#define PORT3_IN_P1_Msk                                       (0x01UL << PORT3_IN_P1_Pos)                               /*!< PORT3 IN: P1 Mask                   */
#define PORT3_IN_P2_Pos                                       2                                                         /*!< PORT3 IN: P2 Position               */
#define PORT3_IN_P2_Msk                                       (0x01UL << PORT3_IN_P2_Pos)                               /*!< PORT3 IN: P2 Mask                   */
#define PORT3_IN_P3_Pos                                       3                                                         /*!< PORT3 IN: P3 Position               */
#define PORT3_IN_P3_Msk                                       (0x01UL << PORT3_IN_P3_Pos)                               /*!< PORT3 IN: P3 Mask                   */
#define PORT3_IN_P4_Pos                                       4                                                         /*!< PORT3 IN: P4 Position               */
#define PORT3_IN_P4_Msk                                       (0x01UL << PORT3_IN_P4_Pos)                               /*!< PORT3 IN: P4 Mask                   */
#define PORT3_IN_P5_Pos                                       5                                                         /*!< PORT3 IN: P5 Position               */
#define PORT3_IN_P5_Msk                                       (0x01UL << PORT3_IN_P5_Pos)                               /*!< PORT3 IN: P5 Mask                   */
#define PORT3_IN_P6_Pos                                       6                                                         /*!< PORT3 IN: P6 Position               */
#define PORT3_IN_P6_Msk                                       (0x01UL << PORT3_IN_P6_Pos)                               /*!< PORT3 IN: P6 Mask                   */
#define PORT3_IN_P7_Pos                                       7                                                         /*!< PORT3 IN: P7 Position               */
#define PORT3_IN_P7_Msk                                       (0x01UL << PORT3_IN_P7_Pos)                               /*!< PORT3 IN: P7 Mask                   */
#define PORT3_IN_P8_Pos                                       8                                                         /*!< PORT3 IN: P8 Position               */
#define PORT3_IN_P8_Msk                                       (0x01UL << PORT3_IN_P8_Pos)                               /*!< PORT3 IN: P8 Mask                   */
#define PORT3_IN_P9_Pos                                       9                                                         /*!< PORT3 IN: P9 Position               */
#define PORT3_IN_P9_Msk                                       (0x01UL << PORT3_IN_P9_Pos)                               /*!< PORT3 IN: P9 Mask                   */
#define PORT3_IN_P10_Pos                                      10                                                        /*!< PORT3 IN: P10 Position              */
#define PORT3_IN_P10_Msk                                      (0x01UL << PORT3_IN_P10_Pos)                              /*!< PORT3 IN: P10 Mask                  */
#define PORT3_IN_P11_Pos                                      11                                                        /*!< PORT3 IN: P11 Position              */
#define PORT3_IN_P11_Msk                                      (0x01UL << PORT3_IN_P11_Pos)                              /*!< PORT3 IN: P11 Mask                  */
#define PORT3_IN_P12_Pos                                      12                                                        /*!< PORT3 IN: P12 Position              */
#define PORT3_IN_P12_Msk                                      (0x01UL << PORT3_IN_P12_Pos)                              /*!< PORT3 IN: P12 Mask                  */
#define PORT3_IN_P13_Pos                                      13                                                        /*!< PORT3 IN: P13 Position              */
#define PORT3_IN_P13_Msk                                      (0x01UL << PORT3_IN_P13_Pos)                              /*!< PORT3 IN: P13 Mask                  */
#define PORT3_IN_P14_Pos                                      14                                                        /*!< PORT3 IN: P14 Position              */
#define PORT3_IN_P14_Msk                                      (0x01UL << PORT3_IN_P14_Pos)                              /*!< PORT3 IN: P14 Mask                  */
#define PORT3_IN_P15_Pos                                      15                                                        /*!< PORT3 IN: P15 Position              */
#define PORT3_IN_P15_Msk                                      (0x01UL << PORT3_IN_P15_Pos)                              /*!< PORT3 IN: P15 Mask                  */

// ---------------------------------------  PORT3_PDR0  -------------------------------------------
#define PORT3_PDR0_PD0_Pos                                    0                                                         /*!< PORT3 PDR0: PD0 Position            */
#define PORT3_PDR0_PD0_Msk                                    (0x07UL << PORT3_PDR0_PD0_Pos)                            /*!< PORT3 PDR0: PD0 Mask                */
#define PORT3_PDR0_PD1_Pos                                    4                                                         /*!< PORT3 PDR0: PD1 Position            */
#define PORT3_PDR0_PD1_Msk                                    (0x07UL << PORT3_PDR0_PD1_Pos)                            /*!< PORT3 PDR0: PD1 Mask                */
#define PORT3_PDR0_PD2_Pos                                    8                                                         /*!< PORT3 PDR0: PD2 Position            */
#define PORT3_PDR0_PD2_Msk                                    (0x07UL << PORT3_PDR0_PD2_Pos)                            /*!< PORT3 PDR0: PD2 Mask                */
#define PORT3_PDR0_PD3_Pos                                    12                                                        /*!< PORT3 PDR0: PD3 Position            */
#define PORT3_PDR0_PD3_Msk                                    (0x07UL << PORT3_PDR0_PD3_Pos)                            /*!< PORT3 PDR0: PD3 Mask                */
#define PORT3_PDR0_PD4_Pos                                    16                                                        /*!< PORT3 PDR0: PD4 Position            */
#define PORT3_PDR0_PD4_Msk                                    (0x07UL << PORT3_PDR0_PD4_Pos)                            /*!< PORT3 PDR0: PD4 Mask                */
#define PORT3_PDR0_PD5_Pos                                    20                                                        /*!< PORT3 PDR0: PD5 Position            */
#define PORT3_PDR0_PD5_Msk                                    (0x07UL << PORT3_PDR0_PD5_Pos)                            /*!< PORT3 PDR0: PD5 Mask                */
#define PORT3_PDR0_PD6_Pos                                    24                                                        /*!< PORT3 PDR0: PD6 Position            */
#define PORT3_PDR0_PD6_Msk                                    (0x07UL << PORT3_PDR0_PD6_Pos)                            /*!< PORT3 PDR0: PD6 Mask                */
#define PORT3_PDR0_PD7_Pos                                    28                                                        /*!< PORT3 PDR0: PD7 Position            */
#define PORT3_PDR0_PD7_Msk                                    (0x07UL << PORT3_PDR0_PD7_Pos)                            /*!< PORT3 PDR0: PD7 Mask                */

// ---------------------------------------  PORT3_PDR1  -------------------------------------------
#define PORT3_PDR1_PD8_Pos                                    0                                                         /*!< PORT3 PDR1: PD8 Position            */
#define PORT3_PDR1_PD8_Msk                                    (0x07UL << PORT3_PDR1_PD8_Pos)                            /*!< PORT3 PDR1: PD8 Mask                */
#define PORT3_PDR1_PD9_Pos                                    4                                                         /*!< PORT3 PDR1: PD9 Position            */
#define PORT3_PDR1_PD9_Msk                                    (0x07UL << PORT3_PDR1_PD9_Pos)                            /*!< PORT3 PDR1: PD9 Mask                */
#define PORT3_PDR1_PD10_Pos                                   8                                                         /*!< PORT3 PDR1: PD10 Position           */
#define PORT3_PDR1_PD10_Msk                                   (0x07UL << PORT3_PDR1_PD10_Pos)                           /*!< PORT3 PDR1: PD10 Mask               */
#define PORT3_PDR1_PD11_Pos                                   12                                                        /*!< PORT3 PDR1: PD11 Position           */
#define PORT3_PDR1_PD11_Msk                                   (0x07UL << PORT3_PDR1_PD11_Pos)                           /*!< PORT3 PDR1: PD11 Mask               */
#define PORT3_PDR1_PD12_Pos                                   16                                                        /*!< PORT3 PDR1: PD12 Position           */
#define PORT3_PDR1_PD12_Msk                                   (0x07UL << PORT3_PDR1_PD12_Pos)                           /*!< PORT3 PDR1: PD12 Mask               */
#define PORT3_PDR1_PD13_Pos                                   20                                                        /*!< PORT3 PDR1: PD13 Position           */
#define PORT3_PDR1_PD13_Msk                                   (0x07UL << PORT3_PDR1_PD13_Pos)                           /*!< PORT3 PDR1: PD13 Mask               */
#define PORT3_PDR1_PD14_Pos                                   24                                                        /*!< PORT3 PDR1: PD14 Position           */
#define PORT3_PDR1_PD14_Msk                                   (0x07UL << PORT3_PDR1_PD14_Pos)                           /*!< PORT3 PDR1: PD14 Mask               */
#define PORT3_PDR1_PD15_Pos                                   28                                                        /*!< PORT3 PDR1: PD15 Position           */
#define PORT3_PDR1_PD15_Msk                                   (0x07UL << PORT3_PDR1_PD15_Pos)                           /*!< PORT3 PDR1: PD15 Mask               */

// ---------------------------------------  PORT3_PDISC  ------------------------------------------
#define PORT3_PDISC_PDIS0_Pos                                 0                                                         /*!< PORT3 PDISC: PDIS0 Position         */
#define PORT3_PDISC_PDIS0_Msk                                 (0x01UL << PORT3_PDISC_PDIS0_Pos)                         /*!< PORT3 PDISC: PDIS0 Mask             */
#define PORT3_PDISC_PDIS1_Pos                                 1                                                         /*!< PORT3 PDISC: PDIS1 Position         */
#define PORT3_PDISC_PDIS1_Msk                                 (0x01UL << PORT3_PDISC_PDIS1_Pos)                         /*!< PORT3 PDISC: PDIS1 Mask             */
#define PORT3_PDISC_PDIS2_Pos                                 2                                                         /*!< PORT3 PDISC: PDIS2 Position         */
#define PORT3_PDISC_PDIS2_Msk                                 (0x01UL << PORT3_PDISC_PDIS2_Pos)                         /*!< PORT3 PDISC: PDIS2 Mask             */
#define PORT3_PDISC_PDIS3_Pos                                 3                                                         /*!< PORT3 PDISC: PDIS3 Position         */
#define PORT3_PDISC_PDIS3_Msk                                 (0x01UL << PORT3_PDISC_PDIS3_Pos)                         /*!< PORT3 PDISC: PDIS3 Mask             */
#define PORT3_PDISC_PDIS4_Pos                                 4                                                         /*!< PORT3 PDISC: PDIS4 Position         */
#define PORT3_PDISC_PDIS4_Msk                                 (0x01UL << PORT3_PDISC_PDIS4_Pos)                         /*!< PORT3 PDISC: PDIS4 Mask             */
#define PORT3_PDISC_PDIS5_Pos                                 5                                                         /*!< PORT3 PDISC: PDIS5 Position         */
#define PORT3_PDISC_PDIS5_Msk                                 (0x01UL << PORT3_PDISC_PDIS5_Pos)                         /*!< PORT3 PDISC: PDIS5 Mask             */
#define PORT3_PDISC_PDIS6_Pos                                 6                                                         /*!< PORT3 PDISC: PDIS6 Position         */
#define PORT3_PDISC_PDIS6_Msk                                 (0x01UL << PORT3_PDISC_PDIS6_Pos)                         /*!< PORT3 PDISC: PDIS6 Mask             */
#define PORT3_PDISC_PDIS7_Pos                                 7                                                         /*!< PORT3 PDISC: PDIS7 Position         */
#define PORT3_PDISC_PDIS7_Msk                                 (0x01UL << PORT3_PDISC_PDIS7_Pos)                         /*!< PORT3 PDISC: PDIS7 Mask             */
#define PORT3_PDISC_PDIS8_Pos                                 8                                                         /*!< PORT3 PDISC: PDIS8 Position         */
#define PORT3_PDISC_PDIS8_Msk                                 (0x01UL << PORT3_PDISC_PDIS8_Pos)                         /*!< PORT3 PDISC: PDIS8 Mask             */
#define PORT3_PDISC_PDIS9_Pos                                 9                                                         /*!< PORT3 PDISC: PDIS9 Position         */
#define PORT3_PDISC_PDIS9_Msk                                 (0x01UL << PORT3_PDISC_PDIS9_Pos)                         /*!< PORT3 PDISC: PDIS9 Mask             */
#define PORT3_PDISC_PDIS10_Pos                                10                                                        /*!< PORT3 PDISC: PDIS10 Position        */
#define PORT3_PDISC_PDIS10_Msk                                (0x01UL << PORT3_PDISC_PDIS10_Pos)                        /*!< PORT3 PDISC: PDIS10 Mask            */
#define PORT3_PDISC_PDIS11_Pos                                11                                                        /*!< PORT3 PDISC: PDIS11 Position        */
#define PORT3_PDISC_PDIS11_Msk                                (0x01UL << PORT3_PDISC_PDIS11_Pos)                        /*!< PORT3 PDISC: PDIS11 Mask            */
#define PORT3_PDISC_PDIS12_Pos                                12                                                        /*!< PORT3 PDISC: PDIS12 Position        */
#define PORT3_PDISC_PDIS12_Msk                                (0x01UL << PORT3_PDISC_PDIS12_Pos)                        /*!< PORT3 PDISC: PDIS12 Mask            */
#define PORT3_PDISC_PDIS13_Pos                                13                                                        /*!< PORT3 PDISC: PDIS13 Position        */
#define PORT3_PDISC_PDIS13_Msk                                (0x01UL << PORT3_PDISC_PDIS13_Pos)                        /*!< PORT3 PDISC: PDIS13 Mask            */
#define PORT3_PDISC_PDIS14_Pos                                14                                                        /*!< PORT3 PDISC: PDIS14 Position        */
#define PORT3_PDISC_PDIS14_Msk                                (0x01UL << PORT3_PDISC_PDIS14_Pos)                        /*!< PORT3 PDISC: PDIS14 Mask            */
#define PORT3_PDISC_PDIS15_Pos                                15                                                        /*!< PORT3 PDISC: PDIS15 Position        */
#define PORT3_PDISC_PDIS15_Msk                                (0x01UL << PORT3_PDISC_PDIS15_Pos)                        /*!< PORT3 PDISC: PDIS15 Mask            */

// ----------------------------------------  PORT3_PPS  -------------------------------------------
#define PORT3_PPS_PPS0_Pos                                    0                                                         /*!< PORT3 PPS: PPS0 Position            */
#define PORT3_PPS_PPS0_Msk                                    (0x01UL << PORT3_PPS_PPS0_Pos)                            /*!< PORT3 PPS: PPS0 Mask                */
#define PORT3_PPS_PPS1_Pos                                    1                                                         /*!< PORT3 PPS: PPS1 Position            */
#define PORT3_PPS_PPS1_Msk                                    (0x01UL << PORT3_PPS_PPS1_Pos)                            /*!< PORT3 PPS: PPS1 Mask                */
#define PORT3_PPS_PPS2_Pos                                    2                                                         /*!< PORT3 PPS: PPS2 Position            */
#define PORT3_PPS_PPS2_Msk                                    (0x01UL << PORT3_PPS_PPS2_Pos)                            /*!< PORT3 PPS: PPS2 Mask                */
#define PORT3_PPS_PPS3_Pos                                    3                                                         /*!< PORT3 PPS: PPS3 Position            */
#define PORT3_PPS_PPS3_Msk                                    (0x01UL << PORT3_PPS_PPS3_Pos)                            /*!< PORT3 PPS: PPS3 Mask                */
#define PORT3_PPS_PPS4_Pos                                    4                                                         /*!< PORT3 PPS: PPS4 Position            */
#define PORT3_PPS_PPS4_Msk                                    (0x01UL << PORT3_PPS_PPS4_Pos)                            /*!< PORT3 PPS: PPS4 Mask                */
#define PORT3_PPS_PPS5_Pos                                    5                                                         /*!< PORT3 PPS: PPS5 Position            */
#define PORT3_PPS_PPS5_Msk                                    (0x01UL << PORT3_PPS_PPS5_Pos)                            /*!< PORT3 PPS: PPS5 Mask                */
#define PORT3_PPS_PPS6_Pos                                    6                                                         /*!< PORT3 PPS: PPS6 Position            */
#define PORT3_PPS_PPS6_Msk                                    (0x01UL << PORT3_PPS_PPS6_Pos)                            /*!< PORT3 PPS: PPS6 Mask                */
#define PORT3_PPS_PPS7_Pos                                    7                                                         /*!< PORT3 PPS: PPS7 Position            */
#define PORT3_PPS_PPS7_Msk                                    (0x01UL << PORT3_PPS_PPS7_Pos)                            /*!< PORT3 PPS: PPS7 Mask                */
#define PORT3_PPS_PPS8_Pos                                    8                                                         /*!< PORT3 PPS: PPS8 Position            */
#define PORT3_PPS_PPS8_Msk                                    (0x01UL << PORT3_PPS_PPS8_Pos)                            /*!< PORT3 PPS: PPS8 Mask                */
#define PORT3_PPS_PPS9_Pos                                    9                                                         /*!< PORT3 PPS: PPS9 Position            */
#define PORT3_PPS_PPS9_Msk                                    (0x01UL << PORT3_PPS_PPS9_Pos)                            /*!< PORT3 PPS: PPS9 Mask                */
#define PORT3_PPS_PPS10_Pos                                   10                                                        /*!< PORT3 PPS: PPS10 Position           */
#define PORT3_PPS_PPS10_Msk                                   (0x01UL << PORT3_PPS_PPS10_Pos)                           /*!< PORT3 PPS: PPS10 Mask               */
#define PORT3_PPS_PPS11_Pos                                   11                                                        /*!< PORT3 PPS: PPS11 Position           */
#define PORT3_PPS_PPS11_Msk                                   (0x01UL << PORT3_PPS_PPS11_Pos)                           /*!< PORT3 PPS: PPS11 Mask               */
#define PORT3_PPS_PPS12_Pos                                   12                                                        /*!< PORT3 PPS: PPS12 Position           */
#define PORT3_PPS_PPS12_Msk                                   (0x01UL << PORT3_PPS_PPS12_Pos)                           /*!< PORT3 PPS: PPS12 Mask               */
#define PORT3_PPS_PPS13_Pos                                   13                                                        /*!< PORT3 PPS: PPS13 Position           */
#define PORT3_PPS_PPS13_Msk                                   (0x01UL << PORT3_PPS_PPS13_Pos)                           /*!< PORT3 PPS: PPS13 Mask               */
#define PORT3_PPS_PPS14_Pos                                   14                                                        /*!< PORT3 PPS: PPS14 Position           */
#define PORT3_PPS_PPS14_Msk                                   (0x01UL << PORT3_PPS_PPS14_Pos)                           /*!< PORT3 PPS: PPS14 Mask               */
#define PORT3_PPS_PPS15_Pos                                   15                                                        /*!< PORT3 PPS: PPS15 Position           */
#define PORT3_PPS_PPS15_Msk                                   (0x01UL << PORT3_PPS_PPS15_Pos)                           /*!< PORT3 PPS: PPS15 Mask               */

// ---------------------------------------  PORT3_HWSEL  ------------------------------------------
#define PORT3_HWSEL_HW0_Pos                                   0                                                         /*!< PORT3 HWSEL: HW0 Position           */
#define PORT3_HWSEL_HW0_Msk                                   (0x03UL << PORT3_HWSEL_HW0_Pos)                           /*!< PORT3 HWSEL: HW0 Mask               */
#define PORT3_HWSEL_HW1_Pos                                   2                                                         /*!< PORT3 HWSEL: HW1 Position           */
#define PORT3_HWSEL_HW1_Msk                                   (0x03UL << PORT3_HWSEL_HW1_Pos)                           /*!< PORT3 HWSEL: HW1 Mask               */
#define PORT3_HWSEL_HW2_Pos                                   4                                                         /*!< PORT3 HWSEL: HW2 Position           */
#define PORT3_HWSEL_HW2_Msk                                   (0x03UL << PORT3_HWSEL_HW2_Pos)                           /*!< PORT3 HWSEL: HW2 Mask               */
#define PORT3_HWSEL_HW3_Pos                                   6                                                         /*!< PORT3 HWSEL: HW3 Position           */
#define PORT3_HWSEL_HW3_Msk                                   (0x03UL << PORT3_HWSEL_HW3_Pos)                           /*!< PORT3 HWSEL: HW3 Mask               */
#define PORT3_HWSEL_HW4_Pos                                   8                                                         /*!< PORT3 HWSEL: HW4 Position           */
#define PORT3_HWSEL_HW4_Msk                                   (0x03UL << PORT3_HWSEL_HW4_Pos)                           /*!< PORT3 HWSEL: HW4 Mask               */
#define PORT3_HWSEL_HW5_Pos                                   10                                                        /*!< PORT3 HWSEL: HW5 Position           */
#define PORT3_HWSEL_HW5_Msk                                   (0x03UL << PORT3_HWSEL_HW5_Pos)                           /*!< PORT3 HWSEL: HW5 Mask               */
#define PORT3_HWSEL_HW6_Pos                                   12                                                        /*!< PORT3 HWSEL: HW6 Position           */
#define PORT3_HWSEL_HW6_Msk                                   (0x03UL << PORT3_HWSEL_HW6_Pos)                           /*!< PORT3 HWSEL: HW6 Mask               */
#define PORT3_HWSEL_HW7_Pos                                   14                                                        /*!< PORT3 HWSEL: HW7 Position           */
#define PORT3_HWSEL_HW7_Msk                                   (0x03UL << PORT3_HWSEL_HW7_Pos)                           /*!< PORT3 HWSEL: HW7 Mask               */
#define PORT3_HWSEL_HW8_Pos                                   16                                                        /*!< PORT3 HWSEL: HW8 Position           */
#define PORT3_HWSEL_HW8_Msk                                   (0x03UL << PORT3_HWSEL_HW8_Pos)                           /*!< PORT3 HWSEL: HW8 Mask               */
#define PORT3_HWSEL_HW9_Pos                                   18                                                        /*!< PORT3 HWSEL: HW9 Position           */
#define PORT3_HWSEL_HW9_Msk                                   (0x03UL << PORT3_HWSEL_HW9_Pos)                           /*!< PORT3 HWSEL: HW9 Mask               */
#define PORT3_HWSEL_HW10_Pos                                  20                                                        /*!< PORT3 HWSEL: HW10 Position          */
#define PORT3_HWSEL_HW10_Msk                                  (0x03UL << PORT3_HWSEL_HW10_Pos)                          /*!< PORT3 HWSEL: HW10 Mask              */
#define PORT3_HWSEL_HW11_Pos                                  22                                                        /*!< PORT3 HWSEL: HW11 Position          */
#define PORT3_HWSEL_HW11_Msk                                  (0x03UL << PORT3_HWSEL_HW11_Pos)                          /*!< PORT3 HWSEL: HW11 Mask              */
#define PORT3_HWSEL_HW12_Pos                                  24                                                        /*!< PORT3 HWSEL: HW12 Position          */
#define PORT3_HWSEL_HW12_Msk                                  (0x03UL << PORT3_HWSEL_HW12_Pos)                          /*!< PORT3 HWSEL: HW12 Mask              */
#define PORT3_HWSEL_HW13_Pos                                  26                                                        /*!< PORT3 HWSEL: HW13 Position          */
#define PORT3_HWSEL_HW13_Msk                                  (0x03UL << PORT3_HWSEL_HW13_Pos)                          /*!< PORT3 HWSEL: HW13 Mask              */
#define PORT3_HWSEL_HW14_Pos                                  28                                                        /*!< PORT3 HWSEL: HW14 Position          */
#define PORT3_HWSEL_HW14_Msk                                  (0x03UL << PORT3_HWSEL_HW14_Pos)                          /*!< PORT3 HWSEL: HW14 Mask              */
#define PORT3_HWSEL_HW15_Pos                                  30                                                        /*!< PORT3 HWSEL: HW15 Position          */
#define PORT3_HWSEL_HW15_Msk                                  (0x03UL << PORT3_HWSEL_HW15_Pos)                          /*!< PORT3 HWSEL: HW15 Mask              */


// ------------------------------------------------------------------------------------------------
// -----                                 PORT4 Position & Mask                                -----
// ------------------------------------------------------------------------------------------------


// ----------------------------------------  PORT4_OUT  -------------------------------------------
#define PORT4_OUT_P0_Pos                                      0                                                         /*!< PORT4 OUT: P0 Position              */
#define PORT4_OUT_P0_Msk                                      (0x01UL << PORT4_OUT_P0_Pos)                              /*!< PORT4 OUT: P0 Mask                  */
#define PORT4_OUT_P1_Pos                                      1                                                         /*!< PORT4 OUT: P1 Position              */
#define PORT4_OUT_P1_Msk                                      (0x01UL << PORT4_OUT_P1_Pos)                              /*!< PORT4 OUT: P1 Mask                  */
#define PORT4_OUT_P2_Pos                                      2                                                         /*!< PORT4 OUT: P2 Position              */
#define PORT4_OUT_P2_Msk                                      (0x01UL << PORT4_OUT_P2_Pos)                              /*!< PORT4 OUT: P2 Mask                  */
#define PORT4_OUT_P3_Pos                                      3                                                         /*!< PORT4 OUT: P3 Position              */
#define PORT4_OUT_P3_Msk                                      (0x01UL << PORT4_OUT_P3_Pos)                              /*!< PORT4 OUT: P3 Mask                  */
#define PORT4_OUT_P4_Pos                                      4                                                         /*!< PORT4 OUT: P4 Position              */
#define PORT4_OUT_P4_Msk                                      (0x01UL << PORT4_OUT_P4_Pos)                              /*!< PORT4 OUT: P4 Mask                  */
#define PORT4_OUT_P5_Pos                                      5                                                         /*!< PORT4 OUT: P5 Position              */
#define PORT4_OUT_P5_Msk                                      (0x01UL << PORT4_OUT_P5_Pos)                              /*!< PORT4 OUT: P5 Mask                  */
#define PORT4_OUT_P6_Pos                                      6                                                         /*!< PORT4 OUT: P6 Position              */
#define PORT4_OUT_P6_Msk                                      (0x01UL << PORT4_OUT_P6_Pos)                              /*!< PORT4 OUT: P6 Mask                  */
#define PORT4_OUT_P7_Pos                                      7                                                         /*!< PORT4 OUT: P7 Position              */
#define PORT4_OUT_P7_Msk                                      (0x01UL << PORT4_OUT_P7_Pos)                              /*!< PORT4 OUT: P7 Mask                  */
#define PORT4_OUT_P8_Pos                                      8                                                         /*!< PORT4 OUT: P8 Position              */
#define PORT4_OUT_P8_Msk                                      (0x01UL << PORT4_OUT_P8_Pos)                              /*!< PORT4 OUT: P8 Mask                  */
#define PORT4_OUT_P9_Pos                                      9                                                         /*!< PORT4 OUT: P9 Position              */
#define PORT4_OUT_P9_Msk                                      (0x01UL << PORT4_OUT_P9_Pos)                              /*!< PORT4 OUT: P9 Mask                  */
#define PORT4_OUT_P10_Pos                                     10                                                        /*!< PORT4 OUT: P10 Position             */
#define PORT4_OUT_P10_Msk                                     (0x01UL << PORT4_OUT_P10_Pos)                             /*!< PORT4 OUT: P10 Mask                 */
#define PORT4_OUT_P11_Pos                                     11                                                        /*!< PORT4 OUT: P11 Position             */
#define PORT4_OUT_P11_Msk                                     (0x01UL << PORT4_OUT_P11_Pos)                             /*!< PORT4 OUT: P11 Mask                 */
#define PORT4_OUT_P12_Pos                                     12                                                        /*!< PORT4 OUT: P12 Position             */
#define PORT4_OUT_P12_Msk                                     (0x01UL << PORT4_OUT_P12_Pos)                             /*!< PORT4 OUT: P12 Mask                 */
#define PORT4_OUT_P13_Pos                                     13                                                        /*!< PORT4 OUT: P13 Position             */
#define PORT4_OUT_P13_Msk                                     (0x01UL << PORT4_OUT_P13_Pos)                             /*!< PORT4 OUT: P13 Mask                 */
#define PORT4_OUT_P14_Pos                                     14                                                        /*!< PORT4 OUT: P14 Position             */
#define PORT4_OUT_P14_Msk                                     (0x01UL << PORT4_OUT_P14_Pos)                             /*!< PORT4 OUT: P14 Mask                 */
#define PORT4_OUT_P15_Pos                                     15                                                        /*!< PORT4 OUT: P15 Position             */
#define PORT4_OUT_P15_Msk                                     (0x01UL << PORT4_OUT_P15_Pos)                             /*!< PORT4 OUT: P15 Mask                 */

// ----------------------------------------  PORT4_OMR  -------------------------------------------
#define PORT4_OMR_PS0_Pos                                     0                                                         /*!< PORT4 OMR: PS0 Position             */
#define PORT4_OMR_PS0_Msk                                     (0x01UL << PORT4_OMR_PS0_Pos)                             /*!< PORT4 OMR: PS0 Mask                 */
#define PORT4_OMR_PS1_Pos                                     1                                                         /*!< PORT4 OMR: PS1 Position             */
#define PORT4_OMR_PS1_Msk                                     (0x01UL << PORT4_OMR_PS1_Pos)                             /*!< PORT4 OMR: PS1 Mask                 */
#define PORT4_OMR_PS2_Pos                                     2                                                         /*!< PORT4 OMR: PS2 Position             */
#define PORT4_OMR_PS2_Msk                                     (0x01UL << PORT4_OMR_PS2_Pos)                             /*!< PORT4 OMR: PS2 Mask                 */
#define PORT4_OMR_PS3_Pos                                     3                                                         /*!< PORT4 OMR: PS3 Position             */
#define PORT4_OMR_PS3_Msk                                     (0x01UL << PORT4_OMR_PS3_Pos)                             /*!< PORT4 OMR: PS3 Mask                 */
#define PORT4_OMR_PS4_Pos                                     4                                                         /*!< PORT4 OMR: PS4 Position             */
#define PORT4_OMR_PS4_Msk                                     (0x01UL << PORT4_OMR_PS4_Pos)                             /*!< PORT4 OMR: PS4 Mask                 */
#define PORT4_OMR_PS5_Pos                                     5                                                         /*!< PORT4 OMR: PS5 Position             */
#define PORT4_OMR_PS5_Msk                                     (0x01UL << PORT4_OMR_PS5_Pos)                             /*!< PORT4 OMR: PS5 Mask                 */
#define PORT4_OMR_PS6_Pos                                     6                                                         /*!< PORT4 OMR: PS6 Position             */
#define PORT4_OMR_PS6_Msk                                     (0x01UL << PORT4_OMR_PS6_Pos)                             /*!< PORT4 OMR: PS6 Mask                 */
#define PORT4_OMR_PS7_Pos                                     7                                                         /*!< PORT4 OMR: PS7 Position             */
#define PORT4_OMR_PS7_Msk                                     (0x01UL << PORT4_OMR_PS7_Pos)                             /*!< PORT4 OMR: PS7 Mask                 */
#define PORT4_OMR_PS8_Pos                                     8                                                         /*!< PORT4 OMR: PS8 Position             */
#define PORT4_OMR_PS8_Msk                                     (0x01UL << PORT4_OMR_PS8_Pos)                             /*!< PORT4 OMR: PS8 Mask                 */
#define PORT4_OMR_PS9_Pos                                     9                                                         /*!< PORT4 OMR: PS9 Position             */
#define PORT4_OMR_PS9_Msk                                     (0x01UL << PORT4_OMR_PS9_Pos)                             /*!< PORT4 OMR: PS9 Mask                 */
#define PORT4_OMR_PS10_Pos                                    10                                                        /*!< PORT4 OMR: PS10 Position            */
#define PORT4_OMR_PS10_Msk                                    (0x01UL << PORT4_OMR_PS10_Pos)                            /*!< PORT4 OMR: PS10 Mask                */
#define PORT4_OMR_PS11_Pos                                    11                                                        /*!< PORT4 OMR: PS11 Position            */
#define PORT4_OMR_PS11_Msk                                    (0x01UL << PORT4_OMR_PS11_Pos)                            /*!< PORT4 OMR: PS11 Mask                */
#define PORT4_OMR_PS12_Pos                                    12                                                        /*!< PORT4 OMR: PS12 Position            */
#define PORT4_OMR_PS12_Msk                                    (0x01UL << PORT4_OMR_PS12_Pos)                            /*!< PORT4 OMR: PS12 Mask                */
#define PORT4_OMR_PS13_Pos                                    13                                                        /*!< PORT4 OMR: PS13 Position            */
#define PORT4_OMR_PS13_Msk                                    (0x01UL << PORT4_OMR_PS13_Pos)                            /*!< PORT4 OMR: PS13 Mask                */
#define PORT4_OMR_PS14_Pos                                    14                                                        /*!< PORT4 OMR: PS14 Position            */
#define PORT4_OMR_PS14_Msk                                    (0x01UL << PORT4_OMR_PS14_Pos)                            /*!< PORT4 OMR: PS14 Mask                */
#define PORT4_OMR_PS15_Pos                                    15                                                        /*!< PORT4 OMR: PS15 Position            */
#define PORT4_OMR_PS15_Msk                                    (0x01UL << PORT4_OMR_PS15_Pos)                            /*!< PORT4 OMR: PS15 Mask                */
#define PORT4_OMR_PR0_Pos                                     16                                                        /*!< PORT4 OMR: PR0 Position             */
#define PORT4_OMR_PR0_Msk                                     (0x01UL << PORT4_OMR_PR0_Pos)                             /*!< PORT4 OMR: PR0 Mask                 */
#define PORT4_OMR_PR1_Pos                                     17                                                        /*!< PORT4 OMR: PR1 Position             */
#define PORT4_OMR_PR1_Msk                                     (0x01UL << PORT4_OMR_PR1_Pos)                             /*!< PORT4 OMR: PR1 Mask                 */
#define PORT4_OMR_PR2_Pos                                     18                                                        /*!< PORT4 OMR: PR2 Position             */
#define PORT4_OMR_PR2_Msk                                     (0x01UL << PORT4_OMR_PR2_Pos)                             /*!< PORT4 OMR: PR2 Mask                 */
#define PORT4_OMR_PR3_Pos                                     19                                                        /*!< PORT4 OMR: PR3 Position             */
#define PORT4_OMR_PR3_Msk                                     (0x01UL << PORT4_OMR_PR3_Pos)                             /*!< PORT4 OMR: PR3 Mask                 */
#define PORT4_OMR_PR4_Pos                                     20                                                        /*!< PORT4 OMR: PR4 Position             */
#define PORT4_OMR_PR4_Msk                                     (0x01UL << PORT4_OMR_PR4_Pos)                             /*!< PORT4 OMR: PR4 Mask                 */
#define PORT4_OMR_PR5_Pos                                     21                                                        /*!< PORT4 OMR: PR5 Position             */
#define PORT4_OMR_PR5_Msk                                     (0x01UL << PORT4_OMR_PR5_Pos)                             /*!< PORT4 OMR: PR5 Mask                 */
#define PORT4_OMR_PR6_Pos                                     22                                                        /*!< PORT4 OMR: PR6 Position             */
#define PORT4_OMR_PR6_Msk                                     (0x01UL << PORT4_OMR_PR6_Pos)                             /*!< PORT4 OMR: PR6 Mask                 */
#define PORT4_OMR_PR7_Pos                                     23                                                        /*!< PORT4 OMR: PR7 Position             */
#define PORT4_OMR_PR7_Msk                                     (0x01UL << PORT4_OMR_PR7_Pos)                             /*!< PORT4 OMR: PR7 Mask                 */
#define PORT4_OMR_PR8_Pos                                     24                                                        /*!< PORT4 OMR: PR8 Position             */
#define PORT4_OMR_PR8_Msk                                     (0x01UL << PORT4_OMR_PR8_Pos)                             /*!< PORT4 OMR: PR8 Mask                 */
#define PORT4_OMR_PR9_Pos                                     25                                                        /*!< PORT4 OMR: PR9 Position             */
#define PORT4_OMR_PR9_Msk                                     (0x01UL << PORT4_OMR_PR9_Pos)                             /*!< PORT4 OMR: PR9 Mask                 */
#define PORT4_OMR_PR10_Pos                                    26                                                        /*!< PORT4 OMR: PR10 Position            */
#define PORT4_OMR_PR10_Msk                                    (0x01UL << PORT4_OMR_PR10_Pos)                            /*!< PORT4 OMR: PR10 Mask                */
#define PORT4_OMR_PR11_Pos                                    27                                                        /*!< PORT4 OMR: PR11 Position            */
#define PORT4_OMR_PR11_Msk                                    (0x01UL << PORT4_OMR_PR11_Pos)                            /*!< PORT4 OMR: PR11 Mask                */
#define PORT4_OMR_PR12_Pos                                    28                                                        /*!< PORT4 OMR: PR12 Position            */
#define PORT4_OMR_PR12_Msk                                    (0x01UL << PORT4_OMR_PR12_Pos)                            /*!< PORT4 OMR: PR12 Mask                */
#define PORT4_OMR_PR13_Pos                                    29                                                        /*!< PORT4 OMR: PR13 Position            */
#define PORT4_OMR_PR13_Msk                                    (0x01UL << PORT4_OMR_PR13_Pos)                            /*!< PORT4 OMR: PR13 Mask                */
#define PORT4_OMR_PR14_Pos                                    30                                                        /*!< PORT4 OMR: PR14 Position            */
#define PORT4_OMR_PR14_Msk                                    (0x01UL << PORT4_OMR_PR14_Pos)                            /*!< PORT4 OMR: PR14 Mask                */
#define PORT4_OMR_PR15_Pos                                    31                                                        /*!< PORT4 OMR: PR15 Position            */
#define PORT4_OMR_PR15_Msk                                    (0x01UL << PORT4_OMR_PR15_Pos)                            /*!< PORT4 OMR: PR15 Mask                */

// ---------------------------------------  PORT4_IOCR0  ------------------------------------------
#define PORT4_IOCR0_PC0_Pos                                   3                                                         /*!< PORT4 IOCR0: PC0 Position           */
#define PORT4_IOCR0_PC0_Msk                                   (0x1fUL << PORT4_IOCR0_PC0_Pos)                           /*!< PORT4 IOCR0: PC0 Mask               */
#define PORT4_IOCR0_PC1_Pos                                   11                                                        /*!< PORT4 IOCR0: PC1 Position           */
#define PORT4_IOCR0_PC1_Msk                                   (0x1fUL << PORT4_IOCR0_PC1_Pos)                           /*!< PORT4 IOCR0: PC1 Mask               */
#define PORT4_IOCR0_PC2_Pos                                   19                                                        /*!< PORT4 IOCR0: PC2 Position           */
#define PORT4_IOCR0_PC2_Msk                                   (0x1fUL << PORT4_IOCR0_PC2_Pos)                           /*!< PORT4 IOCR0: PC2 Mask               */
#define PORT4_IOCR0_PC3_Pos                                   27                                                        /*!< PORT4 IOCR0: PC3 Position           */
#define PORT4_IOCR0_PC3_Msk                                   (0x1fUL << PORT4_IOCR0_PC3_Pos)                           /*!< PORT4 IOCR0: PC3 Mask               */

// ---------------------------------------  PORT4_IOCR4  ------------------------------------------
#define PORT4_IOCR4_PC4_Pos                                   3                                                         /*!< PORT4 IOCR4: PC4 Position           */
#define PORT4_IOCR4_PC4_Msk                                   (0x1fUL << PORT4_IOCR4_PC4_Pos)                           /*!< PORT4 IOCR4: PC4 Mask               */
#define PORT4_IOCR4_PC5_Pos                                   11                                                        /*!< PORT4 IOCR4: PC5 Position           */
#define PORT4_IOCR4_PC5_Msk                                   (0x1fUL << PORT4_IOCR4_PC5_Pos)                           /*!< PORT4 IOCR4: PC5 Mask               */
#define PORT4_IOCR4_PC6_Pos                                   19                                                        /*!< PORT4 IOCR4: PC6 Position           */
#define PORT4_IOCR4_PC6_Msk                                   (0x1fUL << PORT4_IOCR4_PC6_Pos)                           /*!< PORT4 IOCR4: PC6 Mask               */
#define PORT4_IOCR4_PC7_Pos                                   27                                                        /*!< PORT4 IOCR4: PC7 Position           */
#define PORT4_IOCR4_PC7_Msk                                   (0x1fUL << PORT4_IOCR4_PC7_Pos)                           /*!< PORT4 IOCR4: PC7 Mask               */

// ----------------------------------------  PORT4_IN  --------------------------------------------
#define PORT4_IN_P0_Pos                                       0                                                         /*!< PORT4 IN: P0 Position               */
#define PORT4_IN_P0_Msk                                       (0x01UL << PORT4_IN_P0_Pos)                               /*!< PORT4 IN: P0 Mask                   */
#define PORT4_IN_P1_Pos                                       1                                                         /*!< PORT4 IN: P1 Position               */
#define PORT4_IN_P1_Msk                                       (0x01UL << PORT4_IN_P1_Pos)                               /*!< PORT4 IN: P1 Mask                   */
#define PORT4_IN_P2_Pos                                       2                                                         /*!< PORT4 IN: P2 Position               */
#define PORT4_IN_P2_Msk                                       (0x01UL << PORT4_IN_P2_Pos)                               /*!< PORT4 IN: P2 Mask                   */
#define PORT4_IN_P3_Pos                                       3                                                         /*!< PORT4 IN: P3 Position               */
#define PORT4_IN_P3_Msk                                       (0x01UL << PORT4_IN_P3_Pos)                               /*!< PORT4 IN: P3 Mask                   */
#define PORT4_IN_P4_Pos                                       4                                                         /*!< PORT4 IN: P4 Position               */
#define PORT4_IN_P4_Msk                                       (0x01UL << PORT4_IN_P4_Pos)                               /*!< PORT4 IN: P4 Mask                   */
#define PORT4_IN_P5_Pos                                       5                                                         /*!< PORT4 IN: P5 Position               */
#define PORT4_IN_P5_Msk                                       (0x01UL << PORT4_IN_P5_Pos)                               /*!< PORT4 IN: P5 Mask                   */
#define PORT4_IN_P6_Pos                                       6                                                         /*!< PORT4 IN: P6 Position               */
#define PORT4_IN_P6_Msk                                       (0x01UL << PORT4_IN_P6_Pos)                               /*!< PORT4 IN: P6 Mask                   */
#define PORT4_IN_P7_Pos                                       7                                                         /*!< PORT4 IN: P7 Position               */
#define PORT4_IN_P7_Msk                                       (0x01UL << PORT4_IN_P7_Pos)                               /*!< PORT4 IN: P7 Mask                   */
#define PORT4_IN_P8_Pos                                       8                                                         /*!< PORT4 IN: P8 Position               */
#define PORT4_IN_P8_Msk                                       (0x01UL << PORT4_IN_P8_Pos)                               /*!< PORT4 IN: P8 Mask                   */
#define PORT4_IN_P9_Pos                                       9                                                         /*!< PORT4 IN: P9 Position               */
#define PORT4_IN_P9_Msk                                       (0x01UL << PORT4_IN_P9_Pos)                               /*!< PORT4 IN: P9 Mask                   */
#define PORT4_IN_P10_Pos                                      10                                                        /*!< PORT4 IN: P10 Position              */
#define PORT4_IN_P10_Msk                                      (0x01UL << PORT4_IN_P10_Pos)                              /*!< PORT4 IN: P10 Mask                  */
#define PORT4_IN_P11_Pos                                      11                                                        /*!< PORT4 IN: P11 Position              */
#define PORT4_IN_P11_Msk                                      (0x01UL << PORT4_IN_P11_Pos)                              /*!< PORT4 IN: P11 Mask                  */
#define PORT4_IN_P12_Pos                                      12                                                        /*!< PORT4 IN: P12 Position              */
#define PORT4_IN_P12_Msk                                      (0x01UL << PORT4_IN_P12_Pos)                              /*!< PORT4 IN: P12 Mask                  */
#define PORT4_IN_P13_Pos                                      13                                                        /*!< PORT4 IN: P13 Position              */
#define PORT4_IN_P13_Msk                                      (0x01UL << PORT4_IN_P13_Pos)                              /*!< PORT4 IN: P13 Mask                  */
#define PORT4_IN_P14_Pos                                      14                                                        /*!< PORT4 IN: P14 Position              */
#define PORT4_IN_P14_Msk                                      (0x01UL << PORT4_IN_P14_Pos)                              /*!< PORT4 IN: P14 Mask                  */
#define PORT4_IN_P15_Pos                                      15                                                        /*!< PORT4 IN: P15 Position              */
#define PORT4_IN_P15_Msk                                      (0x01UL << PORT4_IN_P15_Pos)                              /*!< PORT4 IN: P15 Mask                  */

// ---------------------------------------  PORT4_PDR0  -------------------------------------------
#define PORT4_PDR0_PD0_Pos                                    0                                                         /*!< PORT4 PDR0: PD0 Position            */
#define PORT4_PDR0_PD0_Msk                                    (0x07UL << PORT4_PDR0_PD0_Pos)                            /*!< PORT4 PDR0: PD0 Mask                */
#define PORT4_PDR0_PD1_Pos                                    4                                                         /*!< PORT4 PDR0: PD1 Position            */
#define PORT4_PDR0_PD1_Msk                                    (0x07UL << PORT4_PDR0_PD1_Pos)                            /*!< PORT4 PDR0: PD1 Mask                */
#define PORT4_PDR0_PD2_Pos                                    8                                                         /*!< PORT4 PDR0: PD2 Position            */
#define PORT4_PDR0_PD2_Msk                                    (0x07UL << PORT4_PDR0_PD2_Pos)                            /*!< PORT4 PDR0: PD2 Mask                */
#define PORT4_PDR0_PD3_Pos                                    12                                                        /*!< PORT4 PDR0: PD3 Position            */
#define PORT4_PDR0_PD3_Msk                                    (0x07UL << PORT4_PDR0_PD3_Pos)                            /*!< PORT4 PDR0: PD3 Mask                */
#define PORT4_PDR0_PD4_Pos                                    16                                                        /*!< PORT4 PDR0: PD4 Position            */
#define PORT4_PDR0_PD4_Msk                                    (0x07UL << PORT4_PDR0_PD4_Pos)                            /*!< PORT4 PDR0: PD4 Mask                */
#define PORT4_PDR0_PD5_Pos                                    20                                                        /*!< PORT4 PDR0: PD5 Position            */
#define PORT4_PDR0_PD5_Msk                                    (0x07UL << PORT4_PDR0_PD5_Pos)                            /*!< PORT4 PDR0: PD5 Mask                */
#define PORT4_PDR0_PD6_Pos                                    24                                                        /*!< PORT4 PDR0: PD6 Position            */
#define PORT4_PDR0_PD6_Msk                                    (0x07UL << PORT4_PDR0_PD6_Pos)                            /*!< PORT4 PDR0: PD6 Mask                */
#define PORT4_PDR0_PD7_Pos                                    28                                                        /*!< PORT4 PDR0: PD7 Position            */
#define PORT4_PDR0_PD7_Msk                                    (0x07UL << PORT4_PDR0_PD7_Pos)                            /*!< PORT4 PDR0: PD7 Mask                */

// ---------------------------------------  PORT4_PDISC  ------------------------------------------
#define PORT4_PDISC_PDIS0_Pos                                 0                                                         /*!< PORT4 PDISC: PDIS0 Position         */
#define PORT4_PDISC_PDIS0_Msk                                 (0x01UL << PORT4_PDISC_PDIS0_Pos)                         /*!< PORT4 PDISC: PDIS0 Mask             */
#define PORT4_PDISC_PDIS1_Pos                                 1                                                         /*!< PORT4 PDISC: PDIS1 Position         */
#define PORT4_PDISC_PDIS1_Msk                                 (0x01UL << PORT4_PDISC_PDIS1_Pos)                         /*!< PORT4 PDISC: PDIS1 Mask             */
#define PORT4_PDISC_PDIS2_Pos                                 2                                                         /*!< PORT4 PDISC: PDIS2 Position         */
#define PORT4_PDISC_PDIS2_Msk                                 (0x01UL << PORT4_PDISC_PDIS2_Pos)                         /*!< PORT4 PDISC: PDIS2 Mask             */
#define PORT4_PDISC_PDIS3_Pos                                 3                                                         /*!< PORT4 PDISC: PDIS3 Position         */
#define PORT4_PDISC_PDIS3_Msk                                 (0x01UL << PORT4_PDISC_PDIS3_Pos)                         /*!< PORT4 PDISC: PDIS3 Mask             */
#define PORT4_PDISC_PDIS4_Pos                                 4                                                         /*!< PORT4 PDISC: PDIS4 Position         */
#define PORT4_PDISC_PDIS4_Msk                                 (0x01UL << PORT4_PDISC_PDIS4_Pos)                         /*!< PORT4 PDISC: PDIS4 Mask             */
#define PORT4_PDISC_PDIS5_Pos                                 5                                                         /*!< PORT4 PDISC: PDIS5 Position         */
#define PORT4_PDISC_PDIS5_Msk                                 (0x01UL << PORT4_PDISC_PDIS5_Pos)                         /*!< PORT4 PDISC: PDIS5 Mask             */
#define PORT4_PDISC_PDIS6_Pos                                 6                                                         /*!< PORT4 PDISC: PDIS6 Position         */
#define PORT4_PDISC_PDIS6_Msk                                 (0x01UL << PORT4_PDISC_PDIS6_Pos)                         /*!< PORT4 PDISC: PDIS6 Mask             */
#define PORT4_PDISC_PDIS7_Pos                                 7                                                         /*!< PORT4 PDISC: PDIS7 Position         */
#define PORT4_PDISC_PDIS7_Msk                                 (0x01UL << PORT4_PDISC_PDIS7_Pos)                         /*!< PORT4 PDISC: PDIS7 Mask             */
#define PORT4_PDISC_PDIS8_Pos                                 8                                                         /*!< PORT4 PDISC: PDIS8 Position         */
#define PORT4_PDISC_PDIS8_Msk                                 (0x01UL << PORT4_PDISC_PDIS8_Pos)                         /*!< PORT4 PDISC: PDIS8 Mask             */
#define PORT4_PDISC_PDIS9_Pos                                 9                                                         /*!< PORT4 PDISC: PDIS9 Position         */
#define PORT4_PDISC_PDIS9_Msk                                 (0x01UL << PORT4_PDISC_PDIS9_Pos)                         /*!< PORT4 PDISC: PDIS9 Mask             */
#define PORT4_PDISC_PDIS10_Pos                                10                                                        /*!< PORT4 PDISC: PDIS10 Position        */
#define PORT4_PDISC_PDIS10_Msk                                (0x01UL << PORT4_PDISC_PDIS10_Pos)                        /*!< PORT4 PDISC: PDIS10 Mask            */
#define PORT4_PDISC_PDIS11_Pos                                11                                                        /*!< PORT4 PDISC: PDIS11 Position        */
#define PORT4_PDISC_PDIS11_Msk                                (0x01UL << PORT4_PDISC_PDIS11_Pos)                        /*!< PORT4 PDISC: PDIS11 Mask            */
#define PORT4_PDISC_PDIS12_Pos                                12                                                        /*!< PORT4 PDISC: PDIS12 Position        */
#define PORT4_PDISC_PDIS12_Msk                                (0x01UL << PORT4_PDISC_PDIS12_Pos)                        /*!< PORT4 PDISC: PDIS12 Mask            */
#define PORT4_PDISC_PDIS13_Pos                                13                                                        /*!< PORT4 PDISC: PDIS13 Position        */
#define PORT4_PDISC_PDIS13_Msk                                (0x01UL << PORT4_PDISC_PDIS13_Pos)                        /*!< PORT4 PDISC: PDIS13 Mask            */
#define PORT4_PDISC_PDIS14_Pos                                14                                                        /*!< PORT4 PDISC: PDIS14 Position        */
#define PORT4_PDISC_PDIS14_Msk                                (0x01UL << PORT4_PDISC_PDIS14_Pos)                        /*!< PORT4 PDISC: PDIS14 Mask            */
#define PORT4_PDISC_PDIS15_Pos                                15                                                        /*!< PORT4 PDISC: PDIS15 Position        */
#define PORT4_PDISC_PDIS15_Msk                                (0x01UL << PORT4_PDISC_PDIS15_Pos)                        /*!< PORT4 PDISC: PDIS15 Mask            */

// ----------------------------------------  PORT4_PPS  -------------------------------------------
#define PORT4_PPS_PPS0_Pos                                    0                                                         /*!< PORT4 PPS: PPS0 Position            */
#define PORT4_PPS_PPS0_Msk                                    (0x01UL << PORT4_PPS_PPS0_Pos)                            /*!< PORT4 PPS: PPS0 Mask                */
#define PORT4_PPS_PPS1_Pos                                    1                                                         /*!< PORT4 PPS: PPS1 Position            */
#define PORT4_PPS_PPS1_Msk                                    (0x01UL << PORT4_PPS_PPS1_Pos)                            /*!< PORT4 PPS: PPS1 Mask                */
#define PORT4_PPS_PPS2_Pos                                    2                                                         /*!< PORT4 PPS: PPS2 Position            */
#define PORT4_PPS_PPS2_Msk                                    (0x01UL << PORT4_PPS_PPS2_Pos)                            /*!< PORT4 PPS: PPS2 Mask                */
#define PORT4_PPS_PPS3_Pos                                    3                                                         /*!< PORT4 PPS: PPS3 Position            */
#define PORT4_PPS_PPS3_Msk                                    (0x01UL << PORT4_PPS_PPS3_Pos)                            /*!< PORT4 PPS: PPS3 Mask                */
#define PORT4_PPS_PPS4_Pos                                    4                                                         /*!< PORT4 PPS: PPS4 Position            */
#define PORT4_PPS_PPS4_Msk                                    (0x01UL << PORT4_PPS_PPS4_Pos)                            /*!< PORT4 PPS: PPS4 Mask                */
#define PORT4_PPS_PPS5_Pos                                    5                                                         /*!< PORT4 PPS: PPS5 Position            */
#define PORT4_PPS_PPS5_Msk                                    (0x01UL << PORT4_PPS_PPS5_Pos)                            /*!< PORT4 PPS: PPS5 Mask                */
#define PORT4_PPS_PPS6_Pos                                    6                                                         /*!< PORT4 PPS: PPS6 Position            */
#define PORT4_PPS_PPS6_Msk                                    (0x01UL << PORT4_PPS_PPS6_Pos)                            /*!< PORT4 PPS: PPS6 Mask                */
#define PORT4_PPS_PPS7_Pos                                    7                                                         /*!< PORT4 PPS: PPS7 Position            */
#define PORT4_PPS_PPS7_Msk                                    (0x01UL << PORT4_PPS_PPS7_Pos)                            /*!< PORT4 PPS: PPS7 Mask                */
#define PORT4_PPS_PPS8_Pos                                    8                                                         /*!< PORT4 PPS: PPS8 Position            */
#define PORT4_PPS_PPS8_Msk                                    (0x01UL << PORT4_PPS_PPS8_Pos)                            /*!< PORT4 PPS: PPS8 Mask                */
#define PORT4_PPS_PPS9_Pos                                    9                                                         /*!< PORT4 PPS: PPS9 Position            */
#define PORT4_PPS_PPS9_Msk                                    (0x01UL << PORT4_PPS_PPS9_Pos)                            /*!< PORT4 PPS: PPS9 Mask                */
#define PORT4_PPS_PPS10_Pos                                   10                                                        /*!< PORT4 PPS: PPS10 Position           */
#define PORT4_PPS_PPS10_Msk                                   (0x01UL << PORT4_PPS_PPS10_Pos)                           /*!< PORT4 PPS: PPS10 Mask               */
#define PORT4_PPS_PPS11_Pos                                   11                                                        /*!< PORT4 PPS: PPS11 Position           */
#define PORT4_PPS_PPS11_Msk                                   (0x01UL << PORT4_PPS_PPS11_Pos)                           /*!< PORT4 PPS: PPS11 Mask               */
#define PORT4_PPS_PPS12_Pos                                   12                                                        /*!< PORT4 PPS: PPS12 Position           */
#define PORT4_PPS_PPS12_Msk                                   (0x01UL << PORT4_PPS_PPS12_Pos)                           /*!< PORT4 PPS: PPS12 Mask               */
#define PORT4_PPS_PPS13_Pos                                   13                                                        /*!< PORT4 PPS: PPS13 Position           */
#define PORT4_PPS_PPS13_Msk                                   (0x01UL << PORT4_PPS_PPS13_Pos)                           /*!< PORT4 PPS: PPS13 Mask               */
#define PORT4_PPS_PPS14_Pos                                   14                                                        /*!< PORT4 PPS: PPS14 Position           */
#define PORT4_PPS_PPS14_Msk                                   (0x01UL << PORT4_PPS_PPS14_Pos)                           /*!< PORT4 PPS: PPS14 Mask               */
#define PORT4_PPS_PPS15_Pos                                   15                                                        /*!< PORT4 PPS: PPS15 Position           */
#define PORT4_PPS_PPS15_Msk                                   (0x01UL << PORT4_PPS_PPS15_Pos)                           /*!< PORT4 PPS: PPS15 Mask               */

// ---------------------------------------  PORT4_HWSEL  ------------------------------------------
#define PORT4_HWSEL_HW0_Pos                                   0                                                         /*!< PORT4 HWSEL: HW0 Position           */
#define PORT4_HWSEL_HW0_Msk                                   (0x03UL << PORT4_HWSEL_HW0_Pos)                           /*!< PORT4 HWSEL: HW0 Mask               */
#define PORT4_HWSEL_HW1_Pos                                   2                                                         /*!< PORT4 HWSEL: HW1 Position           */
#define PORT4_HWSEL_HW1_Msk                                   (0x03UL << PORT4_HWSEL_HW1_Pos)                           /*!< PORT4 HWSEL: HW1 Mask               */
#define PORT4_HWSEL_HW2_Pos                                   4                                                         /*!< PORT4 HWSEL: HW2 Position           */
#define PORT4_HWSEL_HW2_Msk                                   (0x03UL << PORT4_HWSEL_HW2_Pos)                           /*!< PORT4 HWSEL: HW2 Mask               */
#define PORT4_HWSEL_HW3_Pos                                   6                                                         /*!< PORT4 HWSEL: HW3 Position           */
#define PORT4_HWSEL_HW3_Msk                                   (0x03UL << PORT4_HWSEL_HW3_Pos)                           /*!< PORT4 HWSEL: HW3 Mask               */
#define PORT4_HWSEL_HW4_Pos                                   8                                                         /*!< PORT4 HWSEL: HW4 Position           */
#define PORT4_HWSEL_HW4_Msk                                   (0x03UL << PORT4_HWSEL_HW4_Pos)                           /*!< PORT4 HWSEL: HW4 Mask               */
#define PORT4_HWSEL_HW5_Pos                                   10                                                        /*!< PORT4 HWSEL: HW5 Position           */
#define PORT4_HWSEL_HW5_Msk                                   (0x03UL << PORT4_HWSEL_HW5_Pos)                           /*!< PORT4 HWSEL: HW5 Mask               */
#define PORT4_HWSEL_HW6_Pos                                   12                                                        /*!< PORT4 HWSEL: HW6 Position           */
#define PORT4_HWSEL_HW6_Msk                                   (0x03UL << PORT4_HWSEL_HW6_Pos)                           /*!< PORT4 HWSEL: HW6 Mask               */
#define PORT4_HWSEL_HW7_Pos                                   14                                                        /*!< PORT4 HWSEL: HW7 Position           */
#define PORT4_HWSEL_HW7_Msk                                   (0x03UL << PORT4_HWSEL_HW7_Pos)                           /*!< PORT4 HWSEL: HW7 Mask               */
#define PORT4_HWSEL_HW8_Pos                                   16                                                        /*!< PORT4 HWSEL: HW8 Position           */
#define PORT4_HWSEL_HW8_Msk                                   (0x03UL << PORT4_HWSEL_HW8_Pos)                           /*!< PORT4 HWSEL: HW8 Mask               */
#define PORT4_HWSEL_HW9_Pos                                   18                                                        /*!< PORT4 HWSEL: HW9 Position           */
#define PORT4_HWSEL_HW9_Msk                                   (0x03UL << PORT4_HWSEL_HW9_Pos)                           /*!< PORT4 HWSEL: HW9 Mask               */
#define PORT4_HWSEL_HW10_Pos                                  20                                                        /*!< PORT4 HWSEL: HW10 Position          */
#define PORT4_HWSEL_HW10_Msk                                  (0x03UL << PORT4_HWSEL_HW10_Pos)                          /*!< PORT4 HWSEL: HW10 Mask              */
#define PORT4_HWSEL_HW11_Pos                                  22                                                        /*!< PORT4 HWSEL: HW11 Position          */
#define PORT4_HWSEL_HW11_Msk                                  (0x03UL << PORT4_HWSEL_HW11_Pos)                          /*!< PORT4 HWSEL: HW11 Mask              */
#define PORT4_HWSEL_HW12_Pos                                  24                                                        /*!< PORT4 HWSEL: HW12 Position          */
#define PORT4_HWSEL_HW12_Msk                                  (0x03UL << PORT4_HWSEL_HW12_Pos)                          /*!< PORT4 HWSEL: HW12 Mask              */
#define PORT4_HWSEL_HW13_Pos                                  26                                                        /*!< PORT4 HWSEL: HW13 Position          */
#define PORT4_HWSEL_HW13_Msk                                  (0x03UL << PORT4_HWSEL_HW13_Pos)                          /*!< PORT4 HWSEL: HW13 Mask              */
#define PORT4_HWSEL_HW14_Pos                                  28                                                        /*!< PORT4 HWSEL: HW14 Position          */
#define PORT4_HWSEL_HW14_Msk                                  (0x03UL << PORT4_HWSEL_HW14_Pos)                          /*!< PORT4 HWSEL: HW14 Mask              */
#define PORT4_HWSEL_HW15_Pos                                  30                                                        /*!< PORT4 HWSEL: HW15 Position          */
#define PORT4_HWSEL_HW15_Msk                                  (0x03UL << PORT4_HWSEL_HW15_Pos)                          /*!< PORT4 HWSEL: HW15 Mask              */


// ------------------------------------------------------------------------------------------------
// -----                                 PORT5 Position & Mask                                -----
// ------------------------------------------------------------------------------------------------


// ----------------------------------------  PORT5_OUT  -------------------------------------------
#define PORT5_OUT_P0_Pos                                      0                                                         /*!< PORT5 OUT: P0 Position              */
#define PORT5_OUT_P0_Msk                                      (0x01UL << PORT5_OUT_P0_Pos)                              /*!< PORT5 OUT: P0 Mask                  */
#define PORT5_OUT_P1_Pos                                      1                                                         /*!< PORT5 OUT: P1 Position              */
#define PORT5_OUT_P1_Msk                                      (0x01UL << PORT5_OUT_P1_Pos)                              /*!< PORT5 OUT: P1 Mask                  */
#define PORT5_OUT_P2_Pos                                      2                                                         /*!< PORT5 OUT: P2 Position              */
#define PORT5_OUT_P2_Msk                                      (0x01UL << PORT5_OUT_P2_Pos)                              /*!< PORT5 OUT: P2 Mask                  */
#define PORT5_OUT_P3_Pos                                      3                                                         /*!< PORT5 OUT: P3 Position              */
#define PORT5_OUT_P3_Msk                                      (0x01UL << PORT5_OUT_P3_Pos)                              /*!< PORT5 OUT: P3 Mask                  */
#define PORT5_OUT_P4_Pos                                      4                                                         /*!< PORT5 OUT: P4 Position              */
#define PORT5_OUT_P4_Msk                                      (0x01UL << PORT5_OUT_P4_Pos)                              /*!< PORT5 OUT: P4 Mask                  */
#define PORT5_OUT_P5_Pos                                      5                                                         /*!< PORT5 OUT: P5 Position              */
#define PORT5_OUT_P5_Msk                                      (0x01UL << PORT5_OUT_P5_Pos)                              /*!< PORT5 OUT: P5 Mask                  */
#define PORT5_OUT_P6_Pos                                      6                                                         /*!< PORT5 OUT: P6 Position              */
#define PORT5_OUT_P6_Msk                                      (0x01UL << PORT5_OUT_P6_Pos)                              /*!< PORT5 OUT: P6 Mask                  */
#define PORT5_OUT_P7_Pos                                      7                                                         /*!< PORT5 OUT: P7 Position              */
#define PORT5_OUT_P7_Msk                                      (0x01UL << PORT5_OUT_P7_Pos)                              /*!< PORT5 OUT: P7 Mask                  */
#define PORT5_OUT_P8_Pos                                      8                                                         /*!< PORT5 OUT: P8 Position              */
#define PORT5_OUT_P8_Msk                                      (0x01UL << PORT5_OUT_P8_Pos)                              /*!< PORT5 OUT: P8 Mask                  */
#define PORT5_OUT_P9_Pos                                      9                                                         /*!< PORT5 OUT: P9 Position              */
#define PORT5_OUT_P9_Msk                                      (0x01UL << PORT5_OUT_P9_Pos)                              /*!< PORT5 OUT: P9 Mask                  */
#define PORT5_OUT_P10_Pos                                     10                                                        /*!< PORT5 OUT: P10 Position             */
#define PORT5_OUT_P10_Msk                                     (0x01UL << PORT5_OUT_P10_Pos)                             /*!< PORT5 OUT: P10 Mask                 */
#define PORT5_OUT_P11_Pos                                     11                                                        /*!< PORT5 OUT: P11 Position             */
#define PORT5_OUT_P11_Msk                                     (0x01UL << PORT5_OUT_P11_Pos)                             /*!< PORT5 OUT: P11 Mask                 */
#define PORT5_OUT_P12_Pos                                     12                                                        /*!< PORT5 OUT: P12 Position             */
#define PORT5_OUT_P12_Msk                                     (0x01UL << PORT5_OUT_P12_Pos)                             /*!< PORT5 OUT: P12 Mask                 */
#define PORT5_OUT_P13_Pos                                     13                                                        /*!< PORT5 OUT: P13 Position             */
#define PORT5_OUT_P13_Msk                                     (0x01UL << PORT5_OUT_P13_Pos)                             /*!< PORT5 OUT: P13 Mask                 */
#define PORT5_OUT_P14_Pos                                     14                                                        /*!< PORT5 OUT: P14 Position             */
#define PORT5_OUT_P14_Msk                                     (0x01UL << PORT5_OUT_P14_Pos)                             /*!< PORT5 OUT: P14 Mask                 */
#define PORT5_OUT_P15_Pos                                     15                                                        /*!< PORT5 OUT: P15 Position             */
#define PORT5_OUT_P15_Msk                                     (0x01UL << PORT5_OUT_P15_Pos)                             /*!< PORT5 OUT: P15 Mask                 */

// ----------------------------------------  PORT5_OMR  -------------------------------------------
#define PORT5_OMR_PS0_Pos                                     0                                                         /*!< PORT5 OMR: PS0 Position             */
#define PORT5_OMR_PS0_Msk                                     (0x01UL << PORT5_OMR_PS0_Pos)                             /*!< PORT5 OMR: PS0 Mask                 */
#define PORT5_OMR_PS1_Pos                                     1                                                         /*!< PORT5 OMR: PS1 Position             */
#define PORT5_OMR_PS1_Msk                                     (0x01UL << PORT5_OMR_PS1_Pos)                             /*!< PORT5 OMR: PS1 Mask                 */
#define PORT5_OMR_PS2_Pos                                     2                                                         /*!< PORT5 OMR: PS2 Position             */
#define PORT5_OMR_PS2_Msk                                     (0x01UL << PORT5_OMR_PS2_Pos)                             /*!< PORT5 OMR: PS2 Mask                 */
#define PORT5_OMR_PS3_Pos                                     3                                                         /*!< PORT5 OMR: PS3 Position             */
#define PORT5_OMR_PS3_Msk                                     (0x01UL << PORT5_OMR_PS3_Pos)                             /*!< PORT5 OMR: PS3 Mask                 */
#define PORT5_OMR_PS4_Pos                                     4                                                         /*!< PORT5 OMR: PS4 Position             */
#define PORT5_OMR_PS4_Msk                                     (0x01UL << PORT5_OMR_PS4_Pos)                             /*!< PORT5 OMR: PS4 Mask                 */
#define PORT5_OMR_PS5_Pos                                     5                                                         /*!< PORT5 OMR: PS5 Position             */
#define PORT5_OMR_PS5_Msk                                     (0x01UL << PORT5_OMR_PS5_Pos)                             /*!< PORT5 OMR: PS5 Mask                 */
#define PORT5_OMR_PS6_Pos                                     6                                                         /*!< PORT5 OMR: PS6 Position             */
#define PORT5_OMR_PS6_Msk                                     (0x01UL << PORT5_OMR_PS6_Pos)                             /*!< PORT5 OMR: PS6 Mask                 */
#define PORT5_OMR_PS7_Pos                                     7                                                         /*!< PORT5 OMR: PS7 Position             */
#define PORT5_OMR_PS7_Msk                                     (0x01UL << PORT5_OMR_PS7_Pos)                             /*!< PORT5 OMR: PS7 Mask                 */
#define PORT5_OMR_PS8_Pos                                     8                                                         /*!< PORT5 OMR: PS8 Position             */
#define PORT5_OMR_PS8_Msk                                     (0x01UL << PORT5_OMR_PS8_Pos)                             /*!< PORT5 OMR: PS8 Mask                 */
#define PORT5_OMR_PS9_Pos                                     9                                                         /*!< PORT5 OMR: PS9 Position             */
#define PORT5_OMR_PS9_Msk                                     (0x01UL << PORT5_OMR_PS9_Pos)                             /*!< PORT5 OMR: PS9 Mask                 */
#define PORT5_OMR_PS10_Pos                                    10                                                        /*!< PORT5 OMR: PS10 Position            */
#define PORT5_OMR_PS10_Msk                                    (0x01UL << PORT5_OMR_PS10_Pos)                            /*!< PORT5 OMR: PS10 Mask                */
#define PORT5_OMR_PS11_Pos                                    11                                                        /*!< PORT5 OMR: PS11 Position            */
#define PORT5_OMR_PS11_Msk                                    (0x01UL << PORT5_OMR_PS11_Pos)                            /*!< PORT5 OMR: PS11 Mask                */
#define PORT5_OMR_PS12_Pos                                    12                                                        /*!< PORT5 OMR: PS12 Position            */
#define PORT5_OMR_PS12_Msk                                    (0x01UL << PORT5_OMR_PS12_Pos)                            /*!< PORT5 OMR: PS12 Mask                */
#define PORT5_OMR_PS13_Pos                                    13                                                        /*!< PORT5 OMR: PS13 Position            */
#define PORT5_OMR_PS13_Msk                                    (0x01UL << PORT5_OMR_PS13_Pos)                            /*!< PORT5 OMR: PS13 Mask                */
#define PORT5_OMR_PS14_Pos                                    14                                                        /*!< PORT5 OMR: PS14 Position            */
#define PORT5_OMR_PS14_Msk                                    (0x01UL << PORT5_OMR_PS14_Pos)                            /*!< PORT5 OMR: PS14 Mask                */
#define PORT5_OMR_PS15_Pos                                    15                                                        /*!< PORT5 OMR: PS15 Position            */
#define PORT5_OMR_PS15_Msk                                    (0x01UL << PORT5_OMR_PS15_Pos)                            /*!< PORT5 OMR: PS15 Mask                */
#define PORT5_OMR_PR0_Pos                                     16                                                        /*!< PORT5 OMR: PR0 Position             */
#define PORT5_OMR_PR0_Msk                                     (0x01UL << PORT5_OMR_PR0_Pos)                             /*!< PORT5 OMR: PR0 Mask                 */
#define PORT5_OMR_PR1_Pos                                     17                                                        /*!< PORT5 OMR: PR1 Position             */
#define PORT5_OMR_PR1_Msk                                     (0x01UL << PORT5_OMR_PR1_Pos)                             /*!< PORT5 OMR: PR1 Mask                 */
#define PORT5_OMR_PR2_Pos                                     18                                                        /*!< PORT5 OMR: PR2 Position             */
#define PORT5_OMR_PR2_Msk                                     (0x01UL << PORT5_OMR_PR2_Pos)                             /*!< PORT5 OMR: PR2 Mask                 */
#define PORT5_OMR_PR3_Pos                                     19                                                        /*!< PORT5 OMR: PR3 Position             */
#define PORT5_OMR_PR3_Msk                                     (0x01UL << PORT5_OMR_PR3_Pos)                             /*!< PORT5 OMR: PR3 Mask                 */
#define PORT5_OMR_PR4_Pos                                     20                                                        /*!< PORT5 OMR: PR4 Position             */
#define PORT5_OMR_PR4_Msk                                     (0x01UL << PORT5_OMR_PR4_Pos)                             /*!< PORT5 OMR: PR4 Mask                 */
#define PORT5_OMR_PR5_Pos                                     21                                                        /*!< PORT5 OMR: PR5 Position             */
#define PORT5_OMR_PR5_Msk                                     (0x01UL << PORT5_OMR_PR5_Pos)                             /*!< PORT5 OMR: PR5 Mask                 */
#define PORT5_OMR_PR6_Pos                                     22                                                        /*!< PORT5 OMR: PR6 Position             */
#define PORT5_OMR_PR6_Msk                                     (0x01UL << PORT5_OMR_PR6_Pos)                             /*!< PORT5 OMR: PR6 Mask                 */
#define PORT5_OMR_PR7_Pos                                     23                                                        /*!< PORT5 OMR: PR7 Position             */
#define PORT5_OMR_PR7_Msk                                     (0x01UL << PORT5_OMR_PR7_Pos)                             /*!< PORT5 OMR: PR7 Mask                 */
#define PORT5_OMR_PR8_Pos                                     24                                                        /*!< PORT5 OMR: PR8 Position             */
#define PORT5_OMR_PR8_Msk                                     (0x01UL << PORT5_OMR_PR8_Pos)                             /*!< PORT5 OMR: PR8 Mask                 */
#define PORT5_OMR_PR9_Pos                                     25                                                        /*!< PORT5 OMR: PR9 Position             */
#define PORT5_OMR_PR9_Msk                                     (0x01UL << PORT5_OMR_PR9_Pos)                             /*!< PORT5 OMR: PR9 Mask                 */
#define PORT5_OMR_PR10_Pos                                    26                                                        /*!< PORT5 OMR: PR10 Position            */
#define PORT5_OMR_PR10_Msk                                    (0x01UL << PORT5_OMR_PR10_Pos)                            /*!< PORT5 OMR: PR10 Mask                */
#define PORT5_OMR_PR11_Pos                                    27                                                        /*!< PORT5 OMR: PR11 Position            */
#define PORT5_OMR_PR11_Msk                                    (0x01UL << PORT5_OMR_PR11_Pos)                            /*!< PORT5 OMR: PR11 Mask                */
#define PORT5_OMR_PR12_Pos                                    28                                                        /*!< PORT5 OMR: PR12 Position            */
#define PORT5_OMR_PR12_Msk                                    (0x01UL << PORT5_OMR_PR12_Pos)                            /*!< PORT5 OMR: PR12 Mask                */
#define PORT5_OMR_PR13_Pos                                    29                                                        /*!< PORT5 OMR: PR13 Position            */
#define PORT5_OMR_PR13_Msk                                    (0x01UL << PORT5_OMR_PR13_Pos)                            /*!< PORT5 OMR: PR13 Mask                */
#define PORT5_OMR_PR14_Pos                                    30                                                        /*!< PORT5 OMR: PR14 Position            */
#define PORT5_OMR_PR14_Msk                                    (0x01UL << PORT5_OMR_PR14_Pos)                            /*!< PORT5 OMR: PR14 Mask                */
#define PORT5_OMR_PR15_Pos                                    31                                                        /*!< PORT5 OMR: PR15 Position            */
#define PORT5_OMR_PR15_Msk                                    (0x01UL << PORT5_OMR_PR15_Pos)                            /*!< PORT5 OMR: PR15 Mask                */

// ---------------------------------------  PORT5_IOCR0  ------------------------------------------
#define PORT5_IOCR0_PC0_Pos                                   3                                                         /*!< PORT5 IOCR0: PC0 Position           */
#define PORT5_IOCR0_PC0_Msk                                   (0x1fUL << PORT5_IOCR0_PC0_Pos)                           /*!< PORT5 IOCR0: PC0 Mask               */
#define PORT5_IOCR0_PC1_Pos                                   11                                                        /*!< PORT5 IOCR0: PC1 Position           */
#define PORT5_IOCR0_PC1_Msk                                   (0x1fUL << PORT5_IOCR0_PC1_Pos)                           /*!< PORT5 IOCR0: PC1 Mask               */
#define PORT5_IOCR0_PC2_Pos                                   19                                                        /*!< PORT5 IOCR0: PC2 Position           */
#define PORT5_IOCR0_PC2_Msk                                   (0x1fUL << PORT5_IOCR0_PC2_Pos)                           /*!< PORT5 IOCR0: PC2 Mask               */
#define PORT5_IOCR0_PC3_Pos                                   27                                                        /*!< PORT5 IOCR0: PC3 Position           */
#define PORT5_IOCR0_PC3_Msk                                   (0x1fUL << PORT5_IOCR0_PC3_Pos)                           /*!< PORT5 IOCR0: PC3 Mask               */

// ---------------------------------------  PORT5_IOCR4  ------------------------------------------
#define PORT5_IOCR4_PC4_Pos                                   3                                                         /*!< PORT5 IOCR4: PC4 Position           */
#define PORT5_IOCR4_PC4_Msk                                   (0x1fUL << PORT5_IOCR4_PC4_Pos)                           /*!< PORT5 IOCR4: PC4 Mask               */
#define PORT5_IOCR4_PC5_Pos                                   11                                                        /*!< PORT5 IOCR4: PC5 Position           */
#define PORT5_IOCR4_PC5_Msk                                   (0x1fUL << PORT5_IOCR4_PC5_Pos)                           /*!< PORT5 IOCR4: PC5 Mask               */
#define PORT5_IOCR4_PC6_Pos                                   19                                                        /*!< PORT5 IOCR4: PC6 Position           */
#define PORT5_IOCR4_PC6_Msk                                   (0x1fUL << PORT5_IOCR4_PC6_Pos)                           /*!< PORT5 IOCR4: PC6 Mask               */
#define PORT5_IOCR4_PC7_Pos                                   27                                                        /*!< PORT5 IOCR4: PC7 Position           */
#define PORT5_IOCR4_PC7_Msk                                   (0x1fUL << PORT5_IOCR4_PC7_Pos)                           /*!< PORT5 IOCR4: PC7 Mask               */

// ---------------------------------------  PORT5_IOCR8  ------------------------------------------
#define PORT5_IOCR8_PC8_Pos                                   3                                                         /*!< PORT5 IOCR8: PC8 Position           */
#define PORT5_IOCR8_PC8_Msk                                   (0x1fUL << PORT5_IOCR8_PC8_Pos)                           /*!< PORT5 IOCR8: PC8 Mask               */
#define PORT5_IOCR8_PC9_Pos                                   11                                                        /*!< PORT5 IOCR8: PC9 Position           */
#define PORT5_IOCR8_PC9_Msk                                   (0x1fUL << PORT5_IOCR8_PC9_Pos)                           /*!< PORT5 IOCR8: PC9 Mask               */
#define PORT5_IOCR8_PC10_Pos                                  19                                                        /*!< PORT5 IOCR8: PC10 Position          */
#define PORT5_IOCR8_PC10_Msk                                  (0x1fUL << PORT5_IOCR8_PC10_Pos)                          /*!< PORT5 IOCR8: PC10 Mask              */
#define PORT5_IOCR8_PC11_Pos                                  27                                                        /*!< PORT5 IOCR8: PC11 Position          */
#define PORT5_IOCR8_PC11_Msk                                  (0x1fUL << PORT5_IOCR8_PC11_Pos)                          /*!< PORT5 IOCR8: PC11 Mask              */

// ----------------------------------------  PORT5_IN  --------------------------------------------
#define PORT5_IN_P0_Pos                                       0                                                         /*!< PORT5 IN: P0 Position               */
#define PORT5_IN_P0_Msk                                       (0x01UL << PORT5_IN_P0_Pos)                               /*!< PORT5 IN: P0 Mask                   */
#define PORT5_IN_P1_Pos                                       1                                                         /*!< PORT5 IN: P1 Position               */
#define PORT5_IN_P1_Msk                                       (0x01UL << PORT5_IN_P1_Pos)                               /*!< PORT5 IN: P1 Mask                   */
#define PORT5_IN_P2_Pos                                       2                                                         /*!< PORT5 IN: P2 Position               */
#define PORT5_IN_P2_Msk                                       (0x01UL << PORT5_IN_P2_Pos)                               /*!< PORT5 IN: P2 Mask                   */
#define PORT5_IN_P3_Pos                                       3                                                         /*!< PORT5 IN: P3 Position               */
#define PORT5_IN_P3_Msk                                       (0x01UL << PORT5_IN_P3_Pos)                               /*!< PORT5 IN: P3 Mask                   */
#define PORT5_IN_P4_Pos                                       4                                                         /*!< PORT5 IN: P4 Position               */
#define PORT5_IN_P4_Msk                                       (0x01UL << PORT5_IN_P4_Pos)                               /*!< PORT5 IN: P4 Mask                   */
#define PORT5_IN_P5_Pos                                       5                                                         /*!< PORT5 IN: P5 Position               */
#define PORT5_IN_P5_Msk                                       (0x01UL << PORT5_IN_P5_Pos)                               /*!< PORT5 IN: P5 Mask                   */
#define PORT5_IN_P6_Pos                                       6                                                         /*!< PORT5 IN: P6 Position               */
#define PORT5_IN_P6_Msk                                       (0x01UL << PORT5_IN_P6_Pos)                               /*!< PORT5 IN: P6 Mask                   */
#define PORT5_IN_P7_Pos                                       7                                                         /*!< PORT5 IN: P7 Position               */
#define PORT5_IN_P7_Msk                                       (0x01UL << PORT5_IN_P7_Pos)                               /*!< PORT5 IN: P7 Mask                   */
#define PORT5_IN_P8_Pos                                       8                                                         /*!< PORT5 IN: P8 Position               */
#define PORT5_IN_P8_Msk                                       (0x01UL << PORT5_IN_P8_Pos)                               /*!< PORT5 IN: P8 Mask                   */
#define PORT5_IN_P9_Pos                                       9                                                         /*!< PORT5 IN: P9 Position               */
#define PORT5_IN_P9_Msk                                       (0x01UL << PORT5_IN_P9_Pos)                               /*!< PORT5 IN: P9 Mask                   */
#define PORT5_IN_P10_Pos                                      10                                                        /*!< PORT5 IN: P10 Position              */
#define PORT5_IN_P10_Msk                                      (0x01UL << PORT5_IN_P10_Pos)                              /*!< PORT5 IN: P10 Mask                  */
#define PORT5_IN_P11_Pos                                      11                                                        /*!< PORT5 IN: P11 Position              */
#define PORT5_IN_P11_Msk                                      (0x01UL << PORT5_IN_P11_Pos)                              /*!< PORT5 IN: P11 Mask                  */
#define PORT5_IN_P12_Pos                                      12                                                        /*!< PORT5 IN: P12 Position              */
#define PORT5_IN_P12_Msk                                      (0x01UL << PORT5_IN_P12_Pos)                              /*!< PORT5 IN: P12 Mask                  */
#define PORT5_IN_P13_Pos                                      13                                                        /*!< PORT5 IN: P13 Position              */
#define PORT5_IN_P13_Msk                                      (0x01UL << PORT5_IN_P13_Pos)                              /*!< PORT5 IN: P13 Mask                  */
#define PORT5_IN_P14_Pos                                      14                                                        /*!< PORT5 IN: P14 Position              */
#define PORT5_IN_P14_Msk                                      (0x01UL << PORT5_IN_P14_Pos)                              /*!< PORT5 IN: P14 Mask                  */
#define PORT5_IN_P15_Pos                                      15                                                        /*!< PORT5 IN: P15 Position              */
#define PORT5_IN_P15_Msk                                      (0x01UL << PORT5_IN_P15_Pos)                              /*!< PORT5 IN: P15 Mask                  */

// ---------------------------------------  PORT5_PDR0  -------------------------------------------
#define PORT5_PDR0_PD0_Pos                                    0                                                         /*!< PORT5 PDR0: PD0 Position            */
#define PORT5_PDR0_PD0_Msk                                    (0x07UL << PORT5_PDR0_PD0_Pos)                            /*!< PORT5 PDR0: PD0 Mask                */
#define PORT5_PDR0_PD1_Pos                                    4                                                         /*!< PORT5 PDR0: PD1 Position            */
#define PORT5_PDR0_PD1_Msk                                    (0x07UL << PORT5_PDR0_PD1_Pos)                            /*!< PORT5 PDR0: PD1 Mask                */
#define PORT5_PDR0_PD2_Pos                                    8                                                         /*!< PORT5 PDR0: PD2 Position            */
#define PORT5_PDR0_PD2_Msk                                    (0x07UL << PORT5_PDR0_PD2_Pos)                            /*!< PORT5 PDR0: PD2 Mask                */
#define PORT5_PDR0_PD3_Pos                                    12                                                        /*!< PORT5 PDR0: PD3 Position            */
#define PORT5_PDR0_PD3_Msk                                    (0x07UL << PORT5_PDR0_PD3_Pos)                            /*!< PORT5 PDR0: PD3 Mask                */
#define PORT5_PDR0_PD4_Pos                                    16                                                        /*!< PORT5 PDR0: PD4 Position            */
#define PORT5_PDR0_PD4_Msk                                    (0x07UL << PORT5_PDR0_PD4_Pos)                            /*!< PORT5 PDR0: PD4 Mask                */
#define PORT5_PDR0_PD5_Pos                                    20                                                        /*!< PORT5 PDR0: PD5 Position            */
#define PORT5_PDR0_PD5_Msk                                    (0x07UL << PORT5_PDR0_PD5_Pos)                            /*!< PORT5 PDR0: PD5 Mask                */
#define PORT5_PDR0_PD6_Pos                                    24                                                        /*!< PORT5 PDR0: PD6 Position            */
#define PORT5_PDR0_PD6_Msk                                    (0x07UL << PORT5_PDR0_PD6_Pos)                            /*!< PORT5 PDR0: PD6 Mask                */
#define PORT5_PDR0_PD7_Pos                                    28                                                        /*!< PORT5 PDR0: PD7 Position            */
#define PORT5_PDR0_PD7_Msk                                    (0x07UL << PORT5_PDR0_PD7_Pos)                            /*!< PORT5 PDR0: PD7 Mask                */

// ---------------------------------------  PORT5_PDR1  -------------------------------------------
#define PORT5_PDR1_PD8_Pos                                    0                                                         /*!< PORT5 PDR1: PD8 Position            */
#define PORT5_PDR1_PD8_Msk                                    (0x07UL << PORT5_PDR1_PD8_Pos)                            /*!< PORT5 PDR1: PD8 Mask                */
#define PORT5_PDR1_PD9_Pos                                    4                                                         /*!< PORT5 PDR1: PD9 Position            */
#define PORT5_PDR1_PD9_Msk                                    (0x07UL << PORT5_PDR1_PD9_Pos)                            /*!< PORT5 PDR1: PD9 Mask                */
#define PORT5_PDR1_PD10_Pos                                   8                                                         /*!< PORT5 PDR1: PD10 Position           */
#define PORT5_PDR1_PD10_Msk                                   (0x07UL << PORT5_PDR1_PD10_Pos)                           /*!< PORT5 PDR1: PD10 Mask               */
#define PORT5_PDR1_PD11_Pos                                   12                                                        /*!< PORT5 PDR1: PD11 Position           */
#define PORT5_PDR1_PD11_Msk                                   (0x07UL << PORT5_PDR1_PD11_Pos)                           /*!< PORT5 PDR1: PD11 Mask               */
#define PORT5_PDR1_PD12_Pos                                   16                                                        /*!< PORT5 PDR1: PD12 Position           */
#define PORT5_PDR1_PD12_Msk                                   (0x07UL << PORT5_PDR1_PD12_Pos)                           /*!< PORT5 PDR1: PD12 Mask               */
#define PORT5_PDR1_PD13_Pos                                   20                                                        /*!< PORT5 PDR1: PD13 Position           */
#define PORT5_PDR1_PD13_Msk                                   (0x07UL << PORT5_PDR1_PD13_Pos)                           /*!< PORT5 PDR1: PD13 Mask               */
#define PORT5_PDR1_PD14_Pos                                   24                                                        /*!< PORT5 PDR1: PD14 Position           */
#define PORT5_PDR1_PD14_Msk                                   (0x07UL << PORT5_PDR1_PD14_Pos)                           /*!< PORT5 PDR1: PD14 Mask               */
#define PORT5_PDR1_PD15_Pos                                   28                                                        /*!< PORT5 PDR1: PD15 Position           */
#define PORT5_PDR1_PD15_Msk                                   (0x07UL << PORT5_PDR1_PD15_Pos)                           /*!< PORT5 PDR1: PD15 Mask               */

// ---------------------------------------  PORT5_PDISC  ------------------------------------------
#define PORT5_PDISC_PDIS0_Pos                                 0                                                         /*!< PORT5 PDISC: PDIS0 Position         */
#define PORT5_PDISC_PDIS0_Msk                                 (0x01UL << PORT5_PDISC_PDIS0_Pos)                         /*!< PORT5 PDISC: PDIS0 Mask             */
#define PORT5_PDISC_PDIS1_Pos                                 1                                                         /*!< PORT5 PDISC: PDIS1 Position         */
#define PORT5_PDISC_PDIS1_Msk                                 (0x01UL << PORT5_PDISC_PDIS1_Pos)                         /*!< PORT5 PDISC: PDIS1 Mask             */
#define PORT5_PDISC_PDIS2_Pos                                 2                                                         /*!< PORT5 PDISC: PDIS2 Position         */
#define PORT5_PDISC_PDIS2_Msk                                 (0x01UL << PORT5_PDISC_PDIS2_Pos)                         /*!< PORT5 PDISC: PDIS2 Mask             */
#define PORT5_PDISC_PDIS3_Pos                                 3                                                         /*!< PORT5 PDISC: PDIS3 Position         */
#define PORT5_PDISC_PDIS3_Msk                                 (0x01UL << PORT5_PDISC_PDIS3_Pos)                         /*!< PORT5 PDISC: PDIS3 Mask             */
#define PORT5_PDISC_PDIS4_Pos                                 4                                                         /*!< PORT5 PDISC: PDIS4 Position         */
#define PORT5_PDISC_PDIS4_Msk                                 (0x01UL << PORT5_PDISC_PDIS4_Pos)                         /*!< PORT5 PDISC: PDIS4 Mask             */
#define PORT5_PDISC_PDIS5_Pos                                 5                                                         /*!< PORT5 PDISC: PDIS5 Position         */
#define PORT5_PDISC_PDIS5_Msk                                 (0x01UL << PORT5_PDISC_PDIS5_Pos)                         /*!< PORT5 PDISC: PDIS5 Mask             */
#define PORT5_PDISC_PDIS6_Pos                                 6                                                         /*!< PORT5 PDISC: PDIS6 Position         */
#define PORT5_PDISC_PDIS6_Msk                                 (0x01UL << PORT5_PDISC_PDIS6_Pos)                         /*!< PORT5 PDISC: PDIS6 Mask             */
#define PORT5_PDISC_PDIS7_Pos                                 7                                                         /*!< PORT5 PDISC: PDIS7 Position         */
#define PORT5_PDISC_PDIS7_Msk                                 (0x01UL << PORT5_PDISC_PDIS7_Pos)                         /*!< PORT5 PDISC: PDIS7 Mask             */
#define PORT5_PDISC_PDIS8_Pos                                 8                                                         /*!< PORT5 PDISC: PDIS8 Position         */
#define PORT5_PDISC_PDIS8_Msk                                 (0x01UL << PORT5_PDISC_PDIS8_Pos)                         /*!< PORT5 PDISC: PDIS8 Mask             */
#define PORT5_PDISC_PDIS9_Pos                                 9                                                         /*!< PORT5 PDISC: PDIS9 Position         */
#define PORT5_PDISC_PDIS9_Msk                                 (0x01UL << PORT5_PDISC_PDIS9_Pos)                         /*!< PORT5 PDISC: PDIS9 Mask             */
#define PORT5_PDISC_PDIS10_Pos                                10                                                        /*!< PORT5 PDISC: PDIS10 Position        */
#define PORT5_PDISC_PDIS10_Msk                                (0x01UL << PORT5_PDISC_PDIS10_Pos)                        /*!< PORT5 PDISC: PDIS10 Mask            */
#define PORT5_PDISC_PDIS11_Pos                                11                                                        /*!< PORT5 PDISC: PDIS11 Position        */
#define PORT5_PDISC_PDIS11_Msk                                (0x01UL << PORT5_PDISC_PDIS11_Pos)                        /*!< PORT5 PDISC: PDIS11 Mask            */
#define PORT5_PDISC_PDIS12_Pos                                12                                                        /*!< PORT5 PDISC: PDIS12 Position        */
#define PORT5_PDISC_PDIS12_Msk                                (0x01UL << PORT5_PDISC_PDIS12_Pos)                        /*!< PORT5 PDISC: PDIS12 Mask            */
#define PORT5_PDISC_PDIS13_Pos                                13                                                        /*!< PORT5 PDISC: PDIS13 Position        */
#define PORT5_PDISC_PDIS13_Msk                                (0x01UL << PORT5_PDISC_PDIS13_Pos)                        /*!< PORT5 PDISC: PDIS13 Mask            */
#define PORT5_PDISC_PDIS14_Pos                                14                                                        /*!< PORT5 PDISC: PDIS14 Position        */
#define PORT5_PDISC_PDIS14_Msk                                (0x01UL << PORT5_PDISC_PDIS14_Pos)                        /*!< PORT5 PDISC: PDIS14 Mask            */
#define PORT5_PDISC_PDIS15_Pos                                15                                                        /*!< PORT5 PDISC: PDIS15 Position        */
#define PORT5_PDISC_PDIS15_Msk                                (0x01UL << PORT5_PDISC_PDIS15_Pos)                        /*!< PORT5 PDISC: PDIS15 Mask            */

// ----------------------------------------  PORT5_PPS  -------------------------------------------
#define PORT5_PPS_PPS0_Pos                                    0                                                         /*!< PORT5 PPS: PPS0 Position            */
#define PORT5_PPS_PPS0_Msk                                    (0x01UL << PORT5_PPS_PPS0_Pos)                            /*!< PORT5 PPS: PPS0 Mask                */
#define PORT5_PPS_PPS1_Pos                                    1                                                         /*!< PORT5 PPS: PPS1 Position            */
#define PORT5_PPS_PPS1_Msk                                    (0x01UL << PORT5_PPS_PPS1_Pos)                            /*!< PORT5 PPS: PPS1 Mask                */
#define PORT5_PPS_PPS2_Pos                                    2                                                         /*!< PORT5 PPS: PPS2 Position            */
#define PORT5_PPS_PPS2_Msk                                    (0x01UL << PORT5_PPS_PPS2_Pos)                            /*!< PORT5 PPS: PPS2 Mask                */
#define PORT5_PPS_PPS3_Pos                                    3                                                         /*!< PORT5 PPS: PPS3 Position            */
#define PORT5_PPS_PPS3_Msk                                    (0x01UL << PORT5_PPS_PPS3_Pos)                            /*!< PORT5 PPS: PPS3 Mask                */
#define PORT5_PPS_PPS4_Pos                                    4                                                         /*!< PORT5 PPS: PPS4 Position            */
#define PORT5_PPS_PPS4_Msk                                    (0x01UL << PORT5_PPS_PPS4_Pos)                            /*!< PORT5 PPS: PPS4 Mask                */
#define PORT5_PPS_PPS5_Pos                                    5                                                         /*!< PORT5 PPS: PPS5 Position            */
#define PORT5_PPS_PPS5_Msk                                    (0x01UL << PORT5_PPS_PPS5_Pos)                            /*!< PORT5 PPS: PPS5 Mask                */
#define PORT5_PPS_PPS6_Pos                                    6                                                         /*!< PORT5 PPS: PPS6 Position            */
#define PORT5_PPS_PPS6_Msk                                    (0x01UL << PORT5_PPS_PPS6_Pos)                            /*!< PORT5 PPS: PPS6 Mask                */
#define PORT5_PPS_PPS7_Pos                                    7                                                         /*!< PORT5 PPS: PPS7 Position            */
#define PORT5_PPS_PPS7_Msk                                    (0x01UL << PORT5_PPS_PPS7_Pos)                            /*!< PORT5 PPS: PPS7 Mask                */
#define PORT5_PPS_PPS8_Pos                                    8                                                         /*!< PORT5 PPS: PPS8 Position            */
#define PORT5_PPS_PPS8_Msk                                    (0x01UL << PORT5_PPS_PPS8_Pos)                            /*!< PORT5 PPS: PPS8 Mask                */
#define PORT5_PPS_PPS9_Pos                                    9                                                         /*!< PORT5 PPS: PPS9 Position            */
#define PORT5_PPS_PPS9_Msk                                    (0x01UL << PORT5_PPS_PPS9_Pos)                            /*!< PORT5 PPS: PPS9 Mask                */
#define PORT5_PPS_PPS10_Pos                                   10                                                        /*!< PORT5 PPS: PPS10 Position           */
#define PORT5_PPS_PPS10_Msk                                   (0x01UL << PORT5_PPS_PPS10_Pos)                           /*!< PORT5 PPS: PPS10 Mask               */
#define PORT5_PPS_PPS11_Pos                                   11                                                        /*!< PORT5 PPS: PPS11 Position           */
#define PORT5_PPS_PPS11_Msk                                   (0x01UL << PORT5_PPS_PPS11_Pos)                           /*!< PORT5 PPS: PPS11 Mask               */
#define PORT5_PPS_PPS12_Pos                                   12                                                        /*!< PORT5 PPS: PPS12 Position           */
#define PORT5_PPS_PPS12_Msk                                   (0x01UL << PORT5_PPS_PPS12_Pos)                           /*!< PORT5 PPS: PPS12 Mask               */
#define PORT5_PPS_PPS13_Pos                                   13                                                        /*!< PORT5 PPS: PPS13 Position           */
#define PORT5_PPS_PPS13_Msk                                   (0x01UL << PORT5_PPS_PPS13_Pos)                           /*!< PORT5 PPS: PPS13 Mask               */
#define PORT5_PPS_PPS14_Pos                                   14                                                        /*!< PORT5 PPS: PPS14 Position           */
#define PORT5_PPS_PPS14_Msk                                   (0x01UL << PORT5_PPS_PPS14_Pos)                           /*!< PORT5 PPS: PPS14 Mask               */
#define PORT5_PPS_PPS15_Pos                                   15                                                        /*!< PORT5 PPS: PPS15 Position           */
#define PORT5_PPS_PPS15_Msk                                   (0x01UL << PORT5_PPS_PPS15_Pos)                           /*!< PORT5 PPS: PPS15 Mask               */

// ---------------------------------------  PORT5_HWSEL  ------------------------------------------
#define PORT5_HWSEL_HW0_Pos                                   0                                                         /*!< PORT5 HWSEL: HW0 Position           */
#define PORT5_HWSEL_HW0_Msk                                   (0x03UL << PORT5_HWSEL_HW0_Pos)                           /*!< PORT5 HWSEL: HW0 Mask               */
#define PORT5_HWSEL_HW1_Pos                                   2                                                         /*!< PORT5 HWSEL: HW1 Position           */
#define PORT5_HWSEL_HW1_Msk                                   (0x03UL << PORT5_HWSEL_HW1_Pos)                           /*!< PORT5 HWSEL: HW1 Mask               */
#define PORT5_HWSEL_HW2_Pos                                   4                                                         /*!< PORT5 HWSEL: HW2 Position           */
#define PORT5_HWSEL_HW2_Msk                                   (0x03UL << PORT5_HWSEL_HW2_Pos)                           /*!< PORT5 HWSEL: HW2 Mask               */
#define PORT5_HWSEL_HW3_Pos                                   6                                                         /*!< PORT5 HWSEL: HW3 Position           */
#define PORT5_HWSEL_HW3_Msk                                   (0x03UL << PORT5_HWSEL_HW3_Pos)                           /*!< PORT5 HWSEL: HW3 Mask               */
#define PORT5_HWSEL_HW4_Pos                                   8                                                         /*!< PORT5 HWSEL: HW4 Position           */
#define PORT5_HWSEL_HW4_Msk                                   (0x03UL << PORT5_HWSEL_HW4_Pos)                           /*!< PORT5 HWSEL: HW4 Mask               */
#define PORT5_HWSEL_HW5_Pos                                   10                                                        /*!< PORT5 HWSEL: HW5 Position           */
#define PORT5_HWSEL_HW5_Msk                                   (0x03UL << PORT5_HWSEL_HW5_Pos)                           /*!< PORT5 HWSEL: HW5 Mask               */
#define PORT5_HWSEL_HW6_Pos                                   12                                                        /*!< PORT5 HWSEL: HW6 Position           */
#define PORT5_HWSEL_HW6_Msk                                   (0x03UL << PORT5_HWSEL_HW6_Pos)                           /*!< PORT5 HWSEL: HW6 Mask               */
#define PORT5_HWSEL_HW7_Pos                                   14                                                        /*!< PORT5 HWSEL: HW7 Position           */
#define PORT5_HWSEL_HW7_Msk                                   (0x03UL << PORT5_HWSEL_HW7_Pos)                           /*!< PORT5 HWSEL: HW7 Mask               */
#define PORT5_HWSEL_HW8_Pos                                   16                                                        /*!< PORT5 HWSEL: HW8 Position           */
#define PORT5_HWSEL_HW8_Msk                                   (0x03UL << PORT5_HWSEL_HW8_Pos)                           /*!< PORT5 HWSEL: HW8 Mask               */
#define PORT5_HWSEL_HW9_Pos                                   18                                                        /*!< PORT5 HWSEL: HW9 Position           */
#define PORT5_HWSEL_HW9_Msk                                   (0x03UL << PORT5_HWSEL_HW9_Pos)                           /*!< PORT5 HWSEL: HW9 Mask               */
#define PORT5_HWSEL_HW10_Pos                                  20                                                        /*!< PORT5 HWSEL: HW10 Position          */
#define PORT5_HWSEL_HW10_Msk                                  (0x03UL << PORT5_HWSEL_HW10_Pos)                          /*!< PORT5 HWSEL: HW10 Mask              */
#define PORT5_HWSEL_HW11_Pos                                  22                                                        /*!< PORT5 HWSEL: HW11 Position          */
#define PORT5_HWSEL_HW11_Msk                                  (0x03UL << PORT5_HWSEL_HW11_Pos)                          /*!< PORT5 HWSEL: HW11 Mask              */
#define PORT5_HWSEL_HW12_Pos                                  24                                                        /*!< PORT5 HWSEL: HW12 Position          */
#define PORT5_HWSEL_HW12_Msk                                  (0x03UL << PORT5_HWSEL_HW12_Pos)                          /*!< PORT5 HWSEL: HW12 Mask              */
#define PORT5_HWSEL_HW13_Pos                                  26                                                        /*!< PORT5 HWSEL: HW13 Position          */
#define PORT5_HWSEL_HW13_Msk                                  (0x03UL << PORT5_HWSEL_HW13_Pos)                          /*!< PORT5 HWSEL: HW13 Mask              */
#define PORT5_HWSEL_HW14_Pos                                  28                                                        /*!< PORT5 HWSEL: HW14 Position          */
#define PORT5_HWSEL_HW14_Msk                                  (0x03UL << PORT5_HWSEL_HW14_Pos)                          /*!< PORT5 HWSEL: HW14 Mask              */
#define PORT5_HWSEL_HW15_Pos                                  30                                                        /*!< PORT5 HWSEL: HW15 Position          */
#define PORT5_HWSEL_HW15_Msk                                  (0x03UL << PORT5_HWSEL_HW15_Pos)                          /*!< PORT5 HWSEL: HW15 Mask              */


// ------------------------------------------------------------------------------------------------
// -----                                 PORT6 Position & Mask                                -----
// ------------------------------------------------------------------------------------------------


// ----------------------------------------  PORT6_OUT  -------------------------------------------
#define PORT6_OUT_P0_Pos                                      0                                                         /*!< PORT6 OUT: P0 Position              */
#define PORT6_OUT_P0_Msk                                      (0x01UL << PORT6_OUT_P0_Pos)                              /*!< PORT6 OUT: P0 Mask                  */
#define PORT6_OUT_P1_Pos                                      1                                                         /*!< PORT6 OUT: P1 Position              */
#define PORT6_OUT_P1_Msk                                      (0x01UL << PORT6_OUT_P1_Pos)                              /*!< PORT6 OUT: P1 Mask                  */
#define PORT6_OUT_P2_Pos                                      2                                                         /*!< PORT6 OUT: P2 Position              */
#define PORT6_OUT_P2_Msk                                      (0x01UL << PORT6_OUT_P2_Pos)                              /*!< PORT6 OUT: P2 Mask                  */
#define PORT6_OUT_P3_Pos                                      3                                                         /*!< PORT6 OUT: P3 Position              */
#define PORT6_OUT_P3_Msk                                      (0x01UL << PORT6_OUT_P3_Pos)                              /*!< PORT6 OUT: P3 Mask                  */
#define PORT6_OUT_P4_Pos                                      4                                                         /*!< PORT6 OUT: P4 Position              */
#define PORT6_OUT_P4_Msk                                      (0x01UL << PORT6_OUT_P4_Pos)                              /*!< PORT6 OUT: P4 Mask                  */
#define PORT6_OUT_P5_Pos                                      5                                                         /*!< PORT6 OUT: P5 Position              */
#define PORT6_OUT_P5_Msk                                      (0x01UL << PORT6_OUT_P5_Pos)                              /*!< PORT6 OUT: P5 Mask                  */
#define PORT6_OUT_P6_Pos                                      6                                                         /*!< PORT6 OUT: P6 Position              */
#define PORT6_OUT_P6_Msk                                      (0x01UL << PORT6_OUT_P6_Pos)                              /*!< PORT6 OUT: P6 Mask                  */
#define PORT6_OUT_P7_Pos                                      7                                                         /*!< PORT6 OUT: P7 Position              */
#define PORT6_OUT_P7_Msk                                      (0x01UL << PORT6_OUT_P7_Pos)                              /*!< PORT6 OUT: P7 Mask                  */
#define PORT6_OUT_P8_Pos                                      8                                                         /*!< PORT6 OUT: P8 Position              */
#define PORT6_OUT_P8_Msk                                      (0x01UL << PORT6_OUT_P8_Pos)                              /*!< PORT6 OUT: P8 Mask                  */
#define PORT6_OUT_P9_Pos                                      9                                                         /*!< PORT6 OUT: P9 Position              */
#define PORT6_OUT_P9_Msk                                      (0x01UL << PORT6_OUT_P9_Pos)                              /*!< PORT6 OUT: P9 Mask                  */
#define PORT6_OUT_P10_Pos                                     10                                                        /*!< PORT6 OUT: P10 Position             */
#define PORT6_OUT_P10_Msk                                     (0x01UL << PORT6_OUT_P10_Pos)                             /*!< PORT6 OUT: P10 Mask                 */
#define PORT6_OUT_P11_Pos                                     11                                                        /*!< PORT6 OUT: P11 Position             */
#define PORT6_OUT_P11_Msk                                     (0x01UL << PORT6_OUT_P11_Pos)                             /*!< PORT6 OUT: P11 Mask                 */
#define PORT6_OUT_P12_Pos                                     12                                                        /*!< PORT6 OUT: P12 Position             */
#define PORT6_OUT_P12_Msk                                     (0x01UL << PORT6_OUT_P12_Pos)                             /*!< PORT6 OUT: P12 Mask                 */
#define PORT6_OUT_P13_Pos                                     13                                                        /*!< PORT6 OUT: P13 Position             */
#define PORT6_OUT_P13_Msk                                     (0x01UL << PORT6_OUT_P13_Pos)                             /*!< PORT6 OUT: P13 Mask                 */
#define PORT6_OUT_P14_Pos                                     14                                                        /*!< PORT6 OUT: P14 Position             */
#define PORT6_OUT_P14_Msk                                     (0x01UL << PORT6_OUT_P14_Pos)                             /*!< PORT6 OUT: P14 Mask                 */
#define PORT6_OUT_P15_Pos                                     15                                                        /*!< PORT6 OUT: P15 Position             */
#define PORT6_OUT_P15_Msk                                     (0x01UL << PORT6_OUT_P15_Pos)                             /*!< PORT6 OUT: P15 Mask                 */

// ----------------------------------------  PORT6_OMR  -------------------------------------------
#define PORT6_OMR_PS0_Pos                                     0                                                         /*!< PORT6 OMR: PS0 Position             */
#define PORT6_OMR_PS0_Msk                                     (0x01UL << PORT6_OMR_PS0_Pos)                             /*!< PORT6 OMR: PS0 Mask                 */
#define PORT6_OMR_PS1_Pos                                     1                                                         /*!< PORT6 OMR: PS1 Position             */
#define PORT6_OMR_PS1_Msk                                     (0x01UL << PORT6_OMR_PS1_Pos)                             /*!< PORT6 OMR: PS1 Mask                 */
#define PORT6_OMR_PS2_Pos                                     2                                                         /*!< PORT6 OMR: PS2 Position             */
#define PORT6_OMR_PS2_Msk                                     (0x01UL << PORT6_OMR_PS2_Pos)                             /*!< PORT6 OMR: PS2 Mask                 */
#define PORT6_OMR_PS3_Pos                                     3                                                         /*!< PORT6 OMR: PS3 Position             */
#define PORT6_OMR_PS3_Msk                                     (0x01UL << PORT6_OMR_PS3_Pos)                             /*!< PORT6 OMR: PS3 Mask                 */
#define PORT6_OMR_PS4_Pos                                     4                                                         /*!< PORT6 OMR: PS4 Position             */
#define PORT6_OMR_PS4_Msk                                     (0x01UL << PORT6_OMR_PS4_Pos)                             /*!< PORT6 OMR: PS4 Mask                 */
#define PORT6_OMR_PS5_Pos                                     5                                                         /*!< PORT6 OMR: PS5 Position             */
#define PORT6_OMR_PS5_Msk                                     (0x01UL << PORT6_OMR_PS5_Pos)                             /*!< PORT6 OMR: PS5 Mask                 */
#define PORT6_OMR_PS6_Pos                                     6                                                         /*!< PORT6 OMR: PS6 Position             */
#define PORT6_OMR_PS6_Msk                                     (0x01UL << PORT6_OMR_PS6_Pos)                             /*!< PORT6 OMR: PS6 Mask                 */
#define PORT6_OMR_PS7_Pos                                     7                                                         /*!< PORT6 OMR: PS7 Position             */
#define PORT6_OMR_PS7_Msk                                     (0x01UL << PORT6_OMR_PS7_Pos)                             /*!< PORT6 OMR: PS7 Mask                 */
#define PORT6_OMR_PS8_Pos                                     8                                                         /*!< PORT6 OMR: PS8 Position             */
#define PORT6_OMR_PS8_Msk                                     (0x01UL << PORT6_OMR_PS8_Pos)                             /*!< PORT6 OMR: PS8 Mask                 */
#define PORT6_OMR_PS9_Pos                                     9                                                         /*!< PORT6 OMR: PS9 Position             */
#define PORT6_OMR_PS9_Msk                                     (0x01UL << PORT6_OMR_PS9_Pos)                             /*!< PORT6 OMR: PS9 Mask                 */
#define PORT6_OMR_PS10_Pos                                    10                                                        /*!< PORT6 OMR: PS10 Position            */
#define PORT6_OMR_PS10_Msk                                    (0x01UL << PORT6_OMR_PS10_Pos)                            /*!< PORT6 OMR: PS10 Mask                */
#define PORT6_OMR_PS11_Pos                                    11                                                        /*!< PORT6 OMR: PS11 Position            */
#define PORT6_OMR_PS11_Msk                                    (0x01UL << PORT6_OMR_PS11_Pos)                            /*!< PORT6 OMR: PS11 Mask                */
#define PORT6_OMR_PS12_Pos                                    12                                                        /*!< PORT6 OMR: PS12 Position            */
#define PORT6_OMR_PS12_Msk                                    (0x01UL << PORT6_OMR_PS12_Pos)                            /*!< PORT6 OMR: PS12 Mask                */
#define PORT6_OMR_PS13_Pos                                    13                                                        /*!< PORT6 OMR: PS13 Position            */
#define PORT6_OMR_PS13_Msk                                    (0x01UL << PORT6_OMR_PS13_Pos)                            /*!< PORT6 OMR: PS13 Mask                */
#define PORT6_OMR_PS14_Pos                                    14                                                        /*!< PORT6 OMR: PS14 Position            */
#define PORT6_OMR_PS14_Msk                                    (0x01UL << PORT6_OMR_PS14_Pos)                            /*!< PORT6 OMR: PS14 Mask                */
#define PORT6_OMR_PS15_Pos                                    15                                                        /*!< PORT6 OMR: PS15 Position            */
#define PORT6_OMR_PS15_Msk                                    (0x01UL << PORT6_OMR_PS15_Pos)                            /*!< PORT6 OMR: PS15 Mask                */
#define PORT6_OMR_PR0_Pos                                     16                                                        /*!< PORT6 OMR: PR0 Position             */
#define PORT6_OMR_PR0_Msk                                     (0x01UL << PORT6_OMR_PR0_Pos)                             /*!< PORT6 OMR: PR0 Mask                 */
#define PORT6_OMR_PR1_Pos                                     17                                                        /*!< PORT6 OMR: PR1 Position             */
#define PORT6_OMR_PR1_Msk                                     (0x01UL << PORT6_OMR_PR1_Pos)                             /*!< PORT6 OMR: PR1 Mask                 */
#define PORT6_OMR_PR2_Pos                                     18                                                        /*!< PORT6 OMR: PR2 Position             */
#define PORT6_OMR_PR2_Msk                                     (0x01UL << PORT6_OMR_PR2_Pos)                             /*!< PORT6 OMR: PR2 Mask                 */
#define PORT6_OMR_PR3_Pos                                     19                                                        /*!< PORT6 OMR: PR3 Position             */
#define PORT6_OMR_PR3_Msk                                     (0x01UL << PORT6_OMR_PR3_Pos)                             /*!< PORT6 OMR: PR3 Mask                 */
#define PORT6_OMR_PR4_Pos                                     20                                                        /*!< PORT6 OMR: PR4 Position             */
#define PORT6_OMR_PR4_Msk                                     (0x01UL << PORT6_OMR_PR4_Pos)                             /*!< PORT6 OMR: PR4 Mask                 */
#define PORT6_OMR_PR5_Pos                                     21                                                        /*!< PORT6 OMR: PR5 Position             */
#define PORT6_OMR_PR5_Msk                                     (0x01UL << PORT6_OMR_PR5_Pos)                             /*!< PORT6 OMR: PR5 Mask                 */
#define PORT6_OMR_PR6_Pos                                     22                                                        /*!< PORT6 OMR: PR6 Position             */
#define PORT6_OMR_PR6_Msk                                     (0x01UL << PORT6_OMR_PR6_Pos)                             /*!< PORT6 OMR: PR6 Mask                 */
#define PORT6_OMR_PR7_Pos                                     23                                                        /*!< PORT6 OMR: PR7 Position             */
#define PORT6_OMR_PR7_Msk                                     (0x01UL << PORT6_OMR_PR7_Pos)                             /*!< PORT6 OMR: PR7 Mask                 */
#define PORT6_OMR_PR8_Pos                                     24                                                        /*!< PORT6 OMR: PR8 Position             */
#define PORT6_OMR_PR8_Msk                                     (0x01UL << PORT6_OMR_PR8_Pos)                             /*!< PORT6 OMR: PR8 Mask                 */
#define PORT6_OMR_PR9_Pos                                     25                                                        /*!< PORT6 OMR: PR9 Position             */
#define PORT6_OMR_PR9_Msk                                     (0x01UL << PORT6_OMR_PR9_Pos)                             /*!< PORT6 OMR: PR9 Mask                 */
#define PORT6_OMR_PR10_Pos                                    26                                                        /*!< PORT6 OMR: PR10 Position            */
#define PORT6_OMR_PR10_Msk                                    (0x01UL << PORT6_OMR_PR10_Pos)                            /*!< PORT6 OMR: PR10 Mask                */
#define PORT6_OMR_PR11_Pos                                    27                                                        /*!< PORT6 OMR: PR11 Position            */
#define PORT6_OMR_PR11_Msk                                    (0x01UL << PORT6_OMR_PR11_Pos)                            /*!< PORT6 OMR: PR11 Mask                */
#define PORT6_OMR_PR12_Pos                                    28                                                        /*!< PORT6 OMR: PR12 Position            */
#define PORT6_OMR_PR12_Msk                                    (0x01UL << PORT6_OMR_PR12_Pos)                            /*!< PORT6 OMR: PR12 Mask                */
#define PORT6_OMR_PR13_Pos                                    29                                                        /*!< PORT6 OMR: PR13 Position            */
#define PORT6_OMR_PR13_Msk                                    (0x01UL << PORT6_OMR_PR13_Pos)                            /*!< PORT6 OMR: PR13 Mask                */
#define PORT6_OMR_PR14_Pos                                    30                                                        /*!< PORT6 OMR: PR14 Position            */
#define PORT6_OMR_PR14_Msk                                    (0x01UL << PORT6_OMR_PR14_Pos)                            /*!< PORT6 OMR: PR14 Mask                */
#define PORT6_OMR_PR15_Pos                                    31                                                        /*!< PORT6 OMR: PR15 Position            */
#define PORT6_OMR_PR15_Msk                                    (0x01UL << PORT6_OMR_PR15_Pos)                            /*!< PORT6 OMR: PR15 Mask                */

// ---------------------------------------  PORT6_IOCR0  ------------------------------------------
#define PORT6_IOCR0_PC0_Pos                                   3                                                         /*!< PORT6 IOCR0: PC0 Position           */
#define PORT6_IOCR0_PC0_Msk                                   (0x1fUL << PORT6_IOCR0_PC0_Pos)                           /*!< PORT6 IOCR0: PC0 Mask               */
#define PORT6_IOCR0_PC1_Pos                                   11                                                        /*!< PORT6 IOCR0: PC1 Position           */
#define PORT6_IOCR0_PC1_Msk                                   (0x1fUL << PORT6_IOCR0_PC1_Pos)                           /*!< PORT6 IOCR0: PC1 Mask               */
#define PORT6_IOCR0_PC2_Pos                                   19                                                        /*!< PORT6 IOCR0: PC2 Position           */
#define PORT6_IOCR0_PC2_Msk                                   (0x1fUL << PORT6_IOCR0_PC2_Pos)                           /*!< PORT6 IOCR0: PC2 Mask               */
#define PORT6_IOCR0_PC3_Pos                                   27                                                        /*!< PORT6 IOCR0: PC3 Position           */
#define PORT6_IOCR0_PC3_Msk                                   (0x1fUL << PORT6_IOCR0_PC3_Pos)                           /*!< PORT6 IOCR0: PC3 Mask               */

// ---------------------------------------  PORT6_IOCR4  ------------------------------------------
#define PORT6_IOCR4_PC4_Pos                                   3                                                         /*!< PORT6 IOCR4: PC4 Position           */
#define PORT6_IOCR4_PC4_Msk                                   (0x1fUL << PORT6_IOCR4_PC4_Pos)                           /*!< PORT6 IOCR4: PC4 Mask               */
#define PORT6_IOCR4_PC5_Pos                                   11                                                        /*!< PORT6 IOCR4: PC5 Position           */
#define PORT6_IOCR4_PC5_Msk                                   (0x1fUL << PORT6_IOCR4_PC5_Pos)                           /*!< PORT6 IOCR4: PC5 Mask               */
#define PORT6_IOCR4_PC6_Pos                                   19                                                        /*!< PORT6 IOCR4: PC6 Position           */
#define PORT6_IOCR4_PC6_Msk                                   (0x1fUL << PORT6_IOCR4_PC6_Pos)                           /*!< PORT6 IOCR4: PC6 Mask               */
#define PORT6_IOCR4_PC7_Pos                                   27                                                        /*!< PORT6 IOCR4: PC7 Position           */
#define PORT6_IOCR4_PC7_Msk                                   (0x1fUL << PORT6_IOCR4_PC7_Pos)                           /*!< PORT6 IOCR4: PC7 Mask               */

// ----------------------------------------  PORT6_IN  --------------------------------------------
#define PORT6_IN_P0_Pos                                       0                                                         /*!< PORT6 IN: P0 Position               */
#define PORT6_IN_P0_Msk                                       (0x01UL << PORT6_IN_P0_Pos)                               /*!< PORT6 IN: P0 Mask                   */
#define PORT6_IN_P1_Pos                                       1                                                         /*!< PORT6 IN: P1 Position               */
#define PORT6_IN_P1_Msk                                       (0x01UL << PORT6_IN_P1_Pos)                               /*!< PORT6 IN: P1 Mask                   */
#define PORT6_IN_P2_Pos                                       2                                                         /*!< PORT6 IN: P2 Position               */
#define PORT6_IN_P2_Msk                                       (0x01UL << PORT6_IN_P2_Pos)                               /*!< PORT6 IN: P2 Mask                   */
#define PORT6_IN_P3_Pos                                       3                                                         /*!< PORT6 IN: P3 Position               */
#define PORT6_IN_P3_Msk                                       (0x01UL << PORT6_IN_P3_Pos)                               /*!< PORT6 IN: P3 Mask                   */
#define PORT6_IN_P4_Pos                                       4                                                         /*!< PORT6 IN: P4 Position               */
#define PORT6_IN_P4_Msk                                       (0x01UL << PORT6_IN_P4_Pos)                               /*!< PORT6 IN: P4 Mask                   */
#define PORT6_IN_P5_Pos                                       5                                                         /*!< PORT6 IN: P5 Position               */
#define PORT6_IN_P5_Msk                                       (0x01UL << PORT6_IN_P5_Pos)                               /*!< PORT6 IN: P5 Mask                   */
#define PORT6_IN_P6_Pos                                       6                                                         /*!< PORT6 IN: P6 Position               */
#define PORT6_IN_P6_Msk                                       (0x01UL << PORT6_IN_P6_Pos)                               /*!< PORT6 IN: P6 Mask                   */
#define PORT6_IN_P7_Pos                                       7                                                         /*!< PORT6 IN: P7 Position               */
#define PORT6_IN_P7_Msk                                       (0x01UL << PORT6_IN_P7_Pos)                               /*!< PORT6 IN: P7 Mask                   */
#define PORT6_IN_P8_Pos                                       8                                                         /*!< PORT6 IN: P8 Position               */
#define PORT6_IN_P8_Msk                                       (0x01UL << PORT6_IN_P8_Pos)                               /*!< PORT6 IN: P8 Mask                   */
#define PORT6_IN_P9_Pos                                       9                                                         /*!< PORT6 IN: P9 Position               */
#define PORT6_IN_P9_Msk                                       (0x01UL << PORT6_IN_P9_Pos)                               /*!< PORT6 IN: P9 Mask                   */
#define PORT6_IN_P10_Pos                                      10                                                        /*!< PORT6 IN: P10 Position              */
#define PORT6_IN_P10_Msk                                      (0x01UL << PORT6_IN_P10_Pos)                              /*!< PORT6 IN: P10 Mask                  */
#define PORT6_IN_P11_Pos                                      11                                                        /*!< PORT6 IN: P11 Position              */
#define PORT6_IN_P11_Msk                                      (0x01UL << PORT6_IN_P11_Pos)                              /*!< PORT6 IN: P11 Mask                  */
#define PORT6_IN_P12_Pos                                      12                                                        /*!< PORT6 IN: P12 Position              */
#define PORT6_IN_P12_Msk                                      (0x01UL << PORT6_IN_P12_Pos)                              /*!< PORT6 IN: P12 Mask                  */
#define PORT6_IN_P13_Pos                                      13                                                        /*!< PORT6 IN: P13 Position              */
#define PORT6_IN_P13_Msk                                      (0x01UL << PORT6_IN_P13_Pos)                              /*!< PORT6 IN: P13 Mask                  */
#define PORT6_IN_P14_Pos                                      14                                                        /*!< PORT6 IN: P14 Position              */
#define PORT6_IN_P14_Msk                                      (0x01UL << PORT6_IN_P14_Pos)                              /*!< PORT6 IN: P14 Mask                  */
#define PORT6_IN_P15_Pos                                      15                                                        /*!< PORT6 IN: P15 Position              */
#define PORT6_IN_P15_Msk                                      (0x01UL << PORT6_IN_P15_Pos)                              /*!< PORT6 IN: P15 Mask                  */

// ---------------------------------------  PORT6_PDR0  -------------------------------------------
#define PORT6_PDR0_PD0_Pos                                    0                                                         /*!< PORT6 PDR0: PD0 Position            */
#define PORT6_PDR0_PD0_Msk                                    (0x07UL << PORT6_PDR0_PD0_Pos)                            /*!< PORT6 PDR0: PD0 Mask                */
#define PORT6_PDR0_PD1_Pos                                    4                                                         /*!< PORT6 PDR0: PD1 Position            */
#define PORT6_PDR0_PD1_Msk                                    (0x07UL << PORT6_PDR0_PD1_Pos)                            /*!< PORT6 PDR0: PD1 Mask                */
#define PORT6_PDR0_PD2_Pos                                    8                                                         /*!< PORT6 PDR0: PD2 Position            */
#define PORT6_PDR0_PD2_Msk                                    (0x07UL << PORT6_PDR0_PD2_Pos)                            /*!< PORT6 PDR0: PD2 Mask                */
#define PORT6_PDR0_PD3_Pos                                    12                                                        /*!< PORT6 PDR0: PD3 Position            */
#define PORT6_PDR0_PD3_Msk                                    (0x07UL << PORT6_PDR0_PD3_Pos)                            /*!< PORT6 PDR0: PD3 Mask                */
#define PORT6_PDR0_PD4_Pos                                    16                                                        /*!< PORT6 PDR0: PD4 Position            */
#define PORT6_PDR0_PD4_Msk                                    (0x07UL << PORT6_PDR0_PD4_Pos)                            /*!< PORT6 PDR0: PD4 Mask                */
#define PORT6_PDR0_PD5_Pos                                    20                                                        /*!< PORT6 PDR0: PD5 Position            */
#define PORT6_PDR0_PD5_Msk                                    (0x07UL << PORT6_PDR0_PD5_Pos)                            /*!< PORT6 PDR0: PD5 Mask                */
#define PORT6_PDR0_PD6_Pos                                    24                                                        /*!< PORT6 PDR0: PD6 Position            */
#define PORT6_PDR0_PD6_Msk                                    (0x07UL << PORT6_PDR0_PD6_Pos)                            /*!< PORT6 PDR0: PD6 Mask                */
#define PORT6_PDR0_PD7_Pos                                    28                                                        /*!< PORT6 PDR0: PD7 Position            */
#define PORT6_PDR0_PD7_Msk                                    (0x07UL << PORT6_PDR0_PD7_Pos)                            /*!< PORT6 PDR0: PD7 Mask                */

// ---------------------------------------  PORT6_PDISC  ------------------------------------------
#define PORT6_PDISC_PDIS0_Pos                                 0                                                         /*!< PORT6 PDISC: PDIS0 Position         */
#define PORT6_PDISC_PDIS0_Msk                                 (0x01UL << PORT6_PDISC_PDIS0_Pos)                         /*!< PORT6 PDISC: PDIS0 Mask             */
#define PORT6_PDISC_PDIS1_Pos                                 1                                                         /*!< PORT6 PDISC: PDIS1 Position         */
#define PORT6_PDISC_PDIS1_Msk                                 (0x01UL << PORT6_PDISC_PDIS1_Pos)                         /*!< PORT6 PDISC: PDIS1 Mask             */
#define PORT6_PDISC_PDIS2_Pos                                 2                                                         /*!< PORT6 PDISC: PDIS2 Position         */
#define PORT6_PDISC_PDIS2_Msk                                 (0x01UL << PORT6_PDISC_PDIS2_Pos)                         /*!< PORT6 PDISC: PDIS2 Mask             */
#define PORT6_PDISC_PDIS3_Pos                                 3                                                         /*!< PORT6 PDISC: PDIS3 Position         */
#define PORT6_PDISC_PDIS3_Msk                                 (0x01UL << PORT6_PDISC_PDIS3_Pos)                         /*!< PORT6 PDISC: PDIS3 Mask             */
#define PORT6_PDISC_PDIS4_Pos                                 4                                                         /*!< PORT6 PDISC: PDIS4 Position         */
#define PORT6_PDISC_PDIS4_Msk                                 (0x01UL << PORT6_PDISC_PDIS4_Pos)                         /*!< PORT6 PDISC: PDIS4 Mask             */
#define PORT6_PDISC_PDIS5_Pos                                 5                                                         /*!< PORT6 PDISC: PDIS5 Position         */
#define PORT6_PDISC_PDIS5_Msk                                 (0x01UL << PORT6_PDISC_PDIS5_Pos)                         /*!< PORT6 PDISC: PDIS5 Mask             */
#define PORT6_PDISC_PDIS6_Pos                                 6                                                         /*!< PORT6 PDISC: PDIS6 Position         */
#define PORT6_PDISC_PDIS6_Msk                                 (0x01UL << PORT6_PDISC_PDIS6_Pos)                         /*!< PORT6 PDISC: PDIS6 Mask             */
#define PORT6_PDISC_PDIS7_Pos                                 7                                                         /*!< PORT6 PDISC: PDIS7 Position         */
#define PORT6_PDISC_PDIS7_Msk                                 (0x01UL << PORT6_PDISC_PDIS7_Pos)                         /*!< PORT6 PDISC: PDIS7 Mask             */
#define PORT6_PDISC_PDIS8_Pos                                 8                                                         /*!< PORT6 PDISC: PDIS8 Position         */
#define PORT6_PDISC_PDIS8_Msk                                 (0x01UL << PORT6_PDISC_PDIS8_Pos)                         /*!< PORT6 PDISC: PDIS8 Mask             */
#define PORT6_PDISC_PDIS9_Pos                                 9                                                         /*!< PORT6 PDISC: PDIS9 Position         */
#define PORT6_PDISC_PDIS9_Msk                                 (0x01UL << PORT6_PDISC_PDIS9_Pos)                         /*!< PORT6 PDISC: PDIS9 Mask             */
#define PORT6_PDISC_PDIS10_Pos                                10                                                        /*!< PORT6 PDISC: PDIS10 Position        */
#define PORT6_PDISC_PDIS10_Msk                                (0x01UL << PORT6_PDISC_PDIS10_Pos)                        /*!< PORT6 PDISC: PDIS10 Mask            */
#define PORT6_PDISC_PDIS11_Pos                                11                                                        /*!< PORT6 PDISC: PDIS11 Position        */
#define PORT6_PDISC_PDIS11_Msk                                (0x01UL << PORT6_PDISC_PDIS11_Pos)                        /*!< PORT6 PDISC: PDIS11 Mask            */
#define PORT6_PDISC_PDIS12_Pos                                12                                                        /*!< PORT6 PDISC: PDIS12 Position        */
#define PORT6_PDISC_PDIS12_Msk                                (0x01UL << PORT6_PDISC_PDIS12_Pos)                        /*!< PORT6 PDISC: PDIS12 Mask            */
#define PORT6_PDISC_PDIS13_Pos                                13                                                        /*!< PORT6 PDISC: PDIS13 Position        */
#define PORT6_PDISC_PDIS13_Msk                                (0x01UL << PORT6_PDISC_PDIS13_Pos)                        /*!< PORT6 PDISC: PDIS13 Mask            */
#define PORT6_PDISC_PDIS14_Pos                                14                                                        /*!< PORT6 PDISC: PDIS14 Position        */
#define PORT6_PDISC_PDIS14_Msk                                (0x01UL << PORT6_PDISC_PDIS14_Pos)                        /*!< PORT6 PDISC: PDIS14 Mask            */
#define PORT6_PDISC_PDIS15_Pos                                15                                                        /*!< PORT6 PDISC: PDIS15 Position        */
#define PORT6_PDISC_PDIS15_Msk                                (0x01UL << PORT6_PDISC_PDIS15_Pos)                        /*!< PORT6 PDISC: PDIS15 Mask            */

// ----------------------------------------  PORT6_PPS  -------------------------------------------
#define PORT6_PPS_PPS0_Pos                                    0                                                         /*!< PORT6 PPS: PPS0 Position            */
#define PORT6_PPS_PPS0_Msk                                    (0x01UL << PORT6_PPS_PPS0_Pos)                            /*!< PORT6 PPS: PPS0 Mask                */
#define PORT6_PPS_PPS1_Pos                                    1                                                         /*!< PORT6 PPS: PPS1 Position            */
#define PORT6_PPS_PPS1_Msk                                    (0x01UL << PORT6_PPS_PPS1_Pos)                            /*!< PORT6 PPS: PPS1 Mask                */
#define PORT6_PPS_PPS2_Pos                                    2                                                         /*!< PORT6 PPS: PPS2 Position            */
#define PORT6_PPS_PPS2_Msk                                    (0x01UL << PORT6_PPS_PPS2_Pos)                            /*!< PORT6 PPS: PPS2 Mask                */
#define PORT6_PPS_PPS3_Pos                                    3                                                         /*!< PORT6 PPS: PPS3 Position            */
#define PORT6_PPS_PPS3_Msk                                    (0x01UL << PORT6_PPS_PPS3_Pos)                            /*!< PORT6 PPS: PPS3 Mask                */
#define PORT6_PPS_PPS4_Pos                                    4                                                         /*!< PORT6 PPS: PPS4 Position            */
#define PORT6_PPS_PPS4_Msk                                    (0x01UL << PORT6_PPS_PPS4_Pos)                            /*!< PORT6 PPS: PPS4 Mask                */
#define PORT6_PPS_PPS5_Pos                                    5                                                         /*!< PORT6 PPS: PPS5 Position            */
#define PORT6_PPS_PPS5_Msk                                    (0x01UL << PORT6_PPS_PPS5_Pos)                            /*!< PORT6 PPS: PPS5 Mask                */
#define PORT6_PPS_PPS6_Pos                                    6                                                         /*!< PORT6 PPS: PPS6 Position            */
#define PORT6_PPS_PPS6_Msk                                    (0x01UL << PORT6_PPS_PPS6_Pos)                            /*!< PORT6 PPS: PPS6 Mask                */
#define PORT6_PPS_PPS7_Pos                                    7                                                         /*!< PORT6 PPS: PPS7 Position            */
#define PORT6_PPS_PPS7_Msk                                    (0x01UL << PORT6_PPS_PPS7_Pos)                            /*!< PORT6 PPS: PPS7 Mask                */
#define PORT6_PPS_PPS8_Pos                                    8                                                         /*!< PORT6 PPS: PPS8 Position            */
#define PORT6_PPS_PPS8_Msk                                    (0x01UL << PORT6_PPS_PPS8_Pos)                            /*!< PORT6 PPS: PPS8 Mask                */
#define PORT6_PPS_PPS9_Pos                                    9                                                         /*!< PORT6 PPS: PPS9 Position            */
#define PORT6_PPS_PPS9_Msk                                    (0x01UL << PORT6_PPS_PPS9_Pos)                            /*!< PORT6 PPS: PPS9 Mask                */
#define PORT6_PPS_PPS10_Pos                                   10                                                        /*!< PORT6 PPS: PPS10 Position           */
#define PORT6_PPS_PPS10_Msk                                   (0x01UL << PORT6_PPS_PPS10_Pos)                           /*!< PORT6 PPS: PPS10 Mask               */
#define PORT6_PPS_PPS11_Pos                                   11                                                        /*!< PORT6 PPS: PPS11 Position           */
#define PORT6_PPS_PPS11_Msk                                   (0x01UL << PORT6_PPS_PPS11_Pos)                           /*!< PORT6 PPS: PPS11 Mask               */
#define PORT6_PPS_PPS12_Pos                                   12                                                        /*!< PORT6 PPS: PPS12 Position           */
#define PORT6_PPS_PPS12_Msk                                   (0x01UL << PORT6_PPS_PPS12_Pos)                           /*!< PORT6 PPS: PPS12 Mask               */
#define PORT6_PPS_PPS13_Pos                                   13                                                        /*!< PORT6 PPS: PPS13 Position           */
#define PORT6_PPS_PPS13_Msk                                   (0x01UL << PORT6_PPS_PPS13_Pos)                           /*!< PORT6 PPS: PPS13 Mask               */
#define PORT6_PPS_PPS14_Pos                                   14                                                        /*!< PORT6 PPS: PPS14 Position           */
#define PORT6_PPS_PPS14_Msk                                   (0x01UL << PORT6_PPS_PPS14_Pos)                           /*!< PORT6 PPS: PPS14 Mask               */
#define PORT6_PPS_PPS15_Pos                                   15                                                        /*!< PORT6 PPS: PPS15 Position           */
#define PORT6_PPS_PPS15_Msk                                   (0x01UL << PORT6_PPS_PPS15_Pos)                           /*!< PORT6 PPS: PPS15 Mask               */

// ---------------------------------------  PORT6_HWSEL  ------------------------------------------
#define PORT6_HWSEL_HW0_Pos                                   0                                                         /*!< PORT6 HWSEL: HW0 Position           */
#define PORT6_HWSEL_HW0_Msk                                   (0x03UL << PORT6_HWSEL_HW0_Pos)                           /*!< PORT6 HWSEL: HW0 Mask               */
#define PORT6_HWSEL_HW1_Pos                                   2                                                         /*!< PORT6 HWSEL: HW1 Position           */
#define PORT6_HWSEL_HW1_Msk                                   (0x03UL << PORT6_HWSEL_HW1_Pos)                           /*!< PORT6 HWSEL: HW1 Mask               */
#define PORT6_HWSEL_HW2_Pos                                   4                                                         /*!< PORT6 HWSEL: HW2 Position           */
#define PORT6_HWSEL_HW2_Msk                                   (0x03UL << PORT6_HWSEL_HW2_Pos)                           /*!< PORT6 HWSEL: HW2 Mask               */
#define PORT6_HWSEL_HW3_Pos                                   6                                                         /*!< PORT6 HWSEL: HW3 Position           */
#define PORT6_HWSEL_HW3_Msk                                   (0x03UL << PORT6_HWSEL_HW3_Pos)                           /*!< PORT6 HWSEL: HW3 Mask               */
#define PORT6_HWSEL_HW4_Pos                                   8                                                         /*!< PORT6 HWSEL: HW4 Position           */
#define PORT6_HWSEL_HW4_Msk                                   (0x03UL << PORT6_HWSEL_HW4_Pos)                           /*!< PORT6 HWSEL: HW4 Mask               */
#define PORT6_HWSEL_HW5_Pos                                   10                                                        /*!< PORT6 HWSEL: HW5 Position           */
#define PORT6_HWSEL_HW5_Msk                                   (0x03UL << PORT6_HWSEL_HW5_Pos)                           /*!< PORT6 HWSEL: HW5 Mask               */
#define PORT6_HWSEL_HW6_Pos                                   12                                                        /*!< PORT6 HWSEL: HW6 Position           */
#define PORT6_HWSEL_HW6_Msk                                   (0x03UL << PORT6_HWSEL_HW6_Pos)                           /*!< PORT6 HWSEL: HW6 Mask               */
#define PORT6_HWSEL_HW7_Pos                                   14                                                        /*!< PORT6 HWSEL: HW7 Position           */
#define PORT6_HWSEL_HW7_Msk                                   (0x03UL << PORT6_HWSEL_HW7_Pos)                           /*!< PORT6 HWSEL: HW7 Mask               */
#define PORT6_HWSEL_HW8_Pos                                   16                                                        /*!< PORT6 HWSEL: HW8 Position           */
#define PORT6_HWSEL_HW8_Msk                                   (0x03UL << PORT6_HWSEL_HW8_Pos)                           /*!< PORT6 HWSEL: HW8 Mask               */
#define PORT6_HWSEL_HW9_Pos                                   18                                                        /*!< PORT6 HWSEL: HW9 Position           */
#define PORT6_HWSEL_HW9_Msk                                   (0x03UL << PORT6_HWSEL_HW9_Pos)                           /*!< PORT6 HWSEL: HW9 Mask               */
#define PORT6_HWSEL_HW10_Pos                                  20                                                        /*!< PORT6 HWSEL: HW10 Position          */
#define PORT6_HWSEL_HW10_Msk                                  (0x03UL << PORT6_HWSEL_HW10_Pos)                          /*!< PORT6 HWSEL: HW10 Mask              */
#define PORT6_HWSEL_HW11_Pos                                  22                                                        /*!< PORT6 HWSEL: HW11 Position          */
#define PORT6_HWSEL_HW11_Msk                                  (0x03UL << PORT6_HWSEL_HW11_Pos)                          /*!< PORT6 HWSEL: HW11 Mask              */
#define PORT6_HWSEL_HW12_Pos                                  24                                                        /*!< PORT6 HWSEL: HW12 Position          */
#define PORT6_HWSEL_HW12_Msk                                  (0x03UL << PORT6_HWSEL_HW12_Pos)                          /*!< PORT6 HWSEL: HW12 Mask              */
#define PORT6_HWSEL_HW13_Pos                                  26                                                        /*!< PORT6 HWSEL: HW13 Position          */
#define PORT6_HWSEL_HW13_Msk                                  (0x03UL << PORT6_HWSEL_HW13_Pos)                          /*!< PORT6 HWSEL: HW13 Mask              */
#define PORT6_HWSEL_HW14_Pos                                  28                                                        /*!< PORT6 HWSEL: HW14 Position          */
#define PORT6_HWSEL_HW14_Msk                                  (0x03UL << PORT6_HWSEL_HW14_Pos)                          /*!< PORT6 HWSEL: HW14 Mask              */
#define PORT6_HWSEL_HW15_Pos                                  30                                                        /*!< PORT6 HWSEL: HW15 Position          */
#define PORT6_HWSEL_HW15_Msk                                  (0x03UL << PORT6_HWSEL_HW15_Pos)                          /*!< PORT6 HWSEL: HW15 Mask              */


// ------------------------------------------------------------------------------------------------
// -----                                PORT14 Position & Mask                                -----
// ------------------------------------------------------------------------------------------------


// ---------------------------------------  PORT14_OUT  -------------------------------------------
#define PORT14_OUT_P0_Pos                                     0                                                         /*!< PORT14 OUT: P0 Position             */
#define PORT14_OUT_P0_Msk                                     (0x01UL << PORT14_OUT_P0_Pos)                             /*!< PORT14 OUT: P0 Mask                 */
#define PORT14_OUT_P1_Pos                                     1                                                         /*!< PORT14 OUT: P1 Position             */
#define PORT14_OUT_P1_Msk                                     (0x01UL << PORT14_OUT_P1_Pos)                             /*!< PORT14 OUT: P1 Mask                 */
#define PORT14_OUT_P2_Pos                                     2                                                         /*!< PORT14 OUT: P2 Position             */
#define PORT14_OUT_P2_Msk                                     (0x01UL << PORT14_OUT_P2_Pos)                             /*!< PORT14 OUT: P2 Mask                 */
#define PORT14_OUT_P3_Pos                                     3                                                         /*!< PORT14 OUT: P3 Position             */
#define PORT14_OUT_P3_Msk                                     (0x01UL << PORT14_OUT_P3_Pos)                             /*!< PORT14 OUT: P3 Mask                 */
#define PORT14_OUT_P4_Pos                                     4                                                         /*!< PORT14 OUT: P4 Position             */
#define PORT14_OUT_P4_Msk                                     (0x01UL << PORT14_OUT_P4_Pos)                             /*!< PORT14 OUT: P4 Mask                 */
#define PORT14_OUT_P5_Pos                                     5                                                         /*!< PORT14 OUT: P5 Position             */
#define PORT14_OUT_P5_Msk                                     (0x01UL << PORT14_OUT_P5_Pos)                             /*!< PORT14 OUT: P5 Mask                 */
#define PORT14_OUT_P6_Pos                                     6                                                         /*!< PORT14 OUT: P6 Position             */
#define PORT14_OUT_P6_Msk                                     (0x01UL << PORT14_OUT_P6_Pos)                             /*!< PORT14 OUT: P6 Mask                 */
#define PORT14_OUT_P7_Pos                                     7                                                         /*!< PORT14 OUT: P7 Position             */
#define PORT14_OUT_P7_Msk                                     (0x01UL << PORT14_OUT_P7_Pos)                             /*!< PORT14 OUT: P7 Mask                 */
#define PORT14_OUT_P8_Pos                                     8                                                         /*!< PORT14 OUT: P8 Position             */
#define PORT14_OUT_P8_Msk                                     (0x01UL << PORT14_OUT_P8_Pos)                             /*!< PORT14 OUT: P8 Mask                 */
#define PORT14_OUT_P9_Pos                                     9                                                         /*!< PORT14 OUT: P9 Position             */
#define PORT14_OUT_P9_Msk                                     (0x01UL << PORT14_OUT_P9_Pos)                             /*!< PORT14 OUT: P9 Mask                 */
#define PORT14_OUT_P10_Pos                                    10                                                        /*!< PORT14 OUT: P10 Position            */
#define PORT14_OUT_P10_Msk                                    (0x01UL << PORT14_OUT_P10_Pos)                            /*!< PORT14 OUT: P10 Mask                */
#define PORT14_OUT_P11_Pos                                    11                                                        /*!< PORT14 OUT: P11 Position            */
#define PORT14_OUT_P11_Msk                                    (0x01UL << PORT14_OUT_P11_Pos)                            /*!< PORT14 OUT: P11 Mask                */
#define PORT14_OUT_P12_Pos                                    12                                                        /*!< PORT14 OUT: P12 Position            */
#define PORT14_OUT_P12_Msk                                    (0x01UL << PORT14_OUT_P12_Pos)                            /*!< PORT14 OUT: P12 Mask                */
#define PORT14_OUT_P13_Pos                                    13                                                        /*!< PORT14 OUT: P13 Position            */
#define PORT14_OUT_P13_Msk                                    (0x01UL << PORT14_OUT_P13_Pos)                            /*!< PORT14 OUT: P13 Mask                */
#define PORT14_OUT_P14_Pos                                    14                                                        /*!< PORT14 OUT: P14 Position            */
#define PORT14_OUT_P14_Msk                                    (0x01UL << PORT14_OUT_P14_Pos)                            /*!< PORT14 OUT: P14 Mask                */
#define PORT14_OUT_P15_Pos                                    15                                                        /*!< PORT14 OUT: P15 Position            */
#define PORT14_OUT_P15_Msk                                    (0x01UL << PORT14_OUT_P15_Pos)                            /*!< PORT14 OUT: P15 Mask                */

// ---------------------------------------  PORT14_OMR  -------------------------------------------
#define PORT14_OMR_PS0_Pos                                    0                                                         /*!< PORT14 OMR: PS0 Position            */
#define PORT14_OMR_PS0_Msk                                    (0x01UL << PORT14_OMR_PS0_Pos)                            /*!< PORT14 OMR: PS0 Mask                */
#define PORT14_OMR_PS1_Pos                                    1                                                         /*!< PORT14 OMR: PS1 Position            */
#define PORT14_OMR_PS1_Msk                                    (0x01UL << PORT14_OMR_PS1_Pos)                            /*!< PORT14 OMR: PS1 Mask                */
#define PORT14_OMR_PS2_Pos                                    2                                                         /*!< PORT14 OMR: PS2 Position            */
#define PORT14_OMR_PS2_Msk                                    (0x01UL << PORT14_OMR_PS2_Pos)                            /*!< PORT14 OMR: PS2 Mask                */
#define PORT14_OMR_PS3_Pos                                    3                                                         /*!< PORT14 OMR: PS3 Position            */
#define PORT14_OMR_PS3_Msk                                    (0x01UL << PORT14_OMR_PS3_Pos)                            /*!< PORT14 OMR: PS3 Mask                */
#define PORT14_OMR_PS4_Pos                                    4                                                         /*!< PORT14 OMR: PS4 Position            */
#define PORT14_OMR_PS4_Msk                                    (0x01UL << PORT14_OMR_PS4_Pos)                            /*!< PORT14 OMR: PS4 Mask                */
#define PORT14_OMR_PS5_Pos                                    5                                                         /*!< PORT14 OMR: PS5 Position            */
#define PORT14_OMR_PS5_Msk                                    (0x01UL << PORT14_OMR_PS5_Pos)                            /*!< PORT14 OMR: PS5 Mask                */
#define PORT14_OMR_PS6_Pos                                    6                                                         /*!< PORT14 OMR: PS6 Position            */
#define PORT14_OMR_PS6_Msk                                    (0x01UL << PORT14_OMR_PS6_Pos)                            /*!< PORT14 OMR: PS6 Mask                */
#define PORT14_OMR_PS7_Pos                                    7                                                         /*!< PORT14 OMR: PS7 Position            */
#define PORT14_OMR_PS7_Msk                                    (0x01UL << PORT14_OMR_PS7_Pos)                            /*!< PORT14 OMR: PS7 Mask                */
#define PORT14_OMR_PS8_Pos                                    8                                                         /*!< PORT14 OMR: PS8 Position            */
#define PORT14_OMR_PS8_Msk                                    (0x01UL << PORT14_OMR_PS8_Pos)                            /*!< PORT14 OMR: PS8 Mask                */
#define PORT14_OMR_PS9_Pos                                    9                                                         /*!< PORT14 OMR: PS9 Position            */
#define PORT14_OMR_PS9_Msk                                    (0x01UL << PORT14_OMR_PS9_Pos)                            /*!< PORT14 OMR: PS9 Mask                */
#define PORT14_OMR_PS10_Pos                                   10                                                        /*!< PORT14 OMR: PS10 Position           */
#define PORT14_OMR_PS10_Msk                                   (0x01UL << PORT14_OMR_PS10_Pos)                           /*!< PORT14 OMR: PS10 Mask               */
#define PORT14_OMR_PS11_Pos                                   11                                                        /*!< PORT14 OMR: PS11 Position           */
#define PORT14_OMR_PS11_Msk                                   (0x01UL << PORT14_OMR_PS11_Pos)                           /*!< PORT14 OMR: PS11 Mask               */
#define PORT14_OMR_PS12_Pos                                   12                                                        /*!< PORT14 OMR: PS12 Position           */
#define PORT14_OMR_PS12_Msk                                   (0x01UL << PORT14_OMR_PS12_Pos)                           /*!< PORT14 OMR: PS12 Mask               */
#define PORT14_OMR_PS13_Pos                                   13                                                        /*!< PORT14 OMR: PS13 Position           */
#define PORT14_OMR_PS13_Msk                                   (0x01UL << PORT14_OMR_PS13_Pos)                           /*!< PORT14 OMR: PS13 Mask               */
#define PORT14_OMR_PS14_Pos                                   14                                                        /*!< PORT14 OMR: PS14 Position           */
#define PORT14_OMR_PS14_Msk                                   (0x01UL << PORT14_OMR_PS14_Pos)                           /*!< PORT14 OMR: PS14 Mask               */
#define PORT14_OMR_PS15_Pos                                   15                                                        /*!< PORT14 OMR: PS15 Position           */
#define PORT14_OMR_PS15_Msk                                   (0x01UL << PORT14_OMR_PS15_Pos)                           /*!< PORT14 OMR: PS15 Mask               */
#define PORT14_OMR_PR0_Pos                                    16                                                        /*!< PORT14 OMR: PR0 Position            */
#define PORT14_OMR_PR0_Msk                                    (0x01UL << PORT14_OMR_PR0_Pos)                            /*!< PORT14 OMR: PR0 Mask                */
#define PORT14_OMR_PR1_Pos                                    17                                                        /*!< PORT14 OMR: PR1 Position            */
#define PORT14_OMR_PR1_Msk                                    (0x01UL << PORT14_OMR_PR1_Pos)                            /*!< PORT14 OMR: PR1 Mask                */
#define PORT14_OMR_PR2_Pos                                    18                                                        /*!< PORT14 OMR: PR2 Position            */
#define PORT14_OMR_PR2_Msk                                    (0x01UL << PORT14_OMR_PR2_Pos)                            /*!< PORT14 OMR: PR2 Mask                */
#define PORT14_OMR_PR3_Pos                                    19                                                        /*!< PORT14 OMR: PR3 Position            */
#define PORT14_OMR_PR3_Msk                                    (0x01UL << PORT14_OMR_PR3_Pos)                            /*!< PORT14 OMR: PR3 Mask                */
#define PORT14_OMR_PR4_Pos                                    20                                                        /*!< PORT14 OMR: PR4 Position            */
#define PORT14_OMR_PR4_Msk                                    (0x01UL << PORT14_OMR_PR4_Pos)                            /*!< PORT14 OMR: PR4 Mask                */
#define PORT14_OMR_PR5_Pos                                    21                                                        /*!< PORT14 OMR: PR5 Position            */
#define PORT14_OMR_PR5_Msk                                    (0x01UL << PORT14_OMR_PR5_Pos)                            /*!< PORT14 OMR: PR5 Mask                */
#define PORT14_OMR_PR6_Pos                                    22                                                        /*!< PORT14 OMR: PR6 Position            */
#define PORT14_OMR_PR6_Msk                                    (0x01UL << PORT14_OMR_PR6_Pos)                            /*!< PORT14 OMR: PR6 Mask                */
#define PORT14_OMR_PR7_Pos                                    23                                                        /*!< PORT14 OMR: PR7 Position            */
#define PORT14_OMR_PR7_Msk                                    (0x01UL << PORT14_OMR_PR7_Pos)                            /*!< PORT14 OMR: PR7 Mask                */
#define PORT14_OMR_PR8_Pos                                    24                                                        /*!< PORT14 OMR: PR8 Position            */
#define PORT14_OMR_PR8_Msk                                    (0x01UL << PORT14_OMR_PR8_Pos)                            /*!< PORT14 OMR: PR8 Mask                */
#define PORT14_OMR_PR9_Pos                                    25                                                        /*!< PORT14 OMR: PR9 Position            */
#define PORT14_OMR_PR9_Msk                                    (0x01UL << PORT14_OMR_PR9_Pos)                            /*!< PORT14 OMR: PR9 Mask                */
#define PORT14_OMR_PR10_Pos                                   26                                                        /*!< PORT14 OMR: PR10 Position           */
#define PORT14_OMR_PR10_Msk                                   (0x01UL << PORT14_OMR_PR10_Pos)                           /*!< PORT14 OMR: PR10 Mask               */
#define PORT14_OMR_PR11_Pos                                   27                                                        /*!< PORT14 OMR: PR11 Position           */
#define PORT14_OMR_PR11_Msk                                   (0x01UL << PORT14_OMR_PR11_Pos)                           /*!< PORT14 OMR: PR11 Mask               */
#define PORT14_OMR_PR12_Pos                                   28                                                        /*!< PORT14 OMR: PR12 Position           */
#define PORT14_OMR_PR12_Msk                                   (0x01UL << PORT14_OMR_PR12_Pos)                           /*!< PORT14 OMR: PR12 Mask               */
#define PORT14_OMR_PR13_Pos                                   29                                                        /*!< PORT14 OMR: PR13 Position           */
#define PORT14_OMR_PR13_Msk                                   (0x01UL << PORT14_OMR_PR13_Pos)                           /*!< PORT14 OMR: PR13 Mask               */
#define PORT14_OMR_PR14_Pos                                   30                                                        /*!< PORT14 OMR: PR14 Position           */
#define PORT14_OMR_PR14_Msk                                   (0x01UL << PORT14_OMR_PR14_Pos)                           /*!< PORT14 OMR: PR14 Mask               */
#define PORT14_OMR_PR15_Pos                                   31                                                        /*!< PORT14 OMR: PR15 Position           */
#define PORT14_OMR_PR15_Msk                                   (0x01UL << PORT14_OMR_PR15_Pos)                           /*!< PORT14 OMR: PR15 Mask               */

// --------------------------------------  PORT14_IOCR0  ------------------------------------------
#define PORT14_IOCR0_PC0_Pos                                  3                                                         /*!< PORT14 IOCR0: PC0 Position          */
#define PORT14_IOCR0_PC0_Msk                                  (0x1fUL << PORT14_IOCR0_PC0_Pos)                          /*!< PORT14 IOCR0: PC0 Mask              */
#define PORT14_IOCR0_PC1_Pos                                  11                                                        /*!< PORT14 IOCR0: PC1 Position          */
#define PORT14_IOCR0_PC1_Msk                                  (0x1fUL << PORT14_IOCR0_PC1_Pos)                          /*!< PORT14 IOCR0: PC1 Mask              */
#define PORT14_IOCR0_PC2_Pos                                  19                                                        /*!< PORT14 IOCR0: PC2 Position          */
#define PORT14_IOCR0_PC2_Msk                                  (0x1fUL << PORT14_IOCR0_PC2_Pos)                          /*!< PORT14 IOCR0: PC2 Mask              */
#define PORT14_IOCR0_PC3_Pos                                  27                                                        /*!< PORT14 IOCR0: PC3 Position          */
#define PORT14_IOCR0_PC3_Msk                                  (0x1fUL << PORT14_IOCR0_PC3_Pos)                          /*!< PORT14 IOCR0: PC3 Mask              */

// --------------------------------------  PORT14_IOCR4  ------------------------------------------
#define PORT14_IOCR4_PC4_Pos                                  3                                                         /*!< PORT14 IOCR4: PC4 Position          */
#define PORT14_IOCR4_PC4_Msk                                  (0x1fUL << PORT14_IOCR4_PC4_Pos)                          /*!< PORT14 IOCR4: PC4 Mask              */
#define PORT14_IOCR4_PC5_Pos                                  11                                                        /*!< PORT14 IOCR4: PC5 Position          */
#define PORT14_IOCR4_PC5_Msk                                  (0x1fUL << PORT14_IOCR4_PC5_Pos)                          /*!< PORT14 IOCR4: PC5 Mask              */
#define PORT14_IOCR4_PC6_Pos                                  19                                                        /*!< PORT14 IOCR4: PC6 Position          */
#define PORT14_IOCR4_PC6_Msk                                  (0x1fUL << PORT14_IOCR4_PC6_Pos)                          /*!< PORT14 IOCR4: PC6 Mask              */
#define PORT14_IOCR4_PC7_Pos                                  27                                                        /*!< PORT14 IOCR4: PC7 Position          */
#define PORT14_IOCR4_PC7_Msk                                  (0x1fUL << PORT14_IOCR4_PC7_Pos)                          /*!< PORT14 IOCR4: PC7 Mask              */

// --------------------------------------  PORT14_IOCR8  ------------------------------------------
#define PORT14_IOCR8_PC8_Pos                                  3                                                         /*!< PORT14 IOCR8: PC8 Position          */
#define PORT14_IOCR8_PC8_Msk                                  (0x1fUL << PORT14_IOCR8_PC8_Pos)                          /*!< PORT14 IOCR8: PC8 Mask              */
#define PORT14_IOCR8_PC9_Pos                                  11                                                        /*!< PORT14 IOCR8: PC9 Position          */
#define PORT14_IOCR8_PC9_Msk                                  (0x1fUL << PORT14_IOCR8_PC9_Pos)                          /*!< PORT14 IOCR8: PC9 Mask              */
#define PORT14_IOCR8_PC10_Pos                                 19                                                        /*!< PORT14 IOCR8: PC10 Position         */
#define PORT14_IOCR8_PC10_Msk                                 (0x1fUL << PORT14_IOCR8_PC10_Pos)                         /*!< PORT14 IOCR8: PC10 Mask             */
#define PORT14_IOCR8_PC11_Pos                                 27                                                        /*!< PORT14 IOCR8: PC11 Position         */
#define PORT14_IOCR8_PC11_Msk                                 (0x1fUL << PORT14_IOCR8_PC11_Pos)                         /*!< PORT14 IOCR8: PC11 Mask             */

// --------------------------------------  PORT14_IOCR12  -----------------------------------------
#define PORT14_IOCR12_PC12_Pos                                3                                                         /*!< PORT14 IOCR12: PC12 Position        */
#define PORT14_IOCR12_PC12_Msk                                (0x1fUL << PORT14_IOCR12_PC12_Pos)                        /*!< PORT14 IOCR12: PC12 Mask            */
#define PORT14_IOCR12_PC13_Pos                                11                                                        /*!< PORT14 IOCR12: PC13 Position        */
#define PORT14_IOCR12_PC13_Msk                                (0x1fUL << PORT14_IOCR12_PC13_Pos)                        /*!< PORT14 IOCR12: PC13 Mask            */
#define PORT14_IOCR12_PC14_Pos                                19                                                        /*!< PORT14 IOCR12: PC14 Position        */
#define PORT14_IOCR12_PC14_Msk                                (0x1fUL << PORT14_IOCR12_PC14_Pos)                        /*!< PORT14 IOCR12: PC14 Mask            */
#define PORT14_IOCR12_PC15_Pos                                27                                                        /*!< PORT14 IOCR12: PC15 Position        */
#define PORT14_IOCR12_PC15_Msk                                (0x1fUL << PORT14_IOCR12_PC15_Pos)                        /*!< PORT14 IOCR12: PC15 Mask            */

// ----------------------------------------  PORT14_IN  -------------------------------------------
#define PORT14_IN_P0_Pos                                      0                                                         /*!< PORT14 IN: P0 Position              */
#define PORT14_IN_P0_Msk                                      (0x01UL << PORT14_IN_P0_Pos)                              /*!< PORT14 IN: P0 Mask                  */
#define PORT14_IN_P1_Pos                                      1                                                         /*!< PORT14 IN: P1 Position              */
#define PORT14_IN_P1_Msk                                      (0x01UL << PORT14_IN_P1_Pos)                              /*!< PORT14 IN: P1 Mask                  */
#define PORT14_IN_P2_Pos                                      2                                                         /*!< PORT14 IN: P2 Position              */
#define PORT14_IN_P2_Msk                                      (0x01UL << PORT14_IN_P2_Pos)                              /*!< PORT14 IN: P2 Mask                  */
#define PORT14_IN_P3_Pos                                      3                                                         /*!< PORT14 IN: P3 Position              */
#define PORT14_IN_P3_Msk                                      (0x01UL << PORT14_IN_P3_Pos)                              /*!< PORT14 IN: P3 Mask                  */
#define PORT14_IN_P4_Pos                                      4                                                         /*!< PORT14 IN: P4 Position              */
#define PORT14_IN_P4_Msk                                      (0x01UL << PORT14_IN_P4_Pos)                              /*!< PORT14 IN: P4 Mask                  */
#define PORT14_IN_P5_Pos                                      5                                                         /*!< PORT14 IN: P5 Position              */
#define PORT14_IN_P5_Msk                                      (0x01UL << PORT14_IN_P5_Pos)                              /*!< PORT14 IN: P5 Mask                  */
#define PORT14_IN_P6_Pos                                      6                                                         /*!< PORT14 IN: P6 Position              */
#define PORT14_IN_P6_Msk                                      (0x01UL << PORT14_IN_P6_Pos)                              /*!< PORT14 IN: P6 Mask                  */
#define PORT14_IN_P7_Pos                                      7                                                         /*!< PORT14 IN: P7 Position              */
#define PORT14_IN_P7_Msk                                      (0x01UL << PORT14_IN_P7_Pos)                              /*!< PORT14 IN: P7 Mask                  */
#define PORT14_IN_P8_Pos                                      8                                                         /*!< PORT14 IN: P8 Position              */
#define PORT14_IN_P8_Msk                                      (0x01UL << PORT14_IN_P8_Pos)                              /*!< PORT14 IN: P8 Mask                  */
#define PORT14_IN_P9_Pos                                      9                                                         /*!< PORT14 IN: P9 Position              */
#define PORT14_IN_P9_Msk                                      (0x01UL << PORT14_IN_P9_Pos)                              /*!< PORT14 IN: P9 Mask                  */
#define PORT14_IN_P10_Pos                                     10                                                        /*!< PORT14 IN: P10 Position             */
#define PORT14_IN_P10_Msk                                     (0x01UL << PORT14_IN_P10_Pos)                             /*!< PORT14 IN: P10 Mask                 */
#define PORT14_IN_P11_Pos                                     11                                                        /*!< PORT14 IN: P11 Position             */
#define PORT14_IN_P11_Msk                                     (0x01UL << PORT14_IN_P11_Pos)                             /*!< PORT14 IN: P11 Mask                 */
#define PORT14_IN_P12_Pos                                     12                                                        /*!< PORT14 IN: P12 Position             */
#define PORT14_IN_P12_Msk                                     (0x01UL << PORT14_IN_P12_Pos)                             /*!< PORT14 IN: P12 Mask                 */
#define PORT14_IN_P13_Pos                                     13                                                        /*!< PORT14 IN: P13 Position             */
#define PORT14_IN_P13_Msk                                     (0x01UL << PORT14_IN_P13_Pos)                             /*!< PORT14 IN: P13 Mask                 */
#define PORT14_IN_P14_Pos                                     14                                                        /*!< PORT14 IN: P14 Position             */
#define PORT14_IN_P14_Msk                                     (0x01UL << PORT14_IN_P14_Pos)                             /*!< PORT14 IN: P14 Mask                 */
#define PORT14_IN_P15_Pos                                     15                                                        /*!< PORT14 IN: P15 Position             */
#define PORT14_IN_P15_Msk                                     (0x01UL << PORT14_IN_P15_Pos)                             /*!< PORT14 IN: P15 Mask                 */

// --------------------------------------  PORT14_PDISC  ------------------------------------------
#define PORT14_PDISC_PDIS0_Pos                                0                                                         /*!< PORT14 PDISC: PDIS0 Position        */
#define PORT14_PDISC_PDIS0_Msk                                (0x01UL << PORT14_PDISC_PDIS0_Pos)                        /*!< PORT14 PDISC: PDIS0 Mask            */
#define PORT14_PDISC_PDIS1_Pos                                1                                                         /*!< PORT14 PDISC: PDIS1 Position        */
#define PORT14_PDISC_PDIS1_Msk                                (0x01UL << PORT14_PDISC_PDIS1_Pos)                        /*!< PORT14 PDISC: PDIS1 Mask            */
#define PORT14_PDISC_PDIS2_Pos                                2                                                         /*!< PORT14 PDISC: PDIS2 Position        */
#define PORT14_PDISC_PDIS2_Msk                                (0x01UL << PORT14_PDISC_PDIS2_Pos)                        /*!< PORT14 PDISC: PDIS2 Mask            */
#define PORT14_PDISC_PDIS3_Pos                                3                                                         /*!< PORT14 PDISC: PDIS3 Position        */
#define PORT14_PDISC_PDIS3_Msk                                (0x01UL << PORT14_PDISC_PDIS3_Pos)                        /*!< PORT14 PDISC: PDIS3 Mask            */
#define PORT14_PDISC_PDIS4_Pos                                4                                                         /*!< PORT14 PDISC: PDIS4 Position        */
#define PORT14_PDISC_PDIS4_Msk                                (0x01UL << PORT14_PDISC_PDIS4_Pos)                        /*!< PORT14 PDISC: PDIS4 Mask            */
#define PORT14_PDISC_PDIS5_Pos                                5                                                         /*!< PORT14 PDISC: PDIS5 Position        */
#define PORT14_PDISC_PDIS5_Msk                                (0x01UL << PORT14_PDISC_PDIS5_Pos)                        /*!< PORT14 PDISC: PDIS5 Mask            */
#define PORT14_PDISC_PDIS6_Pos                                6                                                         /*!< PORT14 PDISC: PDIS6 Position        */
#define PORT14_PDISC_PDIS6_Msk                                (0x01UL << PORT14_PDISC_PDIS6_Pos)                        /*!< PORT14 PDISC: PDIS6 Mask            */
#define PORT14_PDISC_PDIS7_Pos                                7                                                         /*!< PORT14 PDISC: PDIS7 Position        */
#define PORT14_PDISC_PDIS7_Msk                                (0x01UL << PORT14_PDISC_PDIS7_Pos)                        /*!< PORT14 PDISC: PDIS7 Mask            */
#define PORT14_PDISC_PDIS8_Pos                                8                                                         /*!< PORT14 PDISC: PDIS8 Position        */
#define PORT14_PDISC_PDIS8_Msk                                (0x01UL << PORT14_PDISC_PDIS8_Pos)                        /*!< PORT14 PDISC: PDIS8 Mask            */
#define PORT14_PDISC_PDIS9_Pos                                9                                                         /*!< PORT14 PDISC: PDIS9 Position        */
#define PORT14_PDISC_PDIS9_Msk                                (0x01UL << PORT14_PDISC_PDIS9_Pos)                        /*!< PORT14 PDISC: PDIS9 Mask            */
#define PORT14_PDISC_PDIS12_Pos                               12                                                        /*!< PORT14 PDISC: PDIS12 Position       */
#define PORT14_PDISC_PDIS12_Msk                               (0x01UL << PORT14_PDISC_PDIS12_Pos)                       /*!< PORT14 PDISC: PDIS12 Mask           */
#define PORT14_PDISC_PDIS13_Pos                               13                                                        /*!< PORT14 PDISC: PDIS13 Position       */
#define PORT14_PDISC_PDIS13_Msk                               (0x01UL << PORT14_PDISC_PDIS13_Pos)                       /*!< PORT14 PDISC: PDIS13 Mask           */
#define PORT14_PDISC_PDIS14_Pos                               14                                                        /*!< PORT14 PDISC: PDIS14 Position       */
#define PORT14_PDISC_PDIS14_Msk                               (0x01UL << PORT14_PDISC_PDIS14_Pos)                       /*!< PORT14 PDISC: PDIS14 Mask           */
#define PORT14_PDISC_PDIS15_Pos                               15                                                        /*!< PORT14 PDISC: PDIS15 Position       */
#define PORT14_PDISC_PDIS15_Msk                               (0x01UL << PORT14_PDISC_PDIS15_Pos)                       /*!< PORT14 PDISC: PDIS15 Mask           */

// ---------------------------------------  PORT14_PPS  -------------------------------------------
#define PORT14_PPS_PPS0_Pos                                   0                                                         /*!< PORT14 PPS: PPS0 Position           */
#define PORT14_PPS_PPS0_Msk                                   (0x01UL << PORT14_PPS_PPS0_Pos)                           /*!< PORT14 PPS: PPS0 Mask               */
#define PORT14_PPS_PPS1_Pos                                   1                                                         /*!< PORT14 PPS: PPS1 Position           */
#define PORT14_PPS_PPS1_Msk                                   (0x01UL << PORT14_PPS_PPS1_Pos)                           /*!< PORT14 PPS: PPS1 Mask               */
#define PORT14_PPS_PPS2_Pos                                   2                                                         /*!< PORT14 PPS: PPS2 Position           */
#define PORT14_PPS_PPS2_Msk                                   (0x01UL << PORT14_PPS_PPS2_Pos)                           /*!< PORT14 PPS: PPS2 Mask               */
#define PORT14_PPS_PPS3_Pos                                   3                                                         /*!< PORT14 PPS: PPS3 Position           */
#define PORT14_PPS_PPS3_Msk                                   (0x01UL << PORT14_PPS_PPS3_Pos)                           /*!< PORT14 PPS: PPS3 Mask               */
#define PORT14_PPS_PPS4_Pos                                   4                                                         /*!< PORT14 PPS: PPS4 Position           */
#define PORT14_PPS_PPS4_Msk                                   (0x01UL << PORT14_PPS_PPS4_Pos)                           /*!< PORT14 PPS: PPS4 Mask               */
#define PORT14_PPS_PPS5_Pos                                   5                                                         /*!< PORT14 PPS: PPS5 Position           */
#define PORT14_PPS_PPS5_Msk                                   (0x01UL << PORT14_PPS_PPS5_Pos)                           /*!< PORT14 PPS: PPS5 Mask               */
#define PORT14_PPS_PPS6_Pos                                   6                                                         /*!< PORT14 PPS: PPS6 Position           */
#define PORT14_PPS_PPS6_Msk                                   (0x01UL << PORT14_PPS_PPS6_Pos)                           /*!< PORT14 PPS: PPS6 Mask               */
#define PORT14_PPS_PPS7_Pos                                   7                                                         /*!< PORT14 PPS: PPS7 Position           */
#define PORT14_PPS_PPS7_Msk                                   (0x01UL << PORT14_PPS_PPS7_Pos)                           /*!< PORT14 PPS: PPS7 Mask               */
#define PORT14_PPS_PPS8_Pos                                   8                                                         /*!< PORT14 PPS: PPS8 Position           */
#define PORT14_PPS_PPS8_Msk                                   (0x01UL << PORT14_PPS_PPS8_Pos)                           /*!< PORT14 PPS: PPS8 Mask               */
#define PORT14_PPS_PPS9_Pos                                   9                                                         /*!< PORT14 PPS: PPS9 Position           */
#define PORT14_PPS_PPS9_Msk                                   (0x01UL << PORT14_PPS_PPS9_Pos)                           /*!< PORT14 PPS: PPS9 Mask               */
#define PORT14_PPS_PPS10_Pos                                  10                                                        /*!< PORT14 PPS: PPS10 Position          */
#define PORT14_PPS_PPS10_Msk                                  (0x01UL << PORT14_PPS_PPS10_Pos)                          /*!< PORT14 PPS: PPS10 Mask              */
#define PORT14_PPS_PPS11_Pos                                  11                                                        /*!< PORT14 PPS: PPS11 Position          */
#define PORT14_PPS_PPS11_Msk                                  (0x01UL << PORT14_PPS_PPS11_Pos)                          /*!< PORT14 PPS: PPS11 Mask              */
#define PORT14_PPS_PPS12_Pos                                  12                                                        /*!< PORT14 PPS: PPS12 Position          */
#define PORT14_PPS_PPS12_Msk                                  (0x01UL << PORT14_PPS_PPS12_Pos)                          /*!< PORT14 PPS: PPS12 Mask              */
#define PORT14_PPS_PPS13_Pos                                  13                                                        /*!< PORT14 PPS: PPS13 Position          */
#define PORT14_PPS_PPS13_Msk                                  (0x01UL << PORT14_PPS_PPS13_Pos)                          /*!< PORT14 PPS: PPS13 Mask              */
#define PORT14_PPS_PPS14_Pos                                  14                                                        /*!< PORT14 PPS: PPS14 Position          */
#define PORT14_PPS_PPS14_Msk                                  (0x01UL << PORT14_PPS_PPS14_Pos)                          /*!< PORT14 PPS: PPS14 Mask              */
#define PORT14_PPS_PPS15_Pos                                  15                                                        /*!< PORT14 PPS: PPS15 Position          */
#define PORT14_PPS_PPS15_Msk                                  (0x01UL << PORT14_PPS_PPS15_Pos)                          /*!< PORT14 PPS: PPS15 Mask              */

// --------------------------------------  PORT14_HWSEL  ------------------------------------------
#define PORT14_HWSEL_HW0_Pos                                  0                                                         /*!< PORT14 HWSEL: HW0 Position          */
#define PORT14_HWSEL_HW0_Msk                                  (0x03UL << PORT14_HWSEL_HW0_Pos)                          /*!< PORT14 HWSEL: HW0 Mask              */
#define PORT14_HWSEL_HW1_Pos                                  2                                                         /*!< PORT14 HWSEL: HW1 Position          */
#define PORT14_HWSEL_HW1_Msk                                  (0x03UL << PORT14_HWSEL_HW1_Pos)                          /*!< PORT14 HWSEL: HW1 Mask              */
#define PORT14_HWSEL_HW2_Pos                                  4                                                         /*!< PORT14 HWSEL: HW2 Position          */
#define PORT14_HWSEL_HW2_Msk                                  (0x03UL << PORT14_HWSEL_HW2_Pos)                          /*!< PORT14 HWSEL: HW2 Mask              */
#define PORT14_HWSEL_HW3_Pos                                  6                                                         /*!< PORT14 HWSEL: HW3 Position          */
#define PORT14_HWSEL_HW3_Msk                                  (0x03UL << PORT14_HWSEL_HW3_Pos)                          /*!< PORT14 HWSEL: HW3 Mask              */
#define PORT14_HWSEL_HW4_Pos                                  8                                                         /*!< PORT14 HWSEL: HW4 Position          */
#define PORT14_HWSEL_HW4_Msk                                  (0x03UL << PORT14_HWSEL_HW4_Pos)                          /*!< PORT14 HWSEL: HW4 Mask              */
#define PORT14_HWSEL_HW5_Pos                                  10                                                        /*!< PORT14 HWSEL: HW5 Position          */
#define PORT14_HWSEL_HW5_Msk                                  (0x03UL << PORT14_HWSEL_HW5_Pos)                          /*!< PORT14 HWSEL: HW5 Mask              */
#define PORT14_HWSEL_HW6_Pos                                  12                                                        /*!< PORT14 HWSEL: HW6 Position          */
#define PORT14_HWSEL_HW6_Msk                                  (0x03UL << PORT14_HWSEL_HW6_Pos)                          /*!< PORT14 HWSEL: HW6 Mask              */
#define PORT14_HWSEL_HW7_Pos                                  14                                                        /*!< PORT14 HWSEL: HW7 Position          */
#define PORT14_HWSEL_HW7_Msk                                  (0x03UL << PORT14_HWSEL_HW7_Pos)                          /*!< PORT14 HWSEL: HW7 Mask              */
#define PORT14_HWSEL_HW8_Pos                                  16                                                        /*!< PORT14 HWSEL: HW8 Position          */
#define PORT14_HWSEL_HW8_Msk                                  (0x03UL << PORT14_HWSEL_HW8_Pos)                          /*!< PORT14 HWSEL: HW8 Mask              */
#define PORT14_HWSEL_HW9_Pos                                  18                                                        /*!< PORT14 HWSEL: HW9 Position          */
#define PORT14_HWSEL_HW9_Msk                                  (0x03UL << PORT14_HWSEL_HW9_Pos)                          /*!< PORT14 HWSEL: HW9 Mask              */
#define PORT14_HWSEL_HW10_Pos                                 20                                                        /*!< PORT14 HWSEL: HW10 Position         */
#define PORT14_HWSEL_HW10_Msk                                 (0x03UL << PORT14_HWSEL_HW10_Pos)                         /*!< PORT14 HWSEL: HW10 Mask             */
#define PORT14_HWSEL_HW11_Pos                                 22                                                        /*!< PORT14 HWSEL: HW11 Position         */
#define PORT14_HWSEL_HW11_Msk                                 (0x03UL << PORT14_HWSEL_HW11_Pos)                         /*!< PORT14 HWSEL: HW11 Mask             */
#define PORT14_HWSEL_HW12_Pos                                 24                                                        /*!< PORT14 HWSEL: HW12 Position         */
#define PORT14_HWSEL_HW12_Msk                                 (0x03UL << PORT14_HWSEL_HW12_Pos)                         /*!< PORT14 HWSEL: HW12 Mask             */
#define PORT14_HWSEL_HW13_Pos                                 26                                                        /*!< PORT14 HWSEL: HW13 Position         */
#define PORT14_HWSEL_HW13_Msk                                 (0x03UL << PORT14_HWSEL_HW13_Pos)                         /*!< PORT14 HWSEL: HW13 Mask             */
#define PORT14_HWSEL_HW14_Pos                                 28                                                        /*!< PORT14 HWSEL: HW14 Position         */
#define PORT14_HWSEL_HW14_Msk                                 (0x03UL << PORT14_HWSEL_HW14_Pos)                         /*!< PORT14 HWSEL: HW14 Mask             */
#define PORT14_HWSEL_HW15_Pos                                 30                                                        /*!< PORT14 HWSEL: HW15 Position         */
#define PORT14_HWSEL_HW15_Msk                                 (0x03UL << PORT14_HWSEL_HW15_Pos)                         /*!< PORT14 HWSEL: HW15 Mask             */


// ------------------------------------------------------------------------------------------------
// -----                                PORT15 Position & Mask                                -----
// ------------------------------------------------------------------------------------------------


// ---------------------------------------  PORT15_OUT  -------------------------------------------
#define PORT15_OUT_P0_Pos                                     0                                                         /*!< PORT15 OUT: P0 Position             */
#define PORT15_OUT_P0_Msk                                     (0x01UL << PORT15_OUT_P0_Pos)                             /*!< PORT15 OUT: P0 Mask                 */
#define PORT15_OUT_P1_Pos                                     1                                                         /*!< PORT15 OUT: P1 Position             */
#define PORT15_OUT_P1_Msk                                     (0x01UL << PORT15_OUT_P1_Pos)                             /*!< PORT15 OUT: P1 Mask                 */
#define PORT15_OUT_P2_Pos                                     2                                                         /*!< PORT15 OUT: P2 Position             */
#define PORT15_OUT_P2_Msk                                     (0x01UL << PORT15_OUT_P2_Pos)                             /*!< PORT15 OUT: P2 Mask                 */
#define PORT15_OUT_P3_Pos                                     3                                                         /*!< PORT15 OUT: P3 Position             */
#define PORT15_OUT_P3_Msk                                     (0x01UL << PORT15_OUT_P3_Pos)                             /*!< PORT15 OUT: P3 Mask                 */
#define PORT15_OUT_P4_Pos                                     4                                                         /*!< PORT15 OUT: P4 Position             */
#define PORT15_OUT_P4_Msk                                     (0x01UL << PORT15_OUT_P4_Pos)                             /*!< PORT15 OUT: P4 Mask                 */
#define PORT15_OUT_P5_Pos                                     5                                                         /*!< PORT15 OUT: P5 Position             */
#define PORT15_OUT_P5_Msk                                     (0x01UL << PORT15_OUT_P5_Pos)                             /*!< PORT15 OUT: P5 Mask                 */
#define PORT15_OUT_P6_Pos                                     6                                                         /*!< PORT15 OUT: P6 Position             */
#define PORT15_OUT_P6_Msk                                     (0x01UL << PORT15_OUT_P6_Pos)                             /*!< PORT15 OUT: P6 Mask                 */
#define PORT15_OUT_P7_Pos                                     7                                                         /*!< PORT15 OUT: P7 Position             */
#define PORT15_OUT_P7_Msk                                     (0x01UL << PORT15_OUT_P7_Pos)                             /*!< PORT15 OUT: P7 Mask                 */
#define PORT15_OUT_P8_Pos                                     8                                                         /*!< PORT15 OUT: P8 Position             */
#define PORT15_OUT_P8_Msk                                     (0x01UL << PORT15_OUT_P8_Pos)                             /*!< PORT15 OUT: P8 Mask                 */
#define PORT15_OUT_P9_Pos                                     9                                                         /*!< PORT15 OUT: P9 Position             */
#define PORT15_OUT_P9_Msk                                     (0x01UL << PORT15_OUT_P9_Pos)                             /*!< PORT15 OUT: P9 Mask                 */
#define PORT15_OUT_P10_Pos                                    10                                                        /*!< PORT15 OUT: P10 Position            */
#define PORT15_OUT_P10_Msk                                    (0x01UL << PORT15_OUT_P10_Pos)                            /*!< PORT15 OUT: P10 Mask                */
#define PORT15_OUT_P11_Pos                                    11                                                        /*!< PORT15 OUT: P11 Position            */
#define PORT15_OUT_P11_Msk                                    (0x01UL << PORT15_OUT_P11_Pos)                            /*!< PORT15 OUT: P11 Mask                */
#define PORT15_OUT_P12_Pos                                    12                                                        /*!< PORT15 OUT: P12 Position            */
#define PORT15_OUT_P12_Msk                                    (0x01UL << PORT15_OUT_P12_Pos)                            /*!< PORT15 OUT: P12 Mask                */
#define PORT15_OUT_P13_Pos                                    13                                                        /*!< PORT15 OUT: P13 Position            */
#define PORT15_OUT_P13_Msk                                    (0x01UL << PORT15_OUT_P13_Pos)                            /*!< PORT15 OUT: P13 Mask                */
#define PORT15_OUT_P14_Pos                                    14                                                        /*!< PORT15 OUT: P14 Position            */
#define PORT15_OUT_P14_Msk                                    (0x01UL << PORT15_OUT_P14_Pos)                            /*!< PORT15 OUT: P14 Mask                */
#define PORT15_OUT_P15_Pos                                    15                                                        /*!< PORT15 OUT: P15 Position            */
#define PORT15_OUT_P15_Msk                                    (0x01UL << PORT15_OUT_P15_Pos)                            /*!< PORT15 OUT: P15 Mask                */

// ---------------------------------------  PORT15_OMR  -------------------------------------------
#define PORT15_OMR_PS0_Pos                                    0                                                         /*!< PORT15 OMR: PS0 Position            */
#define PORT15_OMR_PS0_Msk                                    (0x01UL << PORT15_OMR_PS0_Pos)                            /*!< PORT15 OMR: PS0 Mask                */
#define PORT15_OMR_PS1_Pos                                    1                                                         /*!< PORT15 OMR: PS1 Position            */
#define PORT15_OMR_PS1_Msk                                    (0x01UL << PORT15_OMR_PS1_Pos)                            /*!< PORT15 OMR: PS1 Mask                */
#define PORT15_OMR_PS2_Pos                                    2                                                         /*!< PORT15 OMR: PS2 Position            */
#define PORT15_OMR_PS2_Msk                                    (0x01UL << PORT15_OMR_PS2_Pos)                            /*!< PORT15 OMR: PS2 Mask                */
#define PORT15_OMR_PS3_Pos                                    3                                                         /*!< PORT15 OMR: PS3 Position            */
#define PORT15_OMR_PS3_Msk                                    (0x01UL << PORT15_OMR_PS3_Pos)                            /*!< PORT15 OMR: PS3 Mask                */
#define PORT15_OMR_PS4_Pos                                    4                                                         /*!< PORT15 OMR: PS4 Position            */
#define PORT15_OMR_PS4_Msk                                    (0x01UL << PORT15_OMR_PS4_Pos)                            /*!< PORT15 OMR: PS4 Mask                */
#define PORT15_OMR_PS5_Pos                                    5                                                         /*!< PORT15 OMR: PS5 Position            */
#define PORT15_OMR_PS5_Msk                                    (0x01UL << PORT15_OMR_PS5_Pos)                            /*!< PORT15 OMR: PS5 Mask                */
#define PORT15_OMR_PS6_Pos                                    6                                                         /*!< PORT15 OMR: PS6 Position            */
#define PORT15_OMR_PS6_Msk                                    (0x01UL << PORT15_OMR_PS6_Pos)                            /*!< PORT15 OMR: PS6 Mask                */
#define PORT15_OMR_PS7_Pos                                    7                                                         /*!< PORT15 OMR: PS7 Position            */
#define PORT15_OMR_PS7_Msk                                    (0x01UL << PORT15_OMR_PS7_Pos)                            /*!< PORT15 OMR: PS7 Mask                */
#define PORT15_OMR_PS8_Pos                                    8                                                         /*!< PORT15 OMR: PS8 Position            */
#define PORT15_OMR_PS8_Msk                                    (0x01UL << PORT15_OMR_PS8_Pos)                            /*!< PORT15 OMR: PS8 Mask                */
#define PORT15_OMR_PS9_Pos                                    9                                                         /*!< PORT15 OMR: PS9 Position            */
#define PORT15_OMR_PS9_Msk                                    (0x01UL << PORT15_OMR_PS9_Pos)                            /*!< PORT15 OMR: PS9 Mask                */
#define PORT15_OMR_PS10_Pos                                   10                                                        /*!< PORT15 OMR: PS10 Position           */
#define PORT15_OMR_PS10_Msk                                   (0x01UL << PORT15_OMR_PS10_Pos)                           /*!< PORT15 OMR: PS10 Mask               */
#define PORT15_OMR_PS11_Pos                                   11                                                        /*!< PORT15 OMR: PS11 Position           */
#define PORT15_OMR_PS11_Msk                                   (0x01UL << PORT15_OMR_PS11_Pos)                           /*!< PORT15 OMR: PS11 Mask               */
#define PORT15_OMR_PS12_Pos                                   12                                                        /*!< PORT15 OMR: PS12 Position           */
#define PORT15_OMR_PS12_Msk                                   (0x01UL << PORT15_OMR_PS12_Pos)                           /*!< PORT15 OMR: PS12 Mask               */
#define PORT15_OMR_PS13_Pos                                   13                                                        /*!< PORT15 OMR: PS13 Position           */
#define PORT15_OMR_PS13_Msk                                   (0x01UL << PORT15_OMR_PS13_Pos)                           /*!< PORT15 OMR: PS13 Mask               */
#define PORT15_OMR_PS14_Pos                                   14                                                        /*!< PORT15 OMR: PS14 Position           */
#define PORT15_OMR_PS14_Msk                                   (0x01UL << PORT15_OMR_PS14_Pos)                           /*!< PORT15 OMR: PS14 Mask               */
#define PORT15_OMR_PS15_Pos                                   15                                                        /*!< PORT15 OMR: PS15 Position           */
#define PORT15_OMR_PS15_Msk                                   (0x01UL << PORT15_OMR_PS15_Pos)                           /*!< PORT15 OMR: PS15 Mask               */
#define PORT15_OMR_PR0_Pos                                    16                                                        /*!< PORT15 OMR: PR0 Position            */
#define PORT15_OMR_PR0_Msk                                    (0x01UL << PORT15_OMR_PR0_Pos)                            /*!< PORT15 OMR: PR0 Mask                */
#define PORT15_OMR_PR1_Pos                                    17                                                        /*!< PORT15 OMR: PR1 Position            */
#define PORT15_OMR_PR1_Msk                                    (0x01UL << PORT15_OMR_PR1_Pos)                            /*!< PORT15 OMR: PR1 Mask                */
#define PORT15_OMR_PR2_Pos                                    18                                                        /*!< PORT15 OMR: PR2 Position            */
#define PORT15_OMR_PR2_Msk                                    (0x01UL << PORT15_OMR_PR2_Pos)                            /*!< PORT15 OMR: PR2 Mask                */
#define PORT15_OMR_PR3_Pos                                    19                                                        /*!< PORT15 OMR: PR3 Position            */
#define PORT15_OMR_PR3_Msk                                    (0x01UL << PORT15_OMR_PR3_Pos)                            /*!< PORT15 OMR: PR3 Mask                */
#define PORT15_OMR_PR4_Pos                                    20                                                        /*!< PORT15 OMR: PR4 Position            */
#define PORT15_OMR_PR4_Msk                                    (0x01UL << PORT15_OMR_PR4_Pos)                            /*!< PORT15 OMR: PR4 Mask                */
#define PORT15_OMR_PR5_Pos                                    21                                                        /*!< PORT15 OMR: PR5 Position            */
#define PORT15_OMR_PR5_Msk                                    (0x01UL << PORT15_OMR_PR5_Pos)                            /*!< PORT15 OMR: PR5 Mask                */
#define PORT15_OMR_PR6_Pos                                    22                                                        /*!< PORT15 OMR: PR6 Position            */
#define PORT15_OMR_PR6_Msk                                    (0x01UL << PORT15_OMR_PR6_Pos)                            /*!< PORT15 OMR: PR6 Mask                */
#define PORT15_OMR_PR7_Pos                                    23                                                        /*!< PORT15 OMR: PR7 Position            */
#define PORT15_OMR_PR7_Msk                                    (0x01UL << PORT15_OMR_PR7_Pos)                            /*!< PORT15 OMR: PR7 Mask                */
#define PORT15_OMR_PR8_Pos                                    24                                                        /*!< PORT15 OMR: PR8 Position            */
#define PORT15_OMR_PR8_Msk                                    (0x01UL << PORT15_OMR_PR8_Pos)                            /*!< PORT15 OMR: PR8 Mask                */
#define PORT15_OMR_PR9_Pos                                    25                                                        /*!< PORT15 OMR: PR9 Position            */
#define PORT15_OMR_PR9_Msk                                    (0x01UL << PORT15_OMR_PR9_Pos)                            /*!< PORT15 OMR: PR9 Mask                */
#define PORT15_OMR_PR10_Pos                                   26                                                        /*!< PORT15 OMR: PR10 Position           */
#define PORT15_OMR_PR10_Msk                                   (0x01UL << PORT15_OMR_PR10_Pos)                           /*!< PORT15 OMR: PR10 Mask               */
#define PORT15_OMR_PR11_Pos                                   27                                                        /*!< PORT15 OMR: PR11 Position           */
#define PORT15_OMR_PR11_Msk                                   (0x01UL << PORT15_OMR_PR11_Pos)                           /*!< PORT15 OMR: PR11 Mask               */
#define PORT15_OMR_PR12_Pos                                   28                                                        /*!< PORT15 OMR: PR12 Position           */
#define PORT15_OMR_PR12_Msk                                   (0x01UL << PORT15_OMR_PR12_Pos)                           /*!< PORT15 OMR: PR12 Mask               */
#define PORT15_OMR_PR13_Pos                                   29                                                        /*!< PORT15 OMR: PR13 Position           */
#define PORT15_OMR_PR13_Msk                                   (0x01UL << PORT15_OMR_PR13_Pos)                           /*!< PORT15 OMR: PR13 Mask               */
#define PORT15_OMR_PR14_Pos                                   30                                                        /*!< PORT15 OMR: PR14 Position           */
#define PORT15_OMR_PR14_Msk                                   (0x01UL << PORT15_OMR_PR14_Pos)                           /*!< PORT15 OMR: PR14 Mask               */
#define PORT15_OMR_PR15_Pos                                   31                                                        /*!< PORT15 OMR: PR15 Position           */
#define PORT15_OMR_PR15_Msk                                   (0x01UL << PORT15_OMR_PR15_Pos)                           /*!< PORT15 OMR: PR15 Mask               */

// --------------------------------------  PORT15_IOCR0  ------------------------------------------
#define PORT15_IOCR0_PC0_Po